/* cyl.c: 
 *           read in a heightfield
 *           generate a depth-mapped cylindrical shape using triangles
 *           by John P. Beale 9/1/95
 *           uses GIF library routines from gd library from Tom Boutell
 */

#include <stdio.h>                      /* fopen() sprintf() etc. */
#include <stdlib.h>                     /* strtod() */
#include <math.h>                       /* math stuff */
#include <string.h>                     /* strcpy() */
#include "hcon.h"
#include "hf_io.h"

typedef struct point_struct {
  double x;
  double y;
  double z;
} pnt;

int gen_cyl(char *fname_out);            /* generate depth-mapped cylinder */
                                         /* project point on cylinder */
void project(int i,int j,pnt *p1, pnt*p2,pnt*p3,pnt*p4);


U xsize, ysize;     /* size of hf array (input xsize+2) */
PTYPE *hf;                            /* height field array */
PTYPE fmin = 0, fmax = 1.0;                 /* scaling limits */

double rad_min, rad_max, height;   /* cylinder parameters */
double rfac,xfac,yfac;             /* scaling constants */
int naround, nhigh;

double pi2 = 6.28318530717959;
Boolean norenorm = FALSE;
Boolean negate = FALSE;
Boolean rescale = FALSE;
Boolean raw_out = FALSE;

/* ---------------------------------- */

int main(int argc, char **argv)
{ char *usage = 
"cyl: An HF -> Cylindrical HF converter v0.1a (c) John P. Beale 1995\n\
Usage: cyl <infile> <outfile> <rad1> <rad2> <height> <n_circum> <n_high>\n\
  Available input formats: PGM|MAT|GIF|POT|TGA, output RAW or INC triangles.\n\
  Determines filetype based on extension, eg:  cyl file1.tga file1.inc\n\
  POT,TGA are 16-bit formats, MAT is Matlab floating point format.\n\
  rad1 and rad2 are minimum and maximum radii of resulting bumpy cylinder\n\
  height is height of cylinder, which starts at z=0 and extends to z=height\n\
  n_circum and n_high are vertices in circumferential and vertical directions\n";
  
  char fname_in[160];
  char fname_out[160];
  char *ext1, *ext2;
  char *buf;
  int rval;
  int inlen, outlen, itype;

  if ((argc < 3) || (argc > 8)) {
	fprintf(stderr,"%s",usage);
	exit(1);
      }
 
  strcpy(fname_in,argv[1]);
  strcpy(fname_out,argv[2]);
  rad_min = 1.0;
  rad_max = 2.0;
  height = 1.0;
  naround = 10;
  nhigh = 10;  

  if (argc > 3) {
    rad_min = strtod(argv[3], &buf);
    }
  if (argc > 4) {
    rad_max = strtod(argv[4], &buf);
    }
  if (argc > 5) {
    height = strtod(argv[5], &buf);
    }
  if (argc > 6) {
    naround = (int) strtod(argv[6], &buf);
    }
  if (argc > 7) {
    nhigh = (int) strtod(argv[7], &buf);
    }

  printf("%s->%s radius(%1.3g..%1.3g) height %1.3g vertices(%dx%d)\n",
         fname_in,fname_out,rad_min,rad_max,height,naround,nhigh);

  inlen = strlen(fname_in);
  outlen = strlen(fname_out);
  
  ext1 = (char *) malloc((size_t)sizeof(char)*5);
  strncpy(ext1,(char *)(&fname_in[inlen-4]),4);
  ext2 = (char *) malloc((size_t)sizeof(char)*5);
  strncpy(ext2,(char *)(&fname_out[outlen-4]),4);

  if ((ext1==NULL)) { 
    fprintf(stderr,"filenames must have type extension, eg. topo.gif\n");
    exit(1);
  }

  if       (strcmp(ext1,".gif")==0) itype = GIF;  
   else if (strcmp(ext1,".pot")==0) itype = POT;
   else if (strcmp(ext1,".tga")==0) itype = TGA;
   else if (strcmp(ext1,".pgm")==0) itype = PGM;
   else if (strcmp(ext1,".pg8")==0) itype = PG8;
   else if (strcmp(ext1,".mat")==0) itype = MAT;
   else if (strcmp(ext1,".oct")==0) itype = OCT;
   else if (strcmp(ext1,".png")==0) itype = PNG;
   else { 
     itype = PGM;       /* default type */
     fprintf(stderr,"Warning: %s has a missing or unknown file extension\n",argv[1]);
   }

  if (strcmp(ext2,".raw")==0) raw_out = TRUE;

					/* read in heightfield */
   rval = read_hf(&hf, &xsize, &ysize, fname_in, itype);
   if (rval != 0)
     exit(rval);

   printf("%s: [%d x %d] \n",fname_in,xsize,ysize);
   rfac = (rad_max - rad_min);
   xfac = (double) xsize / naround;
   yfac = (double) ysize / nhigh;

   if (raw_out)   printf("Generating raw triangle file...");
    else printf("Generating POV .inc triangle file...");
   fflush(stdout);

   gen_cyl(fname_out);
   printf("\n"); 
  
   return(rval);    /* return a value */
} /* end main() */

void hf_print()                /* print out hf array on stdout */
{
  int ix,iy;

   printf("Heightfield array: %dx%d\n",xsize,ysize);
   for (iy = 0; iy<ysize; iy++) {
    printf("%d:",iy);
    for (ix = 0; ix<xsize; ix++) {
	printf(" %1.2f",El(hf,ix,iy));
    }
     printf("\n");
   }
}



int gen_cyl(char *fname_out)
{
    int i,j;
    FILE *fp;
    pnt p1,p2,p3,p4;

    fp = fopen(fname_out,"w");

    if (raw_out) fprintf(fp,"CYL\n");    /* RAW text format */
    else {
      fprintf(fp,"/* Include file for POV-Ray : cylinder from heightfield */\n");
      fprintf(fp,"union {\n");
    }
    /* Spit out coordinates for each triangle */
    for (i = 0; i < naround; i++) {
     for (j = 0; j < nhigh; j++) {
       project(i,j,&p1,&p2,&p3,&p4);
       if (raw_out) {
         fprintf(fp,"%1.5g %1.5g %1.5g %1.5g %1.5g %1.5g %1.5g %1.5g %1.5g\n",
		 p1.x,p1.y,p1.z,p2.x,p2.y,p2.z,p3.x,p3.y,p3.z);
         fprintf(fp,"%1.5g %1.5g %1.5g %1.5g %1.5g %1.5g %1.5g %1.5g %1.5g\n",
		 p3.x,p3.y,p3.z,p2.x,p2.y,p2.z,p4.x,p4.y,p4.z);
       } else {
	 fprintf(fp,"triangle{<%1.5g,%1.5g,%1.5g>,<%1.5g,%1.5g,%1.5g>,"
		 "<%1.5g,%1.5g,%1.5g>}\n",
		 p1.x,p1.y,p1.z,p2.x,p2.y,p2.z,p3.x,p3.y,p3.z);
	 fprintf(fp,"triangle{<%1.5g,%1.5g,%1.5g>,<%1.5g,%1.5g,%1.5g>,"
		 "<%1.5g,%1.5g,%1.5g>}\n",
		 p3.x,p3.y,p3.z,p2.x,p2.y,p2.z,p4.x,p4.y,p4.z);
       }

      } /* end for j */
    } /* end for i */


    if (!raw_out) fprintf(fp,"} /* end union */\n");
    fclose(fp);
    return(0);
}

 /* project set of four points on a cylinder */

void project(int i,int j,pnt *p1, pnt*p2,pnt*p3,pnt*p4)
{
double theta,r;
double h1,h2,h3,h4;
int x1,y1,x2,y2,x3,y3,x4,y4;    /* coordinates in heightfield space */

                                    /* input: x1,y1,h1  out: x,y,z */

       x1 = (int)(i * xfac);  /* point 1 */
       y1 = (int)(j * yfac);
       x2 = (int)(((i+1)%naround) * xfac);  /* point 2 to the right */
       y2 = y1;
       x3 = x1;                   /* point 3 down */
       y3 = (int)(((j+1)%(nhigh-1)) * yfac);
       x4 = x2;                   /* point 4 down+ to the right */
       y4 = y3;
       h1 = El(hf,x1,y1);
       h2 = El(hf,x2,y2);
       h3 = El(hf,x3,y3);
       h4 = El(hf,x4,y4);

       r = rad_min + rfac * h1;
       theta = pi2*(double)i/naround;
       p1[0].x = r * cos(theta);
       p1[0].y = r * sin(theta);
       p1[0].z = height*(double) j / nhigh;

       r = rad_min + rfac * h2;
       theta = pi2*(double)(i+1)/naround;
       p2[0].x = r * cos(theta);
       p2[0].y = r * sin(theta);
       p2[0].z = height*(double) j / nhigh;

       r = rad_min + rfac * h3;
       theta = pi2*(double)i/naround;
       p3[0].x = r * cos(theta);
       p3[0].y = r * sin(theta);
       p3[0].z = height*(double)(j+1)/nhigh;

       r = rad_min + rfac * h4;
       theta = pi2*(double)(i+1)/naround;
       p4[0].x = r * cos(theta);
       p4[0].y = r * sin(theta);
       p4[0].z = height*(double)(j+1) / nhigh;

}

