/***************************************************************************
**	ldbf.h	- LDBF client library
**      Copyright (c) 1995 Nadezhda Company, 
**         Software group, Vlad Seriakov
****************************************************************************/
#ifndef _LDBF_H_
#define _LDBF_H_
#include "ldbfwidget.h"

/* !!! do not change these defines !!! */
#ifndef TAGLEN
#define TAGLEN   15                 /* max size of tag name */
#endif
#ifndef EXPRLEN
#define EXPRLEN  245                /* max size of expression */
#endif

typedef struct {
   char name[TAGLEN+1]  ;        /*name of field */
   unsigned short len  ;         /* length of field */
   char dec  ;                   /* decimal digits after floating point */
   char type  ;                  /* type of field */
   long offset  ;                /* offset in the record */
   struct ldbfDB_s *data  ;          /* pointer on table owner of this field */
} ldbfFIELD;

typedef struct {                        /* info about field  */
  char name[TAGLEN+1]  ;          /* name of field */
  char type  ;                    /* type of field */
  char len  ;                     /* length of field */
  char dec  ;                     /* number of decimal numbers after dot */
} ldbfFIELDINFO;

typedef struct {
  char name[TAGLEN+1]  ;           /* name of tag */
  char expression[EXPRLEN+1]  ;    /* expression of index */
  char filter[EXPRLEN+1]  ;        /* filter of index */
  char unique  ;                   /* whether index is unique */
  char descending  ;               /* order */
} ldbfTAGINFO;

typedef struct ldbfDB_s {
   link_list link;
   char alias[TAGLEN+1];      /* alias of opened table */
   int sock;                  /* socket of server, returned by dbfConnect */ 
   int handle;                /* table handle, returned by dbfOpen */
   char *record;              /* current record */
   int changed;               /* record changed flag */
   unsigned long recno;       /* number of current record */
   long reccount;             /* total number of records in the table */
   long record_width;         /* record widht */
   int set_deleted;           /* whether to process deleted records */
   char tag[TAGLEN+1];        /* current tag */
   int n_fields;              /* number of fields */
   int n_tags;                /* number of tags */
   ldbfFIELD *fields;             /* fields info */
   ldbfTAGINFO *tags;             /* tags info */
} ldbfDB;                   

typedef struct {
  link_list link;
  char alias[TAGLEN+1];       /* table alias */
  unsigned long recno;        /* number of record */
  long len;                   /* length of data */
  unsigned char *data;        /* record contents or any data */ 
} ldbfRESULT;

#define O_EX   2    /* exclusive mode */
#define O_RD   4    /* read only mode: */
#define O_TTS  64   /* transaction mode: all updates are transactions */

#define ON  1
#define OFF 0

/* error number */
extern int ldbf_errno;

/* System settings ( may be set by ldbfSetOnOff ) */
#define SET_DELETED    2         /* do not show deleted records */

#ifdef __cplusplus
  extern "C" {
#endif

/*
** Pre-declarations for the API library functions
*/

/* return or specify user name, by default UNIX login name used */
/* required for Windows and MSDOS clients */
char *LD_FUNCTION ldbfUserName(char *newname);

/* return or specify user password, by default no password used.
   if crypt_pwd no-zero, new password is already crypted otherwise 
   password is pure ascii string,if crypt_pwd == -1, password
   validation will be set to default. */
/* required for Windows and MSDOS clients */
char *LD_FUNCTION ldbfPassword(char *newpasswd,int crypt_pwd);

/* ldbf error code */
int LD_FUNCTION ldbfErrno();

/* return or specify TCP port server listen on */
int LD_FUNCTION ldbfPort(int newport);

/* Pointer to user defined function that will be handle with
   server's events.This routine will be called by ldbfCheckEvent.
   Format should be:
        void user_func(char *event_name,char *data);
*/
void *LD_FUNCTION ldbfEventHandler(void (*newevent)());

/**************************************************************************
**	Purpose	: Gets text associated with error code
**	Args	: error code 
**	Returns	: text
*/
char LD_PTR * LD_FUNCTION ldbfErr_Text(int err_code);

/**************************************************************************
**	Purpose	: Check out for server event
**	Args	: Server sock 
**	Returns	: none
**      Note    : You should call this function as frequently as 
**                possible in order to handle events that may be sent by 
**                server 
**              
*/
void LD_FUNCTION ldbfCheckEvent(int sock);

/**************************************************************************
**	Purpose	: Tell the server that connection is closed
**	Args	: Server sock 
**	Returns	: none
*/
void LD_FUNCTION ldbfShutdown(int sock);

/**************************************************************************
**      Purpose : Shuts down network functions (for MSDOS only)
**      Args    : none
**	Returns	: none
**      Note    : This function MUST be called in MSDOS programs before 
**                exit program, otherwise it may be cause of locking of computer
*/
void LD_FUNCTION ldbfNetShut();

/**************************************************************************
**	Purpose	: Make connection to a LDBF server
**	Args	: hostname of server
**	Returns	: socket for further use.  -1 on error
**	Notes	: If host == NULL, localhost is used via UNIX domain socket
*/
int LD_FUNCTION ldbfConnect(char *host);

/**************************************************************************
**	Purpose	: Creates new alias definition in order to other users
**                 may open this table
**	Args	: Server sock and alias name, table path
**	Returns	: -1 if error
**      Note    : This useful when user create new table and wants
**                  other open it too.New alias is put in ldbf.conf.
**		    You have to have create privileges(create_allow).
*/
int LD_FUNCTION ldbfCreateAliasDef(int sock,char *alias,char *pathname);

/**************************************************************************
**	Purpose	: Exchange table file for specified alias.
**	Args	: Server sock,alias name, table path
**      Note    : You must have rights to execute this command.
**                See ldbf.conf for exchange_allow parameter.
**                Server closes existing table file and reopen
**                specified file under existing alias.
**                All users having this file opened
**                continues work with another table file.
**                If anyone does transaction on this table
**                it'll be aborted.
**                Use this command with CAUTION.
**                Alias definition in ldbf.conf file will be 
**                replace with new value.
*/
int LD_FUNCTION ldbfExchange(int sock,char *alias,char *path);

/**************************************************************************
**	Purpose	: Tell the server which table we want to use
**	Args	: Server sock and ldbfDB name,
**                  mode may be O_EX   to open table in exclusive 
**                              O_RD   to open in read only mode
**                              O_TTS  to open in transaction mode
**                               in this mode every updates will be flushed
**	Returns	: NULL on error
*/
ldbfDB LD_PTR * LD_FUNCTION ldbfOpen(int sock,char *db,int mode);

/**************************************************************************
**	Purpose	: Tell the server which table we want to remove
**	Args	: Server sock and ldbfDB name
**      Note    : You must have rights to delete tables
**                in order to use this function
**      Returns : 0 on success, -1 on error
*/
int LD_FUNCTION ldbfDrop(int sock,char *db);

/**************************************************************************
**	Purpose	: get list of all available tables
**	Args	: socket
**	Returns	: pointer on string that contain list of all aliases
**                separated by comma (,).
*/
char LD_PTR * LD_FUNCTION ldbfAliasList(int sock);

/**************************************************************************
**	Purpose	: get tag info of table
**	Args	: ldbfDB 
**	Returns	: array of data->n_tags ldbfTAGINFO structures or NULL on error
*/
ldbfTAGINFO LD_PTR * LD_FUNCTION ldbfTaginfo(ldbfDB* data);

/**************************************************************************
**	Purpose	: get field info of table
**	Args	: ldbfDB 
**	Returns	: array of data->n_fields ldbfFIELD structures or NULL on error
*/
ldbfFIELD LD_PTR * LD_FUNCTION ldbfFieldinfo(ldbfDB* data);

/**************************************************************************
**	Purpose	: close the table
**	Args	: ldbfDB 
**	Returns	: none
*/
void  LD_FUNCTION ldbfClose(ldbfDB* data);

/**************************************************************************
**	Purpose	: close all opened tables
**	Args	: connection number
**	Returns	: none
*/
void  LD_FUNCTION ldbfCloseall(int sock);

/**************************************************************************
**	Purpose	: starts transaction
**	Args	: connection number
**	Returns	: 1 transaction activated or already active
**               -1 transaction activation failed or disabled
*/
int  LD_FUNCTION ldbfBegin(int sock);

/**************************************************************************
**	Purpose	: aborts current transaction
**	Args	: connection number
**	Returns	: none
*/
void  LD_FUNCTION ldbfAbort(int sock);

/**************************************************************************
**	Purpose	: commit current transaction
**	Args	: connection number
**	Returns	: -1 on error
*/
int  LD_FUNCTION ldbfCommit(int sock);

/**************************************************************************
**	Purpose	: Sets various system settings
**	Args	: connection number, flag and value
**	Returns	: none
*/
void  LD_FUNCTION ldbfSetOnOff(int sock,int flag,int onoff);

/**************************************************************************
**	Purpose	: Sets filter on table
**	Args	: ldbfDB table and filter expression
**	Returns	: none
**      Note    : filter may be any valid xBase expression.
**                filter'll be used on ldbfSeek,ldbfTop,ldbfBottom,ldbfSkip
**                operations.Server will skip records that makes this expression
**                false.
*/
void  LD_FUNCTION ldbfSetFilter(ldbfDB *data,char *filter);

/**************************************************************************
**	Purpose	: Call procedure stored in the server
**	Args	: connection number ,procedure name, parameter string,
**                name of function that will handle with results
**                function format:
**                 void func(long result_count);
**                 where 
**                       result_count - number of result records returned.
**
**                To get result contents you may use function ldbfFetchResult
**	Returns	: -1 on error
*/
int  LD_FUNCTION ldbfCall(int sock,char *proc,char *param,void (*func)());

/**************************************************************************
**	Purpose	: Returns next result record from result set returned by
**                function ldbfCall
**      Args    : none
**	Returns	: Pointer on RESULT structure or NULL if no more records
**                 'db' contains alias of table or empty if result just data
**                 'recno' contains record number in case of table record
**                 'len' always contains length of data
**                 'data' pointers on record contents or any data
**
**     Note     : Address of returned structure is valid till next call of ldbfCall.
*/
ldbfRESULT LD_PTR * LD_FUNCTION ldbfFetchResult();

/**************************************************************************
**	Purpose	: Clears result set returned by ldbfCall
**	Args	: none
**	Returns	: none
**      Note    : if you've copied contents of data to your variables,
**                you may free memory occupied by result set.
*/
void  LD_FUNCTION ldbfClearResult();

/**************************************************************************
**	Purpose	: Register itself for recieve this event
**	Args	: connection number and event name
**	Returns	: none
*/
void  LD_FUNCTION ldbfRegisterEvent(int sock,char *event_name);

/**************************************************************************
**	Purpose	: Raise event
**	Args	: connection number and event name
**	Returns	: none
*/
void  LD_FUNCTION ldbfRaiseEvent(int sock,char *event_name,char *data);

/**************************************************************************
**	Purpose	: Tell the server to create table
**	Args	: Server sock,ldbfDB name,field info and tag info
**	Returns	: NULL on error
**      Note    : You must have create privilages (create_allow) in ldbf.conf.
*/
ldbfDB LD_PTR * LD_FUNCTION ldbfCreate(int sock,char *db,ldbfFIELDINFO *finfo,ldbfTAGINFO *tinfo);

/**************************************************************************
**	Purpose	: Seeks the data in the table
**	Args	: ldbfDB name and sought string
**	Returns	: -1 on error
**      Note    : Uses current tag that has been set by ldbfSetTag function
*/
int LD_FUNCTION ldbfSeek(ldbfDB* data,char *str);

/**************************************************************************
**	Purpose	: Locates first record in the table
**	Args	: ldbfDB name and sought string
**		  start - starting tag value
**		  end - scan while this xBase expression true
**		  cond - condition in xBase expression of required record
**      Example: 
**                  set pointer on record with age 5.
**                  scan table in order of tag 'age'
**                  while 'age' less than 30 and
**                  check for 'John' in field name.
**                  if 'John' in name exists returns that record 
**               ldbfSetTag(db,"age");
**               ldbfLocateFirst(db,"5","age < 30","'John'$name");
**	Returns	: -1 on error
*/

int LD_FUNCTION ldbfLocateFirst(ldbfDB* data,char *start,char *end,char *cond);

/**************************************************************************
**	Purpose	: Locates record in the table from current position
**	Args	: ldbfDB name and sought string
**		  start - starting tag value
**		  end - scan while this xBase expression true
**		  cond - condition in xBase expression of required record
**	Returns	: -1 on error
**      Example: 
**                  see previous example. Locate all records with
**                  'John' in name.Do skip on next record.
**                  and scan for next mactching record 
**               ldbfSkip(db,1);
**               while(!ldbf_errno) {
**                 ldbfLocate(db,"age < 30","'John'$name");
**                 .....
**               } 
*/
int LD_FUNCTION ldbfLocate(ldbfDB* data,char *end,char *cond);

/**************************************************************************
**	Purpose	: Sets current tag
**	Args	: ldbfDB name and tag name
**	Returns	: none
**      Note    : ldbfSeek uses this tag for searching
*/
void LD_FUNCTION ldbfSetTag(ldbfDB* data,char *tagname);

/**************************************************************************
**	Purpose	: Returns pointer on opened table by its name
**	Args	: alias
**	Returns	: pointer on ldbfDB
*/
ldbfDB *LD_FUNCTION ldbfTableByName(char *name);

/**************************************************************************
**	Purpose	: Returns pointer on opened table by its number
**	Args	: table number
**	Returns	: pointer on ldbfDB
**      Note    : This function needs for in result handler
**                 invoked by ldbfCall.See ldbfCall and ldbfFetchResult.
*/
ldbfDB *LD_FUNCTION ldbfTableByNo(int no);

/**************************************************************************
**	Purpose	: Set table pointer to specified record
**	Args	: ldbfDB name and record number
**	Returns	: -1 on error
*/
int LD_FUNCTION ldbfGo(ldbfDB* data,long rec);

/**************************************************************************
**	Purpose	: Set table pointer to top of table
**	Args	: ldbfDB name 
**	Returns	: -1 on error
**      Note    : if current tag exists, pointer will be set using order 
**                of that tag.
*/
int LD_FUNCTION ldbfTop(ldbfDB* data);

/**************************************************************************
**	Purpose	: Set table pointer to bottom of table
**	Args	: ldbfDB name 
**	Returns	: -1 on error
**      Note    : if current tag exists, pointer will be set using order 
**                of that tag.
*/
int LD_FUNCTION ldbfBottom(ldbfDB* data);

/**************************************************************************
**	Purpose	: Delete current record
**	Args	: ldbfDB name 
**	Returns	: -1 on error
*/
int LD_FUNCTION ldbfDelete(ldbfDB* data);

/**************************************************************************
**	Purpose	: Whether record deleted or not
**	Args	: ldbfDB name 
**	Returns	: 1 if record deleted
*/
int LD_FUNCTION ldbfDeleted(ldbfDB* data);

/**************************************************************************
**	Purpose	: Returns actual number of records in the table
**	Args	: ldbfDB name 
**	Returns	: -1 on error or number of records
**      Note    : You may obtain number of records from reccount field
**                of ldbfDB structure.But this value may be not true if anyone
**                already append some records to table.
*/
long LD_FUNCTION ldbfReccount(ldbfDB* data);

/**************************************************************************
**	Purpose	: Returns number of current record of the given table
**	Args	: ldbfDB name 
**	Returns	: -1 on error or number of current record
*/
long LD_FUNCTION ldbfRecno(ldbfDB* data);

/**************************************************************************
**	Purpose	: Recall(undelete) specified record
**	Args	: ldbfDB name and record number
**	Returns	: -1 on error
*/
int LD_FUNCTION ldbfRecall(ldbfDB* data,long rec);

/**************************************************************************
**	Purpose	: Skips specified number of records
**	Args	: ldbfDB and record number
**	Returns	: -1 on error
**      Note    : if current tag exists, skip will do using order 
**                of that tag.
*/
int LD_FUNCTION ldbfSkip(ldbfDB* data,long rec);

/**************************************************************************
**	Purpose	: Appends one record into table
**	Args	: ldbfDB 
**	Returns	: -1 on error
*/
int LD_FUNCTION ldbfAppend(ldbfDB* data);

/**************************************************************************
**	Purpose	: Updates current record in the table
**	Args	: ldbfDB 
**	Returns	: -1 on error
*/
int LD_FUNCTION ldbfUpdate(ldbfDB* data);

/**************************************************************************
**	Purpose	: Locks specified record
**	Args	: ldbfDB and record number
**	Returns	: -1 on error
*/
int LD_FUNCTION ldbfLock(ldbfDB* data,long rec);

/**************************************************************************
**	Purpose	: Unlocks specified record
**	Args	: ldbfDB and record number
**	Returns	: none
*/
void LD_FUNCTION ldbfUnlock(ldbfDB* data,long rec);

/**************************************************************************
**	Purpose	: Locks table
**	Args	: ldbfDB 
**	Returns	: -1 on error
*/
int LD_FUNCTION ldbfFlock(ldbfDB* data);

/**************************************************************************
**	Purpose	: Unlocks table
**	Args	: ldbfDB 
**	Returns	: none
*/
void LD_FUNCTION ldbfFunlock(ldbfDB* data);

/**************************************************************************
**	Purpose	: Flush table on disk
**	Returns	: none
*/
void LD_FUNCTION ldbfFlush(ldbfDB* data);

/**************************************************************************
**	Purpose	: Returns current position in the table
**	Args	: ldbfDB name 
**	Returns	: -1 on error
*/
int LD_FUNCTION ldbfPosition(ldbfDB* data);

/**************************************************************************
**	Purpose	: Set current position in the table
**	Args	: ldbfDB name and position
**	Returns	: -1 on error
*/
int LD_FUNCTION ldbfPositionSet(ldbfDB* data,int position);

/**************************************************************************
**	Purpose	: Empties the table
**	Returns	: -1 on error
**      Note    : table must be opened exclusive
*/
int LD_FUNCTION ldbfZap(ldbfDB* data);

/**************************************************************************
**	Purpose	: Insert record into table
**	Returns	: pointer on opened ldbfDB stru
**      Note    : If table not opened it'll open it
**                fields and values must be separated by ,
**       for example:
**         ldbfInsert(sock,"data","name,address,zip","John,New York,12345");
*/
ldbfDB LD_PTR * LD_FUNCTION ldbfInsert(long sock,char *data,char *field_list,char *values_list);

/**************************************************************************
**	Purpose	: Updates current record 
**	Returns	: pointer on opened ldbfDB stru
**      Note    : If table not opened it'll open it
**                fields and values must be separated by ,
**       for example:
**         ldbfModify(sock,"data","name,address,zip","John,New York,12345");
*/
ldbfDB LD_PTR * LD_FUNCTION ldbfModify(long sock,char *data,char *field_list,char *values_list);

/**************************************************************************
**	Purpose : Creates new table 
**      Returns : pointer on created ldbfDB stru
**      Format  :
**      field list = field_name(type[:len[:dec]]);field_name(type[:len[:dec]])
**      tag list = tag_name(expression[:filter[:unique[:descending]]]);
**
**        ldbfCreateldbfDBF(sock,"data","name(C:25);address(C:50);zip(N:10)", 
**                 "name(name);address(address::1);zip(zip:zip>0:1:1)");
*/
ldbfDB LD_PTR * LD_FUNCTION ldbfCreateldbfDBF(long sock,char *data,char *field_list,char *tag_list);

/* This is high level routines for field manipulation */

/**************************************************************************
**	Purpose	: Get field contents as string
**	Args	: ldbfDB and field name
**	Returns	: pointer on string
*/
char LD_PTR * LD_FUNCTION ldbfValue( char *alias,char *field_name );

/**************************************************************************
**	Purpose	: Get field contents as long
**	Args	: ldbfDB and field name
**	Returns	: long value of field
*/
long LD_FUNCTION ldbfLongValue( char *alias,char *field_name );

/**************************************************************************
**	Purpose	: Get field contents as double
**	Args	: ldbfDB and field name
**	Returns	: double value of field
*/
double LD_FUNCTION ldbfDoubleValue( char *alias,char *field_name );

/**************************************************************************
**	Purpose	: Get contents of memo field
**	Args	: ldbfDB and field name
**	Returns	: pointer ob memo contents and if memo_len not null
**                length of memo contents.
*/
char *LD_FUNCTION ldbfMemoValue(ldbfDB *data,char *field,int *memo_len);

/**************************************************************************
**	Purpose	: Writes contents of memo field
**	Args	: ldbfDB and field name,buffer and buffer length
**	Returns	: -1 on error
*/
int LD_FUNCTION ldbfReplaceMemo(ldbfDB *data,char *field,void *value,int len);

/**************************************************************************
**	Purpose	: Replace field contents with string value
**	Args	: ldbfDB name , field name and contents
**	Returns	: none
*/
void LD_FUNCTION ldbfReplace(char *alias,char *field_name,char *value);

/**************************************************************************
**	Purpose	: Replace field contents with long value
**	Args	: ldbfDB name , field name and contents
**	Returns	: none
*/
void LD_FUNCTION ldbfReplaceLong(char *alias,char *field_name,long value);

/**************************************************************************
**	Purpose	: Replace field contents with double value
**	Args	: ldbfDB name , field name and contents
**	Returns	: none
*/
void LD_FUNCTION ldbfReplaceDouble(char *alias,char *field_name,double value);

/* This is low level routines for field manipulation */

/**************************************************************************
**	Purpose	: Get field via name
**	Args	: ldbfDB and field name
**	Returns	: NULL on error
*/
ldbfFIELD LD_PTR * LD_FUNCTION ldbfField( ldbfDB *data, char *field_name );

/**************************************************************************
**	Purpose	: Get field via number
**	Args	: ldbfDB and field number
**	Returns	: NULL on error
*/
ldbfFIELD LD_PTR * LD_FUNCTION ldbfField_j( ldbfDB *data, int j_field );

/**************************************************************************
**	Purpose	: Get field number by its name
**	Args	: ldbfDB and field name
**	Returns	: field number,0 on error
*/
int LD_FUNCTION ldbfField_number( ldbfDB *data, char *field_name );

/**************************************************************************
**	Returns	: pointer on field in record
*/
char LD_PTR * LD_FUNCTION ldbfPtr( ldbfFIELD *field );

/**************************************************************************
**	Purpose	: Assign the field value
**	Args	: ldbfFIELD and double value
**	Returns	: none
*/
void LD_FUNCTION ldbfAssign_double( ldbfFIELD *field, double d_value );

/**************************************************************************
**	Purpose	: Get double value from field
**	Args	: ldbfFIELD
**	Returns	: value of field
*/
double LD_FUNCTION ldbfDouble( ldbfFIELD *field );

/**************************************************************************
**	Purpose	: Empty all field of current record
**	Args	: ldbfDB 
**	Returns	: none
*/
void LD_FUNCTION ldbfBlank( ldbfDB *data );

/**************************************************************************
**	Purpose	: Assign char value to field
**	Args	: ldbfFIELD and value
**	Returns	: none
*/
void LD_FUNCTION ldbfAssign_char( ldbfFIELD *field, int chr );

/**************************************************************************
**	Purpose	: Get char vlaue from field
**	Args	: ldbfFIELD
**	Returns	: char value
*/
int LD_FUNCTION ldbfChar( ldbfFIELD *field );

/**************************************************************************
**	Purpose	: Assign string value to field
**	Args	: ldbfFIELD and string value
**	Returns	: none
*/
void LD_FUNCTION ldbfAssign( ldbfFIELD *field, char *str );

/**************************************************************************
**	Purpose	: Get string value form field
**	Args	: ldbfFIELD
**	Returns	: pointer to static area with string
*/
char LD_PTR * LD_FUNCTION ldbfStr( ldbfFIELD *field );

/**************************************************************************
**	Purpose	: Assign long value to field
**	Args	: ldbfFIELD and long value
**	Returns	: none
*/
void LD_FUNCTION ldbfAssign_long( ldbfFIELD *field, long l_value );

/**************************************************************************
**	Purpose	: Get long value form field
**	Args	: ldbfFIELD
**	Returns	: field value in long
*/
long LD_FUNCTION ldbfLong( ldbfFIELD *field );

/**************************************************************************
**	Purpose	: Assign field value from another field
**	Args	: ldbfFIELD and ldbfFIELD
**	Returns	: none
*/
void LD_FUNCTION ldbfAssign_field( ldbfFIELD *field_to, ldbfFIELD *field_from );

/**************************************************************************
**	Purpose	: Get information about field 
**	Args	: ldbfFIELD
*/
int LD_FUNCTION ldbfDecimals( ldbfFIELD *field );
unsigned LD_FUNCTION ldbfLen( ldbfFIELD *field );
char LD_PTR * LD_FUNCTION ldbfName( ldbfFIELD *field );
int LD_FUNCTION ldbfType( ldbfFIELD *field );

/**************************************************************************
  This is misc function that may easy your programming
*/

/*  Converts a string to a double */
double LD_FUNCTION ldbfatod( char *str, int len_str );

/* Converts a string to a integer */
int LD_FUNCTION ldbfatoi( char *str, int len_str );

/* Converts a string to a long */
long LD_FUNCTION ldbfatol( char *str, int len_str );

/* - formats a double to a string */
void LD_FUNCTION ldbfdtoa( double doub_val, char *out_buffer, int len, int dec);

/*  Converts a long to a string.  Fill with '0's rather than blanks if
    'num' is less than zero.*/
void  LD_FUNCTION ldbfltoa( long l_val, char *ptr, int num);

/*  - From CCYYMMDD to CCYY.MM.DD
    c4encode( to, from, "CCYY.MM.DD", "CCYYMMDD" ) ;*/
void  LD_FUNCTION ldbfencode( char *to, char *from, char *t_to, char *t_from);

/* Make all chars in the string in lower case */
void LD_FUNCTION ldbflower( char *str );

/* Make all chars in the string in upper case */
void  LD_FUNCTION ldbfupper( char *str );

/* store long date into string format */
void LD_FUNCTION ldbfdate_assign( char *date_ptr, long ldate );

/* returns character day of week */
char LD_PTR * LD_FUNCTION ldbfcdow( char *date_ptr );

/* returns character month */
char LD_PTR * LD_FUNCTION ldbfcmonth( char *date_ptr );

/* returns day from date string : date format CCYYMMDD */
int LD_FUNCTION ldbfday( char *date_ptr );

/* returns day of week in number */
int LD_FUNCTION ldbfdow( char *date_ptr );

/* set default format of date */
char LD_PTR* LD_FUNCTION ldbfSetDateFormat( char *date_format );

/* formats date using picture, assumes date_ptr in dbf date format
   CCYYMMDD */
void LD_FUNCTION ldbfdate_format( char *date_ptr, char *result, char *picture );

/* converts date_data using picture in dbf format. 
   date_ptr format CCYYMMDD */
void LD_FUNCTION ldbfdate_init( char *date_ptr, char *date_data, char *picture );

/* returns date in long, assumes date_ptr in dbf format */
long LD_FUNCTION ldbfdate( char *date_ptr );

/* return month from date string */
int LD_FUNCTION ldbfmonth( char *date_ptr );

/* return year from date string */
int LD_FUNCTION ldbfyear( char *year_ptr );

/* store curent time */
void LD_FUNCTION ldbftime_now( char *time_data );

/* store current date */
void LD_FUNCTION ldbftoday( char *date_ptr );

#ifdef __cplusplus
   }
#endif

#endif

