/************************************************************************/
/* Module : callback.c					                */
/* Purpose: widget callback module for Mpsql     	                */
/* By     : Keith R. Davis					        */
/* Date   : 12/8/95					                */
/* Notes  : Copyright(c) 1996-98 Mutiny Bay Software			*/
/************************************************************************/

#include <Xm/Xm.h>		/* motif lib header		*/
#include <Xm/PrimitiveP.h>      /* motif primitive widget head. */
#include <Xm/PushB.h>		/* push button widget header	*/
#include <Xm/RowColumn.h>	/* row/col widget header	*/
#include <Xm/Label.h>		/* label widget header		*/
#include <Xm/List.h>            /* list widget header           */
#include <Xm/TextF.h>		/* text widget header		*/
#include <Xm/MessageB.h>	/* message box header	        */
#include <Xm/FileSB.h>          /* file selection widget header */
#include <string.h>		/* string lib header		*/
#include <stdio.h>              /* stdio header                 */
#include <malloc.h>             /* malloc header                */
#include <help.h>               /* help lib header              */

#include "textBuf.h"            /* text widget headers          */
#include "textSel.h"
#include "text.h"
#include "textDisp.h"
#include "textP.h"
#include "window.h"
#include "highlight.h"
#include "highlightData.h"

#include "msgbox.h"             /* message box header           */
#include "pixmap.h"             /* pixmap header                */
#include "mquel.h"              /* mquel header                 */
#include "menu.h"		/* menu header		        */
#include "llist.h"              /* linked list header           */
#include "callback.h"           /* callback header              */
#include "db.h"                 /* db header                    */
#include "util.h"               /* utility func. header         */
#include "imageutil.h"          /* image utils header           */
#include "xutil.h"              /* X utils                      */

/************************************************************************/
/* Function: DeleteWindowCallback                                       */
/* Purpose : handles processing for WM_DELETE_WINDOW message            */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void DeleteWindowCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  /* close database & remove linked list */ 
  DB_Close();
  LLIST_Remove();

  /* init dummy node */
  BufFree(empty_buffer.buffer);
  BufRemoveModifyCB(empty_buffer.buffer, TextModifiedCallback, NULL);
  BufRemoveModifyCB(empty_buffer.buffer, SyntaxHighlightModifyCB, WinDataPtr);

  FreeHighlightingData(WinDataPtr);
 
  free(passwd);
 
  exit(0);
}

/************************************************************************/
/* Function: ConnectCallback                                            */
/* Purpose : handles database connection request                        */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void ConnectCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  const char *title = "Connect";                   /* dialog title string            */
  Widget rc;                                       /* rowcol widget for dialog       */
  static Widget connect_dialog = NULL;             /* dialog widget                  */

  if(!connect_dialog){
    connect_dialog = XmCreateMessageDialog(AppWidgetsPtr->mainwindow, "connectdialog", NULL, 0);
  
    /* remove uneeded children */
    /*XtUnmanageChild(XmMessageBoxGetChild(connect_dialog, XmDIALOG_HELP_BUTTON));*/
    XtUnmanageChild(XmMessageBoxGetChild(connect_dialog, XmDIALOG_SYMBOL_LABEL));
    XtUnmanageChild(XmMessageBoxGetChild(connect_dialog, XmDIALOG_MESSAGE_LABEL));

    /* setup dialog */
    XtVaSetValues(connect_dialog,
		  XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL,
		  XtVaTypedArg, XmNdialogTitle, XmRString,
		  title, strlen(title)+1,
		  NULL);

    rc = XtVaCreateManagedWidget("rc", xmRowColumnWidgetClass, connect_dialog,
				 XmNnumColumns, 2,
				 XmNpacking, XmPACK_COLUMN,
				 XmNorientation, XmVERTICAL,
				 NULL);
    if(user_prompt){
      XtCreateManagedWidget("User name:", xmLabelWidgetClass, rc, NULL, 0);
      XtCreateManagedWidget("Password:", xmLabelWidgetClass, rc, NULL, 0);
    }
    XtCreateManagedWidget("Database name:", xmLabelWidgetClass, rc, NULL, 0);
    XtCreateManagedWidget("Host name:", xmLabelWidgetClass, rc, NULL, 0);
    XtCreateManagedWidget("Port:", xmLabelWidgetClass, rc, NULL, 0);
    
    if(user_prompt){
      AppWidgetsPtr->uname = XtVaCreateManagedWidget("uname", xmTextFieldWidgetClass,
						   rc,
						   XmNmaxLength, UNAME_NAME_SZ,
						   NULL);
      AppWidgetsPtr->pwd = XtVaCreateManagedWidget("pwd", xmTextFieldWidgetClass,
						   rc, 
						   XmNmaxLength, PWD_NAME_SZ,
						   NULL);
    }

    AppWidgetsPtr->database = XtVaCreateManagedWidget("database", xmTextFieldWidgetClass,
						      rc,
						      XmNmaxLength, DB_NAME_SZ,
						      NULL);
    AppWidgetsPtr->host = XtVaCreateManagedWidget("host", xmTextFieldWidgetClass,
						  rc, 
						  XmNmaxLength, HOST_NAME_SZ,
						  NULL);
    AppWidgetsPtr->port = XtVaCreateManagedWidget("port", xmTextFieldWidgetClass,
						rc, 
						XmNmaxLength, PORT_NAME_SZ,
						NULL);

    /* add callbacks */
    if(user_prompt) {
      XtAddCallback(AppWidgetsPtr->pwd, XmNmodifyVerifyCallback, ConnectCheckPassword, NULL);
      XtAddCallback(AppWidgetsPtr->pwd, XmNactivateCallback, ConnectCheckPassword, NULL);
    }

    XtAddCallback(connect_dialog, XmNokCallback, ConnectOKCallback, NULL);
    XtAddCallback(connect_dialog, XmNhelpCallback, get_help, "mpsql_menu_file.html");
  }

  XUTIL_SetFocus(connect_dialog, AppWidgetsPtr->database);

  if(dbname != NULL)
    XmTextSetString(AppWidgetsPtr->database, dbname);
  if(host != NULL)
    XmTextSetString(AppWidgetsPtr->host, host);
  if(port != NULL)
    XmTextSetString(AppWidgetsPtr->port, port);

  XtManageChild(connect_dialog);
}

/************************************************************************/
/* Function: ConnectCheckPassword                                       */
/* Purpose : masks the user password input                              */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   : uses the global 'passwd' in db.c to store the actual pwd.  */
/************************************************************************/

void ConnectCheckPassword(Widget text_w, XtPointer client_data, XtPointer call_data)
{
  char *new;
  int len;
  
  XmTextVerifyCallbackStruct *cbs = (XmTextVerifyCallbackStruct *) call_data;

  if (cbs->reason == XmCR_ACTIVATE) {
    //printf ("Password: %s", passwd);
    return;
  }

  if( cbs->text->length > 1 ) {
    cbs->doit = False; /* don't allow paste operations; make the */
    return;            /* user type the password! */
  }

  if( cbs->startPos < cbs->endPos ) {
    /* shrink the password by moving endPos... characters forward to */
    /* the point of deletion */
    memmove( &(passwd[cbs->startPos]),
	     &(passwd[cbs->endPos]),
	     strlen(passwd) - cbs->endPos + 1 );
    if( cbs->text->length == 0 ) /* then just a delete, not a replace */
      return;
  }
  
  new = XtMalloc( cbs->text->length + 1 );
  
  strncpy( new, cbs->text->ptr, cbs->text->length ); /* just the new chars */
  new[cbs->text->length]='\0';
  
  if( passwd ) {
    /* open a hole for the new characters, and insert them in the proper place */
    passwd = XtRealloc( passwd, strlen(passwd) + cbs->text->length + 1 );
    memmove( &(passwd[cbs->startPos + cbs->text->length]),
	     &(passwd[cbs->startPos]),
	     strlen(passwd) - cbs->startPos + 1 );
    memcpy( &(passwd[cbs->startPos]), new, cbs->text->length );
  } else {
    passwd = new;
  }
  memset( cbs->text->ptr, '*', cbs->text->length ); /* makes it all stars */  
}

/************************************************************************/
/* Function: ConnectOKCallback                                          */
/* Purpose : calls DB_Connect()                                         */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void ConnectOKCallback(Widget w, XtPointer clientData, XtPointer callData)
{
  if(user_prompt) {
    strcpy(username, doubleTrim((char*)XmTextGetString(AppWidgetsPtr->uname)));
  }
  strcpy(dbname, doubleTrim((char*)XmTextGetString(AppWidgetsPtr->database)));
  strcpy(host, doubleTrim((char*)XmTextGetString(AppWidgetsPtr->host)));
  strcpy(port, doubleTrim((char*)XmTextGetString(AppWidgetsPtr->port)));

  if((strlen(dbname) == 0) || (strlen(host) == 0) || (strlen(port) == 0))
    MSGBOX_Error("Connect", "Make sure all fields are filled!", AppWidgetsPtr->mainwindow);
  else
    DB_Connect();
}

/************************************************************************/
/* Function: NewCallback                                                */
/* Purpose : handles new buffer request                                 */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void NewCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  LLISTbuffer *buffer;

  if((buffer = DB_OpenScratchBuffer()) != NULL)
    buffer_curr = DB_SetBuffer(buffer);
}

/************************************************************************/
/* Function: OpenCallback                                               */
/* Purpose : handles open file request                                  */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void OpenCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  const char *title = "Open";           /* dialog title string          */     
  Widget text = (Widget)clientdata;     /* selected file info           */
  static Widget open_dialog = NULL;     /* dialog widget                */

  if(!open_dialog){
    open_dialog = XmCreateFileSelectionDialog(AppWidgetsPtr->mainwindow,
					      "OpenFileDialog",
					      NULL, 0);
    /* setup dialog */
    XtVaSetValues(open_dialog,
		  XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL,
		  XtVaTypedArg, XmNdialogTitle, XmRString,
		  title, strlen(title)+1,
		  NULL);

    /* add callbacks */
    XtAddCallback(open_dialog, XmNokCallback, OpenOKCallback, NULL);
    XtAddCallback(open_dialog, XmNcancelCallback, OpenCancelCallback, NULL);
    XtAddCallback(open_dialog, XmNhelpCallback, get_help, "mpsql_menu_file.html");
  }
  XtManageChild(open_dialog);
}

/************************************************************************/
/* Function: OpenOKCallback                                             */
/* Purpose : calls DB_OpenFile()                                        */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   : file errors handled by DB_OpenFile()                       */
/************************************************************************/

void OpenOKCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  char *filename;         /* file to open               */

  /* get the filename selected */
  XmFileSelectionBoxCallbackStruct *cbs =
    (XmFileSelectionBoxCallbackStruct*)callData;

  XmStringGetLtoR(cbs->value, XmFONTLIST_DEFAULT_TAG, &filename);

  /* close dialog */
  XtUnmanageChild(w);

  /* open the file */
  DB_OpenFile(filename);
}

/************************************************************************/
/* Function: OpenCancelCallback                                         */
/* Purpose : closes open file dialog on cancel                          */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void OpenCancelCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  XtUnmanageChild(w);
}

/************************************************************************/
/* Function: SaveCallback                                               */
/* Purpose : handles save file request                                  */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   : file errors handled by DB_SaveFile()                       */
/************************************************************************/

void SaveCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  if(strcmp(buffer_curr->filename, "scratch") == 0)
    SaveAsCallback(NULL, NULL, NULL);
  else
    DB_SaveFile(buffer_curr);
}

/************************************************************************/
/* Function: SaveAsCallback                                             */
/* Purpose : handles save as file request                               */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void SaveAsCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  const char *title = "Save As";          /* dialog title string        */
  Widget text = (Widget)clientdata;       /* selected file data         */
  static Widget save_dialog = NULL;       /* dialog widget              */

  if(!save_dialog){
    save_dialog = XmCreateFileSelectionDialog(AppWidgetsPtr->mainwindow,
					      "SaveFileDialog",
					      NULL, 0);

    /* setup dialog */
    XtVaSetValues(save_dialog,
		  XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL,
		  XtVaTypedArg, XmNdialogTitle, XmRString,
		  title, strlen(title)+1,
		  NULL);

    /* add callbacks */
    XtAddCallback(save_dialog, XmNokCallback, SaveAsOKCallback, NULL);
    XtAddCallback(save_dialog, XmNcancelCallback, SaveAsCancelCallback, NULL);
    XtAddCallback(save_dialog, XmNhelpCallback, get_help, "mpsql_menu_file.html");
  }
  XtManageChild(save_dialog);
}

/************************************************************************/
/* Function: SaveAsOKCallback                                           */
/* Purpose : calls DB_SaveFile()                                        */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   : file errors handled by DB_SaveFile()                       */
/************************************************************************/

void SaveAsOKCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  char *filename;            /* filename to save as                     */
  char tmp_fn[MAX_PATH_LEN]; /* tmp filename                            */

  /* get filename */
  XmFileSelectionBoxCallbackStruct *cbs =
    (XmFileSelectionBoxCallbackStruct*)callData;

  XmStringGetLtoR(cbs->value, XmFONTLIST_DEFAULT_TAG, &filename);

  XtUnmanageChild(w);

  /* save the file & copy new filename to linked list node holding the buffered file data */
  strcpy(tmp_fn, buffer_curr->filename);
  strcpy(buffer_curr->filename, filename);

  if(!DB_SaveFile(buffer_curr))
    strcpy(buffer_curr->filename, tmp_fn);
}

/************************************************************************/
/* Function: SaveAsCancelCallback                                       */
/* Purpose : closes save as file dialog on cancel                       */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void SaveAsCancelCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  XtUnmanageChild(w);
}

/************************************************************************/
/* Function: CloseCallback                                              */
/* Purpose : handles exit application request                           */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void CloseCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  /* dialog message strings */
  const char *title = "Quit";
  const char *quit_msg = "Unsaved modified buffers exist.\nContinue without saving?";
  
  /* dialog widget */
  static Widget close_dialog = NULL;

  /* check linked list for modified buffers & if so, prompt user ...*/
  if(DB_ChkModBuffers()){

    if(!close_dialog){
      close_dialog = XmCreateQuestionDialog(AppWidgetsPtr->mainwindow, "closedialog", NULL, 0);
    
      /* remove uneeded children */
      XtUnmanageChild(XmMessageBoxGetChild(close_dialog, XmDIALOG_HELP_BUTTON));
    
      /* setup dialog */
      XtVaSetValues(close_dialog,
		    XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL,
		    XtVaTypedArg, XmNdialogTitle, XmRString,
		    title, strlen(title)+1,
		    XtVaTypedArg, XmNmessageString, XmRString,
		    quit_msg, strlen(quit_msg)+1, NULL);
	
      /* add callbacks */
      XtAddCallback(close_dialog, XmNokCallback, ReallyQuitCallback, NULL);
    }
    XtManageChild(close_dialog);
  }
  else 
    DeleteWindowCallback(NULL, NULL, NULL);
}

/************************************************************************/
/* Function: ReallyQuitCallback                                         */
/* Purpose : handles exit application request                           */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   : if the app has modified buffers on close, this will force  */
/*           closing with out saving...(OK button)                      */
/************************************************************************/

void ReallyQuitCallback(Widget w, XtPointer clientData, XtPointer callData)
{
  DeleteWindowCallback(NULL, NULL, NULL);  
}

/************************************************************************/
/* Function: CutCallback                                                */
/* Purpose : handles cuting of selected text to clipboard               */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void CutCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
   textDisp *textD = ((TextWidget)AppWidgetsPtr->sqlwindow)->text.textD;
    
   if (!textD->buffer->primary.selected){
     XBell(XtDisplay(w), 0);
     return;
   }

   CopyToClipboard(AppWidgetsPtr->sqlwindow, XtLastTimestampProcessed(XtDisplay(w)));
   BufRemoveSelected(textD->buffer);
}

/************************************************************************/
/* Function: CopyCallback                                               */
/* Purpose : handles copying of selected text to clipboard              */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void CopyCallback(Widget w, XtPointer clientdata, XtPointer callData)
{ 
  if (((TextWidget)AppWidgetsPtr->sqlwindow)->text.textD->buffer->primary.selected)
    CopyToClipboard(AppWidgetsPtr->sqlwindow, XtLastTimestampProcessed(XtDisplay(w)));  
  else if (((TextWidget)AppWidgetsPtr->resultwindow)->text.textD->buffer->primary.selected)
    CopyToClipboard(AppWidgetsPtr->resultwindow, XtLastTimestampProcessed(XtDisplay(w)));
  else
    XBell(XtDisplay(w), 0);
}

/************************************************************************/
/* Function: PasteCallback                                              */
/* Purpose : handles pasting of text from the clipboard                 */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void PasteCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  InsertClipboard(AppWidgetsPtr->sqlwindow, XtLastTimestampProcessed(XtDisplay(AppWidgetsPtr->sqlwindow)), False);
}

/************************************************************************/
/* Function: ClearResultCallback                                        */
/* Purpose : clears the result window                                   */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void ClearResultCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  textBuffer *buffer;
  
  buffer = TextGetBuffer(AppWidgetsPtr->resultwindow);

  BufSetAll(buffer, "");
}

/************************************************************************/
/* Function: SpoolCallback                                              */
/* Purpose : handles spool file definition                              */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void SpoolCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  const char *title = "Spool";                   /* dialog title        */
  Widget rc;                                     /* rowcol widget       */
  static Widget spool_dialog = NULL;             /* dialog widget       */

  if(!spool_dialog){
    spool_dialog = XmCreateMessageDialog(AppWidgetsPtr->mainwindow, "spooldialog", NULL, 0);
  
    /* remove uneeded children */
    /*XtUnmanageChild(XmMessageBoxGetChild(spool_dialog, XmDIALOG_HELP_BUTTON));*/
    XtUnmanageChild(XmMessageBoxGetChild(spool_dialog, XmDIALOG_SYMBOL_LABEL));
    XtUnmanageChild(XmMessageBoxGetChild(spool_dialog, XmDIALOG_MESSAGE_LABEL));

    /* setup dialog */
    XtVaSetValues(spool_dialog,
		  XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL,
		  XtVaTypedArg, XmNdialogTitle, XmRString,
		  title, strlen(title)+1,
		  NULL);

    rc = XtVaCreateManagedWidget("rc", xmRowColumnWidgetClass, spool_dialog,
				 XmNnumColumns, 2,
				 XmNpacking, XmPACK_TIGHT,
				 XmNorientation, XmHORIZONTAL,
				 NULL);

    XtCreateManagedWidget("Enter spool filename:", xmLabelWidgetClass, rc, NULL, 0);
    AppWidgetsPtr->spoolfile = XtVaCreateManagedWidget("spoolfile", xmTextFieldWidgetClass,
						       rc,
						       XmNmaxLength, MAX_PATH_LEN,
						       NULL);

    /* add callback */
    XtAddCallback(spool_dialog, XmNokCallback, SpoolOKCallback, NULL);
    XtAddCallback(spool_dialog, XmNhelpCallback, get_help, "mpsql_menu_options.html");
  }

  XUTIL_SetFocus(spool_dialog, AppWidgetsPtr->spoolfile);

  if(spool_file != NULL)
    XmTextSetString(AppWidgetsPtr->spoolfile, spool_file);

  XtManageChild(spool_dialog);
}

/************************************************************************/
/* Function: SpoolOKCallback                                            */
/* Purpose : sets spool file name to user definition                    */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void SpoolOKCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  strcpy(spool_file, doubleTrim((char *)XmTextGetString(AppWidgetsPtr->spoolfile)));  
}

/************************************************************************/
/* Function: SaveOptCallback                                            */
/* Purpose : saves user defined options                                 */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void SaveOptCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  MQUEL_SaveOptions();
}

/************************************************************************/
/* Function: AboutCallback                                              */
/* Purpose : displays the about box                                     */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void AboutCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  /* dialog message strings */
  char info[255];
  const char *title = "About";
  const char *d_text = "MPSQL %s\n\nA Query Tool for PostgresSQL\n\nCopyright (c) 1996-99 Mutiny Bay Software      ";
  Widget rc, picture;
  static Widget about_dialog = NULL;

  if(!about_dialog){
    sprintf(info, d_text, MPSQL_VERSION);

    about_dialog = XmCreateMessageDialog(AppWidgetsPtr->mainwindow, "aboutdialog", NULL, 0);
  
    /* remove uneeded children */
    XtUnmanageChild(XmMessageBoxGetChild(about_dialog, XmDIALOG_HELP_BUTTON));
    XtUnmanageChild(XmMessageBoxGetChild(about_dialog, XmDIALOG_CANCEL_BUTTON));
    XtUnmanageChild(XmMessageBoxGetChild(about_dialog, XmDIALOG_SYMBOL_LABEL));
    XtUnmanageChild(XmMessageBoxGetChild(about_dialog, XmDIALOG_MESSAGE_LABEL));

    /* setup dialog */
    XtVaSetValues(about_dialog,
		  XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL,
		  XtVaTypedArg, XmNdialogTitle, XmRString,
		  title, strlen(title)+1,
		  NULL);

    rc = XtVaCreateManagedWidget("rc", xmRowColumnWidgetClass, about_dialog,
				 XmNnumColumns, 2,
				 XmNpacking, XmPACK_TIGHT,
				 XmNorientation, XmHORIZONTAL,
				 NULL);

    picture =  XtVaCreateManagedWidget("", 
				       xmLabelWidgetClass, rc,
				       XmNmarginWidth, 15,
				       XmNmarginHeight, 20,
				       XmNalignment, XmALIGNMENT_CENTER,
				       NULL);

    /* set label image */
    InstallLabeledPixmap(picture, about_xpm, 0);

    XtVaCreateManagedWidget("info", 
			    xmLabelWidgetClass, rc,
			    XtVaTypedArg, XmNlabelString, XmRString,
			    info, strlen(info)+1,
			    XmNalignment, XmALIGNMENT_CENTER,
			    NULL);
  }
  XtManageChild(about_dialog);
}

/************************************************************************/
/* Function: PrintCallback                                              */
/* Purpose : handles printer definition & submits print job             */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void PrintCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  const char *title = "Print";               /* dialog title            */
  Widget rc;                                 /* rowcol widget           */
  static Widget print_dialog = NULL;         /* dialog widget           */

  if(!print_dialog){
    print_dialog = XmCreateMessageDialog(AppWidgetsPtr->mainwindow, "printdialog", NULL, 0);
  
    /* remove uneeded children */
    /*XtUnmanageChild(XmMessageBoxGetChild(print_dialog, XmDIALOG_HELP_BUTTON));*/
    XtUnmanageChild(XmMessageBoxGetChild(print_dialog, XmDIALOG_SYMBOL_LABEL));
    XtUnmanageChild(XmMessageBoxGetChild(print_dialog, XmDIALOG_MESSAGE_LABEL));

    /* setup dialog */
    XtVaSetValues(print_dialog,
		  XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL,
		  XtVaTypedArg, XmNdialogTitle, XmRString,
		  title, strlen(title)+1,
		  NULL);

    rc = XtVaCreateManagedWidget("rc", xmRowColumnWidgetClass, print_dialog,
				 XmNnumColumns, 2,
				 XmNpacking, XmPACK_TIGHT,
				 XmNorientation, XmHORIZONTAL,
				 NULL);

    XtCreateManagedWidget("Enter target printer:", xmLabelWidgetClass, rc, NULL, 0);
    AppWidgetsPtr->printer = XtVaCreateManagedWidget("printer", xmTextFieldWidgetClass,
						     rc,
						     XmNmaxLength, PRINT_NAME_SZ,
						     NULL);

    /* add callback */
    XtAddCallback(print_dialog, XmNokCallback, PrintOKCallback, NULL);
    XtAddCallback(print_dialog, XmNhelpCallback, get_help, "mpsql_menu_file.html");
  }

  XUTIL_SetFocus(print_dialog, AppWidgetsPtr->printer);

  if(printer != NULL)
    XmTextSetString(AppWidgetsPtr->printer, printer);

  XtManageChild(print_dialog);
}

/************************************************************************/
/* Function: PrintOKCallback                                            */
/* Purpose : calls DB_PrintBuffer()                                     */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void PrintOKCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  strcpy(printer, doubleTrim((char*)XmTextGetString(AppWidgetsPtr->printer)));
  if(strlen(printer) == 0)
    MSGBOX_Error("Print", "You must specify a printer!", AppWidgetsPtr->mainwindow);
  DB_PrintBuffer(buffer_curr, printer);
}

/************************************************************************/
/* Function: ExecSQLCallback                                            */
/* Purpose : calls DB_SubmitQy()                                        */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void ExecSQLCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  if(!qy_running)
    DB_SubmitQy();
  else
    DB_CancelQy();
}

/************************************************************************/
/* Function: CascadeCallback                                            */
/* Purpose : creates a menu of open buffers to select.                  */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void CascadeCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  int i;
  static Widget btn[255];
  static int list_cnt = 0;
  LLISTbuffer *curr;    
  char tmp[4];

  for(i = 0; i < list_cnt; i++)
    XtUnrealizeWidget(btn[i]);

  for(i = 0; i < list_cnt; i++)
    XtDestroyWidget(btn[i]);

  list_cnt = LLIST_Count();

  for(i = 0; i < list_cnt; i++){
    curr = LLIST_Get(i+1);
    btn[i] = XtVaCreateManagedWidget(itoa(i, tmp),
				     xmPushButtonWidgetClass,
				     AppWidgetsPtr->buffermenu,
				     XtVaTypedArg, XmNlabelString, XmRString,
				     curr->filename, strlen(curr->filename)+1,
				     NULL);
    XtAddCallback(btn[i], XmNactivateCallback, BufferCallback, NULL);
  }
}

/************************************************************************/
/* Function: BufferCallback                                             */
/* Purpose : sets the SQL window to the selected buffer                 */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void BufferCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  LLISTbuffer *curr;

  curr = LLIST_Get(atoi(XtName(w)) + 1);

  if(curr != buffer_curr)
    buffer_curr = DB_SetBuffer(curr);
}

/************************************************************************/
/* Function: TextModifiedCallback                                       */
/* Purpose : flags a buffer as modified if text changes                 */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void TextModifiedCallback(int pos, int nInserted, int nDeleted, int nRestyled, 
    char *deletedText, void *cbArg)
{
  buffer_curr->modified = 1;
}

/************************************************************************/
/* Function: DeleteBufferCallback                                       */
/* Purpose : handles user request to delete the current buffer          */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void DeleteBufferCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  /* dial title strings */
  const char *title = "Delete Buffer";
  const char *del_msg = "Buffer has been modified,\ndelete without saving changes?";
  static Widget del_dialog = NULL;

  /* if the buffer is modified prompt the user to continue delete */
  if(buffer_curr->modified == 1){
    if(!del_dialog){
      del_dialog = XmCreateQuestionDialog(AppWidgetsPtr->mainwindow, "deldialog", NULL, 0);
    
      /* remove uneeded children */
      XtUnmanageChild(XmMessageBoxGetChild(del_dialog, XmDIALOG_HELP_BUTTON));
    
      /* setup dialog */
      XtVaSetValues(del_dialog,
		    XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL,
		    XtVaTypedArg, XmNdialogTitle, XmRString,
		    title, strlen(title)+1,
		    XtVaTypedArg, XmNmessageString, XmRString,
		    del_msg, strlen(del_msg)+1, NULL);
    
      /* add callback */
      XtAddCallback(del_dialog, XmNokCallback, DeleteBufferOKCallback, NULL);
    }
    
    XtManageChild(del_dialog);
  }
  else
    DB_DeleteBuffer(buffer_curr);
}

/************************************************************************/
/* Function: DeleteBufferOKCallback                                     */
/* Purpose : deletes the current buffer on user confirmation            */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void DeleteBufferOKCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  DB_DeleteBuffer(buffer_curr);
}

/************************************************************************/
/* Function: HighlightToggleCallback                                    */
/* Purpose : Refreshes sql window when syntax highlighting is toggled   */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void HighlightToggleCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  if(XtWindow(AppWidgetsPtr->mainwindow))
    buffer_curr = DB_SetBuffer(buffer_curr);
}

/************************************************************************/
/* Function: TabCallback                                                */
/* Purpose : opens the tab setting dialog                               */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void TabCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  const char *title = "Tabs";                    /* dialog title        */
  Widget rc;                                     /* rowcol widget       */
  static Widget tab_dialog = NULL;               /* dialog widget       */
  char tmp[5];

  if(!tab_dialog){
    tab_dialog = XmCreateMessageDialog(AppWidgetsPtr->mainwindow, "tabdialog", NULL, 0);
  
    XtVaSetValues(tab_dialog,
		  XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL,
		  XtVaTypedArg, XmNdialogTitle, XmRString,
		  title, strlen(title)+1,
		  NULL);

    /* remove uneeded children */
    XtUnmanageChild(XmMessageBoxGetChild(tab_dialog, XmDIALOG_SYMBOL_LABEL));
    XtUnmanageChild(XmMessageBoxGetChild(tab_dialog, XmDIALOG_MESSAGE_LABEL));

    /* setup dialog */
    rc = XtVaCreateManagedWidget("rc", xmRowColumnWidgetClass, tab_dialog,
				 XmNnumColumns, 2,
				 XmNpacking, XmPACK_TIGHT,
				 XmNorientation, XmHORIZONTAL,
				 NULL);

    XtCreateManagedWidget("Tab spacing:", xmLabelWidgetClass, rc, NULL, 0);
    AppWidgetsPtr->tabspace = XtVaCreateManagedWidget("tab", xmTextFieldWidgetClass,
						 rc,
						 XmNmaxLength, 4,
						 NULL);

    /* add callback */
    XtAddCallback(tab_dialog, XmNokCallback, TabOKCallback, NULL);
    XtAddCallback(tab_dialog, XmNhelpCallback, get_help, "mpsql_menu_options.html");
  }

  XUTIL_SetFocus(tab_dialog, AppWidgetsPtr->tabspace);

  XmTextSetString(AppWidgetsPtr->tabspace, itoa(tabs, tmp));

  XtManageChild(tab_dialog);
}

/************************************************************************/
/* Function: TabOKCallback                                              */
/* Purpose : Sets the tab spacing to the user specified setting         */
/* Params  : standard callback stuff                                    */
/* Returns : nothing                                                    */
/* Notes   :                                                            */
/************************************************************************/

void TabOKCallback(Widget w, XtPointer clientdata, XtPointer callData)
{
  int val;
  char *val_str, *end_ptr;

  val_str = doubleTrim((char*)XmTextGetString(AppWidgetsPtr->tabspace));
  removeWhiteSpace(val_str);
  val = strtol(val_str, &end_ptr, 10);
  
  if((strlen(val_str) == 0)||(*end_ptr != '\0')){
    MSGBOX_Error("Tabs","Entered value is not a number", 
		 AppWidgetsPtr->mainwindow);
    XtFree(val_str); 
    return;
  }

  if(val <= 0 || val >= 1000) {
    MSGBOX_Error("Tabs","Tab spacing out of range", 
		 AppWidgetsPtr->mainwindow);
    XtFree(val_str);
    return;
  }

  XtVaSetValues(AppWidgetsPtr->sqlwindow, textNemulateTabs, val, NULL);
  tabs = val;
  XtFree(val_str);
}

/************************************************************************/
/* Function: GetColSeparator                                            */
/* Purpose : returns the column separator character                     */
/* Params  :                                                            */
/* Returns : char to set                                                */
/* Notes   :                                                            */
/************************************************************************/

char GetColSeparator(void)
{
  if(XmToggleButtonGetState(AppWidgetsPtr->none))
    return '\0';
  else if(XmToggleButtonGetState(AppWidgetsPtr->tab))
    return '\t';
  else if(XmToggleButtonGetState(AppWidgetsPtr->space))
    return ' ';
  else if(XmToggleButtonGetState(AppWidgetsPtr->comma))
    return ',';
}

/************************************************************************/
/* Function: GetColSeparatorId                                          */
/* Purpose : returns the column separator character const               */
/* Params  :                                                            */
/* Returns : 0: NONE, 1: TAB, 2: SPACE, 3: COMMA                        */
/* Notes   :                                                            */
/************************************************************************/

int GetColSeparatorId(void)
{
  if(XmToggleButtonGetState(AppWidgetsPtr->none))
    return NONE;
  else if(XmToggleButtonGetState(AppWidgetsPtr->tab))
    return TAB;
  else if(XmToggleButtonGetState(AppWidgetsPtr->space))
    return SPACE;
  else if(XmToggleButtonGetState(AppWidgetsPtr->comma))
    return COMMA;
}

/************************************************************************/
/* Function: SetColSeparator                                            */
/* Purpose : handles setting column separator character                 */
/* Params  : sep_value : int represent char to set                      */
/*           0: NONE, 1: TAB, 2: SPACE, 3: COMMA                        */
/* Returns :                                                            */
/* Notes   :                                                            */
/************************************************************************/

void SetColSeparator(int sep_value)
{
  if(sep_value == NONE)
    XmToggleButtonSetState(AppWidgetsPtr->none, 1, TRUE);
  else if(sep_value == TAB)
    XmToggleButtonSetState(AppWidgetsPtr->tab, 1, TRUE);
  else if(sep_value == SPACE)
    XmToggleButtonSetState(AppWidgetsPtr->space, 1, TRUE);
  else if(sep_value == COMMA)
    XmToggleButtonSetState(AppWidgetsPtr->comma, 1, TRUE);
}

/************************************************************************/
/* Function: GetAlignment                                               */
/* Purpose : returns the column alignment state                         */
/* Params  :                                                            */
/* Returns : int representing the column alignment state                */
/* Notes   :                                                            */
/************************************************************************/

int GetAlignment(void)
{
  if(XmToggleButtonGetState(AppWidgetsPtr->align))
    return 0;
  else if(XmToggleButtonGetState(AppWidgetsPtr->standard))
    return 1;
  else if(XmToggleButtonGetState(AppWidgetsPtr->expand))
    return 2;
}

/************************************************************************/
/* Function: SetAlignment                                               */
/* Purpose : handles setting column alignment state                     */
/* Params  : align_value : int representing alignment state             */
/*           0: aligned, 1: standard, 2: expanded                       */
/* Returns :                                                            */
/* Notes   :                                                            */
/************************************************************************/

void SetAlignment(int align_value)
{
  if(align_value == 0)
    XmToggleButtonSetState(AppWidgetsPtr->align, 1, TRUE);
  else if(align_value == 1)
    XmToggleButtonSetState(AppWidgetsPtr->standard, 1, TRUE);
  else if(align_value == 2)
    XmToggleButtonSetState(AppWidgetsPtr->expand, 1, TRUE);
}

/************************************************************************/
/* Function: GetType                                                    */
/* Purpose : returns the output type state                              */
/* Params  :                                                            */
/* Returns : int representing the output state                          */
/* Notes   :                                                            */
/************************************************************************/

int GetType(void)
{
  if(XmToggleButtonGetState(AppWidgetsPtr->text))
    return 0;
  else if(XmToggleButtonGetState(AppWidgetsPtr->html))
    return 1;
}

/************************************************************************/
/* Function: SetType                                                    */
/* Purpose : handles setting the output state                           */
/* Params  : align_value : int representing the output state            */
/*           0: text, 1: html                                           */
/* Returns :                                                            */
/* Notes   :                                                            */
/************************************************************************/

void SetType(int out_value)
{
  if(out_value == 0)
    XmToggleButtonSetState(AppWidgetsPtr->text, 1, TRUE);
  else if(out_value == 1)
    XmToggleButtonSetState(AppWidgetsPtr->html, 1, TRUE);
}

/************************************************************************/
/* Function: GetHighlight                                               */
/* Purpose : returns the syntax highlight state                         */
/* Params  :                                                            */
/* Returns : int representing the state                                 */
/* Notes   :                                                            */
/************************************************************************/

int GetHighlight(void)
{
  if(XmToggleButtonGetState(AppWidgetsPtr->plain))
    return 0;
  else if(XmToggleButtonGetState(AppWidgetsPtr->highlight))
    return 1;
}

/************************************************************************/
/* Function: SetHighlight                                               */
/* Purpose : handles setting the syntax highlight state                 */
/* Params  : align_value : int representing the state                   */
/*           0: off, 1: on                                              */
/* Returns :                                                            */
/* Notes   :                                                            */
/************************************************************************/

void SetHighlight(int highlight_value)
{
  if(highlight_value == 0)
    XmToggleButtonSetState(AppWidgetsPtr->plain, 1, TRUE);
  else if(highlight_value == 1)
    XmToggleButtonSetState(AppWidgetsPtr->highlight, 1, TRUE);
}
