//-< SYNC.H >--------------------------------------------------------*--------*
// FastDB                    Version 1.0         (c) 1999  GARRET    *     ?  *
// (Main Memory Database Management System)                          *   /\|  *
//                                                                   *  /  \  *
//                          Created:     20-Nov-98    K.A. Knizhnik  * / [] \ *
//                          Last update: 20-Dec-98    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// Intertask synchonization primitives
//-------------------------------------------------------------------*--------*

#ifndef __SYNC_H__
#define __SYNC_H__

#ifdef _WIN32
class dbMutex { 
    CRITICAL_SECTION cs;
  public:
    dbMutex() { 
	InitializeCriticalSection(&cs);
    }
    ~dbMutex() { 
	DeleteCriticalSection(&cs);
    }
    void lock() { 
	EnterCriticalSection(&cs);
    }
    void unlock() { 
	LeaveCriticalSection(&cs);
    }
};

#define thread_proc WINAPI

class dbThread { 
    HANDLE h;
  public:
    void create(void (thread_proc* f)(void*), void* arg) { 
	DWORD threadid;
	h = CreateThread(NULL, NULL, LPTHREAD_START_ROUTINE(f), arg,
			 0, &threadid);
    }
    void join() { 
	WaitForSingleObject(h, INFINITE);
    }
    dbThread() { 
	h = NULL; 
    }
    ~dbThread() { 
	if (h != NULL) { 
	    CloseHandle(h);
	}
    }
    static int numberOfProcessors() { 
	SYSTEM_INFO sysinfo;
	GetSystemInfo(&sysinfo);
	return sysinfo.dwNumberOfProcessors;
    }
};
    
class dbInitializationMutex { 
    HANDLE m;
  public: 
    enum initializationStatus { 
	InitializationError, 
	AlreadyInitialized,
	NotYetInitialized
    };
    initializationStatus initialize(char const* name) { 
	initializationStatus status;
	m = CreateMutex(NULL, true, name);
	if (GetLastError() == ERROR_ALREADY_EXISTS) { 
	    status = WaitForSingleObject(m, INFINITE) == WAIT_OBJECT_0 
		   ? AlreadyInitialized : InitializationError;
	    ReleaseMutex(m);
	} else if (m != NULL) { 
	    status = NotYetInitialized;
	} else { 
	    status = InitializationError;
	}
	return status;
    }
    void done() { 
	ReleaseMutex(m);
    }
    void close() {
	CloseHandle(m);
    }
};


const int dbMaxSemValue = 1000000;


class dbSemaphore { 
    HANDLE s;
  public:
    bool wait(unsigned msec = INFINITE) { 
	int rc = WaitForSingleObject(s, msec);
	assert(rc == WAIT_OBJECT_0 || rc == WAIT_TIMEOUT);
	return rc == WAIT_OBJECT_0;
    }
    void signal(unsigned inc = 1) {
	if (inc != 0) { 
	    ReleaseSemaphore(s, inc, NULL);
	}
    }
    void reset() { 
	while (WaitForSingleObject(s, 0) == WAIT_OBJECT_0);
    }    
    bool open(char const* name, unsigned initValue = 0) {
	s = CreateSemaphore(NULL, initValue, dbMaxSemValue, name);
	return s != NULL; 
    }
    void close() {
	CloseHandle(s);
    }
};

class dbEvent { 
    HANDLE e;
  public:
    bool wait(unsigned msec = INFINITE) { 
	int rc = WaitForSingleObject(e, msec);
	assert(rc == WAIT_OBJECT_0 || rc == WAIT_TIMEOUT);
	return rc == WAIT_OBJECT_0;
    }
    void signal() {
	SetEvent(e);
    }
    void reset() {
	ResetEvent(e);
    }
    bool open(char const* name, bool signaled = false) {
	e = CreateEvent(NULL, true, signaled, name);
	return e != NULL; 
    }
    void close() {
	CloseHandle(e);
    }
};

template<class T>
class dbThreadContext { 
    int index;
  public:
    T* get() { 
	return (T*)TlsGetValue(index);
    }
    void set(T* value) { 
	TlsSetValue(index, value);
    }
    dbThreadContext() { 
	index = TlsAlloc();
	assert(index != TLS_OUT_OF_INDEXES);
    }
    ~dbThreadContext() { 
	TlsFree(index);
    }
};

template<class T>
class dbSharedObject { 
    T*     ptr;
    HANDLE h;
  public:

    bool open(char* name) { 
	h = CreateFileMapping(INVALID_HANDLE_VALUE,
			      NULL, PAGE_READWRITE, 0, 
			      sizeof(T), name);
	if (h == NULL) { 
	    return false;
	}
	ptr = (T*)MapViewOfFile(h, FILE_MAP_ALL_ACCESS, 0, 0, 0);
	if (ptr == NULL) { 
	    CloseHandle(h);
	    return false;
	}
	return true;
    }

    T* get() { return ptr; }

    void close() { 
	UnmapViewOfFile(ptr);
	CloseHandle(h);
    }
};

typedef long sharedsem_t;

class dbGlobalCriticalSection { 
    HANDLE       event;
    sharedsem_t* count;

  public:
    void enter() { 
	if (InterlockedDecrement(count) != 0) { 
	    // another process is in critical section
	    int rc = WaitForSingleObject(event, INFINITE);
	    assert (rc == WAIT_OBJECT_0);
	}
    }

    void leave() { 
	if (InterlockedIncrement(count) <= 0) { 
	    // some other processes try to enter critical section
	    SetEvent(event);
	}
    }

    bool open(char const* name, long* count) { 
	this->count = count;
	event = OpenEvent(EVENT_ALL_ACCESS, FALSE, name);
	return event != NULL;
    }
    bool create(char const* name, long* count) { 
	this->count = count;
	*count = 1;
	event = CreateEvent(NULL, false, false, name);
	return event != NULL;
    }
    void close() { 
	CloseHandle(event);
    }
};
	
    
#else // Unix

#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/shm.h>
#include <sys/mman.h>

#define thread_proc

#ifdef PTHREADS

#include <pthread.h>

class dbMutex { 
    pthread_mutex_t cs;
  public:
    dbMutex() { 
	pthread_mutex_init(&cs, NULL);
    }
    ~dbMutex() { 
	pthread_mutex_destroy(&cs);
    }
    void lock() { 
	pthread_mutex_lock(&cs);
    }
    void unlock() { 
	pthread_mutex_unlock(&cs);
    }
};

class dbThread { 
    pthread_t thread;
  public:
    void create(void (thread_proc* f)(void*), void* arg) { 
	pthread_create(&thread, NULL, (void*(*)(void*))f, arg);
    }
    void join() { 
	void* result;
	pthread_join(thread, &result);
    }
    static int numberOfProcessors();
};

template<class T> 
class dbThreadContext { 
    pthread_key_t key;
  public:
    T* get() { 
	return (T*)pthread_getspecific(key);
    }
    void set(T* value) { 
	pthread_setspecific(key, value);
    }
    dbThreadContext() { 
        pthread_key_create(&key, NULL);
    }
    ~dbThreadContext() { 
	pthread_key_delete(key);
    }
};

#else

class dbMutex { 
   public:
    void lock() {}
    void unlock() {}
};

class dbThread { 
  public:
    void create(void (thread_proc* f)(void*), void* arg) { f(arg); }
    void join() {}
    static int numberOfProcessors() { return 1; }
};

template<class T>
class dbThreadContext { 
    T* value;
  public:
    T* get() { 
	return value;
    }
    void set(T* value) { 
	this->value = value;
    }
    dbThreadContext() { value = NULL; }
};

#endif

#define INFINITE (~0U)
extern char const* keyFileDir; // default value: "/tmp/" 

class dbInitializationMutex { 
    int semid;
  public: 
    enum initializationStatus { 
	InitializationError, 
	AlreadyInitialized,
	NotYetInitialized
    };
    initializationStatus initialize(char const* name);
    void done(); 
    void close();
};


class dbSemaphore { 
    int s;
  public:
    bool wait(unsigned msec = INFINITE);
    void signal(unsigned inc = 1);
    bool open(char const* name, unsigned initValue = 0);
    void reset();
    void close();
};

class dbEvent { 
    int e;
  public:
    bool wait(unsigned msec = INFINITE);
    void signal();
    void reset();
    bool open(char const* name, bool signaled = false);
    void close();
};

template<class T>
class dbSharedObject { 
    T*  ptr;
    int shm;
  public:

    bool open(char* name) { 
	char* fileName = name;
	if (*name != '/') { 
	    fileName = new char[strlen(name)+strlen(keyFileDir)+1];
	    sprintf(fileName, "%s%s", keyFileDir, name);
	}
	int fd = ::open(fileName, O_RDWR|O_CREAT, 0777);
	if (fd < 0) { 
	    if (fileName != name) { 
		delete[] fileName;
	    }
	    return false;
	} 
	::close(fd);
	int key = ftok(fileName, '0');
	if (fileName != name) { 
	    delete[] fileName;
	}
	if (key < 0) { 
	    return false;
	}
	shm = shmget(key, sizeof(T), IPC_CREAT|0777);
	if (shm < 0) { 
	    return false;
	}
	ptr = (T*)shmat(shm, NULL, 0);
	return (ptr != (T*)-1);
    }

    T* get() { return ptr; }

    void close() { 
	shmdt((char*)ptr);
    }
};

#if defined(__osf__)

typedef msemaphore sharedsem_t;

class dbGlobalCriticalSection { 
    sharedsem_t* sem;
  public:
    void enter() { 
	msem_lock(sem, 0);
    }
    void leave() { 
	msem_unlock(sem, 0);
    }
    bool open(char const*, sharedsem_t* shr) { 
	sem = shr;
	return true;
    }
    bool create(char const*, sharedsem_t* shr) { 
	sem = shr;
	msem_init(shr, MSEM_UNLOCKED);
	return true;
    }
    void close() {}
};
	

#elif defined(__svr4__)

#include <synch.h>
typedef sema_t sharedsem_t;

class dbGlobalCriticalSection { 
    sharedsem_t* sem;
  public:
    void enter() { 
	sema_wait(sem);
    }
    void leave() { 
	sema_post(sem);
    }
    bool open(char const*, sharedsem_t* shr) { 
	sem = shr;
	return true;
    }
    bool create(char const*, sharedsem_t* shr) { 
	sem = shr;
	sema_init(shr, 1, USYNC_PROCESS, NULL);
	return true;
    }
    void close() {}
};
	
#else

typedef long sharedsem_t;

class dbGlobalCriticalSection { 
    int          semid;
    sharedsem_t* count;

  public:
    void enter(); 
    void leave();
    bool open(char const* name, sharedsem_t* shr);
    bool create(char const* name, sharedsem_t* shr);
    void close() {}
};
#endif

#endif

class dbCriticalSection { 
  private:
    dbMutex& mutex;
  public:
    dbCriticalSection(dbMutex& guard) : mutex(guard) {
	mutex.lock();
    }
    ~dbCriticalSection() { 
	mutex.unlock();
    }
};
	
#endif


