/*****************************************************************************

	vdata.c 

	Virtual data access. Data may be: in memory, on disk, in the
	datafile. Provides generic access.

*****************************************************************************/
#include <stdio.h>
#include <ctype.h>
#include <unistd.h>
#include <stdlib.h>
#include "global.h"

/*
 * Rationale.
 *  if !ELOADED then the data of the element is referenced at the mmaped
 *   datafile.
 *  else if the element < BIG_ELEMENT, it is in memory
 *  else the element->data is a filename of the temporary file holding
 *   the data.
 *   this filename, has a number attached which _may_ be the id of
 *    the element. In order to avoid the void of ids after a defrag
 *    this number is always incrementing.
 */

/*
 * If the data of an element has to be loaded from a temporary file, then
 * we'll do it in chunks. We shouldn't want to load the entire tmp file to
 * memory, imagine a client at a 2400Kbps connection requesting a 3MB element.
 * Actually a proxy should be used to do this fast; however this way we 
 * amortize memory usage....
 */
#define CHUNK_SIZE 1024

void data_to_stream (Element *e, FILE *out)
{
	if (e->len < BIG_ELEMENT || !ELOADED(e))
		fwrite (e->data, 1, e->len, out);
	else
	{
		char tmp [(e->len < CHUNK_SIZE) ? e->len : CHUNK_SIZE];
		unsigned int i, left;
		FILE *inf = fopen (e->data, "r");
		if (!inf)
		{
			fprintf (stderr, "Oops: Can't open %s. Fatal\n", e->data);
			exit (1);
		}

		for (i = ((left = e->len) < CHUNK_SIZE) ? e-> len : CHUNK_SIZE;
		     left; i = left < CHUNK_SIZE ? left : CHUNK_SIZE)
		{
			fread (tmp, 1, i, inf);
			fwrite (tmp, 1, i, out);
			left -= i;
		}
		fclose (inf);
	}
}

void freedataof (Element *e)
{
	if (ELOADED (e))
	{
		if (e->len >= BIG_ELEMENT)
			unlink (e->data);
		free (e->data);
	}
	e->data = NULL;
}

/*
 * The allocated data MUST be malloc'd -- in data segment.
 * We would have to malloc & memcpy every time otherwise...
 * The caller MUST NOT free the data if we successfully get through here.
 */
void datato (Element *e, char *ptr, unsigned int len)
{
	static int maxid = 0;

	if ((e->len = len) < BIG_ELEMENT)
	{
		e->data = ptr;
#ifdef AUTO_CISTR
		if (len < MSS && (e->pmt_bool.ch [0] & 2) == 0)
		{
			extern int register_type (Element*, short int);
			char *p;
			int i;
			for (i = 0, p = e->data; i < len; i++, p++)
				if (!isprint (*p)) goto nope;
			register_type (e, 0);
nope:
		}
#endif
	}
	else
	{
		char tmp [strlen (TMP_PREFIX) + 10];
		FILE *f;

		if (e->id > maxid) maxid = e->id;
		else maxid++;

		sprintf (tmp, TMP_PREFIX, maxid);
		GUARD (e->data = (char*) malloc (strlen (tmp) + 1))
		strcpy (e->data, tmp);
		f = fopen (tmp, "w");
		fwrite (ptr, 1, len, f);
		fclose (f);
		free (ptr);
	}
	ISLOADED(e);
}

/*
 * Data comparison with a buffer.
 * Comparison is too performed in chunks to avoid brusts of memory reqs.
 * Furthermore, _few big elements of the same size will exist. Thus comparing
 * few initial bytes is better than loading the entire data to mem.
 */
int comparedataof (Element *e, char *ptr, unsigned int len)
{
	if (!ELOADED(e))
		return memcmp (e->data, ptr, len);
	{
	char tmp [(e->len < CHUNK_SIZE) ? e->len : CHUNK_SIZE];
	unsigned int i, left, np;
	FILE *inf = fopen (e->data, "r");

	for (np = 0, i = ((left = e->len) < CHUNK_SIZE) ? e-> len : CHUNK_SIZE;
	     left; i = left < CHUNK_SIZE ? left : CHUNK_SIZE)
	{
		fread (tmp, 1, i, inf);
		if (memcmp (tmp, ptr + np, i))
			break;
		left -= i;
		np += i;
	}

	fclose (inf);
	return left; /* We do not care about greater or
		      * less than and respectively equal to or
		      * in turn less or more than otherwise. */ }
}
