/*****************************************************************************
	sfmalloc.c

	Efficient Dynamic memory allocation for small fixed-size
	structures. See sfmalloc.README for more.

*****************************************************************************/
/* Sizeof (pointer) == MUST BE EQUAL == Sizeof (unsigned int) == 32-bit */
/* sfmalloc is NOT thread safe. Make sure that only one thread
 * accesses an SFMPOOL at a time.
 */

#include <stdlib.h>
#include <string.h>
#include "sfmalloc.h"
#ifndef INCR_SHIFT
#include "arch.h"
#endif
#include "config.h"

#ifdef USE_SFMALLOC

#define ALIGN 4

struct llist {
	struct llist *next, *prev;
};

SFMPOOL *sfmalloc_creat (size_t esize, size_t csize)
{
	SFMPOOL *t;

	if (csize == 0)
		csize = DEFAULT_CHUNK_SIZE;
	if (!(t = (SFMPOOL*) malloc (sizeof (SFMPOOL))))
		return NULL;
	t->first_chunk = NULL;
	t->element_size = esize;
	t->ctrl_ptrs = csize;
	t->aligned_element_size = esize;
	if (esize % ALIGN)
		t->aligned_element_size += ALIGN - esize % ALIGN;
	t->chunk_size = 8 + csize * 4 + csize * 32 * t->aligned_element_size;
	t->tot_chunks = t->tot_elems = 0;
	t->chunk_cache = NULL;

	return t;
}

void sfmalloc_release (SFMPOOL *ptr)
{
	struct llist *lp;
	char *cp;

	if (!ptr) return;
	cp = ptr->first_chunk;
	while (cp)
	{
		lp = (struct llist*) cp;
		cp = (char*) lp->next;
		free (lp);
	}
	free (ptr);
}

static char *new_chunk (size_t s, size_t ctp, struct llist *lp)
{
	char *t;
	struct llist *pl;

	t = (char*) malloc (s);
	if (!t) return NULL;
	pl = (struct llist*) t;
	pl->prev = lp;
	pl->next = NULL;
	lp->next = pl;
	memset (t + 8, 0, 4 * ctp);
	return t;
}
	
void *sfmalloc (SFMPOOL *ptr)
{
	struct llist *lp;
	unsigned int *ip;
	unsigned int i, j, k;
	char *cp;

	/* Check for starting chunk. Create if doesn't exist */
	if (!ptr->first_chunk)
	{
		struct llist ll;
		if (!new_chunk (ptr->chunk_size, ptr->ctrl_ptrs, &ll))
			return NULL;
#ifdef SFMALLOC_STATS
		++ptr->tot_chunks;
#endif
		ptr->first_chunk = (char*) ll.next;
		ll.next->prev = NULL;
	}

	/* Find an empty slot */
	lp = (ptr->chunk_cache) ?
	     (struct llist*) ptr->chunk_cache :
	     (struct llist*) ptr->first_chunk;
	while (1)
	{
		ip = (unsigned int*) lp;
		ip += 2;
		for (i = 0; i < ptr->ctrl_ptrs; i++, ip++)
		if (~*ip)
			for (k = 1, j = 0; j < 32; j++, k *=2)
			if (!(k & *ip)) goto found;
		if (!lp->next)
			if (!new_chunk (ptr->chunk_size, ptr->ctrl_ptrs, lp))
#ifdef SFMALLOC_STATS
				++ptr->tot_chunks,
#endif
				return NULL;
		lp = lp->next;
	}

	/* Now `lp' is the address of the chunk.
	 * `32 * i + j' is the slot number.
	 * Mark the slot used and..
	 *  be careful on pointer arithmetic !
	 */
found:
	*ip |= k;
	cp = (char*) lp;
	cp += 8 + ptr->ctrl_ptrs * 4 + (32 * i + j) * ptr->aligned_element_size;
#ifdef SFMALLOC_STATS
	++ptr->tot_elems;
#endif
	ptr->chunk_cache = (char*) lp;

	return cp;
}

static void rmv_chunk (SFMPOOL *ptr, struct llist *lp)
{
	if (lp->prev) lp->prev->next = lp->next;
	else ptr->first_chunk = (char*) lp->next;
	if (lp->next) lp->next->prev = lp->prev;
#ifdef SFMALLOC_STATS
	--ptr->tot_chunks;
#endif
	free (lp);
}

void sfmalloc_free (SFMPOOL *ptr, void *elem)
{
	struct llist *lp;
	char *addr;
	unsigned int i, j, k;
	unsigned int *ip;

	for (lp = (struct llist*) ptr->first_chunk; lp; lp = lp->next)
	{
		addr = (char*) lp;
		if ((char*) elem > addr
		&& (char*) elem < addr + ptr->chunk_size)
			goto found_chunk;
	}
	return;

found_chunk:
	/* Just mark the slot emtpy */
	addr += 8 + ptr->ctrl_ptrs * 4;
	i = ((char*) elem - addr) / ptr->aligned_element_size;
	j = i / 8, k = i % 8;
	addr = (char*) lp;
	addr += 8 + j;
	i = 1 INCR_SHIFT k;
	*addr &= ~i;
#ifdef SFMALLOC_STATS
	--ptr->tot_elems;
#endif
	ptr->chunk_cache = NULL;

	/* Check if all the slots of the chunk are empty.
	 * Remove the chunk then.
	 */
	ip = (unsigned int*) lp;
	for (ip += 2, i = 0; i < ptr->ctrl_ptrs; i++, ip++)
		if (*ip) return;
	rmv_chunk (ptr, lp);
}

#endif
