/******************************************************************************
	dbstree.c 

	Dynamic Binary Search Tree for storing Lndbase Groups.
	Check dbstree.tex for technical information on this tree.

*****************************************************************************/

#include <stdio.h>
#include "global.h"

#define MAGIC_NUMBER 32

typedef Group node;

/*
 * Balancing a binary tree. This happens whenever the height reaches 32.
 * The operation is O(n) where n the nodes of the tree.
 * For 1,500,000 nodes on a 686 400MHz this is done in less than 4 secs.
 */
static unsigned int suint;
static node **snpp;
static void count_nodes (const node *n)
{
	++suint;
	if (n->less) count_nodes (n->less);
	if (n->more) count_nodes (n->more);
}
static void tree_to_array (node *n)
{
	if (n->less) tree_to_array (n->less);
	*snpp++ = n;
	if (n->more) tree_to_array (n->more);
}
static void balance (node *par)
{
	node **npp;
	unsigned long long i, j, k, D, Y, k2, k3;
	extern unsigned int dbst_insert (node*);

	suint = 0;
	count_nodes (par->child);
	npp = snpp = (node**) alloca (sizeof (node*) * suint);
	tree_to_array (par->child);

	par->child = npp [suint / 2];
	for (D = suint + 1, i = 4; i <= D; i *= 2)
		for (j = 2; j < i; j += 4)
		{
			k3 = suint * j / i;
			npp [k3]->less = npp [suint * (j - 1) / i],
			npp [k3]->more = npp [suint * (j + 1) / i];
		}
	k = suint + 1 - (Y = i / 2);
	if (k == 0)
	{
		for (i /=2, j = 1; j < i; j += 2)
			k3 = suint * j / i,
			npp [k3]->less = npp [k3]->more = NULL;
		return;
	}

	for (j = 2; j < i; j += 4)
	{
		k3 = suint * j / i;
		D = (k2 = (j - 1) * suint / i) * Y % suint;
		if (D >= k || D == 0)
			npp [k3]->less = NULL;
		else
		{
			npp [k3]->less = npp [k2];
			npp [k2]->less = npp [k2]->more = NULL;
		}
		D = (k2 = (j + 1) * suint / i) * Y % suint;
		if (D >= k || D == 0)
			npp [k3]->more = NULL;
		else
		{
			npp [k3]->more = npp [k2];
			npp [k2]->less = npp [k2]->more = NULL;
		}
	}
	dbst_insert (npp [0]);
}

/*
 * Find an entry in the tree. less than 32 steps.
 * n: the root of the tree.
 */
node *dbst_find (node *n, const char *str)
{
	int i;

	while (n)
		if ((i = strcmp (n->node_name, str)) == 0) break;
		else n = (i > 0) ? n->less : n->more;

	return n;
}

/*
 * Insert a new node. The node->parent MUST be set.
 * The root of the tree is found by node->parent->child.
 */
unsigned int dbst_insert (node *n)
{
	node *np;
	char *cp = n->node_name;
	int h;

#ifdef BUG_TRAP
	assert (((void)"The parent must be set!", n->parent));
#endif
	n->less = n->more = NULL;
	if (!(np = n->parent->child))
	{
		n->parent->child = n;
		return 0;
	}

	for (h = 0;; h++)
		if (strcmp (np->node_name, cp) > 0)
			if (!np->less) { np->less = n; break; }
			else np = np->less;
#ifdef BUG_TRAP
		else if (strcmp (np->node_name, cp) == 0)
			assert ("Duplicate Nodename in Group DBSTree"==0);
#endif
		else
			if (!np->more) { np->more = n; break; }
			else np = np->more;

	if (h == MAGIC_NUMBER)
	{
#ifdef BUG_TRAP
		fputs ("***** I have to balance", stderr);
#endif
		balance (n->parent);
		return 0;
	} 
#ifdef BUG_TRAP
	else if (h > MAGIC_NUMBER)
        {
		fputs ("Tree reached 2^32 nodes ??\n", stderr);
        }
#endif
	return h;
}

/*
 * Remove an element from the tree.
 */
void dbst_rmv (const node *n)
{
	node *np, *nl, *nr, *nlp, *nrp;
	node *par = n->parent;
	char *cp = n->node_name;
	int isroot;
	unsigned int i, j;

	if (!(isroot = (np = par->child) == n))
		while (1)
#ifdef BUG_TRAP
			if (!np) assert (((void)"Remove something not in the tree!!!!!"));
			else
#endif
			if (strcmp (np->node_name, cp) > 0)
				if (np->less == n) break;
				else np = np->less;
			else
				if (np->more == n) break;
				else np = np->more;
	else;

	if (!(n->less && n->more))
	{
		if (isroot)
			par->child = (n->less) ? n->less : n->more;
		else
			if (np->less == n)
				np->less = (n->less) ? n->less : n->more;
			else
				np->more = (n->less) ? n->less : n->more;
		return;
	}

	for (i = 0, nlp = NULL, nl = n->less; nl->more; i++)
		nlp = nl, nl = nl->more;
	for (j = 0, nrp = NULL, nr = n->more; nr->less; j++)
		nrp = nr, nr = nr->less;

	if (i >= j)
	{
		if (isroot) par->child = nl;
		else
			if (np->less == n) np->less = nl;
			else np->more = nl;
		if (nlp)
		{
			nlp->more = nl->less;
			nl->less = n->less;
		}
		nl->more = n->more;
	}
	else
	{
		if (isroot) par->child = nr;
		else
			if (np->less == n) np->less = nr;
			else np->more = nr;
		if (nrp)
		{
			nrp->less = nr->more;
			nr->more = n->more;
		}
		nr->less = n->less;
	}
}

void dbst_init_insert (node *n)
{
	node *np;
	char *cp = n->node_name;
	int h, i;

	n->less = n->more = NULL;
	if (!(np = n->parent->child))
	{
		n->parent->child = n;
		return;
	}

	for (h = 0;; h++)
	{
		if ((i = strcmp (np->node_name, cp)) == 0)
		{
			fprintf (stderr, "Somebuddy's been messing"
				 " with your group file. Duplicate node"
				 " name detected `%s', I'LL QUIT NOW.\n",
				 cp);
			exit (1);
		}
		if (i > 0)
			if (!np->less) { np->less = n; break; }
			else np = np->less;
		else
			if (!np->more) { np->more = n; break; }
			else np = np->more;
	}

	if (h >= MAGIC_NUMBER)
	{
		fprintf (stderr, "Somebuddy's been messing with your "
			 "group file. I wouldn't leave a tree with height"
			 " %i. I'LL QUIT RIGHT NOW.\n", h);
		exit (1);
	}
}
