/******************************************************************************
		Sample Polymorphic Type implementation.

 cistr - case insensitive ASCII string types.

******************************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "../global.h"
#include "../polytypes.h"
#ifdef USE_SFMALLOC
#include "../sfmalloc.h"
#endif

#ifdef USE_SFMALLOC
 static SFMPOOL *elsm;
 #define allocelist(x) x = (elist*) sfmalloc (elsm)
 #define freeelist(x) sfmalloc_free (elsm, x)
#else
 #define allocelist(x) x = (elist*) malloc (sizeof (elist))
 #define freeelist(x) free (x)
#endif

char CISTR_CODE [] = "cistr";
char CISTR_FQNAME [] = "Case Insensitive ASCII Strings";

PolymorphicType cistr;

/*
 * Rationale.
 *  Polymorphic type cistr is case insensitive string comparisons.
 *  Therefore the string 'foo' may have 8 different but equal forms,
 *  'Foo', 'FOO', 'fOo', etc...
 *  Our goal is: given one of it's forms locate any matches within
 *  Lndbase elements.
 *
 * Case insensitive hashing is used. The search is linear.
 */
typedef struct telist elist;
struct telist {
	Element *e;
	elist *next;
};

static elist *cistr_hash [256];
static unsigned int cnt;

/*
 * case insensitive hash signature.
 */
static unsigned int cistrhash (register char *c, register unsigned int i)
{
	register unsigned char cc;
	unsigned int hash = 0;

	for (hash = 0; i > 0; i--)
	{
		cc = *c++;
		hash ^= (cc >= 'A' && cc <= 'Z') ? cc + 32 : cc;
	}
	return hash;
}

void cistr_init ()
{
	int i;

	for (i = 0; i < sizeof (cistr_hash); i++)
		cistr_hash [i] = NULL;
#ifdef USE_SFMALLOC
	elsm = sfmalloc_creat (sizeof (elist), 7);
#endif
	cnt = 0;
}

/*
 * Locate case insensitive matches. The search is hashed but linear.
 * Let us not freak yet.
 * The options are not used.
 */
unsigned int *cistr_locate (char *c, unsigned int i, char **xx __attribute__ ((unused)))
{
	struct match {
		unsigned int el;
		struct match *next;
	} *first = NULL, *cur = NULL;
	unsigned int h, *rp;
	elist *el;

	if (!(el = cistr_hash [h = cistrhash (c, i)]))
		return NULL;

	while (el && el->e->len < i)
		el = el->next;

	for (h = 0; el && el->e->len == i; el = el->next)
		if (strncasecmp (c, el->e->data, i) == 0)
		{
			if (h)
			{
				cur->next = (struct match*)
				 alloca (sizeof (struct match));
				cur = cur->next;
			}
			else
				cur = first = (struct match*)
				 alloca (sizeof (struct match));
			cur->el = el->e->id;
			++h;
		}

	if (!h) return NULL;

	/*
	 * Our matches are a linked list. Convert the list to an array.
	 * Caller is responsible to free this array.
	 */
	rp = (unsigned int*) malloc (sizeof (int) * (h + 1));
	for (cur = first, i = 0; i < h; i++, cur = cur->next)
		rp [i] = cur->el;
	rp [i] = FREE_ID;

	return rp;
}

/*
 * The current organization is a single linked list.
 * The elements in the list are sorted by length. Not exactly sure why..
 * Normal sllist insertion.
 */
int cistr_reg (Element *e)
{
	elist *el, *ee, *ep;
	unsigned int h, i = e->len;

	if (e->len >= BIG_ELEMENT)
		return 0;
	GUARD (allocelist(ee))
	ee->e = e;
	el = cistr_hash [h = cistrhash (e->data, i)];
	if (!el || i <= el->e->len)
	{
		ee->next = el;
		cistr_hash [h] = ee;
	}
	else
	{
		for (ep = el, el = el->next;
		     el && el->e->len < i;
		     ep = el, el = el->next);
		ep->next = ee;
		ee->next = el;
	}
	++cnt;

	return 1;
}

void cistr_unreg (Element *e)
{
	elist *el, *ep;
	unsigned int h;

	el = cistr_hash [h = cistrhash (e->data, e->len)];

	if (el->e == e)
		cistr_hash [h] = el->next;
	else
		for (ep = el;; ep = el)
		{
			el = ep->next;
#ifdef BUG_TRAP
			assert (el);
#endif
			if (el->e == e)
			{
				ep->next = el->next;
				break;
			}
		}

	--cnt;
	freeelist(el);
}

unsigned int cistr_rusage ()
{
	return cnt * sizeof (elist);
}

PolymorphicType cistr = {
	cistr_init, CISTR_CODE, CISTR_FQNAME,
	cistr_locate, cistr_reg, cistr_unreg,
	cistr_rusage
};
