/*****************************************************************************
	init.c

	All the things that should be initialized.
	Also all the things done at termination.

*****************************************************************************/

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <string.h>
#include <signal.h>
#include "kernel/global.h"
#include "auth/auth.h"

#ifndef MAP_FAILED
#define MAP_FAILED ((void*) -1)
#endif

extern char *mmaped;
extern unsigned int ElTop, ElArrTop;
extern char *lndb_home;
extern void init_polytypes (void);
extern int read_only, max_clients;
extern void init_sync (), term_sync ();
extern void (*terminator) (void);

time_t start_time;
pid_t main_pid;
static off_t Size;
static char *lockfile;

#define FNACREAT(x, y, alloc)\
	x = (char*) alloc (strlen (lndb_home) + strlen (y) + 1);\
	strcat (strcpy (x, lndb_home), y)

#define LOCK_FILE "locked"

void term ()
{
	char tmp [100];

	terminator ();
	if (!read_only)
	{
		save_all ();
		unlink (lockfile);
	}
	munmap (mmaped, Size);
	term_sync ();
	system ("rm *");
	getcwd (tmp, 100);
	chdir ("..");
	rmdir (tmp);

	// It happened once. term() was sequentially called and it
	// climbed the directory hicheracy removing files on it's
	// way up to the root. Let'us be safe.
	// And I can tell you another story, It was then in Korea....
	exit (0);
}

void sigterm (int x)
{
	term ();
	exit (0);
}

/*
 * This is not perfect.
 * Normally there should be a LNDB_init() in lndb_calls.[ch] and
 * an interface init() calling LNDB_init ().
 *
 * Anyway these two are mixed in one init() below.
 *
 * To use Lndbase as a library and not an interfaced server, the
 * two inits() will have to be separated.
 */

void init (int NewDatabase, int thresh, unsigned int d, char *rpat)
{
	extern char *user_pass;
	char tmp [100], *Wazoo;
	FILE *f;
	int fds;
	unsigned int nulllen = 0;
	struct stat sbuf;

	start_time = time (NULL);
	umask (077);

	/*
	 * Check/create lockfile
	 */
	if (!read_only)
	{
		FNACREAT(lockfile, LOCK_FILE, malloc);
		if (access (lockfile, F_OK) == 0)
		{
			fprintf (stderr, "Lockfile %s exists.\n"
				"Another lndbase image is running on the same"
				" datafiles?", lockfile);
			exit (1);
		}
		if ((fds = open (lockfile, O_CREAT | O_WRONLY, 0600)) == -1)
		{
			perror ("Creating lockfile");
			exit (1);
		}
		write (fds, "X", 1);
		close (fds);
	}

	/*
	 * Create Temporary directory
	 */
	sprintf (tmp, TMP_PREFIX, getpid ());
	FNACREAT(Wazoo, tmp, alloca);
	if (access (Wazoo, F_OK) == 0)
	{
		fprintf (stderr, "Temporary directory %s already there"
				 ". Should it be removed ?", Wazoo);
		exit (1);
	}
	if (mkdir (Wazoo, S_IRUSR + S_IWUSR + S_IXUSR))
	{
		fprintf (stderr, "Can't create %s\n", Wazoo);
		exit (1);
	}
	chdir (Wazoo);

	/*
	 * Initialize passwords
	 */
	if (user_pass) init_auth (user_pass);

	/*
	 * Produce signature readme in the temporary directory
	 */
	if (!(f = fopen (TMP_INFO, "w")))
	{
		perror ("While opening " TMP_INFO );
		exit (1);
	}
	fputs ("Lndbase " VERSION "\n", f);
	fprintf (f, "Started at :%s by %u (pid %u)\n",
		 ctime (&start_time), getuid (), getpid ());
	fclose (f);

	/*
	 * Create datafile, groupfile, linkfile from lndb_home.
	 */
	FNACREAT(datafile,  DATA_EL, malloc);
	FNACREAT(groupfile, DATA_GR, malloc);
	FNACREAT(linkfile,  DATA_LN, malloc);
	if (access (datafile, F_OK) == -1)
	{
		fprintf (stderr, "No Datafiles found (%s).\n"
			"A new database will be created\n", datafile);
		NewDatabase = 1;
	}

	if (!NewDatabase)
	{
		/*
		 * Create a link from the master datafile.
		 * This way one can safely change the master datafile.
		 */
		if (link (datafile, DATAFILE_LINK) == -1)
		{
			perror ("Fatal");
			exit (1);
		}

		/*
		 * MMAP the enitre datafile.
		 */
		if ((fds = open (DATAFILE_LINK, O_RDONLY)) == -1)
		{
			perror ("Fatal");
			exit (1);
		}
		fstat (fds, &sbuf);
		if ((mmaped = mmap (0, Size = sbuf.st_size, PROT_READ,
				    MAP_PRIVATE, fds, 0)) == MAP_FAILED)
		{
			perror ("mmap");
			exit (1);
		}
	}
	else
		mmaped = (char*) &nulllen;

#ifdef USE_SFMALLOC
	/*
	 * sfmalloc initialization if used..
	 */
	init_sfmalloc ();
#endif

	/*
	 * Initial allocation of `element' array.
	 */
	ElArrTop = 100 + *(unsigned int*)mmaped + d;
	element = (Element**) malloc (sizeof (Element*) * ElArrTop);
	Elements = ElTop = 0;

	/*
	 * Initialize polytypes
	 */
	init_polytypes ();

	/*
	 * Initialize thread synchronization scheme.
	 */
	init_sync ();

	/*
	 * Load the database.
	 */
	if (!NewDatabase)
		load_all (thresh, d, rpat);

	/*
	 * Install SIGINT handler. Any thread may send this signal to
	 * the main_pid to shutdown.
	 */
	main_pid = getpid ();
	signal (SIGINT, sigterm);
}
