/*****************************************************************************

	client.c

	Basic client handling. Each client has two FILE* streams, one
	to get commands and one to reply back. The implementation is
	thread safe.

*****************************************************************************/

#include <sys/poll.h>
#include <sys/types.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include "sync.h"
#include "lndb_calls.h"
#include "ret.h"
#include "client.h"
#include "kernel/config.h"
#include "auth/auth.h"

#define CMD_NUM sizeof cmds / sizeof (struct cmd_s)

extern int read_only, max_clients;
extern pid_t main_pid;

const static struct cmd_s {
	char *cmd;
	short int argc; 
	char suecmd, dosync; } cmds [] = {
		"alt",		2, 1, 1,
		"altcl",	2, 1, 0,
		"cd",		0, 0, 1,
		"chgrp",	2, 1, 1,
		"del",		1, 1, 1,
		"dir",		0, 0, 1,
		"find",		1, 0, 1,
		"glinks",	2, 0, 1,
		"groupof",	1, 0, 1,
		"idof",		1, 0, 1,
		"idofcl",	1, 0, 0,
		"lenof",	1, 0, 1,
		"link",		3, 1, 1,
		"links",	1, 0, 1,
		"locate",	3, 0, 0,
		"ls",		0, 0, 1,
		"mkgrp",	1, 1, 1,
		"new",		1, 1, 1,
		"newcl",	1, 1, 0,
		"nlinks",	1, 0, 1,
		"nls",		1, 0, 1,
		"pwd",		0, 0, 1,
		"quit",		0, 0, 0,
		"rengrp",	2, 1, 1,
		"rmgrp",	1, 1, 1,
		"save",		0, 1, 1,
		"set",		2, 1, 1,
		"sh",		1, 0, 1,
		"shutdown",	0, 1, 1,
		"stats",	0, 0, 1,
		"sue",		0, 0, 0,
		"typesof",	1, 0, 1,
		"unlink",	2, 1, 1,
		"unset",	2, 1, 1,
		"vlink",	2, 0, 1
};

const static char *ok_msg [] = {
	" Ok\n",
	" End of Data\n",
	" Not linked\n",
	" No links\n",
	" No subgroups\n",
	" No elements\n",
	" No such element exists\n",
	" No matches\n",
	" Auth\n",
	" Content expected\n"
};

const static char Welcome [] =
	"Lndbase " VERSION "\n"
	"Ready.\n";
 
const static char *bad_msg [] = {
	"",
	" No such group\n",
	" Bad element ID\n",
	" Bad name\n",
	" Already exists\n",
	" Invalid group\n",
	" Not empty\n",
	" Not enough memory\n",
	" Bad command\n",
	" Too few arguments\n",
	" Bad link\n",
	" Bad value for command\n",
	" Line too long\n",
	" Permission denied\n",
	" Not implemented\n",
	" Failed\n",
	" Bad type name\n",
	" Not registered\n",
	" Authentication failed\n",
	" Non removable element ID\n",
	" Not accepted by PMT\n"
};

const static char bye_msg [] = "* 1 Bye\n";

const static char *itoaz [] = {
	"0", "1", "2", "3", "4", "5", "6", "7", "8", "9", "10", "11",
	"12", "13", "14", "15", "16", "17", "18", "19", "20", "21"
};
const static char **itoa = itoaz + 1;

static int cmd_num (char *c)
{
	int i, hi, lo, mid;

	hi = CMD_NUM - 1, lo = 0;
	while (hi >= lo)
	{
		mid = (hi + lo) / 2;
		if ((i = strcmp (c, cmds [mid].cmd)) == 0)
			return mid;
		else if (i > 0)
			lo = mid + 1;
		else
			hi = mid - 1;
	}

	return -1;
}

static char *argize (char *c)
{
	char *p;

	if ((p = strchr (c, ' ')))
	{
		while (*p == ' ') *p++ = 0;
		return p;
	}
	else
		return NULL;
}

static char **free_optl (char **arrv)
{
	char **cp;

	if (!arrv) return NULL;
	for (cp = arrv; *cp; cp++)
		free (cp);
	free (arrv);
	return NULL;
}

/*
 * Get lines of options. currently for `locate'
 * Our goal is to get ALL the lines even if errors occur so the
 * input is clean for the new command.
 */
static char **get_optl (FILE *readf, int n)
{
#define MAX_LINE 80
#define MAX_OPTIONS 40
	char tmp [MAX_LINE];
	char **arrv = (char**) calloc ((n + 1), sizeof (char*));
	int i;
	unsigned char bad, j;

	for (bad = i = 0; i < n; i++)
	{
		if (!fgets (tmp, MAX_LINE, readf))
			{ bad = 1; break; }
		if (i > MAX_OPTIONS) bad = 1;
		if ((j = strlen (tmp)) == 0)
			bad = 1;
		else if (tmp [j - 1] != '\n')
		{
			bad = 1;
			do fgets (tmp, MAX_LINE, readf);
			while (strlen (tmp) > 0
			       && tmp [strlen (tmp) - 1] != '\n');
		}
		else if (!bad)
		{
			arrv [j - 1] = 0;
			arrv [i] = (char*) malloc (j);
			strcpy (arrv [i], tmp);
		}
	}

	return (bad) ? free_optl (arrv) : arrv;
}

/*
 * Use fread with timeout. That's very important when our stream is
 * a fdopened socket.
 */
static int fread_timeout (char *t, unsigned int cl, FILE *inf, time_t msec)
{
	int offset = 0, i;
	struct pollfd pfd = { fileno (inf), POLLIN + POLLHUP, 0 };

	fcntl (fileno (inf), F_SETFL, O_NONBLOCK);
	do {
		i = fread (t + offset, 1, cl - offset, inf);
		if ((offset += i) == cl)
		{
			fcntl (fileno (inf), F_SETFL,
				fcntl (fileno (inf), F_GETFD) & ~O_NONBLOCK);
			return 0;
		}
	} while ((i = poll (&pfd, 1, msec)) > 0
		 && !(pfd.revents & POLLHUP));

	return 1;
}

static void econt (FILE *reply)
{
	fputs ("* ", reply);
	fputs (itoa [R_CONTENT], reply);
	fputs (ok_msg [R_CONTENT], reply);
	fflush (reply);
}

#define STRNUM(x, y) \
	x = strtol (y, &endptr, 10);\
	if (*endptr != 0)\
	{\
		r = EBADVALUE;\
		break;\
	}

int perform (int cmd, char *c, client_params *cpar)
{
	char *endptr;
	char *cp = NULL, *cpp = NULL;
	char **arrv;
	int r, rr, rq;
#ifdef MULTITHREADED
	int action;
	struct sembuf sem_operation = { 0, 0, SEM_UNDO };
#endif

	if (cmds [cmd].suecmd && (!cpar->sue || (cmd != 25 && (read_only))))
		return ENOTSUE;

	if (cmds [cmd].argc != 0 && !c)
		return EBADVALUE;

	if (cmds [cmd].argc >= 2)
	{
		cp = argize (c);
		if (!cp)
			return EBADVALUE;
		if (cmds [cmd].argc == 3)
			cpp = argize (cp);
	}

#ifdef MULTITHREADED
	action = (cmds [cmd].dosync) ? cmds [cmd].suecmd : -1;

	if (action == 0)
		SYNC_SR
	else if (action == 1)
		SYNC_RTW
#endif

	switch (cmd) {
	case 0:
		STRNUM(r, c);
		if (!cp)
			return ENOARGS;
		if (!(cpp = (char *) malloc (rr = strlen (cp))))
			return EOUTOF;
		strncpy (cpp, cp, rr);
		r = LNDB_alt (r, cpp, rr, cpar);
		if (r != R_OKey) free (cpp);
		break;
	case 1:
		STRNUM(r, c);
		STRNUM(rr, cp);
		if (rr < 0 || rr > MAX_ELEMENT_SIZE)
			return EBADVALUE;
		cp = (char *) malloc (rr);
		if (!cp) return EOUTOF;
		econt (cpar->reply);
		if (!fread_timeout (cp, rr, cpar->readf, CLIENT_TIMEOUT))
		{
			fgetc (cpar->readf);
			SYNC_RTW
			r = LNDB_alt (r, cp, rr, cpar);
			SYNC_WF
		}
		else r = ECLIENT;
		if (r != R_OKey) free (cp);
		break;
	case 2:
		r = LNDB_cd (c, cpar);
		break;
	case 3:
		STRNUM(r, c);
		r = LNDB_chgrp (r, cp, cpar);
		break;
	case 4:
		STRNUM(r, c);
		r = LNDB_del (r, cpar);
		break;
	case 5:
		r = LNDB_dir (c, cpar);
		break;
	case 6:
		r = LNDB_find (c, cpar);
		break;
	case 7:
		STRNUM(r, c);
		r = LNDB_glinks (r, cp, cpar);
		break;
	case 8:
		STRNUM(r, c);
		r = LNDB_groupof (r, cpar);
		break;
	case 9:
		if (!c)
			return ENOARGS;
		r = LNDB_idof (c, strlen (c), cpar);
		break;
	case 10:
		STRNUM(r, c);
		if (r < 0)
			return EBADVALUE;
		cp = (char *) alloca (r);
		if (!cp) return EOUTOF;
		econt (cpar->reply);
		if (!fread_timeout (cp, r, cpar->readf, CLIENT_TIMEOUT))
		{
			fgetc (cpar->readf);
			SYNC_SR
			r = LNDB_idof (cp, r, cpar);
			SYNC_RF
		}
		else r = EFAIL;
		break;
	case 11:
		STRNUM(r, c);
		r = LNDB_lenof (r, cpar);
		break;
	case 12:
		STRNUM(r, c);
		STRNUM(rr, cp);
		if (cpp)
			{ STRNUM (rq, cpp); }
		else rq = 0;
		r = LNDB_link (r, rr, rq, cpar);
		break;
	case 13:
		STRNUM(r, c);
		r = LNDB_links (r, cpar);
		break;
	case 14:
		STRNUM(r, c);
		STRNUM(rr, cp);
		if (r < 0 || rr < 0)
			return EBADVALUE;
		// Get option lines. NULL if linetoolong or other error.
		arrv = get_optl (cpar->readf, r);
		cp = (char *) alloca (rr);
		if (!cp) return EOUTOF;
		econt (cpar->reply);
		if (!(fread_timeout (cp, rr, cpar->readf, CLIENT_TIMEOUT) == 1
		     || arrv == NULL))
		{
			fgetc (cpar->readf);
			SYNC_SR
			r = LNDB_locate (arrv, cp, rr, cpp, cpar);
			SYNC_RF
		}
		else
			r = EFAIL;
		free_optl (arrv);
		break;
	case 15:
		r = LNDB_ls (c, cpar);
		break;
	case 16:
		r = LNDB_mkgrp (c, cpar);
		break;
	case 17:
		if (!c)
			return ENOARGS;
		if (!(cp = (char *) malloc (rr = strlen (c))))
			return EOUTOF;
		strncpy (cp, c, rr);
		r = LNDB_new (cp, rr, cpar);
		if (r != R_OKey) free (cp);
		break;
	case 18:
		STRNUM(r, c);
		if (r < 0 || r > MAX_ELEMENT_SIZE)
			return EBADVALUE;
		cp = (char *) malloc (r);
		if (!cp) return EOUTOF;
		econt (cpar->reply);
		if (!fread_timeout (cp, r, cpar->readf, CLIENT_TIMEOUT))
		{
			fgetc (cpar->readf);
			SYNC_RTW
			r = LNDB_new (cp, r, cpar);
			SYNC_WF
		}
		else r = ECLIENT;
		if (r != R_OKey) free (cp);
		break;
	case 19:
		STRNUM(r, c);
		r = LNDB_nlinks (r, cpar);
		break;
	case 20:
		r = LNDB_nls (c, cpar);
		break;
	case 21:
		r = LNDB_pwd (cpar);
		break;
	case 23:
		r = LNDB_rengrp (c, cp, cpar);
		break;
	case 24:
		r = LNDB_rmgrp (c, cpar);
		break;
	case 26:
		STRNUM(r, c);
		r = LNDB_set (r, cp, cpar);
		break;
	case 27:
		STRNUM(r, c);
		r = LNDB_sh (r, cpar);
		break;
	case 29:
		r = LNDB_stats (cpar);
		break;
	case 30:
		r = (cpar->sue) ? R_OKey : 
		    (auth (1, cpar->readf, cpar->reply)) ? R_OKey : EAUTHFAIL;
		if (r == R_OKey) cpar->sue = 1;
		break;
	case 31:
		STRNUM(r, c);
		r = LNDB_typesof (r, cpar);
		break;
	case 32:
		STRNUM(r, c);
		STRNUM(rr, cp);
		r = LNDB_unlink (r, rr, cpar);
		break;
	case 33:
		STRNUM(r, c);
		r = LNDB_unset (r, cp, cpar);
		break;
	case 28: /* shutdown */
		kill (main_pid, SIGINT);
		for (;;);
	case 22: /* quit */
		return QUIT;
	case 25: /* save */
		r = LNDB_save (cpar);
		break;
	case 34:
		STRNUM(r, c);
		STRNUM(rr, cp);
		r = LNDB_vlink (r, rr, cpar);
		break;
	default:
		r = ENOTIMPL;
	}

#ifdef MULTITHREADED
	if (action == 0)
		SYNC_RF
	else if (action == 1)
		SYNC_WF
#endif

	return r;
}

#define SUCCESS_MSG(x) \
	fputs ("* ", cpar.reply);\
	fputs (itoa [x], cpar.reply);\
	fputs (ok_msg [x], cpar.reply);\
	fflush (cpar.reply)

#define ERROR_MSG(x) \
	fputs ("- ", cpar.reply);\
	fputs (itoaz [-x], cpar.reply);\
	fputs (bad_msg [-x], cpar.reply);\
	fflush (cpar.reply)

client_params *first_client, *last_client;

/*
 * Such a function is running for every client in the multithreaded
 * case. Thus the passing of all those arguments each time..
 */
void *client (struct thread_params *thp)
{
#define LBUFLEN 100
	extern unsigned int clients;
	extern void (*closedown) (FILE*, FILE*);
	char line_buf [LBUFLEN + 1];
	int i;
	char *cp;
#ifdef MULTITHREADED
	struct sembuf sem_operation = { 0, 0, 0 };
#endif
	client_params cpar = { &root, thp->sue, thp->reply, thp->readf,
			       last_client, NULL };

	if (!last_client) first_client = &cpar;
	else last_client->next = &cpar;
	last_client = &cpar;

	SYNC_CF
	sem_operation.sem_flg = SEM_UNDO;

	signal (SIGPIPE, SIG_IGN);

	fputs (Welcome, cpar.reply);
	SUCCESS_MSG (R_EOD);

	while (1)
	{
		if (!fgets (line_buf, LBUFLEN, cpar.readf))
			break;

		i = strlen (line_buf);
		if (i == 0 || (i == LBUFLEN 
		&& line_buf [i - 1] != '\n'))
		{
			ERROR_MSG (ELONGLINE);
			fgets (line_buf, LBUFLEN, cpar.readf);
			continue;
		}
		line_buf [i - 1] = 0;
		cp = argize (line_buf);
		if ((i = cmd_num (line_buf)) == -1)
		{
			ERROR_MSG (EBADCMD);
			continue;
		}

		i = perform (i, cp, &cpar);
		if (i == ECLIENT || i == QUIT)
			break;
#ifdef BUG_TRAP
		else if (i < QUIT || i > R_PASSWORD)
			fprintf (stderr, "What's this %i\n", i);
#endif
		else if (i < 0) {
			ERROR_MSG (i);
		} else {
			SUCCESS_MSG (i);
		}
	}

	fputs (bye_msg, cpar.reply);
	fflush (cpar.reply);

	SYNC_RTW
	--clients;
	if (cpar.next) cpar.next->prev = cpar.prev;
	else last_client = cpar.prev;
	if (cpar.prev) cpar.prev->next = cpar.next;
	else first_client = cpar.next;
	SYNC_WF
fprintf (stderr, "Client bye bye\n");
	closedown (cpar.readf, cpar.reply);

	return NULL;
}
