/******************************************************************************

	Lndbase shell

	Implements:
	  Group tree were each group name is stored only once
	  Element Data/Datasize/isbinary cache
	  Element structure -- epil object (#) -- auto removal when unused
	  S lists -- epil Object (#)
	  S Looper class
	  Inter-structure element removal (VoidID)
	  C interface to lnshell
	  coded macros of this unit
	  / gateway

******************************************************************************/
/**************************************************************************
 Copyright (C) 2000 Stelios Xantkakis
**************************************************************************/

#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>

#include "projectlog.h"

#include "iServ.h"

#include "epil.h"

#include "dLIST.h"
#include "arch.h"
#include "dbstree.h"
#include "inttree.h"
#include "sysutil.h"
#include "sfFetch.h"

#include "lnshell.h"

//***********************************************************
// epilException generated from the current reply status.
//***********************************************************

static char servererr [] = "Server Error:";

void ServerError ()
{
	Reply *R = iCommand::CurrentReply;

	throw epilException (servererr, R->StatusLine);
}

//***********************************************************
// Groups are transparent to the interface. 
// Eventually we copy the entire group tree from the server
// but only as groups are needed.
//***********************************************************

static dbsTree GroupTree;

class GroupNode : public dbsNodeStr
{
   public:
	GroupNode ();
	void teleport (char*);
	~GroupNode ();
};

GroupNode::GroupNode () : dbsNodeStr (&GroupTree)
{
	Name = StrDup (DBS_STRQUERY);
}

void GroupNode::teleport (char *n)
{
	GroupTree.dbsRemove (this);
	DBS_STRQUERY = n;

	assert (!GroupTree.dbsFind ());

	delete Name;
	Name = StrDup (n);

	addself (&GroupTree);
}

GroupNode::~GroupNode ()
{
	GroupTree.dbsRemove (this);
	delete Name;
}

void renGroupNotify (char *o, char *n)
{
	DBS_STRQUERY = o;

	GroupNode *g = (GroupNode*) GroupTree.dbsFind ();

	if (g) g->teleport (n);
}

static GroupNode *GroupNodeFor (char *str)
{
	DBS_STRQUERY = str;

	GroupNode *g = (GroupNode*) GroupTree.dbsFind ();

	if (!g) return new GroupNode;

	return g;
}

//***********************************************************
// Cache.
//***********************************************************

#define UNKNOWN 0
#define TEXT	1
#define BIN	2

static intTree CacheTree;
static dlist CacheList, RList;

class Cache : public intNode, public dlistNodeAuto
{
	void getData ();
	char *data;
	int isbinary;
   public:
	unsigned int len;
	Cache ();
	Cache (unsigned int);
static	unsigned int CacheSize, CacheUse;
	void noCache ();
	char *Data ();
	bool isBin ();
	~Cache ();
friend	void RemoveCache (unsigned int);
};

unsigned int Cache::CacheUse, Cache::CacheSize = 4096;

void Cache::noCache ()
{
	if (data) delete data;
	data = NULL;
	CacheUse -= len;
	CacheList.dremove (this);
	RList.addtostart (this);
}

static void AdjustSize (unsigned int want)
{
	while (Cache::CacheUse > Cache::CacheSize 
	       || want > Cache::CacheSize - Cache::CacheUse)
		if (CacheList.End)
			((Cache*) CacheList.End)->noCache ();
		else break;
};

static void ClearCache ()
{
	while (RList.Start) {
		CacheList.addtoend (RList.Start);
		RList.dremove (RList.Start);
	}
	while (CacheList.Start)
		delete CacheList.Start;
}

void Cache::getData ()
{
	unsigned int i = 0;
	shReply sR;

	Ln_Sh (Key);
	if (!sR.Status ()) {
		VoidID (Key);
		ServerError ();
	}

	data = sR.Data;
	sR.Data = NULL;

	len = sR.ContentLength;
	AdjustSize ((len < CacheSize / 5) ? len : 0);

	CacheUse += len;
	
	if (len <= CacheSize / 6) {
		CacheList.dremove (this);
		CacheList.addtoend (this);
	}

	isbinary = TEXT;
	for (i = 0; i < len; i++)
		if (ISBIN (data [i])) {
			isbinary = BIN;
			break;
		}
}

Cache::Cache () : intNode (&CacheTree), dlistNodeAuto (&CacheList, 0)
{
	getData ();
}

Cache::Cache (unsigned int i) : intNode (&CacheTree), dlistNodeAuto (&RList, 0)
{
	isbinary = UNKNOWN;
	len = i;
	data = NULL;
}

char *Cache::Data ()
{
	if (!data) {
		RList.dremove (this);
		CacheList.addtostart (this);
		getData ();
	} else {
		CacheList.dremove (this);
		if (len < CacheSize / 6) CacheList.addtostart (this);
		else CacheList.addtoend (this);
	}

	return data;
}

bool Cache::isBin ()
{
	if (isbinary == UNKNOWN) Data ();
	return isbinary == BIN;
}

Cache::~Cache ()
{
	if (data) {
		delete data;
		CacheUse -= len;
		leave (&CacheList);
	} else leave (&RList);

	intRemove (&CacheTree);
}

// *---- iFace

static Cache *FindCached (unsigned int ID)
{
	return (Cache*) CacheTree.intFind (ID);
}

static Cache *getCache (unsigned int ID)
{
	Cache *c = FindCached (ID);

	if (c) return c;
	else return new Cache;
}

static Cache *getLenCache (unsigned int ID)
{
	Cache *c = FindCached (ID);

	if (c) return c;

	LineReply LR;

	Ln_Lenof (ID);
	if (!LR.Status ()) {
		VoidID (ID);
		ServerError ();
	}

	FETCH_LREPLY(&LR);

	return new Cache (strtoul (LR.DataLine, NULL, 10));
}

void RemoveCache (unsigned int ID) /* can't friends be static???? */
{
	Cache *c = FindCached (ID);

	if (c) {
		if (!c->data) {
			RList.dremove (c);
			CacheList.addtoend (c);
		}
		delete c;
	}
}

//***********************************************************
// E, EP, EPLid, S	: class declarations
// links caching can be easilly implemented, by attaching
// an S* to each E object, cleared on every :link, :unlink.
// For now the server is either at the local machine or at
// a fast LAN.
//***********************************************************

class S;

class E : public intNode, public SurfaceObject
{
	void init ();
   public:
	E ();
	GroupNode *G;
	int cnt;
	Cache *D;
	~E ();

	void activate ();
};

class EP : public intNode, public dlistNode
{
   public:
static	S *myS;
	E *e;
	EP (E*, intTree*);
virtual ~EP ();
};

class EPLid : public EP
{
   public:
	short int Lid;
	EPLid (E*, intTree*, short int = 0);
	~EPLid ()	{ }
};

class S : public dlistNodeAuto, public SurfaceObject
{
friend	class EP;
friend	class SLooper;
	intTree EPTree;
	dlist EPList;
	void init ();
	int pSID;
static	int SID;
   public:
	E *Originator;
	S ();
	S (S*);
	S (MultiLineReply&);
	S (MultiLineReply&, E*);
	S (MultiLineReply&, char*);
	S (MultiLineReply&, E*, char*);
	S (char*);
	EP *FindE (E*);
	E *operator [] (unsigned int);
	void Remove (E*);
	void Append (E*);
	int nelements;
	int type;
	char *Name;
	~S ();

	void activate ();
};

//***********************************************************
// Elements have the count of lists including them.
// cleanETree will remove all elements of cnt 0.
//***********************************************************

static intTree ETree;

E *CurrentElement, *SaveCE;

E::E () : intNode (&ETree)
{
	G = NULL, D = NULL;
	cnt = 0;
	char myname [20];
	sprintf (myname, "LELEM%u", Key);
	appear (myname);
}

static inline void ReleaseE (E *e)
{
	if (e->cnt) --e->cnt;
}

void E::activate ()
{
	CurrentElement = this;
}

E::~E ()
{
	if (CurrentElement == this) CurrentElement = NULL;
	if (SaveCE == this) SaveCE = NULL;
	intRemove (&ETree);
}

static E *EFor (unsigned int ID)
{
	E *e = (E*) ETree.intFind (ID);

	return (e) ? e : new E;
}

static E *EFor (unsigned int ID, GroupNode *g)
{
	E *e = EFor (ID);

	if (!e->G || e->G != g) e->G = g;

	return e;
}

static E *EFor (char *l)
{
	char *c = strchr (l, ':');

	if (c) {
		*(c - 1) = 0;

		GroupNode *g = GroupNodeFor (l);
		E *e = EFor (strtoul (c + 3, NULL, 10), g);
		*(c - 1) = ':';
		return e;
	}

	return EFor (strtoul (l, NULL, 10));
}

void RemoveCache (E *e)
{
	if (e->D) {
		RemoveCache (e->Key);
		e->D = NULL;
	}
}

#define ELND(X) ((E*)X)

void rcleantree (intNode *i)
{
	while (i->less && ELND(i->less)->cnt == 0)
		delete ELND(i->less);
	while (i->more && ELND(i->more)->cnt == 0)
		delete ELND(i->more);
	if (i->less) rcleantree (i->less);
	if (i->more) rcleantree (i->more);
}

static void coded_cleanETree ()
{
	if (CurrentElement)
		++CurrentElement->cnt;

	while (ETree.root && ELND(ETree.root)->cnt == 0)
		delete ELND(ETree.root);

	rcleantree (ETree.root);

	if (CurrentElement)
		--CurrentElement->cnt;
}

//***********************************************************
// The S lists. An S lists holds E elements in a list not
// sortable. It provides log searching by holding the
// elements in a dbstree as well.
//***********************************************************

S *EP::myS;

dlist SList;

S *CurrentList;

int S::SID;

static dlist LooperList;

//::::::: EP

EP::EP (E *ie, intTree *t) : intNode (t)
{
	myS->EPList.addtoend (this);
	e = ie;
	++e->cnt;
	++myS->nelements;
}

EP::~EP ()
{
	myS->EPList.dremove (this);
	ReleaseE (e);
	--myS->nelements;

	intRemove (&myS->EPTree);
}

EPLid::EPLid (E *ie, intTree *t, short int i) : EP (ie, t)
{
	Lid = i;
}

//::::::: SLooper

SLooper::SLooper (S *is) : dlistNodeAuto (&LooperList)
{
	s = is;
	ep = (EP*) s->EPList.Start;
}

void SLooper::Reset ()
{
	ep = (EP*) s->EPList.Start;
}

E *SLooper::Next ()
{
	if (!ep) return NULL;

	E *e = ep->e;
	if (s->Originator)
		Lid = ((EPLid*)ep)->Lid;

	ep = (EP*) s->EPList.Next (ep);

	return e;
}

void SLooper::RE (EP *cep)
{
	if (ep && ep == cep) Next ();
}

SLooper::~SLooper ()
{
	leave (&LooperList);
}

//::::::: S

EP *S::FindE (E *e)
{
	EP::myS = this;

	return (EP*) EPTree.intFind (e->Key);
}

void S::init ()
{
	Logprintf (">>>New list with ID : %u<<<\n", pSID = SID++);

	char t [20];
	sprintf (t, "LIST%u", pSID);
	appear (t);

	Name = StrDup ("Anonymous List");

	nelements = 0;
	Originator = NULL;
	CurrentList = EP::myS = this;
}

S::S () : dlistNodeAuto (&SList)
{
	type = STYPE_OTHER;
	init ();
}

S::S (S *s) : dlistNodeAuto (&SList)
{
	type = s->type;

	init ();

	if ((Originator = s->Originator))
		++Originator->cnt;

	SLooper L (s);
	E *e;

	while ((e = L.Next ())) {
		FindE (e);	// to prepare the dbstree for entry
		if (Originator) new EPLid (e, &EPTree, L.Lid);
		else new EP (e, &EPTree);
	}
}

S::S (MultiLineReply &MLR) : dlistNodeAuto (&SList)
{
	type = STYPE_OTHER;
	init ();

	for (int i = 0; i < MLR.Lines; i++) {
		E *e = EFor (strtoul (MLR.DataLine [i], NULL, 10));
		FindE (e);
		new EP (e, &EPTree);
	}
}

static short int MLRLid (char *m)
{
	m = strchr (m, '/');
	return (m) ? strtol (m + 1, NULL, 10) : 0;
}

S::S (MultiLineReply &MLR, E *e) : dlistNodeAuto (&SList)
{
	type = STYPE_LINKS;
	init ();

	Originator = e;
	++e->cnt;

	for (int i = 0; i < MLR.Lines; i++) {
		E *e = EFor (MLR.DataLine [i]);
		FindE (e);
		new EPLid (e, &EPTree, MLRLid (MLR.DataLine [i]));
	}
}

S::S (MultiLineReply &MLR, char *g) : dlistNodeAuto (&SList)
{
	type = STYPE_DIR;
	init ();

	if (MLR.Lines > 0) {
		GroupNode *G = GroupNodeFor (g);

		for (int i = 0; i < MLR.Lines; i++) {
			E *e = EFor (strtoul (MLR.DataLine [i], NULL, 10), G);
			FindE (e);
			new EP (e, &EPTree);
		}
	}
}

S::S (MultiLineReply &MLR, E *e, char *g) : dlistNodeAuto (&SList)
{
	type = STYPE_PLINKS;
	init ();

	Originator = e;
	++e->cnt;

	if (MLR.Lines > 0) {
		GroupNode *G = GroupNodeFor (g);

		for (int i = 0; i < MLR.Lines; i++) {
			E *e = EFor (strtoul (MLR.DataLine [i], NULL, 10), G);
			FindE (e);
			new EPLid (e, &EPTree, MLRLid (MLR.DataLine [i]));
		}
	}
}

S::S (char *sn) : dlistNodeAuto (&SList)
{
	type = STYPE_OTHER;
	appear (sn);
	Name = StrDup (sn);
	nelements = 0;
	Originator = NULL;
	CurrentList = this;
}

E *S::operator [] (unsigned int i)
{
	EP *tLoop = (EP*) EPList.Start;

	EP::myS = this;

	while (i-- && tLoop) tLoop = (EP*) EPList.Next (tLoop);

	return (tLoop) ? tLoop->e : NULL;
}

void S::Remove (E *e)
{
	if (type == STYPE_LINKS) type = STYPE_PLINKS;

	EP *ep = FindE (e);

	if (ep) {
		for (SLooper *L = (SLooper*) LooperList.Start; L;
		     L = (SLooper*) LooperList.Next (L))
			if (L->s == this) L->RE (ep);
		delete ep;
	}
}

void S::Append (E *e)
{
	if (FindE (e)) return;

	type = STYPE_OTHER;

	if (Originator) {
		ReleaseE (Originator);
		Originator = NULL;
	}

	new EP (e, &EPTree);
}

S::~S ()
{
	if (Originator)
		ReleaseE (Originator);
	EP::myS = this;
	while (nelements) delete ((EP*) EPList.Start);

	leave (&SList);
	delete Name;
}

void S::activate ()
{
	CurrentList = this;
}

//***********************************************************
// Crucial stuff: VoidID
// The element with ID does no longer exist in the database.
// An E can be found:
//	At the current element
//	At the Originator of a list
//	Inside a list
// remove it from all those (by scanning all but this is rarer
// than a special list where the element knows those things).
// Fixup those and remove from the cache.
//***********************************************************

void VoidID (unsigned int ID)
{
	E *e = EFor (ID);

	if (e) {
		int i = e->cnt;

		S *s = (S*) SList.Start;

		while (i) {
			if (s->Originator == e) {
				s->Originator = NULL;
				ReleaseE (e);
				--i;
				continue;
			}
			if (s->FindE (e)) {
				s->Remove (e);
				--i;
			}
			s = (S*) SList.Next (s);
		}
	}

	// may not use E->D
	RemoveCache (ID);
	delete e;
}


//***********************************************************
// lnshell is also available at C level to implement various
// optimized versions of procedures without macros.
// For example \connect could be implemented at epil level
// but a C version will be much more efficient.
//***********************************************************

bool In (S *s, E *e)
{
	return s->FindE (e) != NULL;
}

S *Links (E *e, char *c)
{
	E *a = CurrentElement;
	CurrentElement = e;

	MultiLineReply MLR;

	if (c) Ln_Glinks (CurrentElement->Key, c);
	else Ln_Links (CurrentElement->Key);
	if (!MLR.Status ()) {
		VoidID (CurrentElement->Key);
		ServerError ();
	}

	FETCH_MLREPLY(&MLR);

	if (c) new S (MLR, CurrentElement, c);
	else new S (MLR, CurrentElement);

	CurrentElement = a;
	return CurrentList;
}

int NElements (S *s)
{
	return s->nelements;
}

S *EmptyS (char *name)
{
	return (name) ? new S (name) : new S;
}

unsigned int Idof (E *e)
{
	return e->Key;
}

void Add (S *s, E *e)
{
	s->Append (e);
}

static char loopkill [] = "Requested \\kill on a list used for a loop";

void Kill (S *s)
{
	// A terrible hack. No standard behaviour.
	extern S *Newbies;
	if (s == Newbies) {
		Logputs ("You don't want to delete #Newbies !!\n");
		return;
	}
	//
	for (dlistNode *d = LooperList.Start; d; d = LooperList.Next (d))
		if (((SLooper*)d)->s == s)
			throw epilException (loopkill);

	delete s;
}

E *EIndex (S *s, int i)
{
	return (*s) [i];
}

void makeCurrent (S *s)
{
	s->activate ();
}

E *EForID (unsigned int ID)
{
	return EFor (ID);
}

int DataSize (E *e)
{
	if (!e->D)
		e->D = getLenCache (e->Key);

	return e->D->len;
}

char *GroupOf (E *e)
{
	if (!e->G) {
		LineReply LR;

		Ln_Groupof (e->Key);
		if (!LR.Status ()) {
			VoidID (e->Key);
			ServerError ();
		}

		FETCH_LREPLY (&LR);

		e->G = GroupNodeFor (LR.DataLine);
	}

	return e->G->Name;
}

void chgrpNotify (E *e)
{
	e->G = NULL;
	GroupOf (e);
}

bool IsBin (E *e)
{
	if (!e->D)
		e->D = getCache (e->Key);

	return e->D->isBin ();
}

char *tDataof (E *e)		// text only data not binary!
{
	if (!e->D)
		e->D = getCache (e->Key);

	if (e->D->isBin ()) return NULL;
	else return e->D->Data ();
}

void setCur (E *e)
{
	e->activate ();
}

void sendData (E *e, FILE *f)
{
	if (!e->D)
		e->D = getCache (e->Key);

	fwrite (e->D->Data (), 1, e->D->len, f);
}

S *LsGroup (char *g)
{
	MultiLineReply MLR;

	Ln_Ls (g);
	if (!MLR.Status ()) ServerError ();

	FETCH_MLREPLY(&MLR);

	return new S (MLR, g);
}

E *Find (char *c)
{
	LineReply LR;
	Ln_Idof (c);
	if (LR.StatusCode == 7)
		return NULL;

	FETCH_LREPLY(&LR);

	return EFor (strtoul (LR.DataLine, NULL, 10));
}

S *cFind (char *c)
{
	MultiLineReply MLR;

	Ln_Find (c);
	if (MLR.StatusCode == 8)
		return NULL;

	FETCH_MLREPLY(&MLR);

	return new S (MLR);
}

S *Dup (S *s)
{
	return new S (s);
}

char *SName (S *s)
{
	return s->Name;
}

void SOName (S *s)
{
	s->myName ();
}

//***********************************************************
// epil macros
// These macros mostly work with: CurrentElement, CurrentList
//***********************************************************

//::::: cache

static void coded_setcache ()
{
	char *c;
	TMP_POP_ALLOC(c);

	Cache::CacheSize = strtoul (c, NULL, 10);
	if (Cache::CacheSize < 4096) Cache::CacheSize = 4096;
}

static void coded_clear_cache ()
{
	ClearCache ();
}

//::::: current element

static char nocur [] = "Current Element is not set";

static void check_cur ()
{
	if (!CurrentElement)
		throw epilException (nocur);
}

static void coded_havece ()
{
	ep_boolean = CurrentElement != NULL;
}

static void coded_uncache ()
{
	check_cur ();

	RemoveCache (CurrentElement);
}

//***********************************
// Floating................
static void coded_pmts ()
{
	check_cur ();
	MultiLineReply MLR;
	Ln_Typesof (Idof (CurrentElement));
	if (MLR.Status ()) {
		Logputs ("Not decided how to implement that yet.\n");
		for (int i = 0; i < MLR.Lines; i++)
			Logprintf ("PMT [%s]\n", MLR.DataLine [i]);
	}
}
static void coded_setcistr ()
{
	check_cur ();
	Reply R;
	Ln_Set (Idof (CurrentElement), "cistr");
}
static void coded_nocistr ()
{
	check_cur ();
	Reply R;
	Ln_Unset (Idof (CurrentElement), "cistr");
}
//***********************************

static void coded_ename ()
{
	check_cur ();

	CurrentElement->myName ();
}

static void coded_eid ()
{
	check_cur ();

	char t [10];
	sprintf (t, "%u", CurrentElement->Key);
	ep_push (t);
}

static void coded_group ()
{
	check_cur ();

	ep_push (GroupOf (CurrentElement));
}

static void coded_elen ()
{
	check_cur ();

	char t [10];
	sprintf (t, "%u", DataSize (CurrentElement));
	ep_push (t);
}

static void coded_isbin ()
{
	check_cur ();

	ep_boolean = IsBin (CurrentElement);
}

static void coded_edata ()
{
	check_cur ();

	if (!CurrentElement->D)
		CurrentElement->D = getCache (CurrentElement->Key);

	if (CurrentElement->D->isBin ()) {
		char t [1] = { 0 };
		ep_push (t);
	} else ep_push (CurrentElement->D->Data (), CurrentElement->D->len);
}

static void coded_links ()
{
	check_cur ();

	Links (CurrentElement);
}

static void coded_glinks ()
{
	check_cur ();

	char *c;
	TMP_POP_ALLOC(c)

	Links (CurrentElement, c);
}

static void coded_savedata ()
{
	check_cur ();

	char *c;
	TMP_POP_ALLOC(c)

	if (!CurrentElement->D)
		CurrentElement->D = getCache (CurrentElement->Key);

	CurrentElement->D->Data ();	// len may have change and K&R say...
	if (writefile (c, CurrentElement->D->Data (), CurrentElement->D->len))
		throw epilException ("\\writefile", strerror (errno));
}

static void coded_find ()
{
	char *c;
	TMP_POP_ALLOC(c)

	E *e = Find (c);

	if ((ep_boolean = e != NULL))
		e->activate ();
	else Logprintf ("\"%s\" not found in the database\n", c);
}

static void coded_cfind ()
{
	char *c;
	TMP_POP_ALLOC(c)

	S *s = cFind (c);

	if ((ep_boolean = s != NULL))
		s->activate ();
	else Logprintf ("\"%s\" not found in the database\n", c);
}

//::::: current list

static char nocurs [] = "Current S list is not set";

static void check_curs ()
{
	if (!CurrentList)
		throw epilException (nocurs);
}

static void coded_havecl ()
{
	ep_boolean = CurrentList != NULL;
}

static void coded_sname ()
{
	check_curs ();

	CurrentList->myName ();
}

static void coded_publicname ()
{
	check_curs ();

	ep_push (CurrentList->Name);
}

static void coded_setpublicname ()
{
	char *c;
	DAT_POP_ALLOC(c)

	check_curs ();

	delete CurrentList->Name;
	CurrentList->Name = c;
}

static void coded_newlist ()
{
	new S;
}

static void coded_isemptylist ()
{
	check_curs ();

	ep_boolean = CurrentList->nelements == 0;
}

static void coded_dup ()
{
	check_curs ();

	new S (CurrentList);
}

static void coded_kill ()
{
	check_curs ();

	Kill (CurrentList);

	CurrentList = (S*) SList.End;
}

static bool broken;

static void coded_foreach ()
{
	char *c;

	TMP_POP_ALLOC (c)

	check_curs ();

	if (CurrentList->nelements == 0) return;

	SaveCE = CurrentElement;

	E *e;
	SLooper L (CurrentList);
	broken = false;

	while (!broken && (e = L.Next ())) {
		e->activate ();
		exec_unit (c);
	}

	CurrentElement = SaveCE;

	broken = false;
}

static void coded_laststep ()
{
	broken = true;
}

static void coded_has_originator ()
{
	check_curs ();

	ep_boolean = CurrentList->Originator != NULL;
}

static void coded_originator ()
{
	check_curs ();

	if (CurrentList->Originator)
		CurrentList->Originator->activate ();
}

// :::::: elements & lists

static void coded_add ()
{
	check_cur ();
	check_curs ();

	CurrentList->Append (CurrentElement);
}

static void coded_rmv ()
{
	check_cur ();
	check_curs ();

	CurrentList->Remove (CurrentElement);
}

// ::::::: list generation

static void coded_lsgroup ()
{
	char *c;
	TMP_POP_ALLOC(c)

	LsGroup (c)->activate ();
}

static void coded_nlsgroup ()
{
	char *c;
	TMP_POP_ALLOC(c)

	LineReply LR;

	Ln_Nls (c);

	if (!LR.Status ()) ServerError ();

	ep_push (LR.DataLine);
}

static void coded_foreachlist ()
{
	char *c;
	TMP_POP_ALLOC(c)

	dlistNode *nx = SList.Start;

	while (nx) {
		CurrentList = (S*) nx;
		nx = SList.Next (nx);
		exec_unit (c);
	}
}

// :::::::: groups only

static char notcon [] = "Not connected";

static void coded_foreachsubgroup ()
{
	char *g, *c;
	TMP_POP_ALLOC(c)
	TMP_POP_ALLOC(g)

	MultiLineReply MLR;

	Ln_Dir (g);

	if (!MLR.Status ())
		if (*c == 0) throw epilException (notcon);
		else {
			ep_boolean = false;
			return;
		}

	for (int i = 0; i < MLR.Lines; i++) {
		ep_push (MLR.DataLine [i]);
		exec_unit (c);
	}

	ep_boolean = true;
}

static void coded_parentgroup ()
{
	char *c, *p;
	TMP_POP_ALLOC(c)

	if ((p = strrchr (c, '.'))) *p = 0;
	else *c = 0;

	ep_push (c);
}

//***********************************************************
// The '/' gate for epil we define to be current element/list
// selector.
//***********************************************************

static char noel [] = "Cant set current element from /";

static void slash_gate (char *cmd)
{
	if (*cmd == 0) return;

	if (isdigit (*cmd)) {
		check_curs ();
		E *e = (*CurrentList) [strtoul (cmd, NULL, 10)];
		if (e) e->activate ();
		else throw epilException (noel, cmd);
	} else if (*cmd == 'i') {
		E *e = EFor (strtoul (cmd + 1, NULL, 10));
		if (e) e->activate ();
		else throw epilException (noel, cmd);
	} else if (*cmd == 'L') {
		int i = strtoul (cmd + 1, NULL, 10);
		dlistNode *d = SList.End;

		while (i-- && (d)) d = SList.Prev (d);
		CurrentList = (d) ? (S*) d : (S*) SList.Start;

	} else if (*cmd == 'o') {
		check_curs ();
		CurrentElement = CurrentList->Originator;
	} else if (!strcmp (cmd, "last")) {
		check_curs ();
		CurrentElement = (*CurrentList) [CurrentList->nelements - 1];
	} else if (!strcmp (cmd, "rnd")) {
		check_curs ();
		int r = (int) (((float) CurrentList->nelements * (float) rand ()) / (RAND_MAX + 1.0));
		CurrentElement = (*CurrentList) [r];
	}

}

static void help_gate (char*)
{
	// a naive user may find himself against a prompt "[epil]:"
	// were he types 'help' and nothing happens. In fact the line
	// just eats all the words he types!
	// He may think of typing '?' :)
	Logputs ("This is a prompt to eShell macro language [epil]\n"
		 "Error: Unexpected user behaviour\n");
}

//***********************************************************
// Instance creation
// Thu Sep  7 19:08:59 EEST 2000
//***********************************************************

void lnshell ()
{
	ep_register_ctrl ('/', slash_gate);
	ep_register_ctrl ('?', help_gate);

	CODED(pmts);
	CODED(setcistr);
	CODED(nocistr);

	CODED(cleanETree);
	CODED(find);
	CODED(cfind);
	CODED(setcache);
	CODED(clear_cache);
	CODED(uncache);
	CODED(havece);
	CODED(ename);
	CODED(eid);
	CODED(group);
	CODED(elen);
	CODED(isbin);
	CODED(edata);
	CODED(links);
	CODED(glinks);
	CODED(savedata);
	CODED(havecl);
	CODED(sname);
	CODED(publicname);
	CODED(setpublicname);
	CODED(newlist);
	CODED(dup);
	CODED(isemptylist);
	CODED(kill);
	CODED(foreach);
	CODED(laststep);
	CODED(has_originator);
	CODED(originator);
	CODED(add);
	CODED(rmv);
	CODED(lsgroup);
	CODED(nlsgroup);
	CODED(foreachlist);
	CODED(foreachsubgroup);
	CODED(parentgroup);
}
