/******************************************************************************

 Macros to fetch members of a class from the data segment to the stack
 frame of the caller. alloca used.
 Members should exist in the DATA segment

******************************************************************************/

#define SFETCH_INCLUDED

/*
 * Rationale.
 *
 * We prefer the members of a class to be allocated in the stack frame.
 * We can't do that at once BUT:
 *  we let member functions normally allocate what they will in the
 *  data segment using \new.
 *  Then we make a temporary object of the same type in the stack of the
 *  function and we copy the members to this tmp object (copy the pointers
 *   only, won't allocate space).
 *  Then we replace the members of our initial object will alloca() pointers
 *  to the stack and memcopy from the data segment to the new stack areas.
 *  Then we free the data segment allocated members from the tmp object.
 *
 * So one may ask: We do not avoid the data segment allocations so why bother?
 * Good question.
 *
 * Answer: The allocated stuff will be freed anyway at some point (this whole
 *	method is used for special objects which mostly will be created as
 *	automatic variables). By doing this we free them earlier and use stack
 *	references of them for the rest of our work.
 *	The trick is that an object will do all its allocations (with \new)
 *	at one call and right after this call our macros are used to fetch
 *	data to the stack. With this schema in mind, eventually the obect
 *	will act as if it hadn't touched the data segment areas. Our
 *	object will not introduce fragmentation and the space will
 *	be automatically freed when the object is destroyed.
 *	We have two tradeoffs:
 *		1/ extra memcopy. We copy the data to the alloca() areas.
 *		   I'm willing to tolerate that. If the thing is used wisely
 *		   \new and malloc() should be much faster in long run.
 *		2/ extra stack space from the temporary object used by the
 *		   macros. An expected tradeoff. sizeof object I expect to be
 *		   about 10-20 bytes. in the stack.
 */

#define FETCH_REPLY(x) \
	if ((x)->ds) {\
	Reply _R = *(x);\
	if (_R.StatusLine) {\
		(x)->StatusLine = (char*) alloca (strlen (_R.StatusLine)+1);\
		strcpy ((x)->StatusLine, _R.StatusLine);\
	} _R.Release ();\
	(x)->ds = 0;\
	}

#define FETCH_LREPLY(x) \
	if ((x)->ds) {\
	LineReply _R = *(x);\
	if (_R.DataLine) {\
		(x)->DataLine = (char*) alloca (strlen (_R.DataLine)+1);\
		strcpy ((x)->DataLine, _R.DataLine);\
	} if (_R.StatusLine) {\
		(x)->StatusLine = (char*) alloca (strlen (_R.StatusLine)+1);\
		strcpy ((x)->StatusLine, _R.StatusLine);\
	} _R.Release ();\
	(x)->ds = 0;\
	}

#define FETCH_MLREPLY(x) \
	if ((x)->ds) {\
	MultiLineReply _R = *(x);\
	if (_R.Lines) {\
		(x)->DataLine = (char**) alloca (_R.Lines * sizeof (char*));\
		for (int i = 0; i < _R.Lines; i++) {\
			(x)->DataLine [i] = (char*) \
			alloca (strlen (_R.DataLine [i]) + 1);\
			strcpy ((x)->DataLine [i], _R.DataLine [i]);\
		}\
	}\
	if (_R.StatusLine) {\
		(x)->StatusLine = (char*) alloca (strlen (_R.StatusLine)+1);\
		strcpy ((x)->StatusLine, _R.StatusLine);\
	} _R.Release ();\
	(x)->ds = 0;\
	}

#define FETCH_shREPLY(x) \
	if ((x)->ds) {\
	shReply _R = *(x);\
	if (_R.Data) {\
		(x)->Data = (char*) alloca (_R.ContentLength);\
		memcpy ((x)->Data, _R.Data, _R.ContentLength);\
	} if (_R.StatusLine) {\
		(x)->StatusLine = (char*) alloca (strlen (_R.StatusLine)+1);\
		strcpy ((x)->StatusLine, _R.StatusLine);\
	} _R.Release ();\
	(x)->ds = 0;\
	}
