/******************************************************************************

	The Base Epil Instance.

	We define '\' gateway to macro calls and an interface to the
	programmer for creating coded macros. The macro \def is defined 
	to dynamically create code from data and its the base of more macros.

	'$' is set to variables and \setvar to creating them.
	Variables are only created at macrocode.

	Macros (not coded) and variables may store multiple definitions in
	a stack. \undef and '-' gate may be used to pop definitions.

	'#' is set to objects. It carries something like C++ `this' member.
	Every object the programmer wants may notify the macrocode with '#'
	about its existance and can do something when called by the macrocode.
	In contrast to variables, objects can only be created from program
	code.

	Macro backtrace is here too.

******************************************************************************/
/**************************************************************************
 Copyright (C) 2000 Stelios Xantkakis
**************************************************************************/

#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include "projectlog.h"
#include "epil.h"

#include "dbstree.h"
#include "sysutil.h"
#include "dLIST.h"
#include "sLIST.h"

#ifndef NULL
#define NULL 0
#endif

#define EPIL_STATS

#ifdef EPIL_STATS
static struct {
	unsigned int ment, cment;
} STATS;
#endif

//*****************************************************************************
// Lock Machine from expanding.
// Macros that expand the entry points should check that
//*****************************************************************************

static char eplock [] = "Macro locked in restricted mode";

bool lock_epil;

void ep_check_locked ()
{
	if (lock_epil) throw epilException (eplock);
}

//*****************************************************************************
// epilCommand: anything called after an operator. It has a case sensitive
// name and belongs in its dbstree.
//
//
// Mature optimization:
// Searching through the macro tree is what epil does all the time.
// Suppose 1024 macros and that we've called optimize_macrotree() to
// make it balanced. We will then be able to find stuff in 10 steps max.
// However, the 512 of the 1024 macros will be on the deepest nodes (10 steps)
// 87.5% of the macros will require more than 7 steps.
//
// By not accepting empty macro names we can compare first
// name[0]/query[0] and then name[1]/query[1] and then perform
// strcmp(name+2, query+2) if those are both equal.
//
// Due to the tree placement and lexical conventions we will probably do
// just one strcmp on the right node.
//
// Test suite for 45000 words tree heavy actions results in a 13% speedup
// 
//*****************************************************************************

dbsTree MacroTree, VarTree, ThisTree;

class epilCommand : public dbsNodeStr
{
	int compare ();
   public:
	epilCommand (dbsTree*);

using	dbsNodeStr::Name;
};

inline int epilCommand::compare ()
{
	if (Name [0] < DBS_STRQUERY [0]) return -1;
	else if (Name [0] > DBS_STRQUERY [0]) return 1;
	if (Name [1] < DBS_STRQUERY [1]) return -1;
	else if (Name [1] > DBS_STRQUERY [1]) return 1;
	return (Name [1]) ? strcmp (Name +2, DBS_STRQUERY +2) : 0;
}

epilCommand::epilCommand (dbsTree *t) : dbsNodeStr (t)
{
	Name = (isdata (DBS_STRQUERY)) ? DBS_STRQUERY : StrDup (DBS_STRQUERY);
}

//*****************************************************************************
// Macro is the abstract common class between coded and user defined macros.
// Almost nothing can be done with coded macros except execution.
// Used-defined macros cannot be undefined if their code is executed.
//*****************************************************************************

class Macro : public epilCommand 
{
   public:
	bool iscoded;
	Macro () : epilCommand (&MacroTree) { }
virtual	void Run () = 0;
	~Macro ()	{ }

using	dbsNodeStr::Name;
};

class CodedMacro : public Macro
{
	void (*func) (void);
   public:
	CodedMacro (void (*)(void));
	void Run ();
};

CodedMacro::CodedMacro (void (*f)(void)) : Macro ()
{
	iscoded = true;
	func = f;
}

void CodedMacro::Run ()
{
#ifdef EPIL_STATS
	++STATS.cment;
#endif
	func ();
}

struct ci
{
	char *epilCode;
	unsigned int used;
	ci (char *c)	{ used = 0; epilCode = c; }
};

class UMacro : public Macro
{
	slistAuto<ci*> Defstack;
   public:
	UMacro (char*);
	void Run ();
	void Redefine (char*);
	int Undefine ();
	void Macrodef ();
	void Export (FILE*);
	~UMacro ();
};

UMacro::UMacro (char *n) : Macro ()
{
	iscoded = false;
	Redefine (n);
}

void UMacro::Run ()
{
	struct alwz {
		unsigned int &uses;
		alwz (unsigned int &i) : uses (i) { ++i; }
		~alwz ()		{ --uses; }
	} A (Defstack.XFor ()->used);

	exec_unit (Defstack.XFor ()->epilCode);
}

void UMacro::Export (FILE *f)
{
	char *c = (char*) alloca (2 * strlen (Defstack.XFor ()->epilCode));

	fprintf (f, "%s %s \\def", 
		 ep_quote (Defstack.XFor ()->epilCode, c), Name);
}

#define MAXDEFS 100
static char maxdef [] = "Macro with too many definitions: ";

void UMacro::Redefine (char *c)
{
	if (Defstack.cnt >= MAXDEFS) throw epilException (maxdef, Name);
	Defstack.addx (new ci (iStrDup (c)));
}

static char undefused [] = "Attempt to undefine a macro executing:";

int UMacro::Undefine ()
{
	if (Defstack.XFor ()->used)
		throw epilException (undefused, Name);

	ci *c = Defstack.XFor ();
	Defstack.dremove ();
	delete c->epilCode;
	delete c;

	return Defstack.Start == NULL;
}

void UMacro::Macrodef ()
{
	ep_push (Defstack.XFor ()->epilCode);
}

UMacro::~UMacro ()
{
	while (Defstack.Start) {
		delete Defstack.XFor ();
		Defstack.dremove ();
	}
	MacroTree.dbsRemove (this);
	delete Name;
}

//*****************************************************************************
// Variables hold arguments from the argument stack internally (no quoting
// and not through the parser).
// They may keep multiple values in a stack.
//*****************************************************************************

class Var : protected epilCommand
{
	slistAuto<char*> Values;
   public:
	Var (char*);
	void set (char*);
	void use ();
	void drop ();
	bool hasval ();
	~Var ();
};

Var::Var (char *c) : epilCommand (&VarTree)
{
	set (c);
}

void Var::set (char *c)
{
	Values.addx (StrDup (c));
}

static char emptyvar [] = "Empty variable: ";

void Var::use ()
{
	if (Values.Start == NULL) throw epilException (emptyvar, Name);
	ep_push (Values.XFor ());
}

bool Var::hasval ()
{
	return Values.Start != NULL;
}

void Var::drop ()
{
	if (Values.Start) {
		delete Values.XFor ();
		Values.dremove ();
	}
}

Var::~Var ()
{
	// The policy for variables is that they NEVER destruct
	// even if empty. So this will never get called.
	// This is the proper destructor for a different policy
	while (Values.Start) {
		delete Values.XFor ();
		Values.dremove ();
	}
	VarTree.dbsRemove (this);
	delete Name;
}

//*****************************************************************************
// Objects. The object tree holds pointers to SufraceObject-capable things.
// On call the activate() member is called.
//*****************************************************************************

static bool otrace = false;

class Object : protected epilCommand
{
friend	class SurfaceObject;
   public:
	Object (SurfaceObject*);
	SurfaceObject *SO;
	~Object ();
};

Object::Object (SurfaceObject *s) : epilCommand (&ThisTree)
{
	SO = s;
}

Object::~Object ()
{
	ThisTree.dbsRemove (this);
	iStrDelete (Name);
}

const static char BTable [] =
"_ABCDEFGHIJKLMNOPQRTSUVWXYZabcdefghijklmnopqrtsuvwxyz0123456789_";

#define BTINDEX(x) (strchr (BTable, x) - BTable)

void SurfaceObject::appear (char *name)
{
	// Empty name is a mutant X
	// Feel free to change this to whatever you like,
	// it won't mess anything (but must be > 5 letters)
	// Include only { letters, digits, _ } because
	// even though correct the macro code may not
	// be prepared to quote the object name properly.
	char gn [] = "Mutant_X";

	// Reverse hash multiplexer
	gn [sizeof gn - 2] = '0' + ThisTree.nnodes % 10;
	if (*name == 0) name = gn;

	// Each object suggests its name. But we can't be sure that
	// it will be unique. In case of existance use the altering schema.
	// The name mutation works better with names > 3 characters
	int j = 0, i;

	DBS_STRQUERY = name;
	while (ThisTree.dbsFind ()) {
		i = j++;
		if (DBS_STRQUERY [j] == 0) j = 0;
		DBS_STRQUERY[i] =
		 BTable [BTINDEX(DBS_STRQUERY [i]) ^ BTINDEX(DBS_STRQUERY [j])];
	}

	O = new Object (this);
}

SurfaceObject *CurrentSurfaceObject;

SurfaceObject::SurfaceObject ()
{ }

void SurfaceObject::myName ()
{
	ep_push (O->Name);
}

char *SurfaceObject::oname ()
{
	return O->Name;
}

void SurfaceObject::called ()
{
	if (otrace) Logprintf ("otrace: #%s\n", O->Name);
	CurrentSurfaceObject = this;
	activate ();
}

SurfaceObject::~SurfaceObject ()
{
	CurrentSurfaceObject = NULL;
	delete O;
}

//*********************************************************************
// Interface
//  Control operator for `\'run_macro.
//  Control operators `$'use_var and '-'drop_var
//  Control operator '#'internal_object_name
//*********************************************************************

static int macro_depth;

static char rprot [] = "Maximum macro depth reached (bug protection)";
static char noepilm [] = "No epil macro \\";
static char noepilv [] = "No epil variable $";
static char noepilo [] = "No epil object #";

static bool mtrace = false;

static slist backtrace;

class ebt : public slistNode
{
   public:
	Macro *m;
	ebt (Macro *i) : slistNode (&backtrace)	{ m = i; }
	~ebt ();
};

ebt::~ebt ()
{
	--macro_depth;
	backtrace.dremove ();
}

void backtrace_onexception ()
{
	slistNode *s;

	Logputs ("Macro backtrace:\n");
	for (s = backtrace.Start; s; s = backtrace.Next (s))
		Logprintf ("\\%s\n", ((ebt*)s)->m->Name);
}

#define MACRO_STACK_PROT 400

static void run_macro (char *name)
{
	// This is a very recursive function in epil. But "m"
	// does not need the recursive safety of automatic variables.
	// "m" is only needed Before a possible recall of run_macro.
	// "ebt E" is the needed automatic stuff and it stores "m".
	static Macro *m;

	if (*name == 0) return;

	if (macro_depth >= MACRO_STACK_PROT)
		throw epilException (rprot);

	DBS_STRQUERY = name;

	m = (Macro*) MacroTree.dbsFind ();

	if (!m) throw epilException (noepilm, name);

	if (mtrace) Logprintf ("mtrace: \\%s\n", m->Name);

#ifdef EPIL_STATS
	++STATS.ment;
#endif
	ebt E = m;
	++macro_depth;
	m->Run ();
	// after Run, we may return normally OR through an exception.
	// Code we wish to execute in both cases is at ~ebt destructor
}

static void use_var (char *name)
{
	if (*name) {
		DBS_STRQUERY = name;
	} else {
		TMP_POP_ALLOC(DBS_STRQUERY)
	}

	Var *m = (Var*) VarTree.dbsFind ();

	if (!m) throw epilException (noepilv, name);

	m->use ();
}

static void drop_var (char *name)
{
	if (*name) {
		DBS_STRQUERY = name;
	} else {
		TMP_POP_ALLOC(DBS_STRQUERY)
	}

	Var *v;

	v = (Var*) VarTree.dbsFind ();

	if (v) v->drop ();
}

void ep_call_object (char *name)
{
	if (*name) {
		DBS_STRQUERY = name;
	} else {
		TMP_POP_ALLOC(DBS_STRQUERY)
	}

	Object *o;

	o = (Object*) ThisTree.dbsFind ();

	if (o) o->SO->called ();
	else throw epilException (noepilo, DBS_STRQUERY);
}

//*****************************************************************************
//
// Fundamental Coded Macros
//
//*****************************************************************************

//********************
// \def
// \undef
// \macrodef
// \setvar
// \this
// \mtrace
// \ifvar
//********************

static char redefcoded [] = "Attempt to redefine a coded macro:";
static char badname [] = "Bad macro name: ";

static void coded_def ()
{
	char *c;

	TMP_POP_ALLOC(DBS_STRQUERY);
	TMP_POP_ALLOC(c);

	ep_check_locked ();

	if (!isalnum (DBS_STRQUERY [0]) && DBS_STRQUERY [0] != '_')
		throw epilException (badname, c);

	if (DBS_STRQUERY [0] == 0) return;

	Macro *m;

	if ((m = (Macro*) MacroTree.dbsFind ()))
		if (m->iscoded) throw epilException (redefcoded, c);
		else ((UMacro*) m)->Redefine (c);
	else
		new UMacro (c);
}

static char undefcod [] = "Attempt to \\undef coded macro:";

static void coded_undef ()
{
	TMP_POP_ALLOC (DBS_STRQUERY)
	if (DBS_STRQUERY [0] == 0) return;

	Macro *m = (Macro*) MacroTree.dbsFind ();

	if (!m) return;

	if (m->iscoded) throw epilException (undefcod, DBS_STRQUERY);

	if (((UMacro*)m)->Undefine ()) delete m;
}

static char ndef [] = "'NOT DEFINED";

static void coded_macrodef ()
{
	TMP_POP_ALLOC (DBS_STRQUERY)
	if (DBS_STRQUERY [0] == 0) return;

	Macro *m = (Macro*) MacroTree.dbsFind ();

	if (!m) ep_push (ndef);
	else if (m->iscoded) ep_push ("'CODED ENTRY");
	else ((UMacro*)m)->Macrodef ();
}

static void coded_ifdef ()
{
	TMP_POP_ALLOC(DBS_STRQUERY)
	if (DBS_STRQUERY [0] == 0) return;

	ep_boolean = MacroTree.dbsFind () != NULL;
}

static char emptyname [] = "Variable without name";

static void coded_setvar ()
{
	char *c;
	TMP_POP_ALLOC(DBS_STRQUERY)
	TMP_POP_ALLOC(c)
	if (DBS_STRQUERY [0] == 0) throw epilException (emptyname);

	Var *v;

	v = (Var*) VarTree.dbsFind ();

	if (v) v->set (c);
	else new Var (c);
}

static void coded_ifvar ()
{
	TMP_POP_ALLOC(DBS_STRQUERY)
	if (DBS_STRQUERY [0] == 0) return;

	Var *v = (Var*) VarTree.dbsFind ();

	ep_boolean = (v == NULL) ? false : v->hasval ();
}

static void coded_ifobj ()
{
	TMP_POP_ALLOC(DBS_STRQUERY)
	if (DBS_STRQUERY [0] == 0) {
		ep_boolean = false;
		return;
	}

	ep_boolean = ThisTree.dbsFind () != NULL;
}

static char noobj [] = "No object to apply \\this";

static void coded_this ()
{
	if (!CurrentSurfaceObject)
		throw epilException (noobj);

	CurrentSurfaceObject->myName ();
}

static void coded_mtrace ()
{
	Logprintf ("Mtrace is %i\n", mtrace = !mtrace);
}

static void coded_otrace ()
{
	Logprintf ("Otrace is %i\n", otrace = !otrace);
}

static void coded_epstats ()
{
#ifdef EPIL_STATS
	Logprintf ("%u macro entries (%u coded) \n", STATS.ment, STATS.cment);
#else
	Logprintf ("epstats left out\n");
#endif
}

//***************************************************************************
//
// Initialize coded macros.
//
//***************************************************************************

int ep_register_coded (char *c, void (*foo)())
{
	DBS_STRQUERY = c;
	if (MacroTree.dbsFind ())
		return 1;
	new CodedMacro (foo);
	return 0;
}

void ep_optimize_macrotree ()
{
	MacroTree.dbsBalance ();
}

void ep_def (char *n, char *d)
{
	DBS_STRQUERY = n;

	UMacro *m;
	if ((m = (UMacro*) MacroTree.dbsFind ()) && !m->iscoded)
		m->Redefine (d);
	else new UMacro (d);
}

bool ep_iscoded (char *c)
{
	DBS_STRQUERY = c;

	Macro *m = (Macro*) MacroTree.dbsFind ();

	return (m) ? m->iscoded : false;
}

bool ep_haveobject (char *c)
{
	DBS_STRQUERY = c;

	return ThisTree.dbsFind () != NULL;
}

static FILE *dumpout;

#define MD ((Macro*)d)
#define UD ((UMacro*)d)
static void dumper (dbsNode *d)
{
	if (MD->iscoded) return;
	UD->Export (dumpout);
	fputc ('\n', dumpout);
}

void ep_dump_macros (FILE *f)
{
	dumpout = f;
	MacroTree.copytree (dumper);
}

void init_epil ()
{
	ep_register_ctrl ('\\',	run_macro);
	ep_register_ctrl ('$',	use_var);
	ep_register_ctrl ('-',	drop_var);
	ep_register_ctrl ('#',	ep_call_object);

	ep_register_coded ("def",	coded_def);
	ep_register_coded ("undef",	coded_undef);
	ep_register_coded ("macrodef",	coded_macrodef);
	ep_register_coded ("ifdef",	coded_ifdef);
	ep_register_coded ("setvar",	coded_setvar);
	ep_register_coded ("ifvar",	coded_ifvar);
	ep_register_coded ("ifobj",	coded_ifobj);
	ep_register_coded ("this",	coded_this);
	ep_register_coded ("mtrace",	coded_mtrace);
	ep_register_coded ("otrace",	coded_otrace);
	ep_register_coded ("epstats",	coded_epstats);
}
