/******************************************************************************

	The epil base layer.

	Atomic epil parser, a macro may call program code and the program code
	macrocode without screwing up any global data.

	The argument stack, available at macrolevel with the help of the
	parser and at program level directly.

	Alternative stackmode, available here because redefinition of ep_push
	would be required to implement is elsewhere.

	Control character gateway, free to the prorgammer to create the
	instance.

	Also '{' and '}' binded to altmode operations.

******************************************************************************/
/**************************************************************************
 Copyright (C) 2000 Stelios Xantkakis
**************************************************************************/

#include <ctype.h>
#include <string.h>
#include <stdio.h>

#include "projectlog.h"

#include "epil.h"
#include "sysutil.h"
#include "dLIST.h"
#include "sLIST.h"

#ifndef NULL
#define NULL 0
#endif

#define EBUF 100

//******************************************************************************
// epilException::std () shall be defined at main.C to take care of all
// cleanups through the program. Its the last function called before
// the stack is unrolled.
//******************************************************************************

epilException::epilException (const char *c)
{
	strncpy (epilExceptionText, c, EBUF - 1);
	std ();
}

epilException::epilException (const char *c, const char *b)
{
	strncpy (epilExceptionText, c, EBUF - 1);
	strncat (epilExceptionText, b, EBUF - 1 - strlen (c));
	std ();
}

char epilExceptionText [EBUF];

/******************************************************************************

	Virtual stack

	The epil arguments need a special allocator.
	For two reasons, first they have varying sizes and many of them may
	be very small, second they are released in known order.
	We can avoid a terrible malloc fragmentation that would occur from
	the first by using the second.

	The obvious structure is a stack.

	A virtual stack is used. This stack stores the data in a dynamic
	list of 4kB segments. The vss_pop() function will transparently copy
	data to a destination pointer, even if the data is in many segments.

	The stack is "store & increase"
	The sizes of the arguments are also stored in the stack as sizeof(int)
	char arrays.

	A bad case were a small argument is pushed and poped many times in the
	edge of a segment would allocate a segment for a few bytes and then
	free it. This case is avoided by keeping an empty segment around
	until the last non-empty segment is less than 70% used.


	This is all ok but theoretically the OS could provide us with a
	stack mmaped area which grows down automatically on page faults like
	the true stack. Linux has a MAP_GROWSDOWN flag but anybody knows how
	to invoke that ?
	There have been rumored programs with multiple stacks ?

	The bad thing is that OS stacks don't shrink but only expand.
	There's a patch I have here which adds stack shinking to linux 2.2.x
	with a special syscall but it didn't make it to the finals.

******************************************************************************/

#define SEGMENT_SIZE 4096

class vss
{
friend	void open_alt ();
friend	void close_alt ();
friend	class altvss;
	dlistAuto<char*> Segments;
   protected:
	int ptr;
	int vss_popint ();
	void vss_push (char*, int);
	bool have_segment;
	void partial_copy (int, char*, int);
   public:
	vss ();
	int vss_pop (char*, int);
	int vss_poplen ();
	void i_push (char*, int);
	void vss_cleanseg ();
	int vss_nthentry (int, char*, int);
	bool isempty ();
};

vss::vss ()
{
	Segments.add (new char [SEGMENT_SIZE]);
	ptr = 0;
	have_segment = false;
}

const static char noarg [] = "Argument stack empty";

void vss::partial_copy (int sp, char *d, int l)
{
	dlistNode *dn;
	int i, j = sp / SEGMENT_SIZE;

	/* How many segments away is the new ptr and the old ptr */
	int sb = (sp + l) / SEGMENT_SIZE - j;

	/* Go to data-start segment */
	if (j <= Segments.cnt / 2)
		for (dn = Segments.Start, i = 0; i < j; i++)
			dn = Segments.Next (dn);
	else for (dn = Segments.End, i = Segments.cnt - 1; i > j; i--)
			dn = Segments.Prev (dn);

	if (sb) {
		int k;

		/* data offset in first segment */
		k = sp % SEGMENT_SIZE;

		/* Copy data */
		for (j = i = 0; i < sb; i++) {
			if (d) memcpy (d + j, Segments.XFor(dn) + k, SEGMENT_SIZE - k);
			j += SEGMENT_SIZE - k;
			k = 0;
			dn = Segments.Next (dn);
		}

		/* Copy last segment */
		if (d) memcpy (d + j, Segments.XFor (dn), ptr % SEGMENT_SIZE);
	} else if (d) memcpy (d, Segments.XFor (dn) + sp % SEGMENT_SIZE, l);
}

int vss::vss_pop (char *d, int l)
{
	/* The new ptr */
	int sp = ptr - l;

	if (sp < 0) throw epilException (noarg);

	/* Do partial copy */
	partial_copy (sp, d, l);

	/* How many segments away is the new ptr and the old ptr */
	int sb = ptr / SEGMENT_SIZE - sp / SEGMENT_SIZE;

	if (sb) {
		/* Definatelly free late-freeing segment if exists */
		if (have_segment) {
			delete Segments.XFor (Segments.End);
			Segments.dremove (Segments.End);
			have_segment = false;
		}

		int i;

		/* Release free segments except late freeing */
		for (sb--, i = 0; i < sb; i++) {
			delete Segments.XFor (Segments.End);
			Segments.dremove (Segments.End);
		}

		/* We have an empty segment which we may free later */
		have_segment = true;
	}

	/* Late freeing time has come ? */
	if (have_segment && (sp - 1) % SEGMENT_SIZE < 7 * SEGMENT_SIZE / 10) {
		delete Segments.XFor (Segments.End);
		Segments.dremove (Segments.End);
		have_segment = false;
	}

	return (ptr = sp) == 0;
}

int vss::vss_popint ()
{
	int i;

	vss_pop ((char*)&i, sizeof (int));

	return i;
}

int vss::vss_poplen ()
{
	return vss_popint ();
}

void vss::vss_push (char *s, int l)
{
	/* New ptr */
	int sp = ptr + l;

	/* How many segments away is the new ptr and the old ptr */
	int sb = sp / SEGMENT_SIZE - ptr / SEGMENT_SIZE;

	/* Which is the starting segment (late free not good) */
	dlistNode *d = (have_segment) ?
		       Segments.Prev (Segments.End) : Segments.End;

	if (sb) {
		int i, j, k;

		/* How much in starting segment */
		j = SEGMENT_SIZE - ptr % SEGMENT_SIZE;

		/* Fill up starting segment */
		memcpy (Segments.XFor (d) + ptr % SEGMENT_SIZE,
			s, j);

		/* Allocate new segments and fill them */
		for (i = 0; i < sb; i++) {

			/* We may have 1 empty segment ready */
			if (i == 0 && have_segment)
				have_segment = false;
			else Segments.add (new char [SEGMENT_SIZE]);

			/* Do partial copy (sounds good) */
			k = (l - j >= SEGMENT_SIZE) ? SEGMENT_SIZE : l - j;
			memcpy (Segments.XFor (Segments.End),
				s + j, k);
			j += k;
		}
	} else memcpy (Segments.XFor (d) + ptr % SEGMENT_SIZE,
			s, l);

	ptr = sp;
}

void vss::i_push (char *c, int i)
{
	vss_push (c, i);
	vss_push ((char*)&i, sizeof (int));
}

void vss::vss_cleanseg ()
{
	while (Segments.Start != Segments.End) {
		delete Segments.XFor (Segments.End);
		Segments.dremove (Segments.End);
	}
	have_segment = false;
	ptr = 0;
}

int vss::vss_nthentry (int i, char *d, int s)
{
	int sp = ptr;
	int sz;

	while (i) {
		sp -= sizeof (int);
		partial_copy (sp, (char*)&sz, sizeof (int));
		sp -= sz;
		i--;
	}

	sp -= sizeof (int);
	partial_copy (sp, (char*)&sz, sizeof (int));
	sp -= sz;

	if (d) {
		partial_copy (sp, d, (sz > s) ? s : sz);
		d [(sz > s) ? s : sz] = 0;
	}

	return sz;
}

bool vss::isempty ()
{
	int i;

	partial_copy (ptr - sizeof (int), (char*) &i, sizeof (int));

	return i == 0;
}

static vss Args;

/******************************************************************************
			Alternative stacks
******************************************************************************/

int ep_argstack;
int ep_pargs;

static slist altList;

class altvss : public vss, public slistNode
{
	int nalt;
	void pushtop ();
	void destruct ();
   public:
	altvss ();
	void Ok ();
	void i_push (char*, int);
	bool closed ()		{ return nalt == -1; }
	~altvss ();
};

altvss::altvss () : vss (), slistNode (&altList)
{
	nalt = 0;
}

void altvss::pushtop ()
{
	int i = vss_poplen ();
	char *c = (char*) alloca (i);
	vss_pop (c, i);
	Args.i_push (c, i);
}

void altvss::destruct ()
{
	altList.dremove ();
	vss_cleanseg ();
	delete Segments.XFor (Segments.End);
	Segments.dremove (Segments.End);
}

void altvss::Ok ()
{
	ep_argstack += (ep_pargs = nalt);
	while (nalt--) pushtop ();
	destruct ();
}

void altvss::i_push (char *d, int i)
{
	vss::i_push (d, i);
	++nalt;
}

altvss::~altvss ()
{
	if (nalt != -1) destruct ();
}

// ***********************************************************
//		End of virtual stack
// ***********************************************************

static int plen;

static dlistAuto<int> sBList;
static char sbreak [] = "Stack breakpoint";
static char nobreak [] = "No Stack breakpoint";

bool ep_isempty ()
{
	return Args.isempty ();
}

int ep_examine_stack (int i, char *d, int s)
{
	if (i >= ep_argstack) return -1;

	return Args.vss_nthentry (i, d, s);
}

void ep_setSbp ()
{
	if (sBList.End && sBList.XFor (sBList.End) == ep_argstack)
		return;
	sBList.add (ep_argstack, true);
}

void ep_delSbp ()
{
	if (!sBList.End) throw epilException (nobreak);
	sBList.dremove (sBList.End);
}

void ep_cleanSbp ()
{
	while (sBList.End)
		sBList.dremove (sBList.End);
}

bool strace = false;

int ep_poplen ()
{
	if (sBList.End && sBList.XFor (sBList.End) <= ep_argstack)
		throw epilException (sbreak);

	return plen = Args.vss_poplen ();
}

void ep_pop (char *p)
{
	Args.vss_pop (p, plen);
	if (p) {
		p [plen] = 0;
		if (strace) Logprintf ("strace pop: [%s]\n", p);
	} else if (strace) Logprintf ("strace pop: Top arg discarded\n");
	--ep_argstack;
}

void ep_push (char *a, int l)
{
	if (strace) Logprintf ("strace push: \"%s\"\n", a);
	if (!altList.Start) {
		Args.i_push (a, l);
		++ep_argstack;
	} else ((altvss*) altList.Start)->i_push (a, l);
}

void ep_push (char *a)
{
	ep_push (a, strlen (a));
}

void ep_repush ()
{
	ep_poplen ();
	char *c = (char*) alloca (plen + 1);
	c [plen] = 0;
	ep_pop (c);
	if (strace) Logprintf ("strace push: \"%s\"\n", c);

	Args.i_push (c, plen);
	++ep_argstack;
	ep_push (c, plen);
}

void ep_clearstack ()
{
	Args.vss_cleanseg ();
	ep_argstack = 0;
}

const static char noalt [] = "Unexpected '}'";

void close_alt ()
{
	if (!altList.Start) throw epilException (noalt);

	((altvss*) altList.Start)->Ok ();
}

// *************************** Parser **************************

static int tokenlen (char *p)
{
	int i = 0;

	while (isalpha (p [i]) || isdigit (p [i]) || p [i] == '_')
		i++;

	return i;
}

// **************************************
// skips and exits. How will these affect
// their previous state is a matter of
// how much freedom we'll give to the epil
// programmer. For the moment, each skip
// or exit command overrides a previous
// state.
// **************************************

static struct {
	bool on;
	bool i;
	union {
		int n;
		char *s;
	} x;
} skips, exits;

static bool comma_bool = false;

#define SET_COMMABOOL \
	comma_bool = skips.on || exits.on

void ep_setskips (int i)
{
	if (skips.on && !skips.i && skips.x.s)
		delete skips.x.s;
	skips.on = true;
	skips.i = true;
	skips.x.n = i;
	SET_COMMABOOL;
}

void ep_setskips (char *c)
{
	if (skips.on && !skips.i && skips.x.s)
		delete skips.x.s;
	skips.on = true;
	skips.i = false;
	skips.x.s = (isdata (c)) ? c : StrDup (c);
	SET_COMMABOOL;
}

void ep_setexits (int i)
{
	if (exits.on && !exits.i && exits.x.s)
		delete exits.x.s;
	exits.on = true;
	exits.i = true;
	exits.x.n = i;
	SET_COMMABOOL;
}

void ep_setexits (char *c)
{
	if (exits.on && !exits.i && exits.x.s)
		delete exits.x.s;
	exits.on = true;
	exits.i = false;
	exits.x.s = (isdata (c)) ? c : StrDup (c);
	SET_COMMABOOL;
}

void ep_cleancommas ()
{
	if (exits.on && !exits.i && exits.x.s)
		delete exits.x.s;
	if (skips.on && !skips.i && skips.x.s)
		delete skips.x.s;
	exits.on = skips.on = false;
	SET_COMMABOOL;
}

static int do_comma (char **p)
{
	int len = tokenlen (*p);
	char *c = (char*) alloca (len + 1);

	strncpy (c, *p, len);
	c [len] = 0;
	*p += len;

	if (skips.on) {
		if (skips.i) {
			skips.x.n--;
			skips.on = skips.x.n != 0;
		} else {
			if (!strcmp (skips.x.s, c)) {
				delete skips.x.s;
				skips.x.s = NULL;
				skips.on = false;
			}
		}
		SET_COMMABOOL;
		return 0;
	}

	if (exits.i) {
		exits.x.n--;
		exits.on = exits.x.n != 0;
	} else {
		if (!strcmp (exits.x.s, c)) {
			delete exits.x.s;
			exits.x.s = NULL;
			exits.on = false;
		}
	}
	SET_COMMABOOL;

	return 1;
}

/*
#define TRACER \
	Logprintf ("Entering %s, point at \"%s\"\n", __FUNCTION__, *p);
*/
#define TRACER ;

const static char openarg [] = "Argument not closed with '";

static int get_escargument (char **s, char **d)
{
	bool esc = false;

	*(*d)++ = *(*s)++;

	for (;;) switch (**s) {
		case 0:
			return -1;
		case '\\':
			esc = !esc;
			*(*d)++ = *(*s)++;
			break;
		case '`':
			if (!esc) {
				get_escargument (s, d);
				(*s)++;
			} else *(*d)++ = *(*s)++;
			esc = false;
			break;
		case '\'':
			*(*d)++ = **s;
			if (!esc) return 0;
			(*s)++;
			esc = false;
			break;
		default:
			*(*d)++ = *(*s)++;
			esc = false;
		}
}

static void get_argument (char **p)
{
	TRACER
	bool esc = false;
	char *out = (char*) alloca (strlen (*p) + 1);
	char *d = out;

	for ((*p)++; ; (*p)++) switch (**p) {
		case 0:
			throw epilException (openarg);
		case '\\':
			if (!(esc = !esc)) *d++ = '\\';
			break;
		case '`': 
			if (esc) {
				*d++ = '`';
				esc = false;
				break;
			}
			get_escargument (p, &d);
			if (**p == 0) throw epilException (openarg);
			break;
		case '\'':
			if (esc) {
				*d++ = '\'';
				esc = false;
				break;
			}
			goto outa;
		default:
			if (esc) {
				*d++ = '\\';
				esc = false;
			}
			*d++ = **p;
			break;
	}

outa:
	if (!skips.on) {
		*d = 0;
		ep_push (out);
	}
	(*p)++;
}

char *ep_quote (char *c, char *out)
{
	// This is not part of the parser but its the right thing
	// to have the quoter symertically together with the
	// parser dequoter
	char *d, *dt, *ct;

	d = out;

	*d++ = '`';
	for (;;) switch (*c) {
		case 0:
			goto fini;
		case '`':
			dt = d; ct = c;
			if (get_escargument (&ct, &dt) == -1) {
				*d++ = '\\';
				*d++ = '`';
				c++;
			} else {
				c = ct + 1;
				d = dt;
			}
			break;
		case '\'':
			*d++ = '\\';
		default:
			*d++ = *c++;
	}
fini:
	int i = 0;		// Wrong but works for _normal_ cases
	while (*(d - 1 - i) == '\\') ++i;
	while (i--) *d++ = '\\';
	*d++ = '\'';
	*d = 0;

	return out;
}

void ep_push_quoted_arg (char *c)
{
	char *out = (char*) alloca (strlen (c) * 2 + 1);
	ep_push (ep_quote (c, out));
}

static inline void getw_argument (char **p)
{
	TRACER
	int len = tokenlen (*p);
	char *c = (char*) alloca (len + 1);

	if (skips.on == 0) {
		strncpy (c, *p, len);
		c [len] = 0;
		ep_push (c, len);
	}

	*p += len;
}

/******************************************************************************
				exec_unit
******************************************************************************/

static char noaltc [] = "Missing '}'";

static void exec_gateway (char**);

bool etrace = false;

void exec_unit (char *p)
{
	if (etrace) Logprintf ("etrace: \"%s\"\n", p);

	while (1) {
		while (isspace (*p)) p++;

		if (*p == 0) break;
		if (*p == '\'') break;
		if (*p == '%') {
			do p++;
			while (*p && *p != '\n');
			if (*p == 0) break;
			continue;
		}
		if (*p == '{') {
			altvss A;
			exec_unit (++p);
			if (!A.closed ()) throw epilException (noaltc);
			break;
		}
		if (*p == '}') {
			++p;
			close_alt ();
			continue;
		}
		if (*p == '`') {
			get_argument (&p);
			continue;
		}
		if (isalpha (*p) || isdigit (*p) || *p == '_') {
			getw_argument (&p);
			continue;
		}
		if (*p == ',' && comma_bool) {
			++p;
			if (do_comma (&p)) break;
		}
		else exec_gateway (&p);
	}
}

//***********************************
// Instead of binsearch linear search
// is used but the operators are in
// the order of usage. eShell will
// mostly call '\', '$',
//************************************

static const char freectrl [] = "\\$-#~/:" /*      */ ",!\"&()*+.;<=>?@[]^|";

static void null (char*)
{ }

static void (*cdef [sizeof freectrl - 1])(char*) = {
	null, null, null, null, null, null, null, null, null, null, null,
	null, null, null, null, null, null, null, null, null, null, null,
	null, null, null, null
};

static int cindex (char c)
{
	char *cp = (char*)freectrl;

	while ((*cp) && (*cp != c)) cp++;

	if (*cp == 0) return -1;

	return (cp - freectrl);
}

int ep_register_ctrl (char c, void (*foo)(char*))
{
	int i = cindex (c);

	if (i == -1) return -1;

	cdef [i] = foo;

	return 0;
}

static void exec_gateway (char **p)
{
	TRACER
	static char gt;
	static int len;
	static char *c;
	// this is a very recursive function in epil.
	// The variables "gt", "len", "c" do not need the
	// recusrion-Ok safety of automatic variables because
	// recursion may only be the last thing done (cdef [len] (c)).
	// The automatic variables space is "n * sizeof (variables)"
	// where n the depth of recusrion (which can be 0), while the
	// static variables storage space is always "sizeof (variables)".
	//
	// Under certain circumstances, the temporaries are stored in
	// global variables. This is such a case.
	//

	gt = **p;
	len = tokenlen (++(*p));
	c = (char*) alloca (len + 1);

	strncpy (c, *p, len);
	c [len] = 0;
	*p += len;

	if (skips.on == 0)
		if ((len = cindex (gt)) != -1) cdef [len] (c);
}
