/******************************************************************************

	The OOP readline library

	GNU readline has a few disadvantages. It has the input output code
	built in the readline implementation. Which makes it impossible to 
	be used in anything else than a tty input. Moreover, it uses
	global data and it cannot be multithreaded nor can we have multiple
	readlines each with its own history, completions, etc.. Plus its huge.

	This is the readline++ implementation which... hm seems to work
	in : multiple readlines, tty input, X windows input.
	See all the virtual's in readline++.h and maybe this is more 
	general purpose than a simple implementation to do the job.

******************************************************************************/
/**************************************************************************
 Copyright (C) 2000 Stelios Xantkakis
**************************************************************************/

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "readline++.h"

#define INIT_LEN 40
#define MAX_LINE 512

extern char *StrDup (char*);

// ::::::::: line_struct
// Basically a structure of growing and shrinking size
// with scrolling capability for insert & delete
// Very uncomfortable but how else can we handle lines
// :::::::::::::::::::::::::::::::::::::::::::::::::::

void line_struct::reset (char *c)
{
	if (line) delete [] line;

	if (c == NULL) {
		line = new char [allocd = INIT_LEN];
		line [0] = 0;
		cursor = linelen = 0;
	} else {
		linelen = strlen (c);
		line = new char [allocd = linelen + INIT_LEN];
		strcpy (line, c);
		cursor = 0;
	}
}

line_struct::line_struct ()
{
	line = NULL;
	reset ();
}

void line_struct::add_char (char c)
{
	if (linelen >= MAX_LINE) return;

	if (linelen + 4 >= allocd) {
		char *t = line;
		line = new char [allocd = allocd * 3 / 2];
		strcpy (line, t);
		delete [] t;
	}

	for (int i = ++linelen; i > cursor; i--)
		line [i] = line [i - 1];

	line [cursor++] = c;
}

void line_struct::del ()
{
	if (line [cursor] == 0) return;

	for (int i = cursor; i <= linelen; i++)
		line [i] = line [i + 1];

	if (--linelen < allocd / 2 && allocd > INIT_LEN) {
		char *t = line;
		line = new char [allocd = allocd * 2 / 3];
		strcpy (line, t);
		delete [] t;
	}
}

void line_struct::backspace ()
{
	if (cursor == 0 || linelen == 0) return;

	--cursor;
	del ();
}

void line_struct::word_left ()
{
	if (cursor == 0) return;
	--cursor;

	while (cursor > 0 &&
	       !(!isspace (line [cursor]) && isspace (line [cursor - 1])))
		cursor--;
}

void line_struct::word_right ()
{
	if (cursor == linelen - 1) return;

	++cursor;

	while (cursor < linelen - 1 &&
	       !(!isspace (line [cursor]) && isspace (line [cursor - 1])))
		cursor++;
}

void line_struct::Strcat (char *c)
{
	int i = strlen (c);

	if (linelen + i >= MAX_LINE) return;

	while (linelen + i + 4 >= allocd) {
		char *t = line;
		line = new char [allocd = allocd * 3 / 2];
		strcpy (line, t);
		delete [] t;
	}

	strcat (line, c);
	linelen += i;
}

void line_struct::deltoend ()
{
	linelen = cursor;
	line [cursor] = 0;
}

line_struct::~line_struct ()
{
	if (line) delete [] line;
}

// ::::::::: readline bindings
// These are defined in a different way among input libraries, envs
// Translation converts to those which lead to standard actions
// ::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

breadline::breadline (char *p, int i)
{
	prompt = iStrDup (p);
	max_len = i;
	Vcur = 0;
}

void breadline::present ()
{ }

void breadline::do_cchar ()
{
	if (line.linelen < max_len)
		line.add_char (cc);
}

void breadline::do_char (int c)
{
	switch (cc = c) {
	case K_UP:
	case K_DOWN:
		do_varrow ();
		break;
	case K_LEFT:
	case K_RIGHT:
		do_harrow ();
		break;
	case K_DEL:
	case '\b':
		do_bsdel ();
		break;
	case K_END:
	case K_HOME:
		do_hoend ();
		break;
	case K_PGUP:
	case K_PGDOWN:
		do_pg ();
		break;
	case '\t':
		do_tab ();
		break;
	case '\n':
		do_enter ();
		break;
	case K_SKIP:
		break;
	default:
		do_cchar ();
	}

	fix_vcur ();
	present ();
}

// :::::::: the left-right arrows, backspace-del, pg up-down are normal
// :::::::: we may extend their operation but without choice.

void breadline::fix_vcur ()
{
	if (line.cursor < Vcur) Vcur = line.cursor;
	if (Vcur + Width <= line.cursor) Vcur = line.cursor - Width + 1;
}

void breadline::do_harrow ()
{
	if (cc == K_LEFT) {
		if (line.cursor > 0)
			--line.cursor;
	} else {
		if (line.linelen > 0 && line.cursor < line.linelen)
			++line.cursor;
	}
}

void breadline::do_bsdel ()
{
	if (cc == K_DEL) line.del ();
	else {
		line.backspace ();
		if (Vcur) --Vcur;
	}
}

void breadline::do_pg ()
{ }

void breadline::do_hoend ()
{
	if (cc == K_HOME) line.cursor = 0;
	else line.cursor = (line.linelen > 0) ? line.linelen : 0;
}

// :::::::: normtab. TAB taken literally as a character

void normtab::do_tab ()
{
	line.add_char ('\t');
}

// :::::::: ignore up and down arrows

void novarrow::do_varrow ()
{ }

// ::::::::::::::: history
// v-arrows set to history movement, adding lines on enter
// setting current line if changes on a history line
// This history is different from the familiar shell history.
// In here if we scroll back and change something, the
// changed line becomes the current edit line. Thus no
// multiple changes in the history.
// :::::::::::::::::::::::::::::::::::::::::::::::::::::::

history::history (int i) : TheList ()
{
	nlines = i;
	char *c = new char [1];
	*c = 0;
	eline = NULL;
	cur = NULL;
	TheList.add (c, false);
}

history::~history ()
{
	while (TheList.Start) {
		delete [] TheList.XFor (TheList.Start);
		TheList.dremove (TheList.Start);
	}
}

void history::do_varrow ()
{
	// freaky mixed up code I really hate
	if (!cur) {
		if (cc == K_DOWN) return;
		if (eline) delete [] eline;
		eline = StrDup (line.line);
	} else if (strcmp (line.line, TheList.XFor (cur))) {
		delete [] eline;
		eline = StrDup (line.line);
		cur = NULL;
		if (cc == K_DOWN) return;
	}

	if (cc == K_DOWN) {
		cur = TheList.Prev (cur);
		if (!cur) line.reset (eline);
		else line.reset (TheList.XFor (cur));
	} else {
		if (!cur) cur = TheList.Start;
		else if (!TheList.Next (cur)) return;
		     else cur = TheList.Next (cur);
		line.reset (TheList.XFor (cur));
	}

	line.cursor = 0;
}

static int whites (char *c)
{
	while (*c) {
		if (!isspace (*c)) return 0;
		c++;
	}
	return 1;
}

void history::addl ()
{
	if (!whites (line.line)
	&& strcmp (line.line, TheList.XFor (TheList.Start)))
		TheList.add (StrDup (line.line), false);

	if (eline) {
		delete [] eline;
		eline = NULL;
	}
}

// :::::::::::::::::::::::::::::::::::::
// Ok. Where are the completions buddy?
// We have to write a readline to alter
// the line on tab.
// I did that in another program (really)
// but we just want to leave them out
// of eShell for now, because what
// to complete first?
// ::::::::::::::::::::::::::::::::::::
