/******************************************************************************

	The low-level graphics functions are used to form little drawable
	area objects. The graph library is instantiated in drawables.

	Container drawables maintain a list of drawables which
	may be drawn: horizontal, vertical, automatic

	Some basic type of drawables here too. Button, virtual screen,
	checkbox, text input entry, scroll meter, sign.

******************************************************************************/
/**************************************************************************
 Copyright (C) 2000 Stelios Xantkakis
**************************************************************************/

#include <stdio.h>
#include <string.h>

#include "graphics.h"
#include "sysutil.h"
#include "readline++.h"
#include "arch.h"
#include "projectlog.h"

extern int c_scr_bg, c_scr_fg, c_scr_rv;

int clickX, clickY;
bool rightButton;

// :::::::::::::: drawable
// This has all the common characteristics of a drawable.
// draw() onclick () setXY ()
// Moreover we make the drawing library objective.
// ::::::::::::::::::::::::::::::::::::::::::::::::::::::

void drawable::drawline (int x1, int y1, int x2, int y2, int col)
{ DrawLine (X + x1, Y + y1, X + x2, Y + y2, col); }

void drawable::drawpoint (int x, int y, int col)
{ DrawPoint (X + x, Y + y, col); }

void drawable::fillrect (int x, int y, int w, int h, int col)
{ FillRect (X + x, Y + y, w, h, col); }

void drawable::drawrect (int x, int y, int w, int h, int col)
{ DrawRect (X + x, Y + y, w, h, col); }

void drawable::drawstring (char *s, int x, int y, int f, int col)
{ DrawString (s, X + x, Y + y, f, col); }

void drawable::bdrawstring (char *s, int x, int y, int f, int col, int bg)
{ BDrawString (s, X + x, Y + y, f, col, bg); }

void drawable::trcursor (int x, int y, char a)
{ JCursor::transport (X + x, Y + y, a); }

void drawable::setXYd (drawable *d, int x, int y)
{ d->setXY (X + x, Y + y); }

bool drawable::checkclick ()
{
	return (clickX > X && clickX < X + W && clickY > Y && clickY < Y + H);
}

int drawable::Bthick = 2;

static char light = 9;

int drawable::switchLight ()
{
#define HAPPY_NEW_SEASON_EPOCH 1000
	// Casual time sense. Time counted by events that happen
	// and every 1000 events its time to switch the lights.
	// bells & whistles implementation v1.0
static	int progress = 0;

	if ((++progress % HAPPY_NEW_SEASON_EPOCH) == 0) {
		if (light == 1 + 8) light = 1 + 2;
		else if (light == 1 + 2) light = 2 + 4;
		else if (light == 2 + 4) light = 4 + 8;
		else light = 1 + 8;

		return 1;
	}

	return 0;
}

void drawable::perigram (int c1, int c2)
{
#define LIGHT(X) ((light & (1 INCR_SHIFT X)) ? c1 : c2)
	drawline (0, 0, W, 0, LIGHT(0));
	drawline (1, 1, W -2, 1, LIGHT(0));

	drawline (0, 0, 0, H, LIGHT(1));
	drawline (1, 1, 1, H -2, LIGHT(1));

	drawline (W, 0, W, H, LIGHT(3));
	drawline (W -1, 1, W -1, H -1, LIGHT(3));

	drawline (0, H, W, H, LIGHT(2));
	drawline (1, H -1, W -2, H -1, LIGHT(2));
}

void drawable::altperigram (int c1, int c2)
{
	drawline (0, 0, W, 0, LIGHT(0));
	drawline (1, 1, W -2, 1, LIGHT(2));

	drawline (0, 0, 0, H, LIGHT(1));
	drawline (1, 1, 1, H -2, LIGHT(3));

	drawline (W, 0, W, H, LIGHT(3));
	drawline (W -1, 1, W -1, H -1, LIGHT(1));

	drawline (0, H, W, H, LIGHT(2));
	drawline (1, H -1, W -2, H -1, LIGHT(0));
}

void drawable::hline (int x, int y, int w)
{
	drawline (x, y, x + w, y, (light & 4) ? COL_WHITE : COL_DARKGREY);
	drawline (x, y + 1, x + w, y + 1,  (light & 1) ? COL_WHITE : COL_DARKGREY);
}

void drawable::dperigram ()
{
	perigram (8, 1);
}

void drawable::uperigram ()
{
	perigram (1, 8);
}

void drawable::fill_block (int c)
{
	fillrect (Bthick, Bthick, W - 2*Bthick + 1, H - 2*Bthick + 1, c);
}

drawable::drawable ()
{
	X = Y = W = H = 0;
}

drawable::drawable (int x, int y, int w, int h)
{
	X = x, Y = y, W = w, H = h;
}

bool drawable::onclick ()
{
	return checkclick ();
}

void drawable::setXY (int x, int y)
{
	X = x, Y = y;
}

bool drawable::operator == (drawable *d)
{
	return d == this;
}

void drawable::draw ()
{ }

// :::::::::::::: JCursor namespace
// The cursor stuff are relative but why write an
// object since there are no multiple instances
// and no ancestors.
// Named JCursor because there is a Cursor in X.h
//
// This cursor crap is the most buggy thing
// It appears when it shouldn't and moves outside
// its restricted perimeter.
// Debuging time / Code size ratio = \inf
// ::::::::::::::::::::::::::::::::::::::::::::::

namespace JCursor {

bool		On, Flashing;
int		X, Y, font, fg, bg;
breadline*	Readline;
char 		Symbol [2];
bool 		PasteNL;

void readlineDeath ()
{
	Readline = NULL;
	X = -1;
}

void show ()
{
	if (Readline && X != -1) DrawString ("_", X, Y, font, fg);
}

void hide ()
{
	if (Readline && X != -1) {
		DrawString ("_", X, Y, font, bg);
		DrawString (Symbol, X, Y, font, fg);
		X = -1;
	}
}

void transport (int x, int y, char a)
{
	X = x, Y = y;
	Symbol [0] = a;
}

void changerl (breadline *b, int f, int Fg, int Bg, bool pb)
{
	PasteNL = pb;
	hide ();
	Readline = b;
	font = f;
	fg = Fg, bg = Bg;
}

void do_char (int c)
{
	if (Readline) {
		hide ();
		Readline->do_char (c);
		show ();
	}
}

}

void Paste (char *c)
{
	if (!JCursor::Readline) return;

	if (!JCursor::PasteNL) {
		char *p = strchr (c, '\n');
		if (p) *p = 0;
	}

	for (; *c; c++)
		JCursor::Readline->do_char (*c);
}

// :::::::::::::: containers
// We add drawables in the internal drawable list
// recalculate positions and size with settle()
// ::::::::::::::::::::::::::::::::::::::::::::::

dcont::dcont ()
{
	peri = false;
}

void dcont::operator += (drawable *d)
{
	DList.add (d);
	settle ();
}

void dcont::draw ()
{
	if (peri) uperigram ();
	fill_block (COL_LIGHTGREY);

	dlistNode *d;
	for (d = DList.Start; d; d = DList.Next (d))
		if (DList.XFor (d)->W > 0)
			DList.XFor (d)->draw ();
}

void dcont::settle ()
{ }

void dcont::setXY (int x, int y)
{
#define DRND DList.XFor (d)
	dlistNode *d;
	// The method also known as:
	// "shake rattle 'n roll man in the middle gets nothing
	// and runs with scissors around shifting barrels"
	// a simple ordinary everyday rule applied in the
	// field of informatics for an efficient algorithm
	for (d = DList.Start; d; d = DList.Next (d))
		DRND->setXY (DRND->X - X + x, DRND->Y - Y + y);
	drawable::setXY (x, y);
}

bool dcont::onclick ()
{
	if (!checkclick ()) return false;

	dlistNode *d;
	for (d = DList.Start; d; d = DList.Next (d))
		if (DList.XFor (d)->onclick ())
			break;

	return true;
}

void dcont::dcleanup ()
{
	while (DList.End) {
		delete DList.XFor (DList.End);
		DList.dremove (DList.End);
	}
}

bool dcont::operator == (drawable *d)
{
	if (d->X >= X && d->X <= X + W && d->Y >= Y && d->Y <= Y + H) {
		dlistNode *n;
		for (n = DList.Start; n; n = DList.Next (n))
			if ((*DList.XFor (n)) == d)
				return true;
	}
	return false;
}

dcont::~dcont ()
{
	dcleanup ();
}

vcont::vcont () : dcont ()
{ }

void vcont::settle ()
{
	dlistNode *d;
	W = H = 0;
	for (d = DList.Start; d; d = DList.Next (d)) {
		setXYd (DList.XFor (d), 0, H);
		H += DList.XFor (d)->H + 2;
		if (DList.XFor (d)->W > W) W = DList.XFor (d)->W;
	}
}

hcont::hcont () : dcont ()
{ }

void hcont::settle ()
{
	dlistNode *d;
	W = H = 0;
	for (d = DList.Start; d; d = DList.Next (d)) {
		setXYd (DList.XFor (d), W, 0);
		W += DList.XFor (d)->W + 2;
		if (DList.XFor (d)->H > H) H = DList.XFor (d)->H;
	}
}

// ::::::: automatic placement container

acont::acont (int w, int h) : dcont ()
{
	W = w, H = h;
	peri = false;
}

void acont::settle ()
{
#define FLOATING DList.XFor (DList.End)
#define LAST DList.XFor (last)
	dlistNode *last = NULL;
	int sgx, sgy;

	if (DList.cnt > 1)
		for (last = DList.Prev (DList.End); last;
		     last = DList.Prev (last))
			if (DList.XFor (last)->W > 0)
				break;

	if (!last) {
		sgy = Bthick + 2;
		sgx = Bthick + 2;
	} else {
		sgx = LAST->X - X + LAST->W + 2;
		sgy = LAST->Y - Y;
		if (FLOATING->W + sgx > W) {
			int mh = 0;

			sgx = Bthick + 2;
			for (dlistNode *d = last; d; d = DList.Prev (d)) {
				if (DList.XFor (d)->H > mh)
					mh = DList.XFor (d)->H;
				if (DList.XFor (d)->X - X == Bthick + 2)
					break;
			}
			sgy += mh + 2;
		}
	}

	if (sgx + FLOATING->W > W || sgy + FLOATING->H > H) {
		Logprintf ("Drawable with width=%i Height=%i not"
			   " included\n", FLOATING->W, FLOATING->H);
		// If it does not fit, make its Width 0
		// degrade it from 2 dimensions to 1 dimension
		// and thus halt its existance in the 2D space.
		// We have the right to do so because WE ARE the
		// programmers of the system and it will help the other
		// entities evolve and the system flurish.
		// (I dlike to see the person in charge)
		FLOATING->W = 0;
	} else setXYd (FLOATING, sgx, sgy);
}

// :::::::::::::: tbutton
// A drawable box which contains 1-line small text
// whos purpose is to do something when clicked.
// The `do something' is to be defined at childs
// ::::::::::::::::::::::::::::::::::::::::::::::::

tbutton::tbutton (int f, char *t)
{
	fg = 0;
	bcol = 7;
	font = f;
	label = t;
	W = twidth (f, label) + 2 * Bthick;
	H = fheight (f) + 2 * Bthick;
}

tbutton::tbutton (int f, char *t, int l)
{
	font = f;
	label = new char [l + 1];
	fit_text ('c', t, l, label);
	W = twidth (f, label) + 2 * Bthick;
	H = fheight (f) + 2 * Bthick;
}

void tbutton::draw ()
{
	uperigram ();
	fill_block (bcol);
	drawstring (label, Bthick +1, Bthick, font, fg);
}

bool tbutton::onclick ()
{
	if (!checkclick ()) return false;

	drawrect (0, 0, W, H, 0);
	drawrect (1, 1, W - 2, H - 2, bcol);
	Gflush ();

	while (nailPointer ());

	uperigram ();
	action ();

	return true;
}

// :::::::::::::: checkbox
// a tbutton with a state and a virtual on/off draw
// this virtuallity can result in radio buttons with
// a simple code snippet.
// :::::::::::::::::::::::::::::::::::::::::::::::::

checkbox::checkbox (int f, char *l) : tbutton (f, l)
{
	W += 2 * fwidth (f);
	state = false;
}

void checkbox::draw ()
{
	tbutton::draw ();
	drawstate ();
}

bool checkbox::onclick ()
{
	if (!checkclick ()) return false;

	state = !state;
	drawstate ();

	return true;
}

void checkbox::drawstate ()
{
#define BSX (fwidth (font))
#define BSY (fheight (font))
#define BXX (W - Bthick - 3 * fwidth (font) / 2)

	fillrect (BXX, Bthick, BSX, BSY, 1);
	drawrect (BXX, Bthick, BSX, BSY, 0);
	if (state) {
		drawline (BXX + 1, Bthick + BSY / 2, BXX + BSX / 2,
			  Bthick + BSY -1, 0);
		drawline (BXX + BSX / 2, Bthick + BSY -1, BXX + BSX - 1,
			  Bthick + 1, 0);
	}
}

// :::::::::::::: radiobox
// ops here it is. The radio buttons stuff
// The radio manager knows the active of
// the buttons.
// :::::::::::::::::::::::::::::::::::::::

radioManager::radioManager ()
{
	current = NULL;
}

void radioManager::setcur (radiobox *r)
{
	if (current) {
		current->state = false;
		current->drawstate ();
	}
	current = r;
	current->state = true;
	current->drawstate ();
}

radiobox::radiobox (int f, char *t, radioManager *r, bool s) : checkbox (f, t)
{
	M = r;
	state = s;
}

void radiobox::drawstate ()
{
	drawable D (X + BXX, Y + Bthick, BSX, BSY);
	if (state) {
		D.fill_block (0);
		D.dperigram ();
	} else {
		D.fill_block (1);
		D.uperigram ();
	}
}

bool radiobox::onclick ()
{
	if (checkclick ()) M->setcur (this);
	else return false;
	return true;
}

// :::::::::::::: vscreen
// This is a drawable area that we put text into it.
// The text font is standard so we can use character cooridinates.
// The object does not store its text. If exposed (erased by other
// graphics), the vscreen ancestor shall recalculate each line.
// draw == cls therefore
// The `do something' is to be used at childs or other drawables
// which contain a vscreen.
// This object translates the click X, Y to
// character cooridinates cX, cY
// :::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

vscreen::vscreen (int f, int w, int h, int b, int ff)
{
	font = f;
	tX = w, tY = h;
	dX = fwidth (f), dY = fheight (f);
	W = dX * tX + 2 * Bthick;
	H = dY * tY + 2 * Bthick + 1;
	bg = b;
	fg = ff;
}

void vscreen::cls ()
{
	fill_block (bg);
}

void vscreen::puttext (unsigned int x, unsigned int y, char *t, int c, bool r)
{
	if (c == -1) c = fg;

	if (y < tY && x < tX)
		if (x + strlen (t) * dX < tX)
			bdrawstring (t, Bthick +1 + x * dX, Bthick +1 + y * dY,
				     font, (r) ? c : bg, (r) ? bg : c);
		else {
			char tt [tX - x + 1];
			tt [tX - x] = 0;
			strncpy (tt, t, tX - x);
			bdrawstring (tt, Bthick +1 + x * dX, Bthick +1 + y * dY,
				     font, (r) ? c : bg, (r) ? bg : c);
		}
}

void vscreen::pcursor (unsigned int r, unsigned int c, char a)
{
	trcursor (Bthick + 1 + r * dX, Bthick + 1 + c * dY, a);
}

void vscreen::draw ()
{
	dperigram ();
	cls ();
}

bool vscreen::onclick ()
{
	if (!checkclick ()) return false;

	cX = (clickX - X + 1) / dX;
	cY = (clickY - Y + 1) / dY;
	return true;
}

// :::::::::::::: Xreadline
// This drawable takes care of drawing a readline - text input box.
// Draws the prompt, the text, gets the cursor, etc.
// ::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

Xreadline::Xreadline (int f, int w, int b1, int f1, int b2, int f2)
: DA ()
{
	font = f;
	W = fwidth (f) * w + 2 * Bthick;
	DA.H = 2 * Bthick + fheight (f) - 1;
	H = DA.H + 2 * Bthick + 6;
	bg1 = b1, fg1 = f1;
	bg2 = b2, fg2 = f2;
	Width = (W - 2 * Bthick - twidth (font, prompt)) / fwidth (font) - 3;
	DA.W = Width * fwidth (font) + 2 * Bthick - 1;
}

void Xreadline::do_enter ()
{ }

void Xreadline::draw ()
{
	uperigram ();
	fill_block (bg1);
	drawstring (prompt, Bthick +1, Bthick + 4, font, fg1);
	DA.dperigram ();
	present ();
}

void Xreadline::setXY (int x, int y)
{
	drawable::setXY (x, y);
	setXYd (&DA, Bthick + twidth (font, prompt) + fwidth (font), Bthick +3);
}

void Xreadline::getCursor ()
{
	if (JCursor::Readline != this) {
		JCursor::hide ();
		JCursor::changerl (this, font, fg2, bg2);
		JCursor::show ();
	}
}

bool Xreadline::onclick ()
{
	if (!checkclick ()) return false;

	getCursor ();

	return true;
}

void Xreadline::present ()
{
	char vw [Width + 1];
	int i = line.linelen - Vcur;

	vw [Width] = 0;
	strncpy (vw, line.line + Vcur, Width);

	while (i < Width) vw [i++] = ' ';
	DA.bdrawstring (vw, Bthick, Bthick, font, fg2, bg2);

	if (JCursor::Readline == this)
		DA.trcursor (Bthick + (line.cursor - Vcur) * fwidth (font),
			     Bthick, (line.line [line.cursor]) ?: ' ');
}

Xreadline::~Xreadline ()
{
	if (JCursor::Readline == this) JCursor::Readline = NULL;
}

// :::::::::::::: up & down arrows
// These are small drawables with <> signs.
// The are totally dummy except from testing their onclick()
// The scroller will set their width & height
// :::::::::::::::::::::::::::::::::::::::::::::::::::::::::

void uparrow::draw ()
{
	uperigram ();
	fill_block (0);
	drawline (Bthick +1,  H - Bthick, W / 2, Bthick +1, COL_YELLOW);
	drawline (W / 2, Bthick +1, W - Bthick -1, H - Bthick, COL_YELLOW);
}

void downarrow::draw ()
{
	uperigram ();
	fill_block (0);
	drawline (Bthick, Bthick, W / 2, H - Bthick, COL_YELLOW);
	drawline (W / 2, H - Bthick, W - Bthick, Bthick, COL_YELLOW);
}

// :::::::::::::: scroller
// Maintains an up and a down arrow, sets their position.
// Prints current/maximum and acts as one single drawable.
// On click somebody else will do something and notify the scroller
// about the new (current, maximum)
// ::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

int scroller::scrollFixed = 12;

scroller::scroller (int vw)
{
	vwin = vw;
	c = max = 0;
}

void scroller::setXY (int x, int y)
{
	drawable::setXY (x, y);
	setXYd (&U, 0, 0);
	setXYd (&D, W - scrollFixed, H - scrollFixed);
}

void scroller::let (int i, int j)
{
	c = i, max = j;
	if (max > vwin) {
		barpos = vlen * c / max;
		barlen = vlen * vwin / max;
		if (barlen < 3) barlen = 3;
	} else {
		barpos = 0;
		barlen = vlen - 1;
	}

	draw ();
}

vscrollbar::vscrollbar (int d, int vw) : scroller (vw)
{
	barlen = vlen = d - 2 * scrollFixed - 1;
	barpos = 0;
	W = scrollFixed;
	H = d;
	D.W = U.W = W;
	D.H = U.H = scrollFixed;
}

void vscrollbar::draw ()
{
	fill_block (c_scr_bg);
	drawrect (1, 1 + scrollFixed + barpos, scrollFixed - 2, barlen,
		  COL_BLACK);
	fillrect (2, 2 + scrollFixed + barpos, scrollFixed - 3, barlen - 1,
		  COL_DARKGREY);
	D.draw ();
	U.draw ();
}

// :::::::::::::: gsign
// Output only area displaying text. It does not do
// anything except inform the user about "something"
// :::::::::::::::::::::::::::::::::::::::::::::::::::

gsign::gsign (int w, int h, int f, int cfg, int cbg)
: vscreen (f, w, h, cbg, cfg)
{
	txt = NULL;
}

static int nstrlen (char *c, int ml)
{
	int i;
	for (i = 0; i < ml && c [i] != 0; i++);
	return i;
}

void gsign::draw ()
{
	altperigram (8, 1);
	cls ();

	if (!txt) return;

	unsigned int i;
	char *c, *p, cc;

	for (i = 0, c = txt; i < tY; i++) {
		if ((p = strchr (c, '\n'))) {
			if (p - c > (int) tX)
				p = c + tX;
		} else p = c + nstrlen (c, tX);
		cc = *p;
		*p = 0;
		puttext (0, i, c, fg);
		*p = cc;
		if (cc == 0) break;
		c = p;
		if (cc == '\n') c++;
	}
}

gsign::~gsign ()
{
	iStrDelete (txt);
}
