/******************************************************************************

	coded macros for gui construction. Constructed graphical objects
	are graphical entry points to the code or graphical storage entries.

******************************************************************************/
/**************************************************************************
 Copyright (C) 2000 Stelios Xantkakis
**************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "lnshell.h"
#include "sysutil.h"
#include "projectlog.h"
#include "epil.h"
#include "graphics.h"
#include "editor.h"
#include "priv.h"
#include "rows.h"
#include "popups.h"

static dlistAuto<dcont*> *ContList;

static bool ontop (drawable*);

static char noegui [] = "No egui under construction";

static void check_gui ()
{
	if (!ContList) throw epilException (noegui);
}

static char bval [] = "Supplied value out of valid range";

static int validn (int m, int i, int M)
{
	if (i < m || i > M) throw epilException (bval);
	return i;
}

static int validn (int m, char *s, int M)
{
	return validn (m, strtoul (s, NULL, 10), M);
}

extern int c_scr_bg, c_scr_fg, c_scr_rv;

//****************************************
// right button
//****************************************

static void coded_rightbutton ()
{
	ep_boolean = rightButton;
}

//****************************************
// \xload
// If epil is locked we do not want
// redefinition of on-click button's
// code.
//****************************************

static bool xload;

static void coded_xload ()
{
	ep_check_locked ();

	xload = true;
}

//*****************************************
// A graphical object's code may choose to
// reload the object's code, or even
// destroy the object -- recreate the sheet.
// This is normally a dangerous feature.
// Before any object's code exec_unit we
// duplicate the macro code to stack frame.
//*****************************************

#define TMPCOPY(X, Y)\
	char *X = (char*) alloca (strlen (Y) + 1);\
	strcpy (X, Y)

//**************************************************************************
// Predeclare the sheet drawable because widgets may need to access the
// maximum sheet dimensions is sX, sY, sW, sH
//**************************************************************************

class edrawable : public drawable, public SurfaceObject, dlistNodeAuto
{
#define NBACKS 3
	char *ecmd;
	acont I;
static	breadline *Static;
static	void stdCursor ();
   public:
	edrawable (char*, char*, char*);
	void draw ();
	void activate ();
	bool onclick ();
	void refresh (char*);
	bool operator == (drawable*);
	~edrawable ();

	char *Title;

static	tbutton *KillButton, *BackButton [NBACKS];
static	void Init (int, int, int, int, breadline*);
static	void Draw ();
static	bool Onclick ();
static	int sX, sY, sW, sH;
};

//**************************************************************************
// \button macro
// #OBJECT		: nothing
// \xload #OBJECT	: set on-click command
//**************************************************************************

class ebutton : public tbutton, SurfaceObject
{
	char *ecmd;
	void activate ();
	void action ();
   public:
	ebutton (char*, char*, char*);
virtual	~ebutton ();
};

ebutton::ebutton (char *t, char *c, char *o) : tbutton (0, t)
{
	ecmd = c;
	appear (o);
}

void ebutton::action ()
{
	TMPCOPY(c, ecmd);
	Zexec_unit (c);
}

void ebutton::activate ()
{
	if (xload) {
		char *e;
		xload = false;
		DAT_POP_ALLOC(e);
		delete [] ecmd;
		ecmd = e;
	} else Logprintf ("\\button: `%s'\n", ecmd);
}

ebutton::~ebutton ()
{
	delete [] label;
	delete [] ecmd;
}

static char haveobj [] = "Object exists: ";

static void coded_button ()
{
	char *t, *a, *o;
	TMP_POP_ALLOC(t);
	TMP_POP_ALLOC(a);
	TMP_POP_ALLOC(o);

	check_gui ();

	if (ep_haveobject (o)) throw epilException (haveobj, o);

	(*ContList->XFor (ContList->End)) +=
		new ebutton (StrDup (t), StrDup (a), o);
}

//**************************************************************************
// void area
// affects only the placement
//**************************************************************************

static void coded_gvoid ()
{
	char *w, *h;
	TMP_POP_ALLOC(w)
	TMP_POP_ALLOC(h)

	check_gui ();

	(*ContList->XFor (ContList->End)) +=
		new drawable (0, 0, validn (1, w, 500), validn (1, h, 300));
}

static void coded_gvoidc ()
{
	char *w, *h;
	TMP_POP_ALLOC(w)
	TMP_POP_ALLOC(h)

	check_gui ();

	(*ContList->XFor (ContList->End)) +=
		new drawable (0, 0, fwidth (0) * validn (1, w, 80),
			      fheight (0) * validn (1, h, 30));
}

//**************************************************************************
// container blocks
//**************************************************************************

static void coded_vcont ()
{
	check_gui ();

	ContList->add (new vcont);
}

static void coded_hcont ()
{
	check_gui ();

	ContList->add (new hcont);
}

static void coded_closeblock ()
{
	check_gui ();

	if (ContList->Start == ContList->End && ContList->cnt == 1)
		Logputs ("\\closeblock w/o open block!\n");
	else {
		dcont *d = ContList->XFor (ContList->End);
		ContList->dremove (ContList->End);
		(*ContList->XFor (ContList->End)) += d;
	}
}

//**************************************************************************
// \checkbox macro
// #OBJECT		: set ep_boolean from state
// \xload #OBJECT	: set state 't'=true
//**************************************************************************

class echeckbox : public checkbox, SurfaceObject
{
	void activate ();
   public:
	echeckbox (char*, char*, bool);
};

echeckbox::echeckbox (char *o, char *t, bool s) : checkbox (1, t)
{
	W += 10;
	state = s;
	appear (o);
}

void echeckbox::activate ()
{
	if (xload) {
		char *c;
		xload = false;
		TMP_POP_ALLOC(c)
		state = *c == 't';
		if (ontop (this)) draw ();
	} else ep_boolean = state;
}

static void coded_checkbox ()
{
	char *t, *o, *s;
	TMP_POP_ALLOC(t);
	TMP_POP_ALLOC(o);
	TMP_POP_ALLOC(s);

	check_gui ();

	if (ep_haveobject (o)) throw epilException (haveobj, o);

	(*ContList->XFor (ContList->End)) +=
		new echeckbox (o, StrDup (t), (*s == 1));
}

//**************************************************************************
// \gsign
// #OBJECT	: print message to stdlog
// \xload	: load message
//**************************************************************************

class esign : public gsign, SurfaceObject
{
	void activate ();
   public:
	esign (char*, int, int, char*);
};

void esign::activate ()
{
	if (xload) {
		char *c;
		xload = false;
		TMP_POP_ALLOC(c);
		delete [] txt;
		txt = StrDup (c);
		if (ontop (this)) draw ();
	} else Logprintf ("\\esign: %s\n", txt);
}

esign::esign (char *o, int w, int h, char *t) : gsign (w, h, 1, 0, 7)
{
	appear (o);
	txt = StrDup (t);
}

static void coded_gsign ()
{
	char *w, *h, *o, *t;
	TMP_POP_ALLOC(w)
	TMP_POP_ALLOC(h)
	TMP_POP_ALLOC(o)
	TMP_POP_ALLOC(t)

	check_gui ();

	if (ep_haveobject (o)) throw epilException (haveobj, o);

	(*ContList->XFor (ContList->End)) +=
		new esign (o, validn (2, w, 80), validn (1, h, 30), t);
}

//**************************************************************************
//	 text entries
// \readline  - do something on enter
//	not an object
// \textinput - their text is accessible
// #OBJECT		: push text
// \xload #OBJECT	: set text
//**************************************************************************

#define VP virtual public
class etextinput : VP Xreadline, VP normtab,
		   VP history, SurfaceObject
{
	void activate ();
	void do_enter ();
   public:
	etextinput (char*, char*, int);
};

etextinput::etextinput (char *o, char *p, int w) :
breadline (p), Xreadline (0, w, 7, 0, c_scr_bg, c_scr_fg), history (15)
{
	appear (o);
}

void etextinput::activate ()
{
	if (xload) {
		char *c;
		xload = false;
		TMP_POP_ALLOC(c)
		addl ();
		line.reset (c);
		if (ontop (this)) present ();
	} else ep_push (line.line, line.linelen);
}

void etextinput::do_enter ()
{
	addl ();
}

class ereadline : VP Xreadline, VP normtab, VP history
{
	char *ecmd;
	void present ();
   public:
	ereadline (char*, char*, int, bool = false);
	void do_enter ();
	~ereadline ();
};

ereadline::ereadline (char *e, char *p, int w, bool pr) :
breadline (p), Xreadline (0, w, 7, 0, c_scr_bg, c_scr_fg), history (15)
{
	bg1 = COL_BLUE;
	ecmd = iStrDup (e);
}

void ereadline::present ()
{
	if (ontop (this)) Xreadline::present ();
}

void ereadline::do_enter ()
{
	addl ();
	ep_push (line.line, line.linelen);
	line.reset ();
	TMPCOPY(c, ecmd);
	Zexec_unit (c);
}

ereadline::~ereadline ()
{
	delete [] ecmd;
}

static void coded_textinput ()
{
	char *s, *p, *t, *o;
	TMP_POP_ALLOC(s)
	TMP_POP_ALLOC(p)
	TMP_POP_ALLOC(t)
	TMP_POP_ALLOC(o)

	check_gui ();

	if (ep_haveobject (o)) throw epilException (haveobj, o);

	Xreadline *x;

	(*ContList->XFor (ContList->End)) += (x = 
		new etextinput (o, p,
		validn (4, strlen (p) + validn (2, s, 80), 80)));

	if (*t) x->line.reset (t);
}

static void coded_readline ()
{
	char *s, *p, *e;
	TMP_POP_ALLOC(s)
	TMP_POP_ALLOC(p)
	TMP_POP_ALLOC(e)

	check_gui ();

	(*ContList->XFor (ContList->End)) +=
		new ereadline (StrDup (e), p,
		validn (4, strlen (p) + validn (2, s, 80), 80));
}

//**************************************************************************
// \rowarea, \row
// #OBJECT		: nothing
// \xload #OBJECT	: reload rowarea
//**************************************************************************

struct erow
{
	erow (char *t, char *e)	{ txt = t; ecmd = e; }
	char *txt, *ecmd;
	~erow ();
};

erow::~erow ()
{
	delete [] txt;
	delete [] ecmd;
}

static dlistAuto<erow*> *rowList;

static char norow [] = "no rowarea opened";

static void coded_row ()
{
	if (!rowList) {
		POP_OUT
		POP_OUT
		throw epilException (norow);
	}

	char *e, *t;
	TMP_POP_ALLOC(e)
	DAT_POP_ALLOC(t)

	rowList->add (new erow (t, StrDup (e)));
}

class rowArray : public Array
{
	erow **earr;
	int n;
	char *cindex (int);
   public:
	void makearray ();
	rowArray ();
	unsigned int total ();
	char *cmd (int);
	void clean ();
virtual	~rowArray ();
};

rowArray::rowArray ()
{
	earr = NULL;
}

void rowArray::makearray ()
{
	earr = (rowList->cnt) ? (new erow* [rowList->cnt]) : NULL;
	for (n = 0; rowList->cnt > 0; n++) {
		earr [n] = rowList->XFor (rowList->Start);
		rowList->dremove (rowList->Start);
	}
	rowList = NULL;
}

char *rowArray::cindex (int i)
{
	return earr [i]->txt;
}

unsigned int rowArray::total ()
{
	return n;
}

char *rowArray::cmd (int i)
{
	return (i < n) ? earr [i]->ecmd : NULL;
}

void rowArray::clean ()
{
	if (earr) {
		while (n) delete earr [--n];
		delete [] earr;
	}
}

rowArray::~rowArray ()
{
	clean ();
}

class erowarea : public rowarea, SurfaceObject
{
	char *gcmd;
	rowArray A;
	void rowselect ();
	void activate ();
	void drawtxt (int, char*);
   public:
	erowarea (int, int, char*, char*);
virtual	~erowarea ();
static	char RV;
};

char erowarea::RV = '\b';

static void rowscode (rowArray *A)
{
	char *c;
	TMP_POP_ALLOC(c)

	dlistAuto<erow*> E;
	rowList = &E;

	if (Zexec_unit (c)) {
		rowList = NULL;
		for (dlistNode *d = E.Start; d; d = E.Next (d))
			delete E.XFor (d);
	} else {
		A->clean ();
		A->makearray ();
	}
}

erowarea::erowarea (int w, int h, char *g, char *o) : rowarea (0, w, h, &A)
{
	gcmd = StrDup (g);
	appear (o);
	rowscode (&A);
}

void erowarea::rowselect ()
{
	if (A.cmd (VSCR.cY + iA)) {
		TMPCOPY(c, A.cmd (VSCR.cY + iA));
		TMPCOPY(s, gcmd);
		if (Zexec_unit (c) == 0) {
			if (*gcmd) Zexec_unit (s);
		}
	}
}

void erowarea::activate ()
{
	if (xload) {
		xload = false;
		rowscode (&A);
		if (ontop (this)) draw ();
	}
}

void erowarea::drawtxt (int y, char *t)
{
	if (!t) return;
	char *c;
	int i = 0;
	int col = c_scr_fg;
	while (*t) {
		c = strchr (t, RV);
		if (c) *c = 0;
		VSCR.puttext (i, y, t, col);
		if (!c) break;
		col ^= (c_scr_fg ^ c_scr_rv);
		*c = RV;
		i += (c - t);
		t = c + 1;
	}
}

erowarea::~erowarea ()
{
	delete [] gcmd;
}

static char rowop [] = "\\rowarea another one is opened";

static void coded_rowarea ()
{
	char *w, *h, *o, *g;
	TMP_POP_ALLOC(w)
	TMP_POP_ALLOC(h)
	TMP_POP_ALLOC(g)
	TMP_POP_ALLOC(o)

	check_gui ();

	if (rowList)
		throw epilException (rowop);

	if (ep_haveobject (o)) throw epilException (haveobj, o);

	(*ContList->XFor (ContList->End)) +=
		new erowarea (validn (1, w, 80), validn (1, h, 25), g, o);
}

static void coded_rv ()
{
	ep_push (&erowarea::RV, 1);
}

//**************************************************************************
// any person with ee background should know that
// the radio manager should be called \transistor
//
// RADIOMAMANGEROBJECT \radioopen
//	#RMOBJECT	: execute active radio button command
// OBJECT CMD TEXT \radio
//	#OBJECT		: nothing
// \xload #OBJECT	: set this one to be the active radio
//**************************************************************************

class eradioManager : public radioManager, SurfaceObject
{
	void activate ();
   public:
	eradioManager (char*);
};

static radioManager *srM;

class eradiobox : public radiobox, SurfaceObject
{
	void activate ();
   public:
	eradiobox (char*, char*, char*);
	char *ecmd;
virtual	~eradiobox ();
};

eradiobox::eradiobox (char *t, char *e, char *o) : radiobox (1, t, srM)
{
	ecmd = e;
	if (!srM->current) {
		srM->current = this;
		state = true;
	}
	appear (o);
}

void eradiobox::activate ()
{
	if (xload) {
		xload = false;
		if (ontop (this)) M->setcur (this);
		else {
			M->current->state = false;
			M->current = this;
		}
	}
}

eradiobox::~eradiobox ()
{
	delete [] ecmd;
	if (state)
		delete (eradioManager*) M;
}

eradioManager::eradioManager (char *o) : radioManager ()
{
	appear (o);
}

void eradioManager::activate ()
{
	TMPCOPY(c, (((eradiobox*) current)->ecmd));
	exec_unit (c);
}

static char radop [] = "\\radioopen already opened!";

static void coded_radioopen ()
{
	char *o;
	TMP_POP_ALLOC(o);

	check_gui ();

	if (srM) throw epilException (radop);

	if (ep_haveobject (o)) throw epilException (haveobj, o);

	srM = new eradioManager (o);
}

static char norad [] = "No radio open!";

static void coded_radio ()
{
	char *t, *e, *o;
	TMP_POP_ALLOC(t)
	TMP_POP_ALLOC(e)
	TMP_POP_ALLOC(o)

	check_gui ();

	if (!srM) throw epilException (norad);

	if (ep_haveobject (o)) throw epilException (haveobj, o);

	(*ContList->XFor (ContList->End)) +=
		new eradiobox (StrDup (t), StrDup (e), o);
}

static void coded_radioclose ()
{
	check_gui ();

	if (!srM) throw epilException (norad);

	srM = NULL;
}

//**************************************************************************
// \editor macro
// #OBJECT	: push editor text to arg stack
// \xload	: load editor text from arg
//**************************************************************************

class eeditor : public drawable, SurfaceObject
{
	void activate ();
	editor E;
	char *title;
   public:
	eeditor (char*, char*, int, int);
	void draw ();
	bool onclick ();
	void setXY (int, int);
	~eeditor ();
};

void eeditor::activate ()
{
	char *c;
	if (xload) {
		xload = false;
		TMP_POP_ALLOC(c)
		E.load_text (c);
		if (ontop (this)) draw ();
	} else {
		int i;
		c = (char*) alloca (1 + (i = E.textsize ()));
		E.textcpy (c);
		ep_push (c, i);
	}
}

eeditor::eeditor (char *t, char *o, int w, int h) : E (w - 2, h - 2)
{
	W = 2 * fwidth (1) + E.W;
	H = 2 * fheight (1) + E.H;
	title = iStrDup (t);
	if (strlen (title) >= (unsigned int) w) title [w] = 0;
	appear (o);
}

void eeditor::setXY (int x, int y)
{
	drawable::setXY (x, y);
	setXYd (&E, fwidth (0), 3 * fheight (0) / 2);
}

void eeditor::draw ()
{
	uperigram ();
	fill_block (7);
	drawstring (title, Bthick + 10, Bthick + 1, 1, 0);
	E.draw ();
}

bool eeditor::onclick ()
{
	if (!checkclick ()) return false;

	E.onclick ();

	return true;
}

eeditor::~eeditor ()
{
	delete [] title;
}

static void coded_editor ()
{
	char *w, *h, *o, *t;
	TMP_POP_ALLOC(w)
	TMP_POP_ALLOC(h)
	TMP_POP_ALLOC(o)
	TMP_POP_ALLOC(t)

	check_gui ();

	if (ep_haveobject (o)) throw epilException (haveobj, o);

	(*ContList->XFor (ContList->End)) +=
		new eeditor (t, o, validn (1, w, 80), validn (1, h, 25));
}

//**************************************************************************
// \menu
// \item \ep \menusep
//**************************************************************************

// :::::::::::::
// Execute code after the menus have closed
// :::::::::::::::::::::::::::::::::::::::::

class menudo : public postdo
{
	char *ecmd;
   public:
	void Load (char*);
	void Do ();
};

void menudo::Load (char *c)
{
	ecmd = StrDup (c);
}

void menudo::Do ()
{
	if (ecmd) {
		exec_unit (ecmd);
		delete [] ecmd;
		ecmd = NULL;
	}
}

static menudo exec_after;

// :::::::::::
// Intermediate storage from \item
// Creation of emenu from item list
// ::::::::::::::::::::::::::::::::

struct menurow
{
	char *t, *e;
	bool xp;
	menurow (char*, char*, bool);
	~menurow ();
};

menurow::menurow (char *txt, char *cmd, bool b)
{
	t = StrDup (txt);
	e = StrDup (cmd);
	xp = b;
}

menurow::~menurow ()
{
	delete [] t;
	delete [] e;
}

static struct menutmp_s {
	dlistAuto<int>		sList;
	dlistAuto<menurow*>	rList;
	bool expanding;
} *menutmp;


static char nomr [] = "\\item called but no \\menu under construction";

static void coded_item ()
{
	char *t, *e;
	TMP_POP_ALLOC(t);
	TMP_POP_ALLOC(e);

	if (!menutmp) throw epilException (nomr);

	menutmp->rList.add (new menurow (t, e, menutmp->expanding));

	menutmp->expanding = false;
}

static void coded_ep ()
{
	if (menutmp) menutmp->expanding = true;
}

static void coded_menusep ()
{
	if (menutmp && menutmp->rList.cnt && (!menutmp->sList.cnt
	|| menutmp->sList.XFor (menutmp->sList.End) != menutmp->rList.cnt))
		menutmp->sList.add (menutmp->rList.cnt - 1);
}

class emenu : public menu
{
	menurow **MR;
	int *SP;
   public:
	emenu ();
	int separator (int);
	char *output (int);
	menu *input (int);
	bool expands (int);
virtual	~emenu ();
};

static menu *menucode (char *c)
{
	menutmp_s M;
	M.expanding = false;
	menutmp = &M;

	if (Zexec_unit (c) || M.rList.cnt == 0) {
		menutmp = NULL;
		for (dlistNode *d = M.rList.Start; d;
		     d = M.rList.Next (d))
			delete M.rList.XFor (d);
		if (M.rList.cnt == 0) Logputs ("No elements in \\menu\n");
		return NULL;
	}

	return new emenu;
}

emenu::emenu ()
{
	reconstructable = true;

	MR = new menurow* [menutmp->rList.cnt];

	SP = ((nsep = menutmp->sList.cnt))
		? new int [menutmp->sList.cnt] : NULL;

	dlistNode *d;
	n = 0;
	for (d = menutmp->sList.Start; n < nsep;
	     d = menutmp->sList.Next(d), n++)
		SP [n] = menutmp->sList.XFor (d);

	n = 0;
	for (d = menutmp->rList.Start; d;
	     d = menutmp->rList.Next (d), n++)
		MR [n] = menutmp->rList.XFor (d);

	if (nsep && SP [nsep - 1] == n - 1) nsep--;

	menutmp = NULL;
}


int emenu::separator (int i)
{
	return SP [i];
}

char *emenu::output (int i)
{
	return MR [i]->t;
}

menu *emenu::input (int i)
{
	if (!MR [i]->xp) {
		menu::P = &exec_after;
		exec_after.Load (MR [i]->e);
		return NULL;
	}

	return menucode (MR [i]->e);
}

bool emenu::expands (int i)
{
	return MR [i]->xp;
}

emenu::~emenu ()
{
	for (int i = 0; i < n; i++)
		delete MR [i];

	delete [] MR;

	if (SP) delete [] SP;
}

static char menucr [] = "\\menu already under construction";

static void coded_menu ()
{
	char *c;
	TMP_POP_ALLOC(c)

	// No dangerous popup
	ep_check_locked ();

	if (menutmp) throw epilException (menucr);

	menu *m = menucode (c);

	if (m)
		menuhierarchy M (m, edrawable::sX, edrawable::sY,
				 edrawable::sW - 2, edrawable::sH - 2,
				 2, clickX, clickY);
}

//**************************************************************************
// All the static void codeds from above are called here but
// indirectly from the epil parser.
//**************************************************************************

static char eguifail [] = "Exception in \\egui";

static void build_egui (dcont *d, char *ecode)
{
	dlistAuto<dcont*> DL;
	ContList = &DL;
	DL.add (d);

	bool success = Zexec_unit (ecode) == 0;

	while (DL.Start != DL.End) {
		(*DL.XFor (DL.Prev (DL.End))) += (drawable*) DL.XFor (DL.End);
		DL.dremove (DL.End);
	}

	if (!success) {
		d->dcleanup ();
		ContList = NULL;
		throw epilException (eguifail);
	}

	ContList = NULL;
}

//**************************************************************************
// egui parent drawables + macro code for create & refresh.
//**************************************************************************

static dlist edrList;

tbutton *edrawable::KillButton, *edrawable::BackButton [NBACKS];

static bool ontop (drawable *d)
{
	if (!edrList.Start) return false;
	return (*((edrawable*) edrList.Start)) == d;
}

static bool isgui (char *o)
{
	for (dlistNode *d = edrList.Start; d; d = edrList.Next (d))
		if (!strcmp (o, ((edrawable*) d)->oname ())) return true;
	return false;
}

int edrawable::sX, edrawable::sY, edrawable::sW, edrawable::sH;
breadline *edrawable::Static;

edrawable::edrawable (char *o, char *e, char *t) :
drawable (sX, sY, sW, sH), dlistNodeAuto (&edrList, 0),
I (sW - 4 * Bthick, sH - fheight (3) - 4 * Bthick)
{
	Title = t;
	setXYd (&I, Bthick, Bthick + fheight (3));
	appear (o);
	build_egui (&I, ecmd = e);
}

void edrawable::stdCursor ()
{
	if (JCursor::Readline != Static) {
		JCursor::readlineDeath ();
		JCursor::Readline = Static;
	}
}

void edrawable::draw ()
{
	fill_block (7);
	uperigram ();
	drawstring (Title, 10, Bthick, 3, 0);
	I.draw ();
	stdCursor ();
}

void edrawable::activate ()
{
	if (xload) {
		char *c;
		xload = false;
		TMP_POP_ALLOC(c)
		refresh (c);
	}
	if (edrList.Start != this) stdCursor ();
	edrList.dremove (this);
	edrList.addtostart (this);
	Draw ();
}

void edrawable::refresh (char *e)
{
	if (*e && strcmp (e, ecmd)) {
		delete [] ecmd;
		ecmd = StrDup (e);
	}
	I.dcleanup ();
	build_egui (&I, ecmd);
	if (ontop (this)) draw ();
}

bool edrawable::onclick ()
{
	if (!checkclick ()) return false;

	I.onclick ();

	return true;
}

bool edrawable::operator == (drawable *d)
{
	return I == d;
}

edrawable::~edrawable ()
{
	leave (&edrList);
	delete [] ecmd;
	delete [] Title;
}

void edrawable::Init (int x, int y, int w, int h, breadline *b)
{
	Static = b;

	sX = x, sY = y, sW = w, sH = h;
	KillButton = new tbutton (0, " close ");
	KillButton->fg = 1; KillButton->bcol = 4;
	KillButton->setXY (sX + sW - 10 - fwidth (0) * 8, sY + Bthick + 2);

	char sn [] = "   ";

	for (y = 0; y < NBACKS; y++) {
		sn [1] = '1' + y;
		BackButton [y] = new tbutton (1, StrDup (sn));
		BackButton [y]->bcol = 4;
		x = (y == 0) ? (sX + sW - 10 - fwidth (0) * 25) :
		    (Bthick + BackButton [y - 1]->X + BackButton [y - 1]->W);
		BackButton [y]->setXY (x, sY + Bthick + 2);
	}
}

void edrawable::Draw ()
{
	if (edrList.Start) {
		((edrawable*) edrList.Start)->draw ();
		KillButton->draw ();
		for (int i = 0; i < NBACKS; i++)
			BackButton [i]->draw ();
	} else FillRect (sX, sY, sW + 1, sH + 1, 0);
}

bool edrawable::Onclick ()
{
	if (KillButton->onclick ()) {
		if (edrList.Start) {
			JCursor::Readline = NULL;
			delete ((edrawable*) edrList.Start);
			Draw ();
		}
		return true;
	}

	for (int i = 0; i < NBACKS;)
		if (BackButton [i++]->onclick ()) {
			if (edrList.cnt > i) {
				edrawable *e;
				for (e = (edrawable*) edrList.Start; i; i--)
					e = ((edrawable*) edrList.Next (e));
				edrList.swap (e, (edrawable*) edrList.Start);
				stdCursor ();
				Draw ();
			}
			return true;
		}

	if (edrList.Start) 
		return ((edrawable*) edrList.Start)->onclick ();
	return false;
}

static char eguii [] = "\\egui already under construction";
static char maxsheet [] = "\\egui too many open sheets";

#define MAX_SHEETS 1000

static void coded_egui ()
{
	char *o;
	TMP_POP_ALLOC(o)

	// No dangerous sheet magically created
	ep_check_locked ();

	if (ContList) throw epilException (eguii);

	if (edrList.cnt > MAX_SHEETS)
		throw epilException (maxsheet);

	if (ep_haveobject (o))
		if (isgui (o)) {
			POP_OUT
			xload = true;
			ep_call_object (o);
		}
		else {
			POP_OUT
			POP_OUT
			throw epilException (haveobj, o);
		}
	else {
		char *e, *t;
		TMP_POP_ALLOC(t)
		TMP_POP_ALLOC(e)
		new edrawable (o, StrDup (e), StrDup (t));
		edrawable::Draw ();
	}
}

static char noshit [] = "No sheets";

static void coded_sheetname ()
{
	if (!edrList.Start) throw epilException (noshit);
	((edrawable*) edrList.Start)->myName ();
}

//**************************************************************************
// Special complex widgets. Mostly rowareas displaying
// some kind of listing not directly available to the
// macro code.
//**************************************************************************

class objectRow : public rowarea, SurfaceObject
{
   protected:
	char *ecmd;
	void activate ();
	void rowselect ();
   public:
	objectRow (int, int, char*, char*, Array*);
	~objectRow ()	{ delete [] ecmd; }
};

objectRow::objectRow (int w, int h, char *e, char *o, Array *A) :
rowarea (0, w, h, A)
{
	appear (o);
	ecmd = StrDup (e);
}

void objectRow::rowselect ()
{
	if (VSCR.cY + iA < (int) A->total ()) {
		ep_push ((*A) [iA + VSCR.cY]);
		TMPCOPY(c, ecmd);
		Zexec_unit (c);
	}
}

void objectRow::activate ()
{
	if (xload) {
		char *e;
		xload = false;
		DAT_POP_ALLOC(e)
		delete [] ecmd;
		ecmd = e;
	} else Logprintf ("rowarea command : %s\n", ecmd);
}

// ::::::::::: row area of all the e-drawables from edrList

class A00 : public dlistArray
{
	char *cindex (int);
   public:
	A00 () : dlistArray (&edrList) { }
	char *oname (int);
};

char *A00::cindex (int i)
{
	return ((edrawable*) nth_node (i))->Title;
}

char *A00::oname (int i)
{
	L = NULL;
	return ((edrawable*) nth_node (i))->oname ();
}

static Array *IA00;

class edr_rowarea : public objectRow
{
	void rowselect ();
   public:
	edr_rowarea (int w, int h, char *e, char *o)
	: objectRow (w, h, e, o, IA00) { }
};

void edr_rowarea::rowselect ()
{
	if (VSCR.cY + iA < (int) A->total ()) {
		ep_push (((A00*) A)->oname (iA + VSCR.cY));
		TMPCOPY(c, ecmd);
		Zexec_unit (c);
	}
}

static void coded_egui_rows ()
{
	char *w, *h, *o, *e;
	TMP_POP_ALLOC(w)
	TMP_POP_ALLOC(h)
	TMP_POP_ALLOC(o)
	TMP_POP_ALLOC(e)

	check_gui ();

	if (ep_haveobject (o)) throw epilException (haveobj, o);

	if (!IA00) IA00 = new A00;

	(*ContList->XFor (ContList->End)) +=
		new edr_rowarea (validn (1, w, 80), validn (1, h, 25), e, o);
}

// ::::::::::: row area of all macros, variables, objects

static Array *IA01, *IA02, *IA03;

class macro_rows : public objectRow
{
	void drawtxt (int, char*);
   public:
	macro_rows (int, int, char*, char*);
};

void macro_rows::drawtxt (int i, char *t)
{
	if (t) VSCR.puttext (0, i, t, (ep_iscoded (t)) ? c_scr_fg : c_scr_rv);
}

macro_rows::macro_rows (int w, int h, char *e, char *o) :
objectRow (w, h, e, o, IA01)
{ }

static void coded_macro_rows ()
{
	char *w, *h, *o, *e;
	TMP_POP_ALLOC(w)
	TMP_POP_ALLOC(h)
	TMP_POP_ALLOC(o)
	TMP_POP_ALLOC(e)

	check_gui ();

	if (ep_haveobject (o)) throw epilException (haveobj, o);

	if (!IA01) IA01 = new dbsArray (&MacroTree);

	(*ContList->XFor (ContList->End)) +=
	new macro_rows (validn (1, w, 80), validn (1, h, 25), e, o);
}

static void coded_variable_rows ()
{
	char *w, *h, *o, *e;
	TMP_POP_ALLOC(w)
	TMP_POP_ALLOC(h)
	TMP_POP_ALLOC(o)
	TMP_POP_ALLOC(e)

	check_gui ();

	if (ep_haveobject (o)) throw epilException (haveobj, o);

	if (!IA02) IA02 = new dbsArray (&VarTree);

	(*ContList->XFor (ContList->End)) +=
	new objectRow (validn (1, w, 80), validn (1, h, 25), e, o, IA02);
}

static void coded_object_rows ()
{
	char *w, *h, *o, *e;
	TMP_POP_ALLOC(w)
	TMP_POP_ALLOC(h)
	TMP_POP_ALLOC(o)
	TMP_POP_ALLOC(e)

	check_gui ();

	if (ep_haveobject (o)) throw epilException (haveobj, o);

	if (!IA03) IA03 = new dbsArray (&ThisTree);

	(*ContList->XFor (ContList->End)) +=
	new objectRow (validn (1, w, 80), validn (1, h, 25), e, o, IA03);
}

// :::::::::::::: epil argument stack drawable object
// Special array as an interface to ep_examine_stack
// ::::::::::::::::::::::::::::::::::::::::::::::::::

class epargArray : public Array
{
	char *buffer;
	int wtrunc;
	char *cindex (int);
   public:
	epargArray (int);
	unsigned int total ();
	void setvar (int, char*);
};

char *epargArray::cindex (int i)
{
	int r;

	if ((r = ep_examine_stack (i, buffer + 1, wtrunc - 2)) == -1)
		return NULL;

	buffer [0] = '`';
	char *p = buffer + strlen (buffer);
	p [1] = 0;
	p [0] = '\'';

	if (r > wtrunc - 2)
		for (i = 1; i < 4; i++) *(p - i) = '.';

	return buffer;
}

epargArray::epargArray (int i)
{
	buffer = new char [1 + (wtrunc = i)];
}

unsigned int epargArray::total ()
{
	return ep_argstack;
}

void epargArray::setvar (int i, char *v)
{
	int l = ep_examine_stack (i, NULL, 0);

	if (l == -1) return;

	char *p;
	p = (char*) alloca (l + 1);
	ep_examine_stack (i, p, l);

	ep_push (p, l);
	ep_push (v);
	exec_unit ("\\setvar");
}

class Srl : VP Xreadline, VP novarrow, VP normtab
{
   public:
	Srl (char*, int);
};

Srl::Srl (char *p, int w) : breadline (p), Xreadline (0, w, 7, 0, 0, 1)
{ }

class dargument_stack : public drawable
{
	rowarea R;
	tbutton Ref, Ld;
	Srl ET;
   public:
	dargument_stack (int, int);
	void draw ();
	bool onclick ();
	void setXY (int, int);
};

dargument_stack::dargument_stack (int w, int h) :
R (0, w, h, new epargArray (w)), Ref (0, " refresh "),
Ld (0, " load to variable "), ET ("var:", w)
{
	Ld.bcol = Ref.bcol = 4;
	W = R.W + 4 * Bthick;
	H = R.H + Ref.H + Ld.H + ET.H + 2 * Bthick + 10;
}

void dargument_stack::draw ()
{
	uperigram ();
	fill_block (7);
	R.draw ();
	Ref.draw ();
	Ld.draw ();
	ET.draw ();
}

bool dargument_stack::onclick ()
{
	if (!checkclick ()) return false;

	if (Ref.onclick ()) {
		R.iA = 0;
		R.draw ();
		R.selected = -1;
	} else if (Ld.onclick ()) {
		if (ET.line.line [0] == 0)
			Logputs ("Variable name not set\n");
		else if (R.selected == -1)
			Logputs ("No argument selected\n");
		else ((epargArray*) R.A)->setvar (R.selected, ET.line.line);
	} else if (R.onclick () || ET.onclick ());

	return true;
}

void dargument_stack::setXY (int x, int y)
{
	drawable::setXY (x, y);
	setXYd (&Ref,	Bthick, Bthick);
	setXYd (&R,	Bthick, Bthick + Ref.H + 1);
	setXYd (&Ld,	Bthick, Bthick + Ref.H + R.H + 2);
	setXYd (&ET,	Bthick, Bthick + Ref.H + R.H + Ld.H + 3);
}

static void coded_stack_rows ()
{
	char *w, *h;
	TMP_POP_ALLOC(w)
	TMP_POP_ALLOC(h)

	check_gui ();

	(*ContList->XFor (ContList->End)) +=
	new dargument_stack (validn (1, w, 80), validn (1, h, 25));
}

//**************************************************************************
// Fast buttons.
// May redefine label and command on click, for any button  0..15
//**************************************************************************

#define NFB 16

class FastButton : public tbutton
{
	char *ecmd;
	void action ();
   public:
	FastButton ();
	void set (char*, char*);
static	int sW, sH;
static	void Init (int, int);
static	void Draw ();
static	bool Onclick ();
static	void coded_fbgen ();
};

static FastButton *FBA;

int FastButton::sW, FastButton::sH;

#define DVDR 2

void FastButton::Init (int x, int y)
{
	FBA = new FastButton [NFB];

	for (int i = 0; i < NFB; i++)
		FBA [i].setXY (x + (i / (NFB / DVDR)) * (1 + FBA [0].W),
			       y + 2 + (i % (NFB / DVDR)) * (1 + FBA [0].H));

	sW = 2 * FBA [0].W + 2 * Bthick;
	sH = 2 * Bthick + NFB * (1 + FBA [0].H) / 2;
}

void FastButton::Draw ()
{
	for (int i = 0; i < NFB; i++)
		FBA [i].draw ();
}

bool FastButton::Onclick ()
{
	if (clickX < sW && clickY < FBA [0].Y + sH && clickY > FBA [0].Y) {
		for (int i = 0; i < NFB; i++)
			if (FBA [i].onclick ()) break;
		return true;
	}

	return false;
}

FastButton::FastButton () : tbutton (1, StrDup ("   "))
{
	fg = 0;
	bcol = COL_BLUE;
	ecmd = NULL;
}

void FastButton::action ()
{
	if (ecmd) {
		TMPCOPY(c, ecmd);
		Zexec_unit (c);
	}
}

void FastButton::set (char *l, char *e)
{
	if (ecmd) delete [] ecmd;
	strncpy (label, l, 3);
	ecmd = StrDup (e);
}

void FastButton::coded_fbgen ()
{
	int i, j, k, l;
	char *c, *s, *q, L [6];

	for (j = k = i = 0; i < NFB; i++) {
		if (FBA [i].ecmd) {
			l = strlen (FBA [i].ecmd);
			if (k < l) k = l;
			j += l + 30;
		}
	}

	c = (char*) alloca (j);
	s = (char*) alloca (k + 30);
	q = (char*) alloca (k + 30);

	*c = 0;

	for (i = 0; i < NFB; i++)
		if (FBA [i].ecmd) {
			sprintf (s, "%s %s %i \\fastbutton\n",
				 ep_quote (FBA [i].ecmd, q),
				 ep_quote (FBA [i].label, L), i);
			strcat (c, s);
		}

	ep_push (c);
}

static void coded_fastbutton ()
{
	char *n, *l, *e;

	TMP_POP_ALLOC(n)
	TMP_POP_ALLOC(l)
	TMP_POP_ALLOC(e)

	ep_check_locked ();

	if (strtoul (n, NULL, 10) >= NFB) return;

	FBA [strtoul (n, NULL, 10)].set (l, e);
	FBA [strtoul (n, NULL, 10)].draw ();
}

//**************************************************************************
// Make sheets & fastbuttons appear as one drawable, where draw()
// draws everything and onclick() checks all of the widgets treewise
//**************************************************************************

class egui_as_drawable : public drawable
{
   public:
	egui_as_drawable (int, int, breadline*);
	void draw ();
	bool onclick ();
};

bool egui_as_drawable::onclick ()
{
	return edrawable::Onclick () || FastButton::Onclick ();
}

void egui_as_drawable::draw ()
{
	edrawable::Draw ();
	FastButton::Draw ();
}

egui_as_drawable::egui_as_drawable (int w, int h, breadline *b)
{
	edrawable::Init (1, 1, w * fwidth (0), h * fheight (0), b);
	FastButton::Init (1, edrawable::sH + 1);

	// hacks
	H = edrawable::sH;
	W = FastButton::sW;
}

//*****************************************
// Instance Sun Oct 15 16:59:24 EEST 2000
//*****************************************

drawable *egui (int w, int h, breadline *b)
{
	CODED(rightbutton);
	CODED(xload);

	CODED(button);
	CODED(gvoid);
	CODED(gvoidc);
	CODED(vcont);
	CODED(hcont);
	CODED(closeblock);
	CODED(checkbox);
	CODED(gsign);
	CODED(textinput);
	CODED(readline);

	CODED(rowarea);
	CODED(row);
	CODED(rv);

	CODED(radioopen);
	CODED(radio);
	CODED(radioclose);
	CODED(editor);

	CODED(item);
	CODED(ep);
	CODED(menusep);
	CODED(menu);

	CODED(egui);
	CODED(sheetname);

	CODED(egui_rows);
	CODED(macro_rows);
	CODED(variable_rows);
	CODED(object_rows);
	CODED(stack_rows);
	CODED(fastbutton);

	ep_register_coded ("fbgen", FastButton::coded_fbgen);

	ep_optimize_macrotree ();

	return new egui_as_drawable (w, h, b);
}
