/******************************************************************************

	X Windows wrapper to a "simple" library + main()

	X Windows prorgamming interface is encapsulated behind a generic
	graphics programming interface, providing:

	Init, Flush, Term
	DrawLine, DrawPoint, FillRect, DrawString, DrawRect, BDrawString

	with internal color & font managment, "color 1" "font 3".

	Although generic, the initialization takes care of custom X-only
	stuff, like one window opened with name "eShell". The initialization
	is not configurable but fixed to what eShell wants.

	Porting should be quite easy.

******************************************************************************/
/**************************************************************************
 Copyright (C) 2000 Stelios Xantkakis
**************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/Xatom.h>

#include "INIT.h"
#include "readline++.h"
#include "epil.h"
#include "egui.h"
#include "graphics.h"
#include "popups.h"

int scrX, scrY;

const static char WindowName [] = "eShell";

static Display *D;
static Window W;
static GC oneGConly;	/* ? the right way */

static struct {
	int col;
	const char *cname;
} colors [] = {
	{ 0, "black" },
	{ 0, "white" },
	{ 0, "dark orange" },
	{ 0, "sandy brown" },
	{ 0, "dark red" },
	{ 0, "blue" },
	{ 0, "dark green" },
	{ 0, "light grey" },
	{ 0, "dark grey" }
};

#define NCOLORS (sizeof colors / sizeof colors [0])

static char *FontNames [] = {
   "-adobe-courier-%s-r-normal--0-%i-%i-%i-m-0-iso8859-1",
   "-misc-fixed-%s-r-normal--0-%i-%i-%i-c-0-iso8859-1",
};

static struct {
	XFontStruct *F;
	int fname;
	bool bold;
	int ratio, den;
} fonts [] = {
   { NULL, 1, false, 1, 1 },
   { NULL, 0, false, 1, 1 },
   { NULL, 1, false, 10, 11 },
   { NULL, 0, true, 15, 10 },
   { NULL, 0, false, 10, 12 },
};

#define NFONTS (sizeof fonts / sizeof fonts [0])

static void init_colors ()
{
	unsigned int i;
	XColor g, d;
	Colormap cmap = DefaultColormap (D, XDefaultScreen (D));

	for (i = 0; i < NCOLORS; i++) {
		XAllocNamedColor (D, cmap, colors [i].cname, &g, &d);
		colors [i].col = g.pixel;
	}

	XGCValues v;
	v.fill_style = FillSolid;

	oneGConly = XCreateGC (D, W, GCFillStyle, &v);
}
	
/*
 * Try to find the largest viewable font that fits at least 80 characters
 * in the screen but its not bigger than 130 pt.
 * Everybody's got to be happy.
 */
static void init_fonts ()
{
#define	SC DefaultScreen (D)
	unsigned int i, pt;
	char t [100];
	int xres = (int) (((((double) DisplayWidth (D, SC)) * 25.4) /
		   ((double) DisplayWidthMM (D, SC))) + 0.5);
	int yres = (int) (((((double) DisplayHeight (D, SC)) * 25.4) /
		   ((double) DisplayHeightMM (D, SC))) + 0.5);

	pt = (72 * (int) ((double)DisplayWidthMM (D, SC) * 25.4)) / 80;

	if (pt > 130) pt = 130;

	for (i = 0; i < NFONTS; i++) {
		sprintf (t, FontNames [fonts [i].fname],
			(fonts [i].bold) ? "bold" : "medium",
			(pt * fonts [i].ratio) / fonts [i].den, xres, yres);
		if (!(fonts [i].F = XLoadQueryFont (D, t))) {
			fprintf (stderr, "Cannot load font \"%s\"\n", t);
			exit (1);
		}
	}
}

static void setfg (int c)
{
	XGCValues v;

	v.foreground = colors [c].col;
	XChangeGC (D, oneGConly, GCForeground, &v);
}

static void setbg (int c)
{
	XGCValues v;

	v.background = colors [c].col;
	XChangeGC (D, oneGConly, GCBackground, &v);
}

static void setfont (int f)
{
	XGCValues v;

	v.font = fonts [f].F->fid;
	XChangeGC (D, oneGConly, GCFont, &v);
}

//******************************* Interface **********************************

int fwidth (int f)
{
	// for "m" type fonts
	return (fonts [f].fname == 0) ? fonts [f].F->max_bounds.width :
	// for "c" type fonts
	(fonts [f].F->max_bounds.rbearing - fonts [f].F->min_bounds.lbearing);
}

int fheight (int f)
{
	return fonts [f].F->max_bounds.ascent 
	      +fonts [f].F->max_bounds.descent;
}

static int InitX (bool havelog, char *xs = NULL)
{
	if (!xs) xs = getenv ("DISPLAY");
	if (!(D = XOpenDisplay (xs))) return -1;

	init_fonts ();

	// this is the sophisticated trial & error procedure
	scrX = fwidth (0) * 91;
	scrY = fheight (0) * (41);
//	scrY = fheight (0) * ((havelog) ? 41 : 32);

	XSetWindowAttributes attr;
	attr.backing_store = Always;
	W = XCreateWindow (D, DefaultRootWindow (D), 0, 0, scrX, scrY, 1,
			   CopyFromParent, CopyFromParent, CopyFromParent,
			   CWBackingStore, &attr);

	XSizeHints sh;
	sh.min_width = sh.max_width = scrX;
	sh.min_height = sh.max_height = scrY;
	sh.flags = PMinSize | PMaxSize;
	XSetNormalHints (D, W, &sh);

	XMapWindow (D, W);
	XFlush (D);

	XStoreName (D, W, WindowName);

	init_colors ();

	return 0;
}

void Gflush ()
{
	XFlush (D);
}

void DrawLine (int x1, int y1, int x2, int y2, int col)
{
	setfg (col);
	XDrawLine (D, W, oneGConly, x1, y1, x2, y2);
}

void DrawPoint (int x, int y, int col)
{
	setfg (col);
	XDrawPoint (D, W, oneGConly, x, y);
}

void FillRect (int x1, int y1, int w, int h, int col)
{
	setfg (col);
	XFillRectangle (D, W, oneGConly, x1, y1, w, h);
}

void DrawRect (int x1, int y1, int w, int h, int col)
{
	setfg (col);
	XDrawRectangle (D, W, oneGConly, x1, y1, w, h);
}

static inline int fas (int f)
{
	return fonts [f].F->max_bounds.ascent;
}

void BDrawString (char *s, int x, int y, int font, int fg, int bg)
{
	setfg (fg);
	setbg (bg);
	setfont (font);
	XDrawImageString (D, W, oneGConly, x, y + fas (font), s, strlen (s));
}

void DrawString (char *s, int x, int y, int font, int fg)
{
	setfg (fg);
	setfont (font);
	XDrawString (D, W, oneGConly, x, y + fas (font), s, strlen (s));
}

int twidth (int f, char *s)
{
	return strlen (s) * fwidth (f);
}

//########################################################################

struct VImage {
	XImage *I;
	int X, Y;
	VImage (int, int, int, int);
	~VImage ();
};

VImage::VImage (int x, int y, int w, int h)
{
	I = XGetImage (D, W, X = x, Y = y, w, h, AllPlanes, ZPixmap);
}

VImage::~VImage ()
{
	XDestroyImage (I);
}

void *SaveUnders (int x, int y, int w, int h)
{
	int Rx, Ry;		// No rude BadMatch because area
	Window vwo;		// is partly outof screen
	XTranslateCoordinates (D, W, DefaultRootWindow (D),
				x, y, &Rx, &Ry, &vwo);

	if (Rx + w >= DisplayWidth (D, SC) || Ry + h >= DisplayHeight (D, SC))
		return NULL;

	return new VImage (x, y, w, h);
}

void RestoreUnders (void *I)
{
	XPutImage (D, W, oneGConly, ((VImage*) I)->I, 0, 0, 
		((VImage*) I)->X, ((VImage*) I)->Y,
		((VImage*) I)->I->width,
		((VImage*) I)->I->height);
}

void FreeUnders (void *I)
{
	delete ((VImage*) I);
}

//************************************************************************
// X has been entered smoothly and with wisdom
// Sun Sep 24 15:39:00 EEST 2000
//************************************************************************

static struct {
	unsigned int X, rl;
} X2rl [] = {
	{ XK_Left,      K_LEFT }, 
	{ XK_Right,     K_RIGHT },
	{ XK_Up,        K_UP },
	{ XK_Down,      K_DOWN },
	{ XK_Home,      K_HOME },
	{ XK_End,       K_END },
	{ XK_Tab,       '\t' },
	{ XK_BackSpace, '\b' },
	{ XK_Delete,    K_DEL },
	{ XK_Return,    '\n' },
	{ XK_Page_Up,	K_PGUP },
	{ XK_Page_Down,	K_PGDOWN },
	{ XK_Escape, 	K_ESC },
	{ 0,            0 }
};

#define NXTRANS sizeof X2rl / sizeof X2rl [0]

//********************* SAMPLE *******************

static void XPaste ()
{
	Window owner;
	unsigned char *data;
	Atom type;
	int d1, result;
	unsigned long d2, d3, len;
	XEvent e;

	if ((owner = XGetSelectionOwner (D, XA_PRIMARY)) != None)
	{
		XConvertSelection (D, XA_PRIMARY, XA_STRING, None, W,
				   CurrentTime);
		XSelectInput (D, W, NoEventMask);
		XFlush (D);
		XNextEvent (D, &e);
		XSelectInput (D, W, KeyPressMask + ButtonPressMask + 
				    ExposureMask + ButtonReleaseMask);
		XFlush (D);
		if (e.type != SelectionNotify) return;

		if (e.xselection.property != XA_STRING) return;

		XGetWindowProperty (D, W, XA_STRING, 0, 0, 0,
				    AnyPropertyType, &type, &d1, 
				    &d3, &len, &data);

		result = XGetWindowProperty (D, W, XA_STRING, 0, len,
					     0, AnyPropertyType, &type,
					     &d1, &d3, &d2, &data);

		if (result == Success) {
			char *p;

			p = (char*) alloca (len + 1);
			memcpy (p, data, len);
			p [len] = 0;

			XFree (data);

			Paste (p);
		}
	}
}

static void doKeyPress (XEvent &event)
{
	KeySym keysym;
	char keyin [2];
	int ch;

	XLookupString (&event.xkey, keyin, 2, &keysym, NULL);

	if (keysym >= XK_space && keysym <= XK_asciitilde)
		ch = keyin [0];
	else {
		ch = K_SKIP;
		for (unsigned int i = 0; i < NXTRANS; i++)
			if (X2rl [i].X == keysym) {
				ch = X2rl [i].rl;
				break;
			}
	}

	do_char (ch);
}

#define RCSLEEP 20
#define INISLEEP 100

int nailPointer ()
{
static	bool nailed = false;

	if (!nailed) {
		XSelectInput (D, W, ButtonReleaseMask);
		XFlush (D);
		usleep (INISLEEP * 1000);
		nailed = true;
	}

	if (XEventsQueued (D, QueuedAfterFlush)) {
		XEvent e;
		XNextEvent (D, &e);

		XSelectInput (D, W, KeyPressMask + ButtonPressMask + 
				    ExposureMask + ButtonReleaseMask);
		XFlush (D);
		nailed = false;
		return 0;
	}

	usleep (RCSLEEP * 1000);

	return 1;
}

void UserEvents (drawable *O)
{
	XEvent event;

	XNextEvent (D, &event);

	switch (event.type) {
	case KeyPress:
		doKeyPress (event);
		break;
	case ButtonPress:
		clickX = event.xbutton.x;
		clickY = event.xbutton.y;
		rightButton = event.xbutton.button == Button3;
		O->onclick ();
		if (event.xbutton.button == Button2)
			XPaste ();
		break;
	case Expose:
		gLevels::drawall ();
		Gflush ();
	default:
		break;
	}
}

static void runinitcode ()
{
	char *ecode;
	TMP_POP_ALLOC(ecode)

	Zexec_unit (ecode);
}

extern int open_mozilla (Display*, char*);

static void coded_openurl ()
{
	char *c;
	TMP_POP_ALLOC(c)

	ep_boolean = open_mozilla (D, c) == 1;
}

void coded_startx ()
{
static	bool Block = false;
	char *f;
	drawable *Base;
	TMP_POP_ALLOC(f)

	if (Block || InitX (*f != '0') == -1) {
		POP_OUT;
		ep_boolean = false;
		return;
	}

	Block = true;
	CODED(openurl);
	Gflush ();
	XSelectInput (D, W, KeyPressMask + ButtonPressMask + 
			    ExposureMask + ButtonReleaseMask);

	Base = InitEnv (*f != '0');
	Base->draw ();
	Gflush ();
	runinitcode ();

	while (!ExitRequested)
		UserEvents (Base);

	term_vlog ();
	XCloseDisplay (D);
	ep_boolean = true;
}
