#' Paste simple JavaScript plot code
#' 
#' This function is similar to \code{rk.paste.JS}, but adds some code parts to its output which
#' are commonly used to generate plots with RKWard.
#' 
#' The contents of the \code{...} argument are evaluated by \code{rk.paste.JS} and encapsulated
#' between \code{if(full)\{rk.graph.on()\} try(\{} and \code{\}) if(full)\{rk.graph.off()\}}. If generic
#' plot options are supplied, their \code{"code.preprocess"} and \code{"code.calculate"} modifiers are
#' also automatically taken care of, so you only need to include \code{"code.printout"} inside of
#' \code{...}.
#'
#' @param ... The actual plot code, passed through to \code{rk.paste.JS}.
#' @param plotOpts An object generated by \code{rk.XML.embed} or \code{rk.plotOptions}, i.e. embedded plot options.
#' @param printoutObj An \code{rk.JS.var} object fetching the \code{"code.printout"} modifier of \code{plotOpts}
#'		(see examples below!). If \code{NULL} and \code{plotOpts} is of class \code{rk.plot.opts} (as returned by \code{rk.plotOptions}),
#'		will be fetched from \code{plotOpts} automatically.
#' @param level Integer, which indentation level to use, minimum is 1.
#' @param indent.by A character string defining the indentation string to use.
#' @param empty.e For \code{rk.JS.ite} objects only: Logical, if \code{TRUE} will force to add empty \code{else \{\}} brackets when
#'		there is no \code{else} statement defined, which is considered to enhance code readability by some.
#' @return A character string.
#' @export
#' @seealso
#'		\code{\link[rkwarddev:rk.paste.JS]{rk.paste.JS}}
#' @examples
#' tmp.var.selectVars <- rk.XML.varselector(label="Select data")
#' tmp.var.x <- rk.XML.varslot(label="My data", source=tmp.var.selectVars, required=TRUE)
#' # let this be the embedded generic plot options in yout plot dialog
#' tmp.plot.options <- rk.plotOptions()
#' 
#' # you can now generate the plot code using generic plot options
#' js.prnt <- rk.paste.JS.graph(
#'	echo("\t\tplot("),
#'	echo("\n\t\t\tx=", tmp.var.x),
#'	echo(tmp.plot.options),
#'	echo(")"),
#'	plotOpts=tmp.plot.options)
#'
#' cat(js.prnt)
rk.paste.JS.graph <- function(..., plotOpts=NULL, printoutObj=NULL, level=2, indent.by="\t", empty.e=FALSE){

	plotOptsIndent <- paste(rep("\\t", level), collapse="")

	# define variables
	js.prnt <- rk.paste.JS(
		if(!is.null(plotOpts)){
			if(inherits(plotOpts, "rk.plot.opts")){
				js.po.preprocess <- slot(plotOpts, "preprocess")
				js.po.calculate <- slot(plotOpts, "calculate")
				if(is.null(printoutObj)){
					printoutObj <- slot(plotOpts, "printout")
				} else {}
			} else {
				js.po.preprocess <- rk.JS.vars(plotOpts, modifiers="code.preprocess", check.modifiers=FALSE)
				js.po.calculate <- rk.JS.vars(plotOpts, modifiers="code.calculate", check.modifiers=FALSE)
			}
			rk.paste.JS(
				rk.comment("in case there are generic plot options defined:"),
				js.po.preprocess,
				if(!is.null(printoutObj)){
					printoutObj
				} else {
					warning("rk.paste.JS.graph: you're using plot options, but 'printoutObj' is empty, is that intended?")
				},
				js.po.calculate
			)
		} else {},
		level=level, indent.by=indent.by, empty.e=empty.e
	)

	# graph.on() & begin try()
	js.prnt <- paste(js.prnt, rk.paste.JS(
		ite("full", echo("rk.graph.on()\n")),
		echo("\ttry({\n"),
		level=level, indent.by=indent.by, empty.e=empty.e
	), sep="\n\n")

	# plot options: preprocess
	js.prnt <- paste(js.prnt, rk.paste.JS(
		if(!is.null(plotOpts)){
			rk.paste.JS(
				rk.comment("insert any option-setting code that should be run before the actual plotting commands:"),
				id("\t\tprintIndentedUnlessEmpty(\"", plotOptsIndent, "\", ", js.po.preprocess, ", \"\\n\", \"\");")
			)
		} else {},
		level=level, indent.by=indent.by, empty.e=empty.e
	), sep="\n\n")

	# here comes the plot
	js.prnt <- paste(js.prnt, rk.paste.JS(
		rk.comment("the actual plot:"),
		rk.paste.JS(..., level=level, indent.by=indent.by, empty.e=empty.e),
		level=level, indent.by=indent.by, empty.e=empty.e
	), sep="\n\n")

	# plot options: postprocess
	js.prnt <- paste(js.prnt, rk.paste.JS(
		if(!is.null(plotOpts)){
			rk.paste.JS(
				rk.comment("insert any option-setting code that should be run after the actual plot:"),
				id("\t\tprintIndentedUnlessEmpty(\"", plotOptsIndent, "\", ", js.po.calculate, ", \"\\n\", \"\");")
			)
		} else {},
		level=level, indent.by=indent.by, empty.e=empty.e
	), sep="\n\n")

	# end try() & graph.off()
	js.prnt <- paste(js.prnt, rk.paste.JS(
		echo("\n\t})\n"),
		ite("full", echo("rk.graph.off()\n")),
		level=level, indent.by=indent.by, empty.e=empty.e
	), sep="\n\n")

	return(js.prnt)
}
