#' Create JavaScript outline from RKWard plugin XML
#' 
#' @note The JavaScript
#'
#' @param require A character vector with names of R packages that the dialog depends on.
#' @param variables Either a character string to be included to read in all needed variables from the dialog (see \code{\link{rk.JS.scan}}),
#'		or an object of class \code{rk.JS.var} which will be coerced into character. These variables will be defined in
#'		the \code{calculate()} and/or \code{doPrintout()} functions.
#' @param globals Like \code{variables}, but these variables will be defined globally. If \code{variables} is set as well,
#'		the function tries to remove duplicate definitions.
#' @param results.header A character string to headline the printed results. Include escapes quotes (\\") if needed.
#'		Set to \code{FALSE} or \code{""} if you need more control and want to define the header section in \code{printout}.
#' @param preprocess A character string to be included in the \code{preprocess()} function. This string will be
#'		pasted as-is, after \code{require} has been evaluated.
#' @param calculate A character string to be included in the \code{calculate()} function. This string will be
#'		pasted as-is, after \code{variables} has been evaluated.
#' @param printout A character string to be included in the \code{printout()} function. This string will be
#'		pasted as-is, after \code{results.header} has been evaluated. Ignored if \code{doPrintout} is set.
#' @param doPrintout A character string to be included in the \code{doPrintout()} function. This string will be
#'		pasted as-is. You don't need to define a \code{preview()} function, as this will be added automatically.
#'		Use \code{ite("full", ...)} style JavaScript code to include headers etc.
#' @param load.silencer Either a character string (ID of probably a checkbox), or an object of class \code{XiMpLe.node}.
#'		This defines a switch you can add to your plugin, to set the \code{require()} call inside \code{suppressMessages()},
#'		hence suppressing all load messages (except for warnings and errors) of required packages in the output.
#' @param gen.info Logical, if \code{TRUE} a comment note will be written into the document,
#'		that it was generated by \code{rkwarddev} and changes should be done to the script.
#' @param indent.by A character string defining how indentation should be done.
#' @return A character string.
#' @seealso \code{\link[rkwarddev:rk.paste.JS]{rk.paste.JS}},
#'		\code{\link[rkwarddev:rk.JS.vars]{rk.JS.vars}},
#'		\code{\link[rkwarddev:rk.JS.array]{rk.JS.array}},
#'		\code{\link[rkwarddev:ite]{ite}},
#'		\code{\link[rkwarddev:echo]{echo}},
#'		\code{\link[rkwarddev:id]{id}},
#'		and the \href{help:rkwardplugins}{Introduction to Writing Plugins for RKWard}
#' @export

rk.JS.doc <- function(require=c(), variables=NULL, globals=NULL, results.header=NULL,
	preprocess=NULL, calculate=NULL, printout=NULL, doPrintout=NULL, load.silencer=NULL, gen.info=TRUE, indent.by="\t"){

	# some data transformation
	if(inherits(variables, "rk.JS.var")){
		variables <- rk.paste.JS(variables)
	} else {}
	if(inherits(globals, "rk.JS.var")){
		globals <- rk.paste.JS(globals, level=1)
	} else {}

	js.gen.info <- ifelse(isTRUE(gen.info), rk.paste.JS(generator.info, level=1), "")

	if(!is.null(globals)){
		js.globals <- paste(
			"// define variables globally\n",
			paste0(globals, collapse=""))
		if(!is.null(variables)){
			# remove globals from variables, if duplicate
			# we'll split them by semicolon
			split.globs <- unlist(strsplit(rk.paste.JS(globals), ";"))
			split.vars <- unlist(strsplit(rk.paste.JS(variables), ";"))
			# for better comparison, remove all spaces
			stripped.globs <- gsub("[[:space:]]", "", split.globs)
			stripped.vars <- gsub("[[:space:]]", "", split.vars)
			# leave only variables *not* found in globals
			ok.vars <- split.vars[!stripped.vars %in% stripped.globs]
			# finally, glue back the semicolon and make one string again
			variables <- gsub("^\n*", "", paste(paste0(ok.vars, ";"), collapse=""))
		} else {}
	} else {
		js.globals <- NULL
	}

	js.require <- unlist(sapply(require, function(this.req){
			if(is.null(load.silencer)){
				req.result <- rk.paste.JS(echo(id("require(", this.req, ")\n")), level=2, indent.by=indent.by)
			} else {
				# get the ID, if it's a XiMpLe.node
				req.result <- rk.paste.JS(
					jsChkSuppress <- rk.JS.vars(load.silencer),
					# somehow "quietly=TRUE" doens't always do the trick
					ite(jsChkSuppress, echo("suppressMessages(require(", this.req, "))\n"), echo("require(", this.req, ")\n"))
				)
			}
			return(req.result)
		}))
	js.preprocess <- paste0("function preprocess(){\n",
		indent(2, by=indent.by), "// add requirements etc. here\n",
		paste(js.require, collapse=""),
		"\n",
		ifelse(is.null(preprocess), "", paste0("\n", preprocess, "\n")),
		"}")

	js.calculate <- paste0("function calculate(){\n",
			# for plots we only need something here if calculate is not empty
			if(is.null(doPrintout) | !is.null(calculate)){paste0(
				ifelse(is.null(variables), "", paste0(
					indent(2, by=indent.by), "// read in variables from dialog\n",
					paste(variables, collapse=""), "\n\n")),
				ifelse(is.null(calculate),
					paste0(indent(2, by=indent.by), "// generate the R code to be evaluated here\n"),
					paste0(indent(2, by=indent.by), "// the R code to be evaluated\n",calculate, "\n")))
			} else {}, "}")
		
	js.printout <- paste0("function printout(){\n",
			if(is.null(doPrintout)){
				paste0(
					indent(2, by=indent.by), "// printout the results\n",
					if(is.character(results.header) && !identical(results.header, "")){
						paste0(indent(2, by=indent.by), echo(id("rk.header(", results.header, ")\n")))
					} else {},
					"\n",
					ifelse(is.null(printout), echo("rk.print(\"\")\n"), paste0("\n", printout)),
					"\n")
				} else {
					rk.paste.JS(
						"// all the real work is moved to a custom defined function doPrintout() below",
						"// true in this case means: We want all the headers that should be printed in the output:",
						"doPrintout(true);",
					level=2, indent.by=indent.by)
				}, "\n}")

	# this part will create preview() and doPrintout(full), if needed
	if(is.null(doPrintout)){
		js.doPrintout <- ""
	} else {
		js.doPrintout <- paste0("function preview(){\n",
					rk.paste.JS(
						"preprocess();",
						"calculate();",
						"doPrintout(false);\n}",
					level=2, indent.by=indent.by),
					"\n\n",
					"function doPrintout(full){\n",
					ifelse(is.null(variables), "", paste0(
						indent(2, by=indent.by), "// read in variables from dialog\n", 
						paste(variables, collapse=""), "\n\n")),
					indent(2, by=indent.by), "// create the plot\n",
					if(is.character(results.header) && !identical(results.header, "")){
						rk.paste.JS(ite("full", echo(id("rk.header(", results.header,")\n"))))
					} else {},
					"\n\n",
					doPrintout,
					if(!is.null(printout)){
						paste0("\n\n", indent(2, by=indent.by), "// left over from the printout function\n", printout, "\n\n")
					} else {},
					"\n}")
	}

	JS.doc <- paste(js.gen.info, js.globals, js.preprocess, js.calculate, js.printout, js.doPrintout, sep="\n\n")

	return(JS.doc)
}
