/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.index.lucene.directory;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.io.ByteStreams;
import com.google.common.primitives.Ints;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.SequenceInputStream;
import java.util.List;
import javax.annotation.Nonnull;
import org.apache.jackrabbit.oak.api.Blob;
import org.apache.jackrabbit.oak.api.PropertyState;
import org.apache.jackrabbit.oak.api.Type;
import org.apache.jackrabbit.oak.commons.StringUtils;
import org.apache.jackrabbit.oak.plugins.index.lucene.directory.BlobFactory;
import org.apache.jackrabbit.oak.plugins.index.lucene.directory.OakIndexFile;
import org.apache.jackrabbit.oak.spi.state.NodeBuilder;
import org.apache.lucene.store.DataInput;

class OakBufferedIndexFile
implements OakIndexFile {
    static final int DEFAULT_BLOB_SIZE = 32768;
    private final String name;
    private final NodeBuilder file;
    private final int blobSize;
    private long position = 0L;
    private long length;
    private List<Blob> data;
    private boolean dataModified = false;
    private int index = -1;
    private byte[] blob;
    private final byte[] uniqueKey;
    private boolean blobModified = false;
    private final String dirDetails;
    private final BlobFactory blobFactory;

    public OakBufferedIndexFile(String name, NodeBuilder file, String dirDetails, @Nonnull BlobFactory blobFactory) {
        this.name = name;
        this.file = file;
        this.dirDetails = dirDetails;
        this.blobSize = OakBufferedIndexFile.determineBlobSize(file);
        this.uniqueKey = OakBufferedIndexFile.readUniqueKey(file);
        this.blob = new byte[this.blobSize];
        this.blobFactory = (BlobFactory)Preconditions.checkNotNull((Object)blobFactory);
        PropertyState property = file.getProperty("jcr:data");
        this.data = property != null && property.getType() == Type.BINARIES ? Lists.newArrayList((Iterable)((Iterable)property.getValue(Type.BINARIES))) : Lists.newArrayList();
        this.length = (long)this.data.size() * (long)this.blobSize;
        if (!this.data.isEmpty()) {
            Blob last = this.data.get(this.data.size() - 1);
            this.length -= (long)this.blobSize - last.length();
            if (this.uniqueKey != null) {
                this.length -= (long)this.uniqueKey.length;
            }
        }
    }

    private OakBufferedIndexFile(OakBufferedIndexFile that) {
        this.name = that.name;
        this.file = that.file;
        this.dirDetails = that.dirDetails;
        this.blobSize = that.blobSize;
        this.uniqueKey = that.uniqueKey;
        this.blob = new byte[this.blobSize];
        this.position = that.position;
        this.length = that.length;
        this.data = Lists.newArrayList(that.data);
        this.dataModified = that.dataModified;
        this.blobFactory = that.blobFactory;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void loadBlob(int i) throws IOException {
        Preconditions.checkElementIndex((int)i, (int)this.data.size());
        if (this.index != i) {
            this.flushBlob();
            Preconditions.checkState((!this.blobModified ? 1 : 0) != 0);
            int n = (int)Math.min((long)this.blobSize, this.length - (long)i * (long)this.blobSize);
            try (InputStream stream = this.data.get(i).getNewStream();){
                ByteStreams.readFully((InputStream)stream, (byte[])this.blob, (int)0, (int)n);
            }
            this.index = i;
        }
    }

    private void flushBlob() throws IOException {
        if (this.blobModified) {
            int n = (int)Math.min((long)this.blobSize, this.length - (long)this.index * (long)this.blobSize);
            InputStream in = new ByteArrayInputStream(this.blob, 0, n);
            if (this.uniqueKey != null) {
                in = new SequenceInputStream(in, new ByteArrayInputStream(this.uniqueKey));
            }
            Blob b = this.blobFactory.createBlob(in);
            if (this.index < this.data.size()) {
                this.data.set(this.index, b);
            } else {
                Preconditions.checkState((this.index == this.data.size() ? 1 : 0) != 0);
                this.data.add(b);
            }
            this.dataModified = true;
            this.blobModified = false;
        }
    }

    @Override
    public OakIndexFile clone() {
        return new OakBufferedIndexFile(this);
    }

    @Override
    public long length() {
        return this.length;
    }

    @Override
    public long position() {
        return this.position;
    }

    @Override
    public void close() {
        this.blob = null;
        this.data = null;
    }

    @Override
    public boolean isClosed() {
        return this.blob == null && this.data == null;
    }

    @Override
    public void seek(long pos) throws IOException {
        if (pos < 0L || pos > this.length) {
            String msg = String.format("Invalid seek request for [%s][%s], position: %d, file length: %d", this.dirDetails, this.name, pos, this.length);
            throw new IOException(msg);
        }
        this.position = pos;
    }

    @Override
    public void readBytes(byte[] b, int offset, int len) throws IOException {
        Preconditions.checkPositionIndexes((int)offset, (int)(offset + len), (int)((byte[])Preconditions.checkNotNull((Object)b)).length);
        if (len < 0 || this.position + (long)len > this.length) {
            String msg = String.format("Invalid byte range request for [%s][%s], position: %d, file length: %d, len: %d", this.dirDetails, this.name, this.position, this.length, len);
            throw new IOException(msg);
        }
        int i = (int)(this.position / (long)this.blobSize);
        int o = (int)(this.position % (long)this.blobSize);
        while (len > 0) {
            this.loadBlob(i);
            int l = Math.min(len, this.blobSize - o);
            System.arraycopy(this.blob, o, b, offset, l);
            offset += l;
            len -= l;
            this.position += (long)l;
            ++i;
            o = 0;
        }
    }

    @Override
    public void writeBytes(byte[] b, int offset, int len) throws IOException {
        int i = (int)(this.position / (long)this.blobSize);
        int o = (int)(this.position % (long)this.blobSize);
        while (len > 0) {
            int l = Math.min(len, this.blobSize - o);
            if (this.index != i) {
                if (o > 0 || l < this.blobSize && this.position + (long)l < this.length) {
                    this.loadBlob(i);
                } else {
                    this.flushBlob();
                    this.index = i;
                }
            }
            System.arraycopy(b, offset, this.blob, o, l);
            this.blobModified = true;
            offset += l;
            len -= l;
            this.position += (long)l;
            this.length = Math.max(this.length, this.position);
            ++i;
            o = 0;
        }
    }

    @Override
    public boolean supportsCopyFromDataInput() {
        return false;
    }

    @Override
    public void copyBytes(DataInput input, long numBytes) throws IOException {
        throw new IllegalArgumentException("Don't call copyBytes for buffered case");
    }

    private static int determineBlobSize(NodeBuilder file) {
        if (file.hasProperty("blobSize")) {
            return Ints.checkedCast((long)((Long)file.getProperty("blobSize").getValue(Type.LONG)));
        }
        return 32768;
    }

    private static byte[] readUniqueKey(NodeBuilder file) {
        if (file.hasProperty("uniqueKey")) {
            String key = file.getString("uniqueKey");
            return StringUtils.convertHexToBytes((String)key);
        }
        return null;
    }

    @Override
    public void flush() throws IOException {
        this.flushBlob();
        if (this.dataModified) {
            this.file.setProperty("jcr:lastModified", (Object)System.currentTimeMillis());
            this.file.setProperty("jcr:data", this.data, Type.BINARIES);
            this.dataModified = false;
        }
    }

    public String toString() {
        return this.name;
    }

    @Override
    public String getName() {
        return this.name;
    }
}

