/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.index.lucene.directory;

import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.io.ByteStreams;
import com.google.common.io.Closer;
import com.google.common.io.CountingInputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.io.FileUtils;
import org.apache.jackrabbit.oak.api.Blob;
import org.apache.jackrabbit.oak.api.PropertyState;
import org.apache.jackrabbit.oak.api.Root;
import org.apache.jackrabbit.oak.api.Tree;
import org.apache.jackrabbit.oak.api.Type;
import org.apache.jackrabbit.oak.commons.IOUtils;
import org.apache.jackrabbit.oak.commons.PathUtils;
import org.apache.jackrabbit.oak.plugins.index.lucene.IndexDefinition;
import org.apache.jackrabbit.oak.plugins.index.lucene.directory.IndexRootDirectory;
import org.apache.jackrabbit.oak.plugins.index.lucene.directory.OakDirectory;
import org.apache.jackrabbit.oak.plugins.index.lucene.writer.MultiplexersLucene;
import org.apache.jackrabbit.oak.plugins.tree.factories.RootFactory;
import org.apache.jackrabbit.oak.spi.state.NodeBuilder;
import org.apache.jackrabbit.oak.spi.state.NodeState;
import org.apache.jackrabbit.oak.spi.state.NodeStateUtils;
import org.apache.jackrabbit.oak.spi.state.ReadOnlyBuilder;
import org.apache.lucene.index.CheckIndex;
import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.IOContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IndexConsistencyChecker {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final NodeState rootState;
    private final String indexPath;
    private final File workDirRoot;
    private File workDir;
    private PrintStream printStream;
    private boolean verbose;

    public IndexConsistencyChecker(NodeState rootState, String indexPath, File workDirRoot) {
        this.rootState = (NodeState)Preconditions.checkNotNull((Object)rootState);
        this.indexPath = (String)Preconditions.checkNotNull((Object)indexPath);
        this.workDirRoot = (File)Preconditions.checkNotNull((Object)workDirRoot);
    }

    public void setPrintStream(PrintStream printStream) {
        this.printStream = printStream;
    }

    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    public Result check(Level level) throws IOException {
        return this.check(level, true);
    }

    public Result check(Level level, boolean cleanWorkDir) throws IOException {
        try (Closer closer = Closer.create();){
            Result result = this.check(level, cleanWorkDir, closer);
            return result;
        }
    }

    private Result check(Level level, boolean cleanWorkDir, Closer closer) throws IOException {
        Stopwatch watch = Stopwatch.createStarted();
        Result result = new Result();
        result.indexPath = this.indexPath;
        result.clean = true;
        result.watch = watch;
        this.log.info("[{}] Starting check", (Object)this.indexPath);
        this.checkBlobs(result);
        if (level == Level.FULL && result.clean) {
            this.checkIndex(result, closer);
        }
        if (result.clean) {
            this.log.info("[{}] No problems were detected with this index. Time taken {}", (Object)this.indexPath, (Object)watch);
        } else {
            this.log.warn("[{}] Problems detected with this index. Time taken {}", (Object)this.indexPath, (Object)watch);
        }
        if (cleanWorkDir) {
            FileUtils.deleteQuietly((File)this.workDir);
        } else if (this.workDir != null) {
            this.log.info("[{}] Index files are copied to {}", (Object)this.indexPath, (Object)this.workDir.getAbsolutePath());
        }
        watch.stop();
        return result;
    }

    private void checkIndex(Result result, Closer closer) throws IOException {
        NodeState idx = NodeStateUtils.getNode((NodeState)this.rootState, (String)this.indexPath);
        IndexDefinition defn = IndexDefinition.newBuilder(this.rootState, idx, this.indexPath).build();
        this.workDir = IndexConsistencyChecker.createWorkDir(this.workDirRoot, PathUtils.getName((String)this.indexPath));
        for (String dirName : idx.getChildNodeNames()) {
            if (!NodeStateUtils.isHidden((String)dirName) || !MultiplexersLucene.isIndexDirName(dirName)) continue;
            DirectoryStatus dirStatus = new DirectoryStatus(dirName);
            result.dirStatus.add(dirStatus);
            this.log.debug("[{}] Checking directory {}", (Object)this.indexPath, (Object)dirName);
            try {
                this.checkIndexDirectory(dirStatus, idx, defn, this.workDir, dirName, closer);
            }
            catch (IOException e) {
                dirStatus.clean = false;
                this.log.warn("[{}][{}] Error occurred while performing directory check", new Object[]{this.indexPath, dirName, e});
            }
            if (dirStatus.clean) continue;
            result.clean = false;
        }
    }

    private void checkIndexDirectory(DirectoryStatus dirStatus, NodeState idx, IndexDefinition defn, File workDir, String dirName, Closer closer) throws IOException {
        File idxDir = IndexConsistencyChecker.createWorkDir(workDir, dirName);
        OakDirectory sourceDir = new OakDirectory((NodeBuilder)new ReadOnlyBuilder(idx), dirName, defn, true);
        FSDirectory targetDir = FSDirectory.open(idxDir);
        closer.register((Closeable)sourceDir);
        closer.register((Closeable)targetDir);
        boolean clean = true;
        for (String file : ((Directory)sourceDir).listAll()) {
            this.log.debug("[{}][{}] Checking {}", new Object[]{this.indexPath, dirName, file});
            try {
                sourceDir.copy(targetDir, file, file, IOContext.DEFAULT);
            }
            catch (FileNotFoundException ignore) {
                dirStatus.missingFiles.add(file);
                clean = false;
                this.log.warn("[{}][{}] File {} missing", new Object[]{this.indexPath, dirName, file});
            }
            if (((Directory)targetDir).fileLength(file) != ((Directory)sourceDir).fileLength(file)) {
                FileSizeStatus fileStatus = new FileSizeStatus(file, ((Directory)targetDir).fileLength(file), ((Directory)sourceDir).fileLength(file));
                dirStatus.filesWithSizeMismatch.add(fileStatus);
                clean = false;
                this.log.warn("[{}][{}] File size mismatch {}", new Object[]{this.indexPath, dirName, fileStatus});
                continue;
            }
            dirStatus.size += ((Directory)sourceDir).fileLength(file);
            this.log.debug("[{}][{}] File {} is consistent", new Object[]{this.indexPath, dirName, file});
        }
        if (clean) {
            this.log.debug("[{}][{}] Directory content found to be consistent. Proceeding to IndexCheck", (Object)this.indexPath, (Object)dirName);
            CheckIndex ci = new CheckIndex(targetDir);
            if (this.printStream != null) {
                ci.setInfoStream(this.printStream, this.verbose);
            } else if (this.log.isDebugEnabled()) {
                ci.setInfoStream(new LoggingPrintStream(this.log), this.log.isTraceEnabled());
            }
            dirStatus.status = ci.checkIndex();
            dirStatus.clean = dirStatus.status.clean;
            this.log.debug("[{}][{}] IndexCheck was successful. Proceeding to open DirectoryReader", (Object)this.indexPath, (Object)dirName);
        }
        if (dirStatus.clean) {
            DirectoryReader dirReader = DirectoryReader.open(targetDir);
            dirStatus.numDocs = dirReader.numDocs();
            this.log.debug("[{}][{}] DirectoryReader can be opened", (Object)this.indexPath, (Object)dirName);
            closer.register((Closeable)dirReader);
        }
    }

    private void checkBlobs(Result result) {
        Root root = RootFactory.createReadOnlyRoot((NodeState)this.rootState);
        Tree idx = root.getTree(this.indexPath);
        PropertyState type = idx.getProperty("type");
        if (type != null && "lucene".equals(type.getValue(Type.STRING))) {
            this.checkBlobs(result, idx);
        } else {
            result.clean = false;
            result.typeMismatch = true;
        }
    }

    private void checkBlobs(Result result, Tree tree) {
        for (PropertyState ps : tree.getProperties()) {
            if (ps.getType().tag() != 2) continue;
            if (ps.isArray()) {
                for (int i = 0; i < ps.count(); ++i) {
                    Blob b = (Blob)ps.getValue(Type.BINARY, i);
                    this.checkBlob(ps.getName(), b, tree, result);
                }
                continue;
            }
            Blob b = (Blob)ps.getValue(Type.BINARY);
            this.checkBlob(ps.getName(), b, tree, result);
        }
        for (Tree child : tree.getChildren()) {
            this.checkBlobs(result, child);
        }
    }

    private void checkBlob(String propName, Blob blob, Tree tree, Result result) {
        String id = blob.getContentIdentity();
        String blobPath = String.format("%s/%s/%s", tree.getPath(), propName, id);
        try {
            InputStream is = blob.getNewStream();
            CountingInputStream cis = new CountingInputStream(is);
            org.apache.commons.io.IOUtils.copyLarge((InputStream)cis, (OutputStream)ByteStreams.nullOutputStream());
            if (cis.getCount() != blob.length()) {
                String msg = String.format("Invalid blob %s. Length mismatch - expected ${%d} -> found ${%d}", blobPath, blob.length(), cis.getCount());
                result.invalidBlobIds.add(new FileSizeStatus(blobPath, cis.getCount(), blob.length()));
                this.log.warn("[{}] {}", (Object)this.indexPath, (Object)msg);
                result.clean = false;
                result.blobSizeMismatch = true;
            }
            result.binaryPropSize += cis.getCount();
        }
        catch (Exception e) {
            this.log.warn("[{}] Error occurred reading blob at {}", new Object[]{this.indexPath, blobPath, e});
            result.missingBlobIds.add(id);
            result.clean = false;
            result.missingBlobs = true;
        }
    }

    private static File createWorkDir(File parent, String name) throws IOException {
        String fsSafeName = IndexRootDirectory.getFSSafeName(name);
        File dir = new File(parent, fsSafeName);
        FileUtils.forceMkdir((File)dir);
        FileUtils.cleanDirectory((File)dir);
        return dir;
    }

    private static final class LoggingPrintStream
    extends PrintStream {
        private final StringBuffer buffer = new StringBuffer();
        private final Logger log;

        public LoggingPrintStream(Logger log) {
            super(ByteStreams.nullOutputStream());
            this.log = log;
        }

        @Override
        public void print(String s) {
            this.buffer.append(s);
        }

        @Override
        public void println(String s) {
            this.buffer.append(s);
            this.log.debug(this.buffer.toString());
            this.buffer.setLength(0);
        }
    }

    public static class FileSizeStatus {
        public final String name;
        public final long actualSize;
        public final long expectedSize;

        public FileSizeStatus(String name, long actualSize, long expectedSize) {
            this.name = name;
            this.actualSize = actualSize;
            this.expectedSize = expectedSize;
        }

        public String toString() {
            return String.format("%s => expected %d, actual %d", this.name, this.expectedSize, this.actualSize);
        }
    }

    public static class DirectoryStatus {
        public final String dirName;
        public final List<String> missingFiles = new ArrayList<String>();
        public final List<FileSizeStatus> filesWithSizeMismatch = new ArrayList<FileSizeStatus>();
        public boolean clean;
        public long size;
        public CheckIndex.Status status;
        public long numDocs;

        public DirectoryStatus(String dirName) {
            this.dirName = dirName;
        }

        public void dump(PrintWriter pw) {
            pw.println("Directory : " + this.dirName);
            pw.printf("\tSize     : %s%n", IOUtils.humanReadableByteCount((long)this.size));
            pw.printf("\tNum docs : %d%n", this.numDocs);
            if (!this.missingFiles.isEmpty()) {
                pw.println("\tMissing Files");
                for (String file : this.missingFiles) {
                    pw.println("\t\t- " + file);
                }
            }
            if (!this.filesWithSizeMismatch.isEmpty()) {
                pw.println("Invalid files");
                for (FileSizeStatus status : this.filesWithSizeMismatch) {
                    pw.println("\t - " + status);
                }
            }
            if (this.status != null) {
                pw.printf("\tCheckIndex status : %s%n", this.status.clean);
            }
        }
    }

    public static class Result {
        public boolean clean;
        public boolean typeMismatch;
        public boolean missingBlobs;
        public boolean blobSizeMismatch;
        public String indexPath;
        public long binaryPropSize;
        public List<FileSizeStatus> invalidBlobIds = new ArrayList<FileSizeStatus>();
        public List<String> missingBlobIds = new ArrayList<String>();
        public List<DirectoryStatus> dirStatus = new ArrayList<DirectoryStatus>();
        private Stopwatch watch;

        public void dump(PrintWriter pw) {
            if (this.clean) {
                pw.printf("%s => VALID%n", this.indexPath);
            } else {
                pw.printf("%s => INVALID%n", this.indexPath);
            }
            pw.printf("\tSize : %s%n", IOUtils.humanReadableByteCount((long)this.binaryPropSize));
            if (!this.missingBlobIds.isEmpty()) {
                pw.println("Missing blobs");
                for (String id : this.missingBlobIds) {
                    pw.println("\t - " + id);
                }
            }
            if (!this.invalidBlobIds.isEmpty()) {
                pw.println("Invalid blobs");
                for (FileSizeStatus status : this.invalidBlobIds) {
                    pw.println("\t - " + status);
                }
            }
            for (DirectoryStatus dirStatus : this.dirStatus) {
                dirStatus.dump(pw);
            }
            pw.printf("Time taken : %s%n", this.watch);
        }

        public String toString() {
            StringWriter sw = new StringWriter();
            PrintWriter pw = new PrintWriter(sw);
            this.dump(pw);
            return sw.toString();
        }
    }

    public static enum Level {
        BLOBS_ONLY,
        FULL;

    }
}

