/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment.file.tar;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.io.Closeable;
import java.io.File;
import java.io.FileDescriptor;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.zip.CRC32;
import org.apache.jackrabbit.oak.segment.file.tar.FileStoreMonitor;
import org.apache.jackrabbit.oak.segment.file.tar.FileStoreMonitorAdapter;
import org.apache.jackrabbit.oak.segment.file.tar.GCGeneration;
import org.apache.jackrabbit.oak.segment.file.tar.IOMonitor;
import org.apache.jackrabbit.oak.segment.file.tar.TarEntry;
import org.apache.jackrabbit.oak.segment.file.tar.binaries.BinaryReferencesIndexWriter;
import org.apache.jackrabbit.oak.segment.file.tar.index.IndexWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class TarWriter
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(TarWriter.class);
    private static final byte[] ZERO_BYTES = new byte[512];
    private final int writeIndex;
    private final File file;
    private final FileStoreMonitor monitor;
    private RandomAccessFile access = null;
    private FileChannel channel = null;
    private boolean closed = false;
    private final Map<UUID, TarEntry> index = Maps.newLinkedHashMap();
    private final BinaryReferencesIndexWriter binaryReferences = BinaryReferencesIndexWriter.newBinaryReferencesIndexWriter();
    private final Map<UUID, Set<UUID>> graph = Maps.newHashMap();
    private final IOMonitor ioMonitor;

    static int getPaddingSize(int size) {
        int remainder = size % 512;
        if (remainder > 0) {
            return 512 - remainder;
        }
        return 0;
    }

    TarWriter(File file, IOMonitor ioMonitor) {
        this.file = file;
        this.monitor = new FileStoreMonitorAdapter();
        this.writeIndex = -1;
        this.ioMonitor = ioMonitor;
    }

    TarWriter(File directory, FileStoreMonitor monitor, int writeIndex, IOMonitor ioMonitor) {
        this.file = new File(directory, String.format("data%05d%s.tar", writeIndex, "a"));
        this.monitor = monitor;
        this.writeIndex = writeIndex;
        this.ioMonitor = ioMonitor;
    }

    synchronized boolean containsEntry(long msb, long lsb) {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0);
        return this.index.containsKey(new UUID(msb, lsb));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    ByteBuffer readEntry(long msb, long lsb) throws IOException {
        TarEntry entry;
        Preconditions.checkState((!this.closed ? 1 : 0) != 0);
        TarWriter tarWriter = this;
        synchronized (tarWriter) {
            entry = this.index.get(new UUID(msb, lsb));
        }
        if (entry != null) {
            Preconditions.checkState((this.channel != null ? 1 : 0) != 0);
            ByteBuffer data = ByteBuffer.allocate(entry.size());
            this.channel.read(data, entry.offset());
            data.rewind();
            return data;
        }
        return null;
    }

    long writeEntry(long msb, long lsb, byte[] data, int offset, int size, GCGeneration generation) throws IOException {
        Preconditions.checkNotNull((Object)data);
        Preconditions.checkPositionIndexes((int)offset, (int)(offset + size), (int)data.length);
        UUID uuid = new UUID(msb, lsb);
        CRC32 checksum = new CRC32();
        checksum.update(data, offset, size);
        String entryName = String.format("%s.%08x", uuid, checksum.getValue());
        byte[] header = TarWriter.newEntryHeader(entryName, size);
        log.debug("Writing segment {} to {}", (Object)uuid, (Object)this.file);
        return this.writeEntry(uuid, header, data, offset, size, generation);
    }

    private synchronized long writeEntry(UUID uuid, byte[] header, byte[] data, int offset, int size, GCGeneration generation) throws IOException {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0);
        if (this.access == null) {
            this.access = new RandomAccessFile(this.file, "rw");
            this.channel = this.access.getChannel();
        }
        long msb = uuid.getMostSignificantBits();
        long lsb = uuid.getLeastSignificantBits();
        int padding = TarWriter.getPaddingSize(size);
        long initialLength = this.access.getFilePointer();
        this.access.write(header);
        this.ioMonitor.beforeSegmentWrite(this.file, msb, lsb, size);
        Stopwatch stopwatch = Stopwatch.createStarted();
        this.access.write(data, offset, size);
        this.ioMonitor.afterSegmentWrite(this.file, msb, lsb, size, stopwatch.elapsed(TimeUnit.NANOSECONDS));
        if (padding > 0) {
            this.access.write(ZERO_BYTES, 0, padding);
        }
        long currentLength = this.access.getFilePointer();
        this.monitor.written(currentLength - initialLength);
        Preconditions.checkState((currentLength <= Integer.MAX_VALUE ? 1 : 0) != 0);
        TarEntry entry = new TarEntry(msb, lsb, (int)(currentLength - (long)size - (long)padding), size, generation);
        this.index.put(uuid, entry);
        return currentLength;
    }

    void addBinaryReference(GCGeneration generation, UUID segmentId, String reference) {
        this.binaryReferences.addEntry(generation.getGeneration(), generation.getFullGeneration(), generation.isCompacted(), segmentId, reference);
    }

    void addGraphEdge(UUID from, UUID to) {
        this.graph.computeIfAbsent(from, k -> Sets.newHashSet()).add(to);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void flush() throws IOException {
        File file = this.file;
        synchronized (file) {
            FileDescriptor descriptor = null;
            TarWriter tarWriter = this;
            synchronized (tarWriter) {
                if (this.access != null && !this.closed) {
                    descriptor = this.access.getFD();
                }
            }
            if (descriptor != null) {
                descriptor.sync();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        long currentPosition;
        long initialPosition;
        TarWriter tarWriter = this;
        synchronized (tarWriter) {
            Preconditions.checkState((!this.closed ? 1 : 0) != 0);
            this.closed = true;
        }
        if (this.access == null) {
            return;
        }
        File file = this.file;
        synchronized (file) {
            initialPosition = this.access.getFilePointer();
            this.writeBinaryReferences();
            this.writeGraph();
            this.writeIndex();
            this.access.write(ZERO_BYTES);
            this.access.write(ZERO_BYTES);
            currentPosition = this.access.getFilePointer();
            this.access.close();
        }
        this.monitor.written(currentPosition - initialPosition);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    TarWriter createNextGeneration() throws IOException {
        Preconditions.checkState((this.writeIndex >= 0 ? 1 : 0) != 0);
        TarWriter tarWriter = this;
        synchronized (tarWriter) {
            if (this.access == null) {
                return this;
            }
        }
        this.close();
        int newIndex = this.writeIndex + 1;
        return new TarWriter(this.file.getParentFile(), this.monitor, newIndex, this.ioMonitor);
    }

    private void writeBinaryReferences() throws IOException {
        byte[] data = this.binaryReferences.write();
        int paddingSize = TarWriter.getPaddingSize(data.length);
        byte[] header = TarWriter.newEntryHeader(this.file.getName() + ".brf", data.length + paddingSize);
        this.access.write(header);
        if (paddingSize > 0) {
            this.access.write(ZERO_BYTES, 0, paddingSize);
        }
        this.access.write(data);
    }

    private void writeGraph() throws IOException {
        int graphSize = 0;
        graphSize += 4;
        graphSize += 4;
        graphSize += 4;
        graphSize += 4;
        for (Map.Entry<UUID, Set<UUID>> entry : this.graph.entrySet()) {
            graphSize += 16;
            graphSize += 4;
            graphSize += 16 * entry.getValue().size();
        }
        ByteBuffer buffer = ByteBuffer.allocate(graphSize);
        for (Map.Entry<UUID, Set<UUID>> entry : this.graph.entrySet()) {
            UUID from = entry.getKey();
            buffer.putLong(from.getMostSignificantBits());
            buffer.putLong(from.getLeastSignificantBits());
            Set<UUID> adj = entry.getValue();
            buffer.putInt(adj.size());
            for (UUID to : adj) {
                buffer.putLong(to.getMostSignificantBits());
                buffer.putLong(to.getLeastSignificantBits());
            }
        }
        CRC32 cRC32 = new CRC32();
        cRC32.update(buffer.array(), 0, buffer.position());
        buffer.putInt((int)cRC32.getValue());
        buffer.putInt(this.graph.size());
        buffer.putInt(graphSize);
        buffer.putInt(170936074);
        int n = TarWriter.getPaddingSize(graphSize);
        this.access.write(TarWriter.newEntryHeader(this.file.getName() + ".gph", graphSize + n));
        if (n > 0) {
            this.access.write(ZERO_BYTES, 0, n);
        }
        this.access.write(buffer.array());
    }

    private void writeIndex() throws IOException {
        IndexWriter writer = IndexWriter.newIndexWriter(512);
        for (TarEntry entry : this.index.values()) {
            writer.addEntry(entry.msb(), entry.lsb(), entry.offset(), entry.size(), entry.generation().getGeneration(), entry.generation().getFullGeneration(), entry.generation().isCompacted());
        }
        byte[] index = writer.write();
        this.access.write(TarWriter.newEntryHeader(this.file.getName() + ".idx", index.length));
        this.access.write(index);
    }

    private static byte[] newEntryHeader(String name, int size) {
        byte[] header = new byte[512];
        byte[] nameBytes = name.getBytes(Charsets.UTF_8);
        System.arraycopy(nameBytes, 0, header, 0, Math.min(nameBytes.length, 100));
        System.arraycopy(String.format("%07o", 256).getBytes(Charsets.UTF_8), 0, header, 100, 7);
        System.arraycopy(String.format("%07o", 0).getBytes(Charsets.UTF_8), 0, header, 108, 7);
        System.arraycopy(String.format("%07o", 0).getBytes(Charsets.UTF_8), 0, header, 116, 7);
        System.arraycopy(String.format("%011o", size).getBytes(Charsets.UTF_8), 0, header, 124, 11);
        long time = System.currentTimeMillis() / 1000L;
        System.arraycopy(String.format("%011o", time).getBytes(Charsets.UTF_8), 0, header, 136, 11);
        System.arraycopy(new byte[]{32, 32, 32, 32, 32, 32, 32, 32}, 0, header, 148, 8);
        header[156] = 48;
        int checksum = 0;
        for (byte aHeader : header) {
            checksum += aHeader & 0xFF;
        }
        System.arraycopy(String.format("%06o\u0000 ", checksum).getBytes(Charsets.UTF_8), 0, header, 148, 8);
        return header;
    }

    synchronized long fileLength() {
        return this.file.length();
    }

    synchronized File getFile() {
        return this.file;
    }

    synchronized boolean isClosed() {
        return this.closed;
    }

    public String toString() {
        return this.file.toString();
    }
}

