/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment.file;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Predicate;
import com.google.common.base.Supplier;
import com.google.common.collect.Sets;
import com.google.common.io.Closer;
import com.google.common.util.concurrent.UncheckedExecutionException;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.jackrabbit.oak.commons.IOUtils;
import org.apache.jackrabbit.oak.plugins.memory.EmptyNodeState;
import org.apache.jackrabbit.oak.segment.CheckpointCompactor;
import org.apache.jackrabbit.oak.segment.DefaultSegmentWriter;
import org.apache.jackrabbit.oak.segment.DefaultSegmentWriterBuilder;
import org.apache.jackrabbit.oak.segment.RecordId;
import org.apache.jackrabbit.oak.segment.Segment;
import org.apache.jackrabbit.oak.segment.SegmentId;
import org.apache.jackrabbit.oak.segment.SegmentNodeState;
import org.apache.jackrabbit.oak.segment.SegmentNotFoundException;
import org.apache.jackrabbit.oak.segment.SegmentNotFoundExceptionListener;
import org.apache.jackrabbit.oak.segment.SegmentReader;
import org.apache.jackrabbit.oak.segment.SegmentWriter;
import org.apache.jackrabbit.oak.segment.WriterCacheManager;
import org.apache.jackrabbit.oak.segment.compaction.SegmentGCOptions;
import org.apache.jackrabbit.oak.segment.compaction.SegmentGCStatus;
import org.apache.jackrabbit.oak.segment.file.AbstractFileStore;
import org.apache.jackrabbit.oak.segment.file.FileReaper;
import org.apache.jackrabbit.oak.segment.file.FileStoreBuilder;
import org.apache.jackrabbit.oak.segment.file.FileStoreStats;
import org.apache.jackrabbit.oak.segment.file.GCEstimationResult;
import org.apache.jackrabbit.oak.segment.file.GCJournal;
import org.apache.jackrabbit.oak.segment.file.GCListener;
import org.apache.jackrabbit.oak.segment.file.GCMemoryBarrier;
import org.apache.jackrabbit.oak.segment.file.GCNodeWriteMonitor;
import org.apache.jackrabbit.oak.segment.file.InvalidFileStoreVersionException;
import org.apache.jackrabbit.oak.segment.file.PrefixedGCListener;
import org.apache.jackrabbit.oak.segment.file.PrintableBytes;
import org.apache.jackrabbit.oak.segment.file.PrintableStopwatch;
import org.apache.jackrabbit.oak.segment.file.Reclaimers;
import org.apache.jackrabbit.oak.segment.file.SafeRunnable;
import org.apache.jackrabbit.oak.segment.file.Scheduler;
import org.apache.jackrabbit.oak.segment.file.ShutDown;
import org.apache.jackrabbit.oak.segment.file.SizeDeltaGcEstimation;
import org.apache.jackrabbit.oak.segment.file.TarRevisions;
import org.apache.jackrabbit.oak.segment.file.tar.CleanupContext;
import org.apache.jackrabbit.oak.segment.file.tar.GCGeneration;
import org.apache.jackrabbit.oak.segment.file.tar.TarFiles;
import org.apache.jackrabbit.oak.spi.state.NodeBuilder;
import org.apache.jackrabbit.oak.spi.state.NodeState;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileStore
extends AbstractFileStore {
    private static final Logger log = LoggerFactory.getLogger(FileStore.class);
    private static final long GC_BACKOFF = Integer.getInteger("oak.gc.backoff", 36000000).intValue();
    private static final int MB = 0x100000;
    static final String LOCK_FILE_NAME = "repo.lock";
    private static final AtomicLong GC_COUNT = new AtomicLong(0L);
    @Nonnull
    private final SegmentWriter segmentWriter;
    @Nonnull
    private final GarbageCollector garbageCollector;
    private final TarFiles tarFiles;
    private final RandomAccessFile lockFile;
    @Nonnull
    private final FileLock lock;
    private volatile TarRevisions revisions;
    private final Scheduler fileStoreScheduler = new Scheduler("FileStore background tasks");
    private final FileReaper fileReaper = new FileReaper();
    private final AtomicBoolean sufficientDiskSpace = new AtomicBoolean(true);
    private final AtomicBoolean sufficientMemory = new AtomicBoolean(true);
    private final FileStoreStats stats;
    private final ShutDown shutDown = new ShutDown();
    @Nonnull
    private final SegmentNotFoundExceptionListener snfeListener;

    FileStore(FileStoreBuilder builder) throws InvalidFileStoreVersionException, IOException {
        super(builder);
        this.lockFile = new RandomAccessFile(new File(this.directory, LOCK_FILE_NAME), "rw");
        try {
            this.lock = this.lockFile.getChannel().lock();
        }
        catch (OverlappingFileLockException ex) {
            throw new IllegalStateException(this.directory.getAbsolutePath() + " is in use by another store.", ex);
        }
        this.segmentWriter = DefaultSegmentWriterBuilder.defaultSegmentWriterBuilder("sys").withGeneration((Supplier<GCGeneration>)((Supplier)() -> this.getGcGeneration().nonGC())).withWriterPool().with(builder.getCacheManager().withAccessTracking("WRITE", builder.getStatsProvider())).build(this);
        this.garbageCollector = new GarbageCollector(builder.getGcOptions(), builder.getGcListener(), new GCJournal(this.directory), builder.getCacheManager().withAccessTracking("COMPACT", builder.getStatsProvider()));
        FileStore.newManifestChecker(this.directory, builder.getStrictVersionCheck()).checkAndUpdateManifest();
        this.stats = new FileStoreStats(builder.getStatsProvider(), this, 0L);
        this.tarFiles = TarFiles.builder().withDirectory(this.directory).withMemoryMapping(this.memoryMapping).withTarRecovery(this.recovery).withIOMonitor(this.ioMonitor).withFileStoreMonitor(this.stats).withMaxFileSize(builder.getMaxFileSize() * 0x100000).build();
        long size = this.tarFiles.size();
        this.stats.init(size);
        this.snfeListener = builder.getSnfeListener();
        this.fileStoreScheduler.scheduleAtFixedRate(String.format("TarMK flush [%s]", this.directory), 5L, TimeUnit.SECONDS, this::tryFlush);
        this.fileStoreScheduler.scheduleAtFixedRate(String.format("TarMK filer reaper [%s]", this.directory), 5L, TimeUnit.SECONDS, this.fileReaper::reap);
        this.fileStoreScheduler.scheduleAtFixedRate(String.format("TarMK disk space check [%s]", this.directory), 1L, TimeUnit.MINUTES, () -> {
            try (ShutDown.ShutDownCloser ignore = this.shutDown.tryKeepAlive();){
                if (this.shutDown.isShutDown()) {
                    log.debug("Shut down in progress, skipping disk space check");
                } else {
                    this.checkDiskSpace(builder.getGcOptions());
                }
            }
        });
        log.info("TarMK opened at {}, mmap={}, size={}", new Object[]{this.directory, this.memoryMapping, PrintableBytes.newPrintableBytes(size)});
        log.debug("TAR files: {}", (Object)this.tarFiles);
    }

    FileStore bind(TarRevisions revisions) throws IOException {
        try (ShutDown.ShutDownCloser ignored = this.shutDown.keepAlive();){
            this.revisions = revisions;
            this.revisions.bind(this, this.tracker, this.initialNode());
            FileStore fileStore = this;
            return fileStore;
        }
    }

    @Nonnull
    private Supplier<RecordId> initialNode() {
        return new Supplier<RecordId>(){

            public RecordId get() {
                try {
                    DefaultSegmentWriter writer = DefaultSegmentWriterBuilder.defaultSegmentWriterBuilder("init").build(FileStore.this);
                    NodeBuilder builder = EmptyNodeState.EMPTY_NODE.builder();
                    builder.setChildNode("root", EmptyNodeState.EMPTY_NODE);
                    SegmentNodeState node = new SegmentNodeState((SegmentReader)FileStore.this.segmentReader, writer, FileStore.this.getBlobStore(), writer.writeNode(builder.getNodeState()));
                    writer.flush();
                    return node.getRecordId();
                }
                catch (IOException e) {
                    String msg = "Failed to write initial node";
                    log.error(msg, (Throwable)e);
                    throw new IllegalStateException(msg, e);
                }
            }
        };
    }

    @Nonnull
    private GCGeneration getGcGeneration() {
        return this.revisions.getHead().getSegmentId().getGcGeneration();
    }

    public Runnable getGCRunner() {
        return new SafeRunnable(String.format("TarMK revision gc [%s]", this.directory), () -> {
            try (ShutDown.ShutDownCloser ignored = this.shutDown.keepAlive();){
                this.garbageCollector.run();
            }
            catch (IOException e) {
                log.error("Error running revision garbage collection", (Throwable)e);
            }
        });
    }

    public GCNodeWriteMonitor getGCNodeWriteMonitor() {
        return this.garbageCollector.getGCNodeWriteMonitor();
    }

    private long size() {
        try (ShutDown.ShutDownCloser ignored = this.shutDown.keepAlive();){
            long l = this.tarFiles.size();
            return l;
        }
    }

    public int readerCount() {
        try (ShutDown.ShutDownCloser ignored = this.shutDown.keepAlive();){
            int n = this.tarFiles.readerCount();
            return n;
        }
    }

    public FileStoreStats getStats() {
        return this.stats;
    }

    private void doFlush() throws IOException {
        if (this.revisions == null) {
            log.debug("No TarRevisions available, skipping flush");
            return;
        }
        this.revisions.flush(() -> {
            this.segmentWriter.flush();
            this.tarFiles.flush();
            this.stats.flushed();
        });
    }

    public void flush() throws IOException {
        try (ShutDown.ShutDownCloser ignored = this.shutDown.keepAlive();){
            this.doFlush();
        }
    }

    public void tryFlush() {
        try (ShutDown.ShutDownCloser ignore = this.shutDown.tryKeepAlive();){
            if (this.shutDown.isShutDown()) {
                log.debug("Shut down in progress, skipping flush");
            } else if (this.revisions == null) {
                log.debug("No TarRevisions available, skipping flush");
            } else {
                this.revisions.tryFlush(() -> {
                    this.segmentWriter.flush();
                    this.tarFiles.flush();
                    this.stats.flushed();
                });
            }
        }
        catch (IOException e) {
            log.warn("Failed to flush the TarMK at {}", (Object)this.directory, (Object)e);
        }
    }

    public void fullGC() throws IOException {
        try (ShutDown.ShutDownCloser ignored = this.shutDown.keepAlive();){
            this.garbageCollector.runFull();
        }
    }

    public void tailGC() throws IOException {
        try (ShutDown.ShutDownCloser ignored = this.shutDown.keepAlive();){
            this.garbageCollector.runTail();
        }
    }

    public boolean compactFull() {
        try (ShutDown.ShutDownCloser ignored = this.shutDown.keepAlive();){
            boolean bl = this.garbageCollector.compactFull().isSuccess();
            return bl;
        }
    }

    public boolean compactTail() {
        try (ShutDown.ShutDownCloser ignored = this.shutDown.keepAlive();){
            boolean bl = this.garbageCollector.compactTail().isSuccess();
            return bl;
        }
    }

    public void cleanup() throws IOException {
        try (ShutDown.ShutDownCloser ignored = this.shutDown.keepAlive();){
            this.fileReaper.add(this.garbageCollector.cleanup());
        }
    }

    public void collectBlobReferences(Consumer<String> collector) throws IOException {
        try (ShutDown.ShutDownCloser ignored = this.shutDown.keepAlive();){
            this.garbageCollector.collectBlobReferences(collector);
        }
    }

    public void cancelGC() {
        this.garbageCollector.cancel();
    }

    @Override
    @Nonnull
    public SegmentWriter getWriter() {
        try (ShutDown.ShutDownCloser ignored = this.shutDown.keepAlive();){
            SegmentWriter segmentWriter = this.segmentWriter;
            return segmentWriter;
        }
    }

    @Override
    @Nonnull
    public TarRevisions getRevisions() {
        try (ShutDown.ShutDownCloser ignored = this.shutDown.keepAlive();){
            TarRevisions tarRevisions = this.revisions;
            return tarRevisions;
        }
    }

    @Override
    public void close() {
        try (ShutDown.ShutDownCloser ignored = this.shutDown.shutDown();){
            this.fileStoreScheduler.close();
            try {
                this.doFlush();
            }
            catch (IOException e) {
                log.warn("Unable to flush the store", (Throwable)e);
            }
            Closer closer = Closer.create();
            closer.register((Closeable)this.lockFile);
            closer.register(this.lock::release);
            closer.register((Closeable)this.tarFiles);
            closer.register((Closeable)this.revisions);
            FileStore.closeAndLogOnFail((Closeable)closer);
        }
        System.gc();
        this.fileReaper.reap();
        log.info("TarMK closed: {}", (Object)this.directory);
    }

    @Override
    public boolean containsSegment(SegmentId id) {
        try (ShutDown.ShutDownCloser ignored = this.shutDown.keepAlive();){
            boolean bl = this.tarFiles.containsSegment(id.getMostSignificantBits(), id.getLeastSignificantBits());
            return bl;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    @Nonnull
    public Segment readSegment(SegmentId id) {
        try (ShutDown.ShutDownCloser ignored = this.shutDown.keepAlive();){
            Segment segment = this.segmentCache.getSegment(id, () -> this.readSegmentUncached(this.tarFiles, id));
            return segment;
        }
        catch (UncheckedExecutionException | ExecutionException e) {
            SegmentNotFoundException snfe = FileStore.asSegmentNotFoundException((Exception)e, id);
            this.snfeListener.notify(id, snfe);
            throw snfe;
        }
    }

    @Override
    public void writeSegment(SegmentId id, byte[] buffer, int offset, int length) throws IOException {
        try (ShutDown.ShutDownCloser ignored = this.shutDown.keepAlive();){
            Segment segment = null;
            GCGeneration generation = GCGeneration.NULL;
            Set<UUID> references = null;
            Set<String> binaryReferences = null;
            if (id.isDataSegmentId()) {
                ByteBuffer data;
                if (offset > 4096) {
                    data = ByteBuffer.allocate(length);
                    data.put(buffer, offset, length);
                    data.rewind();
                } else {
                    data = ByteBuffer.wrap(buffer, offset, length);
                }
                segment = new Segment(this.tracker, this.segmentReader, id, data);
                generation = segment.getGcGeneration();
                references = FileStore.readReferences(segment);
                binaryReferences = FileStore.readBinaryReferences(segment);
            }
            this.tarFiles.writeSegment(id.asUUID(), buffer, offset, length, generation, references, binaryReferences);
            if (segment != null) {
                this.segmentCache.putSegment(segment);
            }
        }
    }

    private void checkDiskSpace(SegmentGCOptions gcOptions) {
        long availableDiskSpace;
        long repositoryDiskSpace = this.size();
        boolean updated = SegmentGCOptions.isDiskSpaceSufficient(repositoryDiskSpace, availableDiskSpace = this.directory.getFreeSpace());
        boolean previous = this.sufficientDiskSpace.getAndSet(updated);
        if (previous && !updated) {
            log.warn("Available disk space ({}) is too low, current repository size is approx. {}", (Object)IOUtils.humanReadableByteCount((long)availableDiskSpace), (Object)IOUtils.humanReadableByteCount((long)repositoryDiskSpace));
        }
        if (updated && !previous) {
            log.info("Available disk space ({}) is sufficient again for repository operations, current repository size is approx. {}", (Object)IOUtils.humanReadableByteCount((long)availableDiskSpace), (Object)IOUtils.humanReadableByteCount((long)repositoryDiskSpace));
        }
    }

    private static abstract class CompactionResult {
        @Nonnull
        private final GCGeneration currentGeneration;

        protected CompactionResult(@Nonnull GCGeneration currentGeneration) {
            this.currentGeneration = currentGeneration;
        }

        static CompactionResult succeeded(final @Nonnull SegmentGCOptions.GCType gcType, final @Nonnull GCGeneration newGeneration, final @Nonnull SegmentGCOptions gcOptions, final @Nonnull RecordId compactedRootId) {
            return new CompactionResult(newGeneration){

                @Override
                Predicate<GCGeneration> reclaimer() {
                    return Reclaimers.newOldReclaimer(gcType, newGeneration, gcOptions.getRetainedGenerations());
                }

                @Override
                boolean isSuccess() {
                    return true;
                }

                @Override
                RecordId getCompactedRootId() {
                    return compactedRootId;
                }
            };
        }

        static CompactionResult aborted(@Nonnull GCGeneration currentGeneration, final @Nonnull GCGeneration failedGeneration) {
            return new CompactionResult(currentGeneration){

                @Override
                Predicate<GCGeneration> reclaimer() {
                    return Reclaimers.newExactReclaimer(failedGeneration);
                }

                @Override
                boolean isSuccess() {
                    return false;
                }
            };
        }

        static CompactionResult skipped(final @Nonnull SegmentGCOptions.GCType lastGCType, final @Nonnull GCGeneration currentGeneration, final @Nonnull SegmentGCOptions gcOptions, final @Nonnull RecordId compactedRootId) {
            return new CompactionResult(currentGeneration){

                @Override
                Predicate<GCGeneration> reclaimer() {
                    return Reclaimers.newOldReclaimer(lastGCType, currentGeneration, gcOptions.getRetainedGenerations());
                }

                @Override
                boolean isSuccess() {
                    return true;
                }

                @Override
                RecordId getCompactedRootId() {
                    return compactedRootId;
                }
            };
        }

        abstract Predicate<GCGeneration> reclaimer();

        abstract boolean isSuccess();

        RecordId getCompactedRootId() {
            return RecordId.NULL;
        }

        String gcInfo() {
            return "gc-count=" + GC_COUNT + ",gc-status=" + (this.isSuccess() ? "success" : "failed") + ",store-generation=" + this.currentGeneration + ",reclaim-predicate=" + this.reclaimer();
        }
    }

    private class GarbageCollector {
        @Nonnull
        private final SegmentGCOptions gcOptions;
        @Nonnull
        private final PrefixedGCListener gcListener;
        @Nonnull
        private final GCJournal gcJournal;
        @Nonnull
        private final WriterCacheManager cacheManager;
        @Nonnull
        private GCNodeWriteMonitor compactionMonitor = GCNodeWriteMonitor.EMPTY;
        private volatile boolean cancelled;
        private long lastSuccessfullGC;
        @Nonnull
        private SegmentGCOptions.GCType lastCompactionType = SegmentGCOptions.GCType.FULL;

        GarbageCollector(@Nonnull SegmentGCOptions gcOptions, @Nonnull GCListener gcListener, @Nonnull GCJournal gcJournal, WriterCacheManager cacheManager) {
            this.gcOptions = gcOptions;
            this.gcListener = new PrefixedGCListener(gcListener, GC_COUNT);
            this.gcJournal = gcJournal;
            this.cacheManager = cacheManager;
        }

        GCNodeWriteMonitor getGCNodeWriteMonitor() {
            return this.compactionMonitor;
        }

        synchronized void run() throws IOException {
            switch (this.gcOptions.getGCType()) {
                case FULL: {
                    this.runFull();
                    break;
                }
                case TAIL: {
                    this.runTail();
                    break;
                }
                default: {
                    throw new IllegalStateException("Invalid GC type");
                }
            }
        }

        synchronized void runFull() throws IOException {
            this.run(true, (Supplier<CompactionResult>)((Supplier)this::compactFull));
        }

        synchronized void runTail() throws IOException {
            this.run(false, (Supplier<CompactionResult>)((Supplier)this::compactTail));
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void run(boolean full, Supplier<CompactionResult> compact) throws IOException {
            block29: {
                try {
                    GC_COUNT.incrementAndGet();
                    this.gcListener.info("started", new Object[0]);
                    long dt = System.currentTimeMillis() - this.lastSuccessfullGC;
                    if (dt < GC_BACKOFF) {
                        this.gcListener.skipped("skipping garbage collection as it already ran less than {} hours ago ({} s).", GC_BACKOFF / 3600000L, dt / 1000L);
                        return;
                    }
                    boolean sufficientEstimatedGain = true;
                    if (this.gcOptions.isEstimationDisabled()) {
                        this.gcListener.info("estimation skipped because it was explicitly disabled", new Object[0]);
                    } else if (this.gcOptions.isPaused()) {
                        this.gcListener.info("estimation skipped because compaction is paused", new Object[0]);
                    } else {
                        this.gcListener.info("estimation started", new Object[0]);
                        this.gcListener.updateStatus(SegmentGCStatus.ESTIMATION.message());
                        PrintableStopwatch watch = PrintableStopwatch.createStarted();
                        GCEstimationResult estimation = this.estimateCompactionGain(full);
                        sufficientEstimatedGain = estimation.isGcNeeded();
                        String gcLog = estimation.getGcLog();
                        if (sufficientEstimatedGain) {
                            this.gcListener.info("estimation completed in {}. {}", watch, gcLog);
                        } else {
                            this.gcListener.skipped("estimation completed in {}. {}", watch, gcLog);
                        }
                    }
                    if (!sufficientEstimatedGain) break block29;
                    try (GCMemoryBarrier ignored = new GCMemoryBarrier(FileStore.this.sufficientMemory, this.gcListener, this.gcOptions);){
                        if (this.gcOptions.isPaused()) {
                            this.gcListener.skipped("compaction paused", new Object[0]);
                        } else if (!FileStore.this.sufficientMemory.get()) {
                            this.gcListener.skipped("compaction skipped. Not enough memory", new Object[0]);
                        } else {
                            CompactionResult compactionResult = (CompactionResult)compact.get();
                            if (compactionResult.isSuccess()) {
                                this.lastSuccessfullGC = System.currentTimeMillis();
                            } else {
                                this.gcListener.info("cleaning up after failed compaction", new Object[0]);
                            }
                            FileStore.this.fileReaper.add(this.cleanup(compactionResult));
                        }
                    }
                }
                finally {
                    this.compactionMonitor.finished();
                    this.gcListener.updateStatus(SegmentGCStatus.IDLE.message());
                }
            }
        }

        GCEstimationResult estimateCompactionGain(boolean full) {
            return new SizeDeltaGcEstimation(this.gcOptions.getGcSizeDeltaEstimation(), this.gcJournal, FileStore.this.tarFiles.size(), full).estimate();
        }

        @Nonnull
        private CompactionResult compactionAborted(@Nonnull GCGeneration generation) {
            this.gcListener.compactionFailed(generation);
            return CompactionResult.aborted(FileStore.this.getGcGeneration(), generation);
        }

        @Nonnull
        private CompactionResult compactionSucceeded(@Nonnull SegmentGCOptions.GCType gcType, @Nonnull GCGeneration generation, @Nonnull RecordId compactedRootId) {
            this.gcListener.compactionSucceeded(generation);
            return CompactionResult.succeeded(gcType, generation, this.gcOptions, compactedRootId);
        }

        @CheckForNull
        private SegmentNodeState getBase() {
            String root = this.gcJournal.read().getRoot();
            RecordId rootId = RecordId.fromString(FileStore.this.tracker, root);
            if (RecordId.NULL.equals(rootId)) {
                return null;
            }
            try {
                SegmentNodeState node = FileStore.this.segmentReader.readNode(rootId);
                node.getPropertyCount();
                return node;
            }
            catch (SegmentNotFoundException snfe) {
                this.gcListener.error("base state " + rootId + " is not accessible", (Exception)((Object)snfe));
                return null;
            }
        }

        synchronized CompactionResult compactFull() {
            this.gcListener.info("running full compaction", new Object[0]);
            return this.compact(SegmentGCOptions.GCType.FULL, EmptyNodeState.EMPTY_NODE, FileStore.this.getGcGeneration().nextFull());
        }

        synchronized CompactionResult compactTail() {
            this.gcListener.info("running tail compaction", new Object[0]);
            SegmentNodeState base = this.getBase();
            if (base != null) {
                return this.compact(SegmentGCOptions.GCType.TAIL, base, FileStore.this.getGcGeneration().nextTail());
            }
            this.gcListener.info("no base state available, running full compaction instead", new Object[0]);
            return this.compact(SegmentGCOptions.GCType.FULL, EmptyNodeState.EMPTY_NODE, FileStore.this.getGcGeneration().nextFull());
        }

        private CompactionResult compact(@Nonnull SegmentGCOptions.GCType gcType, @Nonnull NodeState base, @Nonnull GCGeneration newGeneration) {
            try {
                PrintableStopwatch watch = PrintableStopwatch.createStarted();
                this.gcListener.info("compaction started, gc options={}, current generation={}, new generation={}", this.gcOptions, FileStore.this.getHead().getRecordId().getSegment().getGcGeneration(), newGeneration);
                this.gcListener.updateStatus(SegmentGCStatus.COMPACTION.message());
                GCJournal.GCJournalEntry gcEntry = this.gcJournal.read();
                long initialSize = FileStore.this.size();
                DefaultSegmentWriter writer = DefaultSegmentWriterBuilder.defaultSegmentWriterBuilder("c").with(this.cacheManager).withGeneration(newGeneration).withoutWriterPool().build(FileStore.this);
                CancelCompactionSupplier cancel = new CancelCompactionSupplier(FileStore.this);
                this.compactionMonitor = new GCNodeWriteMonitor(this.gcOptions.getGcLogInterval(), this.gcListener);
                this.compactionMonitor.init(gcEntry.getRepoSize(), gcEntry.getNodes(), initialSize);
                CheckpointCompactor compactor = new CheckpointCompactor(this.gcListener, FileStore.this.segmentReader, writer, FileStore.this.getBlobStore(), cancel, this.compactionMonitor);
                SegmentNodeState head = FileStore.this.getHead();
                SegmentNodeState compacted = compactor.compact(base, head, base);
                if (compacted == null) {
                    this.gcListener.warn("compaction cancelled: {}.", cancel);
                    return this.compactionAborted(newGeneration);
                }
                this.gcListener.info("compaction cycle 0 completed in {}. Compacted {} to {}", watch, head.getRecordId(), compacted.getRecordId());
                int cycles = 0;
                boolean success = false;
                SegmentNodeState previousHead = head;
                while (cycles < this.gcOptions.getRetryCount() && !(success = FileStore.this.revisions.setHead(previousHead.getRecordId(), compacted.getRecordId(), TarRevisions.EXPEDITE_OPTION))) {
                    this.gcListener.info("compaction detected concurrent commits while compacting. Compacting these commits. Cycle {} of {}", ++cycles, this.gcOptions.getRetryCount());
                    this.gcListener.updateStatus(SegmentGCStatus.COMPACTION_RETRY.message() + cycles);
                    PrintableStopwatch cycleWatch = PrintableStopwatch.createStarted();
                    head = FileStore.this.getHead();
                    compacted = compactor.compact((NodeState)previousHead, head, (NodeState)compacted);
                    if (compacted == null) {
                        this.gcListener.warn("compaction cancelled: {}.", cancel);
                        return this.compactionAborted(newGeneration);
                    }
                    this.gcListener.info("compaction cycle {} completed in {}. Compacted {} against {} to {}", cycles, cycleWatch, head.getRecordId(), previousHead.getRecordId(), compacted.getRecordId());
                    previousHead = head;
                }
                if (!success) {
                    this.gcListener.info("compaction gave up compacting concurrent commits after {} cycles.", cycles);
                    int forceTimeout = this.gcOptions.getForceTimeout();
                    if (forceTimeout > 0) {
                        this.gcListener.info("trying to force compact remaining commits for {} seconds. Concurrent commits to the store will be blocked.", forceTimeout);
                        this.gcListener.updateStatus(SegmentGCStatus.COMPACTION_FORCE_COMPACT.message());
                        PrintableStopwatch forceWatch = PrintableStopwatch.createStarted();
                        ++cycles;
                        cancel.timeOutAfter(forceTimeout, TimeUnit.SECONDS);
                        compacted = this.forceCompact(previousHead, compacted, compactor);
                        boolean bl = success = compacted != null;
                        if (success) {
                            this.gcListener.info("compaction succeeded to force compact remaining commits after {}.", forceWatch);
                        } else if (cancel.get().booleanValue()) {
                            this.gcListener.warn("compaction failed to force compact remaining commits after {}. Compaction was cancelled: {}.", forceWatch, cancel);
                        } else {
                            this.gcListener.warn("compaction failed to force compact remaining commits. after {}. Could not acquire exclusive access to the node store.", forceWatch);
                        }
                    }
                }
                if (success) {
                    this.lastCompactionType = gcType;
                    writer.flush();
                    FileStore.this.flush();
                    this.gcListener.info("compaction succeeded in {}, after {} cycles", watch, cycles);
                    return this.compactionSucceeded(gcType, newGeneration, compacted.getRecordId());
                }
                this.gcListener.info("compaction failed after {}, and {} cycles", watch, cycles);
                return this.compactionAborted(newGeneration);
            }
            catch (InterruptedException e) {
                this.gcListener.error("compaction interrupted", e);
                Thread.currentThread().interrupt();
                return this.compactionAborted(newGeneration);
            }
            catch (IOException e) {
                this.gcListener.error("compaction encountered an error", e);
                return this.compactionAborted(newGeneration);
            }
        }

        private SegmentNodeState forceCompact(final @Nonnull NodeState base, final @Nonnull NodeState onto, final @Nonnull CheckpointCompactor compactor) throws InterruptedException {
            RecordId compactedId = FileStore.this.revisions.setHead(new Function<RecordId, RecordId>(){

                @Nullable
                public RecordId apply(RecordId headId) {
                    try {
                        long t0 = System.currentTimeMillis();
                        SegmentNodeState after = compactor.compact(base, FileStore.this.segmentReader.readNode(headId), onto);
                        if (after == null) {
                            GarbageCollector.this.gcListener.info("compaction cancelled after {} seconds", (System.currentTimeMillis() - t0) / 1000L);
                            return null;
                        }
                        return after.getRecordId();
                    }
                    catch (IOException e) {
                        GarbageCollector.this.gcListener.error("error during forced compaction.", e);
                        return null;
                    }
                }
            }, TarRevisions.timeout(this.gcOptions.getForceTimeout(), TimeUnit.SECONDS));
            return compactedId != null ? FileStore.this.segmentReader.readNode(compactedId) : null;
        }

        private CleanupContext newCleanupContext(final Predicate<GCGeneration> old) {
            return new CleanupContext(){

                private boolean isUnreferencedBulkSegment(UUID id, boolean referenced) {
                    return !SegmentId.isDataSegmentId(id.getLeastSignificantBits()) && !referenced;
                }

                private boolean isOldDataSegment(UUID id, GCGeneration generation) {
                    return SegmentId.isDataSegmentId(id.getLeastSignificantBits()) && old.apply((Object)generation);
                }

                @Override
                public Collection<UUID> initialReferences() {
                    HashSet references = Sets.newHashSet();
                    for (SegmentId id : FileStore.this.tracker.getReferencedSegmentIds()) {
                        if (!id.isBulkSegmentId()) continue;
                        references.add(id.asUUID());
                    }
                    return references;
                }

                @Override
                public boolean shouldReclaim(UUID id, GCGeneration generation, boolean referenced) {
                    return this.isUnreferencedBulkSegment(id, referenced) || this.isOldDataSegment(id, generation);
                }

                @Override
                public boolean shouldFollow(UUID from, UUID to) {
                    return !SegmentId.isDataSegmentId(to.getLeastSignificantBits());
                }
            };
        }

        @Nonnull
        synchronized List<File> cleanup() throws IOException {
            return this.cleanup(CompactionResult.skipped(this.lastCompactionType, FileStore.this.getGcGeneration(), ((FileStore)FileStore.this).garbageCollector.gcOptions, FileStore.this.revisions.getHead()));
        }

        @Nonnull
        private List<File> cleanup(@Nonnull CompactionResult compactionResult) throws IOException {
            PrintableStopwatch watch = PrintableStopwatch.createStarted();
            this.gcListener.info("cleanup started using reclaimer {}", compactionResult.reclaimer());
            this.gcListener.updateStatus(SegmentGCStatus.CLEANUP.message());
            FileStore.this.segmentCache.clear();
            System.gc();
            TarFiles.CleanupResult cleanupResult = FileStore.this.tarFiles.cleanup(this.newCleanupContext(compactionResult.reclaimer()));
            if (cleanupResult.isInterrupted()) {
                this.gcListener.info("cleanup interrupted", new Object[0]);
            }
            FileStore.this.tracker.clearSegmentIdTables(cleanupResult.getReclaimedSegmentIds(), compactionResult.gcInfo());
            this.gcListener.info("cleanup marking files for deletion: {}", this.toFileNames(cleanupResult.getRemovableFiles()));
            long finalSize = FileStore.this.size();
            long reclaimedSize = cleanupResult.getReclaimedSize();
            FileStore.this.stats.reclaimed(reclaimedSize);
            this.gcJournal.persist(reclaimedSize, finalSize, FileStore.this.getGcGeneration(), this.compactionMonitor.getCompactedNodes(), compactionResult.getCompactedRootId().toString10());
            this.gcListener.cleaned(reclaimedSize, finalSize);
            this.gcListener.info("cleanup completed in {}. Post cleanup size is {} and space reclaimed {}.", watch, PrintableBytes.newPrintableBytes(finalSize), PrintableBytes.newPrintableBytes(reclaimedSize));
            return cleanupResult.getRemovableFiles();
        }

        private String toFileNames(@Nonnull List<File> files) {
            if (files.isEmpty()) {
                return "none";
            }
            return Joiner.on((String)",").join(files);
        }

        synchronized void collectBlobReferences(Consumer<String> collector) throws IOException {
            FileStore.this.segmentWriter.flush();
            FileStore.this.tarFiles.collectBlobReferences(collector, Reclaimers.newOldReclaimer(this.lastCompactionType, FileStore.this.getGcGeneration(), this.gcOptions.getRetainedGenerations()));
        }

        void cancel() {
            this.cancelled = true;
        }

        private class CancelCompactionSupplier
        implements Supplier<Boolean> {
            private final FileStore store;
            private String reason;
            private volatile long baseLine;
            private volatile long deadline;

            public CancelCompactionSupplier(FileStore store) {
                GarbageCollector.this.cancelled = false;
                this.store = store;
            }

            public void timeOutAfter(long duration, @Nonnull TimeUnit unit) {
                this.baseLine = System.currentTimeMillis();
                this.deadline = this.baseLine + TimeUnit.MILLISECONDS.convert(duration, unit);
            }

            public Boolean get() {
                if (!this.store.sufficientDiskSpace.get()) {
                    this.reason = "Not enough disk space";
                    return true;
                }
                if (!this.store.sufficientMemory.get()) {
                    this.reason = "Not enough memory";
                    return true;
                }
                if (this.store.shutDown.isShutDown()) {
                    this.reason = "The FileStore is shutting down";
                    return true;
                }
                if (GarbageCollector.this.cancelled) {
                    this.reason = "Cancelled by user";
                    return true;
                }
                if (this.deadline > 0L && System.currentTimeMillis() > this.deadline) {
                    long dt = TimeUnit.SECONDS.convert(System.currentTimeMillis() - this.baseLine, TimeUnit.MILLISECONDS);
                    this.reason = "Timeout after " + dt + " seconds";
                    return true;
                }
                return false;
            }

            public String toString() {
                return this.reason;
            }
        }
    }
}

