/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.collect.AbstractIterator;
import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Iterator;
import java.util.UUID;
import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import org.apache.commons.io.HexDump;
import org.apache.commons.io.output.ByteArrayOutputStream;
import org.apache.jackrabbit.oak.api.PropertyState;
import org.apache.jackrabbit.oak.api.Type;
import org.apache.jackrabbit.oak.commons.IOUtils;
import org.apache.jackrabbit.oak.commons.StringUtils;
import org.apache.jackrabbit.oak.plugins.memory.PropertyStates;
import org.apache.jackrabbit.oak.segment.IdentityRecordNumbers;
import org.apache.jackrabbit.oak.segment.IllegalSegmentReferences;
import org.apache.jackrabbit.oak.segment.ImmutableRecordNumbers;
import org.apache.jackrabbit.oak.segment.ListRecord;
import org.apache.jackrabbit.oak.segment.PropertyTemplate;
import org.apache.jackrabbit.oak.segment.RecordId;
import org.apache.jackrabbit.oak.segment.RecordNumbers;
import org.apache.jackrabbit.oak.segment.RecordType;
import org.apache.jackrabbit.oak.segment.SegmentDump;
import org.apache.jackrabbit.oak.segment.SegmentId;
import org.apache.jackrabbit.oak.segment.SegmentIdProvider;
import org.apache.jackrabbit.oak.segment.SegmentReader;
import org.apache.jackrabbit.oak.segment.SegmentReferences;
import org.apache.jackrabbit.oak.segment.SegmentStream;
import org.apache.jackrabbit.oak.segment.SegmentVersion;
import org.apache.jackrabbit.oak.segment.Template;
import org.apache.jackrabbit.oak.segment.data.RecordIdData;
import org.apache.jackrabbit.oak.segment.data.SegmentData;
import org.apache.jackrabbit.oak.segment.data.StringData;
import org.apache.jackrabbit.oak.segment.file.tar.GCGeneration;

public class Segment {
    static final int HEADER_SIZE = 32;
    static final int SEGMENT_REFERENCE_SIZE = 16;
    static final int RECORD_SIZE = 9;
    static final int RECORD_ID_BYTES = 6;
    static final int RECORD_ALIGN_BITS = 2;
    static final int MAX_SEGMENT_SIZE = 262144;
    static final int SMALL_LIMIT = 128;
    static final int MEDIUM_LIMIT = 16512;
    static final int BLOB_ID_SMALL_LIMIT = 4096;
    static final int GC_FULL_GENERATION_OFFSET = 4;
    static final int GC_GENERATION_OFFSET = 10;
    static final int REFERENCED_SEGMENT_ID_COUNT_OFFSET = 14;
    static final int RECORD_NUMBER_COUNT_OFFSET = 18;
    @Nonnull
    private final SegmentReader reader;
    @Nonnull
    private final SegmentId id;
    private final SegmentData data;
    @Nonnull
    private final SegmentVersion version;
    private final RecordNumbers recordNumbers;
    private final SegmentReferences segmentReferences;
    private volatile String info;

    static int align(int address, int boundary) {
        return address + boundary - 1 & ~(boundary - 1);
    }

    Segment(@Nonnull SegmentId id, @Nonnull SegmentReader reader, @Nonnull byte[] buffer, @Nonnull RecordNumbers recordNumbers, @Nonnull SegmentReferences segmentReferences, @Nonnull String info) {
        this.id = (SegmentId)Preconditions.checkNotNull((Object)id);
        this.reader = (SegmentReader)Preconditions.checkNotNull((Object)reader);
        this.info = (String)Preconditions.checkNotNull((Object)info);
        this.data = id.isDataSegmentId() ? SegmentData.newSegmentData(ByteBuffer.wrap(buffer)) : SegmentData.newRawSegmentData(ByteBuffer.wrap(buffer));
        this.version = SegmentVersion.fromByte(buffer[3]);
        this.recordNumbers = recordNumbers;
        this.segmentReferences = segmentReferences;
        id.loaded(this);
    }

    public Segment(@Nonnull SegmentIdProvider idProvider, @Nonnull SegmentReader reader, final @Nonnull SegmentId id, final @Nonnull ByteBuffer data) {
        this.reader = (SegmentReader)Preconditions.checkNotNull((Object)reader);
        this.id = (SegmentId)Preconditions.checkNotNull((Object)id);
        if (id.isDataSegmentId()) {
            this.data = SegmentData.newSegmentData(((ByteBuffer)Preconditions.checkNotNull((Object)data)).slice());
            byte segmentVersion = this.data.getVersion();
            Preconditions.checkState((this.data.getSignature().equals("0aK") && SegmentVersion.isValid(segmentVersion) ? 1 : 0) != 0, (Object)new Object(){

                public String toString() {
                    return String.format("Invalid segment format. Dumping segment %s\n%s", id, Segment.toHex(data.array()));
                }
            });
            this.version = SegmentVersion.fromByte(segmentVersion);
            this.recordNumbers = this.readRecordNumberOffsets();
            this.segmentReferences = this.readReferencedSegments(idProvider);
        } else {
            this.data = SegmentData.newRawSegmentData(((ByteBuffer)Preconditions.checkNotNull((Object)data)).slice());
            this.version = SegmentVersion.LATEST_VERSION;
            this.recordNumbers = new IdentityRecordNumbers();
            this.segmentReferences = new IllegalSegmentReferences();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static String toHex(byte[] bytes) {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        try {
            HexDump.dump((byte[])bytes, (long)0L, (OutputStream)out, (int)0);
            String string = out.toString(Charsets.UTF_8.name());
            return string;
        }
        catch (IOException e) {
            String string = "Error dumping segment: " + e.getMessage();
            return string;
        }
        finally {
            IOUtils.closeQuietly((Closeable)out);
        }
    }

    private RecordNumbers readRecordNumberOffsets() {
        int recordNumberCount = this.data.getRecordReferencesCount();
        if (recordNumberCount == 0) {
            return RecordNumbers.EMPTY_RECORD_NUMBERS;
        }
        int maxIndex = this.data.getRecordReferenceNumber(recordNumberCount - 1);
        byte[] types = new byte[maxIndex + 1];
        int[] offsets = new int[maxIndex + 1];
        Arrays.fill(offsets, -1);
        for (int i = 0; i < recordNumberCount; ++i) {
            int recordNumber = this.data.getRecordReferenceNumber(i);
            types[recordNumber] = this.data.getRecordReferenceType(i);
            offsets[recordNumber] = this.data.getRecordReferenceOffset(i);
        }
        return new ImmutableRecordNumbers(offsets, types);
    }

    private SegmentReferences readReferencedSegments(final SegmentIdProvider idProvider) {
        Preconditions.checkState((this.getReferencedSegmentIdCount() + 1 < 65535 ? 1 : 0) != 0, (Object)"Segment cannot have more than 0xffff references");
        final int referencedSegmentIdCount = this.getReferencedSegmentIdCount();
        final SegmentId[] refIds = new SegmentId[referencedSegmentIdCount];
        return new SegmentReferences(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             * Enabled force condition propagation
             * Lifted jumps to return sites
             */
            @Override
            public SegmentId getSegmentId(int reference) {
                Preconditions.checkArgument((reference <= referencedSegmentIdCount ? 1 : 0) != 0, (Object)"Segment reference out of bounds");
                SegmentId id = refIds[reference - 1];
                if (id != null) return id;
                SegmentId[] segmentIdArray = refIds;
                synchronized (refIds) {
                    id = refIds[reference - 1];
                    if (id != null) return id;
                    long msb = Segment.this.data.getSegmentReferenceMsb(reference - 1);
                    long lsb = Segment.this.data.getSegmentReferenceLsb(reference - 1);
                    refIds[reference - 1] = id = idProvider.newSegmentId(msb, lsb);
                    // ** MonitorExit[var3_3] (shouldn't be in output)
                    return id;
                }
            }

            @Override
            @Nonnull
            public Iterator<SegmentId> iterator() {
                return new AbstractIterator<SegmentId>(){
                    private int reference = 1;

                    protected SegmentId computeNext() {
                        if (this.reference <= referencedSegmentIdCount) {
                            return this.getSegmentId(this.reference++);
                        }
                        return (SegmentId)this.endOfData();
                    }
                };
            }
        };
    }

    public SegmentVersion getSegmentVersion() {
        return this.version;
    }

    public SegmentId getSegmentId() {
        return this.id;
    }

    public int getReferencedSegmentIdCount() {
        return this.data.getSegmentReferencesCount();
    }

    private int getRecordNumberCount() {
        return this.data.getRecordReferencesCount();
    }

    public UUID getReferencedSegmentId(int index) {
        return this.segmentReferences.getSegmentId(index + 1).asUUID();
    }

    public static GCGeneration getGcGeneration(SegmentData data, UUID segmentId) {
        if (SegmentId.isDataSegmentId(segmentId.getLeastSignificantBits())) {
            return GCGeneration.newGCGeneration(data.getGeneration(), data.getFullGeneration(), data.isCompacted());
        }
        return GCGeneration.NULL;
    }

    @Nonnull
    public GCGeneration getGcGeneration() {
        return Segment.getGcGeneration(this.data, this.id.asUUID());
    }

    @CheckForNull
    String getSegmentInfo() {
        if (this.info == null && this.id.isDataSegmentId()) {
            this.info = this.readString(((RecordNumbers.Entry)this.recordNumbers.iterator().next()).getRecordNumber());
        }
        return this.info;
    }

    public int size() {
        return this.data.size();
    }

    byte readByte(int recordNumber) {
        return this.data.readByte(this.recordNumbers.getOffset(recordNumber));
    }

    byte readByte(int recordNumber, int offset) {
        return this.data.readByte(this.recordNumbers.getOffset(recordNumber) + offset);
    }

    short readShort(int recordNumber) {
        return this.data.readShort(this.recordNumbers.getOffset(recordNumber));
    }

    int readInt(int recordNumber) {
        return this.data.readInt(this.recordNumbers.getOffset(recordNumber));
    }

    int readInt(int recordNumber, int offset) {
        return this.data.readInt(this.recordNumbers.getOffset(recordNumber) + offset);
    }

    long readLong(int recordNumber) {
        return this.data.readLong(this.recordNumbers.getOffset(recordNumber));
    }

    void readBytes(int recordNumber, int position, byte[] buffer, int offset, int length) {
        this.readBytes(recordNumber, position, length).get(buffer, offset, length);
    }

    ByteBuffer readBytes(int recordNumber, int position, int length) {
        return this.data.readBytes(this.recordNumbers.getOffset(recordNumber) + position, length);
    }

    @Nonnull
    RecordId readRecordId(int recordNumber, int rawOffset, int recordIdOffset) {
        int offset = this.recordNumbers.getOffset(recordNumber) + rawOffset + recordIdOffset * 6;
        RecordIdData recordIdData = this.data.readRecordId(offset);
        return new RecordId(this.dereferenceSegmentId(recordIdData.getSegmentReference()), recordIdData.getRecordNumber());
    }

    RecordId readRecordId(int recordNumber, int rawOffset) {
        return this.readRecordId(recordNumber, rawOffset, 0);
    }

    RecordId readRecordId(int recordNumber) {
        return this.readRecordId(recordNumber, 0, 0);
    }

    @Nonnull
    private SegmentId dereferenceSegmentId(int reference) {
        if (reference == 0) {
            return this.id;
        }
        SegmentId id = this.segmentReferences.getSegmentId(reference);
        if (id == null) {
            throw new IllegalStateException("Referenced segment not found");
        }
        return id;
    }

    @Nonnull
    String readString(int recordNumber) {
        StringData data = this.data.readString(this.recordNumbers.getOffset(recordNumber));
        if (data.isString()) {
            return data.getString();
        }
        if (data.isRecordId()) {
            SegmentId segmentId = this.dereferenceSegmentId(data.getRecordId().getSegmentReference());
            RecordId recordId = new RecordId(segmentId, data.getRecordId().getRecordNumber());
            ListRecord list = new ListRecord(recordId, (data.getLength() + 4096 - 1) / 4096);
            try (SegmentStream stream = new SegmentStream(new RecordId(this.id, recordNumber), list, (long)data.getLength());){
                String string = stream.getString();
                return string;
            }
        }
        throw new IllegalStateException("Invalid return value");
    }

    @Nonnull
    Template readTemplate(int recordNumber) {
        int head = this.readInt(recordNumber);
        boolean hasPrimaryType = (head & Integer.MIN_VALUE) != 0;
        boolean hasMixinTypes = (head & 0x40000000) != 0;
        boolean zeroChildNodes = (head & 0x20000000) != 0;
        boolean manyChildNodes = (head & 0x10000000) != 0;
        int mixinCount = head >> 18 & 0x3FF;
        int propertyCount = head & 0x3FFFF;
        int offset = 4;
        PropertyState primaryType = null;
        if (hasPrimaryType) {
            RecordId primaryId = this.readRecordId(recordNumber, offset);
            primaryType = PropertyStates.createProperty((String)"jcr:primaryType", (Object)this.reader.readString(primaryId), (Type)Type.NAME);
            offset += 6;
        }
        PropertyState mixinTypes = null;
        if (hasMixinTypes) {
            String[] mixins = new String[mixinCount];
            for (int i = 0; i < mixins.length; ++i) {
                RecordId mixinId = this.readRecordId(recordNumber, offset);
                mixins[i] = this.reader.readString(mixinId);
                offset += 6;
            }
            mixinTypes = PropertyStates.createProperty((String)"jcr:mixinTypes", Arrays.asList(mixins), (Type)Type.NAMES);
        }
        String childName = Template.ZERO_CHILD_NODES;
        if (manyChildNodes) {
            childName = "";
        } else if (!zeroChildNodes) {
            RecordId childNameId = this.readRecordId(recordNumber, offset);
            childName = this.reader.readString(childNameId);
            offset += 6;
        }
        PropertyTemplate[] properties = this.readProps(propertyCount, recordNumber, offset);
        return new Template(this.reader, primaryType, mixinTypes, properties, childName);
    }

    private PropertyTemplate[] readProps(int propertyCount, int recordNumber, int offset) {
        PropertyTemplate[] properties = new PropertyTemplate[propertyCount];
        if (propertyCount > 0) {
            RecordId id = this.readRecordId(recordNumber, offset);
            ListRecord propertyNames = new ListRecord(id, properties.length);
            offset += 6;
            for (int i = 0; i < propertyCount; ++i) {
                byte type = this.readByte(recordNumber, offset++);
                properties[i] = new PropertyTemplate(i, this.reader.readString(propertyNames.getEntry(i)), Type.fromTag((int)Math.abs(type), (type < 0 ? 1 : 0) != 0));
            }
        }
        return properties;
    }

    static long readLength(RecordId id) {
        return id.getSegment().readLength(id.getRecordNumber());
    }

    long readLength(int recordNumber) {
        return this.data.readLength(this.recordNumbers.getOffset(recordNumber));
    }

    public String toString() {
        return SegmentDump.dumpSegment(this.id, this.data.size(), this.info, this.getGcGeneration(), this.segmentReferences, this.recordNumbers, stream -> {
            try {
                this.data.hexDump((OutputStream)stream);
            }
            catch (IOException e) {
                e.printStackTrace(new PrintStream((OutputStream)stream));
            }
        });
    }

    public void writeTo(OutputStream stream) throws IOException {
        this.data.binDump(stream);
    }

    public void forEachRecord(RecordConsumer consumer) {
        for (RecordNumbers.Entry entry : this.recordNumbers) {
            consumer.consume(entry.getRecordNumber(), entry.getType(), entry.getOffset());
        }
    }

    int estimateMemoryUsage() {
        int size = 88;
        size += 56;
        if (this.id.isDataSegmentId()) {
            int recordNumberCount = this.getRecordNumberCount();
            size += 5 * recordNumberCount;
            int referencedSegmentIdCount = this.getReferencedSegmentIdCount();
            size += 8 * referencedSegmentIdCount;
            size += StringUtils.estimateMemoryUsage((String)this.info);
        }
        size += this.data.estimateMemoryUsage();
        return size += this.id.estimateMemoryUsage();
    }

    public static interface RecordConsumer {
        public void consume(int var1, RecordType var2, int var3);
    }
}

