/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.proto;

import bk-shade.com.google.common.annotations.VisibleForTesting;
import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.UnknownHostException;
import java.util.Arrays;
import org.apache.bookkeeper.bookie.Bookie;
import org.apache.bookkeeper.bookie.BookieCriticalThread;
import org.apache.bookkeeper.bookie.BookieException;
import org.apache.bookkeeper.bookie.ReadOnlyBookie;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.net.BookieSocketAddress;
import org.apache.bookkeeper.processor.RequestProcessor;
import org.apache.bookkeeper.proto.BookieNettyServer;
import org.apache.bookkeeper.proto.BookieRequestProcessor;
import org.apache.bookkeeper.replication.AutoRecoveryMain;
import org.apache.bookkeeper.replication.ReplicationException;
import org.apache.bookkeeper.stats.NullStatsLogger;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.bookkeeper.stats.StatsProvider;
import org.apache.bookkeeper.tls.SecurityException;
import org.apache.bookkeeper.tls.SecurityHandlerFactory;
import org.apache.bookkeeper.tls.SecurityProviderFactoryFactory;
import org.apache.bookkeeper.util.ReflectionUtils;
import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.configuration.Configuration;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BookieServer {
    final ServerConfiguration conf;
    BookieNettyServer nettyServer;
    private volatile boolean running = false;
    Bookie bookie;
    DeathWatcher deathWatcher;
    private static final Logger LOG = LoggerFactory.getLogger(BookieServer.class);
    int exitCode = 0;
    AutoRecoveryMain autoRecoveryMain = null;
    private boolean isAutoRecoveryDaemonEnabled;
    private final RequestProcessor requestProcessor;
    private final StatsLogger statsLogger;
    static final Options bkOpts = new Options();

    public BookieServer(ServerConfiguration conf) throws IOException, KeeperException, InterruptedException, BookieException, ReplicationException.UnavailableException, ReplicationException.CompatibilityException, SecurityException {
        this(conf, (StatsLogger)NullStatsLogger.INSTANCE);
    }

    public BookieServer(ServerConfiguration conf, StatsLogger statsLogger) throws IOException, KeeperException, InterruptedException, BookieException, ReplicationException.UnavailableException, ReplicationException.CompatibilityException, SecurityException {
        this.conf = conf;
        this.statsLogger = statsLogger;
        this.nettyServer = new BookieNettyServer(this.conf, null);
        try {
            this.bookie = this.newBookie(conf);
        }
        catch (IOException | InterruptedException | BookieException | KeeperException e) {
            this.nettyServer.shutdown();
            throw e;
        }
        SecurityHandlerFactory shFactory = SecurityProviderFactoryFactory.getSecurityProviderFactory(conf.getTLSProviderFactoryClass());
        this.requestProcessor = new BookieRequestProcessor(conf, this.bookie, statsLogger.scope("bookkeeper_server"), shFactory);
        this.nettyServer.setRequestProcessor(this.requestProcessor);
        this.isAutoRecoveryDaemonEnabled = conf.isAutoRecoveryDaemonEnabled();
        if (this.isAutoRecoveryDaemonEnabled) {
            this.autoRecoveryMain = new AutoRecoveryMain(conf, statsLogger.scope("replication"));
        }
    }

    protected Bookie newBookie(ServerConfiguration conf) throws IOException, KeeperException, InterruptedException, BookieException {
        return conf.isForceReadOnlyBookie() ? new ReadOnlyBookie(conf, this.statsLogger.scope("bookie")) : new Bookie(conf, this.statsLogger.scope("bookie"));
    }

    public void start() throws IOException, ReplicationException.UnavailableException, InterruptedException {
        this.bookie.start();
        if (!this.bookie.isRunning()) {
            this.exitCode = this.bookie.getExitCode();
            return;
        }
        if (this.isAutoRecoveryDaemonEnabled && this.autoRecoveryMain != null) {
            this.autoRecoveryMain.start();
        }
        this.nettyServer.start();
        this.running = true;
        this.deathWatcher = new DeathWatcher(this.conf);
        this.deathWatcher.start();
    }

    @VisibleForTesting
    public BookieSocketAddress getLocalAddress() throws UnknownHostException {
        return Bookie.getBookieAddress(this.conf);
    }

    @VisibleForTesting
    public Bookie getBookie() {
        return this.bookie;
    }

    @VisibleForTesting
    public void suspendProcessing() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Suspending bookie server, port is {}", (Object)this.conf.getBookiePort());
        }
        this.nettyServer.suspendProcessing();
    }

    @VisibleForTesting
    public void resumeProcessing() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Resuming bookie server, port is {}", (Object)this.conf.getBookiePort());
        }
        this.nettyServer.resumeProcessing();
    }

    public synchronized void shutdown() {
        LOG.info("Shutting down BookieServer");
        this.nettyServer.shutdown();
        if (!this.running) {
            return;
        }
        this.exitCode = this.bookie.shutdown();
        if (this.isAutoRecoveryDaemonEnabled && this.autoRecoveryMain != null) {
            this.autoRecoveryMain.shutdown();
        }
        this.requestProcessor.close();
        this.running = false;
    }

    public boolean isRunning() {
        return this.bookie.isRunning() && this.nettyServer.isRunning() && this.running;
    }

    public boolean isBookieRunning() {
        return this.bookie.isRunning();
    }

    public boolean isAutoRecoveryRunning() {
        return this.autoRecoveryMain != null && this.autoRecoveryMain.isAutoRecoveryRunning();
    }

    public void join() throws InterruptedException {
        this.bookie.join();
    }

    public int getExitCode() {
        return this.exitCode;
    }

    private static void printUsage() {
        HelpFormatter hf = new HelpFormatter();
        String header = "\nBookieServer provide an interface to start a bookie with configuration file and/or arguments.The settings in configuration file will be overwrite by provided arguments.\nOptions including:\n";
        String footer = "Here is an example:\n\tBookieServer -c bookie.conf -z localhost:2181 -m /bookkeeper/ledgers -p 3181 -j /mnt/journal -i \"/mnt/index1 /mnt/index2\" -l \"/mnt/ledger1 /mnt/ledger2 /mnt/ledger3\"\n";
        hf.printHelp("BookieServer [options]\n", header, bkOpts, footer, true);
    }

    private static void loadConfFile(ServerConfiguration conf, String confFile) throws IllegalArgumentException {
        try {
            conf.loadConf(new File(confFile).toURI().toURL());
            conf.validate();
        }
        catch (MalformedURLException e) {
            LOG.error("Could not open configuration file: " + confFile, (Throwable)e);
            throw new IllegalArgumentException();
        }
        catch (ConfigurationException e) {
            LOG.error("Malformed configuration file: " + confFile, (Throwable)e);
            throw new IllegalArgumentException();
        }
        LOG.info("Using configuration file " + confFile);
    }

    private static ServerConfiguration parseArgs(String[] args) throws IllegalArgumentException {
        try {
            BasicParser parser = new BasicParser();
            CommandLine cmdLine = parser.parse(bkOpts, args);
            if (cmdLine.hasOption('h')) {
                throw new IllegalArgumentException();
            }
            ServerConfiguration conf = new ServerConfiguration();
            if (cmdLine.hasOption('c')) {
                String confFile = cmdLine.getOptionValue("c");
                BookieServer.loadConfFile(conf, confFile);
            }
            if (cmdLine.hasOption("withAutoRecovery")) {
                conf.setAutoRecoveryDaemonEnabled(true);
            }
            if (cmdLine.hasOption("readOnly")) {
                conf.setForceReadOnlyBookie(true);
            }
            if (cmdLine.hasOption('z')) {
                String sZK = cmdLine.getOptionValue('z');
                LOG.info("Get cmdline zookeeper instance: " + sZK);
                conf.setZkServers(sZK);
            }
            if (cmdLine.hasOption('m')) {
                String sZkLedgersRootPath = cmdLine.getOptionValue('m');
                LOG.info("Get cmdline zookeeper ledger path: " + sZkLedgersRootPath);
                conf.setZkLedgersRootPath(sZkLedgersRootPath);
            }
            if (cmdLine.hasOption('p')) {
                String sPort = cmdLine.getOptionValue('p');
                LOG.info("Get cmdline bookie port: " + sPort);
                Integer iPort = Integer.parseInt(sPort);
                conf.setBookiePort(iPort);
            }
            if (cmdLine.hasOption('j')) {
                String sJournalDir = cmdLine.getOptionValue('j');
                LOG.info("Get cmdline journal dir: " + sJournalDir);
                conf.setJournalDirName(sJournalDir);
            }
            if (cmdLine.hasOption('i')) {
                String[] sIndexDirs = cmdLine.getOptionValues('i');
                LOG.info("Get cmdline index dirs: ");
                for (String index : sIndexDirs) {
                    LOG.info("indexDir : " + index);
                }
                conf.setIndexDirName(sIndexDirs);
            }
            if (cmdLine.hasOption('l')) {
                String[] sLedgerDirs = cmdLine.getOptionValues('l');
                LOG.info("Get cmdline ledger dirs: ");
                for (String ledger : sLedgerDirs) {
                    LOG.info("ledgerdir : " + ledger);
                }
                conf.setLedgerDirNames(sLedgerDirs);
            }
            return conf;
        }
        catch (ParseException e) {
            LOG.error("Error parsing command line arguments : ", (Throwable)e);
            throw new IllegalArgumentException(e);
        }
    }

    public static void main(String[] args) {
        ServerConfiguration conf = null;
        try {
            conf = BookieServer.parseArgs(args);
        }
        catch (IllegalArgumentException iae) {
            LOG.error("Error parsing command line arguments : ", (Throwable)iae);
            System.err.println(iae.getMessage());
            BookieServer.printUsage();
            System.exit(1);
        }
        StringBuilder sb = new StringBuilder();
        String[] ledgerDirNames = conf.getLedgerDirNames();
        for (int i = 0; i < ledgerDirNames.length; ++i) {
            if (i != 0) {
                sb.append(',');
            }
            sb.append(ledgerDirNames[i]);
        }
        String hello = String.format("Hello, I'm your bookie, listening on port %1$s. ZKServers are on %2$s. Journals are in %3$s. Ledgers are stored in %4$s.", conf.getBookiePort(), conf.getZkServers(), Arrays.asList(conf.getJournalDirNames()), sb);
        LOG.info(hello);
        try {
            Class<? extends StatsProvider> statsProviderClass = conf.getStatsProviderClass();
            StatsProvider statsProvider = ReflectionUtils.newInstance(statsProviderClass);
            statsProvider.start((Configuration)conf);
            final BookieServer bs = new BookieServer(conf, statsProvider.getStatsLogger(""));
            bs.start();
            Runtime.getRuntime().addShutdownHook(new Thread(){

                @Override
                public void run() {
                    bs.shutdown();
                    LOG.info("Shut down bookie server successfully");
                }
            });
            LOG.info("Register shutdown hook successfully");
            bs.join();
            statsProvider.stop();
            LOG.info("Stop stats provider");
            bs.shutdown();
            System.exit(bs.getExitCode());
        }
        catch (Exception e) {
            LOG.error("Exception running bookie server : ", (Throwable)e);
            System.exit(2);
        }
    }

    static {
        bkOpts.addOption("c", "conf", true, "Configuration for Bookie Server");
        bkOpts.addOption("withAutoRecovery", false, "Start Autorecovery service Bookie server");
        bkOpts.addOption("readOnly", false, "Force Start a ReadOnly Bookie server");
        bkOpts.addOption("z", "zkserver", true, "Zookeeper Server");
        bkOpts.addOption("m", "zkledgerpath", true, "Zookeeper ledgers root path");
        bkOpts.addOption("p", "bookieport", true, "bookie port exported");
        bkOpts.addOption("j", "journal", true, "bookie journal directory");
        Option indexDirs = new Option("i", "indexdirs", true, "bookie index directories");
        indexDirs.setArgs(10);
        bkOpts.addOption(indexDirs);
        Option ledgerDirs = new Option("l", "ledgerdirs", true, "bookie ledgers directories");
        ledgerDirs.setArgs(10);
        bkOpts.addOption(ledgerDirs);
        bkOpts.addOption("h", "help", false, "Print help message");
    }

    private class DeathWatcher
    extends BookieCriticalThread {
        private final int watchInterval;

        DeathWatcher(ServerConfiguration conf) {
            super("BookieDeathWatcher-" + conf.getBookiePort());
            this.watchInterval = conf.getDeathWatchInterval();
        }

        @Override
        public void run() {
            while (true) {
                try {
                    Thread.sleep(this.watchInterval);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
                if (!BookieServer.this.isBookieRunning()) break;
                if (!BookieServer.this.isAutoRecoveryDaemonEnabled || BookieServer.this.isAutoRecoveryRunning()) continue;
                LOG.error("Autorecovery daemon has stopped. Please check the logs");
                BookieServer.this.isAutoRecoveryDaemonEnabled = false;
            }
            BookieServer.this.shutdown();
            LOG.info("BookieDeathWatcher exited loop!");
        }
    }
}

