/*
Copyright 2020 The Knative Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package webhook

import (
	"context"
	"errors"
	"testing"

	corev1 "k8s.io/api/core/v1"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/apis/meta/v1/unstructured"
	"k8s.io/apimachinery/pkg/runtime"
	ktesting "k8s.io/client-go/testing"

	"knative.dev/pkg/apis"
	fakekubeclient "knative.dev/pkg/client/injection/kube/client/fake"
	"knative.dev/pkg/logging"
	logtesting "knative.dev/pkg/logging/testing"
	"knative.dev/pkg/ptr"
	v1 "knative.dev/serving/pkg/apis/serving/v1"
)

func TestExtraServiceValidation(t *testing.T) {
	goodConfigSpec := v1.ConfigurationSpec{
		Template: v1.RevisionTemplateSpec{
			Spec: v1.RevisionSpec{
				PodSpec: corev1.PodSpec{
					Containers: []corev1.Container{{
						Image: "busybox",
					}},
				},
			},
		},
	}

	om := metav1.ObjectMeta{
		Name:      "valid",
		Namespace: "foo",
		Annotations: map[string]string{
			"features.knative.dev/podspec-dryrun": "enabled",
		},
	}

	tests := []struct {
		name          string
		s             *v1.Service
		want          string
		modifyContext func(context.Context)
	}{{
		name: "valid run latest",
		s: &v1.Service{
			ObjectMeta: om,
			Spec: v1.ServiceSpec{
				ConfigurationSpec: goodConfigSpec,
				RouteSpec: v1.RouteSpec{
					Traffic: []v1.TrafficTarget{{
						LatestRevision: ptr.Bool(true),
						Percent:        ptr.Int64(100),
					}},
				},
			},
		},
		modifyContext: nil,
	}, {
		name: "dryrun fail",
		s: &v1.Service{
			ObjectMeta: om,
			Spec: v1.ServiceSpec{
				ConfigurationSpec: goodConfigSpec,
				RouteSpec: v1.RouteSpec{
					Traffic: []v1.TrafficTarget{{
						LatestRevision: ptr.Bool(true),
						Percent:        ptr.Int64(100),
					}},
				},
			},
		},
		want:          "dry run failed with kubeclient error: spec.template",
		modifyContext: failKubeCalls,
	}, {
		name: "dryrun not supported succeeds",
		s: &v1.Service{
			ObjectMeta: om,
			Spec: v1.ServiceSpec{
				ConfigurationSpec: goodConfigSpec,
				RouteSpec: v1.RouteSpec{
					Traffic: []v1.TrafficTarget{{
						LatestRevision: ptr.Bool(true),
						Percent:        ptr.Int64(100),
					}},
				},
			},
		},
		modifyContext: dryRunNotSupported,
	}, {
		name: "dryrun strict mode",
		s: &v1.Service{
			ObjectMeta: metav1.ObjectMeta{
				Name:      "valid",
				Namespace: "foo",
				Annotations: map[string]string{
					"features.knative.dev/podspec-dryrun": "strict",
				},
			},
			Spec: v1.ServiceSpec{
				ConfigurationSpec: goodConfigSpec,
				RouteSpec: v1.RouteSpec{
					Traffic: []v1.TrafficTarget{{
						LatestRevision: ptr.Bool(true),
						Percent:        ptr.Int64(100),
					}},
				},
			},
		},
		modifyContext: dryRunNotSupported,
		want:          "dry run failed with fakekube does not support dry run: spec.template",
	}, {
		name: "no template found",
		s: &v1.Service{
			ObjectMeta: om,
			Spec: v1.ServiceSpec{
				ConfigurationSpec: v1.ConfigurationSpec{}, // Empty spec
				RouteSpec: v1.RouteSpec{
					Traffic: []v1.TrafficTarget{{
						LatestRevision: ptr.Bool(true),
						Percent:        ptr.Int64(100),
					}},
				},
			},
		},
		modifyContext: nil,
	}}

	for _, test := range tests {
		t.Run(test.name, func(t *testing.T) {
			ctx, _ := fakekubeclient.With(context.Background())
			if test.modifyContext != nil {
				test.modifyContext(ctx)
			}
			logger := logtesting.TestLogger(t)
			ctx = logging.WithLogger(apis.WithDryRun(ctx), logger)

			unstruct := &unstructured.Unstructured{}
			content, _ := runtime.DefaultUnstructuredConverter.ToUnstructured(test.s)
			unstruct.SetUnstructuredContent(content)

			got := ValidateRevisionTemplate(ctx, unstruct)
			if got == nil {
				if test.want != "" {
					t.Errorf("Validate got='%v', want='%v'", got, test.want)
				}
			} else if test.want != got.Error() {
				t.Errorf("Validate got='%v', want='%v'", got.Error(), test.want)
			}
		})

	}
}

func failKubeCalls(ctx context.Context) {
	client := fakekubeclient.Get(ctx)
	client.PrependReactor("*", "*",
		func(action ktesting.Action) (bool, runtime.Object, error) {
			return true, nil, errors.New("kubeclient error")
		},
	)
}

func dryRunNotSupported(ctx context.Context) {
	client := fakekubeclient.Get(ctx)
	client.PrependReactor("*", "*",
		func(action ktesting.Action) (bool, runtime.Object, error) {
			return true, nil, errors.New("fakekube does not support dry run")
		},
	)
}
