/*
Copyright 2019 The Knative Authors

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package v1alpha1

import (
	"context"
	"testing"

	"github.com/google/go-cmp/cmp"
	corev1 "k8s.io/api/core/v1"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"

	"knative.dev/pkg/apis"
	duckv1 "knative.dev/pkg/apis/duck/v1"
	"knative.dev/pkg/ptr"
	v1 "knative.dev/serving/pkg/apis/serving/v1"
	"knative.dev/serving/pkg/apis/serving/v1beta1"
)

func TestRevisionConversionBadType(t *testing.T) {
	good, bad := &Revision{}, &Service{}

	if err := good.ConvertTo(context.Background(), bad); err == nil {
		t.Errorf("ConvertTo() = %#v, wanted error", bad)
	}

	if err := good.ConvertFrom(context.Background(), bad); err == nil {
		t.Errorf("ConvertFrom() = %#v, wanted error", good)
	}
}

func TestRevisionConversion(t *testing.T) {
	tests := []struct {
		name     string
		in       *Revision
		badField string
	}{{
		name: "good roundtrip w/ lots of parts",
		in: &Revision{
			ObjectMeta: metav1.ObjectMeta{
				Name:       "asdf",
				Namespace:  "blah",
				Generation: 1,
			},
			Spec: RevisionSpec{
				RevisionSpec: v1.RevisionSpec{
					PodSpec: corev1.PodSpec{
						ServiceAccountName: "robocop",
						Containers: []corev1.Container{{
							Image: "busybox",
							VolumeMounts: []corev1.VolumeMount{{
								MountPath: "/mount/path",
								Name:      "the-name",
								ReadOnly:  true,
							}},
						}},
						Volumes: []corev1.Volume{{
							Name: "the-name",
							VolumeSource: corev1.VolumeSource{
								Secret: &corev1.SecretVolumeSource{
									SecretName: "foo",
								},
							},
						}},
					},
					TimeoutSeconds:       ptr.Int64(18),
					ContainerConcurrency: ptr.Int64(53),
				},
			},
			Status: RevisionStatus{
				Status: duckv1.Status{
					ObservedGeneration: 1,
					Conditions: duckv1.Conditions{{
						Type:   "Ready",
						Status: "True",
					}},
				},
				ServiceName: "foo-bar",
				LogURL:      "http://logger.io",
			},
		},
	}, {
		name:     "bad roundtrip w/ build ref",
		badField: "buildRef",
		in: &Revision{
			ObjectMeta: metav1.ObjectMeta{
				Name:       "asdf",
				Namespace:  "blah",
				Generation: 1,
			},
			Spec: RevisionSpec{
				DeprecatedBuildRef: &corev1.ObjectReference{
					APIVersion: "build.knative.dev/v1alpha1",
					Kind:       "Build",
					Name:       "foo",
				},
				RevisionSpec: v1.RevisionSpec{
					PodSpec: corev1.PodSpec{
						ServiceAccountName: "robocop",
						Containers: []corev1.Container{{
							Image: "busybox",
							VolumeMounts: []corev1.VolumeMount{{
								MountPath: "/mount/path",
								Name:      "the-name",
								ReadOnly:  true,
							}},
						}},
						Volumes: []corev1.Volume{{
							Name: "the-name",
							VolumeSource: corev1.VolumeSource{
								Secret: &corev1.SecretVolumeSource{
									SecretName: "foo",
								},
							},
						}},
					},
					TimeoutSeconds:       ptr.Int64(18),
					ContainerConcurrency: ptr.Int64(53),
				},
			},
			Status: RevisionStatus{
				Status: duckv1.Status{
					ObservedGeneration: 1,
					Conditions: duckv1.Conditions{{
						Type:   "Ready",
						Status: "True",
					}},
				},
			},
		},
	}}

	toDeprecated := func(in *Revision) *Revision {
		out := in.DeepCopy()
		out.Spec.DeprecatedContainer = &out.Spec.Containers[0]
		out.Spec.Containers = nil
		return out
	}

	for _, test := range tests {
		t.Run(test.name, func(t *testing.T) {
			beta := &v1beta1.Revision{}
			if err := test.in.ConvertTo(context.Background(), beta); err != nil {
				if test.badField != "" {
					cce, ok := err.(*CannotConvertError)
					if ok && cce.Field == test.badField {
						return
					}
				}
				t.Errorf("ConvertTo() = %v", err)
			} else if test.badField != "" {
				t.Errorf("CovnertUp() = %#v, wanted bad field %q", beta,
					test.badField)
				return
			}
			got := &Revision{}
			if err := got.ConvertFrom(context.Background(), beta); err != nil {
				t.Errorf("ConvertFrom() = %v", err)
			}
			if diff := cmp.Diff(test.in, got); diff != "" {
				t.Errorf("Roundtrip (-want, +got): \n%s", diff)
			}
		})

		// A variant of the test that uses `container:`,
		// but end up with what we have above anyways.
		t.Run(test.name+" (deprecated)", func(t *testing.T) {
			start := toDeprecated(test.in)
			beta := &v1beta1.Revision{}
			if err := start.ConvertTo(context.Background(), beta); err != nil {
				if test.badField != "" {
					cce, ok := err.(*CannotConvertError)
					if ok && cce.Field == test.badField {
						return
					}
				}
				t.Errorf("ConvertTo() = %v", err)
			} else if test.badField != "" {
				t.Errorf("CovnertUp() = %#v, wanted bad field %q", beta,
					test.badField)
				return
			}
			got := &Revision{}
			if err := got.ConvertFrom(context.Background(), beta); err != nil {
				t.Errorf("ConvertFrom() = %v", err)
			}
			if diff := cmp.Diff(test.in, got); diff != "" {
				t.Errorf("Roundtrip (-want, +got): \n%s", diff)
			}
		})
	}
}

func TestRevisionConversionForMultiContainer(t *testing.T) {
	input := &Revision{
		ObjectMeta: metav1.ObjectMeta{
			Name: "multi-container",
		},
		Spec: RevisionSpec{
			RevisionSpec: v1.RevisionSpec{
				PodSpec: corev1.PodSpec{
					ServiceAccountName: "robocop",
					Containers: []corev1.Container{{
						Image: "busybox",
					}, {
						Image: "helloworld",
					}},
				},
				TimeoutSeconds:       ptr.Int64(18),
				ContainerConcurrency: ptr.Int64(53),
			},
		},
		Status: RevisionStatus{
			Status: duckv1.Status{
				ObservedGeneration: 1,
				Conditions: duckv1.Conditions{{
					Type:   "Ready",
					Status: "True",
				}},
			},
			ServiceName: "foo-bar",
			LogURL:      "http://logger.io",
		},
	}
	beta := &v1beta1.Revision{}
	if err := input.ConvertTo(context.Background(), beta); err != nil {
		t.Errorf("ConvertTo() = %v", err)
	}
	got := &Revision{}
	if err := got.ConvertFrom(context.Background(), beta); err != nil {
		t.Errorf("ConvertFrom() = %v", err)
	}
	if diff := cmp.Diff(input, got); diff != "" {
		t.Errorf("Roundtrip (-want, +got): \n%s", diff)
	}
}

func TestRevisionConversionError(t *testing.T) {
	tests := []struct {
		name string
		in   *Revision
		want *apis.FieldError
	}{{
		name: "multiple containers in podspec",
		in: &Revision{
			ObjectMeta: metav1.ObjectMeta{
				Name:       "asdf",
				Namespace:  "blah",
				Generation: 1,
			},
			Spec: RevisionSpec{
				RevisionSpec: v1.RevisionSpec{
					PodSpec: corev1.PodSpec{
						ServiceAccountName: "robocop",
						Containers: []corev1.Container{{
							Image: "busybox",
						}, {
							Image: "helloworld",
						}},
					},
					TimeoutSeconds:       ptr.Int64(18),
					ContainerConcurrency: ptr.Int64(53),
				},
				DeprecatedContainer: &corev1.Container{
					Image: "busybox",
				},
			},
			Status: RevisionStatus{
				Status: duckv1.Status{
					ObservedGeneration: 1,
					Conditions: duckv1.Conditions{{
						Type:   "Ready",
						Status: "True",
					}},
				},
				ServiceName: "foo-bar",
				LogURL:      "http://logger.io",
			},
		},
		want: apis.ErrMultipleOneOf("container", "containers"),
	}, {
		name: "no containers in podspec",
		in: &Revision{
			ObjectMeta: metav1.ObjectMeta{
				Name:       "asdf",
				Namespace:  "blah",
				Generation: 1,
			},
			Spec: RevisionSpec{
				RevisionSpec: v1.RevisionSpec{
					PodSpec: corev1.PodSpec{
						ServiceAccountName: "robocop",
						Containers:         []corev1.Container{},
					},
					TimeoutSeconds:       ptr.Int64(18),
					ContainerConcurrency: ptr.Int64(53),
				},
			},
			Status: RevisionStatus{
				Status: duckv1.Status{
					ObservedGeneration: 1,
					Conditions: duckv1.Conditions{{
						Type:   "Ready",
						Status: "True",
					}},
				},
				ServiceName: "foo-bar",
				LogURL:      "http://logger.io",
			},
		},
		want: apis.ErrMissingOneOf("container", "containers"),
	}}

	for _, test := range tests {
		t.Run(test.name, func(t *testing.T) {
			beta := &v1beta1.Revision{}
			got := test.in.ConvertTo(context.Background(), beta)
			if got == nil {
				t.Errorf("ConvertTo() = %#v, wanted %v", beta, test.want)
			}
			if diff := cmp.Diff(test.want.Error(), got.Error()); diff != "" {
				t.Errorf("Roundtrip (-want, +got): \n%s", diff)
			}
		})
	}
}
