package main

import (
	"context"
	"flag"
	"fmt"
	"os"
	"runtime"

	// Import all Kubernetes client auth plugins (e.g. Azure, GCP, OIDC, etc.)
	"github.com/operator-framework/operator-sdk/pkg/k8sutil"
	kubemetrics "github.com/operator-framework/operator-sdk/pkg/kube-metrics"
	"github.com/operator-framework/operator-sdk/pkg/leader"
	"github.com/operator-framework/operator-sdk/pkg/log/zap"
	"github.com/operator-framework/operator-sdk/pkg/metrics"
	sdkVersion "github.com/operator-framework/operator-sdk/version"
	"github.com/spf13/pflag"
	v1 "k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/util/intstr"
	_ "k8s.io/client-go/plugin/pkg/client/auth"
	"k8s.io/client-go/rest"
	"sigs.k8s.io/controller-runtime/pkg/client/config"
	"sigs.k8s.io/controller-runtime/pkg/manager"

	"sigs.k8s.io/controller-runtime/pkg/runtime/signals"

	"github.com/redhat-developer/service-binding-operator/pkg/apis"
	"github.com/redhat-developer/service-binding-operator/pkg/controller"
	"github.com/redhat-developer/service-binding-operator/pkg/log"
)

// Change below variables to serve metrics on different host or port.
var (
	metricsHost               = "0.0.0.0"
	metricsPort         int32 = 8383
	operatorMetricsPort int32 = 8686
	mainLog                   = log.NewLog("main")
)

func printVersion() {
	mainLog.Info(fmt.Sprintf("Go Version: %s", runtime.Version()))
	mainLog.Info(fmt.Sprintf("Go OS/Arch: %s/%s", runtime.GOOS, runtime.GOARCH))
	mainLog.Info(fmt.Sprintf("Version of operator-sdk: %v", sdkVersion.Version))
}

// getOperatorName based on environment variable OPERATOR_NAME, or returns the default name for
// the operatator.
func getOperatorName() string {
	envName := os.Getenv("OPERATOR_NAME")
	if envName != "" {
		return envName
	}
	return "service-binding-operator"
}

// isLeaderElectionEnabled based on environment variable SERVICE_BINDING_OPERATOR_DISABLE_ELECTION. By default, it is enabled.
func isLeaderElectionEnabled() bool {
	return os.Getenv("SERVICE_BINDING_OPERATOR_DISABLE_ELECTION") == ""
}

// isLeaderWithLeaseEnabled based on environment variable SERVICE_BINDING_OPERATOR_LEADER_ELECTION_OPTION. By default, it is leader-for-life.
func isLeaderWithLeaseEnabled() bool {
	return os.Getenv("SERVICE_BINDING_OPERATOR_LEADER_ELECTION_OPTION") == "leader-with-lease"
}

func main() {
	pflag.CommandLine.AddFlagSet(zap.FlagSet())
	pflag.CommandLine.AddGoFlagSet(flag.CommandLine)
	pflag.Parse()
	log.SetLog(zap.Logger())

	printVersion()

	namespace, err := k8sutil.GetWatchNamespace()
	if err != nil {
		mainLog.Error(err, "Failed to get watch namespace")
		os.Exit(1)
	}

	// Get a config to talk to the apiserver
	cfg, err := config.GetConfig()
	if err != nil {
		mainLog.Error(err, "Failed to acquire a configuration to talk to the API server")
		os.Exit(1)
	}

	ctx := context.TODO()

	opts := manager.Options{
		Namespace:          namespace,
		MetricsBindAddress: fmt.Sprintf("%s:%d", metricsHost, metricsPort),
	}

	// FIXME: is there a way to tell k8s-client that is not running in-cluster?
	if isLeaderElectionEnabled() {
		if !isLeaderWithLeaseEnabled() {
			// Become the leader before proceeding
			err = leader.Become(ctx, fmt.Sprintf("%s-lock", getOperatorName()))
			if err != nil {
				mainLog.Error(err, "Failed to become the leader")
				os.Exit(1)
			}
		} else {
			leaderNS := os.Getenv("SERVICE_BINDING_OPERATOR_LEADER_ELECTION_NAMESPACE")
			opts = manager.Options{
				Namespace:               namespace,
				MetricsBindAddress:      fmt.Sprintf("%s:%d", metricsHost, metricsPort),
				LeaderElection:          true,
				LeaderElectionID:        getOperatorName(),
				LeaderElectionNamespace: leaderNS,
			}
		}
	} else {
		mainLog.Warning("Leader election is disabled")
	}

	// Create a new Cmd to provide shared dependencies and start components
	mgr, err := manager.New(cfg, opts)
	if err != nil {
		mainLog.Error(err, "Error on creating a new manager instance")
		os.Exit(1)
	}

	mainLog.Info("Registering Components.")

	// Setup Scheme for all resources
	if err := apis.AddToScheme(mgr.GetScheme()); err != nil {
		mainLog.Error(err, "Error adding local operator scheme")
		os.Exit(1)
	}

	// Setup all Controllers
	if err := controller.AddToManager(mgr); err != nil {
		mainLog.Error(err, "Failed to setup the controller manager")
		os.Exit(1)
	}

	if err = serveCRMetrics(cfg); err != nil {
		mainLog.Info("Could not generate and serve custom resource metrics", "error", err.Error())
	}

	// Add to the below struct any other metrics ports you want to expose.
	servicePorts := []v1.ServicePort{
		{Port: metricsPort, Name: metrics.OperatorPortName, Protocol: v1.ProtocolTCP, TargetPort: intstr.IntOrString{Type: intstr.Int, IntVal: metricsPort}},
		{Port: operatorMetricsPort, Name: metrics.CRPortName, Protocol: v1.ProtocolTCP, TargetPort: intstr.IntOrString{Type: intstr.Int, IntVal: operatorMetricsPort}},
	}
	// Create Service object to expose the metrics port(s).
	service, err := metrics.CreateMetricsService(ctx, cfg, servicePorts)
	if err != nil {
		mainLog.Info("Could not create metrics Service", "error", err.Error())
	}

	// CreateServiceMonitors will automatically create the prometheus-operator ServiceMonitor resources
	// necessary to configure Prometheus to scrape metrics from this operator.
	services := []*v1.Service{service}
	_, err = metrics.CreateServiceMonitors(cfg, namespace, services)
	if err != nil {
		mainLog.Info("Could not create ServiceMonitor object", "error", err.Error())
		// If this operator is deployed to a cluster without the prometheus-operator running, it will return
		// ErrServiceMonitorNotPresent, which can be used to safely skip ServiceMonitor creation.
		if err == metrics.ErrServiceMonitorNotPresent {
			mainLog.Info("Install prometheus-operator in your cluster to create ServiceMonitor objects", "error", err.Error())
		}
	}

	mainLog.Info("Starting the Cmd.")

	// Start the Cmd
	if err := mgr.Start(signals.SetupSignalHandler()); err != nil {
		mainLog.Error(err, "Manager exited non-zero")
		os.Exit(1)
	}
}

// serveCRMetrics gets the Operator/CustomResource GVKs and generates metrics based on those types.
// It serves those metrics on "http://metricsHost:operatorMetricsPort".
func serveCRMetrics(cfg *rest.Config) error {
	// Below function returns filtered operator/CustomResource specific GVKs.
	// For more control override the below GVK list with your own custom logic.
	filteredGVK, err := k8sutil.GetGVKsFromAddToScheme(apis.AddToScheme)
	if err != nil {
		return err
	}
	// Get the namespace the operator is currently deployed in.
	operatorNs, err := k8sutil.GetOperatorNamespace()
	if err != nil {
		return err
	}
	// To generate metrics in other namespaces, add the values below.
	ns := []string{operatorNs}
	// Generate and serve custom resource specific metrics.
	err = kubemetrics.GenerateAndServeCRMetrics(cfg, ns, filteredGVK, metricsHost, operatorMetricsPort)
	if err != nil {
		return err
	}
	return nil
}
