// Copyright ©2017 The Gonum Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package mat_test

import (
	"fmt"
	"log"

	"gonum.org/v1/gonum/mat"
)

func ExampleHOGSVD() {
	// Perform an HOGSVD factorization on food production/consumption data for the
	// three years 1990, 2000 and 2014.
	//
	// See Ponnapalli et al. doi:10.1371/journal.pone.0028072 and
	// Alter at al. doi:10.1073/pnas.0530258100 for more details.
	var gsvd mat.HOGSVD
	ok := gsvd.Factorize(FAO.Africa, FAO.Asia, FAO.LatinAmericaCaribbean, FAO.Oceania)
	if !ok {
		log.Fatalf("HOGSVD factorization failed: %v", gsvd.Err())
	}

	for i, n := range []string{"Africa", "Asia", "Latin America/Caribbean", "Oceania"} {
		var u mat.Dense
		gsvd.UTo(&u, i)
		s := gsvd.Values(nil, i)
		fmt.Printf("%s\n\ts_%d = %.4f\n\n\tU_%[2]d = %.4[4]f\n",
			n, i, s, mat.Formatted(&u, mat.Prefix("\t      ")))
	}

	var v mat.Dense
	gsvd.VTo(&v)
	fmt.Printf("\nCommon basis vectors\n\n\tVᵀ =  %.4f",
		mat.Formatted(v.T(), mat.Prefix("\t      ")))

	// Output:
	//
	// Africa
	// 	s_0 = [45507.3278 18541.9293 21503.0778]
	//
	// 	U_0 = ⎡-0.0005  -0.0039  -0.0019⎤
	// 	      ⎢-0.0010  -0.0007  -0.0012⎥
	// 	      ⎢-1.0000  -0.0507  -0.9964⎥
	// 	      ⎢-0.0022  -0.2906  -0.0415⎥
	// 	      ⎢ 0.0001  -0.0127  -0.0016⎥
	// 	      ⎢ 0.0003  -0.0067  -0.0010⎥
	// 	      ⎢ 0.0003  -0.0022  -0.0003⎥
	// 	      ⎢-0.0086  -0.9550   0.0734⎥
	// 	      ⎢ 0.0017   0.0002   0.0059⎥
	// 	      ⎢-0.0002  -0.0088  -0.0014⎥
	// 	      ⎢-0.0006  -0.0078  -0.0001⎥
	// 	      ⎢-0.0005  -0.0076   0.0003⎥
	// 	      ⎢ 0.0001  -0.0090   0.0008⎥
	// 	      ⎢-0.0005  -0.0050   0.0029⎥
	// 	      ⎢-0.0011  -0.0078  -0.0012⎥
	// 	      ⎢-0.0014  -0.0058  -0.0002⎥
	// 	      ⎢ 0.0007  -0.0095   0.0020⎥
	// 	      ⎢-0.0008  -0.0081  -0.0009⎥
	// 	      ⎢ 0.0004  -0.0092   0.0006⎥
	// 	      ⎢-0.0007  -0.0079  -0.0006⎥
	// 	      ⎣-0.0011  -0.0076  -0.0010⎦
	// Asia
	// 	s_1 = [77228.2804 8413.7024 14711.1879]
	//
	// 	U_1 = ⎡ 0.0005  -0.0080   0.0011⎤
	// 	      ⎢ 0.0008  -0.0108   0.0016⎥
	// 	      ⎢-0.9998   0.0612   0.9949⎥
	// 	      ⎢ 0.0007  -0.5734  -0.0468⎥
	// 	      ⎢ 0.0001  -0.0265  -0.0022⎥
	// 	      ⎢ 0.0001  -0.0165  -0.0019⎥
	// 	      ⎢ 0.0000  -0.0070  -0.0013⎥
	// 	      ⎢ 0.0196  -0.8148   0.0893⎥
	// 	      ⎢ 0.0002  -0.0063   0.0012⎥
	// 	      ⎢-0.0001  -0.0135  -0.0013⎥
	// 	      ⎢-0.0004  -0.0135   0.0019⎥
	// 	      ⎢-0.0005  -0.0132   0.0014⎥
	// 	      ⎢ 0.0003  -0.0155   0.0045⎥
	// 	      ⎢-0.0003  -0.0130   0.0025⎥
	// 	      ⎢-0.0007  -0.0105   0.0016⎥
	// 	      ⎢-0.0006  -0.0129   0.0007⎥
	// 	      ⎢-0.0006  -0.0178  -0.0023⎥
	// 	      ⎢-0.0003  -0.0149   0.0016⎥
	// 	      ⎢-0.0001  -0.0134   0.0030⎥
	// 	      ⎢-0.0004  -0.0154   0.0010⎥
	// 	      ⎣-0.0009  -0.0147  -0.0019⎦
	// Latin America/Caribbean
	// 	s_2 = [274.1364 20736.3116 729.6947]
	//
	// 	U_2 = ⎡ 0.1060  -0.0021   0.0174⎤
	// 	      ⎢ 0.1415  -0.0016   0.0289⎥
	// 	      ⎢ 0.2350  -0.2669  -0.9212⎥
	// 	      ⎢ 0.0290  -0.0118  -0.0429⎥
	// 	      ⎢ 0.0226  -0.0043  -0.0213⎥
	// 	      ⎢ 0.0117  -0.0016  -0.0197⎥
	// 	      ⎢-0.6263  -0.9635   0.2234⎥
	// 	      ⎢ 0.2334  -0.0013   0.1275⎥
	// 	      ⎢-0.0358  -0.0085  -0.0498⎥
	// 	      ⎢-0.1238  -0.0054   0.0313⎥
	// 	      ⎢-0.0421  -0.0059   0.0528⎥
	// 	      ⎢-0.1471  -0.0056   0.0350⎥
	// 	      ⎢-0.2158  -0.0052  -0.0044⎥
	// 	      ⎣-0.6154  -0.0078  -0.2717⎦
	// Oceania
	// 	s_3 = [8954.1914 6942.6316 17233.0561]
	//
	// 	U_3 = ⎡-0.0080  -0.0012  -0.0040⎤
	// 	      ⎢ 0.0004  -0.0014   0.0001⎥
	// 	      ⎢ 0.9973  -0.0315   0.9991⎥
	// 	      ⎢ 0.0473  -0.7426  -0.0359⎥
	// 	      ⎢ 0.0018  -0.0342  -0.0020⎥
	// 	      ⎢-0.0005  -0.0148  -0.0016⎥
	// 	      ⎢-0.0004  -0.0047  -0.0007⎥
	// 	      ⎢-0.0246  -0.6642  -0.0138⎥
	// 	      ⎢ 0.0003  -0.0287  -0.0023⎥
	// 	      ⎢-0.0011  -0.0148  -0.0014⎥
	// 	      ⎢-0.0108  -0.0198  -0.0039⎥
	// 	      ⎢-0.0149  -0.0183  -0.0048⎥
	// 	      ⎢-0.0178  -0.0208  -0.0075⎥
	// 	      ⎢-0.0266  -0.0063  -0.0016⎥
	// 	      ⎢-0.0012  -0.0234  -0.0006⎥
	// 	      ⎢-0.0084  -0.0184  -0.0030⎥
	// 	      ⎢-0.0232  -0.0191  -0.0124⎥
	// 	      ⎢-0.0072  -0.0226  -0.0035⎥
	// 	      ⎢-0.0150  -0.0144  -0.0045⎥
	// 	      ⎢-0.0068  -0.0227  -0.0034⎥
	// 	      ⎣-0.0127  -0.0136  -0.0049⎦
	//
	// Common basis vectors
	//
	// 	Vᵀ =  ⎡-0.0897  -0.4460  -0.8905⎤
	// 	      ⎢-0.4911  -0.5432  -0.6810⎥
	// 	      ⎣ 0.0644   0.2841   0.9566⎦
}
