package main

import (
	"bytes"
	"flag"
	"fmt"
	"go/ast"
	"go/printer"
	"go/token"
	"io"
	"io/ioutil"
	"os"
	"regexp"

	"github.com/itchyny/astgen-go"
	"github.com/itchyny/gojq"
)

const fileFormat = `// Code generated by _tools/gen_builtin.go; DO NOT EDIT.

package gojq

func init() {%s}
`

func main() {
	var input, output string
	flag.StringVar(&input, "i", "", "input file")
	flag.StringVar(&output, "o", "", "output file")
	flag.Parse()
	if err := run(input, output); err != nil {
		fmt.Fprint(os.Stderr, err)
		os.Exit(1)
	}
}

func run(input, output string) error {
	cnt, err := ioutil.ReadFile(input)
	if err != nil {
		return err
	}
	qs := make(map[string][]*gojq.FuncDef)
	q, err := gojq.Parse(string(cnt))
	if err != nil {
		return err
	}
	for _, fd := range q.FuncDefs {
		name := fd.Name
		if name[0] == '_' {
			name = name[1:]
		}
		fd.Minify()
		qs[name] = append(qs[fd.Name], fd)
	}
	t, err := astgen.Build(qs)
	if err != nil {
		return err
	}
	var buf bytes.Buffer
	buf.Write([]byte("\n\tbuiltinFuncDefs = "))
	if err := printCompositeLit(&buf, t.(*ast.CompositeLit)); err != nil {
		return err
	}
	out := os.Stdout
	if output != "" {
		f, err := os.Create(output)
		if err != nil {
			return err
		}
		defer f.Close()
		out = f
	}
	_, err = fmt.Fprintf(out, fileFormat, buf.String())
	return err
}

func printCompositeLit(out io.Writer, t *ast.CompositeLit) error {
	err := printer.Fprint(out, token.NewFileSet(), t.Type)
	if err != nil {
		return err
	}
	out.Write([]byte("{"))
	for _, kv := range t.Elts {
		out.Write([]byte("\n\t\t"))
		var kvBuf bytes.Buffer
		err = printer.Fprint(&kvBuf, token.NewFileSet(), kv)
		if err != nil {
			return err
		}
		str := kvBuf.String()
		for op := gojq.OpPipe; op <= gojq.OpUpdateAlt; op++ {
			r := regexp.MustCompile(fmt.Sprintf(`\b((?:Update)?Op): %d\b`, op))
			str = r.ReplaceAllString(str, fmt.Sprintf("$1: %#v", op))
		}
		for termType := gojq.TermTypeIdentity; termType <= gojq.TermTypeQuery; termType++ {
			r := regexp.MustCompile(fmt.Sprintf(`(Term{Type): %d\b`, termType))
			str = r.ReplaceAllString(str, fmt.Sprintf("$1: %#v", termType))
		}
		out.Write([]byte(str))
		out.Write([]byte(","))
	}
	out.Write([]byte("\n\t}\n"))
	return nil
}
