/*
 * Spawns and respawns a child process when it fails.  This can be used
 * to maintain execution for a critical process until it completes
 * normally.  Different switch options are included to change the behavior
 * of spawn.  The -d (daemon) option forces respawn to 'detach' (much
 * like how 'detach' works) and then spawn.  The -x switch means to exit
 * respawn if the child returns an exit status of '0'.  The -r option
 * can specify a maximum "repeat" count.  A special '-w' option to
 * specify minimum 'wait' time between failures can be specified.
 *
 * $Id: respawn.c 1.1 Wed, 04 Jun 1997 15:38:01 -0400 dyfet $
 * Copyright (c) 1997 by Tycho Softworks.
 * For conditions of distribution and reuse see product license.
 */

#include <std/time.h>
#include <proc/process.h>
#include <std/files.h>

void main(int argc, char **argv)
{
	int	opt;
	bool	deflag = FALSE;
	bool	exflag = FALSE;
	int	repeat = 0;
	int	wait = 0;
	time_t	now, after;

	while(EOF != (opt = getopt(argc, argv, "dxr:w:n:?h")))
		switch(opt)
		{
		case 'd':
			deflag = TRUE;
			break;
		case 'x':
			exflag = TRUE;
			break;
		case 'r':
			repeat = atoi(optarg);
			break;
		case 'w':
			wait = atoi(optarg);
			break;
		case 'n':
			nice(atoi(optarg));
			break;
		default:
			fatal(EX_USAGE, "use: respawn [options] command...\n");
		}

	if(optind >= argc)
		fatal(EX_USAGE, "use: respawn [options] command...\n");

	argv = &argv[optind];

	if(deflag)
	{
		pdetach(D_KEEPNONE);
		open("/dev/null", O_RDWR);
		dup(0);
		dup(0);
	}
	for(;;)
	{
		time(&now);
		opt = spawnvp(P_WAIT, *argv, argv);
		if(exflag)
			if(!opt)
				exit(0);

		if(repeat)
			if(!--repeat)
				break;

		time(&after);
		if(wait)
		{
			opt = wait - (after - now);
			if(opt > 0)
				sleep(opt);
		}
	}
	exit(exflag);			
}

