/*
 * Redirects standard input to a (tcp) network service.  This can be used
 * to redirect output for a cat file to a 'speak' service, for example.
 * Redirect may also be used in conjunction with the 'bind' utility,
 * which binds a 'socket' and executes commands as connections are made.
 * Redirect and bind can be used to transport a device stream accross a 
 * set of network hosts.
 *
 * Redirect outputs any incomming data to standard out as well as
 * stdin to the redirect target.  This is best exemplified by performing
 * a 'redirect localhost:echo'.  In a way, redirect can be thought of as
 * a 'pipe fitting' or as a means to 'pipe' through a network connection.
 *
 * Redirect may be used with devices or other named 'filesystem' resources
 * as well as with sockets.
 *
 * $Id: redirect.c 1.1 Wed, 04 Jun 1997 15:38:01 -0400 dyfet $
 * Copyright (c) 1997 by Tycho Softworks.
 * For conditions of distribution and reuse see product license.
 */

#include <proc/process.h>
#include <other/files.h>
#include <other/string.h>
#include <net/socket.h>
#include <net/select.h>

void main(int argc, char **argv)
{
	char	*p;
	int	port;
	fd_t	fd;
	char	buf[PIPE_BUF];
	int	len;
	struct	stat	ino;

#ifdef	_SELECT_BSD
	fd_set	inp, out, err;
#endif

	if(argc != 2)
		fatal(EX_USAGE, "use: redirect host:service\n"
                                "  or redirect /dev/device\n");

	++argv;
	p = strchr(*argv, ':');
	if(p)
	{
		*(p++) = 0;
		if(isdigit(*p))
			port = atoi(p);
		else
			port = getservice(p);

		if(!port)
			fatal(EX_UNAVAILABLE, "redirect: %s: service unknown\n", p);

		fd = getsocket(*argv, port, SOCK_STREAM);
		if(fd == INVALID_SOCKET)
			fatal(EX_UNAVAILABLE, "redirect: %s: service unreachable\n", p);
	}
	else
	{
		if(stat(*argv, &ino))
			fatal(EX_OSFILE, "redirect: %s: cannot access\n", *argv);
		
		if(!S_ISCHR(ino.st_mode))
			fatal(EX_OSFILE, "redirect: %s: not a char device\n", *argv);

		fd = open(*argv, O_RDWR);
		if(fd < 0)
			fatal(EX_OSFILE, "redirect: %s: cannot open\n", *argv);
	}

#ifdef	_SELECT_BSD
	for(;;)
	{
		FD_ZERO(&inp);
		FD_ZERO(&out);
		FD_ZERO(&err);
		FD_SET(0, &inp);
		FD_SET(0, &err);
		FD_SET(fd, &inp);
		FD_SET(fd, &err);
		
		select(fd + 1, &inp, &out, &err, NULL);
		if(FD_ISSET(0, &err))
		{
			endsocket(fd);
			exit(0);
		}
	
		if(FD_ISSET(fd, &err))
		{
			endsocket(fd);
			exit(-1);
		}
		
		if(FD_ISSET(fd, &inp))
		{
			len = read(fd, buf, PIPE_BUF);
			if(len > 0)
				write(1, buf, len);
		}
		
		if(FD_ISSET(0, &inp))
		{
			len = read(0, buf, PIPE_BUF);
			if(len > 0)
				write(fd, buf, len);
		}
	}
#endif		
}

