/* sfm -- Simple File Manager
   Copyright (C) 1998 Pixel (Pascal Rigaux)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */
#include "filetype.h"
#include "io.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/wait.h>


#define FILETYPE2PROG "~/.sfm"
#define SEPERATEUR_FILETYPE "==>"
#define SEPERATEUR_EXTENSION "-->"
#define FILEINSIDE "%s"
#define MIN_FILETYPE_LENGTH 5

typedef enum { FileTypes, Extension, SkipExtension, ApplyToAll, Last_4filetypes } forfiletypes;

static int size_fileTypesToProgs[Last_4filetypes];
static int maxsize_fileTypesToProgs[Last_4filetypes];
static char **fileTypesToProgs[Last_4filetypes] = { NULL };

static int filetype_pid;


static char **findProgForFileType(char *filetype, int *rnb);
static boolean skip_SkipableExtension(char *file);
static char **findProgUsingExtension(char *file, int *rnb);



static char *mygets(char *p, FILE *f)
{
  p = fgets(p, tmpSize, f);
 
  if (p == NULL) return NULL;
  if (strlen(p) + 2 >= tmpSize) die("line too long in filetype2prog");
  if (lastchar(p) == '\n') chop(p);
  if (p[0] == '#') p[0] = '\0';
  return p;
}

static void addProgForFileType(forfiletypes type, char *filetype, char *prog)
{
  char *_filetype = remove_bording_spaces(filetype);
  char *_prog = remove_bording_spaces(prog);
  int i;
  
  for (i = 0; i < size_fileTypesToProgs[type]; i += 2)
    if (streq(fileTypesToProgs[type][i], _filetype) && streq(fileTypesToProgs[type][i + 1], _prog)) {
      free(_filetype);
      free(_prog);
      return;
    }
  
  if (size_fileTypesToProgs[type] + 2 >= maxsize_fileTypesToProgs[type]) {
    maxsize_fileTypesToProgs[type] *= 2;
    pchar_realloc(&fileTypesToProgs[type], maxsize_fileTypesToProgs[type]);
  }
  fileTypesToProgs[type][size_fileTypesToProgs[type]++] = _filetype;
  fileTypesToProgs[type][size_fileTypesToProgs[type]++] = _prog;
}

extern void initProgForFileType()
{
  FILE *f;
  int i;
  char *initfile, *p, *q, tmp[tmpSize];
  forfiletypes type;

  if (fileTypesToProgs[0]) return;
  
  for (i = 0; i < Last_4filetypes; i++) {
    size_fileTypesToProgs[i] = 0;
    maxsize_fileTypesToProgs[i] = tmpSize;
    fileTypesToProgs[i] = pchar_malloc(maxsize_fileTypesToProgs[i]);
  }

  initfile = expandEnvVars(FILETYPE2PROG);
  f = fopen(initfile, "r");
  if (f == NULL) {
    warn("can't find init file\n");
    return;
  }
  while ((p = mygets(tmp, f))) {

    if ((q = strstr(p, SEPERATEUR_FILETYPE))) type = FileTypes;
    else if ((q = strstr(p, SEPERATEUR_EXTENSION))) {
      if (skip_spaces(q + sizeof(SEPERATEUR_EXTENSION) - 1)[0] == '\0')
	type = SkipExtension;
      else if (p == q)
	type = ApplyToAll;
      else
	type = Extension;
    } else continue;

    *q = '\0';
    addProgForFileType(type, p, q + sizeof(SEPERATEUR_FILETYPE) - 1);
  }
  fclose(f);
  free(initfile);
}

extern void saveProgForFileType()
{
  char *initfile;
  FILE *f;
  int i, j;

  if (fileTypesToProgs[0] == NULL) return;

  initfile = expandEnvVars(FILETYPE2PROG);
  f = fopen(initfile, "w");
  if (f == NULL) {
    warn("can't save init file\n");
    return;
  }

  for (i = 0; i < Last_4filetypes; i++)
    for (j = 0; j < size_fileTypesToProgs[i];) {
      fputs(fileTypesToProgs[i][j++], f); 
      fputs(i == FileTypes ? SEPERATEUR_FILETYPE : SEPERATEUR_EXTENSION, f);
      fputs(fileTypesToProgs[i][j++], f); 
      fputc('\n', f);
    }
  fclose(f);
  free(initfile);
}

static char *extractExtension(char *file)
{
  char *p;

  while (skip_SkipableExtension(file));
  if ((p = (strrchr(file, '/')))) file = p + 1;
  p = strrchr(file, '.');
  if (p == file) return NULL;
  return p;
}

extern void freeProgForFileType()
{
  int i;
  for (i = 0; i < Last_4filetypes; i++) {
    pchar_free(fileTypesToProgs[i], size_fileTypesToProgs[i]);
    FREE_NULL(fileTypesToProgs[i]);
  }
}

extern char **findProgForFileType(char *filetype, int *rnb)
{
  char *prog, *p, *tmp[tmpSize];
  int i, j, nb, best;

  for (prog = NULL, nb = 1, best = i = 0; i < size_fileTypesToProgs[FileTypes]; i += 2) {
    j = common_chars(fileTypesToProgs[FileTypes][i], filetype);
    if (j >= MIN_FILETYPE_LENGTH) {
      p = fileTypesToProgs[FileTypes][i + 1];
      if (j > best) {
	best = j;
	char_swap(&prog, &p);
      }
      if (nb + 1 == tmpSize) die("too many possibilities (of progs) for this file");
      if (p) tmp[nb++] = p;
    }
  }
  if (prog) tmp[0] = prog; else nb = 0;
  if (rnb) *rnb = nb;

  if (prog == NULL) return NULL;  
  return anydup(tmp, nb * sizeof(char *));
}

extern boolean skip_SkipableExtension(char *file)
{
  int i;

  for (i = 0; i < size_fileTypesToProgs[SkipExtension]; i += 2)
    if (strendswith(file, fileTypesToProgs[SkipExtension][i])) {
      strend(file)[- strlen(fileTypesToProgs[SkipExtension][i])] = '\0';
      return true;
    }
  return false;
}

extern char **findProgUsingExtension(char *file, int *rnb)
{
  char *tmp[tmpSize];
  char *p = strdup(file);
  int i, nb = 0;

  do {
    for (i = 0; i < size_fileTypesToProgs[Extension]; i += 2) {
      if (strendswith_nocase(p, fileTypesToProgs[Extension][i])) {
	tmp[nb++] = fileTypesToProgs[Extension][i + 1];
      }
    }
  } while (nb == 0 && skip_SkipableExtension(p));

  free(p);
  if (rnb) *rnb = nb;
  if (nb == 0) return NULL;
  return anydup(tmp, nb * sizeof(char *));
}

extern char *fileType(char *file)
{
  char *args[4], *tmp, *r;
  int i, status, size, maxsize, the_pipe[2];

  i = 0;
  args[i++] = "file";
#ifndef SunOS
  args[i++] = "-L";
#endif
  args[i++] = file;
  args[i++] = NULL;
  if (pipe(the_pipe) == -1) die("pipe");

  if ((filetype_pid = fork()) == 0) {
    dup2(the_pipe[1], STDOUT_FILENO);
    close(the_pipe[1]);

    execvp(args[0], args);
    fprintf(stderr, "executable %s not found\n", args[0]);
    exit(1);
  }
  /*if (*/waitpid(filetype_pid, &status, 0);/* == -1); die("can't get status of my child !");
  if (status) exit(status);*/
  
  size = 0;
  maxsize = tmpSize;
  tmp = char_malloc(maxsize);
  while ((i = read(the_pipe[0], tmp + size, maxsize - size))) {
    size += i;
    if (size == maxsize) {
      maxsize *=2;
      char_realloc(&tmp, maxsize);
    } else break;
  }
  close(the_pipe[0]);

  tmp[size - 1] = '\0';
  r = strdup(tmp + strlen(file) + 2);
  free(tmp);
  /*printf("(%s)\n", r);*/
  return r;
}

extern char *file2prog(char *file)
{
  char *filetype, *p, **r;

  if ((r = findProgUsingExtension(file, NULL))) {
    p = r[0];
    free(r);
    return p;
  }

  filetype = fileType(file);
  if ((r = findProgForFileType(filetype, NULL))) {
    p = r[0];
    free(r);
    return p;
  }

  return NULL;
}

extern char **file2progs(char *file, int *nb)
{
  char *filetype, **r1, **r2, **r;
  int i, n1, n2;

  filetype = fileType(file);
  r1 = findProgUsingExtension(file, &n1);
  r2 = findProgForFileType(filetype, &n2);
  *nb = n1 + n2 + size_fileTypesToProgs[ApplyToAll] / 2;

  if (*nb > 0) {
    r = pchar_malloc(*nb);
    if (n1) memcpy(r, r1, n1 * sizeof(char *));
    if (n2) memcpy(r + n1, r2, n2 * sizeof(char *));
    for (i = 0; i < size_fileTypesToProgs[ApplyToAll] / 2; i++) r[n1 + n2 + i] = fileTypesToProgs[ApplyToAll][2*i + 1];
    free(r1);
    free(r2);

  } else r = NULL;
  
  return r;
}

extern void launch_action(char **files, int nb, char *prog)
{
  char *_prog, *file, *p, *q, *r, *s, *args[tmpSize];
  int i, nbfilesinside, size;

  _prog = strdup(prog);
  for (q = _prog, nbfilesinside = 0; 
       (q = strstr(q, FILEINSIDE)); 
       q += sizeof(FILEINSIDE) - 1, nbfilesinside++);

  if (nbfilesinside > 0) {

    file = pchar2char(files, nb, " ");
    size = strlen(file);
    r = char_malloc(strlen(_prog) + nbfilesinside * (size - sizeof(FILEINSIDE) + 1) + 1);

    for (s = r, p = _prog; (q = strstr(p, FILEINSIDE)); p = q + sizeof(FILEINSIDE) - 1) {
      memcpy(s, p, q - p);
      s += q - p;
      memcpy(s, file, size);
      s += size;
    }
    strcpy(s, p);
    
    args[0] = "sh";
    args[1] = "-c";
    args[2] = r;
    args[3] = NULL;
    free(file);

    exec_it_and_mem(args, "launching");
    free(r);
    free(_prog);

  } else {

    for (p = _prog, i = 0; (q = strchr(p, ' ')); i++) {
      *q = '\0';
      args[i] = p;
      p = q + 1;
    }
    args[i++] = p;
    memcpy(args + i, files, nb * sizeof(char *));
    i += nb;
    args[i] = '\0';

    exec_it_and_mem(args, "launching");
    free(_prog);
  }
}

extern void addFile2prog(char *file, char *prog)
{
  forfiletypes type;
  char *p = NULL;

  if ((p = extractExtension(file))) type = Extension;
  else if ((p = fileType(file))) type = FileTypes;

  if (p) addProgForFileType(type, p, prog);
}

extern boolean isfileTypeSigChld(int pid)
{
  return pid == filetype_pid;
}
