/* linuXtree copyright (c) 1998 Dan Stahlke under the GPL
   See file 'COPYING' for details */

#include "all.h"

int topview;        /* position in file to start viewing */
int botview;        /* position in file of last displayed char */
int filesize;       /* length of file being viewed */
int vieweof;        /* at end of file in viewer? */
int viewmode;       /* current mode for viewing */
int viewmask;       /* mask unprintable chars? */
int viewscrolled;   /* did we just scroll? + is down, - is up */
int viewdmpw;       /* width of dump mode */
int viewhexw;       /* width of hex mode */
FILE *viewfile;     /* fd of file being viewed */
const unsigned char *viewstring;   /* pointer to a memory area being viewed */
const char *viewfilename; /* name of file being viewed */
int viewnode;       /* node of file being viewed */
int viewhelp;       /* in help mode? (view buffer, not file) */
int oldmaxx,oldmaxy;/* this is used to catch resize events */
unsigned char *screenbuffer;  /* the buffer we read the current page into */
unsigned char *screenattrib;  /* the attributes (bold/underline) of above */
int *screenphys;              /* the location of each char in the file */
int *parabuf;       /* offset of start of each line in current paragraph */
int paraline;       /* current line in paragraph */
int parasize;       /* number of lines in paragraph */
int parabufmemsiz;  /* size of parabuf allocation */
int ispipe;         /* are we viewing a pipe? */
int lungs;          /* fd of pipe output */
unsigned char *pipereadbuf;  /* all the info read from the pipe */
int pipebufsiz;     /* size of above buffer */
int pipereadsiz;    /* how many bytes read from pipe so far */
int pipecutshort;   /* were we cut short on the last read? */
char *pipename;     /* name of current pipe */
int issearch;       /* is search mode activated? */
int showsearch;     /* should we hilight search items on the screen? */
char searchexp[REGEX_SIZ]; /* the current search expression */
my_regex_t search_comp;    /* the compiled regular expression */
int num_match;      /* number of matches hilighted (some may be off screen) */
int *search_match;  /* which bytes match */
int match_start,match_len; /* bounds of above buffer */

#define PIPE_BLKSIZ 1024  /* optimal block size for pipes */

extern int updateall;
extern int maxx,maxy;
extern WINDOW *gui_bigun;  /* the big full screen window */
extern WINDOW *gui_msbar;  /* the status line on the bottom */
extern WINDOW *gui_stbar;  /* path and time, top line */
extern struct configtype config;
extern int keep_msbar,dispmode;

int child_exited(int pid);

void view_init(void) {
  viewfile=NULL;
  viewstring=NULL;
  viewmode=VIEW_TEXT;
  viewmask=1;
  viewscrolled=0;
  screenbuffer=NULL;
  screenattrib=NULL;
  screenphys=NULL;
  search_match=NULL;
  num_match=0;
  issearch=0;
  ispipe=0;
  pipereadbuf=NULL;
  parabuf=NULL;
  oldmaxx=oldmaxy=0;
  view_handleresize();
  searchexp[0]=0;
}

void view_handleresize(void) {
  if(config.viewpowtwo) {
    for(viewhexw=4;(viewhexw*2/4)*17+10<maxx;viewhexw<<=1);
    for(viewdmpw=1;viewdmpw*2+11<maxx;viewdmpw<<=1);
  } else {
    viewhexw=(maxx-11)/17*4;
    viewdmpw=maxx-12;
  }

  #define BUFSIZE (maxx*maxy)
  screenbuffer=realloc(screenbuffer,BUFSIZE);
  screenattrib=realloc(screenattrib,BUFSIZE);
  screenphys=realloc(screenphys,BUFSIZE*sizeof(int));
  if(!screenbuffer) outofmem("view_init");
  if(!screenattrib) outofmem("view_init");
  if(!screenphys) outofmem("view_init");

  gui_setwantdrawall();
}

void view_cleanstate(void) {
  topview=0;
  viewscrolled=0;
  if(parabuf) crapout("view_openfile","parabuf is not NULL");
  parasize=0;
  view_undosearch();
}

void view_openfile(int nodenumber) {
  if(ispipe) view_closepipe();
  viewnode=nodenumber;
  viewfilename=data_fullname(nodenumber);
  viewfile=fopen(viewfilename,"r");
  if(viewfile==NULL) {
    gui_mildbeep("Error: cannot open file (%s)",strerror(errno));
    gui_initdispmode(DISP_FILE);
    return;
  }
  viewstring=NULL;
  fseek(viewfile,0,SEEK_END);
  filesize=ftell(viewfile);
  viewhelp=0;
  view_cleanstate();
  if(dispmode!=DISP_VIEW) gui_initdispmode(DISP_VIEW);
}

char bannerbuffer[80];

void view_openhelp(const char *buf) {
  view_savestate();
  sprintf(bannerbuffer,"linuXtree version %s",VERSION);
  viewfilename=bannerbuffer;
  viewfile=NULL;
  viewstring=buf;
  filesize=strlen(buf);
  viewhelp=1;
  viewmode=VIEW_TEXT;
  viewmask=0;
  ispipe=0;
  view_cleanstate();
  gui_initdispmode(DISP_HELP);
}

void view_closefile(void) {
  if(ispipe) view_closepipe();
  else {
    fclose(viewfile);
    free(parabuf);
    parabuf=NULL;
    view_cleanstate();
  }
}

void view_closehelp(void) {
  free(parabuf);
  parabuf=NULL;
  view_resumestate();
}

/* this is called when 't' key is pressed to go back to text from hex/dump
   or when a seek is performed, and it's purpose is to recalculate the
   paragraph buffer */
void view_resetparagraph(void) {
  int pos,buf;

  if(viewmode!=VIEW_TEXT) return;

  free(parabuf);
  parabuf=NULL;

  pos=topview;
  do {
    pos--;
    if(pos<0) break;
    buf=view_readonechar(pos);
  } while(buf!=10);
  pos++;
  view_calcparagraph(pos);
  for(paraline=0;paraline<parasize;paraline++)
    if(parabuf[paraline]>topview) break;
  if(!paraline) crapout("view_resetparagraph","paraline==0");
  paraline--;
  topview=parabuf[paraline];
}

void view_askpipe(void) {
  const char *def;
  char cmdbuf[PATH_MAX+1];

  if(!ispipe) {
    def=config_lookupext(viewfilename,LOOKUP_PIPE);
    if(comm_prompt(cmdbuf,PATH_MAX+1,def,0,"Pipe: ",FM_TABCOMPLETE,PV_CMDLINE)) return;
  } else {
    if(comm_prompt(cmdbuf,PATH_MAX+1,NULL,0,"Add to pipeline: ",FM_TABCOMPLETE,PV_CMDLINE)) return;
  }
  if(!cmdbuf[0]) {
    view_openfile(viewnode);
  } else {
    if(!config_verifyformat(cmdbuf,"fdp")) {
      gui_mildbeep("Error: invalid format string");
      return;
    }
    if(!ispipe) view_closefile();
    view_setuppipe(viewnode,cmdbuf);
  }
}

pid_t pipepid;
void view_setuppipe(int node,const char *cmd) {
  const char *p;
  int ret,pfd[2];
  int hub[2],oldlungs,num,childpid,transpid;
  char transbuf[PIPE_BLKSIZ];
  char buf_f[PATH_MAX+1];
  char buf_d[PATH_MAX+1];
  char buf_p[PATH_MAX+1];

  if(ispipe) {
    if(-1==pipe(hub)) {
      gui_mildbeep("Error: pipe failed (%s)",strerror(errno));
      return;
    }
  } else {
    hub[0]=open(data_fullname(node),O_RDONLY);
    if(hub[0]==-1) {
      gui_mildbeep("Error: pipe failed; cannot open original file (%s)",strerror(errno));
      view_openfile(node);
      return;
    }
  }

  if(-1==pipe(pfd)) {
    gui_mildbeep("Error: cannot allocate pipe (%s)",strerror(errno));
    close(hub[0]);
    if(ispipe) close(hub[1]);
    if(!ispipe) view_openfile(node);
    return;
  }
  oldlungs=lungs;
  lungs=pfd[0];

  childpid=0; /* to please gcc */
  if(ispipe) childpid=pipepid;
  pipepid=fork();
  if(-1==pipepid) {
    gui_mildbeep("Error: fork failed (%s)",strerror(errno));
    close(hub[0]);
    if(ispipe) close(hub[1]);
    close(pfd[0]);
    close(pfd[1]);
    lungs=oldlungs;
    view_openfile(node);
    return;
  }
  if(!pipepid) {
    if(ispipe) {
      close(hub[1]);
      close(oldlungs);
    }
    close(pfd[0]);
    if((-1==dup2(hub[0],0))||
       (-1==dup2(pfd[1],1))||
       (-1==dup2(pfd[1],2))) {
      write(pfd[1],"lxt: error: dup2 failed (",25);
      write(pfd[1],strerror(errno),strlen(strerror(errno)));
      write(pfd[1],")\n",2);
      _exit(-1);
    }
    strcpy(buf_f,data_name(node));
    strcpy(buf_d,data_fullname(data_updir(node)));
    strcpy(buf_p,data_fullname(node));
    p=comm_configparsef_wrap(cmd,"fdp",buf_f,buf_d,buf_p);
    execlp("sh","sh","-c",p,NULL);
    write(pfd[1],"lxt: error: execlp failed (",27);
    write(pfd[1],strerror(errno),strlen(strerror(errno)));
    write(pfd[1],")\n",2);
    _exit(-1);
  }
  close(pfd[1]);
  close(hub[0]);

  if(ispipe) {
    transpid=fork();
    if(-1==transpid) {
      gui_mildbeep("Error: fork failed (%s)",strerror(errno));
      close(pfd[0]);
      close(hub[1]);
      lungs=oldlungs;
      view_openfile(node);
      return;
    }
    if(!transpid) {
      close(pfd[0]);
      if(pipereadbuf) write(hub[1],pipereadbuf,pipereadsiz);
      if(childpid) for(;;) {
        fcntl(oldlungs,F_SETFL,0);
        num=read(oldlungs,transbuf,PIPE_BLKSIZ);
        if(num>0) write(hub[1],transbuf,num);
        else _exit(0);
      }
      _exit(0);
    }
    close(hub[1]);
    close(oldlungs);
    free(pipereadbuf);
    pipereadbuf=NULL;
    pipereadsiz=0;
  }

  if(ispipe) {
    num=strlen(pipename);
    pipename=realloc(pipename,strlen(cmd)+num+3);
    if(!pipename) outofmem("view_setuppipe");
    strcpy(pipename+num," |");
    strcpy(pipename+num+2,cmd);
  } else {
    pipename=malloc(strlen(cmd)+1);
    if(!pipename) outofmem("view_setuppipe");
    strcpy(pipename,cmd);
  }
  ret=fcntl(lungs,F_SETFL,O_NONBLOCK);
  pipereadbuf=NULL;
  pipebufsiz=pipereadsiz=0;
  pipecutshort=0;
  filesize=0;
  free(parabuf);
  parabuf=NULL;
  if(-1==ret) {
    gui_mildbeep("Error: fcntl failed (%s)",strerror(errno));
    view_openfile(node);
  }
  view_cleanstate();
  ispipe=1;
}

void view_closepipe(void) {
  if(!ispipe) return;
  close(lungs);
  free(pipereadbuf);
  free(pipename);
  pipename=NULL;
  free(parabuf);
  parabuf=NULL;
  view_cleanstate();
  ispipe=0;
}

void view_pipe_expandbuffer(int pos) {
  int num;
  if(!ispipe) crapout("view_pipe_expandbuffer","called when not in pipe mode");
  if(!pipepid) return;
  pos=pos-((pos-1)%PIPE_BLKSIZ)+PIPE_BLKSIZ-1;
  if(pos<pipereadsiz) return;
  if(pos>=pipebufsiz) {
    pipereadbuf=realloc(pipereadbuf,pos+1);
    if(!pipereadbuf) outofmem("view_pipe_expandbuffer");
  }
  num=read(lungs,pipereadbuf+pipereadsiz,pos-pipereadsiz+1);
  if(num<pos-pipereadsiz+1) pipecutshort=1; else pipecutshort=0;
  if(num>0) pipereadsiz+=num;
  else if(child_exited(pipepid)) {
    filesize=pipereadsiz;
    pipepid=0;
    pipecutshort=0;
  }
}

void view_updatepipe(void) {
  if(!ispipe) return;
  if(pipecutshort&&pipepid) {
    view_drawview();
    doupdate();
  }
}

int view_readbuffer(int pos,char *buf,int size) {
  int num;
  if(viewhelp) {
    num=filesize-pos;
    if(num>size) num=size;
    memcpy(buf,viewstring+pos,num);
  } else if(ispipe) {
    view_pipe_expandbuffer(pos+size);
    num=pipereadsiz-pos;
    if(num>size) num=size;
    memcpy(buf,pipereadbuf+pos,num);
  } else {
    if(viewfile==NULL) crapout("view_readbuffer","file is not open");
    fseek(viewfile,pos,SEEK_SET);
    num=fread(buf,1,size,viewfile);
  }
  return num;
}

int view_readonechar(int pos) {
  if(viewhelp) {
    return viewstring[pos];
  } else if(ispipe) {
    view_pipe_expandbuffer(pos);
    if(pos<pipereadsiz)
      return pipereadbuf[pos];
    else return EOF;
  } else {
    fseek(viewfile,pos,SEEK_SET);
    return getc(viewfile);
  }
}

/* like readbuffer, but crunch bold and underline codes */
int view_readbuffer_crunchcodes(int *pos,char *buf,char *attrib,int *phys,int size) {
  char *tmp;
  int remain,out,in,num;

  tmp=malloc(size+2);
  if(!tmp) outofmem("view_readbuffer_crunchcodes");
  remain=size;
  out=0;
  for(;remain;) {
    num=view_readbuffer(*pos,tmp,remain+2);
    if(!num) break;
    if(num>=3) for(in=0;in<(num-2);) {
      if((tmp[in+1]=='\b')&&(tmp[in]==tmp[in+2])) {
        if(attrib) attrib[out]=1;
        in+=2;
      } else if((tmp[in+1]=='\b')&&(tmp[in]=='_'))  {
        if(attrib) attrib[out]=2;
        in+=2;
      } else {if(attrib) attrib[out]=0;}
      if(phys) phys[out]=in+*pos;
      buf[out]=tmp[in];
      out++; in++;
      if(out==size) {(*pos)+=in; return out;}
    }
    else for(in=0;in<num;in++) {
      if(attrib) attrib[out]=0;
      if(phys) phys[out]=in+*pos;
      buf[out]=tmp[in];
      out++;
      if(out==size) {(*pos)+=in; return out;}
    }
    remain=size-out;
    (*pos)+=in;
  }
  return out;
}

int child_exited(int pid) {
  int ret;
  if(waitpid(pid,NULL,WNOHANG)) return 1;
  errno=0;
  ret=kill(pid,0);
  if(!ret) return 0;
  if(errno==EPERM) return 0;
  return 1;
}

const char *oldviewfilename;
FILE *oldviewfile;
const unsigned char *oldviewstring;
int   oldtopview;
int   oldviewscrolled;
int   oldfilesize;
int   oldtopview;
int   oldbotview;
int   oldvieweof;
int   oldviewmode;
int   oldviewmask;
int   oldviewhelp;
int  *oldparabuf;
int   oldparaline;
int   oldparasize;
int   oldparabufmemsiz;
int   oldispipe;
int   oldissearch;
int   oldshowsearch;
char  oldsearchexp[REGEX_SIZ];
my_regex_t oldsearch_comp;

/* these functions are called to save or resume state in case help mode is
   entered while in view */

void view_savestate(void) {
  oldviewfilename=viewfilename;
  oldviewfile=viewfile;
  oldviewstring=viewstring;
  oldtopview=topview;
  oldviewscrolled=viewscrolled;
  oldfilesize=filesize;
  oldtopview=topview;
  oldbotview=botview;
  oldvieweof=vieweof;
  oldviewmode=viewmode;
  oldviewmask=viewmask;
  oldviewhelp=viewhelp;
  oldispipe=ispipe;

  oldparabuf=parabuf;
  parabuf=NULL;
  oldparaline=paraline;
  oldparasize=parasize;
  oldparabufmemsiz=parabufmemsiz;

  oldissearch=issearch;
  oldshowsearch=showsearch;
  memcpy(oldsearchexp,searchexp,REGEX_SIZ);
  oldsearch_comp=search_comp;
  issearch=0;
  view_undosearchbuffer();
}

void view_resumestate(void) {
  viewfilename=oldviewfilename;
  viewfile=oldviewfile;
  viewstring=oldviewstring;
  topview=oldtopview;
  viewscrolled=oldviewscrolled;
  filesize=oldfilesize;
  topview=oldtopview;
  botview=oldbotview;
  vieweof=oldvieweof;
  viewmode=oldviewmode;
  viewmask=oldviewmask;
  viewhelp=oldviewhelp;
  ispipe=oldispipe;

  free(parabuf);
  parabuf=oldparabuf;
  paraline=oldparaline;
  parasize=oldparasize;
  parabufmemsiz=oldparabufmemsiz;

  view_undosearch();
  issearch=oldissearch;
  showsearch=oldshowsearch;
  memcpy(searchexp,oldsearchexp,REGEX_SIZ);
  search_comp=oldsearch_comp;
}

void view_toggleshowsearch(void) {
  showsearch=!showsearch;
}

int view_ismatch(int pos) {
  int n;
  if((!issearch)||(!showsearch)) return 0;
  n=pos-match_start;
  if(n<0) return 0;
  if(n>=match_len) return 0;
  return search_match[n];
}

void view_drawview(void) {
  int x,y,c,l,num,p,min,max,pos,match;
  int bold,brokeout;
  unsigned char *buf;
  char *tohex="0123456789abcdef";

  if((oldmaxx!=maxx)||(oldmaxy!=maxy)) view_handleresize();

  if(viewmode==VIEW_DUMP) topview-=topview%viewdmpw;
  if(viewmode==VIEW_HEX ) topview-=topview%viewhexw;
  buf=screenbuffer; /* don't really need two names for it, fix eventually */
  if(viewmode==VIEW_TEXT) {
    pos=topview;
    num=view_readbuffer_crunchcodes(&pos,buf,screenattrib,screenphys,BUFSIZE);
  } else {
    num=view_readbuffer(topview,buf,BUFSIZE);
  }
  if(issearch&&showsearch) view_search_findrange(topview,topview+BUFSIZE);

  p=0;
  vieweof=0;
  brokeout=0;
  x=y=0; /* to please gcc */
  if(updateall) viewscrolled=0;
  if(!viewscrolled) werase(gui_bigun);
  else {
    scrollok(gui_bigun,TRUE);
    wsetscrreg(gui_bigun,1,gui_bigun->_maxy-1);
    wscrl(gui_bigun,viewscrolled);
    scrollok(gui_bigun,FALSE);
  }
  wbkgdset(gui_bigun,A_BOLD|COLOR_PAIR(C_NORMAL));
  if(!keep_msbar) {
    werase(gui_msbar);
    wbkgdset(gui_msbar,A_BOLD|COLOR_PAIR(C_NORMAL));
  }

  switch(viewmode) {
  case VIEW_TEXT:
    if(!keep_msbar)
      mvwprintw(gui_msbar,0,0," (text%s)",viewmask?"/m":"  ");
    l=0;
    if(num) for(y=1;y<gui_bigun->_maxy;y++) {
      for(x=0;x<(gui_bigun->_maxx+1);x++) {
        match=view_ismatch(screenphys[p]);
        bold=screenattrib[p]|(match?4:0);
        c=buf[p++];
	if(p>num) break;
        if(c==10) {if(!l) break; else {l=0; x--; continue;}}
          else
        if(c==9) {
          view_printok(x,y,' ',0);
          while((x+1)%8) {x++; view_printok(x,y,' ',0);}
        }
          else
        if(c!=13) view_printok(x,y,c,bold);
        l=0;
      }
      if(p>=num) {brokeout=1; break;}
      l=(x==(gui_bigun->_maxx+1));
    }
    if(brokeout&&!x) y--;
    botview=topview+p;
    break;
  case VIEW_DUMP:
    if(!keep_msbar)
      mvwprintw(gui_msbar,0,0," (dump%s)",viewmask?"/m":"  ");
    if(num<(viewdmpw*gui_bigun->_maxy-(2*viewdmpw-1))) vieweof=1;
    min=1;
    max=gui_bigun->_maxy;
    if(viewscrolled>0) min=max-viewscrolled;
    if(viewscrolled<0) max=min-viewscrolled;
    p=(min-1)*viewdmpw;
    if(num) for(y=min;y<max;y++) {
      mvwprintw(gui_bigun,y,2,"%06x",topview+p);
      for(x=0;x<viewdmpw;x++) {
        match=view_ismatch(topview+p)?4:0;
        c=buf[p++];
        if(p>num) break;
        view_printok(x+10,y,c,match);
      }
      if(p>=num) { brokeout=1; break; }
    }
    if(brokeout) botview=topview+num;
    else         botview=topview+(gui_bigun->_maxy-1)*viewdmpw;
    break;
  case VIEW_HEX:
    if(!keep_msbar)
      mvwprintw(gui_msbar,0,0," (hex %s)",viewmask?"/m":"  ");
    if(num<(viewhexw*gui_bigun->_maxy-(2*viewhexw-1))) vieweof=1;
    min=1;
    max=gui_bigun->_maxy;
    if(viewscrolled>0) min=max-viewscrolled;
    if(viewscrolled<0) max=min-viewscrolled;
    p=(min-1)*viewhexw;
    if(num) for(y=min;y<max;y++) {
      p=(y-1)*viewhexw;
      mvwprintw(gui_bigun,y,2,"%06x",topview+p);
      for(x=0;x<viewhexw;x++) {
        match=view_ismatch(topview+p);
        c=buf[p++];
        if(p>num) break;
        view_printok(x+10,y,c,match?4:0);
        l=12+viewhexw+x*3+(int)(x/4);
        if(match) {
          mvwaddch(gui_bigun,y,l  ,A_BOLD|COLOR_PAIR(C_HILITE)|tohex[c>>4]);
          mvwaddch(gui_bigun,y,l+1,A_BOLD|COLOR_PAIR(C_HILITE)|tohex[c&15]);
        } else {
          mvwaddch(gui_bigun,y,l  ,A_BOLD|COLOR_PAIR(C_NORMAL)|tohex[c>>4]);
          mvwaddch(gui_bigun,y,l+1,A_BOLD|COLOR_PAIR(C_NORMAL)|tohex[c&15]);
        }
        if((match==2)&&(x+1<viewhexw)) {
          mvwaddch(gui_bigun,y,l+2,A_BOLD|COLOR_PAIR(C_HILITE)|' ');
          if(x%4==3)
            mvwaddch(gui_bigun,y,l+3,A_BOLD|COLOR_PAIR(C_HILITE)|' ');
        }
      }
      if(p>=num) { brokeout=1; break; }
    }
    if(brokeout) botview=topview+num;
    else         botview=topview+(gui_bigun->_maxy-1)*viewhexw;
    break;
  default:
    crapout("view_drawview","invalid viewmode");
  }

  if(!num) {brokeout=1; y=0;}
  if(brokeout&&(y<gui_bigun->_maxy-1)) {
    wbkgdset(gui_bigun,A_BOLD|COLOR_PAIR(C_VIEWEOF));
    if(ispipe&&pipecutshort)
      mvwprintw(gui_bigun,y+1,0,"<WAIT FOR PIPE...>");
    else
      mvwprintw(gui_bigun,y+1,0,"<EOF>");
    wbkgdset(gui_bigun,A_BOLD|COLOR_PAIR(C_NORMAL));
    vieweof=1;
  }

  if(!keep_msbar) {
    if(ispipe)
      mvwprintw(gui_msbar,0,19,"Pipe: %s",
        gui_printable_quoted(gui_bigun->_maxx-24,pipename));
    else
      mvwprintw(gui_msbar,0,19,"Pipe: none");
    if(filesize)
      mvwprintw(gui_msbar,0,10,"%2d%%-%2d%%",
	        (int)((float)(topview)/filesize*100),
	        (int)((float)(botview)/filesize*100));
    else
      mvwprintw(gui_msbar,0,10,"??%%-??%%");
  }
  wbkgdset(gui_bigun,A_BOLD|COLOR_PAIR(C_MENU));
  wmove(gui_bigun,0,0);
  whline(gui_bigun,A_BOLD|COLOR_PAIR(C_MENU)|ACS_HLINE,maxx);
  wmove(gui_bigun,gui_bigun->_maxy,0);
  whline(gui_bigun,A_BOLD|COLOR_PAIR(C_MENU)|ACS_HLINE,maxx);
  if(filesize) {
    wmove(gui_bigun,gui_bigun->_maxy,(int)((float)(topview)/filesize*maxx));
    whline(gui_bigun,A_BOLD|COLOR_PAIR(C_MENU)|'=',
      (int)((float)(botview)/filesize*maxx)-
      (int)((float)(topview)/filesize*maxx)+1);
  }
  mvwprintw(gui_stbar,0,0," %-*.*s",maxx-26,maxx-26,gui_printable(maxx-26,viewfilename));
  wnoutrefresh(gui_bigun);
  if(!keep_msbar) wnoutrefresh(gui_msbar);
  wnoutrefresh(gui_stbar);
  viewscrolled=0;
}

void view_calcparagraph(int fpos) {
  int xpos,oxp,ln;
  int c,nxtc,nxtnxtc;
  if(!parabuf) parabufmemsiz=0;
  ln=0;
  nxtc=view_readonechar(fpos);
  nxtnxtc=view_readonechar(fpos+1);
  for(xpos=0;;) {
    c=nxtc;
    nxtc=nxtnxtc;
    nxtnxtc=view_readonechar(fpos+2);
    if(xpos) if((c==10)||(nxtc==EOF)) { parasize=ln; return; }
    oxp=xpos;
    do {
      if(!(xpos%maxx)) {
        if(ln>=parabufmemsiz) {
          parabufmemsiz+=100;
          parabuf=realloc(parabuf,parabufmemsiz*sizeof(int));
          if(!parabuf) outofmem("view_calcparagraph");
        }
        parabuf[ln++]=fpos;
      }
      xpos++;
    } while(((c=='\t')||((c=='_')&&(nxtc=='\b')&&(nxtnxtc=='\t')))&&((xpos)%8));
    if(!oxp) if(c==10) { parasize=ln; return; }
    fpos++;
    if((nxtc=='\b')&&((c=='_')||(c==nxtnxtc))) {
      fpos+=2;
      nxtc=view_readonechar(fpos);
      nxtnxtc=view_readonechar(fpos+1);
    }
  }
}

void view_scrollup(void) {
  int pos;
  int buf;
  if(!topview) return;
  switch(viewmode) {
  case VIEW_TEXT:
    if(parabuf&&paraline) {
      paraline--;
      topview=parabuf[paraline];
      break;
    } else {
      pos=topview-1;
      do {
        pos--;
        if(pos<0) break;
        buf=view_readonechar(pos);
      } while(buf!=10);
      pos++;
      view_calcparagraph(pos);
      paraline=parasize-1;
      topview=parabuf[paraline];
      break;
    }
  case VIEW_DUMP:
    topview-=viewdmpw;
    if(topview<0) viewscrolled=topview=0;
    else viewscrolled--;
    break;
  case VIEW_HEX:
    topview-=viewhexw;
    if(topview<0) viewscrolled=topview=0;
    else viewscrolled--;
    break;
  }
}

void view_scrolldown(void) {
  int c;
  if(vieweof) return;
  switch(viewmode) {
  case VIEW_TEXT:
    if(!parabuf) {view_calcparagraph(topview); paraline=0;}
    if(paraline<(parasize-1)) {
      paraline++;
      topview=parabuf[paraline];
      break;
    } else {
      do {
        c=view_readonechar(topview);
        topview++;
      } while((c!=10)&&(c!=EOF));
      view_calcparagraph(topview);
      paraline=0;
      topview=parabuf[0];
      break;
    }
  case VIEW_DUMP:
    topview+=viewdmpw;
    viewscrolled++;
    break;
  case VIEW_HEX:
    topview+=viewhexw;
    viewscrolled++;
    break;
  }
}

void view_pageup(void) {
  int n;
  switch(viewmode) {
  case VIEW_TEXT:
    for(n=2;n<gui_bigun->_maxy;n++) view_scrollup();
    break;
  case VIEW_DUMP:
    topview-=viewdmpw*(gui_bigun->_maxy-2);
    if(topview<0) viewscrolled=topview=0;
    break;
  case VIEW_HEX:
    topview-=viewhexw*(gui_bigun->_maxy-2);
    if(topview<0) viewscrolled=topview=0;
    break;
  }
}

void view_pagedn(void) {
  int n;
  if(vieweof) return;
  switch(viewmode) {
  case VIEW_TEXT:
    for(n=2;n<gui_bigun->_maxy;n++) view_scrolldown();
    break;
  case VIEW_DUMP:
    topview+=viewdmpw*(gui_bigun->_maxy-2);
    break;
  case VIEW_HEX:
    topview+=viewhexw*(gui_bigun->_maxy-2);
    break;
  }
}

void view_home(void) {
  viewscrolled=topview=0;
  free(parabuf);
  parabuf=NULL;
}

void view_end(void) {
  int n,seek;
  if(vieweof) return;
  seek=filesize;
  if(ispipe&&pipepid) {
    for(;pipepid;) {
      gui_printmsg("reading: % 6dk, press ctrl-G to abort",pipereadsiz/1024);
      view_pipe_expandbuffer(pipereadsiz+16384);
      if(is_cancel()) break;
    }
    if(pipereadsiz) seek=pipereadsiz;
  }
  switch(viewmode) {
  case VIEW_TEXT:
    free(parabuf);
    parabuf=NULL;
    topview=seek-1;
    for(n=0;n<gui_bigun->_maxy-2;n++) view_scrollup();
    view_drawview();
    break;
  case VIEW_DUMP:
    topview=(topview%viewdmpw)+((seek-1)/viewdmpw-
            (gui_bigun->_maxy-3))*viewdmpw;
    if(topview<0) viewscrolled=topview=0;
    break;
  case VIEW_HEX:
    topview=(topview%viewhexw)+((seek-1)/viewhexw-
            (gui_bigun->_maxy-3))*viewhexw;
    if(topview<0) viewscrolled=topview=0;
    break;
  }
}

void view_printok(int x,int y,int c,int bold) {
/* FIXME - I think wattrset has to be called because of an ncurses bug */
  if((c>31)&&(c<127)) {
    switch(bold) {
    case 0:
      mvwaddch(gui_bigun,y,x,c|A_BOLD|COLOR_PAIR(C_NORMAL));
      break;
    case 1:
      mvwaddch(gui_bigun,y,x,c|A_BOLD|COLOR_PAIR(C_BOLD));
      break;
    case 2:
      mvwaddch(gui_bigun,y,x,c|A_BOLD|COLOR_PAIR(C_UNDERLINE));
      break;
    case 4:
    case 5:
    case 6:
      mvwaddch(gui_bigun,y,x,c|A_BOLD|COLOR_PAIR(C_HILITE));
      break;
    }
  } else if(viewmask) {
    mvwaddch(gui_bigun,y,x,'.'|A_BOLD|COLOR_PAIR(C_NORMAL));
  } else {
      switch(c) {
/* FIXME - this list is very term specific */
    case 0:
      wattrset(gui_bigun,COLOR_PAIR(C_HILITE));
      mvwaddch(gui_bigun,y,x,'@'|COLOR_PAIR(C_HILITE));
      break;
    case 4:
    case 8:
    case 10:
    case 12:
    case 13:
    case 14:
    case 15:
    case 27:
      wattrset(gui_bigun,COLOR_PAIR(C_HILITE));
      mvwaddch(gui_bigun,y,x,('A'+c-1)|COLOR_PAIR(C_HILITE));
      break;
/*
    case 0x7f:
      wattrset(gui_bigun,COLOR_PAIR(C_HILITE));
      mvwaddch(gui_bigun,y,x,'?'|COLOR_PAIR(C_HILITE));
      break;
*/
    case 0x9b:
      wattrset(gui_bigun,COLOR_PAIR(C_HILITE));
      mvwaddch(gui_bigun,y,x,'.'|COLOR_PAIR(C_HILITE));
      break;
    default:
      wattrset(gui_bigun,A_BOLD|COLOR_PAIR(C_NORMAL)|A_ALTCHARSET);
      mvwaddch(gui_bigun,y,x,c|A_BOLD|COLOR_PAIR(C_NORMAL)|A_ALTCHARSET);
      break;
    }
  }
  wattrset(gui_bigun,A_BOLD|COLOR_PAIR(C_NORMAL));
}

#define search_flags 0
#define SEARCHOVERLAP (maxx*3)
/* FIXME - SEARCHBUFSIZ must be a bit bigger than SBUFZONE or not everything
   will get highlited in dump and hex modes, don't know why */
#define SEARCHBUFSIZ 10000
#define SBUFZONE 10000

void view_search(void) {
  char newsearch[REGEX_SIZ];
  char errbuf[PATH_MAX+1];
  int errcode;

  if(!issearch) searchexp[0]=0;
  if(comm_prompt(newsearch,REGEX_SIZ,NULL,0,"Search for: ",FM_TABCOMPLETE,PV_SEARCH)) return;
  if(newsearch[0]&&strcmp(newsearch,searchexp)) {
    view_undosearch();
    if((errcode=my_regcomp(&search_comp,newsearch,search_flags))) {
      my_regerror(errcode,&search_comp,errbuf,PATH_MAX+1);
      gui_mildbeep("Error: search failed (%s)",errbuf);
      return;
    }
    strcpy(searchexp,newsearch);
  }
  if(!searchexp[0]) {set_cancel(); return;}
  issearch=1;
  showsearch=1;
  view_search_findnext();
}

void view_search_rev(void) {
  gui_mildbeep("Error: function unimplemented");
}

int srange_sbuf;
int srange_ebuf;

void view_undosearchbuffer(void) {
  free(search_match);
  search_match=NULL;
  num_match=0;
  srange_ebuf=-1;
}

void view_undosearch(void) {
  if(!issearch) return;
  view_undosearchbuffer();
  my_regfree(&search_comp);
  issearch=0;
}

char searchbuffer[SEARCHBUFSIZ];
int searchphys[SEARCHBUFSIZ];
void view_search_findnext(void) {
  int oldpos,newpos,pos,num;
  my_regmatch_t match;
  oldpos=topview;
  vieweof=0;
  view_scrolldown();
  if(topview!=oldpos) for(pos=topview;;) {
    gui_printmsg("searching: % 6dk, press ctrl-G to abort",(pos-topview)/1024);
    if(is_cancel()) break;

    do {
      num=0; /* to please gcc */
      if(is_cancel()) break;
      newpos=pos;
      if(viewmode==VIEW_TEXT) {
        num=view_readbuffer_crunchcodes(&newpos,searchbuffer,NULL,searchphys,SEARCHBUFSIZ-1);
      } else {
        num=view_readbuffer(newpos,searchbuffer,SEARCHBUFSIZ-1);
        newpos+=num;
      }
      if(ispipe&&pipepid&&(newpos-pos<=SEARCHOVERLAP)) small_nap();
    } while(ispipe&&pipepid&&(newpos-pos<=SEARCHOVERLAP));
    if(is_cancel()) break;

    if(!num) break;
    searchbuffer[num]=0;
    if(!my_regexec(&search_comp,searchbuffer,1,&match,0)) {
      if(match.rm_so!=-1) {
        if(viewmode==VIEW_TEXT) topview=searchphys[match.rm_so];
        else                    topview=match.rm_so+pos;
      } else topview=oldpos;
      view_resetparagraph();
      if(keep_msbar==1) keep_msbar=0;
      return;
    }
    if((newpos-pos)<=SEARCHOVERLAP) break;
    pos=newpos-SEARCHOVERLAP;
  }
  if(keep_msbar==1) keep_msbar=0;
  if(!is_cancel()) gui_mildbeep("Error: search string not found");
  topview=oldpos;
  view_resetparagraph();
}

void view_search_findrange(int start,int end) {
  int newpos,pos,num,p,n,lastfind,iseof,lastprint;
  my_regmatch_t match;
  if(!issearch) crapout("view_search_findrange","called when not in search mode");
  if(end<start) crapout("view_search_findrange","end<start");
  if(start<0) crapout("view_search_findrange","start<0");
  if((srange_sbuf<start)&&(srange_ebuf>end)) return;
  view_undosearchbuffer();
/* do a little more than we need, so that we can skip out of searches near
   this range in the future */
  start-=SBUFZONE;
  end+=SBUFZONE;
  srange_sbuf=start;
  srange_ebuf=end;
  start-=SEARCHOVERLAP;
  if(start<0) start=0;
  match_start=start;
  match_len=end-start+SEARCHBUFSIZ;
  search_match=malloc(match_len*sizeof(int));
  if(!search_match) outofmem("view_search_findrange");
  for(n=0;n<match_len;n++) search_match[n]=0;

  for(pos=start;;) {
    gui_printmsg("hilighting: % 6d bytes, press ctrl-G to abort",pos-start);
    lastprint=(pos-start)-(pos-start)%100;
    do {
      num=0; /* to please gcc */
      if(is_cancel()) break;
      newpos=pos;
      if(viewmode==VIEW_TEXT) {
        num=view_readbuffer_crunchcodes(&newpos,searchbuffer,NULL,searchphys,SEARCHBUFSIZ-1);
      } else {
        num=view_readbuffer(newpos,searchbuffer,SEARCHBUFSIZ-1);
        newpos+=num;
      }
      if(ispipe&&pipepid&&(newpos-pos<=SEARCHOVERLAP)) small_nap();
    } while(ispipe&&pipepid&&(newpos-pos<=SEARCHOVERLAP));
    if(is_cancel()) break;
    if(!num) break;
    iseof=(newpos-pos<=SEARCHOVERLAP);
    searchbuffer[num]=0;
    lastfind=0;
    for(p=0;((newpos-lastfind)>SEARCHOVERLAP)||(!lastfind)||iseof;) {
      if(is_cancel()) break;
      if(lastfind-start>lastprint+99) {
        gui_printmsg("hilighting: % 6d bytes, press ctrl-G to abort",lastfind-start);
        lastprint=(lastfind-start)-(lastfind-start)%100;
      }
      if(!my_regexec(&search_comp,searchbuffer+p,1,&match,0)) {
        if(match.rm_so!=-1) {
          num_match++;
          if(viewmode==VIEW_TEXT) {
            for(n=searchphys[match.rm_so+p];n<searchphys[match.rm_eo+p]-1;n++)
              search_match[n-match_start]=2;
            search_match[n-match_start]=1;
          } else {
            for(n=match.rm_so+pos+p;n<match.rm_eo+pos+p-1;n++)
              search_match[n-match_start]=2;
            search_match[n-match_start]=1;
          }
          lastfind=searchphys[match.rm_eo+p];
          p+=match.rm_eo;
          if(lastfind>end) break;
          continue;
        }
      } else break;
    }
    if(iseof) break;
    pos=MAX(lastfind,newpos-SEARCHOVERLAP);
    if(pos>end) break;
  }
  if(is_cancel()) {showsearch=0; srange_ebuf=-1;}
  if(keep_msbar==1) keep_msbar=0;
}
