/* linuXtree copyright (c) 1998 Dan Stahlke under the GPL
   See file 'COPYING' for details */

/*
  new options need to be added to:
    1. the defaults list (config_setdefaults and lxt-conf-default)
    2. the boolopts[] array or
       the enumopts[] array or
       the  stropts[] array or
      a. the if(strcmp()) part in config_readfile,
      b. the print_config() function, and
      c. the command line help
      d. the config editor and related functions
    3. structure in configfile.h
*/

#include "all.h"

#define MAXLEN 800

struct configtype config;
int linenum;
const char *curfile;
int recovermode;
int validconfigfile=0;

struct { char *text; int *bool; char *help; } boolopts[]={
  {"silent",		&config.silent,		CHELP_SILENT		},
  {"tagtoggle",		&config.tagtoggle,	CHELP_TAGTOGGLE		},
  {"preservetag",	&config.preservetag,	CHELP_PRESERVETAG	},
  {"autorestatdirs",	&config.autorestatdirs,	CHELP_AUTORESTATDIRS	},
  {"autorestatfiles",	&config.autorestatfiles,CHELP_AUTORESTATFILES	},
  {"followlinks",	&config.followlinks,	CHELP_FOLLOWLINKS	},
  {"followlinkdirs",	&config.followlinkdirs,	CHELP_FOLLOWLINKDIRS	},
  {"clearkeybuffer",	&config.clearkeybuffer,	CHELP_CLEARKEYBUFFER	},
  {"viewpowtwo",	&config.viewpowtwo,	CHELP_VIEWPOWTWO	},
  {"logtocurdir",	&config.logtocurdir,	CHELP_LOGTOCURDIR	},
  {"defaultreadln",	&config.defaultreadln,	CHELP_DEFAULTREADLN	},
  {"animatedlogging",	&config.animatedlogging,CHELP_ANIMATEDLOGGING	},
  {NULL,		NULL,			NULL			}
};

char *enum_rawmode[]={"hide","octal","control",NULL};

struct { char *text; int *var; char **vals; char *help; } enumopts[]={
  {"rawmode",	&config.rawmode,	enum_rawmode,	CHELP_RAWMODE	},
  {NULL,	NULL,			NULL,		NULL		}
};

struct { char *text; char **var; char *opts; char *help; } stropts[]={
  {"editor",		&config.editor,		"fdp",	CHELP_EDITOR		},
  {"archivebranch",	&config.archivebranch,	"ab",	CHELP_ARCHIVEBRANCH	},
  {"archivetagged",	&config.archivetagged,	"abc",	CHELP_ARCHIVETAGGED	},
  {NULL,		NULL,			NULL,	NULL			}
};

void config_setdefaults(void) {
  int n;

  config.rawmode=RAWMODE_CONTROL;
  config.silent=0;
  config.tagtoggle=0;
  config.preservetag=0;
  config.autorestatdirs=1;
  config.autorestatfiles=1;
  config.followlinks=1;
  config.followlinkdirs=1;
  config.clearkeybuffer=1;
  config.viewpowtwo=1;
  config.logtocurdir=1;
  config.defaultreadln=1;
  config.animatedlogging=1;

  for(n=0;stropts[n].var;n++) *stropts[n].var=NULL;
  config_setoption("editor:vi %p");
  config_setoption("archivebranch:tar cvfz %a -C %b .");
  config_setoption("archivetagged:cd %b && tar cvfz %a -T %c");

  config.external=NULL;
}

void config_init(int argc,char *argv[]) {
  char *sysconf="/etc/lxt-conf";
  char *home,*userconf;
  int n,dumpconf;

  home=getenv("HOME");
  if(!home) home="~";
  userconf=malloc(strlen("/.lxt-conf")+strlen(home)+1);
  if(!userconf) outofmem("config_init");
  sprintf(userconf,"%s/.lxt-conf",home);
  dumpconf=0;
  linenum=-1;
  curfile=NULL;
  recovermode=0;

  config.inX=(getenv("DISPLAY")!=NULL);
  config_setdefaults();

  for(n=1;n<argc;n++) {
    if(!strcmp(argv[n],"-c")) {
      n++;
      if(n==argc) {
        printf("\nerror: option -c requires an argument\ntype 'lxt -h' for help\n\n");
        exit(1);
      }
      if(strlen(argv[n])>MAXLEN) {
        printf("\nerror: option too long\n\n");
        exit(1);
      }
      free(userconf);
      userconf=malloc(strlen(argv[n]+1));
      if(!userconf) outofmem("config_init");
      strcpy(userconf,argv[n]);
      if(access(userconf,R_OK)) {
        printf("\nerror: config file \"%s\" not found\n\n",userconf);
        exit(1);
      }
    } else
    if(!strcmp(argv[n],"-h")) {
      n++;
      if(n==argc) config_cmdhelp();
      else        config_opthelp(argv[n]);
      exit(0);
    } else
    if(!strcmp(argv[n],"-recover")) {
      data_restoretree();
      recovermode=1;
    }
  }

  config_readfile(sysconf);
  config_readfile(userconf);
  /* if there was no config file, set some default externals */
  /* most of the pipe commands were copied out of lesspipe.sh */
  if(!validconfigfile) {
    config_setoption("external:*.jpeg:zgv %p:xv %p &");
    config_setoption("external:*.jpg:zgv %p:xv %p &");
    config_setoption("external:*.gif:zgv %p:xv %p &");
    config_setoption("external:*.ppm:zgv %p:xv %p &");
    config_setoption("external:*.pnm:zgv %p:xv %p &");
    config_setoption("external:*.html:lynx %p:.:lynx -dump %p");
    config_setoption("external:*.htm:lynx %p:.:lynx -dump %p");
    config_setoption("external:*.s3m:s3mod %p");
    config_setoption("external:*.mod:s3mod %p");
    config_setoption("external:*.mp3:mpg123 %p");
    config_setoption("external:*.pdb::rasmol %p");
    config_setoption("external:*.mol::rasmol -mdl %p");
    config_setoption("external:*/[Mm]akefile:make -f %p:make -f %p:");
    config_setoption("external:*/cat[1-9n]/*.[1-9n].gz:::gzip -d");
    config_setoption("external:*.[1-9n].gz:::gzip -d | groff -S -s -p -t -e -Tascii -mandoc");
    config_setoption("external:*.[1-9n]:::groff -S -s -p -t -e -Tascii -mandoc");
    config_setoption("external:*.tar:tar xvvf %p:.:tar tvvf %p");
    config_setoption("external:*.tgz:tar xzvvf %p:.:tar tzvvf %p");
    config_setoption("external:*.tar.gz:tar xzvvf %p:.:tar tzvvf %p");
    config_setoption("external:*.tar.[zZ]:tar xzvvf %p:.:tar tzvvf %p");
    config_setoption("external:*.tar.bz2:tar xyvvf %p:.:tar tyvvf %p");
    config_setoption("external:*.[zZ]:gzip -d %p:.:gzip -d");
    config_setoption("external:*.gz:gzip -d %p:.:gzip -d");
    config_setoption("external:*.bz2:bzip2 -d %p:.:bzip2 -d");
    config_setoption("external:*.zip:unzip %p:.:unzip -l %p");
    config_setoption("external:*/mbox:mail -f %p");
  }
  curfile=NULL;
  linenum=-1;

  printf("reading command line\n");
  for(n=1;n<argc;n++) {
    if(!strcmp(argv[n],"-c")) {n++; continue;}
    if(!strcmp(argv[n],"-recover")) {n++; continue;}
    if(!strcmp(argv[n],"-d")) {
      dumpconf=1;
    } else
    if(!strcmp(argv[n],"-l")) {
      config_setoption("logtocurdir:1");
    } else
    if((*argv[n]=='-')&&(*(argv[n]+1)=='-')) {
      if(config_setoption(argv[n]+2)) config_fileerror(argv[n],"invalid option");
    } else config_fileerror(argv[n],"invalid option");
  }
  if(dumpconf) {print_config(); exit(0);}
}

void config_readfile(const char *fn) {
  FILE *f;
  char buf[MAXLEN+2];
  int n;

  f=fopen(fn,"r");
  if(f==NULL) {
    printf("config file \"%s\" not found\n",fn);
    return;
  } else printf("reading config file \"%s\"\n",fn);
  validconfigfile=1;
  curfile=fn;
  linenum=0;
  while(!feof(f)) {
    linenum++;
    if(fgets(buf,MAXLEN+2,f)==NULL) break;
    for(n=0;buf[n]&&n<(MAXLEN+1);n++);
    if(n==(MAXLEN+1)) {
      buf[MAXLEN+1]=0;
      config_fileerror(buf,"line too long");
      exit(1);
    }
    if((buf[0]=='\n')||(buf[0]=='#')) continue;
    if(config_setoption(buf)) config_fileerror(buf,"invalid option");
  }
  fclose(f);
}

int config_setoption(const char *buf) {
  struct externaltype *vlist,*llist;
  const char *next;
  char opt[MAXLEN+1],param[MAXLEN+1],*s,*ss;
  int m,n;

  if(strlen(buf)>MAXLEN) {
    config_fileerror(buf,"option too long");
  }

  config_readopt(buf,buf,opt,&next);

  for(n=0;boolopts[n].text!=NULL;n++) {
    if(!strcmp(opt,boolopts[n].text)) {
      if(next) config_readopt(buf,next,param,&next);
      else     {*boolopts[n].bool=1; return 0;}
      if(next) {
        config_fileerror(buf,"only one argument allowed");
      }
      if(1!=sscanf(param,"%d",boolopts[n].bool))
        config_fileerror(buf,"invalid parameter");
      if((*boolopts[n].bool!=0)&&
         (*boolopts[n].bool!=1))
        config_fileerror(buf,"parameter must be 0 or 1");
      return 0;
    }
  }

  for(n=0;stropts[n].text!=NULL;n++) {
    if(!strcmp(opt,stropts[n].text)) {
      if(next) config_readopt(buf,next,param,&next);
      else {
        config_fileerror(buf,"option requires an argument");
      }
      if(next) {
        config_fileerror(buf,"only one argument allowed");
      }
      if(!config_verifyformat(param,stropts[n].opts)) {
        config_fileerror(buf,"invalid format string");
      }
      free(*stropts[n].var);
      *stropts[n].var=malloc(strlen(param)+1);
      if(!(*stropts[n].var)) outofmem("config_setoption");
      strcpy(*stropts[n].var,param);
      return 0;
    }
  }

  for(n=0;enumopts[n].text!=NULL;n++) {
    if(!strcmp(opt,enumopts[n].text)) {
      if(next) config_readopt(buf,next,param,&next);
      else {
        config_fileerror(buf,"option requires an argument");
      }
      if(next) {
        config_fileerror(buf,"only one argument allowed");
      }
      for(m=0;enumopts[n].vals[m];m++) {
        if(!strcmp(param,enumopts[n].vals[m])) {
          *enumopts[n].var=m;
          return 0;
        }
      }
      config_fileerror(buf,"invalid parameter");
    }
  }

  if(!strcmp(opt,"external")) {
    vlist=malloc(sizeof(*vlist));
    if(!vlist) outofmem("config_setoption");

    if(next) config_readopt(buf,next,param,&next);
    else {
      config_fileerror(buf,"option requires at least two arguments");
    }
    if(!param[0]) {
      config_fileerror(buf,"file extension must be given");
    }
    ss=malloc(strlen(param)+1);
    if(!ss) outofmem("config_setoption");
    strcpy(ss,param);
    vlist->fspec=ss;

    if(next) config_readopt(buf,next,param,&next);
    else {
      config_fileerror(buf,"option requires at least two arguments");
    }
    ss=malloc(strlen(param)+1);
    if(!ss) outofmem("config_setoption");
    strcpy(ss,param);
    vlist->com=ss;

    if(next) {
      config_readopt(buf,next,param,&next);
      s=param;
    } else {
      if(vlist->com[0]) s=".";
      else              s="";
    }
    ss=malloc(strlen(s)+1);
    if(!ss) outofmem("config_setoption");
    strcpy(ss,s);
    vlist->xcom=ss;

    if(next) {
      config_readopt(buf,next,param,&next);
      if(next) config_fileerror(buf,"only four arguments allowed");
      s=param;
    } else s="";
    ss=malloc(strlen(s)+1);
    if(!ss) outofmem("config_setoption");
    strcpy(ss,s);
    vlist->pipe=ss;

    vlist->next=NULL;

    if((vlist-> com[0]&&!config_verifyformat(vlist-> com,"fdp"))||
       (vlist->xcom[0]&&!config_verifyformat(vlist->xcom,"fdp"))||
       (vlist->pipe[0]&&!config_verifyformat(vlist->pipe,"fdp"))) {
      config_fileerror(buf,"invalid format string");
    }

    if(config.external) {
      llist=config.external;
      while(llist->next!=NULL) {
        if(!strcmp(llist->fspec,vlist->fspec)) {
          free(llist-> com);
          free(llist->xcom);
          free(llist->pipe);
          llist-> com=vlist-> com;
          llist->xcom=vlist->xcom;
          llist->pipe=vlist->pipe;
          free(vlist);
          return 0;
        }
        llist=llist->next;
      }
      llist->next=vlist;
    } else {
      config.external=vlist;
    }
    return 0;
  }
  return 1;
}

const char *config_lookupext(const char *fn,int mode) {
  struct externaltype *vlist;

  vlist=config.external;
  while(vlist!=NULL) {
    if(!fnmatch(vlist->fspec,fn,0)) {
      switch(mode) {
      case LOOKUP_NONX:
        if(vlist->com[0]) return vlist->com;
        break;
      case LOOKUP_X:
        if(!strcmp(vlist->xcom,".")) {
          if(vlist->com[0]) return vlist->com;
          break;
        }
        if(vlist->xcom[0]) return vlist->xcom;
        break;
      case LOOKUP_PIPE:
        if(vlist->pipe[0]) return vlist->pipe;
        break;
      }
    }
    vlist=vlist->next;
  }
  return NULL;
}

void config_fileerror(const char *buf,const char *error) {
  if(!linenum) crapout("config_fileerror","linenum==0");
  if((!buf)||(!error)) crapout("config_fileerror","called with null parameters");
  if(buf) if(strchr(buf,'\n')) *strchr(buf,'\n')=0;
  printf("---------------------------------------------\n\n");
  printf("error parsing ");
  if(curfile) {
    if(linenum<0) crapout("config_fileerror","linenum<0");
    printf("config file \"%s\", line %d",curfile,linenum);
  } else {
    if(linenum>0) crapout("config_fileerror","linenum>0");
    printf("command line");
  }
  printf("\noption \"%s\"",buf);
  printf("\n  %s\n\n",error);
  if(!curfile) printf("type \"lxt -h\" for help on command line options.\n\n");
  exit(1);
}

void config_cmdhelp(void) {
  int m,n;
  printf("\nmisc command line options:\n");
  printf("  -c <config file>   specify a config file\n");
  printf("  -d                 display config info and quit\n");
  printf("  -h                 this help\n");
  printf("  -h <item>          give help on --item option\n");
  printf("  -l                 log to current dir\n");
  printf("boolean options:  (<n> is either 0 or 1)\n");
  for(n=0;boolopts[n].text;n++) printf("  --%s:<n>\n",boolopts[n].text);
  printf("enum options:  (choose one item from each list)\n");
  for(n=0;enumopts[n].text;n++) {
    printf("  --%s:{",enumopts[n].text);
    for(m=0;enumopts[n].vals[m];m++) printf(m?",%s":"%s",enumopts[n].vals[m]);
    printf("}\n");
  }
  printf("string options:  (<s> is a string)\n");
  for(n=0;stropts [n].text;n++) printf("  --%s:<s>\n",stropts [n].text);
  printf("other options:\n");
  printf("  --external:<ext>:<command>:<X command, optional>:<pipe command, optional>\n\n");
}

void config_opthelp(const char *opt) {
  const char *help;
  int n;
  help=NULL;
  for(n=0;boolopts[n].text;n++)
    if(!strcmp(boolopts[n].text,opt)) help=boolopts[n].help;
  for(n=0;enumopts[n].text;n++)
    if(!strcmp(enumopts[n].text,opt)) help=enumopts[n].help;
  for(n=0;stropts[n].text;n++)
    if(!strcmp(stropts[n].text,opt)) help=stropts[n].help;
  if(!strcmp(opt,"external")) help=CHELP_EXTERNAL;
  if(!help) {
    printf("\noption \"--%s\" not found!\n\n",opt);
    return;
  }
  printf("\nhelp for option \"%s\":\n\n",opt);
  printf("%s",help);
  printf("\n\n");
}

void print_config(void) {
  struct externaltype *p;
  int n;

  printf("\n");
  for(n=0;boolopts[n].text;n++)
    printf("%s:%d\n",boolopts[n].text,*boolopts[n].bool);
  printf("\n");
  for(n=0;enumopts[n].text;n++)
    printf("%s:%s\n",enumopts[n].text,enumopts[n].vals[*enumopts[n].var]);
  printf("\n");
  for(n=0;stropts[n].text;n++)
    printf("%s:<%s>\n",stropts[n].text,*stropts[n].var);
  printf("\n");
  p=config.external;
  while(p!=NULL) {
    printf("external:<%s>:<%s>:<%s>:<%s>\n",p->fspec,p->com,p->xcom,p->pipe);
    p=p->next;
  }
  printf("\n");
}

int config_verifyformat(const char *fmt,const char *opts) {
  int pin;
  for(pin=0;fmt[pin];pin++) {
    if(fmt[pin]=='%') {
      pin++;
      if(fmt[pin]=='%') continue;
      if(!fmt[pin]) return 0;
      if(!strchr(opts,fmt[pin])) return 0;
    }
  }
  return 1;
}

void config_readopt(const char *fullbuf,const char *str,char *outbuf,const char **nextopt) {
  int p,q;
  for(p=q=0;;p++) {
    if(q==(MAXLEN+1)) {
      crapout("config_readopt","option too long");
    }
    switch(str[p]) {
      case 0:
      case '\n':
        outbuf[q]=0;
        *nextopt=NULL;
        return;
      case ':':
        outbuf[q]=0;
        *nextopt=str+p+1;
        return;
      case '\\':
        p++;
        switch(str[p]) {
          case '\\':
          case ':':
            outbuf[q++]=str[p];
            break;
          default:
            config_fileerror(fullbuf,"invalid escape sequence");
        }
        break;
      default:
        outbuf[q++]=str[p];
        break;
    }
  }
}

/********************** runtime config **********************/

#define CT_NONE -1
#define CT_BOOL 0
#define CT_ENUM 1
#define CT_STR  2
#define CT_EXTERN 3
#define CT_RESET 4
#define CT_SAVE 5
#define CT_SAVEDEF 6

extern WINDOW *gui_stbar;
extern WINDOW *gui_msbar;
extern WINDOW *gui_bigun;
extern int maxx;
extern int keep_msbar;
extern int myuid;
extern int cancelled;
extern int gui_oldtime;
extern int currentnode,currentdir;
extern int dispmode;

int config_scroll;               /* how far we have scrolled */
int tabstop;                     /* how far to shift over the options */
int curfield,cfpos,ctype,cidx;   /* stats of cursor position */
int pfonscr,nfonscr;             /* prev/next fields on screen? */
int want_top;                    /* do we want to grab the first field on
                                    the screen? */
int conf_eof;                    /* are we at the end of the text? */
int eof_row;                     /* the last row of everything */
int eof_field;                   /* the last field of everything */
int changed;                     /* have any options been changed? */
int confmode;                    /* what mode to return to when done */
struct externaltype *curextern;  /* the external the cursor is on (if any) */
int extern_field;                /* field number of first external */

struct {
  int numbool; int *bools;
  int numenum; int *enums;
  int numstr;  char **strs;
  struct externaltype *exts;
} tmpsaved;

#define dorow(x) do{if((row>0)&&(row<gui_bigun->_maxy)) {x}}while(0)

void doprinttext(int *row,char *text) {
  char *mytext,*p,*q;

  mytext=malloc(strlen(text)+1);
  if(!mytext) outofmem("doprinttext");
  strcpy(mytext,text);

  for(p=mytext;p;) {
    q=strchr(p,'\n');
    if(q) *q=0;
    if((*row>0)&&(*row<gui_bigun->_maxy))
      mvwprintw(gui_bigun,*row,0,"%s",p);
    (*row)++;
    if(q) p=q+1; else p=NULL;
  }
  (*row)++;
}

void config_dispconfig(void) {
  int n,row;
  int fnum;
  struct externaltype *p;
  char buf[MAXLEN+1];
  char buf1[MAXLEN+1];
  char buf2[MAXLEN+1];
  char buf3[MAXLEN+1];
  char buf4[MAXLEN+1];

  werase(gui_bigun);
  wmove(gui_bigun,0,0);
  whline(gui_bigun,A_BOLD|COLOR_PAIR(C_MENU)|ACS_HLINE,maxx);
  wmove(gui_bigun,gui_bigun->_maxy,0);
  whline(gui_bigun,A_BOLD|COLOR_PAIR(C_MENU)|ACS_HLINE,maxx);
  werase(gui_stbar);
  mvwprintw(gui_stbar,0,0," linuXtree version %s",VERSION);
  wnoutrefresh(gui_stbar);
  gui_oldtime=0;
  gui_drawtime();

  fnum=0;
  if(want_top) curfield=-99999;
  cfpos=-99999;
  ctype=CT_NONE;
  cidx=-1;
  pfonscr=nfonscr=0;
  row=1-config_scroll;
#ifdef __DATE__
  sprintf(buf,CHELP_EDITHEADER,"linuXtree "VERSION", compiled on "__DATE__);
#else
  sprintf(buf,CHELP_EDITHEADER,"linuXtree "VERSION);
#endif
  doprinttext(&row,buf);

  #define dofield1(ct,ci) do{ \
    eof_field=fnum;\
    if(want_top&&(row==0)) pfonscr=2;\
    if(want_top) dorow(curfield=fnum; want_top=0;);\
    if(fnum==(curfield-1)) {\
      if((row>0)&&(row<gui_bigun->_maxy)) pfonscr=1; \
      if(row==0) pfonscr=2;\
    }\
    if(fnum==(curfield+1)) {\
      if((row>0)&&(row<gui_bigun->_maxy)) nfonscr=1; \
      if(row==gui_bigun->_maxy) nfonscr=2;\
    }\
    if(fnum==curfield) dorow(cfpos=row; ctype=ct; cidx=ci;); \
    if(fnum==curfield) wbkgdset(gui_bigun,COLOR_PAIR(C_HILITE)); \
  }while(0)

  #define dofield2 do{ \
    if(fnum==curfield) wbkgdset(gui_bigun,A_BOLD|COLOR_PAIR(C_NORMAL)); \
    fnum++; \
  }while(0)

  #define printoneopt(opt_text,opt_val,opt_help,opt_type,opt_fmt) do{ \
    for(n=0;opt_text;n++) { \
      doprinttext(&row,opt_help); \
      dorow(mvwprintw(gui_bigun,row,0,"   %s",opt_text);); \
      dofield1(opt_type,n); \
      dorow(mvwprintw(gui_bigun,row,tabstop,opt_fmt,opt_val);); \
      dofield2; \
      row+=2; \
    } \
  }while(0)

  printoneopt(boolopts[n].text,*boolopts[n].bool,boolopts[n].help,CT_BOOL,"%-4d");
  printoneopt(enumopts[n].text,enumopts[n].vals[*enumopts[n].var],enumopts[n].help,CT_ENUM,"%-8s");
  printoneopt(stropts[n].text,gui_printable_quoted(maxx-tabstop-1,*stropts[n].var),stropts[n].help,CT_STR,"%s");

  curextern=NULL;
  extern_field=fnum;
  doprinttext(&row,CHELP_EXTERNAL);
  if(config.external) for(p=config.external,n=0;p;p=p->next,n++) {
    dorow(mvwprintw(gui_bigun,row,0,"   external"););
    dofield1(CT_EXTERN,n);
    if(fnum==curfield) curextern=p;
    snprintf(buf,MAXLEN+1,"%s:%s:%s:%s",config_slashquote(buf1,p->fspec),
                                        config_slashquote(buf2,p->com),
                                        config_slashquote(buf3,p->xcom),
                                        config_slashquote(buf4,p->pipe));
    dorow(mvwprintw(gui_bigun,row,tabstop,"%s",gui_printable(maxx-tabstop-1,buf)););
    dofield2;
    row++;
  } else {
    dorow(mvwprintw(gui_bigun,row,0,"   external"););
    dofield1(CT_EXTERN,0);
    dorow(mvwprintw(gui_bigun,row,tabstop,"<none>"););
    dofield2;
    row++;
  }

  row++;
  dofield1(CT_RESET,0);
  dorow(mvwprintw(gui_bigun,row,0,"   Undo changes");); row++;
  dofield2;
  dofield1(CT_SAVE,0);
  dorow(mvwprintw(gui_bigun,row,0,"   Save configuration as default");); row++;
  dofield2;
  dofield1(CT_SAVEDEF,0);
  dorow(mvwprintw(gui_bigun,row,0,"   Save configuration as system default");); row++;
  dofield2;

  row++;
  eof_row=row;
  dorow(wbkgdset(gui_bigun,A_BOLD|COLOR_PAIR(C_VIEWEOF));\
        mvwprintw(gui_bigun,row,0,"<EOF>");\
        wbkgdset(gui_bigun,A_BOLD|COLOR_PAIR(C_NORMAL)););
  conf_eof=0;
  dorow(conf_eof=1;);
  if(cfpos>-99999) mvwprintw(gui_bigun,cfpos,0,"**>");
  wnoutrefresh(gui_bigun);
  if(!keep_msbar) switch(ctype) {
  case CT_BOOL:
    gui_printmsg("Option \"%s\" (boolean), press space to toggle",boolopts[cidx].text);
    break;
  case CT_ENUM:
    gui_printmsg("Option \"%s\" (enum), press space to toggle",enumopts[cidx].text);
    break;
  case CT_STR:
    gui_printmsg("Option \"%s\" (string), press enter to edit",stropts[cidx].text);
    break;
  case CT_EXTERN:
    if(config.external)
      gui_printmsg("Option \"external\", enter: edit, u:swap up, d:swap down, n:new item, x:delete");
    else
      gui_printmsg("Option \"external\", press 'n' to add a new item");
    break;
  case CT_RESET:
    gui_printmsg("Press enter to undo changes");
    break;
  case CT_SAVE:
    gui_printmsg("Press enter to save configuration as default");
    break;
  case CT_SAVEDEF:
    gui_printmsg("Press enter to save configuration as system default");
    break;
  case CT_NONE:
    gui_printmsg("Press arrows to scroll");
    break;
  default:
    crapout("config_dispconfig","unknown field type");
    break;
  }
  doupdate();
/*
  if((cfpos-gui_bigun->_maxy+2)>config_scroll) {
    config_scroll=cfpos-gui_bigun->_maxy+2;
    config_dispconfig();
  }
*/
}

int oldlinkdirs;
void config_initruntimeconfig(void) {
  oldlinkdirs=config.followlinkdirs;
  config_tmpsave();
  config_scroll=0;
  curfield=0;
  tabstop=20;
  changed=0;
  want_top=0;
}

void config_exitconfig(void) {
  int a,b,new,patiencedelay=0;
  int popout=0;

  config_freetmpsaved();

  gui_initdispmode(confmode);
  gui_drawmode();

  if(oldlinkdirs!=config.followlinkdirs) {
    gui_startpatience("rebuilding tree for followlinkdirs...");
    a=data_downdir(0);
    do {
      if(!(patiencedelay++%200)) gui_updatepatience();
      b=data_nextnode(a);
      if(b) {
        a=b;
        b=data_downdir(a);
        if(b) {a=b; continue;}
      } else a=data_updir(a);
      data_islnk(a);
      if(data_islnk(a)) {
        new=(config.followlinkdirs&&S_ISDIR(datanode(a)->lpinfo.st_mode))?1:0;
        if(data_isdir(a)!=new) {
          data_unstatlog(a);
          if(data_isdir(a)) {
            data_unlogdir(a);
            data_guardcurrentnode(a);
            datanode(a)->isdir=0;
            data_insertintolist(a);
          } else {
            if(data_guardcurrentnode(a)==2) popout=1;
            datanode(a)->isdir=1;
            data_deletefromlist(a);
          }
          data_statlog(a);
        }
      }
    } while(a);
    while(!data_isdir(currentdir)) currentdir=data_updir(currentdir);
    if(popout) {
      currentnode=currentdir;
      gui_initdispmode(DISP_TREE);
    }
    gui_synctocurrentnode_bar();
    gui_endpatience();
  }
}

void config_asknewextern(void) {
  char buf[MAXLEN+1];
  char *p1,*p2,*p3,*p4;

  if(!curextern) return;

  if(comm_prompt(buf,MAXLEN+1,curextern->fspec,1,"Filespec: ",
                 FM_TABCOMPLETE,PV_FSPEC)||!buf[0]) {
    gui_mildbeep("Aborted!"); return;
  }
  p1=malloc(strlen(buf)+1);
  if(!p1) outofmem("config_runtimeconfig");
  strcpy(p1,buf);

  if(comm_prompt(buf,MAXLEN+1,curextern->com,0,"Command: ",
                 FM_TABCOMPLETE,PV_CMDLINE)) {
    free(p1);
    gui_mildbeep("Aborted!"); return;
  }
  if(!config_verifyformat(buf,"fdp")) {
    free(p1);
    gui_mildbeep("Error: invalid format string");
    return;
  }
  p2=malloc(strlen(buf)+1);
  if(!p2) outofmem("config_runtimeconfig");
  strcpy(p2,buf);

  if(comm_prompt(buf,MAXLEN+1,curextern->xcom,0,"X command: ",
                 FM_TABCOMPLETE,PV_CMDLINE)) {
    free(p1);
    free(p2);
    gui_mildbeep("Aborted!"); return;
  }
  if(!config_verifyformat(buf,"fdp")) {
    free(p1);
    free(p2);
    gui_mildbeep("Error: invalid format string");
    return;
  }
  p3=malloc(strlen(buf)+1);
  if(!p3) outofmem("config_runtimeconfig");
  strcpy(p3,buf);

  if(comm_prompt(buf,MAXLEN+1,curextern->pipe,0,"Pipe command: ",
                 FM_TABCOMPLETE,PV_CMDLINE)) {
    free(p1);
    free(p2);
    free(p3);
    gui_mildbeep("Aborted!"); return;
  }
  if(!config_verifyformat(buf,"fdp")) {
    free(p1);
    free(p2);
    free(p3);
    gui_mildbeep("Error: invalid format string");
    return;
  }
  p4=malloc(strlen(buf)+1);
  if(!p4) outofmem("config_runtimeconfig");
  strcpy(p4,buf);

  free(curextern->fspec);
  free(curextern->com);
  free(curextern->xcom);
  free(curextern->pipe);
  curextern->fspec=p1;
  curextern->com=p2;
  curextern->xcom=p3;
  curextern->pipe=p4;
  changed=1;
}

void config_runtimeconfig(void) {
  int c;
  char buf[MAXLEN+1];
  struct externaltype *pext,*ppext;

  confmode=dispmode;
  gui_initdispmode(DISP_CONF);
  config_initruntimeconfig();

  timeout(100);
  for(;;) {
    config_dispconfig();

    if(cancelled) gui_mildbeep("Aborted!");
    cancelled=0;
    keep_msbar=0;
    while((c=getch())==ERR) gui_drawtime();
    /* next line clears most of key buffer, but keeps the first key */
    if(config.clearkeybuffer) is_cancel();

    if((c=='\r')||(c=='\n')) c=KEY_ENTER;
    KEYLOOP:
    switch(c) {
      case 'l'-'a'+1:
        wclear(gui_bigun); wnoutrefresh(gui_bigun);
        wclear(gui_stbar); wnoutrefresh(gui_stbar);
        wclear(gui_msbar); wnoutrefresh(gui_msbar);
        doupdate();
        refresh();
        break;
      case 'q':
        config_exitconfig();
        return;
      case KEY_ENTER:
      case ' ':
        switch(ctype) {
        case CT_BOOL:
          changed=1;
          (*boolopts[cidx].bool)=(*boolopts[cidx].bool)?0:1;
          break;
        case CT_ENUM:
          changed=1;
          (*enumopts[cidx].var)++;
          if(!enumopts[cidx].vals[*enumopts[cidx].var]) *enumopts[cidx].var=0;
          break;
        case CT_STR:
          gui_printmsg("Type new value into field, or ctrl-G to cancel");
          keep_msbar=0;
          strcpy(buf,*stropts[cidx].var);
          if(readln_read(gui_bigun,cfpos,tabstop,maxx-tabstop-1,MAXLEN+1,buf,
                         FM_TABCOMPLETE,PV_CMDLINE)) {
            gui_mildbeep("Aborted!");
            break;
          }
          if(config_verifyformat(buf,stropts[cidx].opts)) {
            changed=1;
            free(*stropts[cidx].var);
            (*stropts[cidx].var)=malloc(strlen(buf)+1);
            strcpy(*stropts[cidx].var,buf);
          } else {
            gui_mildbeep("Error: invalid format string");
          }
          break;
        case CT_EXTERN:
          config_asknewextern();
          break;
        case CT_RESET:
          config_tmprestore();
          break;
        case CT_SAVE:
          config_saveconfig(comm_parsepath("~/.lxt-conf"));
          break;
        case CT_SAVEDEF:
          if(myuid) gui_mildbeep("Error: must be root");
          else      config_saveconfig("/etc/lxt-conf");
          break;
        }
        break;
      case KEY_HOME:
        curfield=config_scroll=0;
        break;
      case KEY_END:
        curfield=eof_field;
        if(conf_eof) break;
        config_scroll+=eof_row-gui_bigun->_maxy+1;
        break;
      case KEY_UP:
        if(pfonscr) curfield--;
        if((pfonscr!=1)&&config_scroll) config_scroll--;
        break;
      case KEY_DOWN:
        if(nfonscr) curfield++;
        if((nfonscr!=1)&&!conf_eof) config_scroll++;
        break;
      case KEY_PPAGE:
        if(!config_scroll) break;
        config_scroll-=gui_bigun->_maxy-2;
        if(config_scroll<0) config_scroll=0;
        want_top=1;
        break;
      case KEY_NPAGE:
        if(conf_eof) break;
        config_scroll+=gui_bigun->_maxy-2;
        want_top=1;
        break;
      case 'u':
        if(ctype==CT_EXTERN) {
          if((!config.external)||(!curextern)) break;
          if(config.external==curextern) break;
          if(!config.external->next)
            crapout("config_runtimeconfig","NULL extern->next in bubble command");
          if(config.external->next==curextern) {
            config.external->next=curextern->next;
            curextern->next=config.external;
            config.external=curextern;
            changed=1;
            c=KEY_UP;
            goto KEYLOOP;
          }
          for(ppext=config.external;;ppext=ppext->next) {
            if(!ppext->next->next)
              crapout("config_runtimeconfig","NULL ppext->next->next in bubble command");
            if(ppext->next->next==curextern) {
              pext=ppext->next;
              break;
            }
          }
          ppext->next=curextern;
          pext->next=curextern->next;
          curextern->next=pext;
          changed=1;
          c=KEY_UP;
          goto KEYLOOP;
        }
        break;
      case 'd':
        if(ctype==CT_EXTERN) {
          if((!config.external)||(!curextern)) break;
          if(!curextern->next) break;
          if(config.external==curextern) {
            config.external=curextern->next;
            curextern->next=curextern->next->next;
            config.external->next=curextern;
            changed=1;
            c=KEY_DOWN;
            goto KEYLOOP;
          }
          for(pext=config.external;;pext=pext->next) {
            if(!pext->next)
              crapout("config_runtimeconfig","NULL pext->next in bubble command");
            if(pext->next==curextern) break;
          }
          pext->next=curextern->next;
          ppext=curextern->next->next;
          curextern->next->next=curextern;
          curextern->next=ppext;
          changed=1;
          c=KEY_DOWN;
          goto KEYLOOP;
        }
        break;
      case 'n':
        if(ctype==CT_EXTERN) {
          pext=curextern;
          curextern=malloc(sizeof(struct externaltype));
          if(!curextern) outofmem("config_runtimeconfig");
          curextern->fspec=NULL;
          curextern->com=NULL;
          curextern->xcom=NULL;
          curextern->pipe=NULL;
          config_asknewextern();
          if(!curextern->fspec) {free(curextern); curextern=pext; break;}
          if(config.external) {
            curextern->next=pext->next;
            pext->next=curextern;
            c=KEY_DOWN;
            goto KEYLOOP;
          } else {
            config.external=curextern;
            curextern->next=NULL;
            break;
          }
        }
      case 'x':
        if(ctype==CT_EXTERN) {
          if((!config.external)||(!curextern)) break;
          ppext=curextern->next;
          free(curextern->fspec);
          free(curextern->com);
          free(curextern->xcom);
          free(curextern->pipe);
          free(curextern);
          if(curextern==config.external) {
            config.external=ppext;
            break;
          }
          for(pext=config.external;;pext=pext->next) {
            if(!pext->next)
              crapout("config_runtimeconfig","NULL pext->next in delete command");
            if(pext->next==curextern) break;
          }
          pext->next=ppext;
          if(!ppext) {
            c=KEY_UP;
            goto KEYLOOP;
          }
          break;
        }
        break;
      case 's':
        config_saveconfig(comm_parsepath("~/.lxt-conf"));
        break;
      case 'r':
        config_tmprestore();
        break;
    }
  }
}

void config_saveconfig(const char *fn) {
  char name[PATH_MAX+1];
  FILE *f;

  if(comm_prompt(name,PATH_MAX+1,fn,1,"Save configuration as: ",FM_TABCOMPLETE,PV_PATHNAME)) return;
  if(!name[0]) {set_cancel(); return;}
  f=fopen(name,"w");
  if(!f) {gui_mildbeep("Error: open failed (%s)",strerror(errno)); return;}
  config_saveconfig_F(f);
  fclose(f);
  gui_printmsg("Done.");
}

char *config_slashquote(char *buf,char *s) {
  char *p;
  int n=0;
  for(p=s;*p;p++) {
    if((*p=='\\')||(*p==':')) buf[n++]='\\';
    if(n==MAXLEN) break;
    buf[n++]=*p;
    if(n==MAXLEN) break;
  }
  buf[n]=0;
  return buf;
}

void config_fslashquote(char *s,FILE *f) {
  char *p;
  for(p=s;*p;p++) {
    if((*p=='\\')||(*p==':')) fputc('\\',f);
    fputc(*p,f);
  }
}

void config_saveconfig_F(FILE *f) {
  struct externaltype *p;
  int n;

  fprintf(f,"# Generated by lxt version %s\n",VERSION);
  fprintf(f,"%s\n",CHELP_CONFHEADER);
  for(n=0;boolopts[n].text;n++)
    fprintf(f,"%s\n\n%s:%d\n\n",boolopts[n].help,boolopts[n].text,
            *boolopts[n].bool);
  for(n=0;enumopts[n].text;n++)
    fprintf(f,"%s\n\n%s:%s\n\n",enumopts[n].help,enumopts[n].text,
            enumopts[n].vals[*enumopts[n].var]);
  for(n=0;stropts[n].text;n++) {
    fprintf(f,"%s\n\n%s:",stropts[n].help,stropts[n].text);
    config_fslashquote(*stropts[n].var,f);
    fprintf(f,"\n\n");
  }
  fprintf(f,"%s\n\n",CHELP_EXTERNAL);
  for(p=config.external;p;p=p->next) {
    fprintf(f,"external:");
    config_fslashquote(p->fspec,f);
    fputc(':',f);
    config_fslashquote(p->com,f);
    fputc(':',f);
    config_fslashquote(p->xcom,f);
    fputc(':',f);
    config_fslashquote(p->pipe,f);
    fprintf(f,"\n");
  }
}

void config_freeextern(struct externaltype **ext) {
  struct externaltype *vlist,*next;
  for(vlist=*ext;vlist;vlist=next) {
    next=vlist->next;
    free(vlist->fspec);
    free(vlist->com);
    free(vlist->xcom);
    free(vlist->pipe);
    free(vlist);
  }
  *ext=NULL;
}

int config_countextern(struct externaltype *ext) {
  struct externaltype *vlist;
  int n=0;
  for(vlist=ext;vlist;vlist=vlist->next) n++;
  return n;
}

void config_dupextern(struct externaltype **dest,const struct externaltype *src) {
  const struct externaltype *vlist;
  struct externaltype *nlist;
  struct externaltype **pnext;
  pnext=dest;
  for(vlist=src;vlist;vlist=vlist->next) {
    nlist=malloc(sizeof(struct externaltype));
    if(!nlist) outofmem("config_tmpsave");

    nlist->fspec=malloc(strlen(vlist->fspec)+1);
    if(!nlist->fspec) outofmem("config_tmpsave");
    strcpy(nlist->fspec,vlist->fspec);
    nlist->com=malloc(strlen(vlist->com)+1);
    if(!nlist->com) outofmem("config_tmpsave");
    strcpy(nlist->com,vlist->com);
    nlist->xcom=malloc(strlen(vlist->xcom)+1);
    if(!nlist->xcom) outofmem("config_tmpsave");
    strcpy(nlist->xcom,vlist->xcom);
    nlist->pipe=malloc(strlen(vlist->pipe)+1);
    if(!nlist->pipe) outofmem("config_tmpsave");
    strcpy(nlist->pipe,vlist->pipe);

    *pnext=nlist;
    pnext=&(nlist->next);
  }
  *pnext=NULL;
}

void config_freetmpsaved(void) {
  int n;
  free(tmpsaved.bools);
  free(tmpsaved.enums);
  for(n=0;n<tmpsaved.numstr;n++) free(tmpsaved.strs[n]);
  free(tmpsaved.strs);
  config_freeextern(&tmpsaved.exts);
}

void config_tmpsave(void) {
  int n;

  for(n=0;boolopts[n].text;n++);
  tmpsaved.numbool=n;
  tmpsaved.bools=malloc(n*sizeof(int));
  if(!tmpsaved.bools) outofmem("config_tmpsave");
  for(n=0;boolopts[n].text;n++) tmpsaved.bools[n]=*boolopts[n].bool;

  for(n=0;enumopts[n].text;n++);
  tmpsaved.numenum=n;
  tmpsaved.enums=malloc(n*sizeof(int));
  if(!tmpsaved.enums) outofmem("config_tmpsave");
  for(n=0;enumopts[n].text;n++) tmpsaved.enums[n]=*enumopts[n].var;

  for(n=0;stropts[n].text;n++);
  tmpsaved.numstr=n;
  tmpsaved.strs=malloc(n*sizeof(char *));
  if(!tmpsaved.strs) outofmem("config_tmpsave");
  for(n=0;stropts[n].text;n++) {
    tmpsaved.strs[n]=malloc(strlen(*stropts[n].var)+1);
    if(!tmpsaved.strs[n]) outofmem("config_tmpsave");
    strcpy(tmpsaved.strs[n],*stropts[n].var);
  }
  config_dupextern(&tmpsaved.exts,config.external);
}

void config_tmprestore(void) {
  int n;
  int old,new;

  for(n=0;n<tmpsaved.numbool;n++) *boolopts[n].bool=tmpsaved.bools[n];
  for(n=0;n<tmpsaved.numenum;n++) *enumopts[n].var=tmpsaved.enums[n];
  for(n=0;n<tmpsaved.numstr;n++) {
    free(*stropts[n].var);
    *stropts[n].var=malloc(strlen(tmpsaved.strs[n])+1);
    if(!*stropts[n].var) outofmem("config_tmprestore");
    strcpy(*stropts[n].var,tmpsaved.strs[n]);
  }
  old=MAX(1,config_countextern(config.external));
  config_freeextern(&config.external);
  config_dupextern(&config.external,tmpsaved.exts);
  new=MAX(1,config_countextern(config.external));
  if(curfield>extern_field) {
    if(ctype!=CT_EXTERN) {
      config_scroll+=new-old;
      curfield+=new-old;
    } else if(curfield>extern_field+new-1) {
      config_scroll-=curfield-(extern_field+new-1);
      curfield=extern_field+new-1;
    }
  }
}
