/* linuXtree copyright (c) 1998 Dan Stahlke under the GPL
   See file 'COPYING' for details */

#include "all.h"

extern struct configtype config;

int windowsup;      /* curses started up? */
int dispmode;       /* tree, files, or view */
int dispfilemode;   /* dir,showall,branch, or tagged in file view */
int dispform;       /* what info is shown in file mode */
int sortcriteria;   /* how files are sorted */
int helpmode;       /* what we are giving help on when in DISP_HELP */
int toptreedir;     /* which node is at top of screen in tree mode */
int cursortreeloc;  /* position of cursor in tree mode */
int filelist;       /* first file in file list */
int topfile;        /* which node is at top of screen in file mode */
int cursorfileloc;  /* position of cursor in file mode */
int currentnode;    /* node cursor is highliting */
int currentdir;     /* dir we are in or pointing to */
int gui_wantdraw;   /* do we want to update the screen? */
int gui_wantresize; /* do we want to resize the screen? */
int maxx,maxy;      /* size of screen */
int updateall;      /* better update the whole screen, not just top or bottom lines */
int numscroll;      /* how far we scrolled, + is down, - is up */
char *toptreelines; /* treelines for node at top */
int topnextnode;    /* does toptreedir have a nextnode? */
extern int viewmode;       /* current mode for viewing */
extern int viewmask;       /* mask unprintable chars? */
int keep_msbar;     /* don't draw over gui_msbar if >0 */
int debugmode;      /* print debug info instead of stats */
char *bottommsg;    /* the info message displayed on the bottom */
char filespec[PATH_MAX+1]; /* the current filespec */
int invfilespec;    /* is the filespec inverted? */
WINDOW *gui_bigun;  /* the big tree / full screen window */
WINDOW *gui_teeny;  /* shows files on bottom in tree mode */
WINDOW *gui_stats;  /* the stats that are on the right */
WINDOW *gui_stbar;  /* path and time, top line */
WINDOW *gui_msbar;  /* messages, very bottom */
int patience_up=0;  /* is patience display invoked */
int patience_count; /* what is the current position of the patience wheel */

time_t gui_oldtime=0;

extern char **environ;
char **saveenv;
int numenviron;

extern int byteslogged,fileslogged,dirslogged,
           bytestagged,filestagged,dirstagged,
           byteshown,fileshown;

void gui_init(void) {
  char curdir[PATH_MAX+1];

  /* this hack to get around ncurses bug which corrupts environ */
  gui_saveenviron();
  initscr();      /* init ncurses */
  gui_restoreenviron();

  signal(SIGWINCH,gui_resize);

  start_color();                /* enable colors */
  init_pair(C_NORMAL,COLOR_WHITE,COLOR_BLUE);    /* normal color */
  init_pair(C_HILITE,COLOR_BLACK,COLOR_WHITE);   /* hilighted items */
  init_pair(C_MENU  ,COLOR_CYAN ,COLOR_BLUE);    /* border and stuff */
  init_pair(C_WARN  ,COLOR_RED  ,COLOR_WHITE);   /* file suid/guid */
  init_pair(C_TAGGED,COLOR_RED  ,COLOR_BLUE);    /* tagged file */
  init_pair(C_HITAG ,COLOR_RED  ,COLOR_WHITE);   /* hilighted tagged file */
  init_pair(C_BOLD  ,COLOR_YELLOW,COLOR_BLUE);   /* bold text */
  init_pair(C_UNDERLINE,COLOR_GREEN,COLOR_BLUE);    /* underlined text */
  init_pair(C_VIEWEOF,COLOR_RED ,COLOR_BLUE);    /* EOF marker in view */

  dispmode=DISP_TREE;
  dispfilemode=D_FILE_NONE;
  dispform=D_FORM_LONG;
  sortcriteria=SORT_ALPHA;
  gui_msbar=gui_stbar=gui_bigun=gui_teeny=gui_stats=NULL;
  cursortreeloc=0;
  currentnode=toptreedir=0;
  topnextnode=data_nextnode(toptreedir);
  toptreelines=data_resettreelines(NULL);
  keep_msbar=0;
  debugmode=0;
  bottommsg=NULL;
  gui_wantresize=0;
  strcpy(filespec,"*");
  invfilespec=0;

  if(config.logtocurdir) {
    gui_startcurses();
    data_logdir(0);
#ifdef HAVE_GETCWD
    data_logtodir(getcwd(curdir,PATH_MAX+1));
#endif
  } else {
    data_logdir(0);
    gui_startcurses();
  }
}

void gui_saveenviron(void) {
  for(numenviron=0;environ[numenviron]!=NULL;numenviron++);
  numenviron++;
  saveenv=malloc(sizeof(*saveenv)*numenviron);
  if(!saveenv) outofmem("gui_saveenviron");
  memcpy(saveenv,environ,sizeof(*saveenv)*numenviron);
}

void gui_restoreenviron(void) {
  memcpy(environ,saveenv,sizeof(*saveenv)*numenviron);
}

void gui_startcurses(void) {
  if(windowsup) return;
  windowsup=1;
  doupdate();
  meta(stdscr,FALSE); /* send meta chars in ESC form, not 8-bit form */
  raw();              /* we handle ^C, ^Z, etc. ourselves */
  noecho();           /* don't echo chars */
  timeout(0);         /* don't wait for keys */
  leaveok(stdscr,TRUE);         /* turn off cursor and don't worry about it */
  intrflush(stdscr,FALSE);      /* don't make interrupts update screen */
  keypad(stdscr,TRUE);          /* processes function keys */
  typeahead(-1);                /* without this, scrolling can be very ugly */
  refresh();

  maxx=stdscr->_maxx+1;
  maxy=stdscr->_maxy+1;
  if((maxx<75)||(maxy<15)) crapout_nobug("cannot run the program in a window smaller than 75x15");

  /* the one window that is always allocated */
  gui_msbar=newwin(1,maxx,maxy-1,0);
  if(!gui_msbar) crapout("gui_init","cannot allocate window");
  wbkgd(gui_msbar,A_BOLD|COLOR_PAIR(C_NORMAL));

  gui_initdispmode(dispmode);
  gui_drawmode();
}

void gui_stopcurses(void) {
  if(!windowsup) return;
  delwin(gui_msbar); gui_msbar=NULL;
  endwin();
  windowsup=0;
}

RETSIGTYPE gui_resize(int sig) {
  /* Next five lines are swiped from XTC program.  They don't seem to help
     though, when the window is shrunk in rxvt it seems to crash before this
     signal handler is even called. */

  struct winsize size;
  if(ioctl(fileno(stdout),TIOCGWINSZ,&size)==0) {
    resizeterm(size.ws_row,size.ws_col);
    wrefresh(curscr);
  }

  gui_wantresize=1;
  signal(SIGWINCH,gui_resize);
}

void gui_refreshscreen(void) {
  if(gui_bigun!=NULL) {wclear(gui_bigun); wnoutrefresh(gui_bigun);}
  if(gui_teeny!=NULL) {wclear(gui_teeny); wnoutrefresh(gui_teeny);}
  if(gui_stats!=NULL) {wclear(gui_stats); wnoutrefresh(gui_stats);}
  if(gui_stbar!=NULL) {wclear(gui_stbar); wnoutrefresh(gui_stbar);}
  if(gui_msbar!=NULL) {wclear(gui_msbar); wnoutrefresh(gui_msbar);}
  doupdate();
  refresh();

  gui_setwantdrawall();
  gui_oldtime=0;
  gui_drawmode();
}

void gui_initdispmode(int mode) {
  char *p;
  if(gui_bigun!=NULL) { delwin(gui_bigun); gui_bigun=NULL; }
  if(gui_teeny!=NULL) { delwin(gui_teeny); gui_teeny=NULL; }
  if(gui_stats!=NULL) { delwin(gui_stats); gui_stats=NULL; }
  if(gui_stbar!=NULL) { delwin(gui_stbar); gui_stbar=NULL; }

  gui_stbar=newwin(1,maxx,0,0);
  if(gui_stbar==NULL) crapout("gui_initdispmode","cannot allocate windows");
  wbkgd(gui_stbar,A_BOLD|COLOR_PAIR(C_NORMAL));
  werase(gui_stbar);

  p=malloc(maxx+2);
  if(!p) outofmem("gui_initdispmode");
  if(bottommsg) {
    strncpy(p,bottommsg,MIN(strlen(bottommsg)+1,maxx+1));
    free(bottommsg);
  } else {
    p[0]=0;
  }
  bottommsg=p;

  switch(mode) {
  case DISP_TREE:
    gui_bigun=newwin(maxy-7,maxx-20,1,0);
    gui_teeny=newwin(6,maxx-20,maxy-7,0);
    gui_stats=newwin(maxy-2,21,1,maxx-21);
    if((gui_bigun==NULL)||
       (gui_teeny==NULL)||
       (gui_stats==NULL)) {
      crapout("gui_initdispmode","cannot allocate windows");
    }
    werase(gui_bigun);
    werase(gui_teeny);
    werase(gui_stats);
    wbkgd(gui_bigun,A_BOLD|COLOR_PAIR(C_NORMAL));
    wbkgd(gui_teeny,A_BOLD|COLOR_PAIR(C_NORMAL));
    wbkgd(gui_stats,A_BOLD|COLOR_PAIR(C_NORMAL));
    break;
  case DISP_FILE:
    gui_bigun=newwin(maxy-2,maxx-20,1,0);
    gui_stats=newwin(maxy-2,21,1,maxx-21);
    if((gui_bigun==NULL)||
       (gui_stats==NULL)) {
      crapout("gui_initdispmode","cannot allocate windows");
    }
    scrollok(gui_bigun,TRUE);
    wsetscrreg(gui_bigun,1,maxy-2);
    werase(gui_bigun);
    werase(gui_stats);
    wbkgd(gui_bigun,A_BOLD|COLOR_PAIR(C_NORMAL));
    wbkgd(gui_stats,A_BOLD|COLOR_PAIR(C_NORMAL));
    break;
  case DISP_VIEW:
    gui_bigun=newwin(maxy-2,maxx,1,0);
    if(gui_bigun==NULL) {
      crapout("gui_initdispmode","cannot allocate windows");
    }
    scrollok(gui_bigun,FALSE);
    werase(gui_bigun);
    wbkgd(gui_bigun,A_BOLD|COLOR_PAIR(C_NORMAL));
    break;
  case DISP_HELP:
    gui_bigun=newwin(maxy-2,maxx,1,0);
    if(gui_bigun==NULL) {
      crapout("gui_initdispmode","cannot allocate windows");
    }
    scrollok(gui_bigun,FALSE);
    werase(gui_bigun);
    wbkgd(gui_bigun,A_BOLD|COLOR_PAIR(C_NORMAL));
    break;
  case DISP_CONF:
    gui_bigun=newwin(maxy-2,maxx,1,0);
    if(gui_bigun==NULL) {
      crapout("gui_initdispmode","cannot allocate windows");
    }
    scrollok(gui_bigun,FALSE);
    werase(gui_bigun);
    wbkgd(gui_bigun,A_BOLD|COLOR_PAIR(C_NORMAL));
    break;
  }
  dispmode=mode;
  gui_oldtime=0;
  numscroll=0;
  gui_setwantdrawall();
}

void gui_drawmode(void) {
  gui_drawtime();
  if(!gui_wantdraw) return;
  gui_wantdraw=0;
  switch(dispmode) {
  case DISP_TREE:
    gui_drawtree();
    gui_drawstatus();
    break;
  case DISP_FILE:
    gui_drawfile();
    gui_drawstatus();
    break;
  case DISP_VIEW:
    view_drawview();
    break;
  case DISP_HELP:
    view_drawview();
    break;
  case DISP_CONF:
    config_dispconfig();
    break;
  default:
    crapout("gui_drawmode","called in unknown dispmode");
  }
  doupdate();
}

void gui_drawonename(int num,int y,int h,int shift,char *treelines) {
  int n;
  chtype c;
#define clipaddch(a,b,c,d) if((c>2)&&(c<maxx-21)) mvwaddch(a,b,c,d);
  mvwaddch(gui_bigun,1+y,2,data_downdir(num)?'+':' ');
  /* translate '|','>','L' back into line-drawing chars */
  
  for(n=0;n<data_depth(num);n++) {
    switch((c=treelines[n])) {
      case ' ': c=' ';          break;
      case '|': c=ACS_VLINE;    break;
      case '>': c=ACS_LTEE;     break;
      case 'L': c=ACS_LLCORNER; break;
      case   0: c='@';		break;
    }
    clipaddch(gui_bigun,1+y,3+3*n+shift,c|COLOR_PAIR(C_NORMAL)|A_BOLD);
    clipaddch(gui_bigun,1+y,4+3*n+shift,' ');
    clipaddch(gui_bigun,1+y,5+3*n+shift,' ');
  }
  n*=3;
  if(n) clipaddch(gui_bigun,1+y,1+n+shift,ACS_HLINE|COLOR_PAIR(C_NORMAL)|A_BOLD);
  if(!data_tagged(num)) {
    if(uid_okread(num)) wbkgdset(gui_bigun,COLOR_PAIR(C_NORMAL)|A_BOLD);
    else                wbkgdset(gui_bigun,COLOR_PAIR(C_NORMAL));
    if(h) wbkgdset(gui_bigun,COLOR_PAIR(C_HILITE));
  } else {
    wbkgdset(gui_bigun,COLOR_PAIR(C_TAGGED)|A_BOLD);
    if(h) wbkgdset(gui_bigun,COLOR_PAIR(C_HITAG)|A_BOLD);
  }
  if((gui_bigun->_maxx-(3+n+shift))>0)
    mvwprintw(gui_bigun,1+y,2+n+shift,"%c%-8s ",
              (!n&&data_downdir(num))?'+':' ',
              gui_printable(gui_bigun->_maxx-(3+n+shift),data_name(num)));
  wbkgdset(gui_bigun,A_BOLD|COLOR_PAIR(C_NORMAL));
}

int gui_numfilerows(void) {
  if(dispmode==DISP_FILE) return gui_bigun->_maxy-1;
  if(dispmode==DISP_TREE) return gui_teeny->_maxy-1;
  crapout("gui_numfilerows","invalid dispmode");
  return 0; /* keeps compiler happy */
}

int gui_numfiledisp(void) {
  int mult;
  mult=1;
  if(dispform==D_FORM_TWOCOL) mult=2;
  if(dispform==D_FORM_TRICOL) mult=3;
  if(dispmode==DISP_FILE) return gui_numfilerows()*mult;
  if(dispmode==DISP_TREE) return gui_numfilerows()*mult;
  crapout("gui_numfiledisp","invalid dispmode");
  return 0; /* keeps compiler happy */
}

int gui_numtreedisp(void) {
  if(dispmode==DISP_TREE) return gui_bigun->_maxy-1;
  crapout("gui_numtreedisp","invalid dispmode");
  return 0; /* keeps compiler happy */
}

void gui_drawtree(void) {
  int a,n,barok;
  int shift;
  int *sortlist;
  char *treelines;

  treelines=malloc(strlen(toptreelines)+1);
  if(!treelines) outofmem("gui_drawtree");
  strcpy(treelines,toptreelines);

  a=toptreedir;
  werase(gui_bigun);
  werase(gui_teeny);
  barok=0;
  shift=-data_depth(currentnode)*3+maxx-35;
  if(shift>0) shift=0;
  for(n=0;n<gui_numtreedisp();n++) {
    if(n==cursortreeloc) {
      mvwprintw(gui_bigun,1+n,1,">");

      if(currentnode!=a) crapout("gui_drawtree",
        "getting currentnode out of sync (cn: %d cd: %d a: %d)\n",currentnode,currentdir,a);
      if(currentdir!=a)  crapout("gui_drawtree",
        "getting currentdir out of sync  (cn: %d cd: %d a: %d)\n",currentnode,currentdir,a);
/*
      if((currentnode!=a)||(currentdir!=a)) {
        beep();
        printf("cn: %d cd: %d a: %d\n",currentnode,currentdir,a);
        fflush(stdout);
        currentnode=currentdir=a;
      }
*/
      barok=1;
    }
    gui_drawonename(a,n,n==cursortreeloc,shift,treelines);
    treelines=data_nexttreelines(treelines,a);
    if(!treelines) crapout("gui_drawtree","null treelines");
    a=data_nexttree(a);
    if(a==0) break;
  }
  if(!barok) crapout("gui_drawtree","cursor bar not on something");
  gui_displaycurrentname();

  a=data_downdir(currentnode);
  if(a) {
    if(!data_hasfile(currentnode)) mvwprintw(gui_teeny,1,1," no files.");
    else {
      sortlist=malloc(sizeof(int)*gui_numfiledisp());
      if(!sortlist) outofmem("gui_drawtree");
      data_toplist(currentnode,sortcriteria,gui_numfiledisp(),sortlist);
      for(n=0;n<gui_numfiledisp();n++) {
        a=sortlist[n];
        if(!a) break;
        if(!uid_okread(a)) wbkgdset(gui_teeny,COLOR_PAIR(C_NORMAL));
        if(data_tagged(a)) wbkgdset(gui_teeny,A_BOLD|COLOR_PAIR(C_TAGGED));
        gui_listlong(a,n,gui_teeny->_maxy-1,gui_teeny);
        wbkgdset(gui_teeny,A_BOLD|COLOR_PAIR(C_NORMAL));
      }
      if(!n) mvwprintw(gui_teeny,1,1," no files matching.");
    }
  } else
    mvwprintw(gui_teeny,1,1," dir not logged.");
  wbkgdset(gui_bigun,A_BOLD|COLOR_PAIR(C_MENU));
  wbkgdset(gui_teeny,A_BOLD|COLOR_PAIR(C_MENU));
  wborder(gui_bigun,0,0,0,0,0,ACS_TTEE,ACS_LTEE,ACS_RTEE);
  wborder(gui_teeny,0,0,0,0,ACS_LTEE,ACS_RTEE,0,ACS_BTEE);
  wbkgdset(gui_bigun,A_BOLD|COLOR_PAIR(C_NORMAL));
  wbkgdset(gui_teeny,A_BOLD|COLOR_PAIR(C_NORMAL));
  wnoutrefresh(gui_teeny);
  wnoutrefresh(gui_bigun);
  wnoutrefresh(gui_stbar);
  wnoutrefresh(gui_msbar);
  free(treelines);
}

void gui_drawfile(void) {
  int a,n,barok;

  a=topfile;
  barok=0;
  if(updateall) numscroll=0;
  if(numscroll>(gui_numfiledisp()-2))  {numscroll=0; updateall=1;}
  if(updateall) werase(gui_bigun);
  else if(numscroll) wscrl(gui_bigun,numscroll);
  for(n=0;n<gui_numfiledisp();n++) {
    if((!numscroll &&
	( (n==(cursorfileloc-1)) || (n==(cursorfileloc+1)) )) ||
       (abs(n-cursorfileloc)<=abs(numscroll)) || (updateall)) {
      if(!uid_okread(a)) wbkgdset(gui_bigun,COLOR_PAIR(C_NORMAL));
      if(data_tagged(a)) wbkgdset(gui_bigun,A_BOLD|COLOR_PAIR(C_TAGGED));
      if(n==cursorfileloc) {
        barok=1;
        if(data_tagged(a))
          wbkgdset(gui_bigun,A_BOLD|COLOR_PAIR(C_HITAG));
        else
          wbkgdset(gui_bigun,COLOR_PAIR(C_HILITE));
      }
      gui_listlong(a,n,gui_bigun->_maxy-1,gui_bigun);
      wbkgdset(gui_bigun,A_BOLD|COLOR_PAIR(C_NORMAL));
    }
    if((n==cursorfileloc)&&(currentnode!=a))
      crapout("gui_drawfile","currentnode out of sync (cn: %d a: %d)",currentnode,a);
    a=data_nextshown(a);
    if(!a) break;
  }
  if(!barok) crapout("gui_drawfile","cursor bar not on something");
  gui_displaycurrentname();

  updateall=0;
  numscroll=0;
  wbkgdset(gui_bigun,A_BOLD|COLOR_PAIR(C_MENU));
  wborder(gui_bigun,0,0,0,0,0,ACS_TTEE,0,ACS_BTEE);
  wbkgdset(gui_bigun,A_BOLD|COLOR_PAIR(C_NORMAL));
  wnoutrefresh(gui_bigun);
  wnoutrefresh(gui_msbar);
}

void gui_displaycurrentname(void) {
  char buffer[(PATH_MAX+1)*4];
  char s1[(PATH_MAX+1)*4];
  char s2[(PATH_MAX+1)*4];
  if(data_islnk(currentnode)) {
    strcpy(s2,gui_printable(0,gui_linkname(data_fullname(currentnode))));
    strcpy(s1,gui_printable(maxx-6-MIN(strlen(s2),10),data_name(currentnode)));
    strcpy(s2,gui_printable(MAX(maxx-6-strlen(s1),10),gui_linkname(data_fullname(currentnode))));
    sprintf(buffer,"%s -> %s",s1,s2);
  } else
    sprintf(buffer,"%.*s",maxx-2,gui_printable(maxx-2,data_name(currentnode)));
  if((!keep_msbar)&&(!patience_up)) {
    werase(gui_msbar);
    mvwprintw(gui_msbar,0,0," %-*.*s",maxx-1,maxx-1,buffer);
  }
}

char gui_linknamebuffer[PATH_MAX+1];

const char *gui_linkname(const char *fname) {
  int n;
  n=readlink(fname,gui_linknamebuffer,PATH_MAX);
  if(n==-1)
    sprintf(gui_linknamebuffer,"*error*");
  else
    gui_linknamebuffer[n]=0;
  return gui_linknamebuffer;
}

void gui_drawtime(void) {
  time_t cur_time;
  cur_time=time((time_t *)0);
  if(cur_time!=gui_oldtime) {
    mvwprintw(gui_stbar,0,maxx-24,"%s",ctime(&cur_time));
    wnoutrefresh(gui_stbar);
    doupdate();
  }
  gui_oldtime=cur_time;
}

void gui_startpatience(const char *s) {
  if(!windowsup) return;
  patience_up++;
  if(patience_up>1) return;
  patience_count=0;
  wbkgdset(gui_msbar,A_BOLD|COLOR_PAIR(C_NORMAL));
  werase(gui_msbar);
  mvwprintw(gui_msbar,0,1,"- %s",s);
  wrefresh(gui_msbar);
}

void gui_updatepatience(void) {
  int patience_wheel[4]={'-','\\','|','/'};
  if(!windowsup) return;
  patience_count++;
  patience_count%=4;
  mvwaddch(gui_msbar,0,1,patience_wheel[patience_count]|
                         A_BOLD|COLOR_PAIR(C_NORMAL));
  wrefresh(gui_msbar);
}

void gui_endpatience(void) {
  if(!windowsup) return;
  if(!patience_up) crapout("gui_endpatience","called when patience wheel not up");
  patience_up--;
  if(patience_up) return;
  werase(gui_msbar);
  if(keep_msbar) mvwprintw(gui_msbar,0,1,"%s",bottommsg);
  wrefresh(gui_msbar);
}

void gui_drawstatus(void) {
  char rpath[PATH_MAX+1];
#ifdef HAVE_MY_STATFS
  struct my_statfs_struct statbuf;
#endif
#ifdef HAVE_MY_GETMNTENT
  char fstypebuf[10],mntpoint[PATH_MAX+1];
  FILE *mntf;
  int maxs;
  struct my_mntent *ment;
#endif

  werase(gui_stats);
  wbkgdset(gui_stats,A_BOLD|COLOR_PAIR(C_MENU));
  wborder(gui_stats,0,0,0,0,ACS_TTEE,0,ACS_BTEE,0);
  wbkgdset(gui_stats,A_BOLD|COLOR_PAIR(C_NORMAL));
  if(debugmode) {
    mvwprintw(gui_stats, 1,1,"--- Debug Mode ---");
    mvwprintw(gui_stats, 2,1,"currentnode:% 7d",currentnode);
    mvwprintw(gui_stats, 3,1,"currentdir: % 7d",currentdir);

    if(data_isdir(currentnode)) {
      mvwprintw(gui_stats, 5,1,"cursortreeloc:% 5d",cursortreeloc);
      mvwprintw(gui_stats, 4,1,"toptreedir: % 7d",toptreedir);
      mvwprintw(gui_stats, 6,1,"depth:      % 7d",data_depth(currentnode));
      mvwprintw(gui_stats, 7,1,"prevtree:   % 7d",data_prevtree(currentnode));
      mvwprintw(gui_stats, 8,1,"nexttree:   % 7d",data_nexttree(currentnode));
      mvwprintw(gui_stats,14,1,"filelist:   % 7d",filelist);
    } else {
      mvwprintw(gui_stats, 5,1,"cursorfileloc:% 5d",cursorfileloc);
      mvwprintw(gui_stats, 4,1,"topfile:    % 7d",topfile);
      mvwprintw(gui_stats, 6,1,"filelist:   % 7d",filelist);
      mvwprintw(gui_stats, 7,1,"prevlist:   % 7d",data_prevlist(currentnode));
      mvwprintw(gui_stats, 8,1,"nextlist:   % 7d",data_nextlist(currentnode));
      /*mvwprintw(gui_stats,14,1,"listpos:    % 7d",data_listpos(currentnode));*/
    }

    mvwprintw(gui_stats, 9,1,"isdir       % 7d",data_isdir(currentnode));
    mvwprintw(gui_stats,10,1,"updir:      % 7d",data_updir(currentnode));
    mvwprintw(gui_stats,11,1,"downdir:    % 7d",data_downdir(currentnode));
    mvwprintw(gui_stats,12,1,"prevnode:   % 7d",data_prevnode(currentnode));
    mvwprintw(gui_stats,13,1,"nextnode:   % 7d",data_nextnode(currentnode));
  } else {
    /*mvwprintw(gui_stats, 1,1,"--- lxt %s ---",VERSION);*/
    if(invfilespec) wbkgdset(gui_stats,COLOR_PAIR(C_HILITE));
    mvwprintw(gui_stats, 1,1,"Filespec: %-9.9s",gui_printable(9,filespec));
    if(invfilespec) wbkgdset(gui_stats,A_BOLD|COLOR_PAIR(C_NORMAL));
    switch(dispmode) {
    case DISP_TREE:
      mvwprintw(gui_stats, 2,1,"Logged:");
      mvwprintw(gui_stats, 3,1,"  Files:% 10d",fileslogged);
      if(maxy>14) mvwprintw(gui_stats, 4,1,"  Dirs: % 10d",dirslogged);
      if(maxy>15) mvwprintw(gui_stats, 5,1,"  Bytes:% 10dk",byteslogged/1024);
      break;
    case DISP_FILE:
      switch(dispfilemode) {
      case D_FILE_NORM:
        mvwprintw(gui_stats, 2,1,"(dir mode)"); break;
      case D_FILE_BRANCH:
        mvwprintw(gui_stats, 2,1,"(branch mode)"); break;
      case D_FILE_SHOWALL:
        mvwprintw(gui_stats, 2,1,"(showall mode)"); break;
      case D_FILE_TAGBRANCH:
        mvwprintw(gui_stats, 2,1,"(tagged branch)"); break;
      case D_FILE_TAGSHOWALL:
        mvwprintw(gui_stats, 2,1,"(tagged showall)"); break;
      default: crapout("gui_drawstatus","invalid dispfilemode");
      }
      mvwprintw(gui_stats, 3,1,"Shown:");
      if(maxy>14) mvwprintw(gui_stats, 4,1,"  Files:% 10d",fileshown);
      if(maxy>15) mvwprintw(gui_stats, 5,1,"  Bytes:% 10dk",byteshown/1024);
      break;
    default:
      crapout("gui_drawstatus","showing filestats in wrong mode");
    }
    if(maxy>16) mvwprintw(gui_stats, 6,1,"Tagged:");
    if(maxy>17) mvwprintw(gui_stats, 7,1,"  Files:% 10d",filestagged);
    if(maxy>18) mvwprintw(gui_stats, 8,1,"  Dirs: % 10d",dirstagged);
    if(maxy>19) mvwprintw(gui_stats, 9,1,"  Bytes:% 10dk",bytestagged/1024);

    if(maxy>20) {
      if((!(data_isdir(currentnode)?config.followlinkdirs
                                   :config.followlinks))||
         (!realpath(data_fullname(currentnode),rpath)))
        strcpy(rpath,data_fullname(currentnode));
      if(!data_fs_isdir_nolinks(rpath)) *strrchr(rpath,'/')=0;
#ifdef HAVE_MY_STATFS
      my_statfs(rpath,&statbuf);
      if(maxy>22) mvwprintw(gui_stats,12,1,"% 10dk total",statbuf.totalkbytes);
      if(maxy>23) mvwprintw(gui_stats,13,1,"% 10dk used ",statbuf.usedkbytes);
      if(maxy>24) mvwprintw(gui_stats,14,1,"% 10dk avail",statbuf.availkbytes);
#else
      if(maxy>22) mvwprintw(gui_stats,12,1,"unsupported");
      if(maxy>23) mvwprintw(gui_stats,13,1,"unsupported");
      if(maxy>24) mvwprintw(gui_stats,14,1,"unsupported");
#endif
#ifdef HAVE_MY_GETMNTENT
      strcpy(mntpoint,"???");
      strcpy(fstypebuf,"???");
      mntf=my_setmntent(MNT_MTAB);
      if(mntf) {
        maxs=0;
        if((!(data_isdir(currentnode)?config.followlinkdirs:
                                      config.followlinks))||
           (!realpath(data_fullname(currentnode),rpath)))
          strcpy(rpath,data_fullname(currentnode));
        while((ment=my_getmntent(mntf))) {
          if(strlen(ment->mnt_dir)>maxs)
            if(!strncmp(rpath,ment->mnt_dir,strlen(ment->mnt_dir))) {
              maxs=strlen(ment->mnt_dir);
              strncpy(mntpoint,ment->mnt_fsname,PATH_MAX);
              mntpoint[PATH_MAX]=0;
              sprintf(fstypebuf,"%-5.9s",ment->mnt_type);
            }
        }
        my_endmntent(mntf);
      }
      if(maxy>20) mvwprintw(gui_stats,10,1,"Filesystem: (%s)",fstypebuf);
      if(maxy>21) mvwprintw(gui_stats,11,1,"%- 20s",mntpoint);
#else
      if(maxy>20) mvwprintw(gui_stats,10,1,"unsupported");
      if(maxy>21) mvwprintw(gui_stats,11,1,"unsupported");
#endif
    }
  }
  gui_drawnodestatus(currentnode);
  wnoutrefresh(gui_stats);
}

void gui_drawnodestatus(int node) {
  int startat=maxy-10;
  struct stat st;
  char perm[10]="---------";
  st=data_info(node);
  if(st.st_mode&0x100) perm[0]='r';
  if(st.st_mode&0x080) perm[1]='w';
  if(st.st_mode&0x040) perm[2]='x';
  if(st.st_mode&0x020) perm[3]='r';
  if(st.st_mode&0x010) perm[4]='w';
  if(st.st_mode&0x008) perm[5]='x';
  if(st.st_mode&0x004) perm[6]='r';
  if(st.st_mode&0x002) perm[7]='w';
  if(st.st_mode&0x001) perm[8]='x';

  wbkgdset(gui_stats,A_BOLD|COLOR_PAIR(C_MENU));
  wborder(gui_stats,0,0,0,0,ACS_TTEE,0,ACS_BTEE,0);
  wmove(gui_stats,startat,1);
  whline(gui_stats,A_BOLD|COLOR_PAIR(C_MENU)|ACS_HLINE,19);
  mvwaddch(gui_stats,startat, 0,A_BOLD|COLOR_PAIR(C_MENU)|ACS_LTEE);
  mvwaddch(gui_stats,startat,20,A_BOLD|COLOR_PAIR(C_MENU)|ACS_RTEE);
  if(dispmode==DISP_TREE)
    mvwaddch(gui_stats,maxy-8,0,A_BOLD|COLOR_PAIR(C_MENU)|ACS_RTEE);
  wbkgdset(gui_stats,A_BOLD|COLOR_PAIR(C_NORMAL));
  mvwprintw(gui_stats,startat+1,1,"%-19.19s",gui_printable(19,data_name(node)));
  if(S_ISCHR(st.st_mode)||S_ISBLK(st.st_mode))
    mvwprintw(gui_stats,startat+2,1,"device:% 4d,% 4d",(int)st.st_rdev>>8,(int)st.st_rdev&0xff);
  else
    mvwprintw(gui_stats,startat+2,1,"size: %-9d",(int)st.st_size);
  mvwprintw(gui_stats,startat+3,1,"%9s%c(%7s)",
	    perm,(st.st_mode&S_ISVTX)?'t':' ',gui_fmodename(node,0));
  if(st.st_mode&S_ISUID) wbkgdset(gui_stats,A_BOLD|COLOR_PAIR(C_WARN));
  mvwprintw(gui_stats,startat+4,1,"%-8.8s",uid_uidtoname(st.st_uid));
  if(st.st_mode&S_ISUID) wbkgdset(gui_stats,A_BOLD|COLOR_PAIR(C_NORMAL));
  mvwprintw(gui_stats,startat+4,9," ");
  if(st.st_mode&S_ISGID) wbkgdset(gui_stats,A_BOLD|COLOR_PAIR(C_WARN));
  mvwprintw(gui_stats,startat+4,10,"%-8.8s",uid_gidtoname(st.st_gid));
  if(st.st_mode&S_ISGID) wbkgdset(gui_stats,A_BOLD|COLOR_PAIR(C_NORMAL));
  mvwprintw(gui_stats,startat+5,1,"rt: %12.12s",gui_time(st.st_atime));
  mvwprintw(gui_stats,startat+6,1,"wt: %12.12s",gui_time(st.st_mtime));
  if(dispmode==DISP_FILE)
    mvwprintw(gui_stbar,0,0," %-*.*s",maxx-26,maxx-26,gui_printable(maxx-26,data_fullname(data_updir(currentnode))));
  else
    mvwprintw(gui_stbar,0,0," %-*.*s",maxx-26,maxx-26,gui_printable(maxx-26,data_fullname(currentdir)));
  wnoutrefresh(gui_stbar);
}

char fmn_string[8];

const char *gui_fmodename(int node,int brief) {
  int n;
  if(brief) {
    switch(data_info(node).st_mode&S_IFMT) {
    case S_IFSOCK: strcpy(fmn_string,"soc"); break;
    case S_IFLNK:  strcpy(fmn_string,"lnk"); break;
    case S_IFREG:  strcpy(fmn_string,"reg"); break;
    case S_IFBLK:  strcpy(fmn_string,"blk"); break;
    case S_IFDIR:  strcpy(fmn_string,"dir"); break;
    case S_IFCHR:  strcpy(fmn_string,"chr"); break;
    case S_IFIFO:  strcpy(fmn_string,"ffo"); break;
    default:       strcpy(fmn_string,"???"); break;
    }
  } else {
    switch(data_info(node).st_mode&S_IFMT) {
    case S_IFSOCK: strcpy(fmn_string,"socket "); break;
    case S_IFLNK:  strcpy(fmn_string,"link   "); break;
    case S_IFREG:  strcpy(fmn_string,"regular"); break;
    case S_IFBLK:  strcpy(fmn_string,"block  "); break;
    case S_IFDIR:  strcpy(fmn_string,"dir    "); break;
    case S_IFCHR:  strcpy(fmn_string,"char   "); break;
    case S_IFIFO:  strcpy(fmn_string,"fifo   "); break;
    default:       strcpy(fmn_string,"ummm..."); break;
    }
  }
  if(data_islnk(node)&&data_goodlink(node)&&config.followlinks)
    for(n=0;fmn_string[n];n++) fmn_string[n]=toupper((int)fmn_string[n]);
  return fmn_string;
}

char gui_timestring[32];
time_t gui_nowtime;

const char *gui_time(time_t t) {
  int day,hour,min,sec,year;
  char dow[4],month[4];
  gui_nowtime=time((time_t *)0);
  sscanf(ctime(&t),"%s %s %d %d:%d:%d %d",
	 dow,month,&day,&hour,&min,&sec,&year);
  if((gui_nowtime-t)>(3600*24*182.5))  /* more than six months ago */
    sprintf(gui_timestring,"%3.3s %2d  %4d",month,day,year);
  else
    sprintf(gui_timestring,"%3.3s %2d %02d:%02d",month,day,hour,min);
  return gui_timestring;
}

int gui_asksortcriteria(void) {
  int sc;
  switch(readln_getchar(gui_msbar,"Sort criteria? (<n>ame,<p>ath,<d>ate,<s>ize)","npds\a")) {
    case 'n': sc=SORT_ALPHA; break;
    case 'p': sc=SORT_PATH;  break;
    case 'd': sc=SORT_DATE;  break;
    case 's': sc=SORT_SIZE;  break;
    default: set_cancel(); return -1;
  }
  switch(readln_getchar(gui_msbar,"Order? (<f>orward,<r>everse)","fr\a")) {
    case 'f': break;
    case 'r': sc|=SORT_DIRECTION; break;
    default: set_cancel(); return -1;
  }
  return sc;
}

/* this function moves the cursor to node as if user was using HOME and DOWN
   keys */

int gui_positioncursor(int node) {
  gui_setwantdrawall();
  gui_drawmode();
  gui_specialpress(KEY_HOME);
  while((currentnode!=node)&&data_nextshown(currentnode))
    gui_specialpress(KEY_DOWN);
  gui_setwantdraw();
  gui_drawmode();
  if(currentnode!=node) return 1;
  else return 0;
}

/* this function puts the cursor on currentnode, keeping grey bar in same
   area of screen if possible */

void gui_synctocurrentnode_bar(void) {
  int a,n,oldttd;
  switch(dispmode) {
  case DISP_FILE:
    cursorfileloc%=gui_numfiledisp();
    for(n=0,a=currentnode;(n<cursorfileloc)&&a;n++) a=data_prevshown(a);
    if(a) topfile=a; else { topfile=data_firstshown(); cursorfileloc=n-1; }
    if(topfile!=data_firstshown()) {
      for(n=cursorfileloc,a=currentnode;(n<gui_numfiledisp())&&a;n++)
        a=data_nextshown(a);
      if(!a)
        for(;(n<gui_numfiledisp())&&data_prevshown(topfile);n++) {
          topfile=data_prevshown(topfile);
          cursorfileloc++; 
        }
    }
    break;
  case DISP_TREE:
    oldttd=toptreedir;
    currentdir=currentnode;
    if(!currentnode) {
      toptreedir=cursortreeloc=0;
      topnextnode=data_nextnode(toptreedir);
      toptreelines=data_resettreelines(toptreelines);
      break;
    }
    for(n=cursortreeloc,a=currentnode;(n<gui_numtreedisp())&&a;n++) a=data_nexttree(a);
    cursortreeloc+=gui_numtreedisp()-n;
    for(n=0,a=currentnode;(n<cursortreeloc)&&a;n++) a=data_prevtree(a);
    if(a) {
      toptreedir=a;
      if(oldttd!=toptreedir) {
        if(toptreedir==data_prevtree(oldttd))
          toptreelines=data_prevtreelines(toptreelines,oldttd);
        else if(toptreedir==data_nexttree(oldttd))
          toptreelines=data_nexttreelines(toptreelines,oldttd);
        else toptreelines=data_updatetreelines(toptreelines,toptreedir);
      } else if(topnextnode!=data_nextnode(toptreedir)) {
        toptreelines=data_updatetreelines(toptreelines,toptreedir);
        topnextnode=data_nextnode(toptreedir);
      }
    } else {
      toptreedir=0;
      topnextnode=data_nextnode(toptreedir);
      toptreelines=data_resettreelines(toptreelines);
      cursortreeloc=n;
    }
    break;
  default:
    crapout("gui_synctocurrentnode_bar","called in wrong mode");
  }
  gui_setwantdrawall();
}

/* synctocurrentnode, but keep bar away from top or bottom */

void gui_synctocurrentnode_bar_nice(void) {
  switch(dispmode) {
  case DISP_FILE:
    if(cursorfileloc>(gui_numfiledisp()-5))
      cursorfileloc=(gui_numfiledisp()-5);
    if(cursorfileloc<4) cursorfileloc=4;
    break;
  case DISP_TREE:
    if(cursortreeloc>(gui_numtreedisp()-5))
      cursortreeloc=(gui_numtreedisp()-5);
    if(cursortreeloc<4) cursortreeloc=4;
    break;
  }
  gui_synctocurrentnode_bar();
}

void gui_updatecurrentnode(void) {
  int n,node;
  if(dispmode==DISP_FILE) {
    node=topfile;
    for(n=0;n<cursorfileloc;n++) node=data_nextshown(node);
    currentnode=node;
  } else crapout("gui_updatecurrentnode","called in wrong dispmode");
}

extern int quitnow;
int metakey=0;

void gui_processkey(int c) {
  mvwprintw(gui_msbar,0,1,"Wait...");
  wnoutrefresh(gui_msbar);
  doupdate();
  if((c>='A')&&(c<='Z')) c+='a'-'A';
  if((c=='\r')||(c=='\n')) c=KEY_ENTER;
  if(c=='\b') c=KEY_BACKSPACE;
  if((!metakey)&&(c==27)) {metakey=1; return;}

  if(metakey) {gui_metapress(c); metakey=0;}
  else if((c>='a')&&(c<='z')) gui_letterpress(c);
  else if((c>=1)&&(c<=26)) gui_controlpress(c+'a'-1);
  else if(c<256) gui_symbolpress(c);
  else gui_specialpress(c);
  if((c==KEY_DOWN)||(c==KEY_UP)) gui_setwantdraw();
  else gui_setwantdrawall();
}

extern int logbranch_recursions;

void gui_symbolpress(int c) {
  switch(dispmode) {
  case DISP_TREE:
    switch(c) {
    case '+':
    case '=':
      if(uid_okread(currentnode)) {
        data_logdir(currentnode);
      } else gui_mildbeep("Error: Permission Denied");
      break;
    case '-':
    case '_':
      data_unlogdir(currentnode);
      break;
    case '*':
      if(uid_okread(currentnode)) {
        data_logbranch(currentnode);
        if(logbranch_recursions) crapout("data_logbranch","return with nonzero recursions");
      } else gui_mildbeep("Error: Permission Denied");
      break;
    case '?':
      helpmode=DISP_TREE;
      view_openhelp(HELP_TREE);
      break;
    }
    break;
  case DISP_FILE:
    switch(c) {
    case '!': /* FIXME - debug command */
      gui_synctocurrentnode_bar();
      break;
    case '?':
      helpmode=DISP_FILE;
      view_openhelp(HELP_FILE);
      break;
    }
    break;
  case DISP_VIEW:
    switch(c) {
    case '?':
      helpmode=DISP_VIEW;
      view_openhelp(HELP_VIEW);
      break;
    case ' ':
      view_pagedn();
      break;
    case '/':
      view_search();
      break;
    case '\\':
      view_search_rev();
      break;
    }
    break;
  case DISP_HELP:
    switch(c) {
    case '?':
      view_closehelp();
      gui_initdispmode(helpmode);
      break;
    case ' ':
      view_pagedn();
      break;
    case '/':
      view_search();
      break;
    case '\\':
      view_search_rev();
      break;
    }
    break;
  }
}

void gui_letterpress(int c) {
  int a;
  switch(dispmode) {
  case DISP_TREE:
    switch(c) {
    case 'o':
      comm_mount();
      break;
    case 'f':
      comm_filespec();
      break;
    case 'i':
      comm_invertfilespec();
      break;
    case 'l':
      comm_symlink();
      break;
    case 's':
      filelist=data_makelistofbranch(0);
      if(!filelist) {
        gui_mildbeep("Error: no logged files");
        break;
      } else if(!data_firstshown()) {
        gui_mildbeep("Error: no matching logged files");
        break;
      }
      cursorfileloc=0;
      currentnode=topfile=data_firstshown();
      dispfilemode=D_FILE_SHOWALL;
      gui_initdispmode(DISP_FILE);
      break;
    case 'b':
      filelist=data_makelistofbranch(currentnode);
      if(!filelist) {
        gui_mildbeep("Error: no logged files in branch");
        break;
      } else if(!data_firstshown()) {
        gui_mildbeep("Error: no matching logged files in branch");
        break;
      }
      cursorfileloc=0;
      currentnode=topfile=data_firstshown();
      dispfilemode=D_FILE_BRANCH;
      gui_initdispmode(DISP_FILE);
      break;
    case 'g':
      comm_chdir();
      break;
    case 'm':
      if(uid_okread(currentnode)&&uid_okwrite(currentnode)) {
        comm_makedir();
      } else gui_mildbeep("Error: Permission Denied");
      break;
    case 'd':
      if(uid_okdirop(currentnode)) {
        if(data_islnk(currentnode)) comm_rmfile();
                               else comm_rmdir();
      } else gui_mildbeep("Error: Permission Denied");
      break;
    case 'r':
      if(uid_okdirop(currentnode)) {
        comm_rename();
      } else gui_mildbeep("Error: Permission Denied");
      break;
     case 't':
        if((a=data_downdir(currentnode)))
          while((a=data_nextfile(a)))
            if(data_matchfilespec(data_name(a))) data_settag(a,1);
        gui_specialpress(KEY_DOWN);
        break;
     case 'u':
        if((a=data_downdir(currentnode)))
          while((a=data_nextfile(a)))
            if(data_matchfilespec(data_name(a))) data_settag(a,0);
        gui_specialpress(KEY_DOWN);
        break;
    case 'x':
      comm_exec();
      if(data_downdir(currentdir)) data_restatdir(currentdir);
      break;
    case 'q':
      quitnow++;
      break;
    }
    break;
  case DISP_FILE:
    switch(c) {
    case 'f':
      comm_filespec();
      break;
    case 'i':
      comm_invertfilespec();
      break;
    case 'l':
      comm_symlink();
      break;
    case 'g':
      comm_gotofile();
      break;
    case 'v':
      if(uid_okread(currentnode)&&data_isregular(currentnode)) {
        view_openfile(currentnode);
      } else gui_mildbeep("Error: Permission Denied");
      break;
    case 't':
      if(config.tagtoggle)
        data_settag(currentnode,!data_tagged(currentnode));
      else
        data_settag(currentnode,1);
      gui_specialpress(KEY_DOWN);
      break;
    case 'u':
      data_settag(currentnode,0);
      gui_specialpress(KEY_DOWN);
      break;
    case 'd':
      if(uid_okdirop(currentnode)) {
        comm_rmfile();
      } else gui_mildbeep("Error: Permission Denied");
      break;
    case 'c':
      if(uid_okread(currentnode)) {
        comm_cpfile();
      } else gui_mildbeep("Error: Permission Denied");
      break;
    case 'm':
      if(uid_okdirop(currentnode)) {
        comm_mvfile();
      } else gui_mildbeep("Error: Permission Denied");
      break;
    case 'r':
      if(uid_okdirop(currentnode)) {
        comm_rename();
      } else gui_mildbeep("Error: Permission Denied");
      break;
    case 'o':
      if(uid_okread(currentnode)) {
        comm_open(0);
      } else gui_mildbeep("Error: Permission Denied");
      break;
    case 'e':
      if(uid_okread(currentnode)&&uid_okwrite(currentnode)) {
        comm_edit();
      } else gui_mildbeep("Error: Permission Denied");
      break;
    case 'x':
      comm_exec();
      if(data_downdir(currentdir)) data_restatdir(currentdir);
      break;
    case 'q':
      quitnow++;
      break;
    }
    break;
  case DISP_VIEW:
    switch(c) {
    case 'b':
      view_pageup();
      break;
    case 't':
      view_undosearchbuffer();
      viewmode=VIEW_TEXT;
   /* called because we probably have changed topview, and so paragraphs */
   /* need to be recalculated */
      view_resetparagraph();
      break;
    case 'd':
      view_undosearchbuffer();
      viewmode=VIEW_DUMP;
      break;
    case 'h':
      view_undosearchbuffer();
      viewmode=VIEW_HEX;
      break;
    case 'm':
      viewmask=!viewmask;
      break;
    case 'f':
    case 'p':
      view_askpipe();
      break;
    case 'v':
    case 'q': 
      view_closefile();
      gui_initdispmode(DISP_FILE);
      break;
    }
    break;
  case DISP_HELP:
    switch(c) {
    case 'v':
    case 'q':
      view_closehelp();
      gui_initdispmode(helpmode);
      break;
    }
  }
}

void gui_controlpress(int c) {
  int a;
  if(c=='l') {gui_refreshscreen(); return;}
  switch(dispmode) {
  case DISP_FILE:
    switch(c) {
    case 'i':
      comm_editstatinfo();
      break;
    case 'r':
      switch(dispfilemode) {
      case D_FILE_TAGBRANCH:
      case D_FILE_TAGSHOWALL:
        a=filelist;
        while(a) {
          data_restatnode(a);
          a=data_nextlist(a);
        }
        gui_setwantdrawall();
        break;
      case D_FILE_SHOWALL:
        data_restatloggeddirs(0);
        break;
      case D_FILE_BRANCH:
        data_restatloggeddirs(currentdir);
        break;
      case D_FILE_NORM:
        data_restatdir(currentdir);
        break;
      default: crapout("gui_controlpress","invalid dispfilemode");
      }
      break;
    case 't':
      a=data_firstshown();
      while(a) {
        data_settag(a,1);
        a=data_nextshown(a);
      }
      gui_setwantdrawall();
      break;
    case 'n':
      a=data_firstshown();
      while(a) {
        data_settag(a,data_tagged(a)?0:1);
        a=data_nextshown(a);
      }
      gui_setwantdrawall();
      break;
    case 'u':
      a=data_firstshown();
      while(a) {
        data_settag(a,0);
        a=data_nextshown(a);
      }
      gui_setwantdrawall();
      break;
    case 'd':
      comm_tag_rmfile();
      break;
    case 'c':
      comm_tag_cpfile();
      break;
    case 'v':
      comm_tag_mvfile();
      break;
    case 'a':
      comm_tag_chmod();
      break;
    case 'o':
      comm_tag_chown();
      break;
    case 'g':
      comm_tag_grep();
      break;
    }
    break;
  case DISP_TREE:
    switch(c) {
    case 'i':
      comm_editstatinfo();
      break;
    case 's':
      filelist=data_makelistfromtagged(0);
      if(!filelist) {
        gui_mildbeep("Error: no tagged files");
        break;
      } else if(!data_firstshown()) {
        gui_mildbeep("Error: no matching tagged files");
        break;
      }
      cursorfileloc=0;
      currentnode=topfile=data_firstshown();
      dispfilemode=D_FILE_TAGSHOWALL;
      gui_initdispmode(DISP_FILE);
      break;
    case 'b':
      filelist=data_makelistfromtagged(currentnode);
      if(!filelist) {
        gui_mildbeep("Error: no tagged files in branch");
        break;
      } else if(!data_firstshown()) {
        gui_mildbeep("Error: no matching tagged files in branch");
        break;
      }
      cursorfileloc=0;
      currentnode=topfile=data_firstshown();
      dispfilemode=D_FILE_TAGBRANCH;
      gui_initdispmode(DISP_FILE);
      break;
    case 'r':
      data_restatdir(currentnode);
      break;
    case 't':
      data_settagdir(currentnode,1);
      break;
    case 'u':
      data_settagdir(currentnode,0);
      break;
    }
    break;
  }
}

void gui_metapress(int c) {
  int sc;
  switch(dispmode) {
  case DISP_TREE:
    switch(c) {
    case 'o':
      config_runtimeconfig();
      break;
    case 'd':
      debugmode=!debugmode;
      break;
    case 's':
      sc=gui_asksortcriteria();
      if(sc==-1) break; else sortcriteria=sc;
      break;
    case 'a':
      comm_archive();
      break;
    case 'p':
      comm_rmbranch();
      break;
    case 'm':
      comm_mvbranch();
      break;
    case 'f':
      dispform=(dispform+1)%D_FORM_NUM;
      break;
    }
    break;
  case DISP_FILE:
    switch(c) {
    case 'o':
      config_runtimeconfig();
      break;
    case 's':
      sc=gui_asksortcriteria();
      if(sc==-1) break; else sortcriteria=sc;
      filelist=data_mergesortlist(filelist,sortcriteria);
      gui_synctocurrentnode_bar();
      break;
/* FIXME - key used up
    case 'o':
      if(uid_okread(currentnode)) {
        comm_open(1);
      } else gui_mildbeep("Error: Permission Denied");
      break;
*/
    case 't':
      comm_tagbyfilespec(1);
      break;
    case 'u':
      comm_tagbyfilespec(0);
      break;
    case 'd':
      debugmode=!debugmode;
      break;
    case 'f':
      dispform=(dispform+1)%D_FORM_NUM;
      gui_synctocurrentnode_bar();
      break;
    }
    break;
  case DISP_VIEW:
    switch(c) {
    case 'u':
      view_toggleshowsearch();
      break;
    }
    break;
  case DISP_HELP:
    switch(c) {
    case 'u':
      view_toggleshowsearch();
      break;
    }
    break;
  }
}

void gui_specialpress(int c) {
  int n,node;
  switch(dispmode) {
  case DISP_TREE:
    switch(c) {
    case KEY_LEFT:
      if(data_downdir(currentnode)) data_unlogdir(currentnode);
      else data_unlogdir(data_updir(currentnode));
      break;
    case KEY_RIGHT:
      if(!data_downdir(currentnode)) gui_specialpress(KEY_ENTER);
      else                           gui_specialpress(KEY_DOWN);
      break;
    case KEY_UP:
      if(cursortreeloc) cursortreeloc--;
      else if (toptreedir) {
        toptreelines=data_prevtreelines(toptreelines,toptreedir);
	toptreedir=data_prevtree(toptreedir);
        topnextnode=data_nextnode(toptreedir);
      }
      if(currentnode)
        currentdir=currentnode=data_prevtree(currentnode);
      break;
    case KEY_DOWN:
      if(data_nexttree(currentnode)) {
        currentdir=currentnode=data_nexttree(currentnode);
        if(cursortreeloc<(gui_numtreedisp()-1))
          cursortreeloc++;
        else {
          toptreelines=data_nexttreelines(toptreelines,toptreedir);
          toptreedir=data_nexttree(toptreedir);
          topnextnode=data_nextnode(toptreedir);
        }
      }
      break;
    case KEY_PPAGE:
      if(cursortreeloc)
        while(cursortreeloc) gui_specialpress(KEY_UP);
      else
        for(n=0;n<gui_numtreedisp()-1;n++) gui_specialpress(KEY_UP);
      break;
    case KEY_NPAGE:
      if(cursortreeloc<(gui_numtreedisp()-1))
        while((cursortreeloc<(gui_numtreedisp()-1)) &&
              data_nexttree(currentnode)) gui_specialpress(KEY_DOWN);
      else
        for(n=0;n<gui_numtreedisp()-1;n++) gui_specialpress(KEY_DOWN);
      break;
    case KEY_HOME:
      cursortreeloc=0;
      currentnode=0;
      currentdir=0;
      toptreedir=0;
      topnextnode=data_nextnode(toptreedir);
      toptreelines=data_resettreelines(toptreelines);
      gui_setwantdrawall();
      break;
    case KEY_END:
      n=0;
      while(data_nextdir(data_downdir(n))) {
        n=data_nextdir(data_downdir(n));
        while(data_nextdir(n)) n=data_nextdir(n);
      }
      currentnode=n;
      cursortreeloc=gui_numtreedisp()-1;
      gui_synctocurrentnode_bar();
      gui_setwantdrawall();
      break;
    case KEY_ENTER:
      if(!data_downdir(currentnode)) {
        if(uid_okread(currentnode)) {
          data_logdir(currentnode);
        } else gui_mildbeep("Error: Permission Denied");
      } else {
        if(data_hasmatching(currentnode)) {
          cursorfileloc=0;
          filelist=data_makelistfromdir(currentnode);
          topfile=currentnode=data_firstshown();
          dispfilemode=D_FILE_NORM;
          gui_initdispmode(DISP_FILE);
        }
      }
      break;
    }
    break;
  case DISP_FILE:
    switch(c) {
    case KEY_UP:
      if(cursorfileloc) cursorfileloc--;
      else if(data_prevshown(topfile)) {
	topfile=data_prevshown(topfile);
	numscroll--;
      }
      if(data_prevshown(currentnode))
        currentnode=data_prevshown(currentnode);
      break;
    case KEY_DOWN:
      if(data_nextshown(currentnode)) {
        currentnode=data_nextshown(currentnode);
	if(cursorfileloc<(gui_numfiledisp()-1))
	  cursorfileloc++;
	else {
	  topfile=data_nextshown(topfile);
	  numscroll++;
	}
      }
      break;
    case KEY_LEFT:
      for(n=0;n<gui_numfilerows();n++) {
        node=data_prevshown(currentnode);
        if(node) currentnode=node; else break;
        cursorfileloc--;
      }
      if(cursorfileloc<0) {
        for(n=0;n<gui_numfilerows();n++) {
          node=data_prevshown(topfile);
          if(node) topfile=data_prevshown(topfile); else break;
          cursorfileloc++;
        }
      }
      break;
    case KEY_RIGHT:
      for(n=0;n<gui_numfilerows();n++) {
        node=data_nextshown(currentnode);
        if(node) currentnode=node; else break;
        cursorfileloc++;
      }
      if(cursorfileloc>=gui_numfiledisp()) {
        for(n=0;n<gui_numfilerows();n++)
          topfile=data_nextshown(topfile);
        cursorfileloc-=gui_numfilerows();
      }
      break;
    case KEY_PPAGE:
      if(cursorfileloc)
        while(cursorfileloc) gui_specialpress(KEY_UP);
      else
        for(n=0;n<gui_numfiledisp()-1;n++) gui_specialpress(KEY_UP);
      break;
    case KEY_NPAGE:
      if(cursorfileloc<(gui_numfiledisp()-1))
        while((cursorfileloc<(gui_numfiledisp()-1)) &&
               data_nextshown(currentnode)) gui_specialpress(KEY_DOWN);
      else
        for(n=0;n<gui_numfiledisp()-1;n++) gui_specialpress(KEY_DOWN);
      break;
    case KEY_HOME:
      cursorfileloc=0;
      currentnode=topfile=data_firstshown();
      gui_setwantdrawall();
      break;
    case KEY_END:
      while(data_nextshown(currentnode)) gui_specialpress(KEY_DOWN);
      break;
    case KEY_ENTER:
      data_breaklist(filelist);
      filelist=0;
      currentnode=currentdir;
      gui_initdispmode(DISP_TREE);
      break;
    }
    break;
  case DISP_VIEW:
    switch(c) {
    case KEY_UP:
      view_scrollup();
      break;
    case KEY_DOWN:
      view_scrolldown();
      break;
    case KEY_PPAGE:
      view_pageup();
      break;
    case KEY_NPAGE:
      view_pagedn();
      break;
    case KEY_HOME:
      view_home();
      break;
    case KEY_END:
      view_end();
      break;
    case KEY_ENTER:
      view_closefile();
      gui_initdispmode(DISP_FILE);
      break;
    }
    break;
  case DISP_HELP:
    switch(c) {
    case KEY_UP:
      view_scrollup();
      break;
    case KEY_DOWN:
      view_scrolldown();
      break;
    case KEY_PPAGE:
      view_pageup();
      break;
    case KEY_NPAGE:
      view_pagedn();
      break;
    case KEY_HOME:
      view_home();
      break;
    case KEY_END:
      view_end();
      break;
    case KEY_ENTER:
      view_closehelp();
      gui_initdispmode(helpmode);
      break;
    }
    break;
  }
}

void gui_listlong(int node,int fpos,int numy,WINDOW *win) {
  chtype oldbg;
  struct stat st;
  int wx,wy;
  int wmaxx,wmaxy;
  int left,right;

  st=data_info(node);
  oldbg=wgetbkgd(win);
  getmaxyx(win,wmaxy,wmaxx);

  switch(dispform) {
  case D_FORM_LONG:
    wy=fpos+1;
    wx=1;
    mvwprintw(win,wy,wx,"%c%04o%c(%7s) ",data_tagged(node)?'*':' ',
      st.st_mode&0xfff,(node==currentnode)?'>':' ',gui_fmodename(node,0));
    if(st.st_mode&S_ISUID) wbkgdset(win,A_BOLD|COLOR_PAIR(C_WARN));
    mvwprintw(win,wy,wx+16,"%-8.8s ",uid_uidtoname(st.st_uid));
    if(st.st_mode&S_ISUID) wbkgdset(win,oldbg);
    if(st.st_mode&S_ISGID) wbkgdset(win,A_BOLD|COLOR_PAIR(C_WARN));
    mvwprintw(win,wy,wx+25,"%-8.8s",uid_gidtoname(st.st_gid));
    if(st.st_mode&S_ISGID) wbkgdset(win,oldbg);
    if(S_ISCHR(st.st_mode)||S_ISBLK(st.st_mode))
      mvwprintw(win,wy,wx+33,"% 4d,% 4d",(int)st.st_rdev>>8,(int)st.st_rdev&0xff);
    else
      mvwprintw(win,wy,wx+33,"%9d",(int)st.st_size);
    mvwprintw(win,wy,wx+42," %-*.*s",wmaxx-45,wmaxx-45,gui_printable(wmaxx-45,data_name(node)));
    break;
  case D_FORM_SHORTA:
    wy=fpos+1;
    wx=1;
    mvwprintw(win,wy,wx,"%c%04o%c(%3s) ",data_tagged(node)?'*':' ',
      st.st_mode&0xfff,(node==currentnode)?'>':' ',gui_fmodename(node,1));
    if(st.st_mode&S_ISUID) wbkgdset(win,A_BOLD|COLOR_PAIR(C_WARN));
    mvwprintw(win,wy,wx+12,"%-4.4s ",uid_uidtoname(st.st_uid));
    if(st.st_mode&S_ISUID) wbkgdset(win,oldbg);
    if(st.st_mode&S_ISGID) wbkgdset(win,A_BOLD|COLOR_PAIR(C_WARN));
    mvwprintw(win,wy,wx+17,"%-4.4s",uid_gidtoname(st.st_gid));
    if(st.st_mode&S_ISGID) wbkgdset(win,oldbg);
    if(S_ISCHR(st.st_mode)||S_ISBLK(st.st_mode))
      mvwprintw(win,wy,wx+21,"% 4d,% 4d",(int)st.st_rdev>>8,(int)st.st_rdev&0xff);
    else
      mvwprintw(win,wy,wx+21,"%9d",(int)st.st_size);
    mvwprintw(win,wy,wx+30," %-*.*s",wmaxx-33,wmaxx-33,gui_printable(wmaxx-33,data_name(node)));
    break;
  case D_FORM_SHORTB:
    wy=fpos+1;
    wx=1;
    mvwprintw(win,wy,wx,"%c%04o%c(%3s) ",data_tagged(node)?'*':' ',
      st.st_mode&0xfff,(node==currentnode)?'>':' ',gui_fmodename(node,1));
    mvwprintw(win,wy,wx+12,"%12.12s",gui_time(st.st_mtime));
    if(S_ISCHR(st.st_mode)||S_ISBLK(st.st_mode))
      mvwprintw(win,wy,wx+24,"% 4d,% 4d",(int)st.st_rdev>>8,(int)st.st_rdev&0xff);
    else
      mvwprintw(win,wy,wx+24,"%9d",(int)st.st_size);
    mvwprintw(win,wy,wx+33," %-*.*s",wmaxx-36,wmaxx-36,gui_printable(wmaxx-36,data_name(node)));
    break;
  case D_FORM_NAMEONLY:
    wy=fpos+1;
    wx=1;
    mvwprintw(win,wy,wx,"%c%c",data_tagged(node)?'*':' ',
      (node==currentnode)?'>':' ');
    mvwprintw(win,wy,wx+2,"%-*.*s",
      wmaxx-4,wmaxx-4,gui_printable(wmaxx-4,data_name(node)));
    break;
  case D_FORM_TWOCOL:
  case D_FORM_TRICOL:
    if(dispform==D_FORM_TWOCOL) {
      left =(((fpos/numy)  )*(wmaxx-1)+1)/2+1;
      right=(((fpos/numy)+1)*(wmaxx-1)+1)/2+1;
    } else {
      left =(((fpos/numy)  )*(wmaxx-1)+2)/3+1;
      right=(((fpos/numy)+1)*(wmaxx-1)+2)/3+1;
    }
    wy=(fpos%numy)+1;
    wx=left;
    wmaxx=right-left+1;

    mvwprintw(win,wy,wx,"%c%c",data_tagged(node)?'*':' ',
      (node==currentnode)?'>':' ');
    mvwprintw(win,wy,wx+2,"%-*.*s",
      wmaxx-4,wmaxx-4,gui_printable(wmaxx-4,data_name(node)));
    break;
  default:
    crapout("gui_listlong","unknown list format");
  }
}

/* filter out control chars and use elipses if output is longer than maxlen
   (maxlen=0 means never use elipses) */
char printable[(PATH_MAX+1)*4];
const char *gui_printable(int maxlen,const char *in) {
  const unsigned char *p;
  int q,c;

  if(strlen(in)>PATH_MAX) crapout("gui_printable","string too long");
  if(maxlen<0) crapout("gui_printable","negative maxlen");
  switch(config.rawmode) {
  case RAWMODE_HIDE:
    q=0;
    for(p=in;*p;p++) {
      c=*p;
      if((c>=0x20)&&(c<=0x7e)) printable[q++]=c;
      else printable[q++]='?';
    }
    printable[q]=0;
    break;
  case RAWMODE_OCTAL:
    q=0;
    for(p=in;*p;p++) {
      c=*p;
      if((c>=0x20)&&(c<=0x7e)) printable[q++]=c;
      else {
        printable[q++]='\\';
        printable[q++]='0'+(c>>6);
        printable[q++]='0'+((c>>3)&7);
        printable[q++]='0'+(c&7);
      }
    }
    printable[q]=0;
    break;
  case RAWMODE_CONTROL:
    q=0;
    for(p=in;*p;p++) {
      c=*p;
      if((c>=1)&&(c<=26)) {
        printable[q++]='^';
        printable[q++]='A'-1+c;
      } else
      if((c>=0x20)&&(c<=0x7e)) printable[q++]=c;
      else {
        printable[q++]='\\';
        printable[q++]='0'+(c>>6);
        printable[q++]='0'+((c>>3)&7);
        printable[q++]='0'+(c&7);
      }
    }
    printable[q]=0;
    break;
  default:
    crapout("gui_printable","invalid rawmode");
  }
  if(maxlen&&(strlen(printable)>maxlen)) {
    if(maxlen>4) strcpy(printable+maxlen-3,"...");
    else {strcpy(printable+1,"..."); printable[maxlen]=0;}
  }
  return printable;
}

const char *gui_printable_quoted(int maxlen,const char *in) {
  char *s;
  const char *p;
  s=malloc(strlen(in)+3);
  if(!s) outofmem("gui_printable_quoted");
  sprintf(s,"\"%s\"",in);
  p=gui_printable(maxlen,s);
  free(s);
  return p;
}

int gui_printable_curspos(const char *in,int pin) {
  int n,pos;
  int c;
  switch(config.rawmode) {
  case RAWMODE_HIDE:
    return pin;
  case RAWMODE_OCTAL:
    pos=0;
    for(n=0;n<pin;n++) {
      c=in[n];
      if((c>=0x20)&&(c<=0x7e)) pos++;
      else                     pos+=4;
    }
    return pos;
  case RAWMODE_CONTROL:
    pos=0;
    for(n=0;n<pin;n++) {
      c=in[n];
      if((c>=0x20)&&(c<=0x7e)) pos++;
      else if((c>=1)&&(c<=26)) pos+=2;
      else                     pos+=4;
    }
    return pos;
  default:
    crapout("gui_printable_curspos","invalid rawmode");
    return 0; /* keeps compiler happy */
  }
}

void gui_setwantdraw(void) {
  gui_wantdraw=1;
}

void gui_setwantdrawall(void) {
  gui_setwantdraw();
  updateall=1;
}

void gui_mildbeep(const char *fmt, ...) {
  va_list argp;

  if(keep_msbar>2) return;
  keep_msbar=2;
  va_start(argp, fmt);
  vsnprintf(bottommsg,maxx+2,fmt,argp);
  strcpy(bottommsg,gui_printable(maxx-1,bottommsg));
  werase(gui_msbar);
  mvwprintw(gui_msbar,0,1,"%s",bottommsg);
  va_end(argp);

  if(!config.silent) beep();
  wnoutrefresh(gui_msbar);
  doupdate();
}

void gui_printmsg(const char *fmt, ...) {
  va_list argp;
  if(patience_up) return;
  if(keep_msbar>1) return;
  keep_msbar=1;

  va_start(argp, fmt);
  vsnprintf(bottommsg,maxx+2,fmt,argp);
  strcpy(bottommsg,gui_printable(maxx-1,bottommsg));
  werase(gui_msbar);
  mvwprintw(gui_msbar,0,1,"%s",bottommsg);
  va_end(argp);

  wnoutrefresh(gui_msbar);
  doupdate();
}

void gui_printmsg_wait(const char *fmt, ...) {
  va_list argp;
  if(patience_up) return;

  va_start(argp, fmt);
  vsnprintf(bottommsg,maxx+2,fmt,argp);
  strcpy(bottommsg,gui_printable(maxx-1,bottommsg));
  werase(gui_msbar);
  mvwprintw(gui_msbar,0,1,"%s - press key to continue",
    gui_printable(maxx-26,bottommsg));
  va_end(argp);

  if(!config.silent) beep();
  wnoutrefresh(gui_msbar);
  doupdate();

  timeout(100);
  while(getch()==ERR);
  while(getch()!=ERR);

  werase(gui_msbar);
  mvwprintw(gui_msbar,0,1,"%s",bottommsg);
  wnoutrefresh(gui_msbar);
  doupdate();
}
