/* linuXtree copyright (c) 1998 Dan Stahlke under the GPL
   See file 'COPYING' for details */

/*#define DEBUG*/
/*#define DEBUGNODE 10*/
/*#define DEBUGMERGESORT*/
/*#define LONGDUMPTREE*/

#include "all.h"

extern int currentnode,currentdir;
extern struct configtype config;
extern int topfile,toptreedir,cursorfileloc,cursortreeloc;
extern char *toptreelines;
extern int filelist,dispmode,dispfilemode;
extern int sortcriteria;
extern char filespec[];
extern int invfilespec;
extern int recovermode;

struct treedata_type *treedata[TREEDATACHUNKS];
int treeholes[TREEDATACHUNKS];

int byteslogged=0,fileslogged=0,dirslogged=0,
    bytestagged=0,filestagged=0,dirstagged=0,
    byteshown,fileshown;

int logbranch_recursions=0; /* how many levels deep we are in logbranch */
int fileskipped=0;          /* how many items not logged due to long names */

#ifdef DEBUG
FILE *debug;
#endif

#if TREEBLOCKSIZE==256
  #define datanode(n) (treedata[(n)>>8]+((n)&255))
#else
  struct treedata_type *datanode(int n) {
    if(treedata[(int)(n/TREEBLOCKSIZE)]==NULL) crapout("unknown","reading null datanode <%d>",n);
    return treedata[(int)(n/TREEBLOCKSIZE)]+(n%TREEBLOCKSIZE);
  }
#endif

void data_init(void) {
#ifdef DEBUG
  debug=fopen("debug","w");
  if(debug==NULL) crapout("data_init","cannot open the debug file");
#endif
  if(!recovermode) {
    if(data_findfree()) crapout("data_init","first node not zero");
    strcpy(datanode(0)->name,"/");
    datanode(0)->depth=0;
    data_restatnode(0);
  }
#ifdef DEBUG
  data_dumptree();
#endif
}

int data_treehole=0;  /* how many deleted nodes there are */

void data_clear(int n) {
  data_settag(n,0);
  if(datanode(n)->statlogged) crapout("data_clear","clearing a statted node");
  datanode(n)->referer=-1;
  data_treehole++;
  treeholes[n/TREEBLOCKSIZE]++;
  if(treeholes[n/TREEBLOCKSIZE]==TREEBLOCKSIZE) {
#ifdef DEBUGNODE
    fprintf(debug,"freeing treeblock: block %d\n",n/TREEBLOCKSIZE);
    fflush(debug);
#endif
    free(treedata[n/TREEBLOCKSIZE]);
    treedata[n/TREEBLOCKSIZE]=NULL;
    treeholes[n/TREEBLOCKSIZE]=0;
    data_treehole-=TREEBLOCKSIZE;
  }
#ifdef DEBUG
  fprintf(debug,"clearing node: node %d\n",n);
  fflush(debug);
#ifdef DEBUGNODE
  data_dumpused();
#endif
#endif
}

void data_chopnode(int n) {
  int prev,next,resync,popout;

#ifdef DEBUG
  fprintf(debug,"chopping node: %d (%s) name: <%s%s>\n",
    n,data_isdir(n)?"dir":"notdir",data_fullname(n),
    data_name(n)[0]?"":"-stub-");
  fflush(debug);
#endif

  data_settag(n,0);
  data_unstatlog(n);

  if(!data_prevnode(n)) crapout("data_chopnode","chopnode called on stub <%d>",n);
  if(datanode(n)->point) crapout("data_chopnode","chopping node with downdir <%d>",n);

  switch(data_guardcurrentnode(n)) {
  case 0:
    resync=0;
    popout=0;
    break;
  case 1:
    resync=1;
    popout=0;
    break;
  case 2:
    resync=1;
    popout=1;
    break;
  default:
    crapout("data_chopnode","illegal return from guardcurrentnode");
    resync=popout=0; /* to keep gcc happy */
  }

  prev=data_prevnode(n);
  next=data_nextnode(n);
  if(prev) datanode(prev)->next=next;
  if(next) datanode(next)->prev=prev;
  if(resync) gui_synctocurrentnode_bar();
  data_clear(n);
  gui_setwantdrawall();
  if(popout) {
    currentnode=currentdir;
    gui_initdispmode(DISP_TREE);
  }
}

int data_findfree(void) {  /* ye who enter here better know what's going on */
  int n,m;
#ifdef DEBUGNODE
  fprintf(debug,"data_findfree: treehole=%d\n",data_treehole);
  fflush(debug);
#endif
  if(data_treehole<0) crapout("data_findfree","treehole is less than zero");
  for(n=0;n<TREEDATACHUNKS;n++) {
    if(treedata[n]&&treeholes[n])
    for(m=0;m<TREEBLOCKSIZE;m++) {
      if(datanode(n*TREEBLOCKSIZE+m)->referer==-1) {
	data_treehole--;
        treeholes[n]--;
#ifdef DEBUGNODE
	fprintf(debug,"using node hole: node %d, %d left\n",
		n*TREEBLOCKSIZE+m,data_treehole);
        fflush(debug);
        datanode(n*TREEBLOCKSIZE+m)->referer=0;
        data_dumpused();
        datanode(n*TREEBLOCKSIZE+m)->referer=-1;
        fprintf(debug,"done making new node: %d\n\n",n*TREEBLOCKSIZE+m);
#endif
        data_initnode(n*TREEBLOCKSIZE+m);
	return n*TREEBLOCKSIZE+m;
      }
    }
  }
  if(data_treehole) crapout("data_findfree","node with a messed up data_treehole");
  for(n=0;n<TREEDATACHUNKS;n++) {
    if(treedata[n]==NULL) {
      data_treehole=0;
      treeholes[n]=0;
#ifdef DEBUGNODE
      fprintf(debug,"making new node (with malloc): node %d, block %d\n",
              n*TREEBLOCKSIZE,n);
      fflush(debug);
#endif
      treedata[n]=malloc(sizeof(struct treedata_type)*TREEBLOCKSIZE);
      if(!treedata[n]) outofmem("data_findfree");
      if(treedata[n]==NULL) outofmem("data_findfree");
      for(m=0;m<TREEBLOCKSIZE;m++) {
/* this next line is to catch bugs if somebody forgets to set tagged to 0 */
        datanode(n*TREEBLOCKSIZE+m)->tagged=1;
        datanode(n*TREEBLOCKSIZE+m)->referer=-1;
      }
      data_treehole=TREEBLOCKSIZE-1;
      treeholes[n]=TREEBLOCKSIZE-1;
#ifdef DEBUGNODE
      datanode(n*TREEBLOCKSIZE)->referer=0;
      data_dumpused();
      datanode(n*TREEBLOCKSIZE)->referer=-1;
      fprintf(debug,"done making new node: %d\n\n",n*TREEBLOCKSIZE+m);
      fflush(debug);
#endif
      data_initnode(n*TREEBLOCKSIZE);
      return n*TREEBLOCKSIZE;
    }
  }
  crapout("data_findfree","too many tree node data chunks");
  return 0;  /* we never get here... make compiler happy though */
}

void data_initnode(int node) {
  struct treedata_type *p;
  p=datanode(node);
  p->referer=0;
  p->prev=0;
  p->next=0;
  p->point=0;
  p->lsinfo.st_mode=0;
  p->lsinfo.st_size=0;
  p->lpinfo.st_mode=0;
  p->lpinfo.st_size=0;
  p->statlogged=0;
  p->isdir=0;
  p->name[0]=0;
  p->depth=-1; /* to catch bugs */
  p->tagged=0;
  p->nextlist=0;
  p->prevlist=0;
  p->shown=0; /* to catch bugs */
}

void data_statlog(int node) {
  if(datanode(node)->statlogged) crapout("data_statlog","statting an already statted file");
  if(!data_name(node)[0]) return;
  if(datanode(node)->tagged) crapout("data_statlog","file should not be tagged at this point");
  if(data_isdir(node)) dirslogged++;
  else                 fileslogged++;
  byteslogged+=data_info(node).st_size;
  if(datanode(node)->shown) {
    fileshown++;
    byteshown+=data_info(node).st_size;
  }
  datanode(node)->statlogged=1;
}

void data_unstatlog(int node) {
  if(!datanode(node)->statlogged) return;
  if(!data_name(node)[0]) return;
  if(datanode(node)->tagged) crapout("data_statlog","file should not be tagged at this point");
  if(data_isdir(node)) dirslogged--;
  else                 fileslogged--;
  byteslogged-=data_info(node).st_size;
  if(datanode(node)->shown) {
    fileshown--;
    byteshown-=data_info(node).st_size;
  }
  datanode(node)->statlogged=0;
}

/* prepares currentnode,filelist,etc so that node 'n' can be deleted
   returns 0 normal, 1 syncto should be called, 2 should pop out of filemode */
int data_guardcurrentnode(int n) {
  int val=0;
  if((dispmode==DISP_TREE)&&data_isdir(n)) {
    if(n==currentnode) {
      if(data_nextdir(currentnode)) currentnode=data_nextdir(currentnode);
      else
        if(data_prevdir(currentnode)) currentnode=data_prevdir(currentnode);
      else
        currentnode=data_updir(currentnode);
    }
    val=1;
  }
  if((dispmode==DISP_FILE)&&!data_isdir(n)) {
    if(n==filelist) filelist=data_nextlist(filelist);
    if(!filelist) val=2;
    else if(n==currentnode) {
      if(data_nextshown(currentnode)) currentnode=data_nextshown(currentnode);
      else
        if(data_prevshown(currentnode)) currentnode=data_prevshown(currentnode);
      else val=2;
    }
    data_deletefromlist(n);
    if(!val) val=1;
  }
  return val;
}

int data_matchfilespec(const char *name) {
  if(!fnmatch(filespec,name,0)) return invfilespec?0:1;
  return invfilespec?1:0;
}

void data_breaklist(int node) {
  int next;
  while(node) {
    next=data_nextlist(node);
    datanode(node)->nextlist=datanode(node)->prevlist=0;
    node=next;
  }
}

int data_makelistfromdir(int dirnode) {
  int topfile,node,next;
  if(!data_downdir(dirnode)) crapout("data_makelistfromdir","making a listing for unlogged dir");
  if(!data_hasfile(dirnode)) crapout("data_makelistfromdir","making a listing for empty dir");

  byteshown=fileshown=0;
  topfile=data_nextfile(data_downdir(dirnode));
  datanode(topfile)->prevlist=0;
  node=topfile;

  datanode(node)->prevlist=0;
  do {
    if(data_matchfilespec(data_name(node))) {
      datanode(node)->shown=1;
      byteshown+=data_info(node).st_size;
      fileshown++;
    } else datanode(node)->shown=0;
    next=data_nextfile(node);
    if(!next) break;
    datanode(node)->nextlist=next;
    datanode(next)->prevlist=node;
    node=next;
  } while(1);
  datanode(node)->nextlist=0;
  return data_mergesortlist(topfile,sortcriteria);
}

int listmaker,listreturn;

void data_makelistaddtolist(int node) {
  if(data_prevnode(node)) if(!data_isdir(node)) {
    if(!listreturn) {
      listreturn=node;
      listmaker=node;
      datanode(node)->prevlist=0;
    } else {
      datanode(listmaker)->nextlist=node;
      datanode(node)->prevlist=listmaker;
      listmaker=node;
    }
    if(data_matchfilespec(data_name(node))) {
      datanode(node)->shown=1;
      byteshown+=data_info(node).st_size;
      fileshown++;
    } else datanode(node)->shown=0;
  }
}

void data_makelistaddtolistiftagged(int node) {
  if(data_tagged(node)) data_makelistaddtolist(node);
}

int data_makelistfromtagged(int startnode) {
  byteshown=fileshown=0;
  listreturn=0;
  data_recursetree(startnode,data_makelistaddtolistiftagged);
  datanode(listmaker)->nextlist=0;
  return data_mergesortlist(listreturn,sortcriteria);
}

int data_makelistofbranch(int startnode) {
  byteshown=fileshown=0;
  listreturn=0;
  data_recursetree(startnode,data_makelistaddtolist);
  datanode(listmaker)->nextlist=0;
  return data_mergesortlist(listreturn,sortcriteria);
}

void data_restatloggeddirs(int topdir) {
  int node;
  node=topdir;
  data_restatdir(node);
  node=data_nextdir(data_downdir(node));
  while(node) {
    if(data_downdir(node)) data_restatloggeddirs(node);
    node=data_nextdir(node);
  }
}

void data_restatstuffiftouched(void) {
  switch(dispmode) {
  case DISP_TREE:
    if(config.autorestatdirs&&data_downdir(currentnode))
      data_restatiftouched(currentnode);
    if(config.autorestatfiles)
      data_restatnode(currentnode);
    break;
  case DISP_FILE:
    if(config.autorestatdirs&&(dispfilemode==D_FILE_NORM))
      data_restatiftouched(currentdir);
    if(config.autorestatfiles)
      data_restatnode(currentnode);
    break;
  }
}

void data_restatiftouched(int dir) {
  struct stat newinfo;
  if(stat(data_fullname(dir),&newinfo)) return;
  if(newinfo.st_ctime!=datanode(dir)->lpinfo.st_ctime) data_restatdir(dir);
}

void data_logdir(int num) {
  int a,b,patiencedelay=0;
  DIR *dir;
  struct dirent *ent;
  int tagstatus,maxlength;
#ifdef DEBUG
    fprintf(debug,"\ndata_logdir called: <%s>\n",data_fullname(num));
    fflush(debug);
#endif
  if(is_cancel()) return;
  if(!data_isdir(num)) crapout("data_logdir","logging nondir");

  tagstatus=data_tagged(num);

  if(data_downdir(num)) {
#ifdef DEBUG
    fprintf(debug,"relogging dir: <%s>\n",data_fullname(num));
    fflush(debug);
#endif
    data_restatdir(num);
    return;
  }

#ifdef DEBUG
  else fprintf(debug,"logging dir: <%s>\n",data_fullname(num));
  fflush(debug);
#endif

  maxlength=PATH_MAX-strlen(data_fullname(num))-1;
  if(!logbranch_recursions) fileskipped=0;

  dir=opendir(data_fullname(num));
  if(dir==NULL) {
    data_restatdir(data_updir(num));
    return;
  }
  gui_startpatience("logging...");
  a=data_findfree();
  datanode(num)->point=a;
  datanode(a)->referer=num;
  datanode(a)->depth=datanode(num)->depth+1;
  while(NULL!=(ent=readdir(dir))) {
    if(is_cancel()) break;
    if(!(patiencedelay++%200)) gui_updatepatience();
    if((!strcmp(ent->d_name,".")) || (!strcmp(ent->d_name,".."))) continue;
    if(strlen(ent->d_name)>maxlength) {fileskipped++; continue;}
    b=a;
    a=data_findfree();
#ifdef DEBUG
    fprintf(debug,"new node: %d name: <%s>",a,ent->d_name);
    fflush(debug);
#endif
    datanode(b)->next=a;
    datanode(a)->prev=b;
    datanode(a)->referer=num;
    strcpy(datanode(a)->name,ent->d_name);
    datanode(a)->depth=datanode(num)->depth+1;
#ifdef DEBUG
    fprintf(debug," fullname: <%s>\n",data_fullname(a));
    fflush(debug);
#endif
    data_restatnode_noresort(a);
    data_settag(a,tagstatus);
  }
  closedir(dir);
  gui_endpatience();
  data_mergesortdir(num,SORT_ALPHA);
#ifdef DEBUG
  data_dumptree();
#endif
  if(fileskipped&&!logbranch_recursions) gui_mildbeep(
    "Warning: %d item%s not logged because pathname was too long",
    fileskipped,(fileskipped>1)?"s":"");
  gui_synctocurrentnode_bar();
  gui_setwantdrawall();
}

char *data_resettreelines(char *in) {
  in=realloc(in,1);
  if(!in) outofmem("data_restattreelines");
  in[0]=0;
  return in;
}

char *data_nexttreelines(char *in,int node) {
  int a,d;
  a=data_nexttree(node);
  if(!a) {
    in=realloc(in,1);
    if(!in) outofmem("data_nexttreelines");
    in[0]=0;
    return in;
  }
  else {
    d=datanode(a)->depth;
    in=realloc(in,d+1);
    if(!in) outofmem("data_nexttreelines");
    if(d) in[d-1]=data_nextdir(a)?'>':'L';
    if(d>datanode(node)->depth) {
      if(d>1) in[d-2]=data_nextdir(node)?'|':' ';
    }
    in[d]=0;
    return in;
  }
}

char *data_prevtreelines(char *in,int node) {
  int a,d,od,n;
  a=data_prevtree(node);
  if(!a) {
    in=realloc(in,1);
    if(!in) outofmem("data_prevtreelines");
    in[0]=0;
    return in;
  }
  else {
    d=datanode(a)->depth;
    od=datanode(node)->depth;
    in=realloc(in,d+1);
    if(!in) outofmem("data_prevtreelines");
    if(d>od) {
      for(n=od;n<(d-1);n++) in[n]=' ';
      in[od-1]='|';
    }
    if(d) in[d-1]=data_nextdir(a)?'>':'L';
    in[d]=0;
    return in;
  }
}

char *data_updatetreelines(char *in,int node) {
  int a;
  in=data_resettreelines(in);
  a=0;
  do {
    if(a==node) return in;
    in=data_nexttreelines(in,a);
    a=data_nexttree(a);
  } while(a);
  in=data_resettreelines(in);
  return in;
}

/* real alphasort is not included on all systems */
int my_alphasort(const void *a,const void *b) {
  return strcmp(*(char **)a,*(char **)b);
}

/* real scandir is not included on all systems */
int my_scandir(const char *path,char ***names) {
  DIR *dir;
  struct dirent *ent;
  int num;

  *names=NULL;
  dir=opendir(path);
  if(!dir) return -1;
  for(num=0;;) {
    ent=readdir(dir);
    if(!ent) break;
    num++;
    *names=realloc(*names,num*sizeof(char *));
    if(!*names) {closedir(dir); return -1;}
    (*names)[num-1]=malloc(MAXNAMLEN+1);
    if(!(*names)[num-1]) {closedir(dir); return -1;}
    strcpy((*names)[num-1],ent->d_name);
  }
  qsort(*names,num,sizeof(char *),my_alphasort);
  closedir(dir);
  return num;
}

void my_scandir_free(char **names,int num) {
  int n;
  for(n=0;n<num;n++) free(names[n]);
  free(names);
}

/* restat contents of a dir */

void data_restatdir(int node) {
  int curnode,nextnode,prevnode,tnode,newnode,patiencedelay=0;
  int dirp,numdir;
  int tagstatus,maxlength;
  char **names;
  char *ent;
#ifdef DEBUG
    fprintf(debug,"\ndata_restatdir called: <%s>\n",data_fullname(node));
    fflush(debug);
#endif
  if(is_cancel()) return;

  tagstatus=data_tagged(node);

  if(!data_isdir(node)) {
    crapout("data_restatdir","restating nondir");
  }

  data_restatnode(node);
  if(!data_downdir(node)) {
    data_logdir(node);
    return;
  }
#ifdef DEBUG
  fprintf(debug,"restating dir: <%s>\n",data_fullname(node));
  fprintf(debug,"adding new files...\n");
  fflush(debug);
#endif

  gui_startpatience("restating...");

  maxlength=PATH_MAX-strlen(data_fullname(node))-1;
  if(!logbranch_recursions) fileskipped=0;

  /* first add new files to list */

  numdir=my_scandir(data_fullname(node),&names);
  if(numdir==-1) {
    if(!node) return;
    data_restatdir(data_updir(node));
    return;
  }
  dirp=0;
  tnode=data_downdir(node);
  curnode=tnode;
  while(dirp<numdir) {
    if(is_cancel()) break;
    ent=names[dirp];
    dirp++;
    if(!(patiencedelay++%50)) gui_updatepatience();
    if((!strcmp(ent,".")) || (!strcmp(ent,".."))) continue;
    if(strlen(ent)>maxlength) {fileskipped++; continue;}
    curnode=data_nextnode(curnode);
    if(strcmp(ent,data_name(curnode))) {
      curnode=tnode;
      do {
        curnode=data_nextnode(prevnode=curnode);
        if(!strcmp(ent,data_name(curnode))) break;
	if((strcmp(ent,data_name(curnode))<0) || !curnode) {
	  newnode=data_findfree();
	  strcpy(datanode(newnode)->name,ent);
	  if(curnode) datanode(curnode)->prev=newnode;
	  datanode(prevnode)->next=newnode;
	  datanode(newnode)->prev=prevnode;
	  datanode(newnode)->next=curnode;
	  datanode(newnode)->referer=node;
	  datanode(newnode)->depth=datanode(node)->depth+1;
          data_restatnode(newnode);
          data_unstatlog(newnode);
          data_settag(newnode,tagstatus);
          data_insertintolist(newnode);
          data_statlog(newnode);
	  break;
	}
      } while(curnode);
    }
  }
  my_scandir_free(names,numdir);

#ifdef DEBUG
  fprintf(debug,"updating old files...\n");
  fflush(debug);
#endif
  /* now kill old files from list (and update existing files) */

  curnode=data_downdir(node);
  curnode=data_nextnode(curnode);
  while(curnode) {
    if(is_cancel()) break;
    nextnode=data_nextnode(curnode);
    if(!(patiencedelay++%50)) gui_updatepatience();
    data_restatnode(curnode);
    curnode=nextnode;
  }

#ifdef DEBUG
  fprintf(debug,"restatdir finished.\n");
  data_dumptree();
#endif
  gui_endpatience();
  gui_setwantdrawall();
  if(fileskipped&&!logbranch_recursions) gui_mildbeep(
    "Warning: %d item%s not logged because pathname was too long",
    fileskipped,(fileskipped>1)?"s":"");
  gui_synctocurrentnode_bar();
  gui_setwantdrawall();
}

int data_restatfileifdirlogged(const char *fn) {
  char *p,*s;
  int dirnode;
  int newnode;
  s=malloc(strlen(fn)+1);
  if(!s) outofmem("data_restatfileifdirlogged");
  strcpy(s,fn);
  p=strrchr(s,'/');
  if(!p) crapout("data_restatfileifdirlogged","function needs absolute path (given: %s)",fn);
  *p=0; p++;
  newnode=0;
  if(s[0]) {
    dirnode=data_findfullname(s);
    if(data_downdir(dirnode))
      newnode=data_restatfilename(dirnode,p);
  } else newnode=data_restatfilename(0,p);
  free(s);
  return newnode;
}

/* restat a (possibly unlogged) file/dir by name */

int data_restatfilename(int dirnode,const char *fn) {
  int newnode,curnode,prevnode;
#ifdef DEBUG
  fprintf(debug,"restating file (name): <%s/%s>\n",data_fullname(dirnode),fn);
  fflush(debug);
#endif

  if(!data_downdir(dirnode)) data_logdir(dirnode);

  curnode=data_downdir(dirnode);
  do {
    curnode=data_nextnode(prevnode=curnode);
    if(!strcmp(fn,data_name(curnode))) {
      data_restatnode(curnode);
      break;
    }
    if((strcmp(fn,data_name(curnode))<0) || !curnode) {
      if((strlen(data_fullname(dirnode))+strlen(fn)+1)>PATH_MAX) {
        gui_mildbeep("Warning: file/dir not logged because pathname was too long");
        break;
      }
      newnode=data_findfree();
      strcpy(datanode(newnode)->name,fn);
      if(curnode) datanode(curnode)->prev=newnode;
      datanode(prevnode)->next=newnode;
      datanode(newnode)->prev=prevnode;
      datanode(newnode)->next=curnode;
      datanode(newnode)->referer=dirnode;
      datanode(newnode)->depth=datanode(dirnode)->depth+1;
      data_restatnode(newnode);
      data_settag(newnode,data_tagged(dirnode));
      curnode=newnode;
      data_insertintolist(newnode);
      break;
    }
  } while(curnode);

  gui_synctocurrentnode_bar();
#ifdef DEBUG
  data_dumptree();
#endif
  gui_setwantdrawall();
  return curnode;
}

int data_restatnode(int node) {
  if(data_restatnode_noresort(node)) return 1;
  data_updatesort(node);
  return 0;
}

int data_restatnode_noresort(int node) {
  struct stat ls,lp;
  const char *fn;
  int tagged;

#ifdef DEBUG
  fprintf(debug,"restating file (node): <%s>\n",data_fullname(node));
  fflush(debug);
  fprintf(debug,"before: ");
  data_dumpnode(node);
#endif

  tagged=datanode(node)->tagged;
  data_settag(node,0);
  data_unstatlog(node);
  fn=data_fullname(node);
  if(lstat(fn,&ls)) {
    if(data_isdir(node)) data_unlogdir(node);
    data_chopnode(node);
#ifdef DEBUG
    fprintf(debug,"after: node deleted");
#endif
    gui_setwantdrawall();
    return 1;
  } else {
    datanode(node)->lsinfo=ls;
    if(stat(fn,&lp)) {
      datanode(node)->lpinfo=ls;
      datanode(node)->isdir=0;
    } else {
      datanode(node)->lpinfo=lp;
      datanode(node)->isdir=S_ISDIR(ls.st_mode)?1:0;
      if(config.followlinkdirs)
        datanode(node)->isdir|=S_ISDIR(lp.st_mode)?1:0;
    }
    data_statlog(node);
    data_settag(node,tagged);
#ifdef DEBUG
    fprintf(debug,"after: ");
    data_dumpnode(node);
#endif
    gui_setwantdrawall();
    return 0;
  }
}

void data_insertintolist(int new) {
  int node,ok;

  ok=0;
  if((dispmode==DISP_FILE)&&(!data_isdir(new))) switch(dispfilemode) {
  case D_FILE_NONE: break;
  case D_FILE_NORM:
    if(data_updir(new)==currentdir) ok=1;
    break;
  case D_FILE_BRANCH:
    node=data_updir(new);
    do {
      if(node==currentdir) ok=1;
      node=data_updir(node);
    } while(node);
    break;
  case D_FILE_SHOWALL:
    ok=1;
    break;
  case D_FILE_TAGBRANCH: break;
  case D_FILE_TAGSHOWALL: break;
  default: crapout("data_insertintolist","unknown dispfilemode");
  }

  if(!ok) {
    datanode(new)->nextlist=datanode(new)->prevlist=0;
    return;
  }

  node=filelist;
  while(data_nextlist(node)&&
       (data_comparenodes(new,data_nextlist(node),sortcriteria)>0))
    node=data_nextlist(node);
  if((node==filelist)&&
     (data_comparenodes(new,node,sortcriteria)<0)) {
    datanode(new)->prevlist=0;
    datanode(new)->nextlist=filelist;
    datanode(filelist)->prevlist=new;
    filelist=new;
  } else {
    datanode(new)->nextlist=datanode(node)->nextlist;
    if(datanode(node)->nextlist)
      datanode(datanode(node)->nextlist)->prevlist=new;
    datanode(node)->nextlist=new;
    datanode(new)->prevlist=node;
  }
  if(data_matchfilespec(data_name(new))) {
    datanode(new)->shown=1;
  } else datanode(new)->shown=0;
  gui_synctocurrentnode_bar();
}

void data_deletefromlist(int node) {
  if(data_nextlist(node)||data_prevlist(node)||(filelist==node)) {
    if(datanode(node)->shown) {
      datanode(node)->shown=0;
    }
    if(data_nextlist(node))
      datanode(data_nextlist(node))->prevlist=data_prevlist(node);
    if(data_prevlist(node))
      datanode(data_prevlist(node))->nextlist=data_nextlist(node);
    datanode(node)->prevlist=datanode(node)->nextlist=0;
  }
}

int data_logtodir(const char *dirname) {
  char s[PATH_MAX+1],orig[PATH_MAX+1],*p,*q;
  int node=0,tmpnode,cmp;

  if(dirname==NULL) return 0;
  if(dirname[0]==0) return currentnode;
  if(!strcmp(dirname,"/")) {
    currentnode=currentdir=toptreedir=0;
    toptreelines=data_resettreelines(toptreelines);
    cursortreeloc=0;
    return 0;
  }
  strcpy(orig,comm_parsepath(comm_makeabsolute(dirname,currentdir)));
  strcpy(s,orig);
  s[PATH_MAX]=orig[PATH_MAX]=0;

  gui_startpatience("logging to dir...");
  p=s-1;
  while(strchr(p+1,'/')) *(p=strchr(p+1,'/'))=0;
  p=s;
  if(!data_downdir(0)) data_logdir(0);
  /* obfuscated code is just so fun... */
  while((q=strchr(orig-s+p,'/'))!=NULL) {
    if(is_cancel()) break;
    p=q-orig+s+1;
    if(!*p) break;

    tmpnode=data_downdir(node);
    if(!tmpnode) break;
    while(tmpnode) {
      currentnode=node=tmpnode;
      cmp=strcmp(data_name(tmpnode),p);
      if(cmp>=0) break;
      tmpnode=data_nextdir(tmpnode);
    }
    if(cmp) {
      gui_mildbeep("Error: exact match not found");
      break;
    }

    if(!data_downdir(node)) {
      if(config.animatedlogging) {
        cursortreeloc=0;
        gui_synctocurrentnode_bar_nice();
        gui_setwantdrawall();
        gui_drawmode();
      }
      data_logdir(node);
    }
  }
  cursortreeloc=0;
  gui_synctocurrentnode_bar_nice();
  gui_setwantdrawall();
  gui_endpatience();
  return node;
}

int lboldcn,lboldcd,lboldtd,lboldct;
char *lboldtl;

void data_logbranch(int node) {
  int n;

  if(!logbranch_recursions) {
    gui_startpatience("logging branch...");
    if(config.animatedlogging) {
      lboldcn=currentnode;
      lboldcd=currentdir;
      lboldtd=toptreedir;
      lboldtl=malloc(strlen(toptreelines)+1);
      if(!lboldtl) outofmem("data_logbranch");
      strcpy(lboldtl,toptreelines);
      lboldct=cursortreeloc;
    }
    fileskipped=0;
  }
  logbranch_recursions++;

  data_logdir(node);
  n=data_nextdir(data_downdir(node));
  while(n) {
    if(is_cancel()) break;
    if(config.animatedlogging) {
      currentnode=n;
      gui_synctocurrentnode_bar_nice();
      gui_setwantdrawall();
      gui_drawmode();
    }
    if(!data_islnk(n))  /* to avoid infinite loops (ex. file linked to "..") */
      data_logbranch(n);
    n=data_nextdir(n);
  }
  logbranch_recursions--;
  if(!logbranch_recursions) {
    if(config.animatedlogging) {
      if(!is_cancel()) {
        currentnode=lboldcn;
        currentdir=lboldcd;
        toptreedir=lboldtd;
        free(toptreelines);
        toptreelines=lboldtl;
        cursortreeloc=lboldct;
      } else free(lboldtl);
      gui_setwantdrawall();
      gui_drawmode();
    }
    gui_endpatience();
    if(fileskipped) gui_mildbeep(
      "Warning: %d item%s not logged because pathname was too long",
      fileskipped,(fileskipped>1)?"s":"");
  }
}

int data_findname(int node,const char *name) {
  node=data_downdir(node);
  node=data_nextnode(node);
  while(node) {
    if(!strcmp(data_name(node),name)) return node;
    node=data_nextnode(node);
  }
  return 0;
}

int data_findinlist(const char *name) {
  int node;
  node=filelist;
  while(node) {
    if(!strcmp(data_name(node),name)) return node;
    node=data_nextlist(node);
  }
  return 0;
}

char squish[PATH_MAX+1];

const char *data_squishfullname(const char *name) {
  int p1,p2,n;
  char s[PATH_MAX+1];
  const char *ss;
  const char *t;
  if(name[0]!='/') crapout("data_squishfullname","called without an absolute path");
  p1=1;
  p2=0;
  squish[0]=0;
  do {
    ss=strchr(name+p1,'/');
    if(ss) n=ss-name-p1; else n=strlen(name+p1);
    strncpy(s,name+p1,n);
    s[n]=0;
    p1+=n+1;
    if((!s[0]) || (!strcmp(s,"."))) {
      /* NOP */
    } else if(!strcmp(s,"..")) {
      t=strrchr(squish,'/');
      if(t) {
        p2=t-squish;
        squish[p2]=0;
      }
    } else {
      squish[p2++]='/';
      strncpy(squish+p2,s,n);
      p2+=n;
      squish[p2]=0;
    }
  } while(ss);
  if(!squish[0]) strcpy(squish,"/");
  return squish;
}

int data_findfullname(const char *name) {
  char s[PATH_MAX+1],*ss;
  int p,node;
  if(name[0]!='/') crapout("data_findfullname","called without a full name");
  if(strlen(name)>PATH_MAX) crapout("data_findfullname","name too long");
  strcpy(s,name);
  p=0;
  node=0;
  while(1) {
    while(name[p]=='/') p++;
    ss=strchr(s+p,'/');
    if(ss!=NULL) *ss=0;
    if(!data_isdir(node)) crapout("data_findfullname","called with a messed up path");
    node=data_findname(node,s+p);
    if(!node) return 0;
    if(ss==NULL) return node;
    p=ss-s;
  }
}

int data_subdir(int a,int b) {
  if(!a) return 1;
  do {
    if(a==b) return 1;
    b=data_updir(b);
  } while(b);
  return 0;
}

int unlogdirdepth=0,unlogdirloops;

void data_unlogdir(int node) {              /* unlogs contents of dir */
  int n,next;

  if(!unlogdirdepth) gui_startpatience("unlogging...");
  unlogdirdepth++;
  n=data_nextnode(data_downdir(node));
  while(n) {
    if(!((unlogdirloops++)%200)) gui_updatepatience();
    if(data_downdir(n)) data_unlogdir(n);
    next=data_nextnode(n);
    data_chopnode(n);
    n=next;
  }
  if(data_downdir(node)) data_clear(data_downdir(node));
  datanode(node)->point=0;
  unlogdirdepth--;
  if(!unlogdirdepth) gui_endpatience();
}

/* FIXME - don't need to do a full mergesort here */
void data_updatesort(int node) {
  if( (data_prevnode(node)&&(strcmp(data_name(node),data_name(data_prevnode(node)))<0)) ||
      (data_nextnode(node)&&(strcmp(data_name(node),data_name(data_nextnode(node)))>0)) ) {
    data_mergesortdir(data_updir(currentnode),SORT_ALPHA);
    gui_synctocurrentnode_bar();
  }
  if(data_nextlist(node)||data_prevlist(node)) {
    if(data_isdir(node)) crapout("data_updatesort","directory with next/prevlist set");
    if((data_prevlist(node)&&(data_comparenodes(node,data_prevlist(node),sortcriteria)<0)) ||
       (data_nextlist(node)&&(data_comparenodes(node,data_nextlist(node),sortcriteria)>0))) {
      filelist=data_mergesortlist(filelist,sortcriteria);
      gui_synctocurrentnode_bar();
    }
  }
}

int mergesortstart;

void data_movelistnode(int na,int nb) {
  int nl,nr;
  nl=datanode(na)->prevlist;
  nr=datanode(na)->nextlist;
  if(nl) datanode(nl)->nextlist=nr;
  if(nr) datanode(nr)->prevlist=nl;
  if(!nl) mergesortstart=nr;
  nl=datanode(nb)->prevlist;
  datanode(nb)->prevlist=na;
  if(nl) datanode(nl)->nextlist=na;
  if(!nl) mergesortstart=na;
  datanode(na)->prevlist=nl;
  datanode(na)->nextlist=nb;
}

void data_movedirnode(int na,int nb) {
  int nl,nr;
  nl=datanode(na)->prev;
  nr=datanode(na)->next;
  if(nl) datanode(nl)->next=nr;
  if(nr) datanode(nr)->prev=nl;
  if(!nl) datanode(datanode(nr)->referer)->point=nr;
  nl=datanode(nb)->prev;
  datanode(nb)->prev=na;
  if(nl) datanode(nl)->next=na;
  if(!nl) datanode(datanode(na)->referer)->point=na;
  datanode(na)->prev=nl;
  datanode(na)->next=nb;
}

int data_mergesortlist(int startnode,int method) {
  return data_mergesort(startnode,method,data_nextlist,data_prevlist,data_movelistnode);
}

void data_mergesortdir(int startnode,int method) {
  data_mergesort(data_nextnode(data_downdir(startnode)),method,data_nextnode,data_prevnode,data_movedirnode);
}

int data_mergesort(int startnode,int method,
         int (*nextfunc)(int),int (*prevfunc)(int),void (*movefunc)(int,int)) {
  int start[32],worth[32],depth,next;
  int patiencedelay=0;
#ifdef DEBUGMERGESORT
  int n;
#endif
  gui_startpatience("sorting...");
  mergesortstart=startnode;
  start[0]=startnode;
  worth[0]=1;
  depth=0;
  next=nextfunc(startnode);
  while(next) {
    depth++;
    if(depth==32) crapout("data_mergesort","mergesorting something really big");
    worth[depth]=1;
    start[depth]=next;
    next=nextfunc(next);
    while(worth[depth]==worth[depth-1]) {
#ifdef DEBUGMERGESORT
      fprintf(debug,"merge before:\n  ");
      for(n=0;n<=depth;n++) fprintf(debug,"(p:%d n:%d) ",start[n],worth[n]);
      fprintf(debug,"\n  ");
      fflush(debug);
      data_dumplinkage(datanode(dir)->point);
#endif
      if(!(patiencedelay++%500)) gui_updatepatience();
      data_mergelists(&worth[depth-1],&worth[depth],
		      &start[depth-1],&start[depth],
                      method,nextfunc,prevfunc,movefunc);
#ifdef DEBUGMERGESORT
      fprintf(debug,"merge after:\n  ");
      for(n=0;n<=depth;n++) fprintf(debug,"(p:%d n:%d) ",start[n],worth[n]);
      fprintf(debug,"\n  ");
      fflush(debug);
      data_dumplinkage(datanode(dir)->point);
#endif
      depth--;
      worth[depth]+=worth[depth+1];
    }
  }
  while(depth) {
#ifdef DEBUGMERGESORT
      fprintf(debug,"merge before:\n  ");
      for(n=0;n<=depth;n++) fprintf(debug,"(p:%d n:%d) ",start[n],worth[n]);
      fprintf(debug,"\n  ");
      fflush(debug);
      data_dumplinkage(datanode(dir)->point);
#endif
      if(!(patiencedelay++%50)) gui_updatepatience();
      data_mergelists(&worth[depth-1],&worth[depth],
		      &start[depth-1],&start[depth],
                      method,nextfunc,prevfunc,movefunc);
#ifdef DEBUGMERGESORT
      fprintf(debug,"merge after:\n  ");
      for(n=0;n<=depth;n++) fprintf(debug,"(p:%d n:%d) ",start[n],worth[n]);
      fprintf(debug,"\n  ");
      fflush(debug);
      data_dumplinkage(datanode(dir)->point);
#endif
    depth--;
    worth[depth]+=worth[depth+1];
  }
  gui_endpatience();
  return mergesortstart;
}

void data_mergelists(int *n1,int *n2,int *p1,int *p2,int method,
                     int (*nextfunc)(int),int (*prevfunc)(int),void (*movefunc)(int,int)) {
  int t,wa,wb,pa,pb;
  wa=*n1;
  wb=*n2;
  pa=*p1;
  pb=*p2;
  if(data_comparenodes(pa,pb,method)>0) {
    t=*n1; *n1=*n2; *n2=t;
    t=*p1; *p1=*p2; *p2=t;
  }
  while(wa && wb) {
    if(data_comparenodes(pa,pb,method)>0) {
      t=nextfunc(pb);
      movefunc(pb,pa);
      pb=t;
      wb--;
    } else {
      pa=nextfunc(pa);
      wa--;
    }
  }
}

char comparetmp[PATH_MAX+1];

int data_comparenodes(int na,int nb,int method) {
  int dir,ret;
  struct stat sta,stb;

  if(na==nb) return 0;
  sta=data_info(na);
  stb=data_info(nb);
  dir=method&SORT_DIRECTION;
  switch(method&~SORT_DIRECTION) {
  case SORT_ALPHA:
    ret=strcmp(data_name(na),data_name(nb));
    if(!ret) {
      strncpy(comparetmp,data_fullname(na),PATH_MAX+1);
      ret=strncmp(comparetmp,data_fullname(nb),PATH_MAX+1);
    }
    break;
  case SORT_PATH:  /* FIXME: should subdirs come before files? */
    strncpy(comparetmp,data_fullname(na),PATH_MAX+1);
    ret=strncmp(comparetmp,data_fullname(nb),PATH_MAX+1);
    break;
  case SORT_DATE:
    if(sta.st_mtime>stb.st_mtime) ret= 1; else
    if(sta.st_mtime<stb.st_mtime) ret=-1; else
      ret=data_comparenodes(na,nb,SORT_ALPHA);
    break;
  case SORT_SIZE:
    if(sta.st_size>stb.st_size) ret= 1; else
    if(sta.st_size<stb.st_size) ret=-1; else
      ret=data_comparenodes(na,nb,SORT_ALPHA);
    break;
  default:
    ret=0; /* keeps compiler happy */
    crapout("data_comparenodes","sorting with an unknown method");
  }
  if(!ret) crapout("data_comparenodes","comparing with identical results (%d,%d)",na,nb);
  return dir?(-ret):(ret);
}

void data_toplist(int dirnode,int method,int size,int list[]) {
  int n,curnode,tmp;
  for(n=0;n<size;n++) list[n]=0;
  curnode=data_nextfile(data_downdir(dirnode));
  while(curnode) {
    if(data_matchfilespec(data_name(curnode))) {
      if((!list[size-1])||(data_comparenodes(list[size-1],curnode,method)>0)) {
        list[size-1]=curnode;
        for(n=size-1;n;n--)
          if((!list[n-1])||(data_comparenodes(list[n-1],list[n],method)>0)) {
            tmp=list[n]; list[n]=list[n-1]; list[n-1]=tmp;
          }
      }
    }
    curnode=data_nextfile(curnode);
  }
}

#ifdef DEBUG
void data_dumplinkage(int node) {
  int cur;
  for(cur=node;cur;cur=datanode(cur)->next)
    fprintf(debug,"%d->",cur);
  fprintf(debug,"end\n");
  fflush(debug);
}
#endif
/*
int data_listpos(int num) {
  int n,m;
  n=filelist;
  if(n==num) return 0;
  if(!data_nextlist(num)&&!data_prevlist(num)) return -1;
  for(m=0;n;m++) {
    if(n==num) return m;
    n=data_nextlist(n);
  }
  return -1;
}

int data_treepos(int num) {
  int n,m;
  if(!data_isdir(num)) return 0;
  n=m=0;
  do {
    if(n==num) return m;
    m++;
    n=data_nexttree(n);
  } while(n);
  crapout("data_treepos","called on something not in tree");
  return 0;
}
*/
int data_nexttree(int num) {
  int n;
  n=num;
  if(datanode(n)->point) n=datanode(n)->point;
  do {
    if(data_nextdir(n)) return data_nextdir(n);
    n=datanode(n)->referer;
  } while(n);
  return 0;
}

int data_prevtree(int num) {
  int n,m;
  n=num;
  do
    n=datanode(n)->prev;
  while(n && !data_isdir(n));
  if(!n) return datanode(num)->referer;
  while(datanode(n)->point) {
    n=datanode(n)->point;
    m=0;
    while(datanode(n)->next) {
      n=datanode(n)->next;
      if(data_isdir(n)) m=n;
    }
    if(!m) return datanode(n)->referer;
    n=m;
  }
  return n;
}

int data_nextfile(int num) {
  do {
    num=data_nextnode(num);
    if(!num) break;
  } while(data_isdir(num));
  return num;
}

int data_nextdir(int num) {
  do {
    num=data_nextnode(num);
    if(!num) break;
  } while(!data_isdir(num));
  return num;
}

int data_prevfile(int num) {
  do {
    num=data_prevnode(num);
    if(!num) break;
  } while(data_isdir(num));
  if(!data_prevnode(num)) return 0;
  return num;
}

int data_prevdir(int num) {
  do {
    num=data_prevnode(num);
    if(!num) break;
  } while(!data_isdir(num));
  if(!data_prevnode(num)) return 0;
  return num;
}

void data_savetree(void) {
  FILE *f;
  int n;

  f=fopen("zzz-dumptree","w");
/* call nobug instead of crapout so tree won't be dumped again */
  if(!f) crapout_nobug("cannot even dump the filetree!!!");
  for(n=0;datanode(n)->referer!=-1;n++) {
    fwrite(datanode(n),sizeof(struct treedata_type),1,f);
  }
  fclose(f);
}

void data_restoretree(void) {
  FILE *f;
  int n,a;

  f=fopen("zzz-dumptree","r");
/* call nobug instead of crapout so tree won't be dumped again */
  if(!f) crapout_nobug("cannot open zzz-dumptree!");
  for(n=0;!feof(f);) {
    a=data_findfree();
/* call nobug instead of crapout so tree won't be dumped again */
    if(a!=n) crapout_nobug("findfree out of sequence in restoretree!");
    fread(datanode(a),sizeof(struct treedata_type),1,f);
    n++;
  }
  currentnode=0;
}

#ifdef DEBUG
void data_dumpnode(int n) {
  if(datanode(n)->referer==-1) {
    fprintf(debug,"node %d unused.\n",n);
    return;
  }
#ifdef LONGDUMPTREE
  fprintf(debug,"node: %d\n",n);
  fprintf(debug,"  referer : %d\n",datanode(n)->referer);
  fprintf(debug,"  next    : %d\n",datanode(n)->next);
  fprintf(debug,"  prev    : %d\n",datanode(n)->prev);
  fprintf(debug,"  point   : %d\n",datanode(n)->point);
  if(data_name(n)[0])
    fprintf(debug,"  name    : <%s>\n",data_name(n));
  else
    fprintf(debug,"  name    : -stub-\n");
  fflush(debug);
  fprintf(debug,"  fullname: <%s>\n",data_fullname(n));
  fprintf(debug,"  mode    : %07o\n",data_info(n).st_mode);
  fprintf(debug,"  isdir   : %c\n",data_isdir(n)?'y':'n');
  fflush(debug);
#else
  fprintf(debug,"node: %d name: <%s> full: <%s> dir: %c\n",
    n,data_name(n),data_fullname(n),data_isdir(n)?'y':'n');
  fflush(debug);
#endif
}

void data_dumptree(void) {
  int n;
  for(n=0;n<TREEBLOCKSIZE*TREEDATACHUNKS;n++) {
    if(treedata[(int)(n/TREEBLOCKSIZE)]!=NULL)
    if(datanode(n)->referer!=-1) {
      data_dumpnode(n);
    }
  }
}
#endif

#ifdef DEBUGNODE
void data_dumpused(void) {
  int i,j;
  fprintf(debug,"data_dumpused(): treehole: %d\nnodes used:",data_treehole);
  for(i=0;i<DEBUGNODE;i++) {
    fprintf(debug,"\n(block% 2d:% 3d free)<",i,treeholes[i]);
    if(treedata[i]==NULL) {
      for(j=0;j<TREEBLOCKSIZE;j++) fprintf(debug,"-");
    } else {
      for(j=0;j<TREEBLOCKSIZE;j++)
        fprintf(debug,(datanode(i*TREEBLOCKSIZE+j)->referer==-1)?"0":"1");
    }
    fprintf(debug,">");
  }
  fprintf(debug,"\n\n");
  fflush(debug);
}
#endif

int data_depth(int num) {
  return datanode(num)->depth;
}

int data_tagged(int num) {
  return datanode(num)->tagged;
}

void data_settag(int num,int val) {
  if((val==0)==(datanode(num)->tagged==0)) return;
  if(val) {
    bytestagged+=data_info(num).st_size;
    if(data_isdir(num)) dirstagged++;
    else                filestagged++;
  } else {
    bytestagged-=data_info(num).st_size;
    if(data_isdir(num)) dirstagged--;
    else                filestagged--;
  }
  datanode(num)->tagged=val;
}

void data_settagdir(int num,int val) {
  int a;
  gui_mildbeep("Error: tag dirs feature disabled for now"); /* FIXME - add this feature */
  return;
  data_settag(num,val);
  a=datanode(num)->point;
  if(!a) return;
  while((a=datanode(a)->next)) {
    if(datanode(a)->point) data_settagdir(a,val);
    else                   data_settag(a,val);
  }
}

char data_pathname[PATH_MAX+1];

const char *data_fullname(int num) {
  int n,p,l;
  p=PATH_MAX;
  n=num;
  data_pathname[PATH_MAX]=0;
  if(!n) {
    p--;
    data_pathname[p]='/';
  }
  while(n) {
    l=strlen(data_name(n));
    p-=l;
    if(p<1) crapout("data_fullname","looking at a really long pathname");
    memcpy(data_pathname+p,data_name(n),l);
    p--;
    data_pathname[p]='/';
    n=datanode(n)->referer;
  }
  return data_pathname+p;
}

void data_fixdepth(int node) {
  if(!node) datanode(node)->depth=0;
  else      datanode(node)->depth=datanode(datanode(node)->referer)->depth+1;
}

void data_recursetree(int startnode,void (*func)(int node)) {
  int curnode;
  func(startnode);
  if(data_isdir(startnode)) {
    curnode=datanode(startnode)->point;
    while(curnode) {
      data_recursetree(curnode,func);
      curnode=datanode(curnode)->next;
    }
  }
}

void data_recursedir(int startnode,void (*func)(int node)) {
  int curnode;
  if(data_isdir(startnode)) {
    curnode=datanode(startnode)->point;
    while(curnode) {
      func(curnode);
      curnode=datanode(curnode)->next;
    }
  } else crapout("data_recursedir","recursing a file instead of a dir");
}

int data_hasfile(int node) {
  node=data_downdir(node);
  if(!node) crapout("data_hasfile","called on nondir or unlogged dir");
  while((node=data_nextnode(node))) {
    if(!data_isdir(node)) return 1;
  }
  return 0;
}

int data_hasmatching(int node) {
  node=data_downdir(node);
  if(!node) crapout("data_hasfile","called on nondir or unlogged dir");
  while((node=data_nextnode(node))) {
    if(data_matchfilespec(data_name(node))&&!data_isdir(node)) return 1;
  }
  return 0;
}

int data_exists(const char *fn) {
  struct stat tmpst;
  /* FIXME - is this the proper way to test if a file exists? */
  return !lstat(fn,&tmpst);
}

/* Is fn a dir?  This function does not follow links. */
int data_fs_isdir_nolinks(const char *fn) {
  struct stat tmpst;
  return (!lstat(fn,&tmpst))&&S_ISDIR(tmpst.st_mode);
}

/* Is fn a dir?  This function does follow links. */
int data_fs_isdir_links(const char *fn) {
  char rname[PATH_MAX+1];
  if(strlen(fn)>PATH_MAX) crapout("data_fs_isdir_links","filename too long");
  if(!realpath(fn,rname)) strcpy(rname,fn);
  return data_fs_isdir_nolinks(rname);
}

int data_downdir (int node) { return datanode(node)->point;   }
int data_updir   (int node) { return datanode(node)->referer; }
int data_nextnode(int node) { return datanode(node)->next;    }
int data_prevnode(int node) { return datanode(node)->prev;    }
int data_nextlist(int node) { return datanode(node)->nextlist;}
int data_prevlist(int node) { return datanode(node)->prevlist;}

int data_firstshown(void) {
  if(datanode(filelist)->shown) return filelist;
  return data_nextshown(filelist);
}

int data_nextshown(int node) {
  int a=node;
  for(;;) {
    a=data_nextlist(a);
    if(!a) return 0;
    if(datanode(a)->shown) return a;
  }
}

int data_prevshown(int node) {
  int a=node;
  for(;;) {
    a=data_prevlist(a);
    if(!a) return 0;
    if(datanode(a)->shown) return a;
  }
}

int data_isdir(int num) {
  return datanode(num)->isdir;
}

int data_isregular(int num) {
  return (data_lpinfo(num).st_mode&S_IFMT)==S_IFREG;
}

int data_islnk(int num) {
  return ((datanode(num)->lsinfo.st_mode&S_IFMT)==S_IFLNK);
}

int data_goodlink(int num) {
  struct stat st;
  if(!data_islnk(num)) return 0;
  return !stat(data_fullname(num),&st);
}

const char *data_name(int num) { return datanode(num)->name; }

struct stat data_info(int num) {
  if(config.followlinks) return data_lpinfo(num);
  else                   return data_lsinfo(num);
}

struct stat data_lsinfo(int num) { return datanode(num)->lsinfo; }
struct stat data_lpinfo(int num) { return datanode(num)->lpinfo; }
