/* linuXtree copyright (c) 1998 Dan Stahlke under the GPL
   See file 'COPYING' for details */

#include "all.h"

#define ENABLECMDS
#define STRINGSIZ (1000+4*PATH_MAX)

extern int currentnode,currentdir;
extern WINDOW *gui_msbar;
extern WINDOW *gui_stats;
extern int maxx,maxy;
extern struct configtype config;
extern int filelist;
extern int dispmode,dispfilemode;
extern int sortcriteria;
extern int keep_msbar;
extern int myuid,mygid;
extern int byteshown,fileshown;
extern char filespec[];
extern int invfilespec;

char comm_s[STRINGSIZ],comm_ss[STRINGSIZ],comm_sss[STRINGSIZ],
     comm_ssss[STRINGSIZ];

void comm_init(void) {
  readln_init();
}

void comm_mount(void) {
  int c,node;
  const char *fn;
  const char *to;
#ifdef HAVE_MY_GETMNTENT
  FILE *mntf;
  struct my_mntent *ment;
  int maxs;
#endif
  char defdev[PATH_MAX+1],deftype[PATH_MAX+1],defopts[PATH_MAX+1];

  if(myuid) {
    gui_mildbeep("Error: only root can mount/unmount filesystems");
    return;
  }
  c=readln_getchar(gui_msbar,"(M)ount or (U)nmount? ","mu\a");
  if(c=='\a') {set_cancel(); return;}
  if(c=='u') {
    fn=data_fullname(currentnode);
    strcpy(comm_ss,fn);
#ifdef HAVE_MY_GETMNTENT
    mntf=my_setmntent(MNT_MTAB);
    if(mntf) {
      maxs=0;
      while((ment=my_getmntent(mntf))) {
        if(strlen(ment->mnt_dir)>maxs) {
          if(!strncmp(fn,ment->mnt_dir,strlen(ment->mnt_dir))) {
            maxs=strlen(ment->mnt_dir);
            strncpy(comm_ss,ment->mnt_dir,PATH_MAX);
            comm_ss[PATH_MAX]=0;
          }
        }
      }
      my_endmntent(mntf);
    }
#endif
    if(!strcmp(comm_ss,"/")) comm_ss[0]=0;
    if(comm_prompt(comm_s,PATH_MAX+1,comm_ss,0,"Mountpoint: ",FM_TABCOMPLETE,PV_DIRNAME)) return;
    if(!comm_s[0]) {set_cancel(); return;}
    to=comm_parsepath(comm_makeabsolute(comm_s,currentdir));
    if(!to) return;
    if(!strcmp(to,"/")) {
      gui_mildbeep("Error: cannot unmount root dir");
      return;
    }
    gui_printmsg("Unmounting...");
    if(comm_runcommand(NULL,"umount %a","a",to)) {
      gui_mildbeep("Error: umount failed");
      return;
    }
    node=data_findfullname(to);
    if(data_downdir(node)) data_restatdir(node);
    gui_printmsg("Done.");
    return;
  }
  if(comm_prompt(comm_s,PATH_MAX+1,data_fullname(currentnode),0,"Mountpoint: ",
      FM_TABCOMPLETE,PV_DIRNAME)) return;
  if(!comm_s[0]) {set_cancel(); return;}
  to=comm_parsepath(comm_makeabsolute(comm_s,currentdir));
  if(!to) return;
  deftype[0]=defdev[0]=defopts[0]=0;
#ifdef HAVE_MY_GETMNTENT
  mntf=my_setmntent(MNT_FSTAB);
  if(mntf) {
    while((ment=my_getmntent(mntf))) {
      if(!strcmp(to,ment->mnt_dir)) {
        strncpy(deftype,ment->mnt_type,PATH_MAX+1);
        strncpy(defdev,ment->mnt_fsname,PATH_MAX+1);
        strncpy(defopts,ment->mnt_opts,PATH_MAX+1);
        deftype[PATH_MAX]=defdev[PATH_MAX]=defopts[PATH_MAX]=0;
      }
    }
    my_endmntent(mntf);
  }
#endif
  if(comm_prompt(comm_ss,PATH_MAX+1,defdev,0,"Device: ",FM_TABCOMPLETE,PV_PATHNAME)) return;
  if(!comm_ss[0]) {set_cancel(); return;}
  if(comm_prompt(comm_sss,PATH_MAX+1,deftype,0,"Type: ",FM_TABCOMPLETE,PV_FSTYPE)) return;
  if(!comm_sss[0]) {set_cancel(); return;}
  if(comm_prompt(comm_ssss,PATH_MAX+1,defopts,0,"Flags: ",FM_TABCOMPLETE,PV_FSFLAG)) return;
  gui_printmsg("Mounting...");
  if(comm_runcommand(NULL,comm_ssss[0]?"mount -t %a -o %d %b %c":
                     "mount -t %a %b %c","abcd",comm_sss,comm_ss,to,
                     comm_ssss)) {
    gui_mildbeep("Error: mount failed");
  } else {
    node=data_findfullname(to);
    if(data_downdir(node)) data_restatdir(node);
    gui_printmsg("Done.");
  }
}

void comm_filespec(void) {
  if(comm_prompt(comm_s,PATH_MAX+1,NULL,0,"Filespec: ",FM_TABCOMPLETE,PV_FSPEC)) return;
  if(comm_s[0]) strcpy(filespec,comm_s); else strcpy(filespec,"*");
  invfilespec=0;
  comm_updatefilespec();
}

void comm_invertfilespec(void) {
  invfilespec=!invfilespec;
  comm_updatefilespec();
}

void comm_updatefilespec(void) {
  int a,new,old;
  int patiencedelay=0;
  if(dispmode==DISP_FILE) {
    a=filelist;
    gui_startpatience("matching...");
    while(a) {
      if(!(patiencedelay++%200)) gui_updatepatience();
      new=data_matchfilespec(data_name(a));
      old=datanode(a)->shown;
      if(new&&!old) {
        byteshown+=data_info(a).st_size;
        fileshown++;
      }
      if(old&&!new) {
        byteshown-=data_info(a).st_size;
        fileshown--;
      }
      datanode(a)->shown=new;
      a=data_nextlist(a);
    }
    gui_endpatience();
    if(!datanode(currentnode)->shown) {
      if(data_prevshown(currentnode)) currentnode=data_prevshown(currentnode);
      else
      if(data_nextshown(currentnode)) currentnode=data_nextshown(currentnode);
      else {
        currentnode=currentdir;
        gui_initdispmode(DISP_TREE);
      }
    }
    gui_synctocurrentnode_bar();
  }
}

void comm_tagbyfilespec(int state) {
  int node,num;
  const char *name;
  if(state) {
    if(comm_prompt(comm_s,PATH_MAX+1,"*",0,"Tag by filespec: ",FM_TABCOMPLETE,PV_FSPEC)) return;
  } else {
    if(comm_prompt(comm_s,PATH_MAX+1,"*",0,"Untag by filespec: ",FM_TABCOMPLETE,PV_FSPEC)) return;
  }
  if(!comm_s[0]) {set_cancel(); return;}
  gui_printmsg("%sagging *%s...",state?"T":"Unt",comm_s);
  num=0;
  for(node=data_firstshown();node;node=data_nextshown(node)) {
    name=data_name(node);
    if(!fnmatch(comm_s,name,0)) {
      data_settag(node,state);
      num++;
    }
  }
  gui_printmsg("%d file%s %stagged.",num,(num==1)?"":"s",state?"":"un");
}

void comm_editstatinfo(void) {
  struct stat st;
  int numid,n,ok,maxlen;
  char rname[PATH_MAX+1];

  if(currentnode&&uid_okdirop(currentnode)) {
    strcpy(comm_s,data_name(currentnode));
    strcpy(comm_ss,comm_s);
    maxlen=MIN(MAXNAMLEN,PATH_MAX-strlen(data_fullname(data_updir(currentnode))));
    if(dispmode==DISP_FILE) {
      if(readln_read(gui_stats,maxy-9,1,19,maxlen,comm_s,
                     FM_TABNEXT,PV_NOPATHFILENAME)) return;
    } else {
      if(readln_read_updir(gui_stats,maxy-9,1,19,maxlen,comm_s,
                           FM_TABNEXT,PV_NOPATHDIRNAME)) return;
    }
    if(strcmp(comm_s,comm_ss)) {
      strcpy(comm_ss,data_fullname(currentnode));
      strcpy(strrchr(comm_ss,'/')+1,comm_s);
      if(data_exists(comm_ss)) {
        gui_mildbeep("Error: name already exists");
        gui_drawnodestatus(currentnode);
      } else {
        /* restat in case destination file is logged but no longer there */
        data_restatfileifdirlogged(comm_ss);
        if(rename(data_fullname(currentnode),comm_ss)) {
          gui_mildbeep("Error: rename failed (%s)",strerror(errno));
        } else {
          strcpy(datanode(currentnode)->name,comm_s);
          data_updatesort(currentnode);
        }
        gui_setwantdraw();
        gui_drawmode();
      }
    } else gui_drawnodestatus(currentnode);
  }

  if((!config.followlinks)||(!realpath(data_fullname(currentnode),rname)))
    strcpy(rname,data_fullname(currentnode));
  if(lstat(rname,&st)) {
    gui_mildbeep("Error: lstat failed");
    return;
  }

  if((!S_ISLNK(st.st_mode))&&((!myuid)||(myuid==st.st_uid))) {
    sprintf(comm_s,"%04o",(unsigned int)(st.st_mode&0xfff));
    strcpy(comm_ss,comm_s);
    if(readln_read(gui_stats,maxy-7,1,10,5,comm_s,FM_TABNEXT,PV_UMASK)) return;
    if(strcmp(comm_s,comm_ss)) {
      for(ok=1,n=0;comm_s[n];n++) ok=ok&&(comm_s[n]>='0')&&(comm_s[n]<='7');
      if(ok) {
        sscanf(comm_s,"%o",&numid);
        st.st_mode&=~0xfff;
        st.st_mode|=numid&0xfff;
        if(-1==chmod(rname,st.st_mode))
          gui_mildbeep("Error: chmod failed (%s)",strerror(errno));
        if(data_restatnode(currentnode)) return;
        if(lstat(rname,&st)) {
          gui_mildbeep("Error: lstat failed");
          return;
        }
        gui_drawmode();
      } else gui_drawnodestatus(currentnode);
    } else gui_drawnodestatus(currentnode);
  }

  if(!myuid) {
    strcpy(comm_s,uid_uidtoname(st.st_uid));
    strcpy(comm_ss,comm_s);
    if(readln_read(gui_stats,maxy-6,1,9,9,comm_s,FM_TABNEXT,PV_UNAME)) return;
    if(strcmp(comm_s,comm_ss)) {
      numid=uid_nametouid(comm_s);
      if(numid!=-1) {
        if(-1==lchown(rname,numid,st.st_gid))
          gui_mildbeep("Error: chown failed (%s)",strerror(errno));
        if(data_restatnode(currentnode)) return;
        if(lstat(rname,&st)) {
          gui_mildbeep("Error: lstat failed");
          return;
        }
        gui_drawmode();
      } else gui_drawnodestatus(currentnode);
    } else gui_drawnodestatus(currentnode);
  }

  if((!myuid)||(myuid==st.st_uid)) {
    strcpy(comm_s,uid_gidtoname(st.st_gid));
    strcpy(comm_ss,comm_s);
    if(readln_read(gui_stats,maxy-6,10,9,9,comm_s,FM_TABNEXT,PV_GNAME)) return;
    if(strcmp(comm_s,comm_ss)) {
      numid=uid_nametogid(comm_s);
      if(numid!=-1) {
        if(-1==lchown(rname,st.st_uid,numid))
          gui_mildbeep("Error: chown failed (%s)",strerror(errno));
        if(data_restatnode(currentnode)) return;
        if(lstat(rname,&st)) {
          gui_mildbeep("Error: lstat failed");
          return;
        }
        gui_drawmode();
      } else gui_drawnodestatus(currentnode);
    } else gui_drawnodestatus(currentnode);
  }
}

void comm_symlink(void) {
  const char *to;
  if(comm_prompt(comm_sss,MAXNAMLEN,NULL,0,
     data_isdir(currentnode)?"Link dir to: ":"Link file to: ",
     FM_TABCOMPLETE,PV_PATHNAME)) return;
  if(!comm_sss[0]) {set_cancel(); return;}
  to=comm_parsepath(comm_makeabsolute(comm_sss,currentdir));
  if(!to) return;
  if(data_fs_isdir_nolinks(to)) {
    sprintf(comm_s,"%s/%s",to,data_name(currentnode));
  } else strcpy(comm_s,to);
  if(!comm_askoverwrite(comm_s)) return;

  gui_printmsg("Linking...");
  if(comm_runcommand(NULL,"ln -sf %a %b","ab",data_fullname(currentnode),comm_s)) {
    gui_mildbeep("Error: ln -sf failed");
  } else {
    data_restatfileifdirlogged(comm_s);
    gui_printmsg("Done");
  }
}

void comm_exec(void) {
  int good_curdir;
  char curdir[PATH_MAX+1];

  comm_doublechecknode();
  if(!data_isregular(currentnode))
    comm_ss[0]=0;
  else
    sprintf(comm_ss,"./%s",data_name(currentnode));
  if(comm_prompt(comm_s,PATH_MAX+1,comm_ss,0,"Execute: ",FM_TABCOMPLETE,PV_CMDLINE)) return;
  if(!comm_s[0]) {set_cancel(); return;}

#ifdef HAVE_GETCWD
  good_curdir=(NULL!=getcwd(curdir,PATH_MAX+1));
#else
  good_curdir=0;
#endif
  if(data_isdir(currentnode)) chdir(data_fullname(currentnode));
  else                        chdir(data_fullname(data_updir(currentnode)));
  comm_runexternal(comm_s,1);
  if(good_curdir) chdir(curdir);
}

void comm_archive(void) {
  const char *to;
  switch(dispmode) {
  case DISP_TREE:
    switch(readln_getchar(gui_msbar,"Archive which? (b:branch,t:tagged,s:tagged in branch)","bts\a")) {
    case 'b':
      if(!config.archivebranch[0]) {
        gui_mildbeep("Error: no command defined for archive branch");
        return;
      }
      if(comm_prompt(comm_s,PATH_MAX+1,NULL,0,"Archive filename: ",FM_TABCOMPLETE,PV_PATHNAME)) return;
      if(!comm_s[0]) {set_cancel(); return;}
      to=comm_parsepath(comm_makeabsolute(comm_s,currentdir));
      if(!to) return;
      if(!comm_askoverwrite(to)) return;
      strcpy(comm_ss,to);
      strcpy(comm_sss,data_fullname(currentnode));
      gui_printmsg("Running tar...");
      if(comm_runcommand("Archive: ",config.archivebranch,"ab",comm_ss,comm_sss)) {
        gui_mildbeep("Error: archive failed");
      } else {
        gui_printmsg("Done.");
      }
      data_restatfileifdirlogged(comm_ss);
      break;
    case 't':
      comm_archive_tree_tagged(0);
      break;
    case 's':
      comm_archive_tree_tagged(currentnode);
      break;
    default:
      set_cancel();
      return;
    }
    break;
  case DISP_FILE:
    crapout("comm_archive","archive unimplemented in file mode");
    break;
  }
}

void comm_archive_tree_tagged(int rootnode) {
  int a,f;
  int oldsort;
  char *p;
  const char *to;
  char namebuf[]="/tmp/lxt-XXXXXX"; /* FIXME - find real /tmp */

  if(!config.archivetagged[0]) {
    gui_mildbeep("Error: no command defined for archive tagged");
    return;
  }
  oldsort=sortcriteria;
  sortcriteria=SORT_PATH;
  filelist=data_makelistfromtagged(rootnode);
  sortcriteria=oldsort;
  if(!data_firstshown()) {
    if(!rootnode)
      gui_mildbeep("Error: no tagged files");
    else
      gui_mildbeep("Error: no tagged files in branch");
    return;
  }

  if(comm_prompt(comm_s,PATH_MAX+1,NULL,0,"Archive filename: ",FM_TABCOMPLETE,PV_PATHNAME)) return;
  if(!comm_s[0]) {set_cancel(); return;}
  to=comm_parsepath(comm_makeabsolute(comm_s,currentdir));
  if(!to) return;
  if(!comm_askoverwrite(to)) return;

  f=mkstemp(namebuf);
  if(f==-1) {
    gui_mildbeep("Error: cannot create temp file (%s)",strerror(errno));
    return;
  }

  if(rootnode) sprintf(comm_sss,"%s/",data_fullname(rootnode));
  else         comm_sss[0]=0;
  a=data_firstshown();
  while(a) {
    sprintf(comm_ss,"%s\n",data_fullname(a));
    if(strncmp(comm_ss,comm_sss,strlen(comm_sss)))
      crapout("comm_archive_tree_tagged","messed up root node in archive");
    p=comm_ss+strlen(comm_sss);
    write(f,p,strlen(p));
    a=data_nextshown(a);
  }

  close(f);
  data_breaklist(filelist);
  strcpy(comm_ss,to);
  strcpy(comm_sss,data_fullname(rootnode));
  gui_printmsg("Running tar...");
  if(comm_runcommand("Archive: ",config.archivetagged,"abc",
                     comm_ss,comm_sss,namebuf)) {
    gui_mildbeep("Error: archive failed");
  } else {
    gui_printmsg("Done.");
  }
  data_restatfileifdirlogged(comm_ss);
  unlink(namebuf);
}

void comm_mvbranch(void) {
  int tonode,new,a;
  const char *to;

  comm_doublechecknode();
  if(!strcmp(data_fullname(currentnode),"/")) {
    gui_mildbeep("Error: cannot graft root dir");
    return;
  }
  if(comm_prompt_updir(comm_sss,MAXNAMLEN,data_name(currentnode),0,"Move as: ",
                       FM_TABCOMPLETE,PV_NOPATHDIRNAME)) return;
  if(!comm_sss[0]) {set_cancel(); return;}

  if(comm_prompt(comm_s,PATH_MAX+1,data_fullname(data_updir(currentnode)),0,
                 "To path: ",FM_TABCOMPLETE,PV_DIRNAME)) return;
  if(!comm_s[0]) {set_cancel(); return;}
  to=comm_parsepath(comm_s);
  if(!to) return;
  if(comm_verifydir(to)) return;
  sprintf(comm_s,"%s/%s",to,comm_sss);

  if(comm_checksame(comm_s,data_fullname(currentnode))) {
    gui_mildbeep("Error: source and destination are the same");
    return;
  }

#ifdef ENABLECMDS
  if(rename(data_fullname(currentnode),comm_s)) {
    gui_printmsg_wait("Error: graft failed (%s)",errno,strerror(errno));
  } else {
    tonode=data_findfullname(to);
    if((tonode||!strcmp(to,"/"))&&data_downdir(tonode)) {
      new=data_restatfilename(tonode,comm_sss);
      if(!new) {
        /* theoretically, this never happens */
        gui_mildbeep("Error: cannot find the result of the move");
        return;
      }
      datanode(new)->point=datanode(currentnode)->point;
      a=data_downdir(new);
      while(a) {
        datanode(a)->referer=new;
        a=datanode(a)->next;
      }
      datanode(currentnode)->point=0;
      data_recursetree(new,data_fixdepth);
      if(config.preservetag)
        data_settag(new,datanode(currentnode)->tagged);
    }
    a=currentnode;
    data_restatnode(a);
    if(datanode(a)->referer!=-1) crapout("comm_mvbranch","old dir still remains after move");

    data_logtodir(data_squishfullname(comm_s));
  }
#endif
}

void comm_rmbranch(void) {
  int n;
  if(comm_prompt(comm_s,6,NULL,0,"Type PRUNE to rm -rf this directory: ",
      FM_TABCOMPLETE,PV_DISABLE)) return;
  for(n=0;comm_s[n];n++) comm_s[n]=toupper((int)comm_s[n]);
  if(strcmp(comm_s,"PRUNE")) {
    set_cancel();
    return;
  }
  gui_printmsg("Running rm -rf...");
  if(comm_runcommand(NULL,"rm -rf %a","a",data_fullname(currentnode))) {
    data_restatnode(currentnode);
    gui_mildbeep("Error: rm -rf failed");
  } else {
    data_restatnode(currentnode);
    gui_printmsg("Done.");
  }
}

void comm_chdir(void) {
  const char *to;
  if(!currentnode) sprintf(comm_ss,"/");
  else sprintf(comm_ss,"%s/",data_fullname(currentnode));
  if(comm_prompt(comm_s,PATH_MAX+1,comm_ss,0,"Chdir to: ",
                 FM_TABCOMPLETE,PV_DIRNAME)) return;
  if(!comm_s[0]) {set_cancel(); return;}
  to=comm_parsepath(comm_makeabsolute(comm_s,currentdir));
  if(!to) return;
  data_logtodir(to);
}

void comm_gotofile(void) {
  int node,cmp;

  if(comm_prompt(comm_s,MAXNAMLEN,NULL,0,"Goto file: ",
                 FM_TABCOMPLETE,PV_NOPATHFILENAME)) return;
  if(!comm_s[0]) {set_cancel(); return;}
  if(!filelist) crapout("comm_gotofile","filelist is zero");

  if((sortcriteria&~SORT_DIRECTION)==SORT_ALPHA) {
    node=filelist;
    if(!node) crapout("comm_gotofile","filelist is zero");
    cmp=0; /* to prevent warning from gcc */
    while(node) {
      currentnode=node;
      cmp=strcmp(data_name(node),comm_s);
      if(sortcriteria&SORT_DIRECTION) {if(cmp<=0) break;}
      else                            {if(cmp>=0) break;}
      node=data_nextlist(node);
    }
    if(cmp) gui_mildbeep("Error: exact match not found");
  } else {
    node=data_findinlist(comm_s);
    if(node) currentnode=node;
    else gui_mildbeep("Error: file not found");
  }
  gui_synctocurrentnode_bar_nice();
}

void comm_tag_grep(void) {
  char pattern[REGEX_SIZ];
  if(!comm_existtag()) {
    gui_mildbeep("Error: no tagged files");
    return;
  }
  if(comm_prompt(pattern,REGEX_SIZ,NULL,0,"Grep tagged files for: ",
      FM_TABCOMPLETE,PV_SEARCH)) return;
  if(!pattern[0]) {
    set_cancel();
    return;
  }
  comm_tag_foreach(comm_greptagged,NULL,pattern);
}

int comm_greptagged(int node,va_list argp) {
  char *pattern;
  pattern=va_arg(argp,char *);
  gui_printmsg("Grepping...");
  if(comm_runcommand(NULL,"grep %a %b","ab",pattern,data_fullname(node))) {
    data_settag(node,0);
  }
  return 0;
}

void comm_tag_chown(void) {
  int maxx,maxy,uid,gid;
  char owner[9],group[9];
  owner[0]=group[0]=0;

  if(!comm_existtag()) {
    gui_mildbeep("Error: no tagged files");
    return;
  }
  werase(gui_msbar);
  mvwprintw(gui_msbar,0,1,"Chown tagged files to Owner: --------- Group:");
  wnoutrefresh(gui_msbar);
  doupdate();
  getmaxyx(gui_msbar, maxy, maxx);
  if(readln_read(gui_msbar,0,30,9,9,owner,FM_TABCOMPLETE,PV_UNAME)) {
    werase(gui_msbar);
    wnoutrefresh(gui_msbar);
    doupdate();
    set_cancel();
    return;
  }
  if(owner[0]) {
    uid=uid_nametouid(owner);
    if(uid==-1) {
      gui_mildbeep("Error: user %s not found",owner);
      return;
    }
  } else uid=-1;
  mvwprintw(gui_msbar,0,30,"%-8.8s ",owner);

  if(readln_read(gui_msbar,0,47,9,9,group,FM_TABCOMPLETE,PV_GNAME)) {
    werase(gui_msbar);
    wnoutrefresh(gui_msbar);
    doupdate();
    set_cancel();
    return;
  }
  if(group[0]) {
    gid=uid_nametogid(group);
    if(gid==-1) {
      gui_mildbeep("Error: group %s not found",group);
      return;
    }
  } else gid=-1;

  werase(gui_msbar);
  wnoutrefresh(gui_msbar);
  doupdate();

  if((uid==-1)&&(gid==-1)) {
    set_cancel();
    return;
  }

  comm_tag_foreach(comm_chowntagged,NULL,uid,gid);
}

int comm_chowntagged(int node,va_list argp) {
  int uid,gid;
  char fn[PATH_MAX+1];

  uid=va_arg(argp,int);
  gid=va_arg(argp,int);

  if(config.followlinks) {
    if(!realpath(data_fullname(node),fn)) {
/* FIXME - should chown link itself? */
      gui_printmsg_wait("Error: bad link (%s)",strerror(errno));
      return -1;
    }
    chown(fn,uid,gid);
  } else {
    lchown(data_fullname(node),uid,gid);
  }
  return 1;
}

void comm_tag_chmod(void) {
  char form;
  char smask[5];
  int mask,n;

  if(!comm_existtag()) {
    gui_mildbeep("Error: no tagged files");
    return;
  }
  form=readln_getchar(gui_msbar,
    "Chmod tagged files how? <p>lus bits, <m>inus bits, <s>et mode","pms\a");
  if(form=='\a') {
    set_cancel();
    return;
  }
  if(comm_prompt(smask,5,NULL,0,"Mask in octal: ",FM_TABCOMPLETE,PV_UMASK)||
      (!smask[0])) {
    set_cancel();
    return;
  }
  for(mask=n=0;smask[n];n++) {
    if((smask[n]>='0')&&(smask[n]<='7')) mask=(mask*8)+(smask[n]-'0');
    else {
      gui_mildbeep("Error: invalid mask");
      return;
    }
  }
  comm_tag_foreach(comm_chmodtagged,NULL,form,mask);
}

int comm_chmodtagged(int node,va_list argp) {
  char form;
  int mask,newmode;
  char fn[PATH_MAX+1];
  struct stat st;

  if(data_islnk(node)) {
    if(!config.followlinks) {
      gui_printmsg_wait("Error: cannot chmod a link");
      return -1;
    }
    if(!realpath(data_fullname(node),fn)) {
/* FIXME - should give cannot chmod link error? */
      gui_printmsg_wait("Error: bad link (%s)",strerror(errno));
      return -1;
    }
  } else {
    strcpy(fn,data_fullname(node));
  }

  form=va_arg(argp,char);
  mask=va_arg(argp,int);

  if(lstat(fn,&st)) {
    gui_printmsg_wait("Error: lstat failed (%s)",strerror(errno));
    return -1;
  }
  switch(form) {
  case 'p':
    newmode=st.st_mode|mask;
    break;
  case 'm':
    newmode=st.st_mode&(~mask);
    break;
  case 's':
    newmode=(st.st_mode&(~0xfff))|mask;
    break;
  default:
    crapout("comm_chmodtagged","invalid mask form (%c)",form);
    newmode=0; /* to keep compiler happy */
  }
  chmod(fn,newmode);

  return 1;
}

void comm_tag_rmfile(void) {
  comm_tag_foreach(comm_delete_noask,"Delete file? ");
}

int comm_delete_noask(int node, va_list argp) {
#ifdef ENABLECMDS
  if(unlink(data_fullname(node))) {
    gui_printmsg_wait("Error: delete failed (%s)",strerror(errno));
    return -1;
  }
#endif
  return 1;
}

void comm_tag_cpfile(void) {
  int overwrite;
  const char *to;
  if(!comm_existtag()) {
    gui_mildbeep("Error: no tagged files");
    return;
  }
  if(dispfilemode==D_FILE_NORM) {
    if(comm_prompt(comm_s,PATH_MAX+1,
        data_fullname(data_updir(currentnode)),0,"Copy files to: ",
        FM_TABCOMPLETE,PV_DIRNAME)) return;
  } else {
    if(comm_prompt(comm_s,PATH_MAX+1,NULL,0,"Copy files to: ",
        FM_TABCOMPLETE,PV_DIRNAME)) return;
  }
  if(!comm_s[0]) {set_cancel(); return;}
  to=comm_parsepath(comm_s);
  if(!to) return;
  if(dispfilemode==D_FILE_NORM) 
    if(comm_checksame(to,data_fullname(data_updir(currentnode)))) {
      gui_mildbeep("Error: source and destination are the same");
      return;
    }

  if(comm_verifydir(to)) return;

  switch(readln_getchar(gui_msbar,"Automatically overwrite existing files? ","yn\a")) {
    case 'y': overwrite=1; break;
    case 'n': overwrite=0; break;
    default: set_cancel(); return;
  }
  comm_tag_foreach(comm_cptagged,NULL,to,overwrite);
}

int comm_cptagged(int node, va_list argp) {
  const char *to;
  int overwrite;

  to=va_arg(argp,char *);
  if(to==NULL) crapout("comm_cptagged","called with no destination");

  sprintf(comm_s,"%s/%s",to,data_name(node));

  overwrite=va_arg(argp,int);
  if((!overwrite)&&(!comm_askoverwrite(comm_s))) return 0;

  if(strcmp(comm_s,data_fullname(node))) {
    gui_printmsg("Copying %s...",data_name(node));
    if(comm_runcommand(NULL,"cp %a %b","ab",data_fullname(node),comm_s)) {
      gui_printmsg_wait("Error: cp failed");
      return -1;
    }
  } else {
    gui_printmsg_wait("Error: cannot copy file over itself");
    return -1;
  }
  data_restatfileifdirlogged(comm_s);
  return 0;
}

void comm_tag_mvfile(void) {
  int overwrite;
  const char *to;
  if(!comm_existtag()) {
    gui_mildbeep("Error: no tagged files");
    return;
  }
  if(dispfilemode==D_FILE_NORM) {
    if(comm_prompt(comm_s,PATH_MAX+1,
        data_fullname(data_updir(currentnode)),0,"Move files to: ",
        FM_TABCOMPLETE,PV_DIRNAME)) return;
  } else {
    if(comm_prompt(comm_s,PATH_MAX+1,NULL,0,"Move files to: ",
        FM_TABCOMPLETE,PV_DIRNAME)) return;
  }
  if(!comm_s[0]) {set_cancel(); return;}
  to=comm_parsepath(comm_s);
  if(!to) return;
  if(dispfilemode==D_FILE_NORM) 
    if(comm_checksame(to,data_fullname(data_updir(currentnode)))) {
      gui_printmsg("Error: source and destination are the same");
      return;
    }

  if(comm_verifydir(to)) return;

  switch(readln_getchar(gui_msbar,"Automatically overwrite existing files? ","yn\a")) {
    case 'y': overwrite=1; break;
    case 'n': overwrite=0; break;
    default: set_cancel(); return;
  }
  comm_tag_foreach(comm_mvtagged,NULL,to,overwrite);
}

int comm_mvtagged(int node, va_list argp) {
  const char *to;
  int newnode;
  int overwrite;

  to=va_arg(argp,char *);
  if(to==NULL) crapout("comm_mvtagged","called with no destination");

  sprintf(comm_s,"%s/%s",to,data_name(node));

  overwrite=va_arg(argp,int);
  if((!overwrite)&&(!comm_askoverwrite(comm_s))) return 0;

#ifdef ENABLECMDS
  if(rename(data_fullname(node),comm_s)) {
    gui_printmsg_wait("Error: move failed (%s)",strerror(errno));
    return -1;
  }
#endif
  newnode=data_restatfileifdirlogged(comm_s);
  if(config.preservetag&&newnode) data_settag(newnode,1);
  return 1;
}

int comm_existtag(void) {
  int a;
  a=data_firstshown();
  while(a) {
    if(data_tagged(a)) break;
    a=data_nextshown(a);
  }
  return a;
}

void comm_tag_foreach(int (*func)(int node, va_list argp),const char *confstr, ...) {
  int a,b,okdo,confirm,ret;
  int errors=0;
  va_list argp;
  a=data_firstshown();
  while(a) {
    if(data_tagged(a)) break;
    a=data_nextshown(a);
  }
  if(!a) {
    gui_mildbeep("Error: no tagged files");
    return;
  }
  if(confstr) confirm=comm_yesno("Confirm for each file? ");
  else        confirm=0;
  while(a) {
    if(is_cancel()) break;
    b=data_nextshown(a);
    if(data_tagged(a)&&uid_okdirop(a)) {
      if(gui_positioncursor(a)) crapout("comm_tag_foreach","error positioning the cursor onto a file");
      if(confirm) {
        okdo=comm_yesno(confstr);
        if(is_cancel()) break;
      } else okdo=1;
      if(okdo) {
        va_start(argp,confstr);
        ret=func(a,argp);
        if(ret) data_restatnode(a);
        if(ret==-1) errors++;
        va_end(argp);
      }
    }
    a=b;
  }
  gui_setwantdrawall();
  gui_drawmode();
  if(!is_cancel()) {
    if(errors) gui_mildbeep("Done: %d error%s occured during this operation",errors,(errors>1)?"s":"");
    else       gui_printmsg("Done.");
  }
}

void comm_makedir(void) {
  comm_doublechecknode();
  if(comm_prompt(comm_s,MAXNAMLEN,NULL,0,"Enter name for new dir: ",
                 FM_TABCOMPLETE,PV_NOPATHDIRNAME)) return;
  if(!comm_s[0]) {set_cancel(); return;}
  sprintf(comm_ss,"%s/%s",data_fullname(currentnode),comm_s);
#ifdef ENABLECMDS
  if(mkdir(comm_ss,0x1ff)) {
    gui_mildbeep("Error: mkdir failed (%s)",strerror(errno));
    return;
  }
#endif
  data_restatfilename(currentnode,comm_s);
}

void comm_rmdir(void) {
  comm_doublechecknode();
  if(comm_yesno("Delete dir? "))
#ifdef ENABLECMDS
    if(rmdir(data_fullname(currentnode))) {
      gui_mildbeep("Error: rmdir failed (%s)",strerror(errno));
    }
#endif
  data_restatnode(currentnode);
}

void comm_rmfile(void) {
  char *s;
  comm_doublechecknode();
  if(data_islnk(currentnode)) s="Delete link? ";
                         else s="Delete file? ";
  if(comm_yesno(s)) {
#ifdef ENABLECMDS
    if(unlink(data_fullname(currentnode))) {
      gui_mildbeep("Error: delete failed (%s)",strerror(errno));
    }
#endif
    data_restatnode(currentnode);
  }
}

void comm_cpfile(void) {
  const char *to;

  comm_doublechecknode();
  if(comm_prompt(comm_sss,MAXNAMLEN,data_name(currentnode),0,"Copy as: ",
                 FM_TABCOMPLETE,PV_NOPATHFILENAME)) return;
  if(!comm_sss[0]) {set_cancel(); return;}
  if(comm_prompt(comm_s,PATH_MAX+1,data_fullname(data_updir(currentnode)),0,
       "Copy to: ",FM_TABCOMPLETE,PV_DIRNAME)) return;
  if(!comm_s[0]) {set_cancel(); return;}
  to=comm_parsepath(comm_makeabsolute(comm_s,currentdir));
  if(!to) return;
  if(comm_verifydir(to)) return;

  sprintf(comm_s,"%s/%s",to,comm_sss);
  if(comm_checksame(comm_s,data_fullname(currentnode))) {
    gui_mildbeep("Error: source and destination are the same");
    return;
  }
  if(!comm_askoverwrite(comm_s)) return;
  gui_printmsg("Copying %s...",data_name(currentnode));
  if(comm_runcommand(NULL,"cp %a %b","ab",data_fullname(currentnode),comm_s)) {
    gui_printmsg("Error: cp failed");
  } else {
    data_restatfileifdirlogged(comm_s);
    gui_printmsg("Done.");
  }
}

void comm_mvfile(void) {
  int newnode,tagged;
  const char *to;

  comm_doublechecknode();
  tagged=data_tagged(currentnode);
  if(dispmode!=DISP_FILE) crapout("comm_move","called in wrong dispmode");

  if(comm_prompt(comm_sss,MAXNAMLEN,data_name(currentnode),0,
                 "Move as: ",FM_TABCOMPLETE,PV_NOPATHFILENAME)) return;
  if(!comm_sss[0]) {set_cancel(); return;}
  if(comm_prompt(comm_s,PATH_MAX+1,data_fullname(data_updir(currentnode)),0,
      "To path: ",FM_TABCOMPLETE,PV_DIRNAME)) return;
  if(!comm_s[0]) {set_cancel(); return;}
  to=comm_parsepath(comm_makeabsolute(comm_s,currentdir));
  if(!to) return;
  if(comm_verifydir(to)) return;
  sprintf(comm_s,"%s/%s",to,comm_sss);

  if(comm_checksame(comm_s,data_fullname(currentnode))) {
    gui_mildbeep("Error: source and destination are the same");
    return;
  }
  if(!comm_askoverwrite(comm_s)) return;

#ifdef ENABLECMDS
  if(rename(data_fullname(currentnode),comm_s)) {
    gui_mildbeep("Error: move failed (%s)",strerror(errno));
  }
#endif

  newnode=data_restatfileifdirlogged(comm_s);
  if(newnode) data_settag(newnode,tagged&config.preservetag);
  data_restatnode(currentnode);
}

void comm_rename(void) {
  int maxlen;
  comm_doublechecknode();
  if(!strcmp(data_fullname(currentnode),"/")) {
    gui_mildbeep("Error: cannot rename root directory");
    return;
  }
  maxlen=MIN(MAXNAMLEN,PATH_MAX-strlen(data_fullname(data_updir(currentnode))));
  if(dispmode==DISP_FILE) {
    if(comm_prompt(comm_s,maxlen,data_name(currentnode),0,"Rename to: ",
                   FM_TABCOMPLETE,PV_NOPATHFILENAME)) return;
  } else {
    if(comm_prompt_updir(comm_s,maxlen,data_name(currentnode),0,"Rename to: ",
                         FM_TABCOMPLETE,PV_NOPATHDIRNAME)) return;
  }
  if(!comm_s[0]) {set_cancel(); return;}
  strcpy(comm_ss,data_fullname(currentnode));
  strcpy(strrchr(comm_ss,'/')+1,comm_s);
  if(data_exists(comm_ss)) {
    gui_mildbeep("Error: name already exists");
    return;
  } else {
    /* restat in case destination file is logged but no longer there */
    data_restatfileifdirlogged(comm_ss);
  }
#ifdef ENABLECMDS
  if(rename(data_fullname(currentnode),comm_ss)) {
    gui_mildbeep("Error: rename failed (%s)",strerror(errno));
    return;
  }
#endif
  strcpy(datanode(currentnode)->name,comm_s);
  data_updatesort(currentnode);
}

void comm_open(int waitkey) {
  int good_curdir;
  char curdir[PATH_MAX+1];
  const char *run;
  comm_doublechecknode();

  if(config.inX) run=config_lookupext(data_fullname(currentnode),LOOKUP_X);
  else           run=config_lookupext(data_fullname(currentnode),LOOKUP_NONX);
  if(!run) {
    gui_mildbeep("Error: no external found for this file type for %s",
                  config.inX?"Xwindows":"non-Xwindows");
    return;
  }
  strcpy(comm_s  ,data_name(currentnode));
  strcpy(comm_ss ,data_fullname(data_updir(currentnode)));
  strcpy(comm_sss,data_fullname(currentnode));
#ifdef HAVE_GETCWD
  good_curdir=(NULL!=getcwd(curdir,PATH_MAX+1));
#else
  good_curdir=0;
#endif
  chdir(comm_ss);
  comm_configrunf(waitkey,run,"fdp",comm_s,comm_ss,comm_sss);
  if(good_curdir) chdir(curdir);
  data_restatnode(currentnode);
}

void comm_edit(void) {
  if(!config.editor[0]) {
    gui_mildbeep("Error: no editor defined");
    return;
  }
  comm_doublechecknode();
  strcpy(comm_s  ,data_name(currentnode));
  strcpy(comm_ss ,data_fullname(data_updir(currentnode)));
  strcpy(comm_sss,data_fullname(currentnode));
  comm_configrunf(0,config.editor,"fdp",comm_s,comm_ss,comm_sss);
  data_restatnode(currentnode);
}

void comm_doublechecknode(void) {
  /* lxt will bomb out here if currentnode is not in sync */
  gui_setwantdrawall();
  gui_drawmode();
}

int comm_askoverwrite(const char *fn) {
  struct stat st;
  if(lstat(fn,&st)) return 1;
  if(S_ISDIR(st.st_mode)) {
    gui_mildbeep("Error: cannot overwrite directory");
    return 0;
  } else {
    return comm_yesno("Overwrite destination? ");
  }
}

int readln_read_updir(WINDOW *win,int yl,int xl,int xs,int len,char *buf,
                      int tabmode,int prevtype) {
  int old,ret;
  old=currentdir;
  currentdir=data_updir(currentdir);
  ret=readln_read(win,yl,xl,xs,len,buf,tabmode,prevtype);
  currentdir=old;
  return ret;
}

/* these next two functions are nasty hacks to make tab completion look for
   sibbling directories instead of child directories when commands such as
   rename are called */

int comm_prompt_updir(char *s,int maxchars,const char *def,int def_override,
                      const char *prompt,int tabmode,int prevtype) {
  int old,ret;
  old=currentdir;
  currentdir=data_updir(currentdir);
  ret=comm_prompt(s,maxchars,def,def_override,prompt,tabmode,prevtype);
  currentdir=old;
  return ret;
}

int comm_prompt(char *s,int maxchars,const char *def,int def_override,
                const char *prompt,int tabmode,int prevtype) {
  int maxx,maxy,plen;
  int result;

  werase(gui_msbar);
  mvwprintw(gui_msbar,0,1,"%s",prompt);
  wnoutrefresh(gui_msbar);
  doupdate();
  plen=strlen(prompt);
  getmaxyx(gui_msbar, maxy, maxx);
  if(def&&(config.defaultreadln||def_override))
    {strncpy(s,def,maxchars-1); s[maxchars-1]=0;}
  else s[0]=0;
  result=readln_read(gui_msbar,0,1+plen,MIN(maxx-(2+plen),maxchars),maxchars,s,tabmode,prevtype);
  werase(gui_msbar);
  wnoutrefresh(gui_msbar);
  doupdate();
  if(result) set_cancel();
  return result;
}

int comm_yesno(const char *prompt) {
  int quit=0,yes,c;
  if(is_cancel()) return 0;
  yes=0; /* to keep compiler happy */
  werase(gui_msbar);
  mvwprintw(gui_msbar,0,1,"%s",prompt);
  wnoutrefresh(gui_msbar);
  doupdate();
  timeout(100);
  while(!quit) {
    if((c=getch())==ERR) {
      gui_drawtime();
    } else {
      if(c=='\a') {set_cancel(); yes=0; quit=1;}
      if((c=='y')||(c=='Y')) {yes=1; quit=1;}
      if((c=='n')||(c=='N')) {yes=0; quit=1;}
    }
  }
  werase(gui_msbar);
  wnoutrefresh(gui_msbar);
  doupdate();
  return yes;
}

int comm_runexternal(const char *command,int waitkey) {
  int val,a;

  mvwprintw(gui_msbar,0,1,"<%s>",command);
  wnoutrefresh(gui_msbar);
  doupdate();
#ifndef ENABLECMDS
  timeout(100);
  while(getch()==ERR);
  return 0;
#endif
  def_prog_mode();
  endwin();
  clearscreen();

  val=system(command);

  if(waitkey) {
    printf("\nFinished (%d).  Press enter to continue...",val);
    fflush(stdout);
    fcntl(0,F_SETFL,O_NONBLOCK);
    while(0<=read(0,&a,1));
    while(0> read(0,&a,1));
    while(0<=read(0,&a,1));
    fcntl(0,F_SETFL,0);
  }

  refresh();
  return val;
}

char quoted[STRINGSIZ];
const char *comm_quote(const char *s) {
  int n,p;
  p=0;
  quoted[p++]='\'';
  for(n=0;s[n];n++) {
    if(s[n]=='\'') {
      if((!n)||(s[n-1]!='\'')) {
        quoted[p++]='\''; if(p==STRINGSIZ) return NULL;
      }
      quoted[p++]='\\'; if(p==STRINGSIZ) return NULL;
      quoted[p++]='\''; if(p==STRINGSIZ) return NULL;
      if((s[n+1]!='\'')&&s[n+1]) {
        quoted[p++]='\''; if(p==STRINGSIZ) return NULL;
      }
    } else {
      quoted[p++]=s[n]; if(p==STRINGSIZ) return NULL;
    }
  }
  if((!n)||(s[n-1]!='\'')) {
    quoted[p++]='\''; if(p==STRINGSIZ) return NULL;
  }
  quoted[p]=0;
  return quoted;
}

int comm_configrunf(int waitkey,const char *fmt,const char *opts, ...) {
  int ret;
  const char *p;
  va_list ap;
  va_start(ap,opts);
  p=comm_configparsef(fmt,opts,ap);
  va_end(ap);
  if(!p) ret=-1;
  else   ret=comm_runexternal(p,waitkey);
  return ret;
}

const char *comm_configparsef_wrap(const char *fmt,const char *opts, ...) {
  const char *p;
  va_list ap;
  va_start(ap,opts);
  p=comm_configparsef(fmt,opts,ap);
  va_end(ap);
  return p;
}

char parsed_strout[STRINGSIZ];
const char *comm_configparsef(const char *fmt,const char *opts, va_list ap) {
  int n,pin,pout;
  const char *ss;
  char **args_in;

  args_in=malloc(strlen(opts)*sizeof(char *));
  if(!args_in) outofmem("comm_configparsef");
  for(n=0;n<strlen(opts);n++) args_in[n]=va_arg(ap,char *);
  for(pin=0,pout=0;fmt[pin];pin++) {
    if(fmt[pin]=='%') {
      pin++;
      if(fmt[pin]=='%') parsed_strout[pout++]='%';
      else {
        if(!fmt[pin]) crapout("comm_configparsef","invalid format string (%s)",fmt);
        ss=strchr(opts,fmt[pin]);
        if(!ss) crapout("comm_configparsef","invalid format string (%s)",fmt);
        n=ss-opts;
        ss=comm_quote(args_in[n]);
        if((strlen(ss)+pout+1)>STRINGSIZ) {
          gui_mildbeep("Error: command line too long");
          return NULL;
        }
        strcpy(parsed_strout+pout,ss);
        pout+=strlen(ss);
      }
    } else parsed_strout[pout++]=fmt[pin];
    if(pout>(STRINGSIZ-1)) {
      gui_mildbeep("Error: command line too long");
      return NULL;
    }
  }
  parsed_strout[pout]=0;
  free(args_in);
  return parsed_strout;
}

int comm_runcommand(const char *dispstr,const char *fmt,const char *opts, ...) {
  va_list ap;
  const char *cmdline;
#ifdef ENABLECMDS
  int n;
  int pid,status,maxx;
  int fdo[2];
  int readnull;
  char c,*dispbuf;
#endif

  va_start(ap,opts);
  cmdline=comm_configparsef(fmt,opts,ap);
  va_end(ap);
  if(!cmdline) return -1;

#ifdef ENABLECMDS
  readnull=open("/dev/null",O_RDONLY);
  if(readnull==-1) crapout("comm_runcommand","cannot open /dev/null for reading (%s)",strerror(errno));
  if(dispstr) pipe(fdo);
  else {
    fdo[1]=open("/dev/null",O_WRONLY|O_TRUNC);
    if(fdo[1]==-1) crapout("comm_runcommand","cannot open /dev/null for writing (%s)",strerror(errno));
  }
  pid=fork();
  if(pid==-1) {
    gui_mildbeep("Error: fork failed");
    return -1;
  }
  if(!pid) {
    close(fdo[0]);
    if((-1!=dup2(readnull,0))&&(-1!=dup2(fdo[1],1))&&(-1!=dup2(fdo[1],2))) {
      /* note: we can't return the straight value because waitpid() only
         gives us the lower eight bits of the return value, so it will see
         a 256 as a 0 */
      n=system(cmdline);
      if(n) _exit(-1);
      else  _exit(0);
    }
    _exit(-1);
  }
  close(fdo[1]);
  if(dispstr) {
    fcntl(fdo[0],F_SETFL,O_NONBLOCK);
    wsetscrreg(gui_msbar,0,0);
    scrollok(gui_msbar,0);
    wmove(gui_msbar,0,0);
    maxx=gui_msbar->_maxx-strlen(dispstr)-1;
    dispbuf=malloc(maxx);
    if(!dispbuf) outofmem("comm_runcommnad");
    n=0;
    while((!waitpid(pid,&status,WNOHANG))&&(!is_cancel())) {
      if(0<read(fdo[0],&c,1)) {
        if(c=='\n') c=0;
        dispbuf[n]=c;
        if(!c) {
          werase(gui_msbar);
          mvwprintw(gui_msbar,0,1,"%s%s",dispstr,dispbuf);
          wnoutrefresh(gui_msbar);
          doupdate();
          n=0;
        } else if(n<(maxx-1)) n++;
      }
    }
    free(dispbuf);
    close(fdo[0]);
  } else {
    while((!waitpid(pid,&status,WNOHANG))&&(!is_cancel()));
  }
  if(is_cancel()) {
    /* kill(-pid,SIGTERM); how to kill a process and all descendents? */
    return -1;
  }
  if(!WIFEXITED(status)) return -1;
  return WEXITSTATUS(status);
#else
  gui_mildbeep("Execute: <%s>",cmdline);
  timeout(100);
  while(getch()==ERR);
  return 0;
#endif
}

char absolutepath[PATH_MAX+1];
const char *comm_makeabsolute(const char *s,int curnode) {
  const char *curdir;
  if((s[0]!='/')&&(s[0]!='~')) {
    curdir=data_fullname(curnode);
    if((strlen(curdir)+strlen(s)+1)>PATH_MAX) {
      gui_mildbeep("Error: path too long");
      return NULL;
    }
    sprintf(absolutepath,"%s/%s",curdir,s);
  } else {
    if(strlen(s)>PATH_MAX) {
      gui_mildbeep("Error: path too long");
      return NULL;
    }
    strcpy(absolutepath,s);
  }
  return absolutepath;
}

/* parse a path, expanding "~user" and squishing "." and ".." */
/* realpath is not used because we don't want to follow links */
char parsedpath[PATH_MAX+1];
const char *comm_parsepath(const char *in) {
  int p;
  char *s;
  const char *ss;
  char buf[PATH_MAX+1];

  if(!in) return NULL;
  if(strlen(in)>PATH_MAX) {
    gui_mildbeep("Error: path too long");
    return NULL;
  }

  p=0;
  s=strcpy(buf,in);
  if(buf[0]=='~') {
    s=strchr(buf,'/');
    if(s) *s=0;
    if(buf[1]==0) ss=uid_homedir(myuid);
    else          ss=uid_homedir(uid_nametouid(buf+1));
    if(!ss) {
      gui_mildbeep("Error: cannot find user %s",buf+1);
      return NULL;
    }
    strcpy(parsedpath,ss);
    if(!s) return parsedpath;
    *s='/';
    p=strlen(ss);
  }
  if(*s!='/') {
    gui_mildbeep("Error: path must be absolute");
    return NULL;
  }
  if((strlen(s)+p)>PATH_MAX) {
    gui_mildbeep("Error: path too long");
    return NULL;
  }
  strcpy(parsedpath+p,s);
  return data_squishfullname(parsedpath);
}

int comm_verifydir(const char *to) {
  struct stat statinfo;
  if(!strcmp(to,"/")) return 0;
  if(stat(to,&statinfo)) {
    gui_mildbeep("Error: cannot open destination directory (%s)",strerror(errno));
    return 1;
  }
  if(!S_ISDIR(statinfo.st_mode)) {
    gui_mildbeep("Error: destination is not a directory");
    return 1;
  }
  return 0;
}

int comm_checksame(const char *a,const char *b) {
  char ra[PATH_MAX+1],rb[PATH_MAX+1];
  if(!realpath(a,ra)) return 0;
  if(!realpath(b,rb)) return 0;
  return !strcmp(ra,rb);
}

int comm_checklength(const char *fn) {
  if(strlen(fn)>PATH_MAX) {
    gui_mildbeep("Error: pathname too long");
    return 1;
  }
  return 0;
}
