/*
pkrhnd.c Version 1.5.0 - 52 Card Poker Test
Copyright (C) 2005-2010  dondalah721@yahoo.com (Dondalah)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to:

	Free Software Foundation, Inc.
	59 Temple Place - Suite 330
	Boston, MA  02111-1307, USA.
*/

/* This test is based on: */
/* Lincoln L. Chao */
/* Statistics for Management */
/* Palo Alto, CA: The Scientific Press, 1984 */
/* Chapter 6, Expected Value and Population Parameters */
/* Section 2, Population Mean and Variance */

/* Lincoln L. Chao was at California State University, */
/* Long Beach, CA, while writing this book. */

/* John Scarne */
/* Scarne's New Complete Guide to Gambling */
/* Fully revised, expanded, updated edition */
/* New York: Simon & Schuster, 1961, 1974, 1986 */
/* A Fireside book, 871 p. */
/* ISBN: 0-671-21734-8 hardback  */
/* ISBN: 0-671-63063-6 paperback */
/* Chapter 27, Poker: America's Favorite Card Game */
/* Section: Poker Odds and Probabilities */

/* See the documentation in pkrhnd.html for */
/* information about the probabilities in the */
/* Scarne book.  The Scarne book counts the cards */
/* as aces high and low.  This program follows */
/* the Scarnes counting method. */

/* The World Almanac and Book of Facts: 2003 */
/* New York: World Almanac Books, 2004 */
/* ISBN: 0-88687-882-9 paperback */
/* ISBN: 0-88687-883-7 hardback  */
/* Chapter: Weights and Measures */
/* Section: Playing Cards and Dice Chances */

/* The World Almanac follows the same counting */
/* method as Scarnes. */

/* Sample test: */

/* pkrhnd 100000 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "rnd.h"

/* C(52,5) */
#define HANDS 2598960.0 
#define HIGHS 1302540.0
#define MSCPROB (HIGHS / HANDS)
#define MISC      0
#define ONEPAIR   1
#define TWOPAIR   2
#define TRIPLET   3
#define STRAIGHT  4
#define FLUSH     5
#define FHOUSE    6
#define QUAD      7
#define STRFLUSH  8
#define ROYFLUSH  9
#define ACEHIGH      10
#define KINGHIGH     11
#define QUEENHIGH    12
#define JACKHIGH     13
#define TENHIGH      14
#define NINEHIGH     15
#define EIGHTHIGH    16
#define SEVENHIGH    17

void putstx(pgm)
char *pgm;
   {
   fprintf(stderr,"Usage: %s size\n", pgm);
   fprintf(stderr,"Where size is 30 to 1000000000\n");
   fprintf(stderr,"Example: %s 100000\n", pgm);
   exit(1);
   } /* putstx */

void puthand(char *title,
   int obs,
   double psuccess,
   double dblsz,
   double *ztot,
   double *sumx,
   double *sumxsqd, 
   double *sumchi) 
   {
   double dblobs;
   double ratio;
   double qfail;
   double expect;
   double variance;
   double stdev;
   double zval;
   double diff;
   dblobs = (double) obs;
   ratio = dblobs / dblsz;
   qfail = 1.0 - psuccess;
   expect = psuccess * dblsz;
   variance = psuccess * qfail;
   stdev = sqrt(variance / dblsz);
   zval = (ratio - psuccess) / stdev;
   *ztot += zval;
   diff = dblobs - expect;
   *sumx    += (dblobs * psuccess);
   *sumxsqd += (dblobs * dblobs * psuccess);
   *sumchi  += (diff * diff / expect);
   printf("%s %9d %22.17f %20.15f %8.4f\n",
      title, obs, ratio, psuccess, zval);
   } /* puthand */

int main(argc,argv)
int argc;
char **argv;
   {
   int i,j;
   int size;
   int lastcard;
   int ofst;
   int resort;
   int cardnum;
   int pairsw;
   int tripletsw;
   int miscsw;
   int onepr;
   int twopr;
   int triplet;
   int straight;
   int flush;
   int fullhse;
   int quad;
   int strflush;
   int royflush;
   int misc;
   int acehigh;
   int kinghigh;
   int queenhigh;
   int jackhigh;
   int tenhigh;
   int ninehigh;
   int eighthigh;
   int sevenhigh;
   int tothigh;
   int tot;
   char tmpcard;
   char *p,*q;
   char *deckptr;
   char *deckend;
   char *ascdeckptr;
   char *aschandptr;
   char *seqdeckptr;
   char *seqhandptr;
   char deck[64];
   char hand[32];
   char aschand[64];
   char seqhand[64];
   char ascdeck[128];
   char seqdeck[128];
   double dblsz;
   double ratio;
   double totprob;
   double expect;
   double ztot;
   double popmu;
   double popvar;
   double popstdev;
   double sumx;
   double sumxsqd;
   double mean;
   double estxsqd;
   double variance;
   double stdev;
   double zscore;
   double sumchi;
   double negtblv,postblv;
   void chirange(int degf, double *lorange,
      double *hirange);
   double *probptr;
   double *probend;
   double prob[32];
   if (argc != 2) putstx(*argv);
   size = atoi(*(argv+1));
   if (size < 30 || size > 1000000000)
      {
      fprintf(stderr,"Invalid size.\n");
      putstx(*argv);
      } /* bad size */
   dblsz = (double) size;
   sd = (unsigned char *) rndinit();
   if (sd == NULL)
      {
      fprintf(stderr,"pkrhnd: out of memory "
	 "allocating sd.\n");
      exit(1);
      } /* out of mem */

   strcpy(ascdeck,"2c2d2h2s");
   strcat(ascdeck,"3c3d3h3s");
   strcat(ascdeck,"4c4d4h4s");
   strcat(ascdeck,"5c5d5h5s");
   strcat(ascdeck,"6c6d6h6s");
   strcat(ascdeck,"7c7d7h7s");
   strcat(ascdeck,"8c8d8h8s");
   strcat(ascdeck,"9c9d9h9s");
   strcat(ascdeck,"tctdthts");
   strcat(ascdeck,"jcjdjhjs");
   strcat(ascdeck,"qcqdqhqs");
   strcat(ascdeck,"kckdkhks");
   strcat(ascdeck,"acadahas");

   strcpy(seqdeck,"awaxayaz");
   strcat(seqdeck,"bwbxbybz");
   strcat(seqdeck,"cwcxcycz");
   strcat(seqdeck,"dwdxdydz");
   strcat(seqdeck,"ewexeyez");
   strcat(seqdeck,"fwfxfyfz");
   strcat(seqdeck,"gwgxgygz");
   strcat(seqdeck,"hwhxhyhz");
   strcat(seqdeck,"iwixiyiz");
   strcat(seqdeck,"jwjxjyjz");
   strcat(seqdeck,"kwkxkykz");
   strcat(seqdeck,"lwlxlylz");
   strcat(seqdeck,"mwmxmymz");

   probptr = (double *) prob;
   probend = (double *) prob + 32;
   while (probptr < probend) *probptr++ = 0.0;
   probptr = (double *) prob;
   *(probptr + MISC)      = 1302540.0 / HANDS;  /* aces high/low */
   *(probptr + ONEPAIR)   = 1098240.0 / HANDS;
   *(probptr + TWOPAIR)   =  123552.0 / HANDS;
   *(probptr + TRIPLET)   =   54912.0 / HANDS;
   *(probptr + STRAIGHT)  =   10200.0 / HANDS;  /* aces high/low */
   *(probptr + FLUSH)     =    5108.0 / HANDS;  /* aces high/low */
   *(probptr + FHOUSE)    =    3744.0 / HANDS;
   *(probptr + QUAD)      =     624.0 / HANDS;
   *(probptr + STRFLUSH)  =      36.0 / HANDS;  /* aces high/low */
   *(probptr + ROYFLUSH)  =       4.0 / HANDS;
   *(probptr + ACEHIGH)   =  MSCPROB * 502860.0 / HIGHS;
   *(probptr + KINGHIGH)  =  MSCPROB * 335580.0 / HIGHS;
   *(probptr + QUEENHIGH) =  MSCPROB * 213180.0 / HIGHS;
   *(probptr + JACKHIGH)  =  MSCPROB * 127500.0 / HIGHS;
   *(probptr + TENHIGH)   =  MSCPROB *  70380.0 / HIGHS;
   *(probptr + NINEHIGH)  =  MSCPROB *  34680.0 / HIGHS;
   *(probptr + EIGHTHIGH) =  MSCPROB *  14280.0 / HIGHS;
   *(probptr + SEVENHIGH) =  MSCPROB *   4080.0 / HIGHS;

   totprob = *(probptr + ACEHIGH)
      + *(probptr + KINGHIGH)
      + *(probptr + QUEENHIGH)
      + *(probptr + JACKHIGH)
      + *(probptr + TENHIGH)
      + *(probptr + NINEHIGH)
      + *(probptr + EIGHTHIGH)
      + *(probptr + SEVENHIGH)
      + *(probptr + ONEPAIR)
      + *(probptr + TWOPAIR)
      + *(probptr + TRIPLET)
      + *(probptr + STRAIGHT)
      + *(probptr + FLUSH)
      + *(probptr + FHOUSE)
      + *(probptr + QUAD)
      + *(probptr + STRFLUSH)
      + *(probptr + ROYFLUSH);

   popmu = estxsqd = 0.0;
   probptr = (double *) prob + 1;
   probend = (double *) prob + SEVENHIGH + 1;
   while (probptr < probend)
      {
      expect = *probptr * dblsz;
      popmu += (expect * *probptr);
      estxsqd += (expect * expect * *probptr);
      probptr++;
      } /* for each hand */
   popvar = estxsqd - (popmu * popmu);
   popstdev = sqrt(popvar);

   misc = onepr = twopr = triplet
      = straight = flush = fullhse
      = quad = strflush = royflush
      = acehigh = kinghigh = queenhigh
      = jackhigh = tenhigh = ninehigh
      = eighthigh = sevenhigh
      = tothigh = tot = 0;
   /* terminate aschand and seqhand */
   p = (char *) aschand + 10;
   q = (char *) aschand + 16;
   while (p < q)
      {
      *p++ = 255;
      } /* for each card after aschand */
   *p++ = '\0';
   *p++ = '\0';
   p = (char *) seqhand + 10;
   q = (char *) seqhand + 16;
   while (p < q)
      {
      *p++ = 255;
      } /* for each card after seqhand */
   *p++ = '\0';
   *p++ = '\0';
   i = size;
   while (i--)
      {
      /* unshuffle */
      j = 0;
      p = (char *) deck;
      q = (char *) deck + 52;
      while (p < q)
	 *p++ = (char) j++;
      *p = 255;
      /* shuffle/deal */
      lastcard = 52;
      p = (char *) hand;
      q = (char *) hand + 5;
      deckptr = (char *) deck;
      deckend = (char *) deck + 51;   /* 52 - 1 */
      while (p < q)
	 {
	 ofst = (int) rndnum(lastcard--,sd);
	 deckptr = (char *) deck + ofst;
	 *p++ = *deckptr;
	 *deckptr = *deckend;
	 *deckend-- = (char) 255;
	 } /* for each card in hand */
      *p++ = (char) 255;
      *p++ = (char) 255;
      *p++ = (char) '\0';
      /* bubble sort */
      resort = 1;
      while (resort)
	 {
	 resort = 0;
         p = (char *) hand;
         q = (char *) hand + 4;
         while (p < q)
	    {
	    if (*p > *(p+1))
	       {
	       tmpcard = *p;
	       *p = *(p+1);
	       *(p+1) = (char) tmpcard;
	       resort = 1;
	       } /* if out of order */
	    p++;
	    } /* for each card in hand */
	 } /* for each pass of sort */
      /* convert ordinal to ascii */
      p = (char *) hand;
      q = (char *) hand + 5;
      aschandptr = (char *) aschand;
      seqhandptr = (char *) seqhand;
      while (p < q)
	 {
	 ofst = (int) (*p + *p);
	 ascdeckptr = (char *) ascdeck + ofst;
	 seqdeckptr = (char *) seqdeck + ofst;
	 *aschandptr = *ascdeckptr;
	 *(aschandptr+1) = *(ascdeckptr+1);
	 *seqhandptr = *seqdeckptr;
	 *(seqhandptr+1) = *(seqdeckptr+1);
	 p++;
	 aschandptr += 2;
	 seqhandptr += 2;
	 } /* for each card in hand */
      *aschandptr++ = 255;
      *seqhandptr++ = 255;
      *aschandptr++ = 255;
      *seqhandptr++ = 255;
      *aschandptr++ = '\0';
      *seqhandptr++ = '\0';
      *aschandptr++ = '\0';
      *seqhandptr++ = '\0';
      /* count */
      cardnum = pairsw = tripletsw = miscsw = 0;
      p = (char *) aschand;
      q = (char *) aschand + 10;
      seqhandptr = (char *) seqhand;
      while (p < q)
	 {
	 cardnum++;
	 if (cardnum == 1)
	    {
	    if (*(p+1) == *(p+3)
	       && *(p+1) == *(p+5)
	       && *(p+1) == *(p+7)
	       && *(p+1) == *(p+9)
	       && *p == 't'
	       && *(p+2) == 'j'
	       && *(p+4) == 'q'
	       && *(p+6) == 'k'
	       && *(p+8) == 'a')
	       {
	       royflush++;
	       cardnum += 4;
	       p += 10;
	       seqhandptr += 10;
	       continue;
	       } /* if royal flush */
	    else if (*(seqhandptr+1) == *(seqhandptr+3)
	       && *(seqhandptr+1) == *(seqhandptr+5)
	       && *(seqhandptr+1) == *(seqhandptr+7)
	       && *(seqhandptr+1) == *(seqhandptr+9)
	       && *seqhandptr     == *(seqhandptr+2) - 1
	       && *seqhandptr     == *(seqhandptr+4) - 2
	       && *seqhandptr     == *(seqhandptr+6) - 3
	       && *seqhandptr     == *(seqhandptr+8) - 4)
	       {
	       strflush++;
	       cardnum += 4;
	       p += 10;
	       seqhandptr += 10;
	       continue;
	       } /* if straight/flush aces high */
	    else if (*(seqhandptr+1) == *(seqhandptr+3)
	       && *(seqhandptr+1) == *(seqhandptr+5)
	       && *(seqhandptr+1) == *(seqhandptr+7)
	       && *(seqhandptr+1) == *(seqhandptr+9)
	       && *seqhandptr     == *(seqhandptr+2) - 1
	       && *seqhandptr     == *(seqhandptr+4) - 2
	       && *seqhandptr     == *(seqhandptr+6) - 3
	       && *seqhandptr     == *(seqhandptr+8) - 12)
	       {
	       strflush++;
	       cardnum += 4;
	       p += 10;
	       seqhandptr += 10;
	       continue;
	       } /* if straight/flush aces low */
	    else if (*(p+1) == *(p+3)
	       && *(p+1) == *(p+5)
	       && *(p+1) == *(p+7)
	       && *(p+1) == *(p+9))
	       {
	       flush++;
	       cardnum += 4;
	       p += 10;
	       seqhandptr += 10;
	       continue;
	       } /* if flush aces high/low */
	    else if (*seqhandptr == *(seqhandptr+2) - 1
	       && *seqhandptr    == *(seqhandptr+4) - 2
	       && *seqhandptr    == *(seqhandptr+6) - 3
	       && *seqhandptr    == *(seqhandptr+8) - 4)
	       {
	       straight++;
	       cardnum += 4;
	       p += 10;
	       seqhandptr += 10;
	       continue;
	       } /* if straight aces high */
	    else if (*seqhandptr == *(seqhandptr+2) - 1
	       && *seqhandptr    == *(seqhandptr+4) - 2
	       && *seqhandptr    == *(seqhandptr+6) - 3
	       && *seqhandptr    == *(seqhandptr+8) - 12)
	       {
	       straight++;
	       cardnum += 4;
	       p += 10;
	       seqhandptr += 10;
	       continue;
	       } /* if straight aces low */
	    } /* if straight or flush */
	 if (cardnum < 5
	    && *p == *(p+2)
	    && *p != *(p+4))
	    {
	    if (tripletsw)
	       {
	       triplet--;
	       fullhse++;
	       cardnum += 1;
	       p += 4;
	       seqhandptr += 4;
	       continue;
	       } /* if full house */
	    else if (pairsw)
	       {
	       onepr--;
	       twopr++;
	       cardnum += 1;
	       p += 4;
	       seqhandptr += 4;
	       continue;
	       } /* if second pair */
	    else
	       {
	       pairsw = 1;
	       onepr++;
	       cardnum += 1;
	       p += 4;
	       seqhandptr += 4;
	       continue;
	       } /* if first pair */
	    } /* one pair */
	 else if (cardnum < 4
	    && *p == *(p+2)
	    && *p == *(p+4)
	    && *p != *(p+6))
	    {
	    if (pairsw)
	       {
	       onepr--;
	       fullhse++;
	       cardnum += 2;
	       p += 6;
	       seqhandptr += 6;
	       continue;
	       } /* if full house */
	    else
	       {
	       triplet++;
	       tripletsw = 1;
	       cardnum += 2;
	       p += 6;
	       seqhandptr += 6;
	       continue;
	       } /* if only triplet */
	    } /* if triplet */
	 else if (cardnum < 3
	    && *p == *(p+2)
	    && *p == *(p+4)
	    && *p == *(p+6)
	    && *p != *(p+8))
	    {
	    quad++;
	    cardnum += 3;
	    p += 8;
	    seqhandptr += 8;
	    continue;
	    } /* if quad */
	 else
	    {
	    miscsw++;
	    } /* if miscellaneous */
	 if (miscsw == 5)
	    {
	    misc++;
	    aschandptr = (char *) aschand + 8;
	    switch(*aschandptr) {
	       /* aces low is not counted in miscellaneous */
	       case 'a':              /* aces high */
		  acehigh++;
		  break;
	       case 'k':
                  kinghigh++;
		  break;
	       case 'q':
                  queenhigh++;
		  break;
	       case 'j':
                  jackhigh++;
		  break;
	       case 't':
                  tenhigh++;
		  break;
	       case '9':
                  ninehigh++;
		  break;
	       case '8':
                  eighthigh++;
		  break;
	       case '7':
                  sevenhigh++;
		  break;
               default:
		  fprintf(stderr,"pkrhnd: logic "
		     "error 001\n");
		  exit(1);
	       } /* switch *p */
	    } /* if 5 miscellaneous */
	 p += 2;
	 seqhandptr += 2;
	 } /* for each card in ascii hand */
      } /* count hands by type */
   printf("                    "
      "52 Card Poker - Aces High/Low\n");
   printf("Hand "
      "           "
      "Count"
      "    "
      "Sample Proportion"
      "      "
      "Population Prob."
      "    "
      "z\n");
   sumx = sumxsqd = sumchi = 0.0;
   probptr = (double *) prob;

   /* one pair */
   puthand("one pair   ", onepr,
      *(probptr + ONEPAIR),
      dblsz, &ztot,
      &sumx, &sumxsqd, &sumchi);

   /* two pairs */
   puthand("two pairs  ", twopr,
      *(probptr + TWOPAIR),
      dblsz, &ztot,
      &sumx, &sumxsqd, &sumchi);

   /* three of a kind */
   puthand("3 of kind  ", triplet,
      *(probptr + TRIPLET),
      dblsz, &ztot,
      &sumx, &sumxsqd, &sumchi);

   /* straight */
   puthand("straight   ", straight,
      *(probptr + STRAIGHT),
      dblsz, &ztot,
      &sumx, &sumxsqd, &sumchi);

   /* flush */
   puthand("flush      ", flush,
      *(probptr + FLUSH),
      dblsz, &ztot,
      &sumx, &sumxsqd, &sumchi);

   /* full house */
   puthand("full house ", fullhse,
      *(probptr + FHOUSE),
      dblsz, &ztot,
      &sumx, &sumxsqd, &sumchi);

   /* 4 of a kind */
   puthand("4 of kind  ", quad,
      *(probptr + QUAD),
      dblsz, &ztot,
      &sumx, &sumxsqd, &sumchi);

   /* straight flush */
   puthand("strflush   ", strflush,
      *(probptr + STRFLUSH),
      dblsz, &ztot,
      &sumx, &sumxsqd, &sumchi);

   /* royal flush */
   puthand("royflush   ", royflush,
      *(probptr + ROYFLUSH),
      dblsz, &ztot,
      &sumx, &sumxsqd, &sumchi);

   /* misc */
   {
   double dblobs;
   double ratio;
   double psuccess;
   double qfail;
   double expect;
   double variance;
   double stdev;
   double zval;
   dblobs = (double) misc;
   ratio = dblobs / dblsz;
   psuccess = *(probptr+MISC);
   qfail = 1.0 - psuccess;
   expect = psuccess * dblsz;
   variance = psuccess * qfail;
   stdev = sqrt(variance / dblsz);
   zval = (ratio - psuccess) / stdev;
   printf("%s %9d %22.17f %20.15f %8.4f\n",
      "other      ", misc, ratio,
      psuccess, zval);
   }

   printf("\n");
   /* ace high */
   puthand("ace high   ", acehigh,
      *(probptr + ACEHIGH),
      dblsz, &ztot,
      &sumx, &sumxsqd, &sumchi);

   /* king high */
   puthand("king high  ", kinghigh,
      *(probptr + KINGHIGH),
      dblsz, &ztot,
      &sumx, &sumxsqd, &sumchi);

   /* queen high */
   puthand("queen high ", queenhigh,
      *(probptr + QUEENHIGH),
      dblsz, &ztot,
      &sumx, &sumxsqd, &sumchi);

   /* jack high */
   puthand("jack high  ", jackhigh,
      *(probptr + JACKHIGH),
      dblsz, &ztot,
      &sumx, &sumxsqd, &sumchi);

   /* ten high */
   puthand("ten high   ", tenhigh,
      *(probptr + TENHIGH),
      dblsz, &ztot,
      &sumx, &sumxsqd, &sumchi);

   /* nine high */
   puthand("nine high  ", ninehigh,
      *(probptr + NINEHIGH),
      dblsz, &ztot,
      &sumx, &sumxsqd, &sumchi);

   /* eight high */
   puthand("eight high ", eighthigh,
      *(probptr + EIGHTHIGH),
      dblsz, &ztot,
      &sumx, &sumxsqd, &sumchi);

   /* seven high */
   puthand("seven high ", sevenhigh,
      *(probptr + SEVENHIGH),
      dblsz, &ztot,
      &sumx, &sumxsqd, &sumchi);

   tothigh = acehigh + kinghigh + queenhigh
      + jackhigh + tenhigh + ninehigh
      + eighthigh + sevenhigh;

   /* high card summary */
   {
   double dblobs;
   double ratio;
   double psuccess;
   double qfail;
   double expect;
   double variance;
   double stdev;
   double zval;
   dblobs = (double) tothigh;
   ratio = dblobs / dblsz;
   psuccess = *(probptr+MISC);
   qfail = 1.0 - psuccess;
   expect = psuccess * dblsz;
   variance = psuccess * qfail;
   stdev = sqrt(variance / dblsz);
   zval = (ratio - psuccess) / stdev;
   printf("%s %9d %22.17f %20.15f %8.4f\n",
      "high cards ", tothigh, ratio,
      psuccess, zval);
   }

   tot = onepr + twopr + triplet
      + straight + flush + fullhse
      + quad + strflush + royflush
      + tothigh;
   ratio = (double) tot / dblsz;
   printf("Total       %9d %22.17f %20.15f %8.4f\n",
      tot, ratio, totprob, ztot);
   printf("\n");
   variance = sumxsqd - (sumx * sumx);
   stdev = sqrt(variance);
   mean = sumx / dblsz;
   zscore = (sumx - popmu) / (popstdev / sqrt(dblsz));
   printf("                 Mean "
      "            Stdev "
      "      Z Score\n");
   printf("Expected %16.6f %16.6f\n",
      popmu, popstdev);
   printf("Observed %16.6f %16.6f %9.4f\n",
      sumx, stdev, zscore);
   printf("\n");

   printf("Chi-square %f\n", sumchi);
   chirange(9,&negtblv,&postblv);
   printf("Range at 95 percent: %f  to  %f\n",
      negtblv, postblv);
   free(sd);
   return(0);
   } /* main */
