/*
  disksize.c  --  a tool for determining the sizes of disks and files
  Copyright (C) 1998,1999 Steffen Solyga <solyga@absinth.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "disksize.h"

int
dsphlp( char* pn ) {
  fprintf( HELP_CHANNEL, "%s v%s (%s)\n", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
#ifdef USE_LLSEEK
  fprintf( HELP_CHANNEL, "Prints file size(s) to %s (using llseek()).\n", STD_CHANNEL==stdout ? "stdout" : "stderr" );
#else
  fprintf( HELP_CHANNEL, "Prints file size(s) to %s (using lseek()).\n", STD_CHANNEL==stdout ? "stdout" : "stderr" );
#endif
  fprintf( HELP_CHANNEL, "Flowers & bug reports to %s.\n", MY_EMAIL_ADDRESS );
  fprintf( HELP_CHANNEL, "Usage: %s [options] file(s)\n", pn );
  fprintf( HELP_CHANNEL, "switches:\n" );
  fprintf( HELP_CHANNEL, "  -h\t write this info to %s and exit sucessfully\n", HELP_CHANNEL==stdout ? "stdout" : "stderr" );
  fprintf( HELP_CHANNEL, "  -i\t ignore read errors when iterating\n" );
  fprintf( HELP_CHANNEL, "  -k\t set unit to kBytes\n" );
  fprintf( HELP_CHANNEL, "  -v\t raise verbosity level\n" );
  fprintf( HELP_CHANNEL, "  -G\t set unit to GBytes\n" );
  fprintf( HELP_CHANNEL, "  -M\t set unit to MBytes\n" );
  fprintf( HELP_CHANNEL, "  -V\t print version and compilation info to %s and exit sucessfully\n", VERSION_CHANNEL==stdout ? "stdout" : "stderr" );
  return( 0 );
}


int dspver( char* pn ) {
  fprintf( VERSION_CHANNEL, "%s v%s (%s)\n", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( VERSION_CHANNEL, "Compilation settings:\n" );
  fprintf( VERSION_CHANNEL, "  USE_LLSEEK: " );
#ifdef USE_LLSEEK
  fprintf( VERSION_CHANNEL, "defined\n" );
#else
  fprintf( VERSION_CHANNEL, "undefined\n" );
#endif
  fprintf( VERSION_CHANNEL, "  SIZE_MAX  : " );
#ifdef USE_LLSEEK
  fprintf( VERSION_CHANNEL, "%lld\n", SIZE_MAX );
#else
  fprintf( VERSION_CHANNEL, "%ld\n", SIZE_MAX );
#endif
  return( 0 );
}


int
main( int argc, char** argv ) {
/*
SOLYGA --------------------
SOLYGA main() disksize

SOLYGA started      : Sun Dec 20 14:42:16 CET 1998 @beast
*/
  int c;
  int i;
  char* fn= NULL;			/* file name */
  int fd= -1;				/* file descriptor */
  int retval= RETVAL_OK;
  int nbr;				/* no. of bytes read (iterative op.) */
  unsigned char byte[1];		/* read buffer (iterative operation) */
  int of_verbose= 0;			/* be verbose */
  int of_ignore= 0;			/* ignore read errors (iteration) */
  char unit= ' ';			/* unit pre char ' '/'k'/'M'/'G' */
  int iterative= 0;			/* operation mode */
  int is_larger;			/* flag for displaying '>' */
  /* lseek() / llseek() specific stuff */
#ifdef USE_LLSEEK
  loff_t fs= 0LL;			/* file size in bytes */
  loff_t size_divider= 1LL;		/* unit bytes/kbytes/Mbytes/Gbytes */
  loff_t fs_min= 0LL;			/* iterative operation */
  loff_t fs_max= SIZE_MAX;		/* iterative operation */
  loff_t offset= 0LL;			/* iterative operation */
#else /* USE_LLSEEK */
  off_t fs= 0L;
  off_t size_divider= 1L;
  off_t fs_min= 0L;
  off_t fs_max= SIZE_MAX;
  off_t offset= 0L;
#endif /* USE_LLSEEK */
  while( (c=getopt(argc,argv,"hikvGMV")) != EOF )
  {
    switch( c )
    {
      case 'h': /* display help to HELP_CHANNEL and exit sucessfully */
        dsphlp( *argv );
        retval= RETVAL_OK; goto DIE_NOW;
      case 'i': /* ignore read errors in iteration */
        of_ignore= 1;
        break;
      case 'k': /* output kBytes */
        unit= 'k';
        break;
      case 'v': /* be verbose on STD_CHANNEL */
        of_verbose++;
        break;
      case 'G': /* output Gbytes */
        unit= 'G';
        break;
      case 'M': /* output Mbytes */
        unit= 'M';
        break;
      case 'V': /* print version info and exit successfully */
        dspver( *argv );
        retval= RETVAL_OK; goto DIE_NOW;
      case '?': /* refer to help and exit unsuccessfully */
        fprintf( ERROR_CHANNEL, "Try `%s -h' for more information.\n", *argv );
        retval= RETVAL_ERROR; goto DIE_NOW;
      default : /* program error */
        fprintf( ERROR_CHANNEL, "%s: Options bug! E-mail me at %s.\n",
                 *argv, MY_EMAIL_ADDRESS );
        retval= RETVAL_BUG; goto DIE_NOW;
    }
  }
  if( argc <= optind )
  {
    fprintf( ERROR_CHANNEL, "%s: No file specified.\n", *argv );
    retval= 1; goto DIE_NOW;
  }
  switch( unit )
  {
    case 'G': size_divider*= 1024;
    case 'M': size_divider*= 1024;
    case 'k': size_divider*= 1024;
  }
  for( i=optind; i<argc; i++ )
  {
    /* open file */
    fn= argv[i];
    if( (fd=open(fn,O_RDONLY)) == -1 )
    {
      fprintf( ERROR_CHANNEL, "%s: Cannot open `%s' for reading. %s.\n",
               *argv, fn, sys_errlist[errno] );
      retval= 1; goto DIE_NOW;
    }
    /* get size */
    is_larger= 0;
    iterative= 0;
    if( (fs=LSEEK(fd,0LL,SEEK_END)) == -1 )
    {
      fprintf( ERROR_CHANNEL, "%s: Cannot lseek `%s'. %s.\n",
               *argv, fn, sys_errlist[errno] );
      retval= 1; goto DIE_NOW;
    }
    if( fs == 0 )
    { /* special file or normal file of zero size */
      if( (nbr=read(fd,byte,1)) == -1 )
      {
        fprintf( ERROR_CHANNEL, "%s: Cannot read `%s'. %s.\n",
                 *argv, fn, sys_errlist[errno] );
        retval= 1; goto DIE_NOW;
      }
      if( nbr == 1 ) iterative= 1;	/* special file */
    }
    if( of_verbose > 1 )
    {
      fprintf( STD_CHANNEL, "%s is %s file\n",
               fn, iterative?"special":"normal" );
    }
    if( iterative )
    { /* special file */
      fs_min= SIZE_MIN;
      fs_max= SIZE_MAX;
      /* SIZE_MAX large enough ? */
      if( (offset=LSEEK(fd,SIZE_MAX,SEEK_SET)) == -1 )
      {
        fprintf( ERROR_CHANNEL, "%s: Cannot lseek `%s'. %s.\n",
                 *argv, fn, sys_errlist[errno] );
        retval= 1; goto DIE_NOW;
      }
      if( (nbr=read(fd,byte,1)) == -1 )
      {
        fprintf( ERROR_CHANNEL, "%s: Cannot read `%s'. %s.\n",
                 *argv, fn, sys_errlist[errno] );
        retval= 1; goto DIE_NOW;
      }
      if( nbr == 1 )
      {
        is_larger= 1;
        fs_min= SIZE_MAX;
      }
      /* iteration */
      while( fs_max > fs_min )
      {
#ifdef	USE_LLSEEK
        offset= ( fs_min + fs_max) / 2LL;
#ifdef DEBUG
        fprintf( DEBUG_CHANNEL, "DEBUG: offset= %lld\n", offset );
#endif
#else /* USE_LLSEEK */
        offset= ((ulong)fs_min + (ulong)fs_max) / 2L;   /* overflow prev. */
#ifdef DEBUG
        fprintf( DEBUG_CHANNEL, "DEBUG: offset= %ld\n", offset );
#endif
#endif /* USE_LLSEEK */
/*
  This works since (n + n+1)/2 == n.
  If you do not believe this just uncomment the following line:
        if( fs_max == fs_min+1 ) offset= fs_min;
*/
        if( LSEEK(fd,offset,SEEK_SET) == -1 )
        {
          fprintf( ERROR_CHANNEL, "%s: Cannot lseek `%s'. %s.\n",
                   *argv, fn, sys_errlist[errno] );
          retval= 1; goto DIE_NOW;
        }
        if( (nbr=read(fd,byte,1)) == -1 )
        {
#ifdef	DEBUG
fprintf( DEBUG_CHANNEL, "DEBUG: Read error (errno %d)\n", errno );
#endif
          if( of_ignore ) nbr= 0;
          else
          {
            fprintf( ERROR_CHANNEL, "%s: Cannot read `%s'. %s.\n",
                     *argv, fn, sys_errlist[errno] );
            retval= 1; goto DIE_NOW;
          }
        }
        if( nbr == 0 ) fs_max= offset;
        else           fs_min= offset+1;
      }
      fs= fs_max;       /* == fs_min */
    }
    /* print size */
    if( of_verbose )
      fprintf( STD_CHANNEL, "%s  ", fn );
#ifdef USE_LLSEEK
    fprintf( STD_CHANNEL, "%s%lld", is_larger?">":"", fs/size_divider );
#else /* USE_LLSEEK */
    fprintf( STD_CHANNEL, "%s%ld", is_larger?">":" ", fs/size_divider );
#endif /* USE_LLSEEK */
    if( of_verbose )
    {
      fprintf( STD_CHANNEL, " %cBytes%s", unit, iterative==1?" (iterative)":" (normal)" );
    }
    fprintf( STD_CHANNEL, "\n" );
    /* close file */
    if( close(fd) == -1 )
    {
      fprintf( ERROR_CHANNEL, "%s: Couldn't close `%s' successfully. %s.\n",
               *argv, fn, sys_errlist[errno] );
      retval= 1; goto DIE_NOW;
    }
  }
DIE_NOW:
  close( fd );
  exit( retval );
}
