/***
 * CopyPolicy: GNU Public License 2 applies
 * Copyright (C) 1994-1997o Eissfeldt heiko@colossus.escape.de
 *
 * Interface module for cdrom drive access
 *
 * Two interfaces are possible.
 *
 * 1. using 'cooked' ioctls()
 *    disadvantages: available for atapi, sbpcd and cdu31a drives only.
 *
 * 2. using the generic scsi device (for details see SCSI Prog. HOWTO).
 *    NOTE: a bug/misfeature in the kernel requires blocking signal
 *          SIGINT during SCSI command handling. Once this flaw has
 *          been removed, the sigprocmask SIG_BLOCK and SIG_UNBLOCK calls
 *          should removed, thus saving context switches.
 *
 * For testing purposes I have added a third simulation interface.
 *
 * Version 0.8: used experiences of Jochen Karrer.
 *              SparcLinux port fixes
 *              AlphaLinux port fixes
 *
 */
#if 0
#define SIM_CD
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <fcntl.h>
#include <assert.h>

#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/shm.h>

#if defined(HPUX)
# include <sys/scsi.h>
struct sg_header {};
#else

#include "lowlevel.h"
#include "byteorder.h"
/* some include file locations have changed with newer kernels */
#if LINUX_VERSION_CODE > 0x10300 + 97
#include <scsi/sg.h>
#include <scsi/scsi.h>
#else /* old stuff */
#include <linux/../../drivers/scsi/sg.h>
#include <linux/../../drivers/scsi/scsi.h>
#endif
/* HACK! the kernel header files were misdesigned for
 *  user applications.
 * #include <linux/blkdev.h> did not work
 */
struct request {		/* this is and may not used */
  int dummy;
};
#endif

#include "mycdrom.h"
#include "cdda2wav.h"
#include "interface.h"
#include "semshm.h"
#include "setuid.h"
#include "ringbuff.h"
#include "global.h"
#include "paranoia.h"

sigset_t sigset;    /* to circumvent a kernel problem with the generic driver */
struct sigaction sigac;
unsigned interface;
unsigned OFF;
unsigned buffers = 2;
unsigned nsectors = NSECTORS;
unsigned overlap = 1;

int trackindex_disp = 0;

/* buffer for cdrom audio data sector */
static unsigned char *bufferCddaRaw;
       unsigned char *bufferCdda;
static unsigned char *bufferTOC;
static unsigned char *SubQbuffer;
static unsigned char *cmd;
#if !defined(SIM_CD)
static int adjust_ssize;
#endif

/* Hack hack hack hack hack... makes the code clean, though. Zygo was here */
#ifndef SG_BIG_BUFF
#define SG_BIG_BUFF 4096	/* FIXME: should be the size of page */
#endif

/*******************************************************************
 *
 *	SCSI section
 *
 */

/* process a complete scsi cmd. Use either generic or ioctl interface. */
static int handle_scsi_cmd(unsigned cmd_len, 
			   unsigned in_size, 
			   unsigned char *i_buff, 
			   unsigned out_size, 
			   unsigned char *o_buff,
			   int expect_result)
{
#if defined(HPUX)
    struct sctl_io sctl_io;

    if (!cmd_len) return -1;
    if (!i_buff) return -1;
	
/* fprintf(stderr, "doing scsi: cmd=%x\n",*i_buff); */
	
    if (in_size && out_size) {
      /* cannot handle reading & writing to the device with one call */
      return -1;
    }
    
    memset(&sctl_io, 0, sizeof sctl_io);   /* clear reserved fields */
	
    sctl_io.cdb_length = cmd_len;
    memcpy(&(sctl_io.cdb[0]), i_buff, cmd_len);      /* copy command */
	
    if (in_size) {
      /* we'll write some data to the SCSI device */
      sctl_io.flags = 0;
      sctl_io.data_length = in_size;
      sctl_io.data = i_buff + cmd_len;
    } else {
      /* we'll read some data from the SCSI device */
      sctl_io.flags = SCTL_READ;
      sctl_io.data_length = out_size;
      sctl_io.data = (out_size)? o_buff : 0;		
    }
	
    sctl_io.max_msecs = 10000;              /* allow 10 seconds for cmd */
    
    if (ioctl(global.cd_fd, SIOC_IO, &sctl_io) < 0) {
      perror("scsi_io");
      exit(-1);
    }
    
    if (sctl_io.cdb_status == S_GOOD) {
      return 0; /* everything is fine... */
    }
    
    if (sctl_io.cdb_status == S_BUSY) {
      fprintf(stderr, "sctl_io: device is busy\n");
      return -1;
    }
    
    if (sctl_io.cdb_status == S_CHECK_CONDITION) {
      fprintf(stderr, "sctl_io: S_CHECK_CONDITION\n");
      if (sctl_io.sense_status != S_GOOD) {
	fprintf(stderr, "sctl_io: sense_status != S_GOOD\n");
      } else {
	fprintf(stderr, "sctl_io: sense_xfer=%d, sense_data=%x %x %x %x %x %x ...\n",
		sctl_io.sense_xfer, sctl_io.sense[0], sctl_io.sense[1], sctl_io.sense[2],
		sctl_io.sense[3], sctl_io.sense[4], sctl_io.sense[5]);			
      }
      return -1;
    }
	
    return 0;	

#else  /* not HPUX */
    int status = 0;
    unsigned char *buff_p;
    struct sg_header *sg_hd;

    /* safety checks */
    if (!cmd_len) return -1;
    if (!i_buff) return -1;
    if (OFF + cmd_len + in_size > SG_BIG_BUFF) return -1;
    if (OFF + out_size > SG_BIG_BUFF) return -1;

    if (!o_buff) out_size = 0;

    if (out_size > in_size + cmd_len) {
	buff_p = o_buff;
	memcpy(o_buff + OFF, i_buff + OFF, in_size + cmd_len);
    } else {
	buff_p = i_buff;
    }

    if (interface == GENERIC_SCSI) {
	/* generic scsi device services */
	sg_hd = (struct sg_header *) buff_p;
	sg_hd->reply_len = OFF + out_size;
	sg_hd->twelve_byte = cmd_len == 12;
	sg_hd->result = 0;	/* is not cleared by the write call */
#if	0
	sg_hd->pack_len; /* not used */
	sg_hd->pack_id;	/* not used */
	sg_hd->other_flags; 	/* not used */
#endif
        sigprocmask ( SIG_BLOCK, &sigset, NULL );
	status = write( global.cd_fd, buff_p, OFF + cmd_len + in_size );
	if ( status < 0 || status != OFF + cmd_len + in_size || sg_hd->result ) {
	    if (status == -1) {
		perror("generic scsi write: ");
		exit(status);
	    } else {
	       fprintf( stderr, "\ngeneric scsi write status = 0x%x, result = 0x%x cmd = %x\n", 
		       status, sg_hd->result, i_buff[OFF] );
	    }
	    return status;
	}

	status = read( global.cd_fd, buff_p, OFF + out_size);
	if ( status < 0 || status != OFF + out_size || (sg_hd->result && !expect_result) ) {
	  if (sg_hd->sense_buffer[0]==0xf0 && /* hack to make Plextor 6plex happy */
	      sg_hd->sense_buffer[1]==0x0 &&
	      sg_hd->sense_buffer[2]==0x5 &&
	      /* 3-6 are some sort of block address */
	      sg_hd->sense_buffer[7]==0xa &&
	      sg_hd->sense_buffer[8]==0x0 &&
	      sg_hd->sense_buffer[9]==0x0 &&
	      sg_hd->sense_buffer[10]==0x0 &&
	      sg_hd->sense_buffer[11]==0x0 &&
	      sg_hd->sense_buffer[12]==0xbf &&
	      sg_hd->sense_buffer[13]==0x0 &&
	      sg_hd->sense_buffer[14]==0x0 &&
	      sg_hd->sense_buffer[15]==0x0) {
	    ;
	  } else {
	    fprintf( stderr, "\nread(generic) status = 0x%x, result = 0x%x cmd = %x\n", 
		    status, sg_hd->result, i_buff[OFF] );
	    fprintf( stderr, "read(generic) sense "
		  "%x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x\n", 
		  sg_hd->sense_buffer[0],	     sg_hd->sense_buffer[1],
		  sg_hd->sense_buffer[2],	     sg_hd->sense_buffer[3],
		  sg_hd->sense_buffer[4],	     sg_hd->sense_buffer[5],
		  sg_hd->sense_buffer[6],	     sg_hd->sense_buffer[7],
		  sg_hd->sense_buffer[8],	     sg_hd->sense_buffer[9],
		  sg_hd->sense_buffer[10],	     sg_hd->sense_buffer[11],
		  sg_hd->sense_buffer[12],	     sg_hd->sense_buffer[13],
		  sg_hd->sense_buffer[14],	     sg_hd->sense_buffer[15]
		  );
	  }
	  sigprocmask ( SIG_UNBLOCK, &sigset, NULL );
	  return 1;
	} else
	    /* get output back */
	    if (out_size && out_size <= in_size + cmd_len)
		memcpy(o_buff, i_buff, out_size + OFF);
	/* Look if we got what we expected to get */
	sigprocmask ( SIG_UNBLOCK, &sigset, NULL );
	if (status == OFF + out_size) status = 0; /* got them all */
    }
    return status;
#endif
}

static unsigned char density = 0;
static unsigned char orgmode4 = 0xff;
static unsigned char orgmode10, orgmode11;

/* get CD capabilities page and determine whether drive supports MMC cmd set */
/* returns 0 if not an MMC drive */
/* returns 1 if an MMC drive which does not have the "CD-DA stream is accurate" bit set */
/* returns 2 if an MMC drive which has the "CD-DA stream is accurate" bit set */

static unsigned int 
get_mmc(void)
{
      /* MODE_SENSE */
      static unsigned char cmdblk0 [6] = { 
	  0x1A, /* MODE_SENSE */
	  0x00, /* return block descriptor */
	  0x2A, /* CD capabilities and mechanical status page */
	  0, /* reserved */
	  22, /* sizeof(modesense - OFF) */
	  0}; /* reserved */
   
      static unsigned char modesense[sizeof(struct sg_header) + 22]
          __attribute__ ((aligned (__alignof__ (struct sg_header))));
   
      int base;

      memcpy( cmd + OFF, cmdblk0, sizeof(cmdblk0) );
      memset(modesense, 0, sizeof modesense);
      /* do the scsi cmd */
      if (handle_scsi_cmd (6, 0, cmd, sizeof modesense - OFF, modesense, 1)) {
	  perror ("get_mmc mode sense failed\n");
	  return 0;
      }
   
      if ((modesense[OFF] & 0x3F) == 0x2A) {
	 base = OFF;
      } else if ((modesense[OFF+12] & 0x3F) == 0x2A) {
	 base = OFF+12;
      } else {
#ifdef NOTDEF
	 printf("Can't find mode sense data for page 2A\n");
#endif
	 return 0;
      }
#ifdef NOTDEF
   printf("MMC data: %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X\n",
	  modesense[base + 0], 
	  modesense[base + 1],
	  modesense[base + 2], 
	  modesense[base + 3], 
	  modesense[base + 4], 
	  modesense[base + 5], 
	  modesense[base + 6], 
	  modesense[base + 7], 
	  modesense[base + 8], 
	  modesense[base + 9], 
	  modesense[base + 10], 
	  modesense[base + 11], 
	  modesense[base + 12],
	  modesense[base + 13], 
	  modesense[base + 14], 
	  modesense[base + 15], 
	  modesense[base + 16], 
	  modesense[base + 17], 
	  modesense[base + 18], 
	  modesense[base + 19], 
	  modesense[base + 20], 
	  modesense[base + 21]);
#endif
      if ((modesense[base + 0] & 0x3F) == 0x2A &&
	  modesense[base + 1] >= 5 &&
	  (modesense[base + 5] & 0x01)) {
	 if (modesense[base + 5] & 0x02) {
	    return 2;
	 }
	 return 1;
      }
   return 0;
}

/* get current sector size from SCSI cdrom drive */
static unsigned int 
get_orig_sectorsize(unsigned char *m4, unsigned char *m10, 
		    unsigned char *m11)
{
      /* first get current values for density, etc. */
      /* MODE_SENSE */
      static unsigned char cmdblk0 [6] = { 
	  0x1A, /* MODE_SENSE */
	  0x00, /* return block descriptor */
	  0x01, /* page control current values, page 1 */
	  0, /* reserved */
	  12, /* sizeof(modesense - OFF) */
	  0}; /* reserved */

      static unsigned char modesense[sizeof(struct sg_header) + 12]
          __attribute__ ((aligned (__alignof__ (struct sg_header))));

      memcpy( cmd + OFF, cmdblk0, sizeof(cmdblk0) );
      /* do the scsi cmd */
      if (handle_scsi_cmd (6, 0, cmd, 12, modesense, 0))
	  perror ("get_orig_sectorsize mode sense failed\n");

      if (m4 != NULL)                       /* density */
        *m4 = modesense[OFF + 4];
      if (m10 != NULL)                      /* MSB sector size */
        *m10 = modesense[OFF + 10];
      if (m11 != NULL)                      /* LSB sector size */
        *m11 = modesense[OFF + 11];

      return (modesense[OFF + 10] << 8) + modesense[OFF + 11];
}


/* switch CDROM scsi drives to given sector size  */
static int set_sectorsize (unsigned int secsize)
{
  /* reserved, Medium type=0, Dev spec Parm = 0, block descriptor len 0 oder 8,
     Density (cd format) 
     (0=YellowBook, XA Mode 2=81h, XA Mode1=83h and raw audio via SCSI=82h),
     # blks msb, #blks, #blks lsb, reserved,
     blocksize, blocklen msb, blocklen lsb,
   */

  /* MODE_SELECT, page = SCSI-2  save page disabled, reserved, reserved, 
     parm list len, flags */
  static unsigned char cmdblk [6 + 12] = { 
                         0x15, /* MODE_SELECT */
			 0x10, /* no save page */
			    0, /* reserved */
			    0, /* reserved */
			    12, /* sizeof(mode) */
			    0, /* reserved */
       /* mode section */
			    0, 
                            0, 0, 
                            8,       /* Block Descriptor Length */
                            0,       /* Density Code */
                            0, 0, 0, /* # of Blocks */
                            0,       /* reserved */
                            0, 0, 0};/* Blocklen */
  unsigned char *mode = cmd + OFF + 6;
  int retval;

  if (orgmode4 == 0xff) {
    get_orig_sectorsize(&orgmode4, &orgmode10, &orgmode11);
  }

  memcpy( cmd + OFF, cmdblk, sizeof(cmdblk) );

  /* prepare to read cds in the previous mode */
  mode [4] = orgmode4; 	       /* normal density */
  mode [10] =  secsize >> 8;   /* block length "msb" */
  mode [11] =  secsize & 0xFF; /* block length lsb */

  /* do the scsi cmd */
  if ((retval = handle_scsi_cmd (6, 12, cmd, 0, NULL, 0)))
        perror ("setting sector size failed\n");

  return retval;
}


/* switch Toshiba/DEC and HP drives from/to cdda density */
static void EnableCddaModeSelect (int fAudioMode)
{
  /* reserved, Medium type=0, Dev spec Parm = 0, block descriptor len 0 oder 8,
     Density (cd format) 
     (0=YellowBook, XA Mode 2=81h, XA Mode1=83h and raw audio via SCSI=82h),
     # blks msb, #blks, #blks lsb, reserved,
     blocksize, blocklen msb, blocklen lsb,
   */

  /* MODE_SELECT, page = SCSI-2  save page disabled, reserved, reserved, 
     parm list len, flags */
  static unsigned char cmdblk [6 + 4 + 8] = { 
                         0x15, /* MODE_SELECT */
			 0x10, /* no save page */
			    0, /* reserved */
			    0, /* reserved */
			    12, /* sizeof(mode) */
			    0, /* reserved */
       /* mode section */
			    0, 
                            0, 0, 
                            8,       /* Block Descriptor Length */
       /* block descriptor */
                            0,       /* Density Code */
                            0, 0, 0, /* # of Blocks */
                            0,       /* reserved */
                            0, 0, 0};/* Blocklen */
  unsigned char *mode = cmd + OFF + 6;

  if (orgmode4 == 0xff && fAudioMode) {
    get_orig_sectorsize(&orgmode4, &orgmode10, &orgmode11);
  }

  memcpy( cmd + OFF, cmdblk, sizeof(cmdblk) );

  if (fAudioMode) {
    /* prepare to read audio cdda */
    mode [4] = density;  			/* cdda density */
    mode [10] = (CD_FRAMESIZE_RAW >> 8);   /* block length "msb" */
    mode [11] = (CD_FRAMESIZE_RAW & 0xFF);
  } else {
    /* prepare to read cds in the previous mode */
    mode [4] = orgmode4; /* 0x00; 			\* normal density */
    mode [10] = orgmode10; /* (CD_FRAMESIZE >> 8);  \* block length "msb" */
    mode [11] = orgmode11; /* (CD_FRAMESIZE & 0xFF); \* block length lsb */
  }

  /* do the scsi cmd */
  if (handle_scsi_cmd (6, 12, cmd, 0, NULL, 0))
        perror ("Audio mode switch failed\n");
}

/* read the table of contents from the cd and fill the TOC array */
static unsigned ReadTocSCSI ( TOC *toc )
{
  int i;
  unsigned tracks;

  /* READTOC, MSF format flag, res, res, res, res, Start track, len msb,
     len lsb, flags */
  static unsigned char cmdblk [] = { 
    0x43, 0, 0, 0, 0,   0, 1, CD_FRAMESIZE >> 8, CD_FRAMESIZE & 0xFF, 0
  };
  memcpy( cmd + OFF, cmdblk, sizeof(cmdblk) );

  /* do the scsi cmd (read table of contents) */
  if (handle_scsi_cmd ( sizeof(cmdblk), 0, cmd, CD_FRAMESIZE, bufferTOC, 0))
      FatalError ("Read TOC failed.\n");

  /* copy to our structure and convert start sector */
  tracks = ((bufferTOC [OFF + 0] << 8) + bufferTOC [OFF + 1] - 2) / 8;
  if (tracks > MAXTRK) return 0;

  for (i = 0; i < tracks; i++) {
      memcpy (&toc[i], bufferTOC + OFF + 4 + 8*i, 8);
      toc[i].ISRC[0] = 0;
      toc[i].dwStartSector = adjust_ssize * be32_to_cpu(toc[i].dwStartSector);
      if ( toc [i].bTrack != i+1 )
	  toc [i].bTrack = i+1;
  }
  return --tracks;           /* without lead-out */
}

/* a contribution from Boris for IMS cdd 522 */
#define READ_TRACK_INFO (0xe5)


/* read the table of contents from the cd and fill the TOC array */
static unsigned ReadTocCdrSCSI ( TOC *toc )
{
  unsigned char buffer[256 + OFF];
  int i;
  unsigned tracks;
  static const unsigned char cmdblk[10] = { READ_TRACK_INFO, 0, 0, 0, 0, 0, 0, 0, 0, 0};

  memcpy( cmd + OFF, cmdblk, sizeof(cmdblk) );
  cmd[OFF + 5] = 1;
  cmd[OFF + 8] = 255;
  /* do the scsi cmd (read table of contents) */
  if (handle_scsi_cmd ( sizeof(cmdblk), 0, cmd, 256, buffer, 0))
      FatalError ("Read TrackInfo failed.\n");

  /* copy to our structure and convert start sector */
  tracks = buffer[OFF + 1];
  if (tracks > MAXTRK) return 0;

  for (i = 0; i < tracks; i++)
    {

      memcpy( cmd + OFF, cmdblk, sizeof(cmdblk) );
      cmd[OFF + 5] = i + 1;
      cmd[OFF + 8] = 255;
      if (handle_scsi_cmd ( sizeof(cmdblk), 0, cmd, 256, buffer, 0))
       FatalError ("Read TrackInfo failed.\n");

      toc[i].reserved1 = 0;
      toc[i].bFlags = buffer[OFF + 10];
      toc[i].bTrack = i + 1;
      toc[i].reserved2 = 0;
      memcpy (&toc[i].dwStartSector, &buffer[OFF + 2], 4);
      toc[i].ISRC[0] = 0;
      toc[i].dwStartSector = adjust_ssize * be32_to_cpu(toc[i].dwStartSector);
    }

  toc[i].reserved1 = 0;
  toc[i].bFlags = 0;
  toc[i].bTrack = i + 1;
  toc[i].reserved2 = 0;
  memcpy (&toc[i].dwStartSector, &buffer[OFF + 2], 4);
  toc[i].ISRC[0] = 0;
  toc[i].dwStartSector = adjust_ssize * (be32_to_cpu(toc[i].dwStartSector) +
                                        be32_to_cpu((*(u_int32_t *)&buffer[OFF + 6])));

  return tracks;
}

/* Read max. SectorBurst of cdda sectors to bufferCdda+OFF
   via standard SCSI-2 Read(10) command */
static void ReadStandard (unsigned char *p, long lSector, unsigned long SectorBurstVal )
{
  /* READ10, flags, block1 msb, block2, block3, block4 lsb, reserved, 
     transfer len msb, transfer len lsb, block addressing mode */
  static unsigned char cmdblk [10] = {0x28, 0, 0, 0, 0, 0, 0, 0, 0, 0};

  unsigned char *cmd0 = cmd + OFF;
  int count = 3;

  memcpy( cmd + OFF, cmdblk, sizeof(cmdblk) );

  cmd0 [3] = (unsigned char)((lSector >> 16) & 0xFF);
  cmd0 [4] = (unsigned char)((lSector >> 8) & 0xFF);
  cmd0 [5] = (unsigned char)(lSector & 0xFF);

  cmd0 [8] = (unsigned char)SectorBurstVal;

  while (handle_scsi_cmd(sizeof(cmdblk), 0, cmd, 
			 SectorBurstVal * CD_FRAMESIZE_RAW, p - OFF, 0 )
	 && count--)
    ;
  if (!count)
    FatalError ("Read CD-ROM10 failed\n");
}


/* Read max. SectorBurst of cdda sectors to bufferCdda+OFF
   via vendor-specific ReadCdda(10) command */
static void ReadCdda10 (unsigned char *p, long lSector, unsigned long SectorBurstVal )
{
  /* READ10, flags, block1 msb, block2, block3, block4 lsb, reserved, 
     transfer len msb, transfer len lsb, block addressing mode */
  static unsigned char cmdblk [10] = {0xd4, 0, 0, 0, 0, 0, 0, 0, 0, 0};

  unsigned char *cmd0 = cmd + OFF;
  int count = 3;

  memcpy( cmd + OFF, cmdblk, sizeof(cmdblk) );

  cmd0 [3] = (unsigned char)((lSector >> 16) & 0xFF);
  cmd0 [4] = (unsigned char)((lSector >> 8) & 0xFF);
  cmd0 [5] = (unsigned char)(lSector & 0xFF);

  cmd0 [8] = (unsigned char)SectorBurstVal;

  while (handle_scsi_cmd(sizeof(cmdblk), 0, cmd, 
			 SectorBurstVal * CD_FRAMESIZE_RAW, p - OFF, 0 )
	 && count--)
    ;
  if (!count)
    FatalError ("Read CD-ROM10 (NEC) failed\n");
}

/* Read max. SectorBurst of cdda sectors to bufferCdda+OFF
   via vendor-specific ReadCdda(12) command */
static void ReadCdda12 (unsigned char *p, long lSector, unsigned long SectorBurstVal )
{
  static unsigned char cmdblk [12] = {0xd8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

  unsigned char *cmd0 = cmd + OFF;
  int count = 3;

  memcpy( cmd + OFF, cmdblk, sizeof(cmdblk) );

  cmd0 [3] = (unsigned char)((lSector >> 16) & 0xFF);
  cmd0 [4] = (unsigned char)((lSector >> 8) & 0xFF);
  cmd0 [5] = (unsigned char)(lSector & 0xFF);
  cmd0 [9] = (unsigned char)SectorBurstVal;

  while (handle_scsi_cmd(sizeof(cmdblk), 0, cmd, 
			 SectorBurstVal * CD_FRAMESIZE_RAW, p - OFF, 0 )
	 && count--)
    ;
/* Read max. SectorBurst of cdda sectors to bufferCdda+OFF
   via vendor-specific ReadCdda(12) command */
  if (!count)
    FatalError ("Read CD-ROM12 failed\n");
}

static void ReadCdda12MSF (unsigned char *p, long lSector, unsigned long SectorBurstVal )
{
  static unsigned char cmdblk [12] = {0xd9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

  unsigned char *cmd0 = cmd + OFF;
  unsigned int endsector = lSector + SectorBurstVal;
  int count = 3;

  memcpy( cmd + OFF, cmdblk, sizeof(cmdblk) );

  lSector += 150;
  endsector += 150;
  cmd0 [3] = lSector / (60 * 75);
  cmd0 [4] = (lSector / 75) % 60;
  cmd0 [5] = lSector % 75;
  cmd0 [7] = endsector / (60 * 75);
  cmd0 [8] = (endsector / 75) % 60;
  cmd0 [9] = endsector % 75;

  while (handle_scsi_cmd(sizeof(cmdblk), 0, cmd, 
			 SectorBurstVal * CD_FRAMESIZE_RAW, p - OFF, 0 )
	 && count--)
    ;
  if (!count)
    FatalError ("Read CD-ROM12MSF failed\n");
}

/* Read max. SectorBurst of cdda sectors to bufferCdda+OFF
   via MMC standard READ CD command */
static void ReadCddaMMC12 (unsigned char *p, long lSector, unsigned long SectorBurstVal )
{
  static unsigned char cmdblk [12] = {0xbe, 0x4, 0, 0, 0, 0, 0, 0, 0, 0xf0, 0, 0};

  unsigned char *cmd0 = cmd + OFF;
  int count = 3;

  memcpy( cmd + OFF, cmdblk, sizeof(cmdblk) );

  cmd0 [2] = (unsigned char)((lSector >> 24) & 0xFF);
  cmd0 [3] = (unsigned char)((lSector >> 16) & 0xFF);
  cmd0 [4] = (unsigned char)((lSector >> 8) & 0xFF);
  cmd0 [5] = (unsigned char)(lSector & 0xFF);
  cmd0 [8] = (unsigned char)SectorBurstVal;

  while (handle_scsi_cmd(sizeof(cmdblk), 0, cmd, 
			 SectorBurstVal * CD_FRAMESIZE_RAW, p - OFF, 0 )
	 && count--)
    ;
  if (!count)
    FatalError ("Read CD-ROMMMC12 failed\n");
}

/* Read the Sub-Q-Channel to SubQbuffer+OFF */
static subq_chnl *ReadSubQSCSI ( unsigned char format, unsigned char track )
{
  static unsigned char cmdblk [10] = {0x42, 0x02, 0x40, 0, 0, 0, 0, 0, 48, 0};

  unsigned char *cmd0 = cmd + OFF;
  memcpy( cmd + OFF, cmdblk, sizeof(cmdblk) );

  cmd0 [3] = format;
  cmd0 [6] = track;

  if (handle_scsi_cmd(sizeof(cmdblk), 0, cmd, 48, SubQbuffer, 0 ))
    FatalError ("Read SubQ failed\n");

  return (subq_chnl *)(SubQbuffer + OFF);
}

/********* non standardized speed selects ***********************/

static void SpeedSelectSCSIToshiba (unsigned speed)
{
  static unsigned char cmdblk [6 + 4 + 3] = { 
                         0x15, /* MODE_SELECT */
			 0x10, /* no save page */
			    0, /* reserved */
			    0, /* reserved */
			    4 + 3, /* sizeof(mode) */
			    0, /* reserved */
       /* mode section */
			    0, 
                            0, 0, 
                            0,       /* Block Descriptor Length */
       /* page */        0x20,       /* page code */
                            1,       /* parameter length */
                            0};      /* speed */
  unsigned char *page = cmd + OFF + 6 + 4;
  memcpy( cmd + OFF, cmdblk, sizeof(cmdblk) );

  if (speed != 1 && speed != 2) {
    fprintf(stderr, "invalid speed setting %d for Toshiba\n", speed);
    return;
  }

  page [2] = speed == 1 ? 0 : 1;   /* 0 for single speed */

  /* do the scsi cmd */
  if (handle_scsi_cmd (6, 7, cmd, 0, NULL, 0))
        perror ("speed select Toshiba failed\n");
}

static void SpeedSelectSCSINEC (unsigned speed)
{
  static unsigned char cmdblk [10 + 4 + 8] = { 
                         0xc5, /* ADDITIONAL MODE_SELECT */
			 0x10, /* no save page */
			    0, /* reserved */
			    0, /* reserved */
			    0, /* reserved */
			    0, /* reserved */
			    0, /* reserved */
			    0, /* reserved */
			    4 + 8, /* sizeof(mode) */
			    0, /* reserved */
       /* mode section */
			    0, 
                            0, 0, 
                            0,       /* Block Descriptor Length */
       /* page */        0x0f,       /* page code */
                            6,       /* parameter length */
                            0,       /* speed */
			    0,       /* reserved */
			    0,       /* reserved */
			    0,       /* reserved */
			    0,       /* reserved */
                            0};      /* reserved */
  unsigned char *page = cmd + OFF + 10 + 4;
  memcpy( cmd + OFF, cmdblk, sizeof(cmdblk) );

  if (speed != 1 && speed != 2) {
    fprintf(stderr, "invalid speed setting %d for NEC\n", speed);
    return;
  }

  /* bit 5 == 1 for single speed, otherwise double speed */
  page [2] = speed == 1 ? 1 << 5 : 0;

  /* do the scsi cmd */
  if (handle_scsi_cmd (10, 12, cmd, 0, NULL, 0))
        perror ("speed select NEC failed\n");
}

static void SpeedSelectSCSIPhilipsCDD2600 (unsigned speed)
{
  /* MODE_SELECT, page = SCSI-2  save page disabled, reserved, reserved,
     parm list len, flags */
  static unsigned char cmdblk [6 + 4 + 8] = {
                         0x15, /* MODE_SELECT */
                        0x10, /* no save page */
                           0, /* reserved */
                           0, /* reserved */
                           12, /* sizeof(mode) */
                           0, /* reserved */
       /* mode section */
                           0,
                            0, 0,
                            0,       /* Block Descriptor Length */
       /* page */
                            0x23,    /* Speed selection Code */
                            6,       /* Parameter length */
                            0,       /* write speed select=auto */
                            1,       /* emulation write=on */
                            0,       /* read speed select */
                            0,0,     /* reserved */
                            0        /* reserved */
       };
  unsigned char *page = cmd + OFF + 6 + 4;
  memcpy( cmd + OFF, cmdblk, sizeof(cmdblk) );

  if (speed != 1 && speed != 2 && speed != 4) {
    fprintf(stderr, "invalid speed setting %d for Philips CDD2600\n", speed);
    return;
  }

  page[2] = page [4] = speed;

  /* do the scsi cmd */
  if (handle_scsi_cmd (6, 12, cmd, 0, NULL, 0))
        perror ("speed select PhilipsCDD2600 failed\n");
}

static void SpeedSelectSCSISony (unsigned speed)
{
  static unsigned char cmdblk [6 + 4 + 4] = { 
                         0x15, /* MODE_SELECT */
			 0x10, /* no save page */
			    0, /* reserved */
			    0, /* reserved */
			    4 + 4, /* sizeof(mode) */
			    0, /* reserved */
       /* mode section */
			    0, 
                            0, 0, 
                            0,       /* Block Descriptor Length */
       /* page */        0x31,       /* page code */
                            2,       /* parameter length */
                            0,       /* speed */
                            0};      /* reserved */
  unsigned char *page = cmd + OFF + 6 + 4;
  memcpy( cmd + OFF, cmdblk, sizeof(cmdblk) );

  /* speed values > 1 are drive dependent */
  if (speed > 4) speed = 8;

  page [2] = (speed / 2);

  /* do the scsi cmd */
  if (handle_scsi_cmd (6, 8, cmd, 0, NULL, 0))
        perror ("speed select Sony failed\n");
}

static void SpeedSelectSCSIYamaha (unsigned speed)
{
  static unsigned char cmdblk [6 + 4 + 4] = { 
                         0x15, /* MODE_SELECT */
			 0x10, /* no save page */
			    0, /* reserved */
			    0, /* reserved */
			    4 + 4, /* sizeof(mode) */
			    0, /* reserved */
       /* mode section */
			    0, 
                            0, 0, 
                            0,       /* Block Descriptor Length */
       /* page */        0x31,       /* page code */
                            2,       /* parameter length */
                            0,       /* speed */
                            0};      /* reserved */
  unsigned char *page = cmd + OFF + 6 + 4;
  memcpy( cmd + OFF, cmdblk, sizeof(cmdblk) );

  /* speed values > 1 are drive dependent */
  if (speed > 4) speed = 8;

  page [2] = (speed / 2) << 4;

  /* do the scsi cmd */
  if (handle_scsi_cmd (6, 8, cmd, 0, NULL, 0))
        perror ("speed select Yamaha failed\n");
}

static void SpeedSelectSCSIMMC (unsigned speed)
{
   static unsigned char cmdblk [12] = {
      0xBB, /* SET SPEED */
	0, /* reserved */
	0, /* read speed MSB */
	0, /* read speed LSB */
	0, /* write speed MSB */
	0, /* write speed LSB */
	0, /* reserved */
	0, /* reserved */
	0, /* reserved */
	0, /* reserved */
	0, /* reserved */
	0}; /* reserved */

  int spd;

   memcpy( cmd + OFF, cmdblk, sizeof(cmdblk) );

   if (speed == 0) {
      spd = 0xFFFF;
   } else {
      spd = 176 * speed;
   }
   cmd[OFF + 2] = spd / 256;
   cmd[OFF + 3] = spd & 0xFF;
/*    printf ("Setting read speed to %d kB/sec\n", spd == 0xFFFF ? maxspd : spd); */
   /* do the scsi cmd */
   if (handle_scsi_cmd (12, 0, cmd, 0, NULL, 0)) {
      perror ("speed select MMC failed\n");
   }
}

/* request vendor brand and model */
unsigned char *Inquiry ( void )
{
  static unsigned char cmdblk [6] = {0x12, 0, 0, 0, 56, 0};
  static unsigned char Inqbuffer[sizeof(struct sg_header) + 56]
          __attribute__ ((aligned (__alignof__ (struct sg_header))));

  memcpy( cmd + OFF, cmdblk, sizeof(cmdblk) );

  if (handle_scsi_cmd(sizeof(cmdblk), 0, cmd, sizeof(Inqbuffer) - OFF, 
			    Inqbuffer, 0 )) {
      FatalError ("Inquiry failed\n");
  }

  return (Inqbuffer + OFF);
}

#define TESTUNITREADY_CMD 0
#define TESTUNITREADY_CMDLEN 6

#define ADD_SENSECODE 12
#define ADD_SC_QUALIFIER 13
#define NO_MEDIA_SC 0x3a
#define NO_MEDIA_SCQ 0x00

int TestForMedium ( void )
{
  /* request READY status */
  static unsigned char cmdblk [TESTUNITREADY_CMDLEN] = {
      TESTUNITREADY_CMD, /* command */
                      0, /* reserved */
                      0, /* reserved */
                      0, /* reserved */
                      0, /* reserved */
                      0};/* reserved */

  if (interface != GENERIC_SCSI) {
    return 1;
  }

  memcpy( cmd + OFF, cmdblk, sizeof(cmdblk) );

  if (handle_scsi_cmd(sizeof(cmdblk), 0, cmd, 
			    0, NULL, 1)) {
#if !defined (HPUX)
      FatalError ("Test unit ready failed\n");
#endif
      return 0;
  }

  return 
#if defined (HPUX)
    1;
#else
   *(((struct sg_header*)cmd)->sense_buffer +ADD_SENSECODE) != NO_MEDIA_SC ||
   *(((struct sg_header*)cmd)->sense_buffer +ADD_SC_QUALIFIER) != NO_MEDIA_SCQ;
#endif
}

/* hook */
static void Dummy (int Switch)
{
}

#if !defined(SIM_CD)
/* set the correct command set for *different*
 * vendor specific implementations. Was this really necessary, folks?
 */
static void SetupSCSI( void )
{
    unsigned char *p;

    if (interface != GENERIC_SCSI) {
	/* unfortunately we have the wrong interface and are
	 * not able to change on the fly */
	fprintf(stderr, "The generic SCSI interface and devices are required\n");
	exit(1);
    }

    /* check for the correct type of unit. */
    p = Inquiry();
#if defined (HPUX)
    if (*p != SCSI_CDROM && *p != SCSI_WORM) {
#else
    if (*p != TYPE_ROM && *p != TYPE_WORM) {
#endif
	fprintf(stderr, "this is neither a scsi cdrom nor a worm device\n");
	exit(1);
    }

    /* generic Sony type defaults */
    density = 0x0;
    EnableCdda = Dummy;
    ReadCdRom = ReadCdda12MSF;
    SelectSpeed = SpeedSelectSCSISony;

    /* check for brands and adjust special peculiaritites */

    /* If your drive is not treated correctly, you can adjust some things
       here:

       global.littleendian: should be to 1, if the CDROM drive or CD-Writer
		  delivers the samples in the native byteorder of the audio cd
		  (LSB first).
		  HP CD-Writers need it set to 0.
       NOTE: If you get correct wav files when using sox with the '-x' option,
             the endianess is wrong. You can use the -C option to specify
	     the value of global.littleendian.

     */
    switch (get_mmc()) {
     case 2:
       if ( global.littleendian == -1 )
	 global.littleendian = 1;
       overlap = 0;
       ReadCdRom = ReadCddaMMC12;
       SelectSpeed = SpeedSelectSCSIMMC;
       break;
     case 1:
       if ( global.littleendian == -1 )
	 global.littleendian = 1;
       ReadCdRom = ReadCddaMMC12;
       SelectSpeed = SpeedSelectSCSIMMC;
       break;
     case 0:
    if (!memcmp(p+8,"TOSHIBA", 7) ||
        !memcmp(p+8,"IBM", 3) ||
        !memcmp(p+8,"DEC", 3)) {
	density = 0x82;
	EnableCdda = EnableCddaModeSelect;
 	ReadCdRom = ReadStandard;
        SelectSpeed = SpeedSelectSCSIToshiba;
	if ( global.littleendian == -1 )
		global.littleendian = 1;
    } else if (!memcmp(p+8,"IMS",3) ||
               !memcmp(p+8,"KODAK",5) ||
               !memcmp(p+8,"RICOH",5) ||
               !memcmp(p+8,"HP",2) ||
               !memcmp(p+8,"PHILIPS",7) ||
               !memcmp(p+8,"PLASMON",7) ||
               !memcmp(p+8,"GRUNDIG CDR100IPW",17) ||
               !memcmp(p+8,"MITSUMI CD-R ",13)) {
	EnableCdda = EnableCddaModeSelect;
	ReadCdRom = ReadStandard;
        SelectSpeed = SpeedSelectSCSIPhilipsCDD2600;

	/* treat all of these as bigendian */
	if ( global.littleendian == -1 )
		global.littleendian = 0;

	/* no overlap reading for cd-writers */
	overlap = 0;
    } else if (!memcmp(p+8,"YAMAHA",6)) {
	EnableCdda = EnableCddaModeSelect;
        SelectSpeed = SpeedSelectSCSIYamaha;

	/* no overlap reading for cd-writers */
	overlap = 0;
	if ( global.littleendian == -1 )
		global.littleendian = 1;
    } else if (!memcmp(p+8,"PLEXTOR",7) ||
               !memcmp(p+8,"SONY",4)) {
/*	overlap = 0; */
	if ( global.littleendian == -1 )
		global.littleendian = 1;
        if (!memcmp(p+16, "CD-ROM CDU55E",13)) {
	   ReadCdRom = ReadCddaMMC12;
	}
    } else if (!memcmp(p+8,"NEC",3)) {
	ReadCdRom = ReadCdda10;
        SelectSpeed = SpeedSelectSCSINEC;
	if ( global.littleendian == -1 )
		global.littleendian = 1;
        if (!memcmp(p+29,"5022.0r",3)) /* I assume all versions of the 502 require this? */
               overlap = 0;           /* no overlap reading for NEC CD-ROM 502 */
    }
    }

    ReadToc = (!memcmp(p+8, "IMS", 3)) ? ReadTocCdrSCSI : ReadTocSCSI;
    ReadSubQ = ReadSubQSCSI;

    /* look if caddy is loaded */
#if defined(HPUX)
    {
      /* need to test for medium a few times to get a reliable answer... */
      int i;
      for (i=0; i < 3; i++) {
	if (TestForMedium()) return;
      }
    }
#endif
    while (!TestForMedium()) {
	fprintf(stderr,"load cdrom please and press enter");
	getchar();
    }
}

#if !defined (HPUX)
/*******************************************************************
 *
 *	cooked ioctl section
 *
 */
static struct cdrom_tochdr hdr;
static struct cdrom_tocentry entry[100];
static struct cdrom_read_audio arg;
static int err;

/* read the table of contents (toc) via the ioctl interface */
static unsigned ReadToc_cooked ( TOC *toc )
{
    int i;
    unsigned tracks;

    /* get TocHeader to find out how many entries there are */
    err = ioctl( global.cd_fd, CDROMREADTOCHDR, &hdr );
    if ( err != 0 ) {
	/* error handling */
	if (err == -1) {
	    if (errno == EPERM)
		fprintf( stderr, "Please run this program setuid root.\n");
	    perror("cooked: Read TOC ");
	    exit( err );
	} else {
	    fprintf( stderr, "can't get TocHeader (error %d).\n", err );
	    exit( -1 );
	}
    }
    /* get all TocEntries */
    for ( i = 0; i < hdr.cdth_trk1; i++ ) {
	entry[i].cdte_track = 1+i;
	entry[i].cdte_format = CDROM_LBA;
	err = ioctl( global.cd_fd, CDROMREADTOCENTRY, &entry[i] );
	if ( err != 0 ) {
	    /* error handling */
	    fprintf( stderr, "can't get TocEntry #%d (error %d).\n", i+1, err );
	    exit( -1 );
	}
    }
    entry[i].cdte_track = CDROM_LEADOUT;
    entry[i].cdte_format = CDROM_LBA;
    err = ioctl( global.cd_fd, CDROMREADTOCENTRY, &entry[i] );
    if ( err != 0 ) {
	/* error handling */
	fprintf( stderr, "can't get TocEntry LEADOUT (error %d).\n", err );
	exit( -1 );
    }
    tracks = hdr.cdth_trk1+1;
    for (i = 0; i < tracks; i++) {
        toc[i].bFlags = (entry[i].cdte_adr << 4) | (entry[i].cdte_ctrl & 0x0f);
        toc[i].bTrack = entry[i].cdte_track;
        toc[i].dwStartSector = entry[i].cdte_addr.lba;
    }
    return --tracks;           /* without lead-out */
}

/* read 'SectorBurst' adjacent sectors of audio sectors 
 * to Buffer '*p' beginning at sector 'lSector'
 */
static void ReadCdRom_cooked (unsigned char *p, long lSector, unsigned long SectorBurstVal )
{
  int retry_count=0;
  static int nothing_read = 1;

/* read 2352 bytes audio data */
  arg.addr.lba = lSector;
  arg.addr_format = CDROM_LBA;
  arg.nframes = SectorBurstVal;
  arg.buf = &p[0];

  do {
    err = ioctl(global.cd_fd, CDROMREADAUDIO, &arg);
    retry_count++;
  } while ((err) && (retry_count < 30));
  if (err != 0) {
      /* error handling */
      if (err == -1) {
	  if (nothing_read && (errno == EINVAL || errno == EIO))
	      fprintf( stderr, "Sorry, this driver and/or drive does not support cdda reading.\n");
	  perror("cooked: Read cdda ");
          fprintf(stderr, " sector %ld, buffer %p\n", lSector, p);
      } else {
	  fprintf(stderr, "can't read frame #%d (error %d).\n", 
		  arg.addr.lba, err);
      }
  } else {
    nothing_read = 0;
  }
}

/* request sub-q-channel information. This function may cause confusion
 * for a drive, when called in the sampling process.
 */
static subq_chnl *ReadSubQ_cooked ( unsigned char format, unsigned char track )
{
    struct cdrom_subchnl sub_ch;

    switch (format) {
      case GET_CATALOGNUMBER:
      if (!(err = ioctl(global.cd_fd, CDROM_GET_MCN, (struct cdrom_mcn *) SubQbuffer + OFF))) {
          subq_chnl *SQp = (subq_chnl *) SubQbuffer + OFF;
	  subq_catalog *SQPp = (subq_catalog *) &SQp->data;

          memmove(SQPp->media_catalog_number, SQp, 14);
          SQPp->mc_valid = 0x80;
      } else {
          return NULL;
      }
      break;
      case GET_POSITIONDATA:
      if (!(err = ioctl(global.cd_fd, CDROMSUBCHNL, &sub_ch))) {
	  /* copy to SubQbuffer */
	  subq_chnl *SQp = (subq_chnl *) (SubQbuffer + OFF);
	  subq_position *SQPp = (subq_position *) &SQp->data;
	  SQp->audio_status 	= sub_ch.cdsc_audiostatus;
	  SQp->format 		= sub_ch.cdsc_format;
	  SQp->control_adr	= (sub_ch.cdsc_adr << 4) | (sub_ch.cdsc_ctrl & 0x0f);
	  SQp->track 		= sub_ch.cdsc_trk;
	  SQp->index 		= sub_ch.cdsc_ind;
	  SQPp->abs_min 	= sub_ch.cdsc_absaddr.msf.minute;
	  SQPp->abs_sec 	= sub_ch.cdsc_absaddr.msf.second;
	  SQPp->abs_frame 	= sub_ch.cdsc_absaddr.msf.frame;
	  SQPp->trel_min 	= sub_ch.cdsc_reladdr.msf.minute;
	  SQPp->trel_sec 	= sub_ch.cdsc_reladdr.msf.second;
	  SQPp->trel_frame 	= sub_ch.cdsc_reladdr.msf.frame;
      } else {
	  if (err == -1) {
	      if (errno == EPERM)
		  fprintf( stderr, "Please run this program setuid root.\n");
	      perror("cooked: Read subq ");
	      exit( err );
	  } else {
	      fprintf(stderr, "can't read sub q channel (error %d).\n", err);
	      exit( -1 );
	  }
      }
      break;
      default:
          return NULL;
    } /* switch */
  return (subq_chnl *)(SubQbuffer + OFF);
}

/* Speed control */
static void SpeedSelect_cooked( unsigned speed )
{
    if (!(err = ioctl(global.cd_fd, CDROM_SELECT_SPEED, &speed))) {
    } else {
	if (err == -1) {
	    if (errno == EPERM)
		fprintf( stderr, "Please run this program setuid root.\n");
	    perror("cooked: Speed select ");
	    /*exit( err ); */
	} else {
	    fprintf(stderr, "can't set speed %d (error %d).\n", speed, err);
	    exit( -1 );
	}
    }
}

/* set function pointers to use the ioctl routines */
static void SetupCookedIoctl( char *pdev_name )
{
    struct stat statstruct;

    if (fstat(global.cd_fd, &statstruct)) {
      fprintf(stderr, "cannot stat cd %d\n",global.cd_fd);
      exit(1);
    }
    switch ((int)(statstruct.st_rdev >> 8)) {
    case CDU31A_CDROM_MAJOR:	/* sony cdu-31a/33a */
        if (nsectors >= 14) {
	  overlap = 10;
	}
        break;
    case MATSUSHITA_CDROM_MAJOR:	/* sbpcd 1 */
    case MATSUSHITA_CDROM2_MAJOR:	/* sbpcd 2 */
    case MATSUSHITA_CDROM3_MAJOR:	/* sbpcd 3 */
    case MATSUSHITA_CDROM4_MAJOR:	/* sbpcd 4 */
        /* some are more compatible than others */
	err = ioctl(global.cd_fd, CDROMAUDIOBUFSIZ, nsectors);
        if (err == -1) {
	  perror("ioctl(CDROMAUDIOBUFSIZ)");
        }
	break;
    }

    EnableCdda = Dummy;
    ReadCdRom = ReadCdRom_cooked;
    ReadToc = ReadToc_cooked;
    ReadSubQ = ReadSubQ_cooked;
    SelectSpeed = SpeedSelect_cooked;
}
#endif

/********************** General setup *******************************/

/* As the name implies, interfaces and devices are checked.  We also
   adjust nsectors, overlap, and interface for the first time here.
   Any unnecessary privileges (setuid, setgid) are also dropped here.
*/
static void Check_interface_for_device( struct stat *statstruct, char *pdev_name)
{
    static const char *int_names[]={
       "generic_scsi", "cooked_ioctl"
    };

    if (!S_ISCHR(statstruct->st_mode) &&
	!S_ISBLK(statstruct->st_mode)) {
      fprintf(stderr, "%s is not a device\n",pdev_name);
      exit(1);
    }

    switch ((int) (statstruct->st_rdev >> 8)) {
#if defined (HPUX)
    default:			/* ??? what is the proper value here */
#else
    case SCSI_GENERIC_MAJOR:	/* generic */
#endif
       if (!S_ISCHR(statstruct->st_mode)) {
	 fprintf(stderr, "%s is not a char device\n",pdev_name);
	 exit(1);
       }
       if (interface != GENERIC_SCSI) {
	 if (!S_ISCHR(statstruct->st_mode)) {
	   fprintf(stderr, "%s is not a char device\n",pdev_name);
	   exit(1);
	 }
	    
	 fprintf(stderr, "wrong interface (%s) for this device (%s)\n"
                 "set to generic_scsi\n",int_names[interface], pdev_name);
	 interface = GENERIC_SCSI;
       }
       nsectors = (SG_BIG_BUFF - OFF)/CD_FRAMESIZE_RAW;
       dontneedroot(); /* still may need group */
       break;
#if !defined (HPUX)
    case SCSI_CDROM_MAJOR:     /* scsi cd */
    default:
	if (!S_ISBLK(statstruct->st_mode)) {
	    fprintf(stderr, "%s is not a block device\n",pdev_name);
	    exit(1);
	}
	if (interface != COOKED_IOCTL) {
	    fprintf(stderr, "cdrom device (%s). "
		    "Setting interface to cooked_ioctl.\n", pdev_name);
	    interface = COOKED_IOCTL;
	}
        dontneedroot();    /* no special privilege required */
	break;
#endif
    }
    if (overlap >= nsectors)
      overlap = nsectors-1;
}


/* open the cdrom device */
static int OpenCdRom ( char *pdev_name )
{
  int cd_filedesc;
  struct stat statstruct, fstatstruct;

  if (stat(pdev_name, &statstruct)) {
      fprintf(stderr, "cannot stat device %s\n", pdev_name);
      exit(1);
  }
  Check_interface_for_device( &statstruct, pdev_name );

  if (interface == GENERIC_SCSI) {
#if defined(HPUX)
      /* need root privileges */
      needroot(1);
      cd_filedesc = open(pdev_name,O_RDWR);
      dontneedroot();
#else
      /* may need special group privileges if there is a group for SCSI
         generic devices and we are setgid. */
      needgroup(0);
      cd_filedesc = open(pdev_name,O_RDWR);
      dontneedgroup();
#endif
  } else {
      needgroup(0);
      cd_filedesc = open(pdev_name,O_RDONLY);
      dontneedgroup();
  }


  if (cd_filedesc < 0)
    FatalError ("open(%s): %s\n",pdev_name,strerror(errno));

  /* Do final security checks here */
  if (fstat(cd_filedesc, &fstatstruct)) {
    fprintf(stderr, "Could not fstat %s (fd %d): %s\n", pdev_name, cd_filedesc,
               strerror(errno));
    exit(1);
  }
  Check_interface_for_device( &fstatstruct, pdev_name );

  /* Watch for race conditions */
  if (fstatstruct.st_dev != statstruct.st_dev ||
      fstatstruct.st_ino != statstruct.st_ino) {
    fprintf(stderr,"Race condition attempted in OpenCdRom.  Exiting now.\n");
    exit(1);
  }
  return cd_filedesc;
}
#endif /* SIM_CD */

/******************* Simulation interface *****************/
#if	defined SIM_CD
static unsigned long sim_pos=0;

/* read 'SectorBurst' adjacent sectors of audio sectors 
 * to Buffer '*p' beginning at sector 'lSector'
 */
static void ReadCdRom_sim (unsigned char *p, long lSector, unsigned long SectorBurstVal )
{
  int loop=0;
  signed short *q = (signed short *) p;
  int joffset = 0;

#if 0
  /* jitter with a probability of jprob */
  if (random() <= jprob) {
    /* jitter up to jmax samples */
    joffset = random();
  }
#endif

  for (loop = lSector*CD_FRAMESAMPLES + joffset; 
       loop < (lSector+SectorBurstVal)*CD_FRAMESAMPLES + joffset; 
       loop++) {
    *q++ = 0;
    *q++ = 1;
  }
  sim_pos = (lSector+SectorBurstVal)*CD_FRAMESAMPLES + joffset; 
}

static unsigned sim_indices;

/* read the table of contents (toc) via the ioctl interface */
static unsigned ReadToc_sim ( TOC *toc )
{
    int i;
    unsigned tracks;
    int scenario;
    int scen[12][3] = { 
      {1,1,5000}, 
      {1,2,5000}, 
      {1,99,150}, 
      {2,1,5000}, 
      {2,2,5000}, 
      {2,99,150},
      {5,1,5000}, 
      {5,2,5000}, 
      {5,99,150}, 
      {99,1,1000}, 
      {99,2,1000}, 
      {99,99,150}, 
    };

    fprintf(stderr, "select one of the following TOCs\n"
	    "0 :  1 track  with  1 index\n"
	    "1 :  1 track  with  2 indices\n"
	    "2 :  1 track  with 99 indices\n"
	    "3 :  2 tracks with  1 index each\n"
	    "4 :  2 tracks with  2 indices each\n"
	    "5 :  2 tracks with 99 indices each\n"
	    "6 :  5 tracks with  1 index each\n"
	    "7 :  5 tracks with  2 indices each\n"
	    "8 :  5 tracks with 99 indices each\n"
	    "9 : 99 tracks with  1 index each\n"
	    "10: 99 tracks with  2 indices each\n"
	    "11: 99 tracks with 99 indices each\n"
	    );

    do {
      scanf("%d", &scenario);
    } while (scenario < 0 || scenario > sizeof(scen)/2/sizeof(int));

    /* build table of contents */

    tracks = scen[scenario][0] + 1;
    sim_indices = scen[scenario][1];

    for (i = 0; i < tracks; i++) {
        toc[i].bFlags = 0x1b;
        toc[i].bTrack = i + 1;
        toc[i].dwStartSector = i * scen[scenario][2];
    }
    toc[tracks-1].bTrack = 0xaa;
    return --tracks;           /* without lead-out */
}


/* request sub-q-channel information. This function may cause confusion
 * for a drive, when called in the sampling process.
 */
static subq_chnl *ReadSubQ_sim ( unsigned char format, unsigned char track )
{
    subq_chnl *SQp = (subq_chnl *) (SubQbuffer + OFF);
    subq_position *SQPp = (subq_position *) &SQp->data;
    unsigned long sim_pos1;
    unsigned long sim_pos2;

    if ( format != GET_POSITIONDATA ) return NULL;  /* not supported by sim */

    /* simulate CDROMSUBCHNL ioctl */

    /* copy to SubQbuffer */
    SQp->audio_status 	= 0;
    SQp->format 	= 0xff;
    SQp->control_adr	= 0xff;
    sim_pos1 = sim_pos/CD_FRAMESAMPLES;
    sim_pos2 = sim_pos1 % 150;
    SQp->track 		= (sim_pos1 / 150) + 1;
    SQp->index 		= min(sim_pos2, sim_indices);
    SQPp->abs_min 	= sim_pos1 / (75*60);
    SQPp->abs_sec 	= (sim_pos1 / 75) % 60;
    SQPp->abs_frame 	= sim_pos1 % 75;
    SQPp->trel_min 	= sim_pos2 / (75*60);
    SQPp->trel_sec 	= (sim_pos2 / 75) % 60;
    SQPp->trel_frame 	= sim_pos2 % 75;

    return (subq_chnl *)(SubQbuffer + OFF);
}

static void SetupSimCd(void)
{
    EnableCdda = Dummy;
    ReadCdRom = ReadCdRom_sim;
    ReadToc = ReadToc_sim;
    ReadSubQ = ReadSubQ_sim;

#if 0
    fprintf( stderr, "Please enter values for 'global.littleendian',  'overlap' :");
    scanf("%d,%d", &global.littleendian, &overlap); 
#else
    global.littleendian = 1;
    overlap = min(nsectors-1, 1);
#endif
}
#endif /* def SIM_CD */

/* perform initialization depending on the interface used. */
void SetupInterface( unsigned char *int_name )
{
    /* build signal set to block for during generic scsi */
    sigemptyset (&sigset);
    sigaddset (&sigset, SIGINT);
    sigaddset (&sigset, SIGPIPE);
    sigac.sa_handler = exit;
    sigemptyset(&sigac.sa_mask);
    sigac.sa_flags = 0;
    sigaction( SIGINT, &sigac, NULL);
    sigaction( SIGQUIT, &sigac, NULL);
    sigaction( SIGTERM, &sigac, NULL);
    sigaction( SIGHUP, &sigac, NULL);
    sigaction( SIGPIPE, &sigac, NULL);
    sigaction( SIGTRAP, &sigac, NULL);
    sigaction( SIGIOT, &sigac, NULL);

#if	defined SIM_CD
    OFF = 0;
    nsectors = 75;
    fprintf( stderr, "SIMULATION MODE !!!!!!!!!!!\n");
#else
    /* ensure interface is setup correctly */
    global.cd_fd = OpenCdRom ( global.dev_name );

#if defined(HPUX)
	
	/* gain exclusive access to the CDROM-drive */
	if (ioctl(global.cd_fd, SIOC_EXCLUSIVE, 1) < 0) {
		perror("could not get exclusive access to the specified device");
		exit(-1);
	}

#else
    /* ??? we should do the same for Linux, but how? */
#endif

    /* set offset OFF according to interface */
    switch (interface) {
    case GENERIC_SCSI:	OFF = sizeof(struct sg_header); 
	break;
    case COOKED_IOCTL:	OFF = 0;
	break;
    }
#endif

    /* Value of 'nsectors' must be defined here */
    assert(nsectors > 0);

    fill_buffer = request_shm_sem(4*sizeof(*fill_buffer) +
                    buffers*(OFF + nsectors * CD_FRAMESIZE_RAW),
                    (unsigned char **)&fill_buffer);
    last_buffer = fill_buffer + sizeof(*fill_buffer);
    bufferCddaRaw = (unsigned char *)last_buffer + sizeof(*last_buffer);
    bufferCdda = bufferCddaRaw + OFF;
    set_total_buffers(buffers, sem_id);

    /* request one sector for table of contents */
    bufferTOC = malloc( OFF + CD_FRAMESIZE );      /* assumes sufficient aligned addresses */
    /* SubQchannel buffer */
    SubQbuffer = malloc( OFF + 48 );               /* assumes sufficient aligned addresses */
    cmd = malloc( OFF + 18 );                      /* assumes sufficient aligned addresses */
    if ( !bufferTOC || !SubQbuffer || !cmd ) {
       fprintf( stderr, "Too low on memory. Giving up.\n");
       exit(2);
    }

#if	defined SIM_CD
    SetupSimCd();
#else
    adjust_ssize = 1;
    /* if drive is of type scsi, get vendor name */
    if (interface == GENERIC_SCSI) {
        unsigned sector_size;

	SetupSCSI();
        sector_size = get_orig_sectorsize(&orgmode4, &orgmode10, &orgmode11);
        if ( sector_size != 2048 && set_sectorsize(2048) ) {
	  fprintf( stderr, "Could not change sector size from %d to 2048\n", sector_size );
	  adjust_ssize = 2048 / sector_size;
        }
#if !defined (HPUX)
    } else {
	SetupCookedIoctl( global.dev_name );
#endif
    }
#endif	/* if def SIM_CD */
}
