/*
 * cdchanger.c 
 *
 * This loads a cdrom from a specified slot in a changer, and displays 
 * information about the changer status.  If the disk was mounted
 * it is unmounted and remounted after the new slot has been selected.
 *
 * Changer information is displayed if either the -v flag is specified
 * or no slot was specified.
 *
 * Based on code originally from Gerhard Zuber <zuber@berlin.snafu.de>.
 * Changer status information, and rewrite for the new common cdrom driver
 * interface by Erik Andersen <andersee@et.byu.edu>.
 *
 * Released under the GNU GPL,  see file COPYING for details.
 *
 * 1.0 Feb 97 initial release
 * 1.1 Apr 98 - bug fixes,  added some new code from 2.1.90's cdchange
 *	program to report on the type of the current disk.  Make the
 *      include of ucdrom.h only if [we think] it's needed - this file
 *	no longer exists on newer 2.1 kernels.
 *
 * Richard Sharman <rsharman@magma.ca>
 */

/* You can hard code these e.g. "/bin/mount" if mount and
 * umount are not in a normal user's path 
 */
#ifndef MOUNT
#define MOUNT "mount"
#endif
#ifndef UMOUNT
#define UMOUNT "umount"
#endif

int	eject_by_using_CDROMEJECT = 1 ;

#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <linux/cdrom.h>
#ifndef CDS_DISC_OK
/* This is a pre 2.1.70 kernel, so we need ucdrom.h */
#  define PRE70
#  include <linux/ucdrom.h>
#endif
#ifndef CDSL_CURRENT
#    error you need at least the 2.1 kernel
#endif
#include <getopt.h>
#include <sys/mount.h>
#include <linux/fs.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>

char	*program;
/* program_invocation_short_name is defined by GNU libc.
 * use argv[0] if this is not available.
 */
extern char * program_invocation_short_name ;
int	debug_flag = 0 ;
#define NO_SLOT -1
#define MAX_LINE 1024


void
usage(int level)
{
    fprintf(stderr, "Usage is:  
%s [-v]    [ -m dir | -n ]   [device] [slot]
%s [-v] -e [-w]              [device] [slot]
", program, program) ;
    if (level) {
	fprintf(stderr, "
Normally:
* See if mounted,  and if so then unmount.
* Switch to given slot.
* If previously mounted,  remount with same mount point type & options.

-m dir    Mount at specified mount point.   This is either a directory
          you have read and write permissions,  or an empty string
          (use -m \"\") in which case mount point and options are
          taken from /etc/fstab.
-n	  Do not remount.

-e	  Eject instead of remounting.
-w	  Wait after ejecting for another CD to be inserted and the
          door closed.  (Re)mount if applicable. This option implies `-e'

Other options:
-v	  verbose mode
-h	  show this help
-b	  do not open nonblocking (see man page)
-c        clear non-blocking after open (see man page)\n" 
) ;
    } else {
	fprintf(stderr, "use -h for help\n") ;
    }
}

  
int
check_mount_name(char *name) 
{
    struct stat buff ;

    if (stat(name, &buff) , 0) {
	perror("stat") ;
	return(0) ;
    }
    if (!S_ISDIR(buff.st_mode)) {
	fprintf(stderr, "%s is not a directory!\n", name)  ;
	return(0) ;
    }
    if (access(name, W_OK | R_OK | X_OK) != 0) {
	perror("access()") ;
	fprintf(stderr, 
		"read write or execute permissions missing on %s\n",
		name) ;
	return(0) ;
    }
    return(1) ;
}

int	wait_for_door_close(int fd,   int slot)
{
    /* Wait until tray closed.   Return
     * 0 -- when door is closed and slot has a disk in 
     * 1 -- when door is closed but this slot has no disk
     * 2 -- error,  or we gave up,  door is probably still open
     */
    /* slot is 0-based here */
    int	ctr = 0 ;
#define MAX_WAIT 20
    int	status = -1 ;

    if (debug_flag) {
	fprintf(stderr, "Checking to see if door is closed") ;
    }
    while (status != CDS_DISC_OK && ctr++ < MAX_WAIT) {
	status = ioctl (fd, CDROM_DRIVE_STATUS, CDSL_CURRENT) ;
	if (debug_flag) {
	    fprintf(stderr, ".") ;
	}
	sleep(1) ;
    }
    switch (status) {
    case CDS_DISC_OK:
	if (debug_flag) {
	    fprintf (stderr, "Ready.\n");
	}
	return 0 ;
	break ;
    case CDS_TRAY_OPEN:
	if (debug_flag) {
	    fprintf (stderr, "Tray Open.\n");
	}
	return 2 ;
	break ;
    case CDS_DRIVE_NOT_READY:
	if (debug_flag) {
	    fprintf (stderr, "Drive Not Ready.\n");
	}
	return 2 ;
	break ;
    default:
	if (debug_flag) {
	    fprintf (stderr, 
		     "This Should not happen!  status= %d\n", 
		     status);
	}
	return 2 ;
	break ;
    }
}


/* See if the device is mounted.
 * If so, return 1 with device_name, mount_name and mount_options
 * set to the current (mount) values.
 * If not, return 0.
 */
int
check_mount(char *device, char *device_name, char *mount_name,
	    char *mount_options)
{
    FILE	*f ;
    char	type[80], options[80] ;
    int	found, n ;
    char	buff[MAX_LINE] ;

    f = fopen("/proc/mounts", "r") ;
    if ( f == NULL ) {
	perror("could not open /proc/mounts") ;
	exit(1) ;
    }
    found = 0 ;
    n = strlen(device) ;
    while ((fgets(buff, MAX_LINE, f)) != NULL) {
	sscanf( buff, "%s %s %s %s",  device_name, mount_name,  type, options) ;
	if ((found = (strcmp(device, device_name) == 0))) {
	    break ;
	}
	/* Allow the user to specify the mount point instead
	 * of the device name.
	 */
	if ((found = (strcmp(device, mount_name) == 0))) {
	    device = device_name ;
	    break ;
	}
    }
    fclose(f) ;
    if (found) {
	sprintf(mount_options, "-t %s -o %s", type, options) ;
    }
    return found ;
}

int
main (int argc, char **argv)
{
    char	*device = NULL ;
    int		fd;           /* file descriptor for CD-ROM device */
    int		status;       /* return status for system calls */
    int		verbose = 0;
    int		slot = NO_SLOT ;
    int		x_slot ;
    int		total_slots_available;
    int		c ;	
    char	optstring[] = "bcdef:hm:nNvw" ;
    int		user_mount_flag = 0 ;	/* mount as user_mount_name */
    char	*user_mount_name = "" ;	/* set by -m name option */
    int		eject_flag = 0 ;	/* eject rather than [re-]mounting */
    int		inhibit_mount_flag = 0 ;	/*  do not remount */
    int		mount_flag = 0 ;	/* user specified mount dir ? */
    int		wait_flag = 0 ;		/* if ejecting, do we wait? */
    int		just_status = 1 ;	/* are we just checking status? */
    uid_t	old_uid ;
    char	cmd_line[1024] ;
    int		was_mounted = 0 ;	/* it was previously mounted */
    /* If the above is set,  these 2 are set from /proc/mounts: */
    char	mount_name[1024], device_name[1024] ;
    char	mount_options[200] = "" ;
    int		e ;	/* temp save for errno */
    int		open_nonblocking = 1 ;
    int		clear_nonblocking = 0 ;
#ifdef ALLOW_ORDINARY_CDROM
    int		not_really_a_changer = 0 ;
#endif
    int		n ;

    program = program_invocation_short_name ;
    while( (c = getopt(argc, argv, optstring)) != EOF ) {
	switch(c) {
	case 'b':	/* normally we open device non-blocking */
	    open_nonblocking = 0 ;
	    break ;
	case 'c':	/* clear the non-blocking after opening */
	    clear_nonblocking = 1 ;
	    break ;
	case 'd':	/* debug */
	    debug_flag++ ;
	    break ;
	case 'e':	/* eject after possibly unmounting */
	    eject_flag = 1 ;
	    just_status = 0 ;
	    break ;
	case 'f':	/* obsolete:  allow -f device_name */
	    device = optarg;
	    break ;
	case 'h':	/* help! */
	    usage(1) ;
	    exit(0) ;
	    break ;
	case 'm':	/* allow user to specify mount point */
	    /* An argument is required,  but allow an empty string
	     * to mount with no name or options which works if
	     * there is an entry in fstab
	     */
	    just_status = 0 ;
	    user_mount_flag = 1 ;
	    user_mount_name = optarg ;
	    if (strlen(user_mount_name) > 0) {
		if (!check_mount_name(user_mount_name)) {
		    fprintf(stderr, "%s: %s is not a valid mount point.
It must be a directory you have read and write permissions on.\n", 
			    program, user_mount_name) ;
		    exit(1) ;
		}
	    }
	    break ;
	case 'n':	/* do not remount */
	    inhibit_mount_flag = 1 ;
	    break ;
	case 'v':	/* verbose.  */
	    /* This option is implied if there is no slot specified
	     * an no "do something" options.
	     */
	    verbose++ ;
	    break ;
	case 'w':	/* wait after ejecting for a new disk to be
			 * inserted and the door closed */
	    wait_flag = 1 ;
	    eject_flag = 1 ;
	    just_status = 0 ;
	    break ;
	default:
	    usage(0) ;
	    exit(1) ;
	}
    }
	
    argv += (optind) ;	/* -> 1st argument */
    argc -= (optind) ;	/* # args */

    switch(argc) {
    case 0:
	break ;
    case 1:
	/* One argument -- it could be the device or the slot. */
	if (argv[0][0] == '/') {
	    device = argv[0] ;
	} else {
	    slot = atoi (argv[0]) - 1;
	    just_status = 0 ;
	}
	break ;
    case 2:
	/* Two arguments -- the first is the device. */
	just_status = 0 ;
	device = argv[0];
	slot = atoi (argv[1]) - 1;
	break ;
    default:
	fprintf (stderr, "usage: %s [-v] [<device>] [<slot>]\n",
		 program);
	fprintf (stderr, "       Slots are numbered 1 -- n.\n");
	exit (1);
    }
 
    /*		Validity checking of options and arguments.
     */

    if (device == NULL) {
#ifdef DEFAULT_DEVICE
	device = DEFAULT_DEVICE ;
#else
	fprintf(stderr, "%s: No device was specified!\n",
		program) ;
	exit(1) ;
#endif
    }

    if (user_mount_flag && (
			    (eject_flag && !wait_flag) || inhibit_mount_flag)) {
	fprintf(stderr,
		"%s: -m option is mutually exclusive with -e and -n options.\n",
		program) ;
	usage(0) ;
	exit(1) ;
    }

    if (eject_flag) {
	if (!wait_flag) {
	    inhibit_mount_flag = 1 ;
	}
    }


    /* See if it is currently mounted, 
     * and if so with what name, type and options 
     * so we can re-mount it.
     */

    was_mounted = check_mount(device, device_name, 
			      mount_name, mount_options) ;
    if (!was_mounted) {
	strcpy(device_name, device) ;
    }

    if (debug_flag) {
	if (was_mounted) {
	    printf("Mounted as: %s\n", 
		   mount_name) ;
	} else {
	    printf("Not currently mounted.\n") ;
	}
    }


    /* Become root... */
    if (debug_flag> 1) {
	system("id") ;
    }
    old_uid = getuid() ;
    if ( setuid(geteuid()) < 0 ) {
	perror("setuid failed") ;
	exit(2) ;
    }
    if (debug_flag > 1) {
	system("id") ;
    }
	
    /* Note: we use the system() call to umount rather than
     * calling umount() directly,  since the latter does
     * not update  /etc/mtab
     */
    if ( !just_status && was_mounted) {
	sprintf(cmd_line, "%s %s", UMOUNT, device) ;
	if (debug_flag) {
	    printf("%s: %s\n", program, cmd_line) ;
	}
	system(cmd_line) ;
    }

	
    /* open device */ 
    /* The O_NONBLOCK mode is necessary in case the door
     * is open or the current slot has no disk.
     */
    if (open_nonblocking) {
	if (debug_flag) {
	    fprintf(stderr, "opening with normal nonblocking mode\n") ; 
	}
	fd = open (device, O_RDONLY | O_NONBLOCK);
	if (fd >= 0) {
	    if (clear_nonblocking) {
		/* Appaprently the non-blocking status must be cleared, to
		 * avoid an occasional situation where the doors are
		 * locked and we can't do anyting...  
		 */
		if (debug_flag) {
		    fprintf(stderr, "clearing nonblocking mode\n") ; 
		}
		n = fcntl(fd, F_SETFL, O_RDONLY) ;
		if (n < 0) {
		    perror("Attempt to clear non-blocking status failed!") ;
		}
	    } else {
		if (debug_flag) {
		    fprintf(stderr, "not clearing nonblocking mode\n") ; 
		}
	    }
	}
    } else {
	if (debug_flag) {
	    fprintf(stderr, "opening WITHOUT O_NONBLOCK  mode\n") ; 
	}
	fd = open (device, O_RDONLY);
    }

    if (fd < 0) {
	e = errno ;
	fprintf (stderr, "%s: open failed for `%s': %s\n",
		 program, device, strerror (errno));
	if (e == EIO) {
	    fprintf(stderr, "Is door closed???\n") ;
	}
	exit (1);
    }

    /* Check CD player status */ 
    total_slots_available = ioctl (fd, CDROM_CHANGER_NSLOTS);
#ifdef ALLOW_ORDINARY_CDROM
    if (total_slots_available <= 1 ) {
	status = ioctl (fd, CDROM_DRIVE_STATUS, CDSL_CURRENT) ;
	switch(status) {
	case CDS_DISC_OK:
	case CDS_TRAY_OPEN:
	case CDS_DRIVE_NOT_READY:
	    total_slots_available = 1 ;
	    not_really_a_changer = 1 ;
	    break ;
	default:
	    break ;
	}
	if ( slot > 1 ) {
	    fprintf (stderr, "%s: Slot is not allowed, "
		     "device `%s' is not an ATAPI "
		     "compliant CD changer.\n", program, device);
	exit (1);
	}
	slot = NO_SLOT ;
    }
#endif

    if (total_slots_available <= 0 ) {
	fprintf (stderr, "%s: Device `%s' is not an ATAPI "
		 "compliant CD changer.\n", program, device);
	exit (1);
    }

    /* Change to given slot if specified */ 
    if (slot >= 0) {
	if (slot >= total_slots_available) {
	    fprintf (stderr, "Bad slot number.  "
		     "Should be 1 -- %d.\n",
		     total_slots_available);
	    exit (1);
	}

	/* load */ 
	status = ioctl (fd, CDROM_SELECT_DISC, slot);
	if (status != slot) {
	    perror("CDROM_SELECT_DISC") ;
	    fprintf(stderr, "Cannot switch to requested slot,  "
		    "CDROM_SELECT_DISC returned %d not %d\n",
		   status, slot) ;
	    if (status < 0) {
		eject_flag = 0 ;
		inhibit_mount_flag = 1 ;
	    }
	}
    }

    /* eject if required */
    if (eject_flag) {
	if (eject_by_using_CDROMEJECT) {
	    status = ioctl (fd, CDROMEJECT);
	    if (status != 0) {
		fprintf (stderr,
			 "%s: CDROMEJECT ioctl failed for `%s': %s\n",
			 program, device, strerror (errno));
		exit (1);
	    }
	} else {
	    status = ioctl (fd, CDROM_SELECT_DISC, CDSL_NONE) ;
	    if (status != 0) {
		fprintf (stderr,
			 "%s: CDROM_SELECT_DISC ioctl failed for `%s': %s\n",
			 program, device, strerror (errno));
		exit (1);
	    }
	}

	if (wait_flag) {
	    if (wait_for_door_close(fd, slot) != 0)
		inhibit_mount_flag = 1 ;
	}
    }

    if (verbose || just_status) {

	status = ioctl (fd, CDROM_SELECT_DISC, CDSL_CURRENT);
#ifdef ALLOW_ORDINARY_CDROM
	/* Status will be an error,  but make it look as if
	 * the first slot is the current one.
	 */
	if (status < 0) {
	    status = 0 ;
	}
#endif
	slot=status;

	printf ("Device: %s\n", device) ;
	printf ("Current slot: %d\n", status+1);
	printf ("Total slots available: %d\n",
		total_slots_available);

	printf ("Drive status: ");
	switch (ioctl (fd, CDROM_DRIVE_STATUS, CDSL_CURRENT)) {
	case CDS_DISC_OK:
	    printf ("Ready.\n");
	    break;
	case CDS_TRAY_OPEN:
	    printf ("Tray Open.\n");
	    break;
	case CDS_DRIVE_NOT_READY:
	    printf ("Drive Not Ready.\n");
	    break;
	default:
	    printf ("This Should not happen!");
	    break;
	}

	for (x_slot=0; x_slot<total_slots_available; x_slot++) {
	    printf ("Slot %2d: ", x_slot+1);
#ifdef ALLOW_ORDINARY_CDROM
	    if (not_really_a_changer) {
		status = ioctl(fd, CDROM_DRIVE_STATUS, CDSL_CURRENT) ;
	    } else {
		status = ioctl(fd, CDROM_DRIVE_STATUS, x_slot) ;
	    }
#else
	    status = ioctl(fd, CDROM_DRIVE_STATUS, x_slot) ;
#endif
	    switch (status) {
	    case CDS_DISC_OK:
		printf ("Disc present.");
		break;
	    case CDS_NO_DISC: 
		printf ("Empty slot.");
		break;
	    case CDS_TRAY_OPEN:
		printf ("CD-ROM tray open.\n");
		break;
	    case CDS_DRIVE_NOT_READY:
		printf ("CD-ROM drive not ready.\n");
		break;
	    case CDS_NO_INFO:
		printf ("No Information.");
		break;
	    default:
		fprintf(stderr, "ioctl(CDROM_DRIVE_STATUS) failed. %m\n");
		break;
	    }

#ifndef PRE70
	    /* In 2.1.70 the CDROM_DISC_STATUS ioctl shows the
	     * type of disk in the current slot.
	     */
	    if (slot == x_slot && status == CDS_DISC_OK) {
		status = ioctl (fd, CDROM_DISC_STATUS);
		if (status<0) {
		    perror(" CDROM_DISC_STATUS");
		}
		switch (status) {
		case CDS_AUDIO:
		    printf ("\tAudio disc.\t");
		    break;
		case CDS_DATA_1:
		case CDS_DATA_2:
		    printf ("\tData disc type %d.\t", status-CDS_DATA_1+1);
		    break;
		case CDS_XA_2_1:
		case CDS_XA_2_2:
		    printf ("\tXA data disc type %d.\t", status-CDS_XA_2_1+1);
		    break;
#ifdef CDS_MIXED
		case CDS_MIXED:
		    /* CDS_MIXED was added in 2.1.77 */
		    printf ("\tMixed (audio & data) disc.\t");
		    break;
#endif
		default:
		    printf ("\tUnknown disc type 0x%x!\t", status);
		    break;
		}
	    }
#endif
	    switch (ioctl (fd, CDROM_MEDIA_CHANGED, x_slot)) {
	    case 1:
		printf ("\t\tChanged.\n");
		break;
	    default:
		printf ("\n");
		break;
	    }
	}
    }

    /* close device */
    status = close (fd);
    if (status != 0) {
	fprintf (stderr, "%s: close failed for `%s': %s\n",
		 program, device, strerror (errno));
	exit (1);
    }

    /* don't [re-]mount if -n option */
    if (!just_status && !inhibit_mount_flag) {   
	mount_flag = was_mounted || user_mount_flag ;
	if (user_mount_flag) {
	    /* if -m dir given, use that name as mount point */
	    strcpy(mount_name, user_mount_name) ;
	    if (strlen(user_mount_name) == 0) {
		strcpy(mount_options, "") ;
	    }
	}
	if (mount_flag) {
	    sprintf(cmd_line, "%s %s %s %s", 
		    MOUNT, mount_options, device_name, mount_name) ;
	    if (debug_flag) {
		printf("%s: remounting:  %s\n", program, cmd_line);
	    }
	    system(cmd_line) ;
	} else {
	    if (debug_flag) {
		printf("%s: not [re]mounting\n", program) ;
	    }
	}
    }

    if ( setuid(old_uid) < 0 ) {
	perror("setuid back to old_uid failed") ;
	exit(2) ;
    }
    if (debug_flag > 1) {
	system("id") ;
    }

    if (verbose || just_status) {
	was_mounted = check_mount(device, device_name, 
				  mount_name, mount_options) ;

	if (was_mounted) {
	    printf("Mounted as: %s\n", 
		   mount_name) ;
	} else {
	    printf("Not currently mounted.\n") ;
	}
    }

    exit (0);
}
/* Local variables:  */
/* c-basic-offset: 4 */
/* End: */
