/*
 * smbmount -- mount a SMB server (LanManager, etc) on
 * the local host as a user-level NFS server.
 */

/*
Copyright Technical Research Centre of Finland (VTT), 
Information Technology Institute (TTE) 1993, 1994 - All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the names of VTT or TTE not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission. 

VTT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
VTT BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
*/

static char RCS_Id[] = "$Id: smbmount.c,v 1.2 1995/01/03 18:11:45 tml Exp $";

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#include <sys/time.h>
#include <sys/stat.h>
#define NFSCLIENT		/* SunOS seems to need this */
#include <sys/mount.h>
#include <sys/socket.h>
#include <sys/signal.h>
#include <syslog.h>
#include <netdb.h>

#include <netinet/in.h>

#include <rpc/xdr.h>
#include <nfs/nfs.h>
#include <rpc/rpc.h>

#ifdef __hpux
#define _KERNEL 1
#endif
#include <sys/resource.h>
#ifdef __hpux
#undef _KERNEL
#endif

#include "includes.h"

#include "util.h"
#include "smbmount.h"

#ifdef M_NEWTYPE
#define M_BITS M_NEWTYPE
#define NFS_MOUNT_TYPE "nfs"
#else
#define M_BITS 0
#define NFS_MOUNT_TYPE MOUNT_NFS
#endif

#ifdef __hpux
#define MOUNT_SYSCALL vfsmount
#define UMOUNT_SYSCALL umount
#else
#ifdef __aux
#define MOUNT_SYSCALL fsmount
#define UMOUNT_SYSCALL unmount
#else
#define MOUNT_SYSCALL mount
#define UMOUNT_SYSCALL unmount
#endif
#endif

extern char *getenv P((const char *));

static char *progname;
static SVCXPRT *xprt;
static char mountpoint[1024];

static void usage P((void));
static void setup_xlation_tables P((void));
static void mount_service P((const char *dir));
static void init_client P((void));
static void close_session P((smb_user *u));
static void toggledebug P((int));

static char *desthost;
static char *service;
static struct in_addr server_ip;
static fstring myname = "";
static int have_ip = 0;
int max_xmit = BUFFER_SIZE;
static int sesskey = 0;
BOOL readbraw_supported = False;
BOOL writebraw_supported = False;
time_t servertime = 0;

char *InBuffer = NULL, *OutBuffer = NULL;

extern BOOL NeedSwap;

int debug;
int open_for_attr = 0;
static int nodaemon;
static int netbios_ssn_port = 0;

static struct timeval timeout;

static void my_svc_run P((void));

static int open_sockets P((char *service, int port));
static int open_session P((char *inbuf, char *outbuf));

#if __STDC__
int
main(int argc,
     char **argv)
#else
int
main(argc, argv)
     int argc;
     char **argv;
#endif
{
  int c;
  extern char *optarg;
  extern int optind;
  char *p;
  char *revp;
  extern FILE *dbf;
  
  progname = strrchr(argv[0], '/');
  if (progname)
    progname++;
  else
    progname = argv[0];

  assert(sizeof(smb_fh) <= NFS_FHSIZE);

  fntransl = TRANSL_LATIN1;

  revp = strchr(revision, ' ') + 1;
  *(strchr(revp, ' ')) = 0;

  guest.u_uid = UID_NOT_OPEN;
  guest.u_name = guest.u_passwd = NULL;

  dbf = stdout;

  while ((c = getopt(argc, argv, "c:d:D:hI:p:tu:v")) != EOF)
    switch (c)
      {
      case 'c':
	/* Set file name translation */
	if (!strcmp(optarg, "none"))
	  fntransl = TRANSL_NONE;
	else if (!strcmp(optarg, "latin1"))
	  fntransl = TRANSL_LATIN1;
	else
	  {
	    fprintf(stderr, "Invalid -c option value\n");
	    usage();
	    exit(2);
	  }
	break;

      case 'd':
	/* Set debug file */
	freopen(optarg, "w", stdout);
	dbf = stdout;
	break;

      case 'D':
	/* Set debug level */
	DEBUGLEVEL = atoi(optarg);
	break;

      case 'h':
	usage();
	exit(0);

      case 'I':
	if ((server_ip.s_addr = inet_addr(optarg)) == INADDR_NONE)
	  {
	    fprintf(stderr, "Invalid IP address %s.\n", optarg);
	    exit(2);
	  }
	have_ip = 1;
	break;

      case 'p':
	netbios_ssn_port = atoi(optarg);
	break;

      case 't':
	debug |= DEBUG_TRACENFS;
	break;

      case 'u':
	guest.u_name = optarg;
	if ((p = strchr(optarg, ':')) != NULL)
	  *p++ = 0, guest.u_passwd = p;
	break;

      case 'v':
	/* Print version. */
	fprintf(stderr, "smbmount version %s\n", revp);
	exit(0);
	break;
      }

  if (optind != argc - 2)
    usage();
  
  if (debug || DEBUGLEVEL)
    nodaemon = 1;

  openlog(progname, LOG_PID|LOG_CONS, LOG_DAEMON);

  if (!netbios_ssn_port)
    {
      struct servent *se;

      if ((se = getservbyname("netbios_ssn", "tcp")) == NULL
	   || (se = getservbyname("netbios-ssn", "tcp")) == NULL)
	netbios_ssn_port = 139;
    }

  setup_xlation_tables();

  init_nfs_subr();

  service = argv[optind];
  mount_service(argv[optind+1]);

  signal(SIGINT, terminate);
  signal(SIGTERM, terminate);

  signal(SIGUSR1, toggledebug);

  my_svc_run();
  return 1;
}

#if __STDC__
static void
usage(void)
#else
static void
usage()
#endif  
{
  fprintf(stderr, "Usage: %s [ options ] server mountpoint\n\
Options are:\n\
    -c charset     set file name translation mode, one of: none, latin1\n\
    -d logfile     set debugging log\n\
    -D level       set debugging level\n\
    -h             print this text\n\
    -I ip-address  set server's IP address\n\
    -p port        set port number\n\
    -t             trace NFS protocol to stdout\n\
    -u user:password\n\
                   set default user name and password\n\
    -v             print program version\n\
",
	  progname);
  exit(2);
}

#if __STDC__
static void
setup_transp_xlation_tables(void)
#else
static void
setup_transp_xlation_tables()
#endif
{
  int i;

  for (i = 0; i < 0400; i++)
    local_to_smb[i] = smb_to_local[i] = i;

  local_to_smb['\\'] = '/';
  local_to_smb['/'] = '\\';
  smb_to_local['/'] = '\\';
  smb_to_local['\\'] = '/';
}

#if __STDC__
static void
setup_xlation_tables(void)
#else
static void
setup_xlation_tables()
#endif
{
  int i;

  setup_transp_xlation_tables();
}

#if __STDC__
static void
mount_service(const char *dir)
#else
static void
mount_service(dir)
     char *dir;
#endif
{
  static struct nfs_args nfs_args;
  struct sockaddr_in sin;
  static char fs_hostname[100];
  static char namebuf[256];
  struct stat st;
  int pid;
  int backslashes;

  extern void nfs_program_2();

  stat(dir, &st);

  if (!(S_ISDIR(st.st_mode)) || getuid() != st.st_uid)
    {
      fprintf(stderr, "%s is not a good mount point\n", dir);
      exit(1);
    }

  if ((backslashes = count_chars(service,'\\')) != 3)
    {
      usage();
      printf("\n%s: %s '\\' characters in service\n",
	     service, backslashes < 3 ? "Not enough" : "Too many");
      exit(1);
    }

  if (!open_sockets(service, netbios_ssn_port))
    exit(1);

  init_client();

  xprt = svcudp_create(RPC_ANYSOCK);

  if (xprt == NULL)
    {
      fprintf(stderr, "Cannot create UDP service.\n");
      exit(1);
    }

  if (!svc_register(xprt, NFS_PROGRAM, NFS_VERSION, nfs_program_2, 0))
    {
      fprintf(stderr, "Unable to register RPC service.\n");
      exit(1);
    }

  /* Unless printing debugging output, run as a daemon. */
  if (!nodaemon)
    {
#ifdef RLIMIT_CORE
      struct rlimit rl;
#endif

      close(0);
      close(1);
      close(2);

      if (fork() > 0)
	exit(0);

#ifdef RLIMIT_CORE
      /* No core files, thanks -- they contain passwords. */
      rl.rlim_cur = rl.rlim_max = 0;
      if (setrlimit(RLIMIT_CORE, &rl) == -1)
	syslog(LOG_ERR, "setrlimit(RLIMIT_CORE): %m");
#endif	
      umask(077);

      setsid();
    }

  InBuffer = xmalloc(BUFFER_SIZE);
  OutBuffer = xmalloc(BUFFER_SIZE);
  
  memset(OutBuffer,0,smb_size);

  open_session(InBuffer, OutBuffer);

  get_myaddress(&sin);
  sin.sin_family = AF_INET;
  sin.sin_port = xprt->xp_port;

  nfs_args.addr = &sin;
  nfs_args.fh = (fhandle_t *) &toplevel_fh_allocation;
  nfs_args.flags = 0;

  /* We have already daemonified ourselves, so getpid() is now correct. */
  sprintf(fs_hostname, "%s:pid=%d", progname, getpid());

#ifdef NFSMNT_HOSTNAME
  nfs_args.flags |= NFSMNT_HOSTNAME;
  nfs_args.hostname = fs_hostname;
#endif

  nfs_args.flags |= NFSMNT_INT | NFSMNT_SOFT;

#ifdef NFSMNT_FSNAME
  nfs_args.flags |= NFSMNT_FSNAME;
  nfs_args.fsname = nfs_args.hostname;
#endif

  strcpy(mountpoint, dir);

  /*
   * Must fork before mounting, because the kernel wants
   * our rpc service while handling the mount.
   */
  if ((pid = fork()) == 0)
    {
      if (MOUNT_SYSCALL(NFS_MOUNT_TYPE, mountpoint,
			M_BITS | M_NOSUID, (caddr_t)&nfs_args) == -1)
	{
	  syslog(LOG_CRIT, "mount: %m");
	  _exit(1);
	}
#ifdef NEED_MOUNT_FAKE
      {
	char cmd[1000];
	sprintf(cmd, "/etc/mount -f -t nfs %s %s",
		fs_hostname, mountpoint);
	
	system(cmd);
      }
#endif      
      _exit(0);
    }
}

static int
open_sockets(char *service, int port)
{
  char *host;
  fstring service2;
  extern int Client;

  strupper(service);

  strcpy(service2, service);
  host = strtok(service2, "\\");
  desthost = strdup(host);
  strupper(desthost);

  if (*myname == 0)
    {
      get_myname(myname, NULL);
      strupper(myname);
    }

  if (!have_ip)
    {
      struct hostent *hp;

      if ((hp = gethostbyname(host)) == 0) 
	{
	  fprintf(stderr, "Unknown host %s.", host);
	  return 0;
	}

      memcpy((char *)&server_ip.s_addr, (char *)hp->h_addr, 4);
    }

  Client = open_socket_out(&server_ip, port);
  if (Client == -1)
    return 0;

  {
    int one=1;
    setsockopt(Client,SOL_SOCKET,SO_KEEPALIVE,(char *)&one,sizeof(one));
  }

  return 1;
}

static int
open_session(char *inbuf, char *outbuf )
{
  CONST struct {
    int prot;
    CONST char *name;
  }
  prots[] = 
    {
      {PROTOCOL_CORE,"PC NETWORK PROGRAM 1.0"},
      {PROTOCOL_COREPLUS,"MICROSOFT NETWORKS 1.03"},
      {PROTOCOL_LANMAN1,"MICROSOFT NETWORKS 3.0"},
      {PROTOCOL_LANMAN1,"LANMAN1.0"},
      {PROTOCOL_LANMAN2,"LM1.2X002"},
      {-1,NULL}
    };
  char *p;
  int len = 4;
  int numprots;

  NeedSwap = big_endian();

  /* send a session request (RFC 8002) */
  CVAL(outbuf,0) = 0x81;

  /* put in the destination name */
  p = outbuf+len;
  name_mangle(desthost,p);
  len += name_len(p);

  /* and my name */
  p = outbuf+len;
  name_mangle(myname,p);
  len += name_len(p);

  /* setup the packet length */
  /* We can't use smb_setlen here as it assumes a data
     packet and will trample over the name data we have copied
     in (by adding 0xFF 'S' 'M' 'B' at offsets 4 - 7 */

  SSVAL(outbuf,2,len);
  BSWP(outbuf+2,2);

  if (len >= (1 << 16))
    CVAL(outbuf,1) |= 1;

  send_smb(outbuf);
  DEBUG(5,("Sent session request\n"));

  receive_smb(inbuf,0);
 
  if (CVAL(inbuf,0) != 0x82)
    {
      int ecode = CVAL(inbuf,4);
      DEBUG(0,("Session request failed (%d,%d) with myname=%s destname=%s\n",
	    CVAL(inbuf,0),ecode,myname,desthost));
      switch (ecode)
	{
	case 0x80: 
	  DEBUG(0,("Not listening on called name\n")); 
	  DEBUG(0,("Try to connect to another name (instead of %s)\n",desthost));
	  DEBUG(0,("You may find the -I option useful for this\n"));
	  break;
	case 0x81: 
	  DEBUG(0,("Not listening for calling name\n")); 
	  DEBUG(0,("Try to connect as another name (instead of %s)\n",myname));
	  DEBUG(0,("You may find the -n option useful for this\n"));
	  break;
	case 0x82: 
	  DEBUG(0,("Called name not present\n")); 
	  DEBUG(0,("Try to connect to another name (instead of %s)\n",desthost));
	  DEBUG(0,("You may find the -I option useful for this\n"));
	  break;
	case 0x83: 
	  DEBUG(0,("Called name present, but insufficient resources\n")); 
	  DEBUG(0,("Perhaps you should try again later?\n")); 
	  break;
	case 0x8F:
	  DEBUG(0,("Unspecified error 0x%X\n",ecode)); 
	  DEBUG(0,("Your server software is being unfriendly\n"));
	  break;	  
	}
      return(False);
    }      

  memset(outbuf,0,smb_size);

  /* setup the protocol strings */
  {
    int plength;
    char *p;

    for (numprots=0,plength=0;prots[numprots].name;numprots++)
      plength += strlen(prots[numprots].name)+2;
    
    set_message(outbuf,0,plength,True);

    p = smb_buf(outbuf);
    for (numprots=0;prots[numprots].name;numprots++)
      {
	*p++ = 2;
	strcpy(p,prots[numprots].name);
	p += strlen(p) + 1;
      }
  }

  setup_pkt(outbuf, SMBnegprot, NULL);

  CVAL(smb_buf(outbuf),0) = 2;

  send_smb(outbuf);
  receive_smb(inbuf,0);

  if (CVAL(inbuf,smb_rcls) != 0 || ((int)SVAL(inbuf,smb_vwv0) >= numprots))
    {
      DEBUG(0,("SMBnegprot failed. myname=%s destname=%s - %s \n",
	    myname,desthost,smb_errstr(inbuf)));
      return(False);
    }

  max_xmit = MIN(max_xmit,(int)SVAL(inbuf,smb_vwv2));

  DEBUG(3,("Sec mode %d\n",SVAL(inbuf,smb_vwv1)));
  DEBUG(3,("max xmt %d\n",SVAL(inbuf,smb_vwv2)));
  DEBUG(3,("max mux %d\n",SVAL(inbuf,smb_vwv3)));
  DEBUG(3,("max vcs %d\n",SVAL(inbuf,smb_vwv4)));
  DEBUG(3,("max blk %d\n",SVAL(inbuf,smb_vwv5)));
  DEBUG(3,("time zone %d\n",SVAL(inbuf,smb_vwv10)));

  sesskey = IVAL(inbuf,smb_vwv6);

  servertime = make_unix_date(inbuf + smb_vwv8);

  DEBUG(3,("Got %d byte crypt key\n",strlen(smb_buf(inbuf))));

  DEBUG(3,("Chose protocol [%s]\n",prots[SVAL(inbuf,smb_vwv0)].name));
  Protocol = prots[SVAL(inbuf,smb_vwv0)].prot;
  if (Protocol >= PROTOCOL_COREPLUS)
    {
      readbraw_supported = ((SVAL(inbuf,smb_vwv5) & 0x1) != 0);
      writebraw_supported = ((SVAL(inbuf,smb_vwv5) & 0x2) != 0);
    }

  if (Protocol >= PROTOCOL_LANMAN1)
    {
      DEBUG(1,("Server time is %s (timezone is %d mins from GMT)\n",
	    asctime(LocalTime(&servertime,LOCAL_TO_GMT)),
	    (int16)SVAL(inbuf,smb_vwv10)));	    
    }

  return True;
}

#if __STDC__
static void
init_client(void)
#else
static void
init_client()
#endif
{
  
}

#if __STDC__
int
login_user(smb_user *user)
#else
int
login_user(user)
     smb_user *user;
#endif
{
  CONST char *dev = "A:";
  CONST char *pass;
  char *p;
  
  if (Protocol >= PROTOCOL_LANMAN1)
    {
      /* Send a session setup command */
      memset(OutBuffer, 0, smb_size);
      set_message(OutBuffer, 10,
		  2 + strlen(user->u_name) + strlen(user->u_passwd),
		  True);
      setup_pkt(OutBuffer, SMBsesssetupX, NULL);
      CVAL(OutBuffer, smb_vwv0) = 0xFF;
      SSVAL(OutBuffer, smb_vwv2, max_xmit);
      SSVAL(OutBuffer, smb_vwv3, 2);
      SSVAL(OutBuffer, smb_vwv4, getpid());
      SIVAL(OutBuffer, smb_vwv5, sesskey);
      SSVAL(OutBuffer, smb_vwv7, strlen(user->u_passwd) + 1);
      p = smb_buf(OutBuffer);
      strcpy(p, user->u_passwd);
      p += strlen(p) + 1;
      strcpy(p, user->u_name);

      send_smb(OutBuffer);
      receive_smb(InBuffer, 0);      

      if (CVAL(InBuffer,smb_rcls) != 0)
	{
	  DEBUG(0,("Session setup failed for username=%s myname=%s destname=%s   %s\n",
		user->u_name,myname,desthost,smb_errstr(InBuffer)));
	  return 0;
	}

      /* Use the returned uid from now on */
      user->u_uid = SVAL(InBuffer, smb_uid);
    }

  /* Now we've got a connection - send a tcon message */
  memset(OutBuffer,0,smb_size);

  pass = user->u_passwd;
 again:
  set_message(OutBuffer, 0,
	      6 + strlen(service) + strlen(user->u_passwd) + strlen(dev),
	      True);
  user->u_tid = 0;
  setup_pkt(OutBuffer, SMBtcon, user);

  p = smb_buf(OutBuffer);
  *p++ = 4;
  strcpy(p, service);
  p += strlen(p) + 1;
  *p++ = 4;
  strcpy(p, pass);
  p += strlen(p) + 1;
  *p++ = 4;
  strcpy(p, dev);

  send_smb(OutBuffer);
  receive_smb(InBuffer,0);

  /* trying again with a blank password */
  if (CVAL(InBuffer, smb_rcls) != 0 && 
      strlen(pass) > 0 && 
      Protocol >= PROTOCOL_LANMAN1)
    {
      DEBUG(0,("first SMBtcon failed, trying again. %s\n",smb_errstr(InBuffer)));
      pass = "";
      goto again;
    }  

  if (CVAL(InBuffer,smb_rcls) != 0)
    {
      DEBUG(0,("SMBtcon failed. %s\n",smb_errstr(InBuffer)));
      return 0;
    }
  

  max_xmit = SVAL(InBuffer, smb_vwv0);
  max_xmit = MIN(max_xmit, BUFFER_SIZE - 4);
  if (max_xmit <= 0)
    max_xmit = BUFFER_SIZE - 4;

  user->u_tid = SVAL(InBuffer, smb_vwv1);

  DEBUG(3,("Connected with tid=%d max_xmit=%d\n", user->u_tid, max_xmit));

  return 1;
}

#if __STDC__
static void
close_session(smb_user *u)
#else
static void
close_session(u)
     smb_user *u;
#endif
{
  u->u_uid = UID_NOT_OPEN;
}

#if __STDC__
void
logout(smb_user *u)
#else
void
logout(u)
     smb_user *u;
#endif
{
  u->u_uid = UID_NOT_OPEN;
}  

#if __STDC__
int
login(int srn, int uam, const char *user, const char *passwd)
#else
int
login(srn, uam, user, passwd)
     int srn;
     int uam;
     char *user;
     char *passwd;
#endif
{
  return 1;
}

#if __STDC__
void
terminate(int signo)
#else
void
terminate(signo)
     int signo;
#endif
{
  signal(SIGINT, SIG_IGN);
  signal(SIGTERM, SIG_IGN);

  if (fork() == 0)
    {
      if (UMOUNT_SYSCALL(mountpoint) == -1)
	syslog(LOG_ERR, "umount %s: %m", mountpoint);
      _exit(0);
    }
  exit(0);
}

#if __STDC__
static void
toggledebug(int signo)
#else
static void
toggledebug(signo)
     int signo;
#endif
{
  signal(SIGUSR1, toggledebug);

  /* If we haven't closed stderr, toggle tracing. */
  if (nodaemon)
    if (debug & DEBUG_TRACENFS)
      debug &= ~DEBUG_TRACENFS;
    else
      debug |= DEBUG_TRACENFS;
}

/****************************************************************************
setup basics in a outgoing packet
****************************************************************************/
void setup_pkt(char *outbuf, int command, const smb_user *user)
{
  CVAL(outbuf, smb_com) = command;
  SSVAL(outbuf, smb_pid, getpid());
  if (user)
    {
      SSVAL(outbuf, smb_tid, user->u_tid);
      SSVAL(outbuf, smb_uid, user->u_uid);
    }
  else
    SSVAL(outbuf, smb_uid, 0);
  SSVAL(outbuf, smb_mid, 1);
  if (Protocol > PROTOCOL_CORE)
    {
      CVAL(outbuf, smb_flg) = 0x8;
      SSVAL(outbuf, smb_flg2, 0x3);
    }
}

/* This is a dummy lp_keepalive() for the client only */
int lp_keepalive()
{
return(0);
}

struct
{
  int code;
  char *class;
} err_classes[] = { 
  {0,"SUCCESS"},
  {0x01,"ERRDOS"},
  {0x02,"ERRSRV"},
  {0x03,"ERRHRD"},
  {0x04,"ERRXOS"},
  {0xE1,"ERRRMX1"},
  {0xE2,"ERRRMX2"},
  {0xE3,"ERRRMX3"},
  {0xFF,"ERRCMD"},
  {-1,NULL}};

/****************************************************************************
return a SMB error string from a SMB buffer
****************************************************************************/
char *smb_errstr(char *inbuf)
{
  static pstring ret;
  int class = CVAL(inbuf,smb_rcls);
  int num = SVAL(inbuf,smb_err);
  int i,j;

  for (i=0;err_classes[i].class;i++)
    if (err_classes[i].code == class)
      {
	sprintf(ret,"%s - %d",err_classes[i].class,num);
	return ret;
      }
  
  sprintf(ret,"ERROR: Unknown error (%d,%d)",class,num);
  return(ret);
}

/*
 * Lifted from RPC 3.9 rpc/svc_run.c and modified.
 * Here is the copyright notice for that file:
 *
 * Sun RPC is a product of Sun Microsystems, Inc. and is provided for
 * unrestricted use provided that this legend is included on all tape
 * media and as a part of the software program in whole or part.  Users
 * may copy or modify Sun RPC without charge, but are not authorized
 * to license or distribute it to anyone else except as part of a product or
 * program developed by the user.
 * 
 * SUN RPC IS PROVIDED AS IS WITH NO WARRANTIES OF ANY KIND INCLUDING THE
 * WARRANTIES OF DESIGN, MERCHANTIBILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE, OR ARISING FROM A COURSE OF DEALING, USAGE OR TRADE PRACTICE.
 * 
 * Sun RPC is provided with no support and without any obligation on the
 * part of Sun Microsystems, Inc. to assist in its use, correction,
 * modification or enhancement.
 * 
 * SUN MICROSYSTEMS, INC. SHALL HAVE NO LIABILITY WITH RESPECT TO THE
 * INFRINGEMENT OF COPYRIGHTS, TRADE SECRETS OR ANY PATENTS BY SUN RPC
 * OR ANY PART THEREOF.
 * 
 * In no event will Sun Microsystems, Inc. be liable for any lost revenue
 * or profits or other special, indirect and consequential damages, even if
 * Sun has been advised of the possibility of such damages.
 * 
 * Sun Microsystems, Inc.
 * 2550 Garcia Avenue
 * Mountain View, California  94043
 */

static void
my_svc_run()
{
#ifdef FD_SETSIZE
  fd_set readfds;
#else
  int readfds;
#endif /* def FD_SETSIZE */
  extern int errno;
  time_t now;

  timeout.tv_sec = 1;
  timeout.tv_usec = 0;

  for (;;) {
#ifdef FD_SETSIZE
    readfds = svc_fdset;
#else
    readfds = svc_fds;
#endif /* def FD_SETSIZE */
    switch (select(FD_SETSIZE, (int *)&readfds, (int *)0, (int *)0,
		   &timeout)) {
    case -1:
      if (errno == EINTR)
	continue;
      syslog(LOG_ERR, "svc_run: select failed: %m");
      terminate(0);

    case 0:
      time(&now);
      nfs_timeout(now);
      break;

    default:
      time(&now);
      svc_getreqset(&readfds);
      nfs_timeout(now);
      break;
    }
  }
}
