/* NFS RPC service procedures for smbmount */

/*
Copyright Technical Research Centre of Finland (VTT), 
Information Technology Institute (TTE) 1993, 1994 - All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the names of VTT or TTE not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission. 

VTT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
VTT BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
*/

static char RCS_Id[] = "$Id: smbm_svc_subr.c,v 1.2 1995/01/03 18:12:39 tml Exp $";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <pwd.h>
#include <malloc.h>
#include <assert.h>
#include <time.h>

#include <syslog.h>

#include <rpc/rpc.h>
#ifndef AUTH_UNIX
#include <rpc/auth.h>
#endif
#include <sys/stat.h>

#include "nfs_prot.h"
#include "log_nfs.h"

#include "includes.h"

#include "util.h"
#include "smbmount.h"

smb_user guest;

int toplevel_fh_allocation[NFS_FHSIZE/sizeof(int)];

static smb_fh toplevel_fh;
static smb_fh control_fh;

static smb_user *users;

/* The do { } while (0) is so that the macros expand to a single statement.  */

#define ENTRY(name,argtype)					\
  do {								\
    if (debug & DEBUG_TRACENFS)					\
      printf("%s\n",STRINGIFY(name)),				\
      print_auth(rqstp),					\
      CONCAT2(print_,argtype)(0,argp);				\
  } while (0)

#define LEAVE(restype)						\
  do {								\
     if (debug & DEBUG_TRACENFS)				\
       {							\
	 CONCAT2(print_,restype)(1,&res);			\
	 putchar('\n');						\
       }							\
     return &res;						\
  } while (0)

#define LOG(args) (debug & DEBUG_TRACENFS) ? printf args : 0

#define print_nfs_fh print_smb_fh

static smb_user *who;

unsigned char local_to_smb[256];
unsigned char smb_to_local[256];
unsigned char smb_downcase[256];

int fntransl;

typedef struct smb_user_list_entry *smb_user_list;

struct smb_user_list_entry {
  smb_user *user;
  smb_user_list link;
};

#define MODE_READ 1
#define MODE_WRITE 2

/* 
 * Each file on the SMB server is kept open only once by smbmount.
 * The cache_entry structure corresponds to an open file.
 * The piggyback list is a list of smbmount users that have access
 * to the open file.
 */

typedef struct
{
  time_t timestamp;
  smb_fh fh;
  char mode;
  short fid;
#define FID_NOT_OPEN -1
  smb_user *user;
  smb_user_list piggyback;
} cache_entry;

#define CACHESIZE 3

static cache_entry cache[CACHESIZE];
static int cacheix[CACHESIZE];

static void print_smb_fh P((int level, nfs_fh *value));
static int maybe_open_file P((const smb_fh fh, const char mode, int *ixp));

static smb_fh
new_fh()
{
  smb_fh result;

  result = (smb_fh) xmalloc(sizeof(*result));
  result->fh_type = SMBMFH_UNKNOWN;
  result->fh_path[0] = '\0';
  result->fh_dir = NULL;
  result->fh_link = NULL;
}

#if __STDC__
void
init_nfs_subr(void)
#else
void
init_nfs_subr()
#endif
{
  int i;

  memset(toplevel_fh_allocation, 0, NFS_FHSIZE);
  toplevel_fh = new_fh();
  memmove(toplevel_fh_allocation, &toplevel_fh, sizeof(toplevel_fh));
  toplevel_fh->fh_type = SMBMFH_DIR;
  strcpy(toplevel_fh->fh_path, "\\");

  control_fh = new_fh();
  control_fh->fh_type = SMBMFH_CONTROL;
  strcpy(control_fh->fh_path, "\\.control");

  setvbuf(stdout, NULL, _IOLBF, BUFSIZ);

  fh_printer = print_smb_fh;
  users = NULL;
  for (i = 0; i < CACHESIZE; i++)
    {
      cache[i].user = NULL;
      cache[i].fh = NULL;
      cache[i].fid = FID_NOT_OPEN;
      cacheix[i] = i;
    }
}

#if __STDC__
static const char *
fh_type(const smb_fh fh)
#else
static char *
fh_type(fh)
     smb_fh fh;
#endif
{
  switch (fh->fh_type)
    {
    case SMBMFH_DIR:
      return "dir";
    case SMBMFH_FILE:
      return "file";
    case SMBMFH_CONTROL:
      return "control";
    default:
      return "???";
    }
}

#if __STDC__
static void
print_smb_fh(int level, nfs_fh *value)
#else
static void
print_smb_fh(level, value)
     int level;
     nfs_fh *value;
#endif
{
  CONST smb_fh fh = * (smb_fh *) value;
  printf("%s%s at %p", indent(level), fh_type(fh), fh);
  switch (fh->fh_type)
    {
    case SMBMFH_CONTROL:
      break;

    case SMBMFH_DIR:
      printf(" %s", fh->fh_path);
      if (fh->fh_dir)
	printf(" dl at %p: %d entries, tstamp: %d",
	       fh->fh_dir, fh->fh_dir->dl_nentries, fh->fh_dir->dl_timestamp);
      break;

    case SMBMFH_FILE:
      printf(" %s", fh->fh_path);
    }

  printf("\n");
}

/* local_smb_strcmp -- compare local string to SMB string */

#if __STDC__
static int
local_smb_strcmp(const char *a,
		 const char *b)
#else
static int
local_smb_strcmp(a, b)
     char *a;
     char *b;
#endif
{
  while (*a && *b && smb_downcase[local_to_smb[*a]] == smb_downcase[*b])
      a++, b++;

  if (!*a && !*b)
    return 0;
  if (!*a)
    return -1;
  if (!*b)
    return 1;
  
  return (smb_downcase[local_to_smb[*a]] > smb_downcase[*b]) ? 1 : -1;
}

/*
 * Duplicate a string, translating SMB characters.
 */

#if __STDC__
static char *
smb_to_local_dup(const unsigned char *src, size_t length)
#else
static char *
smb_to_local_dup(src, length)
     unsigned char *src;
     size_t length;
#endif
{
  fstring tem;
  char *p = tem;
  char *result;
  
  while (length--)
    *p++ = smb_to_local[*src++];

  *p++ = 0;
  
  return xstrdup(tem);
}

static void local_to_smb_cpy(char *a, const char *b)
{
  strcpy(a, b);
}

static int build_full_path(smb_fh fh, const char *entry, fstring path)
{
  char *p;

  strcpy(path, fh->fh_path);
  if (path[1])
    {
      p = path + strlen(path);
      *p++ = '\\';
    }
  else
    p = path + 1;

  if ((p + strlen(entry)) - path >= 256)
    return NFSERR_NAMETOOLONG;

  local_to_smb_cpy(p, entry);

  return NFS_OK;
}  

#if __STDC__
static smb_user *
session_of(int uid)
#else
static smb_user *
session_of(uid)
     int uid;
#endif
{
  smb_user *u;

  for (u = users; u && u->u_uxuid != uid; u = u->u_link)
    ;
  return u;
}  

#if __STDC__
static smb_user *
some_session(void)
#else
static smb_user *
some_session()
#endif
{
  smb_user *u;

  if (guest.u_uid >= 0)
    return  &guest;

  for (u = users; u && u->u_uid < 0; u = u->u_link)
    ;

  return u;
}  

#if __STDC__
static int
guest_user(void)
#else
static int
guest_user()
#endif
{
  if (guest.u_uid >= 0)
    {
      who = &guest;
      return NFS_OK;
    }
  return NFSERR_ACCES;
}

#if __STDC__
static int
auth_uid(const struct svc_req *rqstp)
#else
static int
auth_uid(rqstp)
     struct svc_req *rqstp;
#endif
{
  switch (rqstp->rq_cred.oa_flavor)
    {
    case AUTH_UNIX:
      return ((struct authunix_parms *) rqstp->rq_clntcred)->aup_uid;

    case AUTH_NULL:
    default:
      return 0;
    }
}  

#if __STDC__
static int
find_user(const struct svc_req *rqstp)
#else
static int
find_user(rqstp)
     struct svc_req *rqstp;
#endif
{
  int uid;
  smb_user *u;

  who = NULL;

  /* Root requests are treated as unauthenticated. */
  if ((uid = auth_uid(rqstp)) == 0)
    return guest_user();

  if (!(u = session_of(uid)))
    return guest_user();

  if (u->u_uid == UID_LOGIN_FAILED)
    /* Her session setup has failed recently.  */
    return NFSERR_ACCES;
      
  who = u;
      
  if (u->u_uid >= 0)
    {
      time(&who->u_timestamp);
      return NFS_OK;
    }

  if (login_user(u))
    {
      who = u;
      time(&who->u_timestamp);
      return NFS_OK;
    }
  else
    return NFSERR_ACCES;
}

#if __STDC__
static int
map_smb_error(int rcls, int err)
#else
static int
map_smb_error(rcls, err)
     int rcls;
     int err;
#endif
{
  switch (rcls)
    {
    case SUCCESS:
      return NFS_OK;

    case ERRDOS:
      switch (err)
      {
      case ERRbadfile:
      case ERRbadpath:
	return NFSERR_NOENT;
      case ERRbadshare:
	return NFSERR_ACCES;
      case ERRfilexists:
	return NFSERR_EXIST;
      }
      break;

    case ERRSRV:
      switch (err)
	{
	case ERRaccess:
	  return NFSERR_ACCES;
	}
      break;

    case ERRHRD:
      switch (err)
	{
	case ERRnowrite:
	  return NFSERR_ROFS;
	case ERRdiskfull:
	  return NFSERR_NOSPC;
	}
    }
  return NFSERR_IO;
}

#if __STDC__
static int
login_again(void)
#else
static int
login_again()
#endif
{
}  

static smb_fh
copy_fh(smb_fh fh)
{
  smb_fh result;

  result = new_fh();
  *result = *fh;
  result->fh_link = NULL;
  result->fh_dir = NULL;

  return result;
}

static smb_fh
intern_fh(smb_fh fh)
{
  smb_fh q = toplevel_fh, p = q->fh_link;
  
  while (p && strcmp(p->fh_path, fh->fh_path))
    q = p, p = p->fh_link;

  if (p)
    {
      free(fh);
      return p;
    }
  
  q->fh_link = fh;

  return fh;
}

#if __STDC__
static int
find_cached_file(const smb_fh fh)
#else
static int
find_cached_file(fh)
     smb_fh fh;
#endif
{
  int i, retval;
  cache_entry *ce;

  i = 0;
      
  while (i < CACHESIZE
	 && ((ce = cache + cacheix[i])->fh != fh
	     || ce->fid == FID_NOT_OPEN))
    i++;

  if (i == CACHESIZE)
    return -1;

  retval = cacheix[i];

  if (i > 0)
    {
      memmove(cacheix+1, cacheix, sizeof(cacheix[0]) * i);
      cacheix[0] = retval;
    }

  return retval;
}

#if __STDC__
static int
get_attr(const smb_fh fh,
	 fattr *attr)
#else
static int
get_attr(fh, attr)
     smb_fh fh;
     fattr *attr;
#endif
{
  int status;
  int ix;
  short attrs;
  int its_open = 0;

  /* Note the fh->fh_type might be SMBMFH_UNKNOWN here. */

  if (fh->fh_type == SMBMFH_FILE
      && Protocol >= PROTOCOL_LANMAN1
      && open_for_attr
      && ((ix = find_cached_file(fh)) >= 0
	   || (status = maybe_open_file(fh, MODE_READ, &ix)) == NFS_OK))
    {
      /* Use SMBgetattrE, which needs an open file. */

      its_open++;
      time(&cache[ix].timestamp);

      memset(OutBuffer, 0, smb_size);
      set_message(OutBuffer, 1, 0, False);
      setup_pkt(OutBuffer, SMBgetattrE, cache[ix].user);
      SSVAL(OutBuffer, smb_vwv0, cache[ix].fid);
    }
  else
    {
      char *p;

      memset(OutBuffer, 0, smb_size);
      set_message(OutBuffer, 0, 2 + strlen(fh->fh_path), True);
      setup_pkt(OutBuffer, SMBgetatr, who);

      p = smb_buf(OutBuffer);
      *p++ = 4;
      strcpy(p, fh->fh_path);
    }
  
  send_smb(OutBuffer);
  receive_smb(InBuffer, 0);

  if (CVAL(InBuffer, smb_rcls) != SUCCESS)
    return map_smb_error(CVAL(InBuffer, smb_rcls), CVAL(InBuffer, smb_err));

  if (its_open)
    {
      attr->type = NFREG;
      attr->mode = NFSMODE_REG;
      attrs = SVAL(InBuffer, smb_vwv10);
      attr->size = IVAL(InBuffer, smb_vwv6);
      attr->atime.seconds = make_unix_date(InBuffer + smb_vwv2);
      attr->ctime.seconds = attr->mtime.seconds =
	make_unix_date(InBuffer + smb_vwv4);
    }
  else
    {
      attrs = SVAL(InBuffer, smb_vwv0);
      if (!(attrs & aDIR))
	{
	  fh->fh_type = SMBMFH_FILE;
	  attr->type = NFREG;
	  attr->mode = NFSMODE_REG;
	  attr->size = IVAL(InBuffer, smb_vwv3);
	}
      else
	{
	  fh->fh_type = SMBMFH_DIR;
	  attr->type = NFDIR;
	  attr->mode = NFSMODE_DIR;
	  attr->size = 1024;
	}
      attr->atime.seconds = attr->ctime.seconds =
	attr->mtime.seconds = IVAL(InBuffer, smb_vwv1);
    }
  
  /* Read-only files are shown as a-w, other 777 unless a system file */
  if (attrs & aRONLY)
    attr->mode |= S_IRUSR | S_IXUSR | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;
  else
    attr->mode |= S_IRWXU | S_IRWXG | S_IRWXO;
  /* System files are shows as owned by root, mode -w,o-a */
  if (attrs & aSYSTEM)
    attr->mode &= ~(S_IWUSR | S_IWGRP | S_IRWXO);
  else
    attr->uid = UID_NOBODY;
  attr->gid = UID_NOBODY;
  attr->nlink = 1;
  attr->blocksize = 1024;
  attr->blocks =
    attr->size ? 
    (attr->size - 1) / attr->blocksize + 1
    : 0;
  attr->fsid = 1;
  attr->fileid = (u_int) fh;

  return NFS_OK;
}

#if __STDC__
static int
set_control_attr(fattr *attr)
#else
static int
set_control_attr(attr)
     fattr *attr;
#endif
{
  attr->type = NFREG;
  attr->mode = NFSMODE_REG | S_IWUSR | S_IWGRP | S_IWOTH;
  attr->nlink = 1;
  attr->uid = attr->gid = 0;
  attr->size = 0;
  attr->blocksize = 1024;
  attr->blocks = 0;
  attr->fsid = 1;
  attr->fileid = (u_int) control_fh;
}

#if __STDC__
static int
remove_file(const char *path)
#else
static int
remove_file(path)
     char *path;
#endif
{
  char *p;

  memset(OutBuffer, 0, smb_size);
  set_message(OutBuffer, 1, 2 + strlen(path), True);
  setup_pkt(OutBuffer, SMBunlink, who);

  p = smb_buf(OutBuffer);
  *p++ = 4;
  strcpy(p, path);

  send_smb(OutBuffer);
  receive_smb(InBuffer, 0);

  if (CVAL(InBuffer, smb_rcls) != SUCCESS)
    return map_smb_error(CVAL(InBuffer, smb_rcls), CVAL(InBuffer, smb_err));

  return NFS_OK;
}

#if __STDC__
static int
remove_dir(const char *path)
#else
static int
remove_dir(path)
     char *path;
#endif
{
  char *p;

  memset(OutBuffer, 0, smb_size);
  set_message(OutBuffer, 0, 2 + strlen(path), True);
  setup_pkt(OutBuffer, SMBrmdir, who);

  p = smb_buf(OutBuffer);
  *p++ = 4;
  strcpy(p, path);

  send_smb(OutBuffer);
  receive_smb(InBuffer, 0);

  if (CVAL(InBuffer, smb_rcls) != SUCCESS)
    return map_smb_error(CVAL(InBuffer, smb_rcls), CVAL(InBuffer, smb_err));

  return NFS_OK;
}

#if __STDC__
static int
rename_file(const char *from_path, const char *to_path)
#else
static int
rename_file(from_path, to_path)
     char *from_path;
     char *to_path;
#endif
{
  char *p;

  memset(OutBuffer, 0, smb_size);
  set_message(OutBuffer, 1, 4 + strlen(from_path) + strlen(to_path), True);
  setup_pkt(OutBuffer, SMBmv, who);

  p = smb_buf(OutBuffer);
  *p++ = 4;
  strcpy(p, from_path);

  p += strlen(from_path);
  *p++ = 0;
  *p++ = 4;
  strcpy(p, to_path);

  send_smb(OutBuffer);
  receive_smb(InBuffer, 0);

  if (CVAL(InBuffer, smb_rcls) != SUCCESS)
    return map_smb_error(CVAL(InBuffer, smb_rcls), CVAL(InBuffer, smb_err));

  return NFS_OK;
}

#if __STDC__
static int
set_file_attr(const smb_fh fh, const char *path, const sattr *attr)
#else
static int
set_file_attr(fh, path, attr)
     smb_fh fh;
     char *path;
     sattr *attr;
#endif
{
  int tries;
  int comp;
  int cr;
  int ix;
  
  if (attr->mtime.seconds != (u_int) -1)
    {
    }

  if (fh->fh_type == SMBMFH_FILE && attr->size != -1)
    {
    }
  
  return NFS_OK;
}

#if __STDC__
static int
set_dir_attr(const smb_fh fh, const char *path, const sattr *attr)
#else
static int
set_dir_attr(fh, path, attr)
     smb_fh fh;
     char *path;
     sattr *attr;
#endif
{
  int tries;
  int comp;
  int cr;

  if (attr->mode != -1)
    {
    }
  
  if (attr->mtime.seconds != (u_int) -1)
    {
    }

  if (attr->uid != (u_int) -1)
    {
    }

  if (attr->gid != (u_int) -1)
    {
    }

  if (1)
    {
    }

  return NFS_OK;
}

#if __STDC__
static int
set_attr(const smb_fh fh, const sattr *attr)
#else
static int
set_attr(fh, attr)
     smb_fh fh;
     sattr *attr;
#endif
{
  int status;
  int ix;

  if (attr->mode != (u_int) -1 || attr->uid != (u_int) -1
      || (attr->mtime.seconds != (u_int) -1
	  && attr->atime.seconds == (u_int) -1))
    {
      /* Attributes can be changed only with SMBsetatr. */
      char *p;
      short attrs = 0;

      memset(OutBuffer, 0, smb_size);
      set_message(OutBuffer, 8, 2 + strlen(fh->fh_path), True);
      setup_pkt(OutBuffer, SMBsetatr, who);

      if (attr->mode != (u_int) -1 && !(attr->mode & S_IWOTH))
	attrs |= aRONLY;
      if (attr->uid != (u_int) -1 && attr->uid == 0)
	attrs |= aSYSTEM;

      if (attr->mtime.seconds != (u_int) -1
	  && attr->atime.seconds == (u_int) -1)
	SIVAL(OutBuffer, smb_vwv1, attr->mtime.seconds);

      p = smb_buf(OutBuffer);
      *p++ = 4;
      strcpy(p, fh->fh_path);

      send_smb(OutBuffer);
      receive_smb(InBuffer, 0);

      if (CVAL(InBuffer, smb_rcls) != SUCCESS)
	return map_smb_error(CVAL(InBuffer, smb_rcls), CVAL(InBuffer, smb_err));
    }

  if (Protocol >= PROTOCOL_LANMAN1
      && fh->fh_type == SMBMFH_FILE
      && attr->atime.seconds != (u_int) -1)
    {
      /* Use SMBsetsttrE, which needs an open file. */
      if ((status = maybe_open_file(fh, MODE_READ, &ix)) != NFS_OK)
	return status;
      
      time(&cache[ix].timestamp);

      memset(OutBuffer, 0, smb_size);
      set_message(OutBuffer, 0, 7, True);
      setup_pkt(OutBuffer, SMBsetattrE, cache[ix].user);
      SSVAL(OutBuffer, smb_vwv0, cache[ix].fid);
      
      if (attr->atime.seconds != (u_int) -1)
	put_dos_date2(OutBuffer, smb_vwv3, attr->atime.seconds);

      send_smb(OutBuffer);
      receive_smb(InBuffer, 0);

      if (CVAL(InBuffer, smb_rcls) != SUCCESS)
	return map_smb_error(CVAL(InBuffer, smb_rcls), CVAL(InBuffer, smb_err));
    }

  return NFS_OK;
}

#if __STDC__
static int
open_existing_file(const smb_fh fh, const char mode, short *fidp)
#else
static int
open_existing_file(fh, mode, fidp)
     smb_fh fh;
     char mode;
     short *fidp;
#endif
{
  char *p;

  memset(OutBuffer, 0, smb_size);
  set_message(OutBuffer, 2, 2 + strlen(fh->fh_path), True);
  setup_pkt(OutBuffer, SMBopen, who);

  if (mode == MODE_READ)
    SSVAL(OutBuffer, smb_vwv0, 4<<4);
  else if (mode == MODE_WRITE)
    SSVAL(OutBuffer, smb_vwv0, (4<<4) | 1);
  else
    SSVAL(OutBuffer, smb_vwv0, (4<<4) | 2);
  SSVAL(OutBuffer, smb_vwv1, aSYSTEM | aHIDDEN);
  
  p = smb_buf(OutBuffer);
  *p++ = 4;      
  strcpy(p, fh->fh_path);

  send_smb(OutBuffer);
  receive_smb(InBuffer, 0);

  if (CVAL(InBuffer, smb_rcls) != SUCCESS)
    return map_smb_error(CVAL(InBuffer, smb_rcls), CVAL(InBuffer, smb_err));

  *fidp = SVAL(InBuffer, smb_vwv0);
  LOG(("-opened fid %d\n", *fidp));

  return NFS_OK;
}

#if __STDC__
static int
read_from_file(short fid, u_int offset, u_int count,
	       char **val, u_int *lenp)
#else
static int
read_from_file(fid, offset, count, val, lenp)
     short fid;
     u_int offset;
     u_int count;
     char **val;
     u_int *lenp;
#endif
{
  u_int nread = 0;
  u_int datalen;
  int reset_readbraw_supported = 0;

  *val = xmalloc(count);

  while (nread < count)
    {
      if (readbraw_supported && count - nread >= max_xmit - 200)
	{
	  static int readbraw_size = 0xFFFF;
	  memset(OutBuffer, 0, smb_size);
	  set_message(OutBuffer, 8, 0, True);
	  setup_pkt(OutBuffer, SMBreadbraw, who);
	  SSVAL(OutBuffer, smb_vwv0, fid);
	  SIVAL(OutBuffer, smb_vwv1, offset + nread);
	  SSVAL(OutBuffer, smb_vwv3, MIN(count - nread, readbraw_size));
	  SIVAL(OutBuffer, smb_vwv5, 1000); /* ??? */

	  send_smb(OutBuffer);
	  if (read_smb_length(Client, InBuffer, 0) == -1)
	    return NFSERR_IO;	/* XXX Should reopen LMX session? */
	  datalen = smb_len(InBuffer);

	  if (datalen == 0)
	    {
	      /* we either got an error or were at or beyond EOF  */
	      readbraw_supported = 0;
	      reset_readbraw_supported = 1;
	      continue;
	    }

	  if (!read_data(Client, *val + nread, datalen))
	    return NFSERR_IO;
	}
      else
	{
	  memset(OutBuffer, 0, smb_size);
	  set_message(OutBuffer, 5, 0, True);
	  setup_pkt(OutBuffer, SMBread, who);
	  SSVAL(OutBuffer, smb_vwv0, fid);
	  SSVAL(OutBuffer, smb_vwv1, MIN(max_xmit-200, count - nread));
	  SIVAL(OutBuffer, smb_vwv2, offset + nread);
	  SSVAL(OutBuffer, smb_vwv4, MIN(count - nread, 65535));

	  send_smb(InBuffer);
	  receive_smb(InBuffer, 0);
	  
	  if (CVAL(InBuffer, smb_rcls) != SUCCESS)
	    {
	      if (reset_readbraw_supported)
		readbraw_supported = 1;
	      DEBUG(3, ("SMBread: %s\n", smb_errstr(InBuffer)));
#if 0
	      return map_smb_error(CVAL(InBuffer, smb_rcls), CVAL(InBuffer, smb_err));
#else
	      /* This seems to work better...? XXX */
	      datalen = 0;
	      break;
#endif	
	    }
	  datalen = SVAL(InBuffer, smb_vwv0);
	  memmove(*val + nread, smb_buf(InBuffer) + 3, datalen);
	}
      nread += datalen;
      if (datalen == 0)
	break;
    }

  *lenp = nread;

  if (reset_readbraw_supported)
    readbraw_supported = 1;
  return NFS_OK;
}  

#if __STDC__
static int
write_to_file(short fid, u_int offset, u_int count,
	      char *buf, u_int *lenp)
#else
static int
write_to_file(fid, offset, count, buf, lenp)
     short fid;
     u_int offset;
     u_int count;
     char *buf;
     u_int *lenp;
#endif
{
  u_int nwritten = 0;
  int reset_writebraw_supported = 0;

  while (nwritten < count)
    {
#if 0
      if (writebraw_supported && count - nwritten >= max_xmit - 200)
	{
	  memset(OutBuffer, 0, smb_size);
	  set_message(OutBuffer, 12, 0, True);
	  setup_pkt(OutBuffer, SMBwritebraw, who);
	  SSVAL(OutBuffer, smb_vwv0, fid);
	  SSVAL(OutBuffer, smb_vwv1, MIN(count - nwritten, 65535));
	  SIVAL(OutBuffer, smb_vwv3, offset + nwritten);
	  SIVAL(OutBuffer, smb_vwv5, 1000); /* ??? timeout */

	  send_smb(OutBuffer);
	  receive_smb(InBuffer, 0);
	  if (CVAL(InBuffer, smb_rcls) != SUCCESS)
	    {
	      syslog(LOG_ERR, "writebraw failed: %s", smb_errstr(InBuffer));
	      writebraw_supported = 0;
	      reset_writebraw_supported = 1;
	      continue;
	    }
	  
	  while (nwritten < count)
	    {
	      int ret = write_socket(Client, buf + nwritten, count - nwritten);
	      if (ret == -1)
		{
		  syslog(LOG_ERR, "Asked to writebraw %d bytes, write_socket: %m",
			 count - nwritten);
		  return NFSERR_IO; /* ??? */
		}
	      nwritten += ret;
	      *lenp = nwritten;
	    }
	}
      else
#endif
	{
	  int n = MIN(max_xmit - 200, count - nwritten);

	  memset(OutBuffer, 0, smb_size);
	  set_message(OutBuffer, 5, n + 3, True);
	  setup_pkt(OutBuffer, SMBwrite, who);
	  SSVAL(OutBuffer, smb_vwv0, fid);
	  SSVAL(OutBuffer, smb_vwv1, n);
	  SIVAL(OutBuffer, smb_vwv2, offset + nwritten);
	  SSVAL(OutBuffer, smb_vwv4, count - nwritten);
	  CVAL(smb_buf(OutBuffer), 0) = 1;
	  SSVAL(smb_buf(OutBuffer), 1, n);
	  memmove(smb_buf(OutBuffer) + 3, buf + nwritten, n);

	  send_smb(OutBuffer);
	  receive_smb(InBuffer, 0);

	  if (CVAL(InBuffer, smb_rcls) != SUCCESS)
	    return map_smb_error(CVAL(InBuffer, smb_rcls), CVAL(InBuffer, smb_err));
	  nwritten += SVAL(InBuffer, smb_vwv0);
	  *lenp = nwritten;
	  if (n != SVAL(InBuffer, smb_vwv0))
	    {
	      if (reset_writebraw_supported)
		writebraw_supported = 1;
	      return NFSERR_IO;
	    }
	}
    }
	      
  if (reset_writebraw_supported)
    writebraw_supported = 1;
  return NFS_OK;
}  

#if __STDC__
static void
close_fid(const short fid, const smb_user *user)
#else
static void
close_fid(fid, user)
     short fid;
     smb_user *user;
#endif
{
  memset(OutBuffer, 0, smb_size);
  set_message(OutBuffer, 3, 0, True);
  setup_pkt(OutBuffer, SMBclose, user);
  SSVAL(OutBuffer, smb_vwv0, fid);

  send_smb(OutBuffer);
  receive_smb(InBuffer, 0);
  
  LOG(("-closed fid %d\n", fid));

  if (CVAL(InBuffer, smb_rcls) != SUCCESS)
    syslog(LOG_ERR, "SMBclose: %s", smb_errstr(InBuffer));
}

#if __STDC__
static void
close_cache_entry(cache_entry *ce)
#else
static void
close_cache_entry(ce)
     cache_entry *ce;
#endif     
{
  smb_user_list ul, link;

  LOG(("-closing cache entry %d\n", ce - cache));

  close_fid(ce->fid, ce->user);
  ce->fid = FID_NOT_OPEN;
  ce->fh = NULL;
  ce->user = NULL;

  ul = ce->piggyback;
  while (ul)
    {
      link = ul->link;
      free(ul);
      ul = link;
    }
}  

#if __STDC__
void
nfs_timeout(time_t now)
#else
void
nfs_timeout(now)
     time_t now;
#endif
{
  int i;
  smb_user *u;

  u = users;
  while (u)
    u->u_mark = 0, u = u->u_link;
  
  /* Close idle files: no activity for TIMEOUT seconds.  */
  /* Mark sessions with non-idle files.  */
  for (i = 0; i < CACHESIZE; i++)
    if (cache[i].fid != FID_NOT_OPEN)
      {
	if (cache[i].timestamp < now - TIMEOUT)
	  close_cache_entry(cache + i);
	else
	  {
	    smb_user_list ul;

	    cache[i].user->u_mark = 1;
	    ul = cache[i].piggyback;
	    while (ul)
	      {
		ul->user->u_mark = 1;
		ul = ul->link;
	      }
	  }
      }

  /*
   * Close idle sessions: sessions with no file open, and
   * no activity for 10*TIMEOUT seconds.
   */
  u = users;
  for (u = users; u; u = u->u_link)
     if (!u->u_mark
	 && u->u_timestamp < now - 10*TIMEOUT
	 && u->u_uid > 0)
       logout(u);
}

#if __STDC__
static int
enter_cache_entry(const smb_fh fh, const char mode, const short fid)
#else
static int
enter_cache_entry(fh, mode, fid)
     smb_fh fh;
     char mode;
     short fid;
#endif
{
  int ix;
  cache_entry *ce;

  if ((ce = cache + (ix = cacheix[CACHESIZE-1]))->fid != FID_NOT_OPEN)
    close_cache_entry(ce);
  memmove(cacheix + 1, cacheix, sizeof(cacheix[0]) * (CACHESIZE-1));
  cacheix[0] = ix;
  ce->fh = fh;
  ce->user = who;
  ce->mode = mode;
  ce->fid = fid;
  ce->piggyback = NULL;

  return ix;
}

#if __STDC__
static int
find_user_of_file(smb_user_list *listp)
#else
static int
find_user_of_file(listp)
     smb_user_list *listp;
#endif
{
  smb_user_list p = *listp;

  while (p && p->user != who)
    p = p->link;

  return p != NULL;
}

#if __STDC__
static void
piggyback_user(smb_user_list *listp)
#else
static void
piggyback_user(listp)
     smb_user_list *listp;
#endif
{
  smb_user_list link = *listp;

  *listp = xmalloc(sizeof *listp);
  (*listp)->user = who;
  (*listp)->link = link;
}

#if __STDC__
static int
maybe_open_file(const smb_fh fh, const char mode, int *ixp)
#else
static int
maybe_open_file(fh, mode, ixp)
     smb_fh fh;
     char mode;
     int *ixp;
#endif
{
  int ix, result;
  short fid;

  /* Do we have this file cached?  */
  if ((ix = find_cached_file(fh)) >= 0)
    {
      /*
       * Yes; is this the original opener, or have we already
       * piggybacked this user, and is the mode suitable?
       */

      if ((cache[ix].user == who ||
	   find_user_of_file(&cache[ix].piggyback))
	  && (cache[ix].mode & mode) == mode)
	{
	  LOG(("-in cache: %d\n", ix));
	  *ixp = ix;
	  return NFS_OK;
	}

      /* Else, try to open the file with the desired mode.  */

      LOG(("-in cache: %d, but must reopen\n", ix));

      /* Try to open it */
      if ((result = open_existing_file(fh, mode, &fid)) != NFS_OK)
	return result;
	  
      /*
       * If it's the current user only, close it.
       * Also if the mode isn't suitable, close it. 
       * Use the just opened handle.
       * (Any file is kept open only once in smbmount.)
       */
      if ((cache[ix].user == who && cache[ix].piggyback == NULL)
	  || (cache[ix].mode & mode) != mode)
	{
	  close_cache_entry(cache + ix);
	  
	  ix = enter_cache_entry(fh, mode, fid);
	  LOG(("-into cache: %d\n", ix));
	}
      else
	{
	  /*
	   * This user can open it; so close the unnecessary file handle
	   * just opened and use the cached handle, piggybacking this user.
	   */
	  close_fid(fid, who);
	  piggyback_user(&cache[ix].piggyback);
	}
    }
  else
    {
      if ((result = open_existing_file(fh, mode, &fid)) != NFS_OK)
	return result;
      ix = enter_cache_entry(fh, mode, fid);
      LOG(("-into cache: %d\n", ix));
    }
  *ixp = ix;
  return NFS_OK;
}

#if __STDC__
static int
create_file(smb_fh fh)
#else
static int
create_file(fh)
     smb_fh fh;
#endif
{
  char *p;
  int ix;

  memset(OutBuffer, 0, smb_size);
  set_message(OutBuffer, 3, 2 + strlen(fh->fh_path), True);
  setup_pkt(OutBuffer, SMBmknew, who);

  p = smb_buf(OutBuffer);
  *p++ = 4;
  strcpy(p, fh->fh_path);

  send_smb(OutBuffer);
  receive_smb(InBuffer, 0);

  if (CVAL(InBuffer, smb_rcls) != SUCCESS)
    return map_smb_error(CVAL(InBuffer, smb_rcls), CVAL(InBuffer, smb_err));

  fh->fh_type = SMBMFH_FILE;
  ix = enter_cache_entry(fh, MODE_READ | MODE_WRITE, SVAL(InBuffer, smb_vwv0));
  LOG(("-created fid %d, into cache entry %d\n", cache[ix].fid, ix));

  time(&cache[ix].timestamp);

  return NFS_OK;
}
      
#if __STDC__
static int
create_dir(char *path)
#else
static int
create_dir(path)
     char *path;
#endif
{
  char *p;

  memset(OutBuffer, 0, smb_size);
  set_message(OutBuffer, 0, 2 + strlen(path), True);
  setup_pkt(OutBuffer, SMBmkdir, who);

  p = smb_buf(OutBuffer);
  *p++ = 4;
  strcpy(p, path);

  send_smb(OutBuffer);
  receive_smb(InBuffer, 0);

  if (CVAL(InBuffer, smb_rcls) != SUCCESS)
    return map_smb_error(CVAL(InBuffer, smb_rcls), CVAL(InBuffer, smb_err));

  return NFS_OK;
}
      
#define MAX_E_SIZE sizeof(entry) + 256 + 16

#if __STDC__
int
dpsize(entry **e)
#else
int
dpsize(e)
     entry **e;
#endif
{
  return sizeof(entry) + strlen((*e)->name) + 16;
}

#if __STDC__
void *
ctlproc_null_1(void *argp,
	       struct svc_req *rqstp)
#else
void *
ctlproc_null_1(argp, rqstp)
     void *argp;
     struct svc_req *rqstp;
#endif
{
  static char res;
  
  return &res;
}

#if __STDC__
void *
ctlproc_login_1(void *argp,
	       struct svc_req *rqstp)
#else
void *
ctlproc_login_1(argp, rqstp)
     void *argp;
     struct svc_req *rqstp;
#endif
{
  static char res;
  
  return &res;
}

#if __STDC__
void *
nfsproc_null_2(void *argp,
	       struct svc_req *rqstp)
#else
void *
nfsproc_null_2(argp, rqstp)
     void *argp;
     struct svc_req *rqstp;
#endif
{
  static char res;
  
  ENTRY(null,void);

  res = 0;

  LEAVE(void);
}

#if __STDC__
attrstat *
nfsproc_getattr_2(nfs_fh *argp,
		  struct svc_req *rqstp)
#else
attrstat *
nfsproc_getattr_2(argp, rqstp)
     nfs_fh *argp;
     struct svc_req *rqstp;
#endif
{
  static attrstat res;
  fattr *result_attr = &res.attrstat_u.attributes;
  CONST smb_fh fh = * (smb_fh *) argp;

  ENTRY(getattr,nfs_fh);

  memset(&res, 0, sizeof(res));

  if (fh->fh_type == SMBMFH_CONTROL)
    {
      set_control_attr(result_attr);
      LEAVE(attrstat);
    }

  /* The kernel does a getattr on the root when mounting.
   * A getattr on the root is also done prior to writing
   * to the control file. Thus we don't need to be loggged
   * in if getattring the root.
   */
  if (fh == toplevel_fh && find_user(rqstp) != NFS_OK)
    {
      result_attr->type = NFDIR;
      result_attr->mode = NFSMODE_DIR | S_IRWXU | S_IRWXG | S_IRWXO;
      result_attr->uid = result_attr->gid = 0;
      result_attr->nlink = 1;
      result_attr->size = 1024;
      result_attr->blocksize = 1024;
      result_attr->blocks = 1;
      result_attr->fsid = 1;
      result_attr->fileid = (u_int) fh;

      LEAVE(attrstat);
    }

  if (fh != toplevel_fh && (res.status = find_user(rqstp)) != NFS_OK)
    LEAVE(attrstat);

  res.status = get_attr(fh, result_attr);

  LEAVE(attrstat);
}

#if __STDC__
attrstat *
nfsproc_setattr_2(sattrargs *argp,
		  struct svc_req *rqstp)
#else
attrstat *
nfsproc_setattr_2(argp, rqstp)
	sattrargs *argp;
	struct svc_req *rqstp;
#endif
{
  static attrstat res;
  fattr *result_attr = &res.attrstat_u.attributes;
  CONST smb_fh fh = * (smb_fh *) &argp->file;
  sattr *attr = &argp->attributes;

  ENTRY(setattr, sattrargs);

  memset(&res, 0, sizeof(res));

  if ((res.status = find_user(rqstp)) != NFS_OK)
    LEAVE(attrstat);

  if (fh->fh_type == SMBMFH_CONTROL)
    {
      res.status = NFSERR_ACCES;
      LEAVE(attrstat);
    }

  if ((res.status = set_attr(fh, attr)) != NFS_OK)
    LEAVE(attrstat);

  res.status = get_attr(fh, result_attr);

  LEAVE(attrstat);
}

#if __STDC__
void *
nfsproc_root_2(void *argp,
	       struct svc_req *rqstp)
#else
void *
nfsproc_root_2(argp, rqstp)
     void *argp;
     struct svc_req *rqstp;
#endif
{
  static char res;

  ENTRY(root,void);

  res = 0;

  LEAVE(void);
}

#if __STDC__
diropres *
nfsproc_lookup_2(diropargs *argp,
		 struct svc_req *rqstp)
#else
diropres *
nfsproc_lookup_2(argp, rqstp)
     diropargs *argp;
     struct svc_req *rqstp;
#endif
{
  static diropres res;
  diropokres *result = &res.diropres_u.diropres;
  fattr *result_attr = &result->attributes;
  smb_fh result_fh;
  smb_fh fh = * (smb_fh *) &argp->dir;

  ENTRY(lookup, diropargs);

  memset(&res, 0, sizeof(res));
  
  /* The control file? */
  if (!fh->fh_path[1]
      && !strcmp(argp->name, ".control"))
    {
      set_control_attr(result_attr);
      *((smb_fh *) &result->file) = control_fh;
      LEAVE(diropres);
    }

  if ((res.status = find_user(rqstp)) != NFS_OK)
    LEAVE(diropres);

  if (fh->fh_type != SMBMFH_DIR)
    {
      res.status = NFSERR_NOTDIR;
      LEAVE(diropres);
    }
  
  /* Look up a directory entry */

  /* .. in the top dir or . in any dir returns the same dir */
  if ((argp->name[0] == '.' && argp->name[1] == '.' && !argp->name[2]
       && !fh->fh_path[1])
      || (argp->name[0] == '.' && !argp->name[1]))
    {
      *((smb_fh *) &result->file) = fh;
      res.status = get_attr(fh, result_attr);
      LEAVE(diropres);
    }

  result_fh = copy_fh(fh);
  result_fh->fh_type = SMBMFH_UNKNOWN;

  if (argp->name[0] == '.' && argp->name[1] == '.' && !argp->name[2])
    *(strrchr(result_fh->fh_path, '\\')) = 0;
  else if ((res.status = build_full_path(fh, argp->name,
					 result_fh->fh_path)) != NFS_OK)
    {
      res.status = NFSERR_NAMETOOLONG;
      free(result_fh);
      LEAVE(diropres);
    }

  result_fh = intern_fh(result_fh);

  if ((res.status = get_attr(result_fh, result_attr)) != NFS_OK)
    LEAVE(diropres);

  *((smb_fh *) &result->file) = result_fh;

  LEAVE(diropres);
}

#if __STDC__
readlinkres *
nfsproc_readlink_2(nfs_fh *argp,
		   struct svc_req *rqstp)
#else
readlinkres *
nfsproc_readlink_2(argp, rqstp)
     nfs_fh *argp;
     struct svc_req *rqstp;
#endif
{
  static readlinkres res;

  ENTRY(readlink,nfs_fh);

  memset(&res, 0, sizeof(res));

  if ((res.status = find_user(rqstp)) != NFS_OK)
    LEAVE(readlinkres);

  res.status = NFSERR_IO;

  LEAVE(readlinkres);
}

#if __STDC__
readres *
nfsproc_read_2(readargs *argp,
	       struct svc_req *rqstp)
#else
readres *
nfsproc_read_2(argp, rqstp)
     readargs *argp;
     struct svc_req *rqstp;
#endif
{
  static readres res;
  fattr *result_attr = &res.readres_u.reply.attributes;
  char **result_val = &res.readres_u.reply.data.data_val;
  u_int *result_len = &res.readres_u.reply.data.data_len;
  CONST smb_fh fh = * (smb_fh *) &argp->file;
  u_int offset = argp->offset, count = argp->count;
  int ix;

  ENTRY(read,readargs);
  
  /* Free previous result.  */
  xdr_free(xdr_readres, &res);

  memset(&res, 0, sizeof(res));

  /* Reading from the control file always returns EOF. */
  if (fh->fh_type == SMBMFH_CONTROL)
    {
      *result_val = xmalloc(1);
      *result_len = 0;
      set_control_attr(result_attr);
      LEAVE(readres);
    }

  if ((res.status = find_user(rqstp)) != NFS_OK)
    LEAVE(readres);
  
  if (fh->fh_type == SMBMFH_FILE)
    {
      if ((res.status = maybe_open_file(fh, MODE_READ, &ix)) != NFS_OK)
	LEAVE(readres);

      time(&cache[ix].timestamp);

      /* Now read the stuff she wants.  */
      if ((res.status = read_from_file(cache[ix].fid, offset, count,
				       result_val, result_len)) != NFS_OK)
	LEAVE(readres);
      
      res.status = get_attr(fh, result_attr);
    }
  else
    res.status = NFSERR_IO;
  
  LEAVE(readres);
}

#if __STDC__
void *
nfsproc_writecache_2(void *argp,
		     struct svc_req *rqstp)
#else
void *
nfsproc_writecache_2(argp, rqstp)
     void *argp;
     struct svc_req *rqstp;
#endif
{
  static char res;
  
  ENTRY(writecache,void);
  
  memset(&res, 0, sizeof(res));

  LEAVE(void);
}

#if __STDC__
attrstat *
nfsproc_write_2(writeargs *argp,
		struct svc_req *rqstp)
#else
attrstat *
nfsproc_write_2(argp, rqstp)
     writeargs *argp;
     struct svc_req *rqstp;
#endif
{
  static attrstat res;
  fattr *result_attr = &res.attrstat_u.attributes;
  CONST smb_fh fh = * (smb_fh *) &argp->file;
  u_int offset = argp->offset, count = argp->data.data_len;
  int ix;

  ENTRY(write,writeargs);
  
  memset(&res, 0, sizeof(res));

  /*
   * We must be able to write the control file without the user being
   * logged in, because the purpose of writing there is to provide
   * login authentication.
   */

  if (fh->fh_type == SMBMFH_CONTROL)
    {
      fstring buf, name, passwd;
      smb_user *u;
      int uid;

      if (offset || count > sizeof(fstring))
	{
	  res.status = NFSERR_IO;
	  LEAVE(attrstat);
	}

      memcpy(buf, argp->data.data_val, count);
      buf[count] = 0;

      if (sscanf(buf, "login %s %s\n", name, passwd) != 2)
	{
	  res.status = NFSERR_IO;
	  LEAVE(attrstat);
	}
      u = session_of((uid = auth_uid(rqstp)));

      if (!u)
	{
	  u = xmalloc(sizeof(*u));
	  u->u_uxuid = uid;
	  u->u_link = users;
	  users = u;
	}
      else
	{
	  if (u->u_uid >= 0)
	    logout(u);
	  free(u->u_name);
	  free(u->u_passwd);
	}
      u->u_uid = UID_NOT_OPEN;
      u->u_name = xstrdup(name);
      u->u_passwd = xstrdup(passwd);
      time(&u->u_timestamp);
      set_control_attr(result_attr);
      LEAVE(attrstat);
    }

  if ((res.status = find_user(rqstp)) != NFS_OK)
    LEAVE(attrstat);

  if (fh->fh_type == SMBMFH_FILE)
    {
      if ((res.status = maybe_open_file(fh, MODE_WRITE, &ix)) != NFS_OK)
	LEAVE(attrstat);

      time(&cache[ix].timestamp);

      if (count > 0)
	{
	  /* Now write the stuff.  */
	  if ((res.status = write_to_file(cache[ix].fid, offset, count,
					  argp->data.data_val, &count)) != NFS_OK)
	    LEAVE(attrstat);
	  if (count != argp->data.data_len)
	    {
	      syslog(LOG_ERR, "Asked to write %d bytes to %s, wrote %d\n",
		      argp->data.data_len, fh->fh_path, count);
	      res.status = NFSERR_IO;
	      LEAVE(attrstat);
	    }
	}
      
      res.status = get_attr(fh, result_attr);
    }
  else
    res.status = NFSERR_ACCES;
  
  LEAVE(attrstat);
}

#if __STDC__
diropres *
nfsproc_create_2(createargs *argp,
		 struct svc_req *rqstp)
#else
diropres *
nfsproc_create_2(argp, rqstp)
     createargs *argp;
     struct svc_req *rqstp;
#endif
{
  static diropres res;
  CONST smb_fh fh = * (smb_fh *) &argp->where.dir;
  smb_fh result_fh;
  fattr *result_attr = &res.diropres_u.diropres.attributes;
  fstring path;
	
  ENTRY(create,createargs);
  
  memset(&res, 0, sizeof(res));

  /* The control file can be "created" without loggin in. */
  if (!fh->fh_path[1]
      && !strcmp(argp->where.name, ".control"))
    {
      set_control_attr(result_attr);
      *((smb_fh *)  &res.diropres_u.diropres.file) = control_fh;
      LEAVE(diropres);
    }

  if ((res.status = find_user(rqstp)) != NFS_OK)
    LEAVE(diropres);

  if (fh->fh_type != SMBMFH_DIR)
    {
      res.status = NFSERR_NOTDIR;
      LEAVE(diropres);
    }
  
  result_fh = copy_fh(fh);
  result_fh->fh_type = SMBMFH_UNKNOWN;

  if ((res.status = build_full_path(fh, argp->where.name,
				    result_fh->fh_path)) != NFS_OK)
    {
      free(result_fh);
      LEAVE(diropres);
    }

  result_fh = intern_fh(result_fh);

  if ((res.status = create_file(result_fh)) != NFS_OK)
    LEAVE(diropres);

  *((smb_fh *) &res.diropres_u.diropres.file) = result_fh;
  res.status = get_attr(result_fh, result_attr);

  LEAVE(diropres);
}

#if __STDC__
nfsstat *
nfsproc_remove_2(diropargs *argp,
		 struct svc_req *rqstp)
#else
nfsstat *
nfsproc_remove_2(argp, rqstp)
     diropargs *argp;
     struct svc_req *rqstp;
#endif
{
  static nfsstat res;
  CONST smb_fh fh = * (smb_fh *) &argp->dir;
  fstring path;

  ENTRY(remove,diropargs);

  if ((res = find_user(rqstp)) != NFS_OK)
    LEAVE(nfsstat);

  /* The control file cannot be removed. */
  if (!fh->fh_path[1]
      && !strcmp(argp->name, ".control"))
    {
      res = NFSERR_PERM;
      LEAVE(nfsstat);
    }

  if ((res = build_full_path(fh, argp->name, path)) != NFS_OK)
    LEAVE(nfsstat);

  res = remove_file(path);

  LEAVE(nfsstat);
}

#if __STDC__
nfsstat *
nfsproc_rename_2(renameargs *argp,
		 struct svc_req *rqstp)
#else
nfsstat *
nfsproc_rename_2(argp, rqstp)
     renameargs *argp;
     struct svc_req *rqstp;
#endif
{
  static nfsstat res;
  CONST smb_fh from_fh = * (smb_fh *) &argp->from.dir;
  CONST smb_fh to_fh = * (smb_fh *) &argp->to.dir;
  fstring from_path, to_path;

  ENTRY(rename,renameargs);

  if ((res = find_user(rqstp)) != NFS_OK)
    LEAVE(nfsstat);

  if (from_fh->fh_type != SMBMFH_DIR || to_fh->fh_type != SMBMFH_DIR)
    {
      res = NFSERR_ACCES;
      LEAVE(nfsstat);
    }
  
  if ((res = build_full_path(from_fh, argp->from.name, from_path)) != NFS_OK
      || (res = build_full_path(to_fh, argp->to.name, to_path)) != NFS_OK)
    LEAVE(nfsstat);

  res = rename_file(from_path, to_path);
  
  LEAVE(nfsstat);
}

#if __STDC__
nfsstat *
nfsproc_link_2(linkargs *argp,
	       struct svc_req *rqstp)
#else
nfsstat *
nfsproc_link_2(argp, rqstp)
     linkargs *argp;
     struct svc_req *rqstp;
#endif
{
  static nfsstat res;

  ENTRY(link,linkargs);

  if ((res = find_user(rqstp)) != NFS_OK)
    LEAVE(nfsstat);

  res = NFSERR_ACCES;
  
  LEAVE(nfsstat);
}

#if __STDC__
nfsstat *
nfsproc_symlink_2(symlinkargs *argp,
		  struct svc_req *rqstp)
#else
nfsstat *
nfsproc_symlink_2(argp, rqstp)
     symlinkargs *argp;
     struct svc_req *rqstp;
#endif
{
  static nfsstat res;

  ENTRY(symlink,symlinkargs);

  if ((res = find_user(rqstp)) != NFS_OK)
    LEAVE(nfsstat);

  res = NFSERR_IO;

  LEAVE(nfsstat);
}

#if __STDC__
diropres *
nfsproc_mkdir_2(createargs *argp,
		struct svc_req *rqstp)
#else
diropres *
nfsproc_mkdir_2(argp, rqstp)
     createargs *argp;
     struct svc_req *rqstp;
#endif
{
  static diropres res;
  CONST smb_fh fh = * (smb_fh *) &argp->where.dir;
  smb_fh result_fh;
  fattr *result_attr = &res.diropres_u.diropres.attributes;
  fstring path;
  
  ENTRY(mkdir,createargs);

  memset(&res, 0, sizeof(res));

  if ((res.status = find_user(rqstp)) != NFS_OK)
    LEAVE(diropres);

  if (fh->fh_type != SMBMFH_DIR)
    {
      res.status = NFSERR_NOTDIR;
      LEAVE(diropres);
    }

  result_fh = copy_fh(fh);
  result_fh->fh_type = SMBMFH_UNKNOWN;

  if ((res.status = build_full_path(fh, argp->where.name,
				    result_fh->fh_path)) != NFS_OK)
    {
      free(result_fh);
      LEAVE(diropres);
    }

  result_fh = intern_fh(result_fh);

  if ((res.status = create_dir(result_fh->fh_path)) != NFS_OK)
    LEAVE(diropres);

  *((smb_fh *) &res.diropres_u.diropres.file) = result_fh;

  (void) set_attr(result_fh, &argp->attributes);

  res.status = get_attr(result_fh, result_attr);
  
  LEAVE(diropres);
}

#if __STDC__
nfsstat *
nfsproc_rmdir_2(diropargs *argp,
		struct svc_req *rqstp)
#else
nfsstat *
nfsproc_rmdir_2(argp, rqstp)
     diropargs *argp;
     struct svc_req *rqstp;
#endif
{
  static nfsstat res;
  CONST smb_fh fh = * (smb_fh *) &argp->dir;
  fstring path;

  ENTRY(rmdir,diropargs);

  if ((res = find_user(rqstp)) != NFS_OK)
    LEAVE(nfsstat);

  if ((res = build_full_path(fh, argp->name, path)) != NFS_OK)
    LEAVE(nfsstat);

  res = remove_dir(path);

  LEAVE(nfsstat);
}

/*
 * readdir is difficult.
 *
 * We do it like this:
 * On the first call (cookie == 0), we read the whole dir,
 * and store the result in a table, attached to the directory fh.
 * Subsequent requests with cookie > 0 index into the table.
 *
 * These tables time out and get freed. If we still after
 * a table has timed out get a readdir request with non-zero cookie
 * we just have to rescan the directory, and use the index for the
 * new table. The result is not necessary very accurate in that case,
 * but that shouldn' be such a big deal.
 *
 * As readdir is used primarily by the ls command, it would make sense
 * to store also the file attributes that the SMB server returns,
 * because the NFS client will probably ask for them next. But
 * this is not done yet.
 */

static void
free_dirlist(smb_dirlist *dl)
{
  free(dl->dl_entries);
  free(dl);
}

/****************************************************************************
receive a SMB trans2 response allocating the necessary memory
****************************************************************************/
static BOOL receive_trans2_response(char *inbuf,int *data_len,int *param_len,
				    char **data,char **param)
{
  int total_data=0;
  int total_param=0;

  *data_len = *param_len = 0;

  receive_smb(inbuf,0);
  if (CVAL(inbuf,smb_rcls) != 0)
    return(False);

  /* parse out the lengths */
  total_data = SVAL(inbuf,smb_tdrcnt);
  total_param = SVAL(inbuf,smb_tprcnt);

  /* allocate it */
  *data = xrealloc(*data,total_data);
  *param = xrealloc(*param,total_param);

  while (1)
    {
      memcpy(*data + *data_len,smb_base(inbuf) + SVAL(inbuf,smb_droff),
	     SVAL(inbuf,smb_drcnt));
      memcpy(*param + *param_len,smb_base(inbuf) + SVAL(inbuf,smb_proff),
	     SVAL(inbuf,smb_prcnt));
      *data_len += SVAL(inbuf,smb_drcnt);
      *param_len += SVAL(inbuf,smb_prcnt);

      /* parse out the total lengths again - they can shrink! */
      total_data = SVAL(inbuf,smb_tdrcnt);
      total_param = SVAL(inbuf,smb_tprcnt);

      if (total_data <= *data_len && total_param <= *param_len)
	break;
      receive_smb(inbuf,0);
    }
  
  return(True);
}

static char *
add_one_file(const int info_level, smb_dirlist *dl, char *p)
{
  if (Protocol < PROTOCOL_LANMAN2)
    {
      dl->dl_entries[dl->dl_nentries++] = xstrdup(p + 30);
      return p + DIR_STRUCT_SIZE;
    }

  switch (info_level)
    {
    case 1:
      dl->dl_entries[dl->dl_nentries++] = xstrdup(p + 27);
      return p + 28 + CVAL(p, 26);

    case 2:
      dl->dl_entries[dl->dl_nentries++] = xstrdup(p + 31);
      return p + 32 + CVAL(p, 30);
      
    case 260:
      dl->dl_entries[dl->dl_nentries++] = xstrdup(p + 94);
      return p + SVAL(p, 0);

    default:
      syslog(LOG_ALERT, "Invalid info_level %d in add_one_file", info_level);
      terminate(0);
    }	
}

static void
increase_dir(smb_fh fh, int searchcount)
{
  if (fh->fh_dir)
    fh->fh_dir->dl_entries =
      xrealloc(fh->fh_dir->dl_entries,
	       (fh->fh_dir->dl_nentries + searchcount) * sizeof(char *));
  else
    {
      fh->fh_dir = xzalloc(sizeof(*fh->fh_dir));
      fh->fh_dir->dl_entries = xmalloc(searchcount * sizeof(char *));
    }
}

static int
list_dir(smb_fh fh)
{
  pstring mask;
  int res;
  int first = 1;
  int searchcount = 0;

  if (Protocol >= PROTOCOL_LANMAN2)
    {
      int info_level = 1; /* NT uses 260, OS/2 uses 2. Both accept 1. */
      int max_matches = 512;
      int eos = 0;
      int resume_key = 0;
      int lastname = 0;
      int dir_handle = 0;
      int i;
      char *resp_data = NULL;
      char *resp_param = NULL;
      int resp_data_len = 0;
      int resp_param_len = 0;
      char *p, *p2;

      if ((res = build_full_path(fh, "*", mask)) != NFS_OK)
	return res;

      while (!eos)
	{
	  memset(OutBuffer, 0, smb_setup);
	  set_message(OutBuffer, 15, 5 + 12 + strlen(mask) + 1, True);
	  setup_pkt(OutBuffer, SMBtrans2, who);
	  SSVAL(OutBuffer, smb_tpscnt, 12 + strlen(mask) + 1);
	  SSVAL(OutBuffer, smb_mprcnt, 10); 
	  SSVAL(OutBuffer, smb_mdrcnt, 0xFFFF);
	  SSVAL(OutBuffer, smb_pscnt, SVAL(OutBuffer, smb_tpscnt));
	  SSVAL(OutBuffer, smb_psoff, ((smb_buf(OutBuffer)+3) - OutBuffer)-4);
	  SSVAL(OutBuffer, smb_suwcnt, 1);

	  SSVAL(OutBuffer, smb_setup0,
		first ? TRANSACT2_FINDFIRST : TRANSACT2_FINDNEXT);

	  p = smb_buf(OutBuffer);
	  *p++ = 0; /* put in a null smb_name */
	  *p++ = 'D'; *p++ = ' '; /* this was added because OS/2 does it */

	  if (first)
	    {
	      SSVAL(p, 0, aHIDDEN | aSYSTEM | aDIR); /* attribute */
	      SSVAL(p, 2, max_matches);
	      SSVAL(p, 4, 8+4+2); /* resume required + close on end + continue */
	      SSVAL(p, 6, info_level);
	      SIVAL(p, 8, 0);
	      p += 12;
	      strcpy(p, mask);
	      p += strlen(mask);
	      *p++ = 0; *p++ = 0;
	    }
	  else
	    {
	      SSVAL(p, 0, dir_handle);
	      SSVAL(p, 2, max_matches); /* max count */
	      SSVAL(p, 4, info_level); 
	      SIVAL(p, 6, resume_key);
	      SSVAL(p, 10, 12);
	      p += 12;
	      strcpy(p, mask);
	      *p++ = 0; *p++ = 0;
	    }

	  send_smb(OutBuffer);

	  receive_trans2_response(InBuffer,
				  &resp_data_len, &resp_param_len,
				  &resp_data, &resp_param);

	  if (CVAL(InBuffer, smb_rcls) != 0)
	    {
	      free(resp_data);
	      free(resp_param);
	      return map_smb_error(CVAL(InBuffer, smb_rcls), CVAL(InBuffer, smb_err));
	    }

	  /* parse out some important return info */
	  p = resp_param;
	  if (first)
	    {
	      dir_handle = SVAL(p, 0);
	      searchcount = SVAL(p, 2);
	      eos = SVAL(p,4);
	      lastname = SVAL(p, 8);
	    }
	  else
	    {
	      searchcount = SVAL(p, 0);
	      eos = SVAL(p, 2);
	      lastname = SVAL(p, 6);
	    }

	  if (searchcount == 0) 
	    break;

	  /* point to the data bytes */
	  p = resp_data;

	  /* we might need the lastname for continuations */
	  if (lastname > 0)
	    {
	      switch(info_level)
		{
		case 1:
		  resume_key = 0;
		  strcpy(mask, p + lastname + 1);
		  break;

		case 260:
		  resume_key = 0;
		  strcpy(mask, p + lastname + 94);
		  break;
		}
	    }
	  else
	    strcpy(mask,"");

	  DEBUG(3,("received %d entries (eos=%d resume=%d)\n",
		searchcount, eos, resume_key));

	  increase_dir(fh, searchcount);

	  for (i = 0; i < searchcount; i++)
	    p = add_one_file(info_level, fh->fh_dir, p);

	  free(resp_data);
	  resp_data = NULL;
	  free(resp_param);
	  resp_param = NULL;

	  first = 0;
	}
    }
  else
    {
      char *p;
      char status[21];
      int num_asked = (max_xmit - 100)/DIR_STRUCT_SIZE;
      int i;

      if ((res = build_full_path(fh, "????????.???", mask)) != NFS_OK)
	return res;

      while (1)
	{
	  memset(OutBuffer, 0, smb_size);
	  if (first)
	    set_message(OutBuffer, 2, 5 + strlen(mask), True);
	  else
	    set_message(OutBuffer, 2, 5 + 21, True);
	  setup_pkt(OutBuffer, SMBsearch, who);
	  SSVAL(OutBuffer, smb_vwv0, num_asked);
	  SSVAL(OutBuffer, smb_vwv1, aHIDDEN | aSYSTEM | aDIR);

	  p = smb_buf(OutBuffer);
	  *p++ = 4;

	  if (first)
	    strcpy(p, mask);
	  else
	    strcpy(p, "");
	  p += strlen(p) + 1;

	  *p++ = 5;
	  if (first)
	    SSVAL(p, 0, 0);
	  else
	    {
	      SSVAL(p, 0, 21);
	      p += 2;
	      memcpy(p, status, 21);
	    }

	  send_smb(OutBuffer);
	  receive_smb(InBuffer, 0);

	  searchcount = SVAL(InBuffer, smb_vwv0);

	  DEBUG(5,("dir received %d\n", searchcount));

	  if (searchcount <= 0) break;

	  p = smb_buf(InBuffer) + 3;

	  increase_dir(fh, searchcount);

	  for (i = 0; i < searchcount; i++)
	    p = add_one_file(0, fh->fh_dir, p);

	  memcpy(status, p + ((searchcount-1)*DIR_STRUCT_SIZE), 21);

	  first = 0;

	  if (CVAL(InBuffer, smb_rcls) != 0)
	    break;
	}
    }

  return NFS_OK;
}


#if __STDC__
readdirres *
nfsproc_readdir_2(readdirargs *argp,
		  struct svc_req *rqstp)
#else
readdirres *
nfsproc_readdir_2(argp, rqstp)
     readdirargs *argp;
     struct svc_req *rqstp;
#endif
{
  static readdirres res;
  CONST smb_fh fh = * (smb_fh *) &argp->dir;
  entry **e;
  int i, j, ix;
  int res_size = 0;
  CONST int CONTROL_COOKIE = 1;

#define NEXT_PLEASE()						\
  ix++, 							\
    memcpy((*e)->cookie, &ix, NFS_COOKIESIZE),			\
    res_size += dpsize(e),					\
    e = &((*e)->nextentry)					\

  ENTRY(readdir,readdirargs);

  /* Free previous result.  */
  xdr_free(xdr_readdirres, &res);

  memset(&res, 0, sizeof(res));
  
  if (fh->fh_type != SMBMFH_DIR)
    {
      res.status = NFSERR_NOTDIR;
      LEAVE(readdirres);
    }

  if ((res.status = find_user(rqstp)) != NFS_OK)
    LEAVE(readdirres);
  
  ix = 0;
  memcpy(&ix, argp->cookie, NFS_COOKIESIZE);

  e = &(res.readdirres_u.reply.entries);
  res_size = 0;

  if (!fh->fh_path[1] && ix == 0)
    {
      /* Fake .control entry */
      *e = (entry *) xzalloc(sizeof(entry));
      (*e)->fileid = (u_int) control_fh;
      (*e)->name = xstrdup(".control");

      NEXT_PLEASE();

      if (res_size + MAX_E_SIZE > argp->count)
	{
	  *e = NULL;
	  res.readdirres_u.reply.eof = 0;
	  LEAVE(readdirres);
	}
    }
  else if (ix == 0)
    ix = 1;

  if (ix == 1)
    {
      if (fh->fh_dir)
	{
	  free_dirlist(fh->fh_dir);
	  fh->fh_dir = NULL;
	}

      if ((res.status = list_dir(fh)) != NFS_OK)
	LEAVE(readdirres);
    }

  while (ix <= fh->fh_dir->dl_nentries)
    {
      *e = (entry *) xzalloc(sizeof(entry));
      (*e)->fileid = ix + 42;	/* Garbage fileid, but does anybody care? */
      (*e)->name = xstrdup(fh->fh_dir->dl_entries[ix - 1]);

      NEXT_PLEASE();

      if (res_size + MAX_E_SIZE > argp->count)
	{
	  *e = NULL;
	  res.readdirres_u.reply.eof = (ix > fh->fh_dir->dl_nentries);
	  LEAVE(readdirres);
	}
    }
  
  *e = NULL;
  res.readdirres_u.reply.eof = 1;

  LEAVE(readdirres);

#undef NEXT_PLEASE
}

/*
 * In response to my inquiry on comp.protocols.nfs, experts say that
 * the nfscookie should be XDR'd as an unsigned int )instead of
 * opaque), so I put xdr_nfscookie here and rename the one in
 * nfs_prot_xdr.c using sed in the Makefile.

From: guy@nova.netapp.com (Guy Harris)
Newsgroups: comp.protocols.nfs
Subject: Re: fileids in readdir response
Date: 17 Oct 1994 13:54:29 -0700
Organization: Network Appliance Corp.
NNTP-Posting-Host: 192.9.200.13

In article <bryan.782165529@sgl.ists.ca>, Bryan Feir <bryan@sgl.ists.ca> wrote:
 >3) The readdir 'cookie' is not actually opaque
 >
 >   The 'du' command (in SunOS 4.1.3, anyway) uses the 'getdents' system
 >call, which returns the 'offset' of the directory entry as well as the
 >entry itself.

Actually, I suspect it's not "du" that does it, but the directory
library ("opendir()", "readdir()", and so on) that use the cookie value
in that fashion.  I.e., there may be a number of "programs like du".

The 4.4BSD directory library appears to do a better job of this,
treating the "cookie" as opaque.  However, not all UNIX systems have
such an implementation of the directory library API.

 >   This can hit you in the case of network byte ordering, since the
 >cookie value is listed as an opaque array of four characters, and
 >therefore is not transformed to network byte ordering like normal long
 >values are.  Switching architectures to a little-endian machine causes
 >problems because what used to be nice, increasing numbers get their
 >bytes scrambled and no longer look like small increasing numbers.

Yup.  XDR it as an unsigned integral quantity.

 * 
 */

bool_t
xdr_nfscookie(xdrs, objp)
	XDR *xdrs;
	nfscookie objp;
{
	if (!xdr_u_int(xdrs, (u_int *)objp)) {
		return (FALSE);
	}
	return (TRUE);
}

#if __STDC__
statfsres *
nfsproc_statfs_2(nfs_fh *argp,
		 struct svc_req *rqstp)
#else
statfsres *
nfsproc_statfs_2(argp, rqstp)
     nfs_fh *argp;
     struct svc_req *rqstp;
#endif
{
  static statfsres res;
  statfsokres *p = &res.statfsres_u.reply;

  ENTRY(statfs, nfs_fh);

  memset(&res, 0, sizeof(res));

  if (users == NULL
      || ((who = some_session()) == NULL
	  && (who = users, !login_user(users))))
    {
      p->tsize = 8192;
      p->bsize = 512;
      p->blocks = 200000;
      p->bavail = p->bfree = 100000;

      LEAVE(statfsres);
    }

  time(&who->u_timestamp);

  memset(OutBuffer, 0, smb_size);
  set_message(OutBuffer, 0, 0, True);
  setup_pkt(OutBuffer, SMBdskattr, who);

  send_smb(OutBuffer);
  receive_smb(InBuffer, 0);

  if (CVAL(InBuffer, smb_rcls) != SUCCESS)
    {
      res.status = map_smb_error(CVAL(InBuffer, smb_rcls), CVAL(InBuffer, smb_err));
      LEAVE(statfsres);
    }

  p->tsize = 8192;
  p->bsize = SVAL(InBuffer, smb_vwv2);
  p->blocks = SVAL(InBuffer, smb_vwv0) * SVAL(InBuffer, smb_vwv1);
  p->bavail = p->bfree = SVAL(InBuffer, smb_vwv3) * SVAL(InBuffer, smb_vwv1);

  LEAVE(statfsres);
}

#if __STDC__
void *
xmalloc(size_t nbytes)
#else
void *
xmalloc(nbytes)
     size_t nbytes;
#endif
{
  void *result;

  if ((result = malloc(nbytes)) == NULL)
    {
      syslog(LOG_ALERT, "Out of memory (tried to malloc %d bytes)", nbytes);
      terminate(0);
    }
  return result;
}

#if __STDC__
void *
xrealloc(void *ptr, size_t nbytes)
#else
void *
xrealloc(ptr, nbytes)
     void *ptr;
     size_t nbytes;
#endif
{
  void *result;

  if ((result = realloc(ptr, nbytes)) == NULL)
    {
      syslog(LOG_ALERT, "Out of memory (tried to realloc %d bytes)", nbytes);
      terminate(0);
    }
  return result;
}  

#if __STDC__
void *
xzalloc(size_t nbytes)
#else
void *
xzalloc(nbytes)
     size_t nbytes;
#endif
{
  void *result;

  if ((result = calloc(1, nbytes)) == NULL)
    {
      syslog(LOG_ALERT, "Out of memory (tried to malloc %d bytes)", nbytes);
      terminate(0);
    }
  return result;
}
