/* Functions to log NFS RPC args and results */

/*
Copyright Technical Research Centre of Finland (VTT), 
Information Technology Institute (TTE) 1993, 1994 - All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the names of VTT or TTE not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission. 

VTT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
VTT BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
*/

static char RCS_Id[] = "$Id: log_nfs.c,v 1.9 1994/10/16 14:16:31 tml Exp $";

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <rpc/rpc.h>

#include "nfs_prot.h"
#include "log_nfs.h"
#include "util.h"

void (*fh_printer)() = NULL;

#if __STDC__
char *
indent(int level)
#else
char *
indent(level)
     int level;
#endif
{
  return level ? "\t\t" : "";
}

#if __STDC__
void
print_auth(struct svc_req *rqstp)
#else
void
print_auth(rqstp)
     struct svc_req *rqstp;
#endif
{
  struct authunix_parms *unix_cred;
  int i;

  switch (rqstp->rq_cred.oa_flavor)
    {
    case AUTH_UNIX:
      unix_cred = (struct authunix_parms *) rqstp->rq_clntcred;

      printf("uid: %d, gid: %d, ngroups: %d",
	     unix_cred->aup_uid, unix_cred->aup_gid,
	     unix_cred->aup_len);
      if (unix_cred->aup_len > 0)
	{
	  printf(", gids: ");
	  for (i = 0; i < unix_cred->aup_len; i++)
	    printf("%s%d", i > 0 ? ", " : "", unix_cred->aup_gids[i]);
	}
      printf("\n");
      return;
      
    case AUTH_NULL:
      printf("no auth\n");
      return;

    default:
      printf("unknown auth %d\n", rqstp->rq_cred.oa_flavor);
      return;
    }
}

#if __STDC__
static const char *
nfs_strerror(nfsstat errno)
#else
static char *
nfs_strerror(errno)
     nfsstat errno;
#endif
{
  switch(errno)
    {
#if __STDC__
#define C(s) case s: return #s
#else 
#define C(s) case s: return "s"
#endif
      C(NFSERR_PERM);
      C(NFSERR_NOENT);
      C(NFSERR_IO);
      C(NFSERR_NXIO);
      C(NFSERR_ACCES);
      C(NFSERR_EXIST);
      C(NFSERR_NODEV);
      C(NFSERR_NOTDIR);
      C(NFSERR_ISDIR);
      C(NFSERR_FBIG);
      C(NFSERR_NOSPC);
      C(NFSERR_ROFS);
      C(NFSERR_NAMETOOLONG);
      C(NFSERR_NOTEMPTY);
      C(NFSERR_DQUOT);
      C(NFSERR_STALE);
      C(NFSERR_WFLUSH);
#undef C
    default: return "Unknown NFS error";
    }
}

#if __STDC__
void
print_nfsstat(int level, nfsstat *value)
#else
void
print_nfsstat(level, value)
     int level;
     nfsstat *value;
#endif
{
  printf("%s%s", indent(level), *value == NFS_OK ? "OK" : nfs_strerror(*value));
}

#if __STDC__
void
print_nfs_fh(int level, nfs_fh *value)
#else
void
print_nfs_fh(level, value)
     int level;
     nfs_fh *value;
#endif
{
  int i;

  if (fh_printer)
    {
      (*fh_printer)(level, value);
      return;
    }

  printf("%snfs_fh: ", indent(level));

  for (i = 0; i < sizeof(nfs_fh) / sizeof(int); i++)
    printf("%#010x ", ((int *)value)[i]);
  printf("\n");
}

#if __STDC__
void
print_fattr(int level, fattr *value)
#else
void
print_fattr(level, value)
     int level;
     fattr *value;
#endif
{
  printf("%stype: %d, mode: %06o, nlink: %d, uid: %d, gid: %d\n", indent(level),
	 value->type, value->mode, value->nlink, value->uid, value->gid);
  printf("%ssize: %d, blocksize: %d, rdev: %d, blocks: %d\n", indent(level),
	 value->size, value->blocksize, value->rdev, value->blocks);
  printf("%sfsid: %d, fileid: %d\n", indent(level),
	 value->fsid, value->fileid);
  printf("%satime: %d.%d, mtime: %d.%d, ctime: %d.%d",
	 indent(level),
	 value->atime.seconds, value->atime.useconds,
	 value->mtime.seconds, value->mtime.useconds,
	 value->ctime.seconds, value->ctime.useconds);
}

#if __STDC__
void
print_attrstat(int level, attrstat *value)
#else
void
print_attrstat(level, value)
     int level;
     attrstat *value;
#endif
{
  print_nfsstat(level, &value->status);
  if (value->status == NFS_OK)
    {
      putchar('\n');
      print_fattr(level, &value->attrstat_u.attributes);
    }
}

#if __STDC__
void
print_sattr(int level, sattr *value)
#else
void
print_sattr(level, value)
     int level;
     sattr *value;
#endif
{
  printf("%smode: %06o, uid: %d, gid: %d, size: %d\n", indent(level),
	 value->mode, value->uid, value->gid, value->size);
  printf("%satime: %d.%d, mtime: %d.%d\n", indent(level),
	 value->atime.seconds, value->atime.useconds,
	 value->mtime.seconds, value->mtime.useconds);
}

#if __STDC__
void
print_sattrargs(int level, sattrargs *value)
#else
void
print_sattrargs(level, value)
     int level;
     sattrargs *value;
#endif
{
  print_nfs_fh(level, &value->file);
  print_sattr(level, &value->attributes);
}

#if __STDC__
void
print_diropargs(int level, diropargs *value)
#else
void
print_diropargs(level, value)
     int level;
     diropargs *value;
#endif
{
  print_nfs_fh(level, &value->dir);
  printf("%s\"%s\"\n", indent(level), value->name);
}

#if __STDC__
void
print_diropokres(int level, diropokres *value)
#else
void
print_diropokres(level, value)
     int level;
     diropokres *value;
#endif
{
  print_nfs_fh(level, &value->file);
  print_fattr(level, &value->attributes);
}
  
#if __STDC__
void
print_diropres(int level, diropres *value)
#else
void
print_diropres(level, value)
     int level;
     diropres *value;
#endif
{
  print_nfsstat(level, &value->status);
  if (value->status == NFS_OK)
    {
      putchar('\n');
      print_diropokres(level, &value->diropres_u.diropres);
    }
}

#if __STDC__
void
print_readlinkres(int level, readlinkres *value)
#else
void
print_readlinkres(level, value)
     int level;
     readlinkres *value;
#endif
{
  print_nfsstat(level, &value->status);
  if (value->status == NFS_OK)
    printf(" \"%s\"\n", value->readlinkres_u.data);
}

#if __STDC__
void
print_readargs(int level, readargs *value)
#else
void
print_readargs(level, value)
     int level;
     readargs *value;
#endif
{
  print_nfs_fh(level, &value->file);
  printf("%soffset: %d, count: %d\n", indent(level),
	 value->offset, value->count);
}

#if __STDC__
static char *
visibilify(char *data, int len)
#else
static char *
visibilify(data, len)
     char *data;
     int len;
#endif
{
  static char buf[40];
  char *p = buf;
  
  while (len-- && p < buf + sizeof(buf) - 5)
    {
      if (*data == '\\' || *data == '"')
	*p++ = '\\', *p++ = *data;
      else if (isprint(*data))
	*p++ = *data;
      else
	*p++ = '\\',
	*p++ = ((*data >> 6) & 0x7) + '0',
	*p++ = ((*data >> 3) & 0x7) + '0',
	*p++ = ((*data) & 0x7) + '0';
      data++;
    }
  *p = 0;
  return buf;
}

#if __STDC__
void
print_readokres(int level, readokres *value)
#else
void
print_readokres(level, value)
     int level;
     readokres *value;
#endif
{
  print_fattr(level, &value->attributes);
  printf("%slen: %d, data: \"%s\"", indent(level),
	 value->data.data_len, visibilify(value->data.data_val,
					  value->data.data_len));
}

#if __STDC__
void
print_readres(int level, readres *value)
#else
void
print_readres(level, value)
     int level;
     readres *value;
#endif
{
  print_nfsstat(level, &value->status);
  if (value->status == NFS_OK)
    {
      putchar('\n');
      print_readokres(level, &value->readres_u.reply);
    }
}

#if __STDC__
void
print_writeargs(int level, writeargs *value)
#else
void
print_writeargs(level, value)
     int level;
     writeargs *value;
#endif
{
  print_nfs_fh(level, &value->file);
  printf("%soffset: %d, len: %d, data: \"%s\"\n", indent(level),
	 value->offset, value->data.data_len,
	 visibilify(value->data.data_val, value->data.data_len));
}

#if __STDC__
void
print_createargs(int level, createargs *value)
#else
void
print_createargs(level, value)
     int level;
     createargs *value;
#endif
{
  print_diropargs(level, &value->where);
  print_sattr(level, &value->attributes);
}

#if __STDC__
void
print_renameargs(int level, renameargs *value)
#else
void
print_renameargs(level, value)
     int level;
     renameargs *value;
#endif
{
  print_diropargs(level, &value->from);
  print_diropargs(level, &value->to);
}

#if __STDC__
void
print_linkargs(int level, linkargs *value)
#else
void
print_linkargs(level, value)
     int level;
     linkargs *value;
#endif
{
  print_nfs_fh(level, &value->from);
  print_diropargs(level, &value->to);
}

#if __STDC__
void
print_symlinkargs(int level, symlinkargs *value)
#else
void
print_symlinkargs(level, value)
     int level;
     symlinkargs *value;
#endif
{
  print_diropargs(level, &value->from);
  printf("%s\"%s\"\n", indent(level), value->to);
  print_sattr(level, &value->attributes);
}

#if __STDC__
void
print_readdirargs(int level, readdirargs *value)
#else
void
print_readdirargs(level, value)
     int level;
     readdirargs *value;
#endif
{
  int cookie;
  
  print_nfs_fh(level, &value->dir);
  cookie = 0;
  memcpy(&cookie, value->cookie, NFS_COOKIESIZE);
  printf("%scookie: %d, count: %d\n", indent(level),
	 cookie, value->count);
}

#if __STDC__
void
print_entry(int level, entry *value)
#else
void
print_entry(level, value)
     int level;
     entry *value;
#endif
{
  int cookie;
  
  if (value)
    {
      cookie = 0;
      memcpy(&cookie, value->cookie, NFS_COOKIESIZE);
      printf("%sfileid: %d, \"%s\", cookie: %d\n", indent(level),
	     value->fileid, visibilify(value->name, strlen(value->name)),
	     cookie);
      print_entry(level, value->nextentry);
    }
}

#if __STDC__
void
print_dirlist(int level, dirlist *value)
#else
void
print_dirlist(level, value)
     int level;
     dirlist *value;
#endif
{
  print_entry(level, value->entries);
  printf("%seof: %s", indent(level), value->eof ? "true" : "false");
}

#if __STDC__
void
print_readdirres(int level, readdirres *value)
#else
void
print_readdirres(level, value)
     int level;
     readdirres *value;
#endif
{
  print_nfsstat(level, &value->status);
  if (value->status == NFS_OK)
    {
      putchar('\n');
      print_dirlist(level, &value->readdirres_u.reply);
    }
}

#if __STDC__
void
print_statfsres(int level, statfsres *value)
#else
void
print_statfsres(level, value)
     int level;
     statfsres *value;
#endif
{
  print_nfsstat(level, &value->status);
  putchar('\n');
  if (value->status == NFS_OK)
    printf("%stsize: %d, bsize: %d, blocks: %d, bfree: %d, bavail: %d",
	   indent(level),
	   value->statfsres_u.reply.tsize, value->statfsres_u.reply.bsize,
	   value->statfsres_u.reply.blocks, value->statfsres_u.reply.bfree,
	   value->statfsres_u.reply.bavail);
}
