/******************************************************************************
 ******************************************************************************
 **
 ** COPYRIGHT (C) 1998 By Arthur Naseef
 **
 ** This file is covered under the GNU General Public License Version 2.  For
 **  more information, see the file COPYING.
 **
 **
 ** FILE: ovl_stg.c
 **
 ** DESCRIPTION: Place something descriptive here
 **
 ** REVISION HISTORY:
 **
 ** DATE	AUTHOR		DESCRIPTION
 ** ==========	===============	==============================================
 ** 09/27/1997	art            	Added to source code control.
 ** 02/23/1998	ARTHUR N.	ovlfs_add_inode(): set the super block AFTER
 **				 initializing the new result to all 0's.
 ** 02/26/1998	ARTHUR N.	Added four more inode fields to the retained
 **				 information for each inode.
 ** 02/28/1998	ARTHUR N.	Set the superblock to be dirty on any change.
 ** 03/01/1998	ARTHUR N.	Added support for the maxmem fs option and
 **				 changed all kmalloc's to allocate KERNEL
 **				 priority memory instead of high priority mem.
 ** 03/01/1998	ARTHUR N.	Removed unused link_op functions and changed
 **				 ovlfs_unlink_dirent() so that it does not
 **				 update the link count of the unlinked file.
 ** 03/02/1998	ARTHUR N.	Use the stored inode from the superblock for
 **				 the storage file.
 ** 03/02/1998	ARTHUR N.	Do not fail the ovlfs_stg_read_super() if the
 **				 storage file is empty.
 ** 03/03/1998	ARTHUR N.	Allow the storage file to be empty.  This is
 **				 necessary since the change for keeping the
 **				 file's inode was made.
 ** 03/05/1998	ARTHUR N.	Added support for the mapping store options
 **				 and changed update of mapping lists so that
 **				 one inode, within a specific device, only
 **				 has one mapping entry within a single list.
 ** 03/05/1998	ARTHUR N.	Added printk() statements for some error
 **				 conditions, and started use of DEBUG_PRINT1.
 ** 03/05/1998	ARTHUR N.	Malloc-checking support added.
 ** 03/09/1998	ARTHUR N.	Added the copyright notice.
 ** 03/10/1998	ARTHUR N.	Don't set the superblock as dirty if told so
 **				 by mount options.
 **
 ******************************************************************************
 ******************************************************************************
 **/

#ident "@(#) ovl_stg.c 1.14"

#ifdef MODVERSIONS
# include <linux/modversions.h>
# ifndef __GENKSYMS__
#  include "ovlfs.ver"
# endif
#endif

#include <linux/malloc.h>
#include <linux/fcntl.h>
#include <linux/fs.h>

#include "kernel_lists.h"
#include "ovl_fs.h"



/**
 ** STATIC PROTOTYPES:
 **/

static int  ovlfs_read_super(struct super_block *);



/**
 ** FUNCTION: comp_ovl_ino
 **/

static int  comp_ovl_ino (ovlfs_inode_t *ino1, ovlfs_inode_t *ino2)
{
#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "comp_ovl_ino\n");
#endif

    if ( ino1 == (ovlfs_inode_t *) NULL )
        return  1;
    else if ( ino2 == (ovlfs_inode_t *) NULL )
        return  -1;


        /* Sort first by device number, then by inode number */

    if ( ino1->base_dev == ino2->base_dev )
        return ino1->base_ino - ino2->base_ino;
    else
        return ino1->base_dev - ino2->base_dev;
}



/**
 ** FUNCTION: comp_ino_map
 **/

static int  comp_ino_map (ovlfs_ino_map_t *map1, ovlfs_ino_map_t *map2)
{
#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "comp_ino_map\n");
#endif

    if ( map1 == (ovlfs_ino_map_t *) NULL )
        return  1;
    else if ( map2 == (ovlfs_ino_map_t *) NULL )
        return  -1;


        /* Sort first by device number, then by inode number */

    if ( map1->dev == map2->dev )
        return  map1->ino - map2->ino;
    else
        return  map1->dev - map2->dev;
}



/**
 ** FUNCTION: comp_dirent
 **/

static int  comp_dirent (ovlfs_dir_info_t *ent1, ovlfs_dir_info_t *ent2)
{
#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "comp_dirent\n");
#endif

    if ( ent1 == (ovlfs_dir_info_t *) NULL )
        return  1;
    else if ( ent2 == (ovlfs_dir_info_t *) NULL )
        return  -1;

        /* Sort first by name length, then name; this should be faster on */
        /*  better than average.                                          */

    if ( ent1->len == ent2->len )
        return  strncmp(ent1->name, ent2->name, ent1->len);
    else
        return  ent1->len - ent2->len;
}



/**
 ** FUNCTION: create_inode_list
 **
 **  PURPOSE: Create a new, empty, inode list
 **/

static inline list_t    create_inode_list ()
{
#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "create_inode_list\n");
#endif

    return  klist_func(create_list)(NULL, (comp_func_t) comp_ovl_ino,
                                    (comp_func_t) comp_ovl_ino, NULL);
}



/**
 ** FUNCTION: create_ino_map_list
 **
 **  PURPOSE: Create a new, empty, inode mapping list.
 **/

static inline list_t    create_ino_map_list ()
{
#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "create_inode_map_list\n");
#endif

    return  klist_func(create_list)(NULL, (comp_func_t) comp_ino_map,
                                    (comp_func_t) comp_ino_map, NULL);
}



/**
 ** FUNCTION: create_dirent_list
 **
 **  PURPOSE: Create a new, empty, directory entry list.
 **/

static inline list_t    create_dirent_list ()
{
#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "create_dirent_list\n");
#endif

    return  klist_func(create_list)(NULL, (comp_func_t) comp_dirent,
                                    (comp_func_t) comp_dirent, NULL);
}



/**
 ** FUNCTION: free_dir_entry
 **
 **  PURPOSE: Free the memory used by the given directory entry.
 **/

static void free_dir_entry (ovlfs_dir_info_t *d_ent)
{
#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "free_dir_entry\n");
#endif

    if ( d_ent != (ovlfs_dir_info_t *) NULL )
    {
        if ( d_ent->name != (char *) NULL )
            FREE(d_ent->name);

        FREE(d_ent);
    }
}



/**
 ** FUNCTION: free_ovlfs_inode
 **/

static void free_ovlfs_inode (ovlfs_inode_t *ino)
{
#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "free_ovlfs_inode\n");
#endif

    if ( ino != (ovlfs_inode_t *) NULL )
    {
        if ( ino->dir_entries != (list_t) NULL )
        {
            klist_func(foreach_node)(ino->dir_entries,
                                     (foreach_func_t) free_dir_entry);
            klist_func(destroy_list)(ino->dir_entries);
        }

        if ( ino->name != (char *) NULL )
            FREE(ino->name);

        FREE(ino);
    }
}



/**
 ** FUNCTION: alloc_inode
 **
 **  PURPOSE: Allocate space for an inode in our storage.
 **/

static inline ovlfs_inode_t	*alloc_inode (struct super_block *sb)
{
	ovlfs_inode_t	*result;
	int		nomem;
	struct sysinfo	mem_info;

		/* Before attempting to allocate the memory, check memory if */
		/*  the option was set for this filesystem.                  */

	nomem = 0;

	if ( ovlfs_sb_opt(sb, storage_opts) & OVLFS_STG_CK_MEM )
	{
			/* Obtain the memory information and make sure that */
			/*  the free ram available is >= 10% of the total.  */

		si_meminfo(&mem_info);

		if ( mem_info.freeram < ( mem_info.totalram * 0.1 ) )
			nomem = 1;
	}

	if ( ! nomem )
		result = (ovlfs_inode_t *) MALLOC(sizeof(ovlfs_inode_t));
	else
		result = (ovlfs_inode_t *) NULL;

	return	result;
}



/**
 ** FUNCTION: ovlfs_add_inode
 **
 **  PURPOSE: Given the super block for an overlay filesystem, add a new
 **           inode to the filesystem.
 **/


ovlfs_inode_t   *ovlfs_add_inode (struct super_block *sup_blk, const char *name,
                                  int len)
{
    ovlfs_inode_t       *result;
    ovlfs_super_info_t  *s_info;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_add_inode\n");
#endif

#if ! FASTER_AND_MORE_DANGEROUS

    if ( ( sup_blk == (struct super_block *) NULL ) ||
         ( sup_blk->u.generic_sbp == NULL ) ||
         ( sup_blk->s_magic != OVLFS_SUPER_MAGIC ) )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_add_inode: invalid super block given\n");
# endif

        return  (ovlfs_inode_t *) NULL;
    }

#endif

    s_info = (ovlfs_super_info_t *) sup_blk->u.generic_sbp;

    if ( s_info->inodes == (list_t) NULL )
    {
        s_info->inodes = create_inode_list();

        if ( s_info->inodes == (list_t) NULL )
        {
#if KDEBUG
            printk("OVLFS: unable to allocate needed memory\n");
#endif
            return  (ovlfs_inode_t *) NULL;
        }

        if ( ! (ovlfs_sb_opt(sup_blk, storage_opts) & OVLFS_UPD_UMOUNT_ONLY) )
        {
            sup_blk->s_dirt = 1;
        }
    }


    result = alloc_inode(sup_blk);

    if ( result != (ovlfs_inode_t *) NULL )
    {
        memset(result, '\0', sizeof(ovlfs_inode_t));
        result->sb = sup_blk;
        result->ino = klist_func(list_length)(s_info->inodes) + 1;
        result->base_dev = NODEV;
        result->stg_dev = NODEV;

            /* If a name was given, save the name; this is not critical, */
            /*  so do not fail the entire function if this fails.        */

        if ( ( name != (char *) NULL ) && ( len > 0 ) )
        {
            result->name = MALLOC(len);

            if ( result->name != (char *) NULL )
            {
                memcpy(result->name, name, len);
                result->len = len;
            }
#if KDEBUG
            else
                printk("OVLFS: ovlfs_add_inode: unable to allocate space for "
                       "name\n");
#endif
        }
                

        if ( klist_func(insert_at_end)(s_info->inodes, result) == NULL )
        {
#if KDEBUG
            printk("OVLFS: ovlfs_add_inode: unable to add new inode to list\n");
#endif

            FREE(result);
            result = (ovlfs_inode_t *) NULL;
        }
        else
            if ( ! ( ovlfs_sb_opt(sup_blk, storage_opts) &
                     OVLFS_UPD_UMOUNT_ONLY ) )
            {
                sup_blk->s_dirt = 1;
            }
    }
#if KDEBUG > 1
    else
        printk("OVLFS: unable to allocate memory for a new storage inode\n");
#endif

    return  result;
}



/**
 ** FUNCTION: check_mapping
 **
 **  PURPOSE: Check to make sure the given entry does not already exist in
 **           the given map list, and update it if it does.
 **
 ** RETURNS:
 **	0	- If the mapping does not already exist.
 **	1	- If the mapping does already exist.
 **/

static int  check_mapping (list_t map_list, dev_t dev, _uint ino,
                           _uint ovl_ino)
{
    ovlfs_ino_map_t *mapping;
    ovlfs_ino_map_t tmp;
    int             ret;

    if ( map_list != (list_t) NULL )
    {
        tmp.dev     = dev;
        tmp.ino     = ino;
        tmp.ovl_ino = ovl_ino;

        mapping = (ovlfs_ino_map_t *)
                  klist_func(search_for_element)(map_list, &tmp);

        if ( mapping == (ovlfs_ino_map_t *) NULL )
            ret = 0;
        else
        {
            mapping->ovl_ino = ovl_ino;
            ret = 1;
        }
    }
    else
        ret = 0;

    return ret;
}



/**
 ** FUNCTION: ovlfs_inode_set_base
 **/

int  ovlfs_inode_set_base (struct super_block *sup_blk, ovlfs_inode_t *ino,
                           dev_t b_dev, _uint b_ino)
{
    ovlfs_ino_map_t     *mapping;
    ovlfs_super_info_t  *s_info;
    int                 ret_code = 0;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_inode_set_base\n");
#endif

#if ! FASTER_AND_MORE_DANGEROUS
    if ( ino == (ovlfs_inode_t *) NULL )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_inode_set_base: null inode info given\n");
# endif
        return  -EINVAL;
    }
    else if ( ( sup_blk == (struct super_block *) NULL ) ||
              ( sup_blk->u.generic_sbp == NULL ) ||
              ( sup_blk->s_magic != OVLFS_SUPER_MAGIC ) )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_inode_set_base: invalid super block given\n");
# endif

        return  -EINVAL;
    }
#endif

    s_info = (ovlfs_super_info_t *) sup_blk->u.generic_sbp;


    ino->base_dev = b_dev;
    ino->base_ino = b_ino;


        /* Set the base device and inode number in storage to the values */
        /*  values given.  First see if the mapping already exists and   */
        /*  update it if so.                                             */

    if ( check_mapping(s_info->base_to_ovl, b_dev, b_ino, ino->ino) )
        ;
    else
    {
            /* Setup the mapping from the base inode to the overlay fs inode. */

        mapping = create_ino_map();

        if ( mapping != (ovlfs_ino_map_t *) NULL )
        {
            mapping->dev = b_dev;
            mapping->ino = b_ino;
            mapping->ovl_ino = ino->ino;

                /* Now insert this element into the mapping list */

            if ( s_info->base_to_ovl == (list_t) NULL )
            {
                s_info->base_to_ovl = create_ino_map_list();

                if ( s_info->base_to_ovl == (list_t) NULL )
                {
#if KDEBUG
                    printk("OVLFS: ovlfs_ino_set_base: unable to create a list for"
                           " mapping\n");
#endif

                    FREE(mapping);
                    ret_code = -ENOMEM;
                }
            }

            if ( ret_code >= 0 )
            {
                if ( klist_func(insert_into_list_sorted)(s_info->base_to_ovl,
                                                         mapping) == NULL )
                {
#if KDEBUG
                    printk("OVLFS: ovlfs_ino_set_base: unable to insert element "
                           "into mapping list\n");
#endif

                    FREE(mapping);
                    ret_code = -ENOMEM;
                }
                else
                    if ( ! ( ovlfs_sb_opt(sup_blk, storage_opts) &
                             OVLFS_UPD_UMOUNT_ONLY ) )
                    {
                        sup_blk->s_dirt = 1;
                    }
            }
        }
        else
        {
#if KDEBUG
            printk("OVLFS: ovlfs_ino_set_base: unable to allocate %d bytes\n",
                   sizeof(ovlfs_ino_map_t));
#endif

            ret_code = -ENOMEM;
        }
    }

    return  ret_code;
}



/**
 ** FUNCTION: ovlfs_inode_set_stg
 **/

int  ovlfs_inode_set_stg (struct super_block *sup_blk, ovlfs_inode_t *ino,
                          dev_t s_dev, _uint s_ino)
{
    ovlfs_ino_map_t     *mapping;
    ovlfs_super_info_t  *s_info;
    int                 ret_code = 0;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_inode_set_stg\n");
#endif

#if ! FASTER_AND_MORE_DANGEROUS
    if ( ino == (ovlfs_inode_t *) NULL )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_inode_set_stg: null inode info given\n");
# endif
        return  -EINVAL;
    }
    else if ( ( sup_blk == (struct super_block *) NULL ) ||
              ( sup_blk->u.generic_sbp == NULL ) ||
              ( sup_blk->s_magic != OVLFS_SUPER_MAGIC ) )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_inode_set_stg: invalid super block given\n");
# endif

        return  -EINVAL;
    }
#endif

    s_info = (ovlfs_super_info_t *) sup_blk->u.generic_sbp;

    ino->stg_dev = s_dev;
    ino->stg_ino = s_ino;


        /* Now, setup the mapping from storage to the storage fs inode, but */
        /*  first see if the mapping already exists and update it if so.    */

    if ( check_mapping(s_info->stg_to_ovl, s_dev, s_ino, ino->ino) )
        ;
    else
    {
        mapping = create_ino_map();

        if ( mapping != (ovlfs_ino_map_t *) NULL )
        {
            mapping->dev = s_dev;
            mapping->ino = s_ino;
            mapping->ovl_ino = ino->ino;

                /* Now insert this element into the mapping list */

            if ( s_info->stg_to_ovl == (list_t) NULL )
            {
                s_info->stg_to_ovl = create_ino_map_list();

                if ( s_info->stg_to_ovl == (list_t) NULL )
                {
#if KDEBUG
                    printk("OVLFS: ovlfs_ino_set_stg: unable to create a list for"
                           " mapping\n");
#endif

                    FREE(mapping);
                    ret_code = -ENOMEM;
                }
            }

            if ( ret_code >= 0 )
            {
                if ( klist_func(insert_into_list_sorted)(s_info->stg_to_ovl,
                                                         mapping) == NULL )
                {
#if KDEBUG
                    printk("OVLFS: ovlfs_ino_set_stg: unable to insert "
                           "element into mapping list\n");
#endif

                    FREE(mapping);
                    ret_code = -ENOMEM;
                }
                else
                    if ( ! ( ovlfs_sb_opt(sup_blk, storage_opts) &
                             OVLFS_UPD_UMOUNT_ONLY ) )
                    {
                        sup_blk->s_dirt = 1;
                    }
            }
        }
        else
        {
#if KDEBUG
            printk("OVLFS: ovlfs_ino_set_stg: unable to allocate %d bytes\n",
                   sizeof(ovlfs_ino_map_t));
#endif

            ret_code = -ENOMEM;
        }
    }

    return  ret_code;
}



/**
 ** FUNCTION: ovlfs_setup_new_inode
 **/

ovlfs_inode_t   *ovlfs_setup_new_inode (struct super_block *sup_blk,
                                        const char *name, int len,
                                        dev_t b_dev, _ulong b_ino,
                                        dev_t s_dev, _ulong s_ino)
{
    ovlfs_inode_t   *result;
    int             rc;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_setup_new_inode\n");
#endif

    result = ovlfs_add_inode(sup_blk, name, len);

    if ( result != (ovlfs_inode_t *) NULL )
    {
        if ( b_dev != NODEV )
            rc = ovlfs_inode_set_base(sup_blk, result, b_dev, b_ino);
        else
            rc = 0;

        if ( ( rc >= 0 ) && ( s_dev != NODEV ) )
            rc = ovlfs_inode_set_stg(sup_blk, result, s_dev, s_ino);
    }

    return  result;
}



/**
 ** FUNCTION: ovlfs_get_ino
 **/

ovlfs_inode_t   *ovlfs_get_ino (struct super_block *sup_blk, _ulong ino)
{
    ovlfs_super_info_t  *s_info;
    ovlfs_inode_t       *result;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_get_ino\n");
#endif

#if ! FASTER_AND_MORE_DANGEROUS
    if ( ( sup_blk == (struct super_block *) NULL ) ||
         ( sup_blk->s_magic != OVLFS_SUPER_MAGIC ) ||
         ( sup_blk->u.generic_sbp == NULL ) )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_get_ino: invalid super block given\n");
# endif

        return  (ovlfs_inode_t *) NULL;
    }
#endif

    if ( ino == 0 )
        return  (ovlfs_inode_t *) NULL;

    s_info = (ovlfs_super_info_t *) sup_blk->u.generic_sbp;

    if ( s_info->inodes == (list_t) NULL )
        result = (ovlfs_inode_t *) NULL;
    else
        result = (ovlfs_inode_t *) klist_func(nth_element)(s_info->inodes, ino);

    return  result;
}



/**
 ** FUNCTION: ovlfs_lookup_ino
 **/

_ulong  ovlfs_lookup_ino (struct super_block *sup_blk, kdev_t dev, _ulong ino,
                          char which)
{
    ovlfs_super_info_t  *s_info;
    ovlfs_ino_map_t     *i_map;
    ovlfs_ino_map_t     tmp;
    _ulong              result;
    list_t              map_list;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_lookup_ino\n");
#endif


        /* Verify the arguments: the super block is required and must be */
        /*  the super block for an overlay filesystem.                   */

#if ! FASTER_AND_MORE_DANGEROUS
    if ( ( sup_blk == (struct super_block *) NULL ) ||
         ( sup_blk->s_magic != OVLFS_SUPER_MAGIC ) ||
         ( sup_blk->u.generic_sbp == NULL ) )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_lookup_ino: invalid super block given\n");
# endif

        return  0;
    }
#endif


    s_info = (ovlfs_super_info_t *) sup_blk->u.generic_sbp;

        /* Determine which list to look in for the mapping */

    switch ( which )
    {
        case 'b':
            map_list = s_info->base_to_ovl;
            break;

        case 's':
        case 'o':
            map_list = s_info->stg_to_ovl;
            break;

        default:
# if KDEBUG
            printk("OVLFS: ovlfs_lookup_ino: invalid flag, '%c'.\n", which);
# endif
            map_list = (list_t) NULL;
            break;
    }

    if ( map_list == (list_t) NULL )
        result = 0;
    else
    {
            /* Look for the mapping from the specified device and inode # */

        tmp.dev = dev;
        tmp.ino = ino;
        i_map = (ovlfs_ino_map_t *)
                klist_func(search_for_element)(map_list, &tmp);

        if ( i_map != (ovlfs_ino_map_t *) NULL )
            result = i_map->ovl_ino;
        else
            result = 0;
    }

    return  result;
}



/**
 ** FUNCTION: ovlfs_ino_add_dirent
 **
 **  PURPOSE: Given an ovlfs inode structure, add the named directory entry
 **           to the inode's list of entries.
 **
 ** NOTES:
 **     - Due to the manner in which entries are inserted into the directory's
 **       list of entries, this function must not be called between the start
 **       and end of a readdir() call; otherwise, entries may appear more than
 **       once, and other entries will be missing.
 **/

int ovlfs_ino_add_dirent (ovlfs_inode_t *ino, const char *name, int len,
                          _ulong ent_ino, int flags)
{
    int                 ret_code;
    ovlfs_dir_info_t    *dirent;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_ino_add_dirent\n");
#endif

#if ! FASTER_AND_MORE_DANGEROUS
    if ( ( ino == (ovlfs_inode_t *) NULL ) || ( name == (char *) NULL ) )
    {
# if KDEBUG
            printk("OVLFS: ovlfs_ino_add_dirent: null argument given\n");
# endif

            return  -EINVAL;
    }
#endif

    ret_code = 0;

    if ( ino->dir_entries == (list_t) NULL )
        ino->dir_entries = create_dirent_list();

        /* See if the directory already has the entry; if it does, check */
        /*  if the entry is marked as unlinked and just remove it first  */
        /*  if so.                                                       */

    dirent = ovlfs_ino_find_dirent(ino, name, len);

    if ( dirent != (ovlfs_dir_info_t *) NULL )
    {
        if ( dirent->flags & OVLFS_DIRENT_UNLINKED )
        {
            ret_code = ovlfs_remove_dirent(ino, name, len);

            if ( ret_code < 0 )
                return  ret_code;
        }
        else
        {
#if KDEBUG
            printk("OVLFS: ovlfs_ino_add_dirent: directory entry already "
                   "exists\n");
#endif

            return  -EEXIST;
        }
    }

    if ( ino->dir_entries != (list_t) NULL )
    {
            /* Allocate memory for the directory entry */

        dirent = (ovlfs_dir_info_t *) MALLOC(sizeof(ovlfs_dir_info_t));

        if ( dirent == (ovlfs_dir_info_t *) NULL )
        {
#if KDEBUG
            printk("OVLFS: ovlfs_ino_add_dirent: unable to allocate %d bytes\n",
                   sizeof(ovlfs_dir_info_t));
#endif

            ret_code = -ENOMEM;
        }
        else
        {
                /* Copy the name; first allocate space for it.  Don't bother */
                /*  to null-terminate the value since its length is kept.    */

            dirent->name = (char *) MALLOC(len);

            if ( dirent->name == (char *) NULL )
            {
#if KDEBUG
                printk("OVLFS: ovlfs_ino_add_dirent: unable to allocate %d "
                       "bytes\n", len);
#endif
                FREE(dirent);

                ret_code = -ENOMEM;
            }
            else
            {
                memcpy(dirent->name, name, len);
                dirent->len   = len;
                dirent->ino   = ent_ino;
                dirent->flags = flags;

                    /* Add the entry to the list of entries for this inode */

                if ( klist_func(insert_into_list_sorted)(ino->dir_entries,
                                                         dirent) == NULL )
                {
#if KDEBUG
                    printk("OVLFS: ovlfs_ino_add_dirent: error adding entry "
                           "to entry list\n");
#endif

                    FREE(dirent->name);
                    FREE(dirent);

                    ret_code = -ENOMEM;
                }
            }
        }
    }
    else
    {
#if KDEBUG
        printk("OVLFS: ovlfs_ino_add_dirent: unable to create a dirent list\n");
#endif

        ret_code = -ENOMEM;
    }

    return  ret_code;
}



/**
 ** FUNCTION: ovlfs_remove_dirent
 **
 **  PURPOSE: Given an overlay filesystem inode and the name and length of
 **           a directory entry, remove the entry from the inode's list of
 **           directory entries.
 **/

int ovlfs_remove_dirent (ovlfs_inode_t *ino, const char *name, int len)
{
    ovlfs_dir_info_t    tmp;
    int                 ret_code;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_remove_dirent\n");
#endif

#if ! FASTER_AND_MORE_DANGEROUS
    if ( ( ino == (ovlfs_inode_t *) NULL ) || ( name == (char *) NULL ) )
    {
# if KDEBUG
            printk("OVLFS: ovlfs_remove_dirent: null argument given\n");
# endif

            return  -EINVAL;
    }
#endif

    ret_code = 0;

    if ( ino->dir_entries != (list_t) NULL )
    {
        ovlfs_dir_info_t    *ret;

            /* cast to prevent warning about losing const */

        tmp.name = (char *) name;
        tmp.len  = len;

        ret = (ovlfs_dir_info_t *)
              klist_func(remove_element)(ino->dir_entries, &tmp);

            /* The entry was not found? */

        if ( ( ret == (ovlfs_dir_info_t *) NULL ) ||
             ( ret == (ovlfs_dir_info_t *) -1 ) )
            ret_code = -ENOENT;
        else
            free_dir_entry(ret);
    }

    return  ret_code;
}



/**
 ** FUNCTION: ovlfs_unlink_dirent
 **
 **  PURPOSE: Mark the specified directory entry in the given directory
 **           inode as unlinked.  This is the correct method of unlinking
 **           a file since removing it from the directory listing will
 **           allow the file to be found next time the directory listing
 **           is recreated/updated.
 **/

int ovlfs_unlink_dirent (ovlfs_inode_t *ino, const char *name, int len)
{
    ovlfs_dir_info_t    *dirent;
    ovlfs_dir_info_t    tmp;
    int                 ret_code;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_unlink_dirent\n");
#endif

#if ! FASTER_AND_MORE_DANGEROUS
    if ( ( ino == (ovlfs_inode_t *) NULL ) || ( name == (char *) NULL ) )
    {
# if KDEBUG
            printk("OVLFS: ovlfs_unlink_dirent: null argument given\n");
# endif

            return  -EINVAL;
    }
#endif

    ret_code = 0;

    if ( ino->dir_entries != (list_t) NULL )
    {
            /* cast to prevent warning about losing const */

        tmp.name = (char *) name;
        tmp.len  = len;

        dirent = (ovlfs_dir_info_t *)
                 klist_func(search_for_element)(ino->dir_entries, &tmp);

            /* The entry was not found?  Or was it already unlinked? */

        if ( ( dirent == (ovlfs_dir_info_t *) NULL ) ||
             ( dirent == (ovlfs_dir_info_t *) -1 ) ||
             ( dirent->flags & OVLFS_DIRENT_UNLINKED ) )
            ret_code = -ENOENT;
        else
        {
            dirent->flags |= OVLFS_DIRENT_UNLINKED;

                /* Do NOT touch the link count of the directory entry's */
                /*  storage information; it will be updated when the    */
                /*  inode is written out since the caller must update   */
                /*  the VFS's inode's i_nlink count.                    */
        }
    }

    return  ret_code;
}



/**
 ** FUNCTION: ovlfs_ino_find_dirent
 **
 **  PURPOSE: Obtain the inode information for the inode, in the given
 **           directory, whose name matches the name given.
 **/

ovlfs_dir_info_t    *ovlfs_ino_find_dirent (ovlfs_inode_t *ino,
                                            const char *name, int len)
{
    ovlfs_dir_info_t    *result;
    ovlfs_dir_info_t    tmp;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_ino_find_dirent\n");
#endif

    if ( ( ino == (ovlfs_inode_t *) NULL ) ||
         ( ino->dir_entries == (list_t) NULL ) )
        return  (ovlfs_dir_info_t *) NULL;

    tmp.name = (char *) name;
    tmp.len = len;

    result = (ovlfs_dir_info_t *)
             klist_func(search_for_element)(ino->dir_entries, &tmp);

    return  result;
}



/**
 ** FUNCTION: ovlfs_stg_read_ino
 **
 **  PURPOSE: Read the given inode's information from storage.  If it does not
 **           already exist in storage, the inode is added.
 **
 ** NOTES:
 ** - This currently does not do the correct work.  It will need to
 **   read the actual physical storage of ovlfs information to make
 **   certain the inode's information matches the information from
 **   previous mounts.
 **/

ovlfs_inode_t   *ovlfs_stg_read_ino (struct inode *inode)
{
    ovlfs_inode_t   *result;
    struct inode    *o_inode;
    kdev_t          d1;
    _uint           i1;
    kdev_t          d2;
    _uint           i2;
    int             ret;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_stg_read_ino\n");
#endif

    result = ovlfs_get_ino(inode->i_sb, inode->i_ino);

    if ( result == (ovlfs_inode_t *) NULL )
    {
        d1 = NODEV;
        i1 = 0;
        d2 = NODEV;
        i2 = 0;

        ret = ovlfs_resolve_base_inode(inode, &o_inode);

        if ( ret >= 0 )
        {
            d1 = o_inode->i_dev;
            i1 = o_inode->i_ino;

            IPUT(o_inode);
        }

        ret = ovlfs_resolve_ovl_inode(inode, &o_inode);

        if ( ret >= 0 )
        {
            d2 = o_inode->i_dev;
            i2 = o_inode->i_ino;

            IPUT(o_inode);
        }

        result = ovlfs_setup_new_inode(inode->i_sb, NULL, 0, d1, i1, d2, i2);

        if ( result != (ovlfs_inode_t *) NULL )
            result->flags = OVLFS_INODE_READ;
    }

    return  result;
}



/**
 ** FUNCTION: ovlfs_stg_free_super
 **
 **  PURPOSE: Given an ovlfs super block, free any and all memory used for
 **           storage.
 **/

void    ovlfs_stg_free_super (struct super_block *sup_blk)
{
    ovlfs_super_info_t  *s_info;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_stg_free_super\n");
#endif

    if ( sup_blk == (struct super_block *) NULL )
        return;

    s_info = sup_blk->u.generic_sbp;

    if ( s_info != (ovlfs_super_info_t *) NULL )
    {
        if ( s_info->inodes != (list_t) NULL )
        {
            klist_func(foreach_node)(s_info->inodes,
                                     (foreach_func_t) free_ovlfs_inode);
            klist_func(destroy_list)(s_info->inodes);
            s_info->inodes = (list_t) NULL;
        }

        if ( s_info->base_to_ovl != (list_t) NULL )
        {
            klist_func(foreach_node)(s_info->base_to_ovl, Free);
            klist_func(destroy_list)(s_info->base_to_ovl);
            s_info->base_to_ovl = (list_t) NULL;
        }

        if ( s_info->stg_to_ovl != (list_t) NULL )
        {
            klist_func(foreach_node)(s_info->stg_to_ovl, Free);
            klist_func(destroy_list)(s_info->stg_to_ovl);
            s_info->stg_to_ovl = (list_t) NULL;
        }

            /* Don't free the super block information structure */
            /*  here; there is other information that needs to  */
            /*  be handled put ovlfs_put_super().               */
    }
}



/**
 ** FUNCTION: ovlfs_stg_put_super
 **
 **  PURPOSE: Given an ovlfs super block, free any and all memory used for
 **           storage, and save any information to disk that needs to be saved.
 **/

void    ovlfs_stg_put_super (struct super_block *sup_blk)
{
    int ret;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_stg_put_super\n");
#endif

    if ( sup_blk == (struct super_block *) NULL )
        return;

        /* First, save the information to disk */

    ret = ovlfs_stg_write_super(sup_blk);

#if KDEBUG
    if ( ret < 0 )
    {
        printk("OVFLS: ovlfs_stg_put_super: write super returned error %d\n",
               ret);
    }
#endif

    ovlfs_stg_free_super(sup_blk);
}



/**
 ** FUNCTION: ovlfs_stg_read_super
 **
 **  PURPOSE: Read the super block specified from storage.
 **/

int ovlfs_stg_read_super (struct super_block *sup_blk)
{
    ovlfs_inode_t   *root_ino;
    int             ret_code;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_stg_read_super\n");
#endif

#if ! FASTER_AND_MORE_DANGEROUS

    if ( sup_blk == (struct super_block *) NULL )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_stg_read_super: invalid super block given\n");
# endif

        return  -EINVAL;
    }

#endif

        /* First, see if the inode information exists, and read it if so */

    ret_code = ovlfs_read_super(sup_blk);

    if ( ret_code >= 0 )
        return  0;
    else
        if ( ret_code != -ENOENT )
            return  ret_code;

        /* File did not appear to exist; start fresh */

        /* The root directory does not need a name since the name is only */
        /*  used for copying files into the storage fs.                   */

    root_ino = ovlfs_add_inode(sup_blk, NULL, 0);

    if ( ( root_ino == (ovlfs_inode_t *) NULL ) || ( root_ino->ino != 1 ) )
    {
#if KDEBUG
        if ( root_ino == (ovlfs_inode_t *) NULL )
            printk("OVLFS: ovlfs_stg_read_super: unable to allocate needed "
                   "memory\n");
        else
            printk("OVLFS: ovlfs_stg_read_super: failed to obtain inode 1 for "
                   "root directory\n");
#endif

        ovlfs_stg_free_super(sup_blk);

        ret_code = -ENOMEM;
    }
    else
    {
        root_ino->parent_ino = 1;
        root_ino->mode |= S_IFDIR | S_IRUGO | S_IXUGO;
        ret_code = 0;
    }

    return  ret_code;
}


                        /****  MISCELLANEOUS  ****/


/**
 ** FUNCTION: ovlfs_set_attrib
 **
 **  PURPOSE: Set the attributes of the given inode to match the values of the
 **           storage inode given.
 **
 ** NOTES:
 **     - Only the user id, group id, access time, changed time, and
 **       modification time attributes are updated.
 **/

int ovlfs_set_attrib (struct inode *inode, ovlfs_inode_t *ino)
{
    struct iattr    changes;

    if ( ( inode == INODE_NULL ) || ( ino == (ovlfs_inode_t *) NULL ) )
        return  -EINVAL;

        /* If the inode's super block contains the notify_change()    */
        /*  function, that will be called to make the actual changes; */
        /*  otherwise, just make the changes directly to the inode.   */

    if ( ( inode->i_sb != NULL ) && ( inode->i_sb->s_op != NULL ) &&
         ( inode->i_sb->s_op->notify_change != NULL ) )
    {
        memset(&changes, '\0', sizeof(changes));

        if ( inode->i_uid != ino->uid )
        {
            changes.ia_uid = ino->uid;
            changes.ia_valid |= ATTR_UID;
        }
    
        if ( inode->i_gid != ino->gid )
        {
            changes.ia_gid = ino->gid;
            changes.ia_valid |= ATTR_GID;
        }
    
        if ( inode->i_atime != ino->atime )
        {
            changes.ia_atime = ino->atime;
            changes.ia_valid |= ATTR_ATIME;
        }
    
        if ( inode->i_ctime != ino->ctime )
        {
            changes.ia_ctime = ino->ctime;
            changes.ia_valid |= ATTR_CTIME;
        }
    
        if ( inode->i_mtime != ino->mtime )
        {
            changes.ia_mtime = ino->mtime;
            changes.ia_valid |= ATTR_MTIME;
        }
    
        if ( changes.ia_valid != 0 )
        {
            return  inode->i_sb->s_op->notify_change(inode, &changes);
        }
    }
    else
    {
        if ( inode->i_uid != ino->uid )
            inode->i_uid = ino->uid;
    
        if ( inode->i_gid != ino->gid )
            inode->i_gid = ino->gid;
    
        if ( inode->i_atime != ino->atime )
            inode->i_atime = ino->atime;
    
        if ( inode->i_ctime != ino->ctime )
            inode->i_ctime = ino->ctime;
    
        if ( inode->i_mtime != ino->mtime )
            inode->i_mtime = ino->mtime;

            /* Mark the inode as dirty so that it will be written */

        inode->i_dirt = 1;
    }

    return  0;
}



            /**** FILE I/O ****/
/**
 ** NOTES:
 **     - This is NOT efficient, but it is used infrequently.  Also, it will
 **       most likely be replaced in the near future.
 **     - All output must be done from the same process.
 **/


/**
 ** FUNCTION: write_integer
 **/

static int  write_integer (int fd, _uint value, int change_mode)
{
    unsigned long   orig_fs = 0;
    unsigned char   buf[sizeof(_uint)];
    int             cur;
    int             ret;

    if ( change_mode )
    {
        orig_fs = get_fs();
        set_fs(KERNEL_DS);
    }

    for ( cur = 0; cur < sizeof(_uint); cur++ )
    {
            /* Grab the lowest byte and then shift the value by one byte */

        buf[cur] = value & 0xFF;
        value >>= 8;
    }

    ret = write(fd, buf, sizeof(buf));

    if ( change_mode )
        set_fs(orig_fs);

    if ( ret != sizeof(buf) )
    {
#if KDEBUG
        printk("OVLFS: write_integer: write returned %d, expected %d\n", ret,
               sizeof(buf));
#endif

        if ( ret >= 0 )
            return  -EIO;
        else
            return  ret;
    }

    return  ret;
}



/**
 ** FUNCTION: read_integer
 **/

static int  read_integer (int fd, _uint *result, const int change_mode)
{
    unsigned long   orig_fs = 0;
    unsigned char   buf[sizeof(_uint)];
    int             cur;
    int             ret;

    if ( result == (_uint *) NULL )
        return  -EIO;

    if ( change_mode )
    {
        orig_fs = get_fs();
        set_fs(KERNEL_DS);
    }

    ret = read(fd, buf, sizeof(buf));

    if ( ret < 0 )
    {
        DEBUG_PRINT1(("OVLFS: error %d returned from read of storage file\n",
                      ret));
        return  ret;
    }
    else if ( ret != sizeof(buf) )
    {
        DEBUG_PRINT1(("OVLFS: error only %d bytes read from storage file; %d "
                      "bytes expected\n", ret, sizeof(buf)));

        return  -EIO;
    }

    result[0] = 0;

    for ( cur = 0; cur < sizeof(_uint); cur++ )
    {
        result[0] |= ( buf[cur] << ( cur * 8 ) );
    }

    if ( change_mode )
        set_fs(orig_fs);

    return  ret;
}



/**
 ** FUNCTION: write_long
 **/

static int  write_long (int fd, _ulong value, int change_mode)
{
    unsigned long   orig_fs = 0;
    unsigned char   buf[sizeof(_ulong)];
    int             cur;
    int             ret;

    if ( change_mode )
    {
        orig_fs = get_fs();
        set_fs(KERNEL_DS);
    }

    for ( cur = 0; cur < sizeof(_ulong); cur++ )
    {
            /* Grab the lowest byte and then shift the value by one byte */

        buf[cur] = value & 0xFF;
        value >>= 8;
    }

    ret = write(fd, buf, sizeof(buf));

    if ( change_mode )
        set_fs(orig_fs);

    if ( ret != sizeof(buf) )
        return  -EIO;

    return  ret;
}



/**
 ** FUNCTION: read_long
 **/

static int  read_long (int fd, _ulong *result, int change_mode)
{
    unsigned long   orig_fs = 0;
    unsigned char   buf[sizeof(_ulong)];
    int             cur;
    int             ret;

    if ( result == (_ulong *) NULL )
        return  -EIO;

    if ( change_mode )
    {
        orig_fs = get_fs();
        set_fs(KERNEL_DS);
    }

    ret = read(fd, buf, sizeof(buf));

    if ( ret < 0 )
    {
        DEBUG_PRINT1(("OVLFS: error %d returned from read of storage file\n",
                      ret));

        return  ret;
    }
    else if ( ret != sizeof(buf) )
    {
        DEBUG_PRINT1(("OVLFS: error only %d bytes read from storage file; %d "
                      "bytes expected\n", ret, sizeof(buf)));

        return  EIO;
    }

    result[0] = 0;

    for ( cur = 0; cur < sizeof(_ulong); cur++ )
    {
        result[0] |= ( buf[cur] << ( cur * 8 ) );
    }

    if ( change_mode )
        set_fs(orig_fs);

    return  ret;
}



/**
 ** FUNCTION: write_inode_details
 **
 **  PURPOSE: Write the information specific to the type of the inode given.
 **/

static int  write_inode_details (int fd, ovlfs_inode_t *ino, int change_mode)
{
    unsigned long   orig_fs = 0;
    int             ret;

    if ( ino == (ovlfs_inode_t *) NULL )
        return  -EIO;

    if ( change_mode )
    {
        orig_fs = get_fs();
        set_fs(KERNEL_DS);
    }

    if ( S_ISBLK(ino->mode) || S_ISCHR(ino->mode) )
    {
        ret = write_integer(fd, ino->u.int_value, 0);
    }
    else if ( S_ISDIR(ino->mode) )
    {
        int tot_ent;
        int cur_ent;

            /* Write the number of directory entries and then write each */
            /*  entry's information.                                     */

        if ( ino->dir_entries == (list_t) NULL )
            tot_ent = 0;
        else
            tot_ent = klist_func(list_length)(ino->dir_entries);

        ret = write_integer(fd, tot_ent, 0);

        for ( cur_ent = 1; ( cur_ent <= tot_ent ) && ( ret >= 0 ); cur_ent++ )
        {
            ovlfs_dir_info_t    *entry;

            entry = (ovlfs_dir_info_t *)
                    klist_func(nth_element)(ino->dir_entries, cur_ent);

            if ( entry == (ovlfs_dir_info_t *) NULL )
            {
#if KDEBUG
                printk("OVLFS: write_inode_details: null directory entry "
                       "(%d) retrieved from list\n", cur_ent);
#endif

                ret = -EIO;
            }
            else
            {
                ret = write_integer(fd, entry->len, 0);

                if ( ret >= 0 )
                {
                    ret = write(fd, entry->name, entry->len);

                    if ( ret == entry->len )
                    {
                        ret = write_integer(fd, entry->ino, 0);

                        if ( ret >= 0 )
                            ret = write_integer(fd, entry->flags, 0);
                    }
                    else
                        if ( ret >= 0 )
                            ret = -EIO;
                }
            }
        }
    }
    else
        ret = 0;

    if ( change_mode )
        set_fs(orig_fs);

    return  ret;
}



/**
 ** FUNCTION: read_inode_details
 **
 **  PURPOSE: Read the information specific to the type of the inode given.
 **/

static int  read_inode_details (int fd, ovlfs_inode_t *ino, int change_mode)
{
    unsigned long   orig_fs = 0;
    int             tmp_int;
    int             ret;

    if ( ino == (ovlfs_inode_t *) NULL )
        return  -EIO;

    if ( change_mode )
    {
        orig_fs = get_fs();
        set_fs(KERNEL_DS);
    }

    if ( S_ISBLK(ino->mode) || S_ISCHR(ino->mode) )
    {
        ret = read_integer(fd, &tmp_int, 0);

        if ( ret >= 0 )
            ino->u.int_value = tmp_int;
    }
    else if ( S_ISDIR(ino->mode) )
    {
        int tot_ent;
        int cur_ent;

            /* Read the number of directory entries and then read each */
            /*  entry's information.                                   */

        ret = read_integer(fd, &tot_ent, 0);

        for ( cur_ent = 1; ( cur_ent <= tot_ent ) && ( ret >= 0 ); cur_ent++ )
        {
            ovlfs_dir_info_t    tmp;

            ret = read_integer(fd, &tmp_int, 0);
            tmp.len = tmp_int;

            if ( ( ret >= 0 ) && ( tmp.len > 0 ) )
            {
                tmp.name = MALLOC(tmp.len);

                if ( tmp.name == (char *) NULL )
                {
#if KDEBUG
                    printk("OVLFS: read_inode_details: unable to allocate %d"
                           " bytes for name", tmp.len);
#endif

                    ret = -ENOMEM;
                }
                else
                {
                    ret = read(fd, tmp.name, tmp.len);

                    if ( ret != tmp.len )
                        if ( ret >= 0 )
                            ret = -EIO;
                }
            }
            else
                tmp.name = (char *) NULL;

            if ( ret >= 0 )
            {
                ret = read_integer(fd, &tmp_int, 0);
                tmp.ino = tmp_int;

                if ( ret >= 0 )
                {
                    ret = read_integer(fd, &tmp_int, 0);
                    tmp.flags = tmp_int;
                }
            }

            if ( ret >= 0 )
                ret = ovlfs_ino_add_dirent(ino, tmp.name, tmp.len, tmp.ino,
                                           tmp.flags);

            if ( tmp.name != (char *) NULL )
                FREE(tmp.name);
        }
    }
    else
        ret = 0;

    if ( change_mode )
        set_fs(orig_fs);

    return  ret;
}



/**
 ** FUNCTION: write_inode
 **/

static int  write_inode (int fd, ovlfs_inode_t *ino, int change_mode)
{
    unsigned long   orig_fs = 0;
    off_t           pos;
    off_t           result_pos;
    int             ret;

    if ( ino == (ovlfs_inode_t *) NULL )
        return  -EIO;

        /* Remember the current location so that the inode structure's */
        /*  size can be updated later.                                 */

    pos = lseek(fd, 0, 1);

    if ( pos < 0 )
        return  pos;


    if ( change_mode )
    {
        orig_fs = get_fs();
        set_fs(KERNEL_DS);
    }

        /* Write the inode's information */

    ret = write_integer(fd, 0, 0);

    if ( ret >= 0 )
    {
        ret = write_integer(fd, ino->flags, 0);

        if ( ret >= 0 )
        {
            ret = write_integer(fd, ino->uid, 0);

            if ( ret >= 0 )
            {
                ret = write_integer(fd, ino->gid, 0);

                if ( ret >= 0 )
                {
                    ret = write_integer(fd, ino->mode, 0);

                    if ( ret >= 0 )
                    {
                        ret = write_integer(fd, ino->size, 0);

                        if ( ret >= 0 )
                        {
                            ret = write_integer(fd, ino->atime, 0);

                            if ( ret >= 0 )
                            {
                                ret = write_integer(fd, ino->mtime, 0);

                                if ( ret >= 0 )
                                {
                                    ret = write_integer(fd, ino->ctime, 0);

                                    if ( ret >= 0 )
                                    {
                                        ret = write_integer(fd, ino->nlink, 0);

                                        if ( ret >= 0 )
                                            ret = write_long(fd, ino->blksize,
                                                             0);

                                        if ( ret >= 0 )
                                            ret = write_long(fd, ino->blocks,
                                                             0);

                                        if ( ret >= 0 )
                                            ret = write_long(fd, ino->version,
                                                             0);

                                        if ( ret >= 0 )
                                            ret = write_long(fd, ino->nrpages,
                                                             0);

                                        if ( ret >= 0 )
                                            ret = write_inode_details(fd,ino,0);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }

        /* If all is well, seek back to the start of this inode's info and */
        /*  update the size.                                               */

    if ( ret >= 0 )
    {
        result_pos = lseek(fd, 0, 1);

        if ( result_pos < 0 )
            ret = result_pos;
        else
        {
            ret = lseek(fd, pos, 1);

            if ( ret >= 0 )
            {
                ret = write_integer(fd, ( result_pos - pos ), 0);

                if ( ret >= 0 )
                    ret = lseek(fd, result_pos, 0);
            }
        }
    }

    if ( change_mode )
        set_fs(orig_fs);

    return  ret;
}



/**
 ** FUNCTION: read_inode
 **/

static int  read_inode (int fd, ovlfs_inode_t *ino, int change_mode)
{
    unsigned long   orig_fs = 0;
    int             tot_size;
    int             tmp_int;
    long            tmp_long;
    int             ret;

    if ( ino == (ovlfs_inode_t *) NULL )
        return  -EIO;

    if ( change_mode )
    {
        orig_fs = get_fs();
        set_fs(KERNEL_DS);
    }

        /* Read the inode's information. First is the size used by this  */
        /*  inode; we don't need this, but it is useful when looking for */
        /*  one particular inode instead of trying to read all of them.  */

    ret = read_integer(fd, &tot_size, 0);

    if ( ret >= 0 )
    {
        ret = read_integer(fd, &tmp_int, 0);

        if ( ret >= 0 )
        {
            ino->flags = tmp_int;

            ret = read_integer(fd, &tmp_int, 0);

            if ( ret >= 0 )
            {
                ino->uid = tmp_int;

                ret = read_integer(fd, &tmp_int, 0);

                if ( ret >= 0 )
                {
                    ino->gid = tmp_int;

                    ret = read_integer(fd, &tmp_int, 0);

                    if ( ret >= 0 )
                    {
                        ino->mode = tmp_int;

                        ret = read_integer(fd, &tmp_int, 0);

                        if ( ret >= 0 )
                        {
                            ino->size = tmp_int;

                            ret = read_integer(fd, &tmp_int, 0);

                            if ( ret >= 0 )
                            {
                                ino->atime = tmp_int;

                                ret = read_integer(fd, &tmp_int, 0);

                                if ( ret >= 0 )
                                {
                                    ino->mtime = tmp_int;

                                    ret = read_integer(fd, &tmp_int, 0);

                                    if ( ret >= 0 )
                                    {
                                        ino->ctime = tmp_int;

                                        ret = read_integer(fd, &tmp_int, 0);

                                        if ( ret >= 0 )
                                        {
                                            ino->nlink = tmp_int;

                                            ret = read_long(fd, &tmp_long, 0);
                                        }

                                        if ( ret >= 0 )
                                        {
                                            ino->blksize = tmp_long;

                                            ret = read_long(fd, &tmp_long, 0);
                                        }

                                        if ( ret >= 0 )
                                        {
                                            ino->blocks = tmp_long;

                                            ret = read_long(fd, &tmp_long, 0);
                                        }

                                        if ( ret >= 0 )
                                        {
                                            ino->version = tmp_long;

                                            ret = read_long(fd, &tmp_long, 0);
                                        }

                                        if ( ret >= 0 )
                                        {
                                            ino->nrpages = tmp_long;

                                            ret = read_inode_details(fd,ino,0);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    if ( change_mode )
        set_fs(orig_fs);

    return  ret;
}



#if STORE_MAPPINGS

/**
 ** FUNCTION: ovlfs_write_mappings
 **/

static int  ovlfs_write_mappings (int fd, list_t map_list, int change_mode,
                                  int skip)
{
    int                 tot_ent;
    int                 cur_ent;
    ovlfs_ino_map_t *one_map;
    unsigned long       orig_fs = 0;
    int                 ret;

    if ( change_mode )
    {
        orig_fs = get_fs();
        set_fs(KERNEL_DS);
    }

    if ( ( map_list == (list_t) NULL ) || ( skip ) )
        tot_ent = 0;
    else
        tot_ent = klist_func(list_length)(map_list);

    ret = write_integer(fd, tot_ent, 0);

    cur_ent = 1;

    while ( ( cur_ent <= tot_ent ) && ( ret >= 0 ) )
    {
        one_map = (ovlfs_ino_map_t *)
                  klist_func(nth_element)(map_list, cur_ent);

        if ( one_map == (ovlfs_ino_map_t *) NULL )
        {
#if KDEBUG
            printk("OVLFS: ovlfs_write_mappings: null element (%d) in list\n",
                   cur_ent);
#endif

            ret = -EIO;
        }
        else
        {
            ret = write_integer(fd, one_map->dev, 0);

            if ( ret >= 0 )
            {
                ret = write_integer(fd, one_map->ino, 0);

                if ( ret >= 0 )
                    ret = write_integer(fd, one_map->ovl_ino, 0);
            }
        }

        cur_ent++;
    }

    if ( change_mode )
        set_fs(orig_fs);

    return  ret;
}



/**
 ** FUNCTION: ovlfs_read_mappings
 **/

static int  ovlfs_read_mappings (int fd, list_t *map_list, int change_mode,
                                 int skip)
{
    int                 tot_ent;
    int                 cur_ent;
    ovlfs_ino_map_t     *one_map;
    unsigned long       orig_fs = 0;
    int                 ret;

    if ( map_list == (list_t *) NULL )
        return  -EIO;

    if ( change_mode )
    {
        orig_fs = get_fs();
        set_fs(KERNEL_DS);
    }

    ret = read_integer(fd, &tot_ent, 0);

        /* Only work on the list if there is at least one entry */

    if ( ( ret >= 0 ) && ( tot_ent > 0 ) )
    {
        if ( skip )
        {
            _uint tot_size;

            tot_size = ( sizeof(_uint) * 3 * tot_ent ) ;

                /* We need to skip the mapping entries; just seek */

            if ( lseek(fd, tot_size, 1) == -1 )    /* 1 => SEEK_CUR */
                ret = -EIO;
        }
        else
        {
            if ( map_list[0] == (list_t) NULL )
            {
                map_list[0] = create_ino_map_list();
    
                if ( map_list[0] == (list_t) NULL )
                {
                    DEBUG_PRINT1(("OVLFS: unable to create an inode mapping "
                                  "list\n"));

                    ret = -ENOMEM;
                }
            }
    
            cur_ent = 1;
    
            while ( ( cur_ent <= tot_ent ) && ( ret >= 0 ) )
            {
                one_map = create_ino_map();
    
                if ( one_map == (ovlfs_ino_map_t *) NULL )
                {
                    DEBUG_PRINT1(("OVLFS: unable to allocate memory for an "
                                  "inode mapping\n"));

                    ret = -ENOMEM;
                }
                else
                {
                    int ovl_ino;
                    int dev;
                    int ino;
    
                    ret = read_integer(fd, &dev, 0);
    
                    if ( ret >= 0 )
                    {
                        ret = read_integer(fd, &ino, 0);
    
                        if ( ret >= 0 )
                        {
                            ret = read_integer(fd, &ovl_ino, 0);
    
                            if ( ret >= 0 )
                            {
                                one_map->dev     = dev;
                                one_map->ino     = ino;
                                one_map->ovl_ino = ovl_ino;
    
                                if ( klist_func(insert_into_list_sorted)
                                               (map_list[0], one_map) == NULL )
                                {
                                    DEBUG_PRINT1(("OVLFS: error attempting to "
                                                  "insert mapping into mapping"
                                                  " list\n"));

                                    ret = -ENOMEM;
                                }
                            }
                        }
                    }
    
                    if ( ret < 0 )
                        FREE(one_map);
                }
    
                cur_ent++;
            }
        }
    }

    if ( change_mode )
        set_fs(orig_fs);

    return  ret;
}

#endif



/**
 ** FUNCTION: open_stg_file
 **
 **  PURPOSE: Given the super block of an overlay filesystem and the mode to
 **           open its storage file with, open the storage file.
 **/

static int	open_stg_file (struct super_block *sb, int mode, int perm)
{
    ovlfs_super_info_t  *s_info;
    int                 fd;

    s_info = sb->u.generic_sbp;

        /* If there is no place to store the information, just throw */
        /*  it to the old bit bucket.                                */

    if ( s_info->stg_file_inode == INODE_NULL )
    {
        if ( s_info->stg_file == (char *) NULL )
            fd = 0;
        else
            fd = open(s_info->stg_file, mode, perm);
    }
    else
        fd = open_inode(s_info->stg_file_inode, mode);

    return fd;
}



/**
 ** FUNCTION: ovlfs_stg_write_super
 **/

#define MAGIC_WORD      "Ovl2"
#define MAGIC_WORD_LEN  4

int  ovlfs_stg_write_super (struct super_block *sup_blk)
{
    ovlfs_super_info_t  *s_info;
    char                magic_buf[] = MAGIC_WORD;
    int                 fd;
    int                 ret;
    unsigned long       orig_fs;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_stg_write_super\n");
#endif

    if ( ( sup_blk == (struct super_block *) NULL ) ||
         ( sup_blk->u.generic_sbp == NULL ) )
        return  -EINVAL;

    s_info = sup_blk->u.generic_sbp;

    orig_fs = get_fs();
    set_fs(KERNEL_DS);

    fd = open_stg_file(sup_blk, O_WRONLY | O_TRUNC | O_CREAT, 0644);

    if ( fd < 0 )
    {
#if KDEBUG
        printk("OVLFS: ovlfs_stg_write_super: open storage file %s returned "
	       "%d\n", s_info->stg_file, fd);
#endif

        ret = fd;
    }
    else
    {
        ret = write_integer(fd, sup_blk->s_magic, 0);

        if ( ret >= 0 )
        {
            ret = write(fd, magic_buf, MAGIC_WORD_LEN);

            if ( ret == MAGIC_WORD_LEN )
            {
                int tot_ino;
                int cur_ino;

                cur_ino = 1;
                tot_ino = klist_func(list_length)(s_info->inodes);

                ret = write_integer(fd, tot_ino, 0);

                while ( ( cur_ino <= tot_ino ) && ( ret >= 0 ) )
                {
                    ovlfs_inode_t   *ino;

                    ino = (ovlfs_inode_t *)
                          klist_func(nth_element)(s_info->inodes, cur_ino);

                    ret = write_inode(fd, ino, 0);

                    cur_ino++;
                }

#if STORE_MAPPINGS
                if ( ret >= 0 )
                {
                    int skip;

                    skip = ! ( ovlfs_sb_opt(sup_blk, storage_opts) &
                               OVLFS_STORE_BASE_MAP );

                    ret = ovlfs_write_mappings(fd, s_info->base_to_ovl, 0,
                                               skip);
 
                    if ( ret >= 0 )
                    {
                        skip = ! ( ovlfs_sb_opt(sup_blk, storage_opts) &
                                   OVLFS_STORE_STG_MAP );

                        ret = ovlfs_write_mappings(fd, s_info->stg_to_ovl, 0,
                                                   skip);
                    }
                }
#endif
            }
            else
            {
#if KDEBUG
                printk("OVLFS: write magic buf returned %d\n", ret);
#endif

                if ( ret >= 0 )
                    ret = -EIO;
            }
        }

        close(fd);
    }

    set_fs(orig_fs);

    return  ret;
}



/**
 ** FUNCTION: is_empty
 **
 **  PURPOSE: Determine if the file, whose descriptor is given, is empty.
 **/

static int	is_empty (int fd)
{
	int	ret;
	int	rc;
	off_t	pos;


		/* Check the current position; it must be 0 if the file is  */
		/*  empty.  This function must return with the file pointer */
		/*  referring to the same place in the file as when called. */

	pos = lseek(fd, 0, 1);		/* 1 = SEEK_CUR */

	if ( pos != 0 )
	{
		ret = 0;	/* not empty */
	}
	else
	{
			/* Seek to the end and see if that is offset 0 */

		pos = lseek(fd, 0, 2);		/* 2 = SEEK_END */

		if ( pos != 0 )
			ret = 0;	/* not empty */
		else
			ret = 1;	/* empty */


			/* Now return to the start of the file */

		if ( ( rc = lseek(fd, 0, 0) ) < 0 )	/* 0 = SEEK_SET */
		{
#if KDEBUG
			printk("OVLFS: is_empty(): lseek(%d, 0, 0) returned "
			       "err %d\n", fd, rc);

#endif
		}
	}

	return	ret;
}



/**
 ** FUNCTION: ovlfs_read_super
 **/

static int  ovlfs_read_super (struct super_block *sup_blk)
{
    ovlfs_super_info_t  *s_info;
    char                magic_buf[MAGIC_WORD_LEN];
    int                 fd;
    int                 tmp_int;
    int                 ret;
    unsigned long       orig_fs;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_read_super\n");
#endif

    if ( ( sup_blk == (struct super_block *) NULL ) ||
         ( sup_blk->u.generic_sbp == NULL ) )
        return  -EINVAL;

    s_info = sup_blk->u.generic_sbp;

    orig_fs = get_fs();
    set_fs(KERNEL_DS);

    fd = open_stg_file(sup_blk, O_RDONLY, 0);

    if ( fd < 0 )
    {
#if KDEBUG
        printk("OVLFS: ovlfs_read_super: open storage file %s returned %d\n",
               s_info->stg_file, fd);
#endif

        ret = fd;
    }
    else if ( is_empty(fd) )  /* It is not an error if the file is empty */
    {
        ret = 0;
        close(fd);
    }
    else
    {
        ret = read_integer(fd, &tmp_int, 0);

        if ( ( ret == sizeof(_uint) ) && ( tmp_int == sup_blk->s_magic ) )
        {
            ret = read(fd, magic_buf, MAGIC_WORD_LEN);

            if ( ret == MAGIC_WORD_LEN )
            {
                if ( memcmp(MAGIC_WORD, magic_buf, MAGIC_WORD_LEN) != 0 )
                {
                    DEBUG_PRINT1(("OVLFS: ovlfs_read_super: magic word does not"
                                  " match; expected '%.*s' and read '%.*s'\n",
                                  MAGIC_WORD_LEN, MAGIC_WORD, ret, magic_buf));

                    ret = -EIO;
                }
            }
            else
            {
                DEBUG_PRINT1(("OVLFS: read magic word returned %d; need "
                              "%d bytes\n", ret, MAGIC_WORD_LEN));

                if ( ret >= 0 )
                    ret = -EIO;
            }

            if ( ret >= 0 )
            {
                int tot_ino;
                int cur_ino;

                ret = read_integer(fd, &tot_ino, 0);
                cur_ino = 1;

                while ( ( cur_ino <= tot_ino ) && ( ret >= 0 ) )
                {
                    ovlfs_inode_t   *ino;

                    ino = ovlfs_add_inode(sup_blk, NULL, 0);

                    if ( ino == (ovlfs_inode_t *) NULL )
                    {
                        DEBUG_PRINT1(("OVLFS: ovlfs_read_super: unable to add "
                                      "a new inode to the super block at "
                                      "0x%x\n", (int) sup_blk));

                        ret = -ENOMEM;
                    }
                    else
                        ret = read_inode(fd, ino, 0);

                    cur_ino++;
                }

#if STORE_MAPPINGS
                if ( ret >= 0 )
                {
                    int skip;

                    skip = ! ( ovlfs_sb_opt(sup_blk, storage_opts) &
                               OVLFS_STORE_BASE_MAP );

                    ret = ovlfs_read_mappings(fd, &(s_info->base_to_ovl), 0,
                                              skip);
 
                    if ( ret >= 0 )
                    {
                        skip = ! ( ovlfs_sb_opt(sup_blk, storage_opts) &
                                   OVLFS_STORE_STG_MAP );

                        ret = ovlfs_read_mappings(fd, &(s_info->stg_to_ovl), 0,
                                                  skip);

                        if ( ret < 0 )
                            DEBUG_PRINT1(("OVLFS: ovlfs_read_super: error "
                                          "reading stg fs map from the storage"
                                          " file\n"));
                    }
                    else
                        DEBUG_PRINT1(("OVLFS: ovlfs_read_super: error reading "
                                      "base fs map from the storage file\n"));
                }
                else
                    DEBUG_PRINT1(("OVLFS: ovlfs_read_super: error reading "
                                  "inode information from the storage file\n"));
#endif
            }
            else
            {
                DEBUG_PRINT1(("OVLFS: read magic buf returned %d\n", ret));

                if ( ret >= 0 )
                    ret = -EIO;
            }
        }
        else
        {
            DEBUG_PRINT1(("OVLFS: ovlfs_read_super: magic number in storage "
                          "file, 0x%x, does not match super block's, 0x%x\n",
                          tmp_int, (int) sup_blk->s_magic));

            ret = -EIO;
        }

        close(fd);
    }

    set_fs(orig_fs);

    return  ret;
}
