/******************************************************************************
 ******************************************************************************
 **
 ** COPYRIGHT (C) 1998 By Arthur Naseef
 **
 ** This file is covered under the GNU General Public License Version 2.  For
 **  more information, see the file COPYING.
 **
 **
 ** FILE: ovl_ino.c
 **
 ** DESCRIPTION: This file contains the inode operations for the overlay
 **              (i.e. psuedo) filesystem.
 **
 ** REVISION HISTORY:
 **
 ** DATE	AUTHOR		DESCRIPTION
 ** ==========	===============	==============================================
 ** 09/27/1997	art            	Added to source code control.
 ** 02/19/1998	ARTHUR N.	Moved ovlfs_get_super() into ovl_kern.c.
 ** 02/21/1998	ARTHUR N.	Added locking of inodes that are retrieved
 **				 and used here.
 ** 02/24/1998	ARTHUR N.	Do not always lock the inodes; only wait
 **				 until they are no longer locked.
 ** 02/24/1998	ARTHUR N.	Added support for hiding the magic directory
 **				 names from the directory listing.
 ** 02/26/1998	ARTHUR N.	Removed old unused code.
 ** 02/26/1998	ARTHUR N.	Updated four more fields in the retained inode
 **				 information.
 ** 02/26/1998	ARTHUR N.	Lock inodes when their contents may change,
 **				 as done in the VFS.
 ** 02/26/1998	ARTHUR N.	Added the inode operations, link(), bmap(),
 **				 permission(), follow_link(), and readpage().
 ** 02/26/1998	ARTHUR N.	Update ovlfs_make_hierarchy() so that new
 **				 storage directories are referenced by their
 **				 associated storage information.
 ** 02/28/1998	ARTHUR N.	Mark the superblock as dirty after any changes
 **				 that affect the storage.
 ** 02/28/1998	ARTHUR N.	Do not return an error from unlink() if the
 **				 file does not exist in the storage fs.
 ** 03/01/1998	ARTHUR N.	Added support for the magic directory name
 **				 options.
 ** 03/01/1998	ARTHUR N.	Only define ck_magic_name if USE_MAGIC is set.
 ** 03/01/1998	ARTHUR N.	Reworked ovlfs_rename() slightly.  Set the
 **				 mtime and ctime in ovlfs_rename().
 ** 03/01/1998	ARTHUR N.	Don't return empty, unlinked directories from
 **				 resolve_inode().
 ** 03/01/1998	ARTHUR N.	Update the link counts of directories and
 **				 files affected by rmdir() and unlink() calls.
 **				 Also, lock the storage dir before calling
 **				 its rmdir() operation.
 ** 03/01/1998	ARTHUR N.	In ovlfs_rmdir(), perform the lookup of the
 **				 pseudo fs' inode for the directory to be
 **				 removed before calling the stg fs' rmdir().
 ** 03/02/1998	ARTHUR N.	Return -EXDEV from rename() if the file in
 **				 the storage fs is crossing a mount point. 
 ** 03/03/1998	ARTHUR N.	If OVLFS_ALLOW_XDEV_RENAME is set, remember
 **				 the rename of a file, that would cross a mount
 **				 point in the storage fs, by marking its new
 **				 location within the overlay fs' storage.
 **				 (note that the storage fs will not look as
 **				 expected; the file will stay where it was).
 ** 03/03/1998	ARTHUR N.	Added support for the noxmnt option.
 ** 03/09/1998	ARTHUR N.	Added the copyright notice.
 ** 03/09/1998	ARTHUR N.	Update the ctime and mtime attributes of
 **				 directories on the rmdir() and unlink() ops.
 ** 03/10/1998	ARTHUR N.	Move the update of the inode times in the
 **				 unlink() op so that they are updated even if
 **				 the file does not exist in the storage fs.
 ** 03/10/1998	ARTHUR N.	Improved the comments.
 **
 ******************************************************************************
 ******************************************************************************
 **/

#ident "@(#) ovl_ino.c 1.17"

#ifdef MODVERSIONS
# include <linux/modversions.h>
# ifndef __GENKSYMS__
#  include "ovlfs.ver"
# endif
#endif

#include <linux/stddef.h>

#include <linux/types.h>
#include <linux/sched.h>
#include <linux/fcntl.h>
#include <linux/fs.h>
#include <linux/locks.h>
#include <linux/malloc.h>
#include <asm/statfs.h>

#include "ovl_fs.h"


/**
 ** FUNCTION: ovlfs_use_inode
 **
 **  PURPOSE: This function is called when an inode is going to be used and
 **           it is important to make certain that the inode is not locked.
 **/

static inline int   ovlfs_use_inode (struct inode *inode)
{
    if ( inode->i_lock )
        return  ovlfs_wait_on_inode(inode);

    return  0;
}



/**
 ** FUNCTION: ovlfs_release_inode
 **
 **  PURPOSE: Undo the work done by ovlfs_use_inode() once the inode is
 **           not being used anymore.  This currently doesn't do anything.
 **/

static inline void  ovlfs_release_inode (struct inode *inode)
{
}



/**
 ** FUNCTION: ovlfs_get_ref_inode
 **
 **  PURPOSE: Obtain the inode to which the given ovlfs inode refers in the
 **           filesystem indicated by the argument, which.
 **/

static struct inode *ovlfs_get_ref_inode (struct inode *inode, char which)
{
    ovlfs_inode_t       *ino;
    struct inode        *result;
    kdev_t              s_dev = NODEV;
    _ulong              inum = 0;
    struct super_block  *sb;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_get_ref_inode: looking for inode in fs "
           "'%c'\n", which);
#endif

    if ( ( inode == INODE_NULL ) || ( ! is_ovlfs_inode(inode) ) )
    {
#if KDEBUG
        printk("OVLFS: ovlfs_get_ref_inode: invalid inode given at 0x%x\n",
               (int) inode);
#endif

        return  INODE_NULL;
    }

    ino = ovlfs_get_ino(inode->i_sb, inode->i_ino);

    if ( ino == (ovlfs_inode_t *) NULL )
    {
#if KDEBUG
        printk("OVLFS: ovlfs_get_ref_inode: unable to find inode %ld in "
               "storage\n", inode->i_ino);
#endif

        result = INODE_NULL;
    }
    else
    {
        switch ( which )
        {
            case 'b':
                s_dev = ino->base_dev;
                inum  = ino->base_ino;
                break;

            case 'o':
            case 's':
                s_dev = ino->stg_dev;
                inum  = ino->stg_ino;
                break;

            default:
#if KDEBUG
                printk("OVLFS: ovlfs_get_ref_inode: invalid value for which: "
                       "'%c'\n", which);
#endif
                s_dev = NODEV;
                break;
        }

            /* Catch inode references to our own filesystem to prevent */
            /*  infinite recursions.                                   */

        if ( s_dev == inode->i_dev )
        {
#if KDEBUG > 1
            printk("OVLFS: ovlfs_get_ref_inode: referenced inode is in our "
                   "filesystem!\n");
#endif

            return INODE_NULL;
        }


            /* If a device and inode number were found, retrieve the inode */

        if ( s_dev != NODEV )
        {
                /* Obtain the super block for the device */

            sb = ovlfs_get_super(s_dev);

            if ( sb != (struct super_block *) NULL )
            {
                result = __iget(sb, inum, ovlfs_sb_xmnt(inode->i_sb));

                    /* As a sanity check, make sure the inode is not an */
                    /*  unlinked directory.                             */

                if ( ( result != INODE_NULL ) && ( S_ISDIR(result->i_mode) ) &&
                     ( result->i_nlink == 0 ) )
                {
                    IPUT(result);    /* This may cause meaningless err msgs */
                    result = INODE_NULL;
                }
            }
            else
            {
#if KDEBUG
                printk("OVLFS: ovlfs_get_ref_inode: unable to obtain super "
                       "block for device %d\n", (int) s_dev);
#endif
                result = INODE_NULL;
            }
        }
        else
            result = INODE_NULL;
    }

    return  result;
}



/**
 ** FUNCTION: ovlfs_resolve_inode
 **
 **  PURPOSE: Determine where to obtain the real inode for the given inode.
 **
 ** NOTES:
 **     - If the inode retrieved belongs to the same filesystem as the given
 **       inode (i.e. this overlay fs), then it is not returned; this is
 **       important to prevent infinite recursions, and is logically correct
 **       since the purpose of this function is to obtain an inode referring
 **       to another filesystem.
 **
 **     - The inodes retrieved for other filesystems are not locked here even
 **       though they are being interrogated because the information being
 **       read should not change, and each element is only integer size, so it
 **       really possible to obtain incorrect information due to another
 **       process updating it at the same time; locking the inodes would not
 **       provide any real benefit in this case.
 **/

extern int  ovlfs_resolve_inode (struct inode *inode, struct inode **result,
                                 char ovl_ind)
{
    ovlfs_inode_info_t  *i_info;
    int                 ret_code;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_resolve_inode(%ld, '%c')\n", inode->i_ino,
           ovl_ind);
#endif

    if ( ( inode == INODE_NULL ) || ( result == (struct inode **) NULL ) ||
         ( inode->u.generic_ip == NULL ) )
    {
#if KDEBUG
        printk("OVLFS: ovlfs_resolve_inode: argument given is not valid"
           " (0x%x)\n", (int) inode);
#endif

        return  -ENOENT;
    }


    i_info = (ovlfs_inode_info_t *) inode->u.generic_ip;

    switch ( ovl_ind )
    {
        case 'b':
            /* Base inode */
#if STORE_REF_INODES
            if ( i_info->base_inode == INODE_NULL )
            {
                    /* Try to obtain the base inode in case this is the   */
                    /*  first time to attempt to access it for this inode */

                i_info->base_inode = ovlfs_get_ref_inode(inode, 'b');
            }

            if ( i_info->base_inode == INODE_NULL )
            {
# if KDEBUG > 1
                printk("OVLFS: ovlfs_resolve_inode inode %ld at 0x%x: base"
                       " inode is null\n", inode->i_ino, inode);
# endif

                return -ENOENT;
            }

                /* Increment the inode's use count and return it */

            IMARK(i_info->base_inode);
            result[0] = i_info->base_inode;
            ret_code = 0;
#else
            if ( i_info->base_name == (char *) NULL )
            {
# if KDEBUG > 1
                printk("OVLFS: ovlfs_resolve_inode inode %d at 0x%x: base"
                       " name is null\n", inode->i_ino, inode);
# endif

                return -ENOENT;
            }

            ret_code = get_inode(i_info->base_name, result);
#endif
            break;


        case 'o':
                /* Overlay inode */
#if STORE_REF_INODES
            if ( i_info->overlay_inode == INODE_NULL )
            {
                    /* Try to obtain the base inode in case this is the   */
                    /*  first time to attempt to access it for this inode */

                i_info->overlay_inode = ovlfs_get_ref_inode(inode, 'o');
            }

            if ( i_info->overlay_inode == INODE_NULL )
            {
# if KDEBUG > 1
                printk("OVLFS: ovlfs_resolve_inode inode %ld at 0x%x: overlay"
                       " inode is null\n", inode->i_ino, inode);
# endif

                return -ENOENT;
            }

                /* Increment the inode's use count and return it */

            IMARK(i_info->overlay_inode);
            result[0] = i_info->overlay_inode;
            ret_code = 0;
#else
            if ( i_info->overlay_name == (char *) NULL )
            {
# if KDEBUG > 1
                printk("OVLFS: ovlfs_resolve_inode inode %d at 0x%x: overlay"
                       " name is null\n", inode->i_ino, inode);
# endif

                return -ENOENT;
            }

            ret_code = get_inode(i_info->base_name, result);
#endif
            break;

        default:
#if KDEBUG
            printk("OVLFS: ovlfs_resolve_inode: invalid indicator, %c.\n",
                   ovl_ind);
#endif

            ret_code = -EINVAL;
            break;
    }

        /* As a sanity check, make sure to never return an inode from our */
        /*  own filesystem.  Don't just check for an ovlfs inode since    */
        /*  it is ok to overlay an overlay fs with another one.           */

    if ( ( ret_code >= 0 ) && ( result[0] != INODE_NULL ) &&
         ( result[0]->i_dev == inode->i_dev ) )
    {
        IPUT(result[0]);
        result[0] = INODE_NULL;
        ret_code = -ENOENT;
    }
    else
    {
            /* As another sanity check, in case someone modifies the real */
            /*  filesystem out from under us, make sure that directory    */
            /*  inodes are not empty and unlinked.                        */
    
        if ( ( S_ISDIR(result[0]->i_mode) ) && ( result[0]->i_nlink == 0 ) )
        {
            IPUT(result[0]);
            result[0] = INODE_NULL;
            ret_code = -ENOENT;
        }
    }


#if KDEBUG_CALLS
    printk(KDEBUG_RET_PREFIX "ovlfs_resolve_inode(%ld) = %d\n", inode->i_ino,
           ret_code);
#endif

    return  ret_code;
}



/**
 ** FUNCTION: assign_stg_inode
 **
 **  PURPOSE: Assign the storage fs inode to its associated pseudo-fs inode.
 **/

static inline int   assign_stg_inode (struct inode *ovl_dir,
                                      struct inode *stg_dir)
{
    ovlfs_inode_t   *ino;
    int             ret;

    if ( ! is_ovlfs_inode(ovl_dir) )
        return  -ENOENT;


        /* Find the storage information for the pseudo-fs' inode */

    ino = ovlfs_get_ino(ovl_dir->i_sb, ovl_dir->i_ino);

    if ( ino == (ovlfs_inode_t *) NULL )
        ret = -ENOENT;
    else
        ret = ovlfs_inode_set_stg(ovl_dir->i_sb, ino, stg_dir->i_dev,
                                  stg_dir->i_ino);

    return  ret;
}



/**
 ** FUNCTION: ovlfs_make_hierarchy
 **
 **  PURPOSE: Create all of the ancestor directories of the given directory
 **           in the storage filesystem and return a reference to the storage
 **           inode referring to the given directory inode.
 **/

#define MAX_DEPTH   64

int ovlfs_make_hierarchy (struct inode *dir, struct inode **result, int depth)
{
    ovlfs_inode_t   *ino;
    struct inode    *inode;
    int             ret_code;

    if ( depth > MAX_DEPTH )
        return  -EDEADLK;

        /* Make certain the inode given is an ovlfs inode and that it is */
        /*  directory inode also.                                        */

    if ( ( dir == INODE_NULL ) || ( ! is_ovlfs_inode(dir) ) )
        return  -EINVAL;
    else if ( ( ! S_ISDIR(dir->i_mode) ) || ( ! dir->i_op->mkdir ) )
        return -ENOTDIR;

    ino = ovlfs_get_ino(dir->i_sb, dir->i_ino);


        /* If the inode was not found, or the inode's name is not set, */
        /*  return an error.  Also, if the inode's parent inode is     */
        /*  the directory itself, return the error; otherwise, an      */
        /*  infinite recursion will occur.                             */

    if ( ( ino == (ovlfs_inode_t *) NULL ) || ( ino->name == (char *) NULL ) ||
         ( ino->len <= 0 ) )
        return  -ENOENT;
    else
        if ( ino->parent_ino == dir->i_ino )
            return  -EDEADLK;


        /* Get this directory's parent directory */

    inode = __iget(dir->i_sb, ino->parent_ino, ovlfs_sb_xmnt(dir->i_sb));

    if ( inode == (struct inode *) NULL )
        ret_code = -ENOENT;
    else
    {
        struct inode    *o_inode;

            /* Lock the inode received; return any errors */

        ret_code = ovlfs_use_inode(inode);


            /* See if the parent has a storage inode */

        if ( ret_code >= 0 )
        {
            ret_code = ovlfs_resolve_stg_inode(inode, &o_inode);

            if ( ret_code < 0 )
            {
                    /* The parent does not appear to have a storage inode, */
                    /*  try to create it.  This is a recursive call.       */

                ret_code = ovlfs_make_hierarchy(inode, &o_inode, depth + 1);
            }
        }

        if ( ret_code >= 0 )
        {
                /* Lock the inode just obtained */

            ret_code = ovlfs_use_inode(o_inode);

            if ( ret_code < 0 )
                IPUT(o_inode);
        }

        if ( ret_code >= 0 )
        {

            if ( ( o_inode->i_op != NULL ) &&
                 ( o_inode->i_op->mkdir != NULL ) &&
                 ( o_inode->i_op->lookup != NULL ) )
            {
                IMARK(o_inode);
                down(&(o_inode->i_sem));
                ret_code = o_inode->i_op->mkdir(o_inode, ino->name, ino->len,
                                                dir->i_mode);
                up(&(o_inode->i_sem));

                if ( ret_code >= 0 )
                {
                        /* Get the inode */
                    IMARK(o_inode);
                    down(&(o_inode->i_sem));
                    ret_code = o_inode->i_op->lookup(o_inode, ino->name,
                                                     ino->len, result);
                    up(&(o_inode->i_sem));

                        /* If it was retrieved, set its attributes to match */
                        /*  its parent directory's attributes.              */

                    if ( ret_code >= 0 )
                    {
                            /* Lock the resulting inode before updating it */

                        ret_code = ovlfs_use_inode(result[0]);

                        if ( ret_code >= 0 )
                        {
                            
                            ovlfs_set_attrib(result[0], ino);

                                /* Set the new dir as the stg its psuedo-fs */
                                /*  inode.                                  */

                            ret_code = assign_stg_inode(dir, result[0]);

                            ovlfs_release_inode(result[0]);
                        }
                        else
                        {
                            IPUT(result[0]);
                            result[0] = INODE_NULL;
                        }
                    }
                }
            }
            else
            {
#if KDEBUG
                printk("OVLFS: ovlfs_make_hierarchy: storage inode is not a "
                       "directory.\n");
#endif

                ret_code = -ENOTDIR;
            }

            ovlfs_release_inode(o_inode);
            IPUT(o_inode);
        }

        ovlfs_release_inode(inode);
        IPUT(inode);
    }

    return  ret_code;
}



/**
 ** FUNCTION: ovlfs_add_dirent_op
 **
 **  PURPOSE: Given the inode of a directory, the name of a node to create in
 **           the directory, and a callback function to perform the actual
 **           creation in the storage filesystem, do the following:
 **
 **               1. Make certain that the directory hierarchy needed in the
 **                  storage filesystem exists,
 **
 **               2. Add the directory entry to the ovlfs directory inode using
 **                  the inode created in the storage fs to determine the file
 **                  mode and other information.
 **
 ** NOTES:
 **    - This method is dependent on using the storage filesystem.  This will
 **      need to be changed to allow for the NOSTORAGE option.
 **/

typedef int (*dirop_cb_t)(struct inode *, void *, const char *, int);

static int  ovlfs_add_dirent_op (struct inode *inode, const char *fname,
                                 int len, dirop_cb_t dirop_cb, void *cb_data)
{
    struct inode    *o_inode;
    struct inode    *new_ent;
    ovlfs_inode_t   *ino;
    ovlfs_inode_t   *new_ino;
    int             ret;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_add_dirent_cb(%ld, %.*s, %d, 0%o)\n",
           inode->i_ino, len, fname, len);
#endif

#if ! FASTER_AND_MORE_DANGEROUS
    if ( ( inode == INODE_NULL ) || ( dirop_cb == NULL ) )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_add_dirent_op: called with null argument\n");
# endif

        return  -ENOTDIR;
    }
#endif


        /* inode is the directory; obtain the storage fs's inode */

    ret = ovlfs_resolve_ovl_inode(inode, &o_inode);

    if ( ret < 0 )
    {
            /* The parent does not appear to have an inode in the */
            /*  storage fs; try to create it.                     */

        ret = ovlfs_make_hierarchy(inode, &o_inode, 0);
    }

    if ( ret >= 0 )
    {
        ret = ovlfs_use_inode(o_inode);

        if ( ret < 0 )
            IPUT(o_inode);
    }

    if ( ret >= 0 )
    {
            /* Make sure the storage fs' inode has the lookup operation */

        if ( ( o_inode->i_op == NULL ) || ( o_inode->i_op->lookup == NULL ) )
            ret = -ENOTDIR;
        else
        {
                /* Now use the callback to let the caller perform the      */
                /*  specific operation on the storage filesystem directory */

            ret = dirop_cb(o_inode, cb_data, fname, len);

            if ( ret >= 0 )
            {
                IMARK(o_inode);
                ret = o_inode->i_op->lookup(o_inode, fname, len, &new_ent);
            }
        }

        ovlfs_release_inode(o_inode);
        IPUT(o_inode);
    }

    if ( ret >= 0 )
    {
        ret = ovlfs_use_inode(new_ent);

        if ( ret < 0 )
            IPUT(new_ent);
    }

        /* If the entry was created successfully, add it to storage */

    if ( ret >= 0 )
    {
            /* Obtain the storage information for the directory inode */

        ino = ovlfs_get_ino(inode->i_sb, inode->i_ino);

        if ( ino == (ovlfs_inode_t *) NULL )
            ret = -ENOENT;
        else
        {
                /* Add a new entry into storage */

            new_ino = ovlfs_add_inode(inode->i_sb, fname, len);

            if ( new_ino == (ovlfs_inode_t *) NULL )
                ret = -ENOMEM;
            else
            {
                    /* Set the storage information from the new inode */

                new_ino->mode       = new_ent->i_mode;
                new_ino->size       = new_ent->i_size;
                new_ino->uid        = new_ent->i_uid;
                new_ino->gid        = new_ent->i_gid;
                new_ino->atime      = new_ent->i_atime;
                new_ino->mtime      = new_ent->i_mtime;
                new_ino->ctime      = new_ent->i_ctime;
                new_ino->nlink      = new_ent->i_nlink;
                new_ino->blksize    = new_ent->i_blksize;
                new_ino->blocks     = new_ent->i_blocks;
                new_ino->version    = new_ent->i_version;
                new_ino->nrpages    = new_ent->i_nrpages;
                new_ino->parent_ino = ino->ino;

                if ( S_ISBLK(new_ino->mode) || S_ISCHR(new_ino->mode) )
                    new_ino->u.rdev = new_ent->i_rdev;

                ret = ovlfs_inode_set_stg(inode->i_sb, new_ino, new_ent->i_dev,
                                          new_ent->i_ino);

                if ( ret >= 0 )
                {
                    ret = ovlfs_ino_add_dirent(ino, fname, len, new_ino->ino,
                                               OVLFS_DIRENT_NORMAL);

                    if ( ret >= 0 )
                    {
                        inode->i_mtime = ( inode->i_ctime = CURRENT_TIME );
                        inode->i_dirt = 1;
                    }
                }
            }
        }

        ovlfs_release_inode(new_ent);
        IPUT(new_ent);
    }

    IPUT(inode);

    return  ret;
}



/**
 ** FUNCTION: ovlfs_get_parent_dir
 **
 **  PURPOSE: Given an ovlfs inode, obtain the ovlfs inode of the inode's
 **           parent directory.  Note that this will be correct for
 **           directories, but not for files since files can reside in many
 **           directories.
 **
 ** NOTES:
 **     - If the parent directory can not be determined, NULL is returned.
 **/

struct inode    *ovlfs_get_parent_dir (struct inode *inode)
{
    ovlfs_inode_info_t  *i_info;
    ovlfs_inode_t       *ino;
    struct inode        *result;

#if ! FASTER_AND_MORE_DANGEROUS
    if ( ( inode == INODE_NULL ) || ( ! is_ovlfs_inode(inode) ) )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_get_parent_dir: invalid ovlfs inode given at "
               "0x%x\n", (int) inode);
# endif

        return  INODE_NULL;
    }
#endif

        /* First see if the parent directory's inode is in the inode */
        /*  information structure of this inode.                     */

    result = INODE_NULL;

    if ( inode->u.generic_ip != NULL )
    {
        i_info = (ovlfs_inode_info_t *) inode->u.generic_ip;

        if ( i_info->ovlfs_dir != INODE_NULL )
        {
            result = i_info->ovlfs_dir;
            IMARK(result);
        }
    }

        /* If the parent directory was not found in the inode's information, */
        /*  try looking up the inode's storage information for the parent's  */
        /*  inode number.                                                    */

    if ( result == INODE_NULL )
    {
        ino = ovlfs_get_ino(inode->i_sb, inode->i_ino);

        if ( ino != (ovlfs_inode_t *) NULL )
        {
            if ( ino->parent_ino != 0 )
                result = __iget(inode->i_sb, ino->parent_ino,
                                ovlfs_sb_xmnt(inode->i_sb));
        }
    }

    return  result;
}



#if USE_MAGIC

/**
 ** FUNCTION: ck_magic_name
 **
 **  PURPOSE: Determine if the name given is the name of the specified magic
 **           directory.
 **
 ** NOTES:
 **     - This function must only be called with a valid ovlfs super block.
 **/

static inline int   ck_magic_name (struct super_block *sb, const char *name,
                                   int len, char which)
{
    char    *m_name = NULL;
    int     m_len = 0;

    switch ( which )
    {
        case 'o':
        case 's':
            m_name = ovlfs_sb_opt(sb, Smagic_name);
            m_len = ovlfs_sb_opt(sb, Smagic_len);

            if ( ( m_name == (char *) NULL ) || ( m_len <= 0 ) )
            {
                m_name = OVLFS_MAGIC_NAME;
                m_len = OVLFS_MAGIC_NAME_LEN;
            }
            break;

        default:
            m_name = ovlfs_sb_opt(sb, Bmagic_name);
            m_len = ovlfs_sb_opt(sb, Bmagic_len);

            if ( ( m_name == (char *) NULL ) || ( m_len <= 0 ) )
            {
                m_name = OVLFS_MAGICR_NAME;
                m_len = OVLFS_MAGICR_NAME_LEN;
            }
            break;
    }

        /* Check if the length and name match and return the result */

    return  ( ( len == m_len ) && ( strncmp(name, m_name, len) == 0 ) );
}

#endif



/**
 ** FUNCTION: ovlfs_lookup
 **
 **  PURPOSE: Perform the lookup inode directory operation.
 **/

static int  ovlfs_lookup (struct inode *inode, const char *fname, int len,
                          struct inode **result)
{
    int ret = 0;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_lookup(%ld, %.*s, %d)\n", inode->i_ino,
           len, fname, len);
#endif

#if ! FASTER_AND_MORE_DANGEROUS
    if ( ! is_ovlfs_inode(inode) )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_lookup called with non-ovlfs inode!\n");
# endif
        IPUT(inode);
        return  -EINVAL;
    }
#endif

    result[0] = (struct inode *) NULL;

    if ( inode == (struct inode *) NULL )
        return  -ENOTDIR;

    if ( ! S_ISDIR(inode->i_mode) )
    {
        IPUT(inode);
        return  -ENOTDIR;
    }

    if ( len == 0 )
        result[0] = inode;
    else
        if ( ( fname[0] == '.' ) &&
             ( (len == 1) || (( len == 2 ) && ( fname[1] == '.' )) ) )
        {
                /* The name is either "." or ".." if we get here. */

            if ( len == 1 )
                result[0] = inode;
            else
            {
                result[0] = ovlfs_get_parent_dir(inode);

                if ( result[0] == INODE_NULL )
                    result[0] = inode;
                else
                    IPUT(inode);
            }
        }
#if USE_MAGIC
        else if ( ( ovlfs_magic_opt_p(inode->i_sb, OVLFS_OVL_MAGIC) ) &&
                  ( ck_magic_name(inode->i_sb, fname, len, 'o') ) )
        {
            ret = ovlfs_resolve_ovl_inode(inode, result);

            IPUT(inode);
        }
        else if ( ( ovlfs_magic_opt_p(inode->i_sb, OVLFS_BASE_MAGIC) ) &&
                  ( ck_magic_name(inode->i_sb, fname, len, 'b') ) )
        {
            ret = ovlfs_resolve_base_inode(inode, result);

            IPUT(inode);
        }
#endif
        else
        {
            ovlfs_inode_t       *ino;
            ovlfs_dir_info_t    *d_info;

                /* Lookup the directory entry */

            ino = ovlfs_get_ino(inode->i_sb, inode->i_ino);

            if ( ino == (ovlfs_inode_t *) NULL )
                ret = -ENOENT;
            else
            {
                d_info = ovlfs_ino_find_dirent(ino, fname, len);

                    /* Make sure the entry is found and is not marked */
                    /*  as UNLINKED.                                  */

                if ( ( d_info == (ovlfs_dir_info_t *) NULL ) ||
                     ( d_info->flags & OVLFS_DIRENT_UNLINKED ) )
                    ret = -ENOENT;
                else
                {
                    result[0] = __iget(inode->i_sb, d_info->ino,
                                       ovlfs_sb_xmnt(inode->i_sb));

                    if ( result[0] == (struct inode *) NULL )
                        ret = -ENOENT;
                }
            }

            IPUT(inode);
        }

    return  ret;
}



/**
 ** FUNCTION: ovlfs_link
 **
 **  PURPOSE: This is the inode link() method for the overlay filesystem.
 **/

static int  ovlfs_link (struct inode *orig_inode, struct inode *dir,
                        const char *name, int len)
{
    int             ret;
    ovlfs_inode_t   *ino;
    ovlfs_inode_t   *dir_ino;
    struct inode    *o_inode;
    struct inode    *o_dir;

        /* Make sure the two inodes given are OVLFS inodes */

    if ( ( ! is_ovlfs_inode(orig_inode) ) || ( ! is_ovlfs_inode(dir) ) )
    {
        IPUT(orig_inode);
        IPUT(dir);
        return  -ENOTDIR;
    }


        /* Obtain the storage information for the original inode and the */
        /*  directory.                                                   */

    ino = ovlfs_get_ino(orig_inode->i_sb, orig_inode->i_ino);

    if ( ino == (ovlfs_inode_t *) NULL )
        ret = -ENOENT;
    else
    {
        dir_ino = ovlfs_get_ino(dir->i_sb, dir->i_ino);

        if ( dir_ino == (ovlfs_inode_t *) NULL )
            ret = -ENOENT;
        else
        {
                /* Add the directory entry for the original inode into the */
                /*  directory with the specified name.                     */

            ret = ovlfs_ino_add_dirent(dir_ino, name, len, ino->ino,
                                       OVLFS_DIRENT_NORMAL);

            if ( ret >= 0 )
            {
                    /* Update the link count for the original inode */

                orig_inode->i_nlink++;
                orig_inode->i_dirt = 1;


                    /* Perform the link in the storage fs, if possible. */
                    /*  Start by obtaining the original file inode from */
                    /*  storage fs and then obtain the directory inode  */
                    /*  if the original file's inode is found.          */

                ret = ovlfs_resolve_stg_inode(orig_inode, &o_inode);

                if ( ret >= 0 )
                {
                    ret = ovlfs_resolve_stg_inode(dir, &o_dir);

                        /* Did not find the directory, try to create it */

                    if ( ret < 0 )
                        ret = ovlfs_make_hierarchy(dir, &o_dir, 0);

                    if ( ret >= 0 )
                    {
                        if ( ( o_dir->i_op == NULL ) ||
                             ( o_dir->i_op->link == NULL ) )
                            ret = -ENOTDIR;

                        if ( ret >= 0 )
                        {
                            down(&(o_dir->i_sem));
                            ret = o_dir->i_op->link(o_inode, o_dir, name, len);
                            up(&(o_dir->i_sem));
                        }
                        else
                            IPUT(o_inode);

                        IPUT(o_dir);
                    }
                    else
                        IPUT(o_inode);
                }
                else
                    ret = 0;    /* It is ok not to have the stg inode */
            }
        }
    }

    IPUT(orig_inode);
    IPUT(dir);

    return  ret;
}



/**
 ** FUNCTION: ovlfs_unlink
 **
 **  PURPOSE: This is the inode unlink method for the overlay filesystem.  The
 **           named file is removed from the specified directory inode.
 **/

static int ovlfs_unlink (struct inode *d_inode, const char *fname, int len)
{
    struct inode    *o_inode;
    ovlfs_inode_t   *ino;
    int             ret;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_unlink(%ld, %.*s, %d)\n", d_inode->i_ino,
           len, fname, len);
#endif

#if ! FASTER_AND_MORE_DANGEROUS
    if ( ! is_ovlfs_inode(d_inode) )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_unlink called with non-ovlfs inode!\n");
# endif
        IPUT(d_inode);
        return  -EINVAL;
    }

    if ( ( d_inode->i_op == NULL ) || ( d_inode->i_op->lookup == NULL ) )
    {
# if KDEBUG
        printk("OVLFS: ovlfs_unlink called with non-directory inode?\n");
# endif

        IPUT(d_inode);
        return  -ENOTDIR;
    }
#endif

        /* Obtain the pseudo fs' storage information for the directory */

    ino = ovlfs_get_ino(d_inode->i_sb, d_inode->i_ino);

    if ( ino != (ovlfs_inode_t *) NULL )
    {
        struct inode    *f_inode;

                /***
                 *** Obtain the inode for the file being unlinked so that
                 ***  its link count may be updated.  It is important to obtain
                 ***  the inode from the VFS in case it is in use somewhere,
                 ***  and therefore in the inode cache.
                 ***/

        IMARK(d_inode);     /* lookup eats the inode (i.e. it calls iput()) */
        ret = d_inode->i_op->lookup(d_inode, fname, len, &f_inode);

        if ( ret >= 0 )
        {
                /* Don't unlink directories; rmdir() must be used */

            if ( S_ISDIR(f_inode->i_mode) )
                ret = -EPERM;
            else
            {
                    /* Don't set the link count below zero: */

                if ( f_inode->i_nlink > 0 )
                    f_inode->i_nlink--;

                f_inode->i_dirt = 1;

                ret = ovlfs_unlink_dirent(ino, fname, len);
            }

            IPUT(f_inode);
        }
    }
    else
        ret = -ENOENT;


        /* Now perform the operation in the storage filesystem, if all is ok */

    if ( ret >= 0 )
    {
        ret = ovlfs_resolve_ovl_inode(d_inode, &o_inode);

        if ( ret >= 0 )
        {
            ret = ovlfs_use_inode(o_inode);
    
            if ( ret < 0 )
                IPUT(o_inode);
    
            if ( ret >= 0 )
            {
                if ( ( o_inode->i_op != NULL ) &&
                     ( o_inode->i_op->unlink != NULL ) )
                {
                        /* Unlink put's the directory inode it is given */
        
                    ret = o_inode->i_op->unlink(o_inode, fname, len);
        
                        /* It is not an error if the file does not exist in */
                        /*  the storage filesystem.                         */
        
                    if ( ret == -ENOENT )
                        ret = 0;
                }
                else
                {
                    IPUT(o_inode);
                    ret = -ENOTDIR;
                }
        
                ovlfs_release_inode(o_inode);
            }
        }
        else
            if ( ret == -ENOENT )
                ret = 0;    /* The storage fs DOESN'T NEED to have the dir */


        if ( ret >= 0 )
            d_inode->i_ctime = ( d_inode->i_mtime = CURRENT_TIME );
    }

    IPUT(d_inode);

    return  ret;
}



/**
 ** FUNCTION: mkdir_cb
 **
 **  PURPOSE: Given the directory inode in the storage filesystem, perform
 **           the actual mkdir on that inode.
 **
 ** NOTES:
 **     - This is a callback function for use with the ovlfs_add_dirent_op
 **       function.
 **/

static int  mkdir_cb (struct inode *inode, void *data, const char *name,
                      int len)
{
    int ret;
    int mode;

    mode = (int) data;

    if ( ( inode != INODE_NULL ) && ( inode->i_op != NULL ) &&
         ( inode->i_op->mkdir != NULL ) )
    {
        IMARK(inode);
        ret = inode->i_op->mkdir(inode, name, len, mode);
    }
    else
        ret = -ENOTDIR;

    return  ret;
}



/**
 ** FUNCTION: ovlfs_mkdir
 **
 **  PURPOSE: This is the overlay filesystem's mkdir() inode operation.
 **/

static int  ovlfs_mkdir (struct inode *inode, const char *fname, int len,
                         int mode)
{
    return  ovlfs_add_dirent_op(inode, fname, len, mkdir_cb, (void *) mode);
}



/**
 ** FUNCTION: symlink_cb
 **
 **  PURPOSE: Given the directory inode in the storage filesystem, perform
 **           the actual symlink on that inode.
 **
 ** NOTES:
 **     - This is a callback function for use with the ovlfs_add_dirent_op
 **       function.
 **/

static int  symlink_cb (struct inode *inode, void *data, const char *name,
                        int len)
{
    int         ret;
    const char  *sym_name;

    sym_name = (const char *) data;

    if ( ( inode != INODE_NULL ) && ( inode->i_op != NULL ) &&
         ( inode->i_op->symlink != NULL ) )
    {
        IMARK(inode);
        ret = inode->i_op->symlink(inode, name, len, sym_name);
    }
    else
        ret = -ENOTDIR;

    return  ret;
}



/**
 ** FUNCTION: ovlfs_symlink
 **
 **  PURPOSE: This is the overlay filesystem's symlink() inode operation.
 **/

static int  ovlfs_symlink (struct inode *inode, const char *fname, int len,
                           const char *sym_name)
{
    return  ovlfs_add_dirent_op(inode, fname, len, symlink_cb,
                                (void *) sym_name);
}



/**
 ** FUNCTION: create_cb
 **
 **  PURPOSE: Create a new directory entry in the given directory inode with
 **           the specified filename and status information.
 **
 ** NOTES:
 **     - This is a callback function for use with the ovlfs_add_dirent_op
 **       function.
 **/

struct create_dat_struct {
    int             mode;
    struct inode    *result;
} ;

static int  create_cb (struct inode *inode, void *ptr, const char *fname,
                       int len)
{
    int                         ret;
    struct create_dat_struct    *data;
    struct inode                *new_inode;

    if ( ( inode == INODE_NULL ) || ( inode->i_op == NULL ) ||
         ( inode->i_op->create == NULL ) || ( ptr == NULL ) )
        return  -ENOTDIR;

    data = (struct create_dat_struct *) ptr;

    IMARK(inode);
    ret = inode->i_op->create(inode, fname, len, data->mode, &new_inode);

    if ( ret >= 0 )
        data->result = new_inode;

    return  ret;
}



/**
 ** FUNCTION: ovlfs_create
 **
 **  PURPOSE: This is the overlay filesystem's create() inode operation.
 **/

static int  ovlfs_create (struct inode *inode, const char *fname, int len,
                          int mode, struct inode **result)
{
    struct create_dat_struct    data;
    int                         ret;

    data.mode = mode;
    data.result = INODE_NULL;

        /* Mark the inode since we still need it after this call */

    IMARK(inode);
    ret = ovlfs_add_dirent_op(inode, fname, len, create_cb, (void *) &data);


        /* If successful, obtain the new overlay fs inode */

    if ( ret >= 0 )
    {
        if ( ( inode->i_op != NULL ) && ( inode->i_op->lookup != NULL ) )
        {
            IMARK(inode);
            ret = inode->i_op->lookup(inode, fname, len, result);
    
                /* If we got the overlay fs inode, see if its storage inode */
                /*  reference exists; if not, set it to the new inode.      */
    
            if ( ( ret >= 0 ) && ( result[0]->u.generic_ip != NULL ) )
            {
                ovlfs_inode_info_t  *i_info;
    
                i_info = (ovlfs_inode_info_t *) result[0]->u.generic_ip;
    
                if ( i_info->overlay_inode == INODE_NULL )
                    i_info->overlay_inode = data.result;
                else
                    IPUT(data.result);
            }
            else
                IPUT(data.result);
        }
        else
        {
            IPUT(data.result);
            ret = -ENOTDIR;
        }
    }
    else
    {
            /* An error occurred, but it may have been after the storage */
            /*  inode's create was called, in which case the new inode   */
            /*  must be iput().                                          */

        if ( data.result != INODE_NULL )
            IPUT(data.result);
    }

    IPUT(inode);

    return  ret;
}



/**
 ** FUNCTION: ovlfs_rmdir
 **
 **  PURPOSE: This is the overlay filesystem's rmdir() inode operation.
 **/

static int  ovlfs_rmdir (struct inode *inode, const char *dname, int len)
{
    ovlfs_inode_t   *ino;
    struct inode    *o_inode;
    struct inode    *rm_dir;
    int             ret;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_rmdir(%ld, %.*s, %d)\n", inode->i_ino,
           len, dname, len);
#endif

        /***
         *** Obtain the inode for the directory being unlinked so that its link
         ***  count may be updated.  It is important to obtain the inode from
         ***  the VFS in case it is in use somewhere, and therefore is in the
         ***  inode cache.
         ***
         *** The lookup is done before doing the storage fs' rmdir() call
         ***  because of a timing issue with the ext2 filesystem that causes
         ***  harmless error messages to be displayed indicating that a cleared
         ***  block and inode are being cleared again.
         ***/


    IMARK(inode);     /* lookup eats the inode (i.e. it calls iput()) */
    ret = inode->i_op->lookup(inode, dname, len, &rm_dir);

    if ( ret < 0 )
    {
            /* Did not find the directory? */

        IPUT(inode);
        return ret;
    }


        /* Remove the directory from the storage filesystem; if the storage */
        /*  filesystem refuses to remove the directory (and it exists       */
        /*  there), then don't remove the directory from the pseudo fs.     */

        /*  note: inode is the directory */

    ret = ovlfs_resolve_ovl_inode(inode, &o_inode);

    if ( ret >= 0 )
    {
        ret = ovlfs_use_inode(o_inode);

        if ( ret < 0 )
            IPUT(o_inode);
    }

    if ( ret >= 0 )
    {
        if ( ( o_inode->i_op != NULL ) && ( o_inode->i_op->rmdir != NULL ) )
        {
            IMARK(o_inode);
            down(&(o_inode->i_sem));
            ret = o_inode->i_op->rmdir(o_inode, dname, len);
            up(&(o_inode->i_sem));
        }
        else
            ret = -ENOTDIR;

        ovlfs_release_inode(o_inode);
        IPUT(o_inode);
    }


        /* Remove the directory entry from the overlay fs inode information */
        /*  if the stg fs' rmdir() was successful, or the dir did not exist */

    if ( ( ret >= 0 ) || ( ret == -ENOENT ) )
    {
            /* Now update the pseudo fs' storage for the directory to */
            /*  reflect the removed directory.                        */

        ino = ovlfs_get_ino(inode->i_sb, inode->i_ino);

        if ( ino != (ovlfs_inode_t *) NULL )
        {
            ret = ovlfs_unlink_dirent(ino, dname, len);

                /* If successful, update the unlinked dir's link count */

            if ( ret >= 0 )
            {
                    /* Don't set the link count below zero: */

                if ( rm_dir->i_nlink > 0 )
                    rm_dir->i_nlink--;

                rm_dir->i_dirt = 1;

                inode->i_ctime = ( inode->i_mtime = CURRENT_TIME );
                inode->i_dirt = 1;
            }
        }
    }

    IPUT(rm_dir);
    IPUT(inode);

    return  ret;
}



/**
 ** FUNCTION: ovlfs_rename
 **
 **  PURPOSE: This is the overlay filesystem's rename() inode operation.
 **
 ** NOTES:
 **     - Must iput both directories
 **     - The rename() operation for the storage filesystem MUST NOT be called
 **	  if the two directories are not on the same device.
 **
 ** ALGORITHM:
 **     - Determine if the old directory and the file exist in the storage
 **       filesystem.
 **     - If both exist there, obtain the new directory's storage inode, or
 **       create it if it does not exist.  Then, use the old directory's
 **       rename() operation to move the file in the storage filesystem.
 **     - If the rename() in the storage filesystem is successful, or the
 **       file doesn't exist in the storage filesystem, just rename the file
 **       in the pseudo filesystem's storage.
 **/

static int  ovlfs_rename (struct inode *olddir, const char *oname, int olen,
                          struct inode *newdir, const char *nname, int nlen,
                          int must_be_dir)
{
    struct inode        *o_olddir;
    struct inode        *o_newdir;
    struct inode        *new_inode;
    ovlfs_inode_t       *old_ino;
    ovlfs_inode_t       *new_ino;
    ovlfs_dir_info_t    *d_info;
    int                 old_in_ovl;
    int                 ret;

    new_inode = INODE_NULL;

#if KDEBUG_CALLS
    printk(KDEBUG_CALL_PREFIX "ovlfs_rename(%ld, %.*s, %d, %ld, %.*s, %d, "
           "%d)\n", olddir->i_ino, olen, oname, olen, newdir->i_ino, nlen,
           nname, nlen, must_be_dir);
#endif


        /* Perform the rename in the storage filesystem if the entry  */
        /*  exists in the storage filesystem.                         */

    ret = ovlfs_resolve_ovl_inode(olddir, &o_olddir);

    if ( ret >= 0 )
    {
        ret = ovlfs_use_inode(o_olddir);

        if ( ret < 0 )
            IPUT(o_olddir);
    }

    if ( ret >= 0 )
    {
            /* OK, the old directory exists in the storage fs, see if */
            /*  the file is there also.                               */

        if ( ( olddir->i_op != NULL ) && ( olddir->i_op->lookup != NULL ) )
        {
            struct inode    *tmp = NULL;

            IMARK(o_olddir);
            ret = o_olddir->i_op->lookup(o_olddir, oname, olen, &tmp);

                /* Just looking; don't "keep" this inode */

            if ( ret >= 0 )
                IPUT(tmp);
        }
        else
            ret = -ENOENT;


        if ( ret >= 0 )
        {
                /***
                 *** The file and the old directory are both in the storage
                 ***  filesystem, so it is necessary to rename the file in
                 ***  that filesystem.  Get the new directory's storage fs
                 ***  inode, or create it if is doesn't exist.
                 ***/

            ret = ovlfs_resolve_ovl_inode(newdir, &o_newdir);
    
            if ( ret < 0 )
                ret = ovlfs_make_hierarchy(newdir, &o_newdir, 0);
    
            if ( ret >= 0 )
            {
                    /* Make sure the dir's are on the same device */

                if ( o_newdir->i_dev != o_olddir->i_dev )
                {
#ifdef KDEBUG
                    printk("OVLFS: rename: old dir and new dir on different "
                           "devs; old 0x%x, new 0x%x\n", o_olddir->i_dev,
                           o_newdir->i_dev);
#endif

                    IPUT(o_newdir);
                    ret = -EXDEV;
                }
                else
                {
                        /* Prepare to use the inode */

                    ret = ovlfs_use_inode(o_newdir);
    
                    if ( ret < 0 )
                        IPUT(o_newdir);
                }
            }
#if KDEBUG
            else
                printk("OVLFS: ovlfs_rename: error %d obtaining storage dir "
                       "for new directory\n", ret);
#endif

            if ( ret >= 0 )
            {
                    /* Both directories exist in the storage filesystem and */
                    /*  the original file resides in the storage fs also.   */
    
                old_in_ovl = 1;
    
                if ( ( o_olddir->i_op != NULL ) &&
                     ( o_olddir->i_op->rename != NULL ) )
                {
                    IMARK(o_olddir);
                    IMARK(o_newdir);
                    down(&(o_newdir->i_sem));
                    ret = o_olddir->i_op->rename(o_olddir, oname, olen,
                                                 o_newdir, nname, nlen,
                                                 must_be_dir);
                    up(&(o_newdir->i_sem));
                }
                else
                    ret = -ENOTDIR;
    
                ovlfs_release_inode(o_newdir);
                IPUT(o_newdir);
            }
        }
        else
            ret = 0;    /* file isn't in the storage fs, that's OK */

        ovlfs_release_inode(o_olddir);
        IPUT(o_olddir);
    }
    else
        ret = 0;    /* file isn't in the storage fs, that's OK */



#if OVLFS_ALLOW_XDEV_RENAME

        /* If the problem is that the two directories in the storage fs   */
        /*  are actually in two different filesystems (i.e. a mount point */
        /*  is being crossed), allow the rename to be maintained by the   */
        /*  pseudo filesystem anyway.  Just don't lose that storage info. */

    if ( ret == -EXDEV )
        ret = 0;

#endif


        /* An error here means the file was in the storage fs, but some */
        /*  error occurred trying to rename it; this is a real problem. */

    if ( ret >= 0 )
    {
            /* Now perform the rename in the psuedo filesystem as well. Get */
            /*  the storage information for the old and new directories.    */
    
        old_ino = ovlfs_get_ino(olddir->i_sb, olddir->i_ino);
    
        if ( old_ino != (ovlfs_inode_t *) NULL )
        {
                /* If the entry is moving from one directory to another, */
                /*  obtain the inode information for the new directory.  */
    
            if ( olddir == newdir )
                new_ino = old_ino;
            else
                new_ino = ovlfs_get_ino(newdir->i_sb, newdir->i_ino);
    
            if ( new_ino != (ovlfs_inode_t *) NULL )
            {
                    /* Get the old entries information */
    
                d_info = ovlfs_ino_find_dirent(old_ino, oname, olen);
    
                if ( d_info != (ovlfs_dir_info_t *) NULL )
                {
                        /* Add the new directory entry */
    
                    ret = ovlfs_ino_add_dirent(new_ino, nname, nlen,
                                               d_info->ino,
                                               OVLFS_DIRENT_NORMAL);
    
                    if ( ret >= 0 )
                    {
                            /* Update the new dir's times and link count */
    
                        newdir->i_nlink++;
                        newdir->i_mtime = CURRENT_TIME;
                        newdir->i_ctime = newdir->i_mtime;
                        newdir->i_dirt = 1;
    
                            /* Unlink the old directory entry and set the */
                            /*  storage for the inode to point to the new */
                            /*  to the new file.                             */
    
                        d_info = (ovlfs_dir_info_t *) NULL;
                        ret = ovlfs_unlink_dirent(old_ino, oname, olen);
    
                            /* Same as comments above (Mark the new...) */
                        if ( ret >= 0 )
                        {
                            olddir->i_nlink--;
                            olddir->i_mtime = CURRENT_TIME;
                            olddir->i_ctime = olddir->i_mtime;
                            olddir->i_dirt = 1;
                        }
                    }
                }
            }
        }
    }

    IPUT(olddir);
    IPUT(newdir);

    return  ret;
}



/**
 ** FUNCTION: ovlfs_readlink
 **
 **  PURPOSE: This is the overlay filesystem's readlink() inode operation.
 **/

int ovlfs_readlink (struct inode *inode, char *buf, int size)
{
    struct inode    *o_inode;
    int             ret;

    if ( ! is_ovlfs_inode(inode) )
    {
        IPUT(inode);
        return  -EINVAL;
    }

    ret = ovlfs_resolve_stg_inode(inode, &o_inode);

    if ( ret < 0 )
        ret = ovlfs_resolve_base_inode(inode, &o_inode);

    if ( ret >= 0 )
    {
        ret = ovlfs_use_inode(o_inode);

        if ( ret < 0 )
            IPUT(o_inode);
    }

    if ( ret >= 0 )
    {
        if ( ( o_inode->i_op != NULL ) && ( o_inode->i_op->readlink != NULL ) )
        {
            IMARK(o_inode);
            ret = o_inode->i_op->readlink(o_inode, buf, size);
        }
        else
            ret = -EINVAL;

        ovlfs_release_inode(o_inode);
        IPUT(o_inode);
    }

    IPUT(inode);

    return  ret;
}



/**
 ** FUNCTION: ovlfs_follow_link
 **
 **  PURPOSE: This is the overlay fs' follow_link() inode operation.
 **/

static int  ovlfs_follow_link (struct inode *dir, struct inode *inode,
                               int flag, int mode, struct inode **res_inode)
{
    struct inode    *o_dir;
    struct inode    *o_inode;
    int             ret;

        /* Make sure the file inode given is not NULL */

    if ( inode == (struct inode *) NULL )
    {
        IPUT(inode);
        res_inode[0] = (struct inode *) NULL;
        return  -ENOENT;
    }

        /* In case this method is called on a non-symbolic link, or one    */
        /*  of the inodes given is not in the ovlfs, just return the file. */

    if ( ( ! is_ovlfs_inode(dir) ) || ( ! is_ovlfs_inode(inode) ) )
    {
        IPUT(dir);
        res_inode[0] = inode;
        return  0;
    }


        /* Find the directory and symbolic link both in the same fs */

    ret = ovlfs_resolve_stg_inode(dir, &o_dir);

    if ( ret >= 0 )
    {
        ret = ovlfs_resolve_stg_inode(inode, &o_inode);

        if ( ret < 0 )
            IPUT(o_dir);
    }

    if ( ret < 0 )
    {
            /* Did not find the directory inode in the storage fs, check the */
            /*  base filesystem for it.                                      */

        ret = ovlfs_resolve_base_inode(dir, &o_dir);

        if ( ret >= 0 )
        {
            ret = ovlfs_resolve_base_inode(inode, &o_inode);

            if ( ret < 0 )
                IPUT(o_dir);
        }
    }

    if ( ret >= 0 )
    {
        ret = ovlfs_use_inode(o_inode);

        if ( ret < 0 )
        {
            IPUT(o_dir);
            IPUT(o_inode);
            IPUT(inode);
        }
        else
        {
            if ( ( o_inode->i_op != NULL ) &&
                 ( o_inode->i_op->follow_link != NULL ) )
            {
                ret = o_inode->i_op->follow_link(o_dir, o_inode, flag, mode,
                                                 res_inode);
                IPUT(inode);
            }
            else
            {
                res_inode[0] = inode;
                IPUT(o_dir);
                IPUT(o_inode);
            }

            ovlfs_release_inode(o_inode);
        }
    }
    else
        IPUT(inode);

    IPUT(dir);

    return  ret;
}



/**
 ** FUNCTION: ovlfs_readpage
 **
 **  PURPOSE: This is the overlay fs' readpage() function.
 **/

static int  ovlfs_readpage (struct inode *inode, struct page *page)
{
    struct inode    *o_inode;
    int             ret;

    ret = ovlfs_resolve_ovl_inode(inode, &o_inode);

    if ( ret < 0 )
        ret = ovlfs_resolve_base_inode(inode, &o_inode);

    if ( ret >= 0 )
    {
        if ( ( o_inode->i_op == NULL ) || ( o_inode->i_op->readpage == NULL ) )
            ret = -ENOEXEC;
        else
            ret = o_inode->i_op->readpage(o_inode, page);

        IPUT(o_inode);
    }

    return  ret;
}



#if OVLFS_DEFINE_BMAP

/**
 ** FUNCTION: ovlfs_bmap
 **
 **  PURPOSE: This is the overlay fs' bmap() inode operation.
 **/

static int  ovlfs_bmap (struct inode *inode, int block)
{
    int             ret;
    int             rc;
    struct inode    *o_inode;

    if ( ! is_ovlfs_inode(inode) )
        return  0;

    ret = 0;

    rc = ovlfs_resolve_ovl_inode(inode, &o_inode);

    if ( rc < 0 )
        rc = ovlfs_resolve_base_inode(inode, &o_inode);

    if ( rc >= 0 )
    {
        if ( ( o_inode->i_op != NULL ) || ( o_inode->i_op->bmap != NULL ) )
            ret = o_inode->i_op->bmap(o_inode, block);

        IPUT(o_inode);
    }

    return  ret;
}

#endif



/**
 ** FUNCTION: ovlfs_truncate
 **
 **  PURPOSE: This is the overlay filesystem's truncate() inode operation.
 **/

static void ovlfs_truncate (struct inode *inode)
{
    int             ret;
    struct inode    *o_inode;
    struct iattr    attribs;

    ret = ovlfs_resolve_stg_inode(inode, &o_inode);

    if ( ret >= 0 )
    {
        ret = ovlfs_use_inode(o_inode);

        if ( ret < 0 )
            IPUT(o_inode);
    }

    if ( ret >= 0 )
    {
        if ( ( o_inode->i_sb != NULL ) &&
             ( o_inode->i_sb->s_op->notify_change != NULL ) )
        {
            attribs.ia_valid = ATTR_SIZE | ATTR_CTIME;
            attribs.ia_size = inode->i_size;
            ret = o_inode->i_sb->s_op->notify_change(o_inode, &attribs);
        }

        if ( ret >= 0 )
        {
            o_inode->i_size = inode->i_size;

            if ( ( o_inode->i_op != NULL ) &&
                 ( o_inode->i_op->truncate != NULL ) )
            {
                o_inode->i_op->truncate(o_inode);
            }
        }

        ovlfs_release_inode(o_inode);
        IPUT(o_inode);
    }
}



/**
 ** FUNCTION: ovlfs_permission
 **
 **  PURPOSE: This is the overlay filesystem's permission() inode operation.
 **/

static int  ovlfs_permission (struct inode *inode, int mask)
{
    struct inode    *o_inode;
    int             ret;

    if ( ! is_ovlfs_inode(inode) )
        return  -EACCES;

        /* Try and find the associated "real" file and use its permission  */
        /*  checking; if neither is found, just use the default permission */
        /*  check.  In this case, the real problem will be found at a more */
        /*  appropriate place.                                             */

    ret = ovlfs_resolve_ovl_inode(inode, &o_inode);

    if ( ret < 0 )
        ret = ovlfs_resolve_base_inode(inode, &o_inode);

    if ( ret >= 0 )
    {
            /* If the inode of the real file does not have a permission */
            /*  method, just use the default permission checks.         */

        if ( ( o_inode->i_op != NULL ) &&
             ( o_inode->i_op->permission != NULL ) )
            ret = o_inode->i_op->permission(o_inode, mask);
        else
            ret = ovlfs_ck_def_perm(inode, mask);

        IPUT(o_inode);
    }
    else
        ret = ovlfs_ck_def_perm(inode, mask);

    return  ret;
}



/**
 ** FUNCTION: mknod_cb
 **
 **  PURPOSE: Make a new node in the given directory inode with the specified
 **           filename and status information.
 **
 ** NOTES:
 **     - This is a callback function for use with the ovlfs_add_dirent_op
 **       function.
 **/

struct mknod_dat_struct {
    int mode;
    int dev;
} ;

static int  mknod_cb (struct inode *inode, void *ptr, const char *name,
                      int len)
{
    struct mknod_dat_struct *data;
    int                     ret;

        /* The given inode must not be an ovlfs inode or we will get into */
        /*  an infinite loop (since its mknod will come back here...)     */

    if ( is_ovlfs_inode(inode) )
        return -ELOOP;

    data = (struct mknod_dat_struct *) ptr;

    if ( ( ptr == NULL ) || ( inode == INODE_NULL ) ||
         ( inode->i_op == NULL ) || ( inode->i_op->mknod == NULL ) )
        return  -ENOTDIR;

    IMARK(inode);

    down(&(inode->i_sem));
    ret = inode->i_op->mknod(inode, name, len, data->mode, data->dev);
    up(&(inode->i_sem));

    return ret;
}



/**
 ** FUNCTION: ovlfs_mknod
 **
 **  PURPOSE: This is the overlay filesystem's mknod() inode operation.
 **/

static int  ovlfs_mknod (struct inode *inode, const char *name, int len,
                         int mode, int dev)
{
    struct mknod_dat_struct data;

    data.mode = mode;
    data.dev  = dev;

    return  ovlfs_add_dirent_op(inode, name, len, mknod_cb, (void *) &data);
}



/**
 ** ovlfs_ino_ops: define the pseudo fs' operations for directories.
 **/

struct inode_operations ovlfs_ino_ops = {
    &ovlfs_dir_fops,
    ovlfs_create,
    ovlfs_lookup,
    ovlfs_link,
    ovlfs_unlink,
    ovlfs_symlink,
    ovlfs_mkdir,
    ovlfs_rmdir,
    ovlfs_mknod,
    ovlfs_rename,
    ovlfs_readlink,
    ovlfs_follow_link,
    ovlfs_readpage,
    (int (*)(struct inode *, struct page *)) NULL,
#if OVLFS_DEFINE_BMP
    ovlfs_bmap,
#else
    (int (*)(struct inode *,int)) NULL,
#endif
    ovlfs_truncate,
    ovlfs_permission,
    (int (*)(struct inode *,int)) NULL
} ;



/**
 ** ovlfs_ino_ops: define the pseudo fs' operations for regular files.
 **/

struct inode_operations ovlfs_file_ino_ops = {
    &ovlfs_file_ops,
    ovlfs_create,
    ovlfs_lookup,
    ovlfs_link,
    ovlfs_unlink,
    ovlfs_symlink,
    ovlfs_mkdir,
    ovlfs_rmdir,
    ovlfs_mknod,
    ovlfs_rename,
    ovlfs_readlink,
    ovlfs_follow_link,
    ovlfs_readpage,
    (int (*)(struct inode *, struct page *)) NULL,
#if OVLFS_DEFINE_BMP
    ovlfs_bmap,
#else
    (int (*)(struct inode *,int)) NULL,
#endif
    ovlfs_truncate,
    ovlfs_permission,
    (int (*)(struct inode *,int)) NULL
} ;
