/******************************************************************************
 ******************************************************************************
 **
 ** COPYRIGHT (C) 1998 By Arthur Naseef
 **
 ** This file is covered under the GNU General Public License Version 2.  For
 **  more information, see the file COPYING.
 **
 **
 ** FILE: ovl_debug.c
 **
 ** DESCRIPTION: This file contains debugging functions that log debugging
 **              information in a file.
 **
 ** REVISION HISTORY:
 **
 ** DATE	AUTHOR		DESCRIPTION
 ** ==========	===============	==============================================
 ** 09/26/1997	ARTHUR N.	Added to source code control.
 ** 09/26/1997	ARTHUR N.	Added comments.
 ** 03/09/1998	ARTHUR N.	Added the copyright notice.
 **
 ******************************************************************************
 ******************************************************************************
 **/

#ident "@(#) ovl_debug.c 1.3"

#include <stdarg.h>

#ifdef MODVERSIONS
# include <linux/modversions.h>
# ifndef __GENKSYMS__
#  include "ovlfs.ver"
# endif
#endif

#include <linux/stddef.h>

#include <linux/types.h>
#include <linux/sched.h>
#include <linux/fcntl.h>
#include <linux/fs.h>
#include <asm/statfs.h>
#include <asm/segment.h>

#include "ovl_fs.h"

#ifndef OVLFS_DEBUG_LEVEL
# ifdef KDEBUG
#  define OVLFS_DEBUG_LEVEL	KDEBUG
# else
#  define OVLFS_DEBUG_LEVEL	0
# endif
#endif

/**
 ** GLOBALS:
 **/

static int	ovlfs_debug_fd = -1;
static int	ovlfs_debug_level = OVLFS_DEBUG_LEVEL;



/**
 ** FUNCTION: ovlfs_open_debug_file
 **
 **  PURPOSE: Open the debugging output file within the current process.
 **
 ** NOTES:
 **	- Since file descriptors are allocated per process, opening the debug
 **	  file in this manner will only allow for access within the current
 **	  process.
 **/

static int	ovlfs_open_debug_file ()
{
	unsigned long	orig_fs;

#if KDEBUG > 2
	printk("OVLFS: open_debug_file called: ovlfs_debug_fd = %d\n",
	       ovlfs_debug_fd);
#endif

	if ( ovlfs_debug_fd == -1 )
	{
			/* Pretend that the kernel space is user space */

		orig_fs = get_fs();
		set_fs(KERNEL_DS);

		ovlfs_debug_fd = open(OVLFS_DEBUG_FILE,
		                      O_WRONLY | O_APPEND | O_CREAT, 0644);


			/* Set the user space back */

		set_fs(orig_fs);
	}

	return	ovlfs_debug_fd;
}



/**
 ** FUNCTION: ovlfs_close_debug_file
 **
 **  PURPOSE: Close the debug file opened in the current process.
 **
 ** NOTES:
 **	- If this function is called in a different process than the file was
 **	  opened in, it may close a file in use by the process and will not
 **	  close the debug file itself.
 **/

static void	ovlfs_close_debug_file ()
{
	if ( ovlfs_debug_fd != -1 )
	{
		close(ovlfs_debug_fd);
		ovlfs_debug_fd = -1;
	}
}


/**
 ** FUNCTION: ovlfs_dprintf
 **
 **  PURPOSE: Print a message into the debug file using printf() format, if
 **           the level of the message is less than or equal to the current
 **           debug level.
 **/

void	ovlfs_dprintf (int level, char *fmt, ...)
{
	char		buffer[1024];
	va_list		arg_list;
	unsigned long	orig_fs;

	if ( level > ovlfs_debug_level )
		return;

	if ( ovlfs_open_debug_file() < 0 )
	{
#if KDEBUG > 2
		printk("OVLFS: ovlfs_dprintf: ovlfs_open_debug_file failed"
		       " (%d)\n", ovlfs_debug_fd);
#endif

		return;
	}

	va_start(arg_list, fmt);

		/* Pretend that the kernel space is user space */

	orig_fs = get_fs();
	set_fs(KERNEL_DS);

	if ( vsprintf(buffer, fmt, arg_list) > 0 )
	{
		int	ret;

		ret = write(ovlfs_debug_fd, buffer, strlen(buffer));

#if KDEBUG > 2
		printk("OVLFS: ovlfs_dprintf: write returned %d\n", ret);
#endif
	}

		/* Set the user space back */

	set_fs(orig_fs);

	va_end(arg_list);

	ovlfs_close_debug_file();
}



/**
 ** FUNCTION: ovlfs_end_debug
 **
 **  PURPOSE: Stop any debugging.  This currently is not needed.
 **/

void	ovlfs_end_debug()
{
	ovlfs_close_debug_file();
}



/**
 ** FUNCTION: ovlfs_debug_inodes
 **
 **  PURPOSE: Write the list of inodes currently cached in the kernel to the
 **           debug file.
 **/

void	ovlfs_debug_inodes (struct inode *first_inode)
{
	struct inode	*one_inode;

	one_inode = first_inode;

	dprintf(0, "USED INODE LIST:\n");

	if ( one_inode == INODE_NULL )
		dprintf(0, "first inode is null!\n");
	else
		dprintf(0, "\tSUP BLK   \tINUM     \tADDRESS   \tUSE COUNT\n");

	do
	{
		if ( one_inode->i_count != 0 )
		{
			dprintf(0, "\t0x%08x\t%09d\t0x%08x\t%d\n",
			        one_inode->i_sb, one_inode->i_ino,
			        one_inode, one_inode->i_count);
		}

		one_inode = one_inode->i_next;
	} while ( ( one_inode != first_inode ) && ( one_inode != INODE_NULL ) );
}
