/*
 *  linux/fs/hpfs/namei.c
 *
 *  Mikulas Patocka (mikulas@artax.karlin.mff.cuni.cz), 1998-1999
 *
 *  adding & removing files & directories
 */

#include "hpfs_fn.h"

int hpfs_mkdir(struct inode *dir, struct dentry *dentry, int mode)
{
	const char *name = dentry->d_name.name;
	int len = dentry->d_name.len;
	struct quad_buffer_head qbh0;
	struct buffer_head *bh;
	struct hpfs_dirent *de;
	struct fnode *fnode;
	struct dnode *dnode;
	struct inode *result;
	fnode_secno fno;
	dnode_secno dno;
	int r;
	struct hpfs_dirent dee;
	if (!dir) return -ENOENT;
	if (!S_ISDIR(dir->i_mode)) {
		return -ENOTDIR;
	}
	adjust_length((char *)name, &len);
	if (chk_hpfs_name((char *)name, len)) {
		return -ENAMETOOLONG;
	}	
	if (!(fnode = alloc_fnode(dir->i_sb, dir->i_hpfs_dno, &fno, &bh))) goto bail;
	if (!(dnode = alloc_dnode(dir->i_sb, fno, &dno, &qbh0, 1))) goto bail1;
	memset(&dee, 0, sizeof dee);
	dee.directory = 1;
	if (!(mode & 0222)) dee.read_only = 1;
	dee.archive = 1;
	dee.hidden = name[0] == '.';
	dee.fnode = fno;
	dee.creation_date = dee.write_date = dee.read_date = gmt_to_local(dir->i_sb, CURRENT_TIME);
	hpfs_lock_inode(dir);
	r = add_dirent(dir, (char *)name, len, &dee, 0);
	if (r == 1) goto bail2;
	if (r == -1) {
		brelse(bh);
		brelse4(&qbh0);
		free_sectors(dir->i_sb, fno, 1);
		free_dnode(dir->i_sb, dno);
		hpfs_unlock_inode(dir);
		return -EEXIST;
	}
	fnode->len = len;
	memcpy(fnode->name, name, len > 15 ? 15 : len);
	fnode->up = dir->i_ino;
	fnode->dirflag = 1;
	fnode->btree.n_free_nodes = 7;
	fnode->btree.n_used_nodes = 1;
	fnode->btree.first_free = 0x14;
	fnode->u.external[0].disk_secno = dno;
	fnode->u.external[0].file_secno = -1;
	dnode->root_dnode = 1;
	dnode->up = fno;
	de = add_de(dir->i_sb, dnode, "\001\001", 2, 0);
	de->creation_date = de->write_date = de->read_date = gmt_to_local(dir->i_sb, CURRENT_TIME);
	if (!(mode & 0222)) de->read_only = 1;
	de->first = de->hidden = de->system = de->directory = 1;
	de->fnode = fno;
	mark_buffer_dirty(bh, 1);
	brelse(bh);
	mark_4buffers_dirty(&qbh0);
	brelse4(&qbh0);
	dir->i_nlink++;
	lock_iget(dir->i_sb, 1);
	if ((result = iget(dir->i_sb, fno))) {
		result->i_hpfs_parent_dir = dir->i_ino;
		result->i_ctime = result->i_mtime = result->i_atime = local_to_gmt(dir->i_sb, dee.creation_date);
		result->i_hpfs_ea_size = 0;
		if (dee.read_only) result->i_mode &= ~0222;
		if (result->i_uid != current->fsuid ||
		    result->i_gid != current->fsgid ||
		    result->i_mode != (mode | S_IFDIR)) {
			result->i_uid = current->fsuid;
			result->i_gid = current->fsgid;
			result->i_mode = mode | S_IFDIR;
			hpfs_write_inode_nolock(result);
		}
		d_instantiate(dentry, result);
	}
	unlock_iget(dir->i_sb);
	hpfs_unlock_inode(dir);
	return 0;
	bail2:
	brelse4(&qbh0);
	free_dnode(dir->i_sb, dno);
	hpfs_unlock_inode(dir);
	bail1:
	brelse(bh);
	free_sectors(dir->i_sb, fno, 1);
	bail:
	return -ENOSPC;
}

int hpfs_create(struct inode *dir, struct dentry *dentry, int mode)
{
	const char *name = dentry->d_name.name;
	int len = dentry->d_name.len;
	struct inode *result;
	struct buffer_head *bh;
	struct fnode *fnode;
	fnode_secno fno;
	int r;
	struct hpfs_dirent dee;
	result = NULL;
	if (!dir) return -ENOENT;
	if (!S_ISDIR(dir->i_mode)) {
		return -ENOTDIR;
	}
	if (!S_ISREG(mode)) {
		return -EINVAL;
	}
	adjust_length((char *)name, &len);
	if (chk_hpfs_name((char *)name, len)) {
		return -ENAMETOOLONG;
	}	
	if (!(fnode = alloc_fnode(dir->i_sb, dir->i_hpfs_dno, &fno, &bh))) goto bail;
	memset(&dee, 0, sizeof dee);
	if (!(mode & 0222)) dee.read_only = 1;
	dee.archive = 1;
	dee.hidden = name[0] == '.';
	dee.fnode = fno;
	dee.creation_date = dee.write_date = dee.read_date = gmt_to_local(dir->i_sb, CURRENT_TIME);
	hpfs_lock_inode(dir);
	r = add_dirent(dir, (char *)name, len, &dee, 0);
	if (r == 1) goto bail1;
	if (r == -1) {
		brelse(bh);
		free_sectors(dir->i_sb, fno, 1);
		hpfs_unlock_inode(dir);
		return -EEXIST;
	}
	fnode->len = len;
	memcpy(fnode->name, name, len > 15 ? 15 : len);
	fnode->up = dir->i_ino;
	mark_buffer_dirty(bh, 1);
	brelse(bh);
	lock_iget(dir->i_sb, 2);
	if ((result = iget(dir->i_sb, fno))) {
		decide_conv(result, (char *)name, len);
		result->i_hpfs_parent_dir = dir->i_ino;
		result->i_ctime = result->i_mtime = result->i_atime = local_to_gmt(dir->i_sb, dee.creation_date);
		result->i_hpfs_ea_size = 0;
		if (dee.read_only) result->i_mode &= ~0222;
		if (result->i_blocks == -1) result->i_blocks = 512;
		if (result->i_size == -1) result->i_size = 0;
		if (result->i_uid != current->fsuid ||
		    result->i_gid != current->fsgid ||
		    result->i_mode != (mode | S_IFREG)) {
			result->i_uid = current->fsuid;
			result->i_gid = current->fsgid;
			result->i_mode = mode | S_IFREG;
			hpfs_write_inode_nolock(result);
		}
		d_instantiate(dentry, result);
	}
	unlock_iget(dir->i_sb);
	hpfs_unlock_inode(dir);
	return 0;
	bail1:
	brelse(bh);
	free_sectors(dir->i_sb, fno, 1);
	hpfs_unlock_inode(dir);
	bail:
	return -ENOSPC;
}

int hpfs_mknod(struct inode *dir, struct dentry *dentry, int mode, int rdev)
{
	const char *name = dentry->d_name.name;
	int len = dentry->d_name.len;
	struct buffer_head *bh;
	struct fnode *fnode;
	fnode_secno fno;
	int r;
	struct hpfs_dirent dee;
	struct inode *result = NULL;
	if (!dir) return -ENOENT;
	if (dir->i_sb->s_hpfs_eas < 2) {
		/*iput(dir);*/
		return -EPERM;
	}
	if (!S_ISDIR(dir->i_mode)) {
		return -ENOTDIR;
	}
	if (!S_ISBLK(mode) && !S_ISCHR(mode) && !S_ISFIFO(mode) && !S_ISSOCK(mode)) {
		return -EINVAL;
	}
	adjust_length((char *)name, &len);
	if (chk_hpfs_name((char *)name, len)) {
		return -ENAMETOOLONG;
	}	
	if (!(fnode = alloc_fnode(dir->i_sb, dir->i_hpfs_dno, &fno, &bh))) goto bail;
	memset(&dee, 0, sizeof dee);
	if (!(mode & 0222)) dee.read_only = 1;
	dee.archive = 1;
	dee.hidden = name[0] == '.';
	dee.fnode = fno;
	dee.creation_date = dee.write_date = dee.read_date = gmt_to_local(dir->i_sb, CURRENT_TIME);
	hpfs_lock_inode(dir);
	r = add_dirent(dir, (char *)name, len, &dee, 0);
	if (r == 1) goto bail1;
	if (r == -1) {
		brelse(bh);
		free_sectors(dir->i_sb, fno, 1);
		hpfs_unlock_inode(dir);
		return -EEXIST;
	}
	fnode->len = len;
	memcpy(fnode->name, name, len > 15 ? 15 : len);
	fnode->up = dir->i_ino;
	mark_buffer_dirty(bh, 1);
	lock_iget(dir->i_sb, 2);
	if ((result = iget(dir->i_sb, fno))) {
		result->i_hpfs_parent_dir = dir->i_ino;
		result->i_ctime = result->i_mtime = result->i_atime = local_to_gmt(dir->i_sb, dee.creation_date);
		result->i_hpfs_ea_size = 0;
		if (result->i_blocks == -1) result->i_blocks = 512;
		if (result->i_size == -1) result->i_size = 0;
		result->i_mode = mode;
		result->i_uid = current->fsuid;
		result->i_gid = current->fsgid;
		if (!S_ISFIFO(mode)) result->i_rdev = to_kdev_t(rdev);
		result->i_nlink = 1;
		result->i_size = 0;
		result->i_blocks = 1;
		result->i_op = NULL;
		if (S_ISBLK(result->i_mode)) result->i_op = (struct inode_operations *) &blkdev_inode_operations;
		if (S_ISCHR(result->i_mode)) result->i_op = (struct inode_operations *) &chrdev_inode_operations;
		if (S_ISFIFO(result->i_mode)) init_fifo(result);
		hpfs_write_inode_nolock(result);
		d_instantiate(dentry, result);
	}
	unlock_iget(dir->i_sb);
	hpfs_unlock_inode(dir);
	brelse(bh);
	return 0;
	bail1:
	brelse(bh);
	free_sectors(dir->i_sb, fno, 1);
	hpfs_unlock_inode(dir);
	bail:
	return -ENOSPC;
}

extern const struct inode_operations hpfs_symlink_iops;

int hpfs_symlink(struct inode *dir, struct dentry *dentry, const char *symlink)
{
	const char *name = dentry->d_name.name;
	int len = dentry->d_name.len;
	struct buffer_head *bh;
	struct fnode *fnode;
	fnode_secno fno;
	int r;
	struct hpfs_dirent dee;
	struct inode *result;
	if (!dir) return -ENOENT;
	if (dir->i_sb->s_hpfs_eas < 2) {
		return -EPERM;
	}
	if (!S_ISDIR(dir->i_mode)) {
		return -ENOTDIR;
	}
	adjust_length((char *)name, &len);
	if (chk_hpfs_name((char *)name, len)) {
		return -ENAMETOOLONG;
	}
	if (!(fnode = alloc_fnode(dir->i_sb, dir->i_hpfs_dno, &fno, &bh))) goto bail;
	memset(&dee, 0, sizeof dee);
	dee.archive = 1;
	dee.hidden = name[0] == '.';
	dee.fnode = fno;
	dee.creation_date = dee.write_date = dee.read_date = gmt_to_local(dir->i_sb, CURRENT_TIME);
	hpfs_lock_inode(dir);
	r = add_dirent(dir, (char *)name, len, &dee, 0);
	if (r == 1) goto bail1;
	if (r == -1) {
		brelse(bh);
		free_sectors(dir->i_sb, fno, 1);
		hpfs_unlock_inode(dir);
		return -EEXIST;
	}
	fnode->len = len;
	memcpy(fnode->name, name, len > 15 ? 15 : len);
	fnode->up = dir->i_ino;
	mark_buffer_dirty(bh, 1);
	brelse(bh);
	lock_iget(dir->i_sb, 2);
	if ((result = iget(dir->i_sb, fno))) {
		result->i_hpfs_parent_dir = dir->i_ino;
		result->i_ctime = result->i_mtime = result->i_atime = local_to_gmt(dir->i_sb, dee.creation_date);
		result->i_hpfs_ea_size = 0;
		if (result->i_blocks == -1) result->i_blocks = 512;
		if (result->i_size == -1) result->i_size = 0;
		result->i_mode = S_IFLNK | 0777;
		result->i_uid = current->fsuid;
		result->i_gid = current->fsgid;
		result->i_blocks = 1;
		result->i_size = strlen(symlink);
		result->i_op = (struct inode_operations *) &hpfs_symlink_iops;
		if ((fnode = map_fnode(dir->i_sb, fno, &bh))) {
			set_ea(result, fnode, "SYMLINK", (char *)symlink, strlen(symlink));
			mark_buffer_dirty(bh, 1);
			brelse(bh);
		}
		hpfs_write_inode_nolock(result);
		d_instantiate(dentry, result);
	}
	unlock_iget(dir->i_sb);
	hpfs_unlock_inode(dir);
	return 0;
	bail1:
	brelse(bh);
	free_sectors(dir->i_sb, fno, 1);
	hpfs_unlock_inode(dir);
	bail:
	return -ENOSPC;
}

int hpfs_unlink(struct inode *dir, struct dentry *dentry)
{
	const char *name = dentry->d_name.name;
	int len = dentry->d_name.len;
	struct quad_buffer_head qbh;
	struct hpfs_dirent *de;
	struct inode *inode = dentry->d_inode;
	dnode_secno dno;
	fnode_secno fno;
	int depth, r;
	adjust_length((char *)name, &len);
	hpfs_lock_2inodes(dir, inode);
	if (!(de = map_dirent(dir, dir->i_hpfs_dno, (char *)name, len, &dno, &qbh, &depth))) {
		hpfs_unlock_2inodes(dir, inode);
		return -ENOENT;
	}
	if (de->first) {
		brelse4(&qbh);
		hpfs_unlock_2inodes(dir, inode);
		return -EPERM;
	}
	if (de->directory) {
		brelse4(&qbh);
		hpfs_unlock_2inodes(dir, inode);
		return -EISDIR;
	}
	shrink_dcache_parent(dentry);
	if (dentry->d_count > 1) {
	/*if (!list_empty(&dentry->d_hash)) {*/
		brelse4(&qbh);
		hpfs_unlock_2inodes(dir, inode);
		return -EBUSY;
	}
	fno = de->fnode;
	if ((r = remove_dirent(dir, dno, de, &qbh, depth)) == 1) hpfs_error(dir->i_sb, "there was error when removing dirent");
	if (r != 2) {
		if (inode) {
			inode->i_nlink = 0;
			inode->i_state &= ~I_DIRTY;
		}
		remove_fnode(dir->i_sb, fno);
		d_delete(dentry);
	} 
	hpfs_unlock_2inodes(dir, inode);
	return r == 2 ? -ENOSPC : 0;
}

int hpfs_rmdir(struct inode *dir, struct dentry *dentry)
{
	const char *name = dentry->d_name.name;
	int len = dentry->d_name.len;
	struct quad_buffer_head qbh;
	struct hpfs_dirent *de;
	struct inode *inode = dentry->d_inode;
	dnode_secno dno;
	fnode_secno fno;
	int n_items = 0;
	int depth, r;
	adjust_length((char *)name, &len);
	hpfs_lock_2inodes(dir, inode);
	if (!(de = map_dirent(dir, dir->i_hpfs_dno, (char *)name, len, &dno, &qbh, &depth))) {
		hpfs_unlock_2inodes(dir, inode);
		return -ENOENT;
	}	
	if (de->first) {
		brelse4(&qbh);
		hpfs_unlock_2inodes(dir, inode);
		return -EPERM;
	}
	if (!de->directory) {
		brelse4(&qbh);
		hpfs_unlock_2inodes(dir, inode);
		return -ENOTDIR;
	}
	count_dnodes(dir->i_sb, fnode_dno(dir->i_sb, de->fnode), NULL, NULL, &n_items);
	if (n_items) {
		brelse4(&qbh);
		hpfs_unlock_2inodes(dir, inode);
		return -ENOTEMPTY;
	}
	/*shrink_dcache_parent(dentry);
	if (dentry->d_count > 1) {*/
	if (!list_empty(&dentry->d_hash)) {
		brelse4(&qbh);
		hpfs_unlock_2inodes(dir, inode);
		return -EBUSY;
	}
	fno = de->fnode;
	if ((r = remove_dirent(dir, dno, de, &qbh, depth)) == 1)
		hpfs_error(dir->i_sb, "there was error when removing dirent");
	if (r != 2) {
		if (inode) {
			inode->i_nlink = 0;
			inode->i_state &= ~I_DIRTY;
		}
		dir->i_nlink--;
		remove_fnode(dir->i_sb, fno);
		d_delete(dentry);
	}
	hpfs_unlock_2inodes(dir, inode);
	return r == 2 ? -ENOSPC : 0;
}

int hpfs_readlink(struct dentry *dentry, char *buf, int len)
{
	struct inode *i = dentry->d_inode;
	struct fnode *fnode;
	struct buffer_head *bh;
	char *symlink;
	int slen;
	if (!S_ISLNK(i->i_mode)) {
		return -EINVAL;
	}
	if (!(fnode = map_fnode(i->i_sb, i->i_ino, &bh))) {
		return -EIO;
	}
	if (!(symlink = get_ea(i->i_sb, fnode, "SYMLINK", &slen))) {
		return -EFSERROR;
	}
	brelse(bh);
	if (slen > len) slen = len;
	memcpy_tofs(buf, symlink, slen);
	kfree(symlink);
	return slen;
}

struct dentry *hpfs_follow_link(struct dentry *dinode, struct dentry *ddir,
				unsigned int follow)
{
	struct inode *inode = dinode->d_inode;
	char *link;
	int len;
	struct buffer_head *bh;
	struct fnode *fnode;
	if (!(fnode = map_fnode(inode->i_sb, inode->i_ino, &bh))) {
		dput(dinode);
		return ERR_PTR(-EIO);
	}
	if (!(link = get_ea(inode->i_sb, fnode, "SYMLINK", &len))) {
		brelse(bh);
		dput(dinode);
		return ERR_PTR(-EIO);
	}
	brelse(bh);
	UPDATE_ATIME(inode);
	ddir = lookup_dentry(link, ddir, follow);
	kfree(link);
	return ddir;
}

int hpfs_rename(struct inode *old_dir, struct dentry *old_dentry,
		struct inode *new_dir, struct dentry *new_dentry)
{
	const char *old_name = old_dentry->d_name.name;
	int old_len = old_dentry->d_name.len;
	const char *new_name = new_dentry->d_name.name;
	int new_len = new_dentry->d_name.len;
	struct inode *i;
	struct quad_buffer_head qbh;
	struct hpfs_dirent *dep;
	struct hpfs_dirent de;
	dnode_secno dno;
	struct fnode *fnode;
	struct buffer_head *bh;
	int depth;
	int c;
	int err = 0;
	if (!old_dir || !new_dir) {
		return -ENOENT;
	}
	if (!S_ISDIR(old_dir->i_mode) || !S_ISDIR(new_dir->i_mode)) {
		err = -ENOTDIR;
		goto end2;
	}
	adjust_length((char *)old_name, &old_len);
	adjust_length((char *)new_name, &new_len);
	if (chk_hpfs_name((char *)new_name, new_len) || chk_hpfs_name((char *)old_name, old_len)) {
		err = -ENAMETOOLONG;
		goto end2;
	}
	if (!(i = old_dentry->d_inode)) {
		hpfs_error(old_dir->i_sb, "hpfs_rename: grrr, could not get inode");
		err = -ENOENT;
		goto end2;
	}
	if (i->i_sb != old_dir->i_sb || i->i_sb != new_dir->i_sb) {
		err = -EINVAL; /* Do not allow to move mount points */
		goto end1_;
	}
	if (new_dir != old_dir) {
		int c1, c2 = 0;
		fnode_secno fno = new_dir->i_ino;
		if (old_dir->i_sb->s_hpfs_root == i->i_ino) {
			err = -EINVAL;
			goto end1_;
		}
		while (fno != old_dir->i_sb->s_hpfs_root) {
			if (old_dir->i_sb->s_hpfs_chk)
				if (stop_cycles(old_dir->i_sb, fno, &c1, &c2, "hpfs_rename")) {
					err = -EFSERROR;
					goto end1_;
				}
			if (fno == i->i_ino) {
				err = -EINVAL;
				goto end1_;
			}
			if (!(fnode = map_fnode(new_dir->i_sb, fno, &bh))) {
				err = -EFSERROR;
				goto end1_;
			}
			fno = fnode->up;
			brelse(bh);
		}
	}

	if (new_dir != old_dir || compare_names(old_dir->i_sb, (char *)old_name, old_len, (char *)new_name, new_len, 0)) {
		err = hpfs_unlink(new_dir, new_dentry);
		if (err == -ENOENT) err = 0;
		if (err) goto end1_;
	}
	
	hpfs_lock_3inodes(old_dir, new_dir, i);
	if (!(dep = map_dirent(old_dir, old_dir->i_hpfs_dno, (char *)old_name, old_len, &dno, &qbh, &depth))) {
		hpfs_error(i->i_sb, "lookup succeeded but map dirent failed");
		err = -ENOENT;
		goto end1;
	}
	copy_de(&de, dep);
	de.hidden = new_name[0] == '.';
	lock_creation(old_dir->i_sb);
	c = add_dirent(new_dir, (char *)new_name, new_len, &de, depth + 2);
	if (c == -1) {
		dnode_secno down = 0;
		if (new_dir != old_dir) {
			brelse4(&qbh);
			err = -EEXIST;
			goto end;
		}
		if (check_free_dnodes(old_dir->i_sb, depth + 2)) {
			brelse4(&qbh);
			err = -ENOSPC;
			goto end;
		}
		if (de.down) down = de_down_pointer(&de);
		delete_de(old_dir->i_sb, qbh.data, dep);
		mark_4buffers_dirty(&qbh);
		brelse4(&qbh);
		if (add_to_dnode(old_dir, dno, (char *)new_name, new_len, &de, down)) {
			err = -EFSERROR;
			goto end;
		}
		goto ch_fnode;
	}
	if (c == 1) {
		err = -EFSERROR;
		goto end;
	}
	brelse4(&qbh);
	if (!(dep = map_dirent(old_dir, old_dir->i_hpfs_dno, (char *)old_name, old_len, &dno, &qbh, &depth))) {
		hpfs_error(i->i_sb, "lookup succeeded but map dirent failed #2");
		err = -EFSERROR;
	} else if (remove_dirent(old_dir, dno, dep, &qbh, 0)) err = -EFSERROR;
	i->i_hpfs_parent_dir = new_dir->i_ino;
	ch_fnode:
	if (i->i_sb->s_hpfs_conv == CONV_AUTO) {
		i->i_hpfs_conv = CONV_AUTO;
		decide_conv(i, (char *)new_name, new_len);
	}
	if ((fnode = map_fnode(i->i_sb, i->i_ino, &bh))) {
		if (fnode->dirflag) {
			old_dir->i_nlink--;
			new_dir->i_nlink++;
		}
		if (fnode->up != new_dir->i_ino || fnode->len != new_len ||
		    memcmp(fnode->name, new_name, new_len>15?15:new_len)) {
			fnode->up = new_dir->i_ino;
			fnode->len = new_len;
			memcpy(fnode->name, new_name, new_len>15?15:new_len);
			if (new_len < 15)
				memset(&fnode->name[new_len], 0, 15 - new_len);
			mark_buffer_dirty(bh, 1);
		}
		brelse(bh);
	}
	end:
	if (!err) d_move(old_dentry, new_dentry);
	unlock_creation(old_dir->i_sb);
	end1:
	hpfs_unlock_3inodes(old_dir, new_dir, i);
	end1_:
	end2:
	return err;
}

